################################################################################
#
#       This is part of CATBox (Combinatorial Algorithm Toolbox) 
#       version 1.0 from 2/25/10. You can find more information at 
#       http://algorithmics.molgen.mpg.de/CATBox/
#
#	file:   MatroidDualKruskal.pro
#	author: Alexander Schliep (alexander@schliep.org)
#
#       Copyright (C) 1998-2010, Winfried Hochstaettler und Alexander Schliep
#	and (c) 2010 Springer Verlag.
#       
#       All rights reserved. Do not distribute. 
#	See http://schliep.org/CATBox for more information.
#




#
#       This file is version $Revision: 1.1 $ 
#                       from $Date: 2003/10/24 15:33:47 $
#             last change by $Author: schliep $.
#
#
################################################################################

#### Options #################################################
self.NeededProperties({'Connected':1,'Simple':1})
breakpoints = [4]
interactive = []
graphDisplays = 1
about = """<HTML>
<HEAD>
<TITLE>Minimal Spanning Tree - Matroid Dual Kruskal</TITLE>
</HEAD>
<BODY>

This algorithm finds a minimal spanning tree in an undirected graph
using the dual approach to Kruskals algorithm.

<H3>Visualization</H3>

<H5>Vertex Colors</H5>

<dl>

<dt><colordef color="blue"></dt> <dd>The spanning subgraph in the input which will be the MST
upon termination of the algorithm.  Initially, all the vertices and
edges are part of the spanning subgraph.  </dd>

</dl>


<H5>Edge Colors</H5>
<dl>

<dt><colordef color="blue"></dt> <dd>Edges which are part of the spanning subgraph.</dd>

<dt><colordef color="yellow"></dt> <dd>Color of the edge being
considered for removal from the spanning subgraph. (transient)</dd>

<dt><colordef color="grey"></dt> <dd>Color of edges which have been removed from
the spanning subgraph.</dd>

</dl>

</BODY></HTML>
"""

#### Wrapper #################################################

class MySubGraph(AnimatedSubGraph):

    def __init__(self, G, A):

        AnimatedSubGraph.__init__(self, G, A)

        # Adds all vertices!
        for (u,v) in G.Edges():
            self.AddEdge(u,v)
        
        #self.bfs = AnimatedSubGraph(G,A,"red")
        self.bfs = SubGraph(G)
        for (u,v) in self.BFSEdges(G, G.vertices[0]):
            self.bfs.AddEdge(u,v)

    def BFSEdges(self, G, root):
        (d,pred) = BFS(G,root)
        result = []
        for v in G.vertices:
            if pred[v] != v: # Non-root
                result.append((v,pred[v]))
        return result


    def BFSReachableNodes(self, H, root):
        (d,pred) = BFS(H,root)                
        reachableNodes = filter(lambda v, dist=d: dist[v] != gInfinity, d.keys())
        return reachableNodes
    
    def RemoveEdge(self,(u,v)):
        self.DeleteEdge(u,v)
        A.SetEdgeColor(u,v,"grey")
        A.DefaultInfo()

    def DisconnectingGraph(self,e):
        if self.Size() == self.Order() -1:
            return 1

        if self.bfs.QEdge(e[0],e[1]):
            # edge e is part of our BFS, check whether we can replace it
            self.bfs.DeleteEdge(e[0],e[1])

            # Compute the two partitions 
            blueNodes = self.BFSReachableNodes(self.bfs, e[0])
            redNodes = self.BFSReachableNodes(self.bfs, e[1])

            # Check whether we have an edge with one blue and one
            # red endpoint
            for u in blueNodes:
                for v in self.Neighborhood(u):
                    if v in redNodes: # Found one
                        if (u,v) != e:
                            self.bfs.AddEdge(u,v)
                            #print "bfs.AddEdge(%d,%d)" %  (u,v)
                            return 0

            # No replacement => Add deleted edge again ...
            self.bfs.AddEdge(e[0],e[1])
            return 1 
            
        else: # edge e is not in our BFS, can be removed
            return 0

 #### Variables ###############################################

weight          = G.edgeWeights[0]
component       = {}
Edges           = G.Edges()

#### Graph management ########################################

T = MySubGraph(G,A)

#### Lambdas #################################################

MergeComponents = lambda src, dest: dest.AddSubGraph(src)
NewComponent    = lambda cm = ComponentMaker(G,A): cm.NewComponent()
Order           = lambda D: D.Order()

DisconnectingGraph = lambda e, t=T: t.DisconnectingGraph(e)
RemoveEdge = lambda e, t=T: t.RemoveEdge(e)


#### Graphinformer ###########################################

class MyMSTGraphInformer(GraphInformer):

    def EdgeInfo(self,tail,head):
        w = self.G.edgeWeights[0][(tail, head)]

        if self.G.edgeWeights[0].QInteger():
            return "Weight of edge (%d,%d): %d - Weight of spanninggraph: %d" % (tail, head, w,
                                                                             T.Weight()) 
        else:
            return "Weight of edge (%d,%d): %0.2f - Weight of spanning graph: %0.2f" % (tail, head, w,
                                                                             T.Weight()) 



    def DefaultInfo(self):
        return "Weight of spanning graph: %f" % float(T.Weight()) 

#### Internal functions ######################################

def byDecreasingWeight(x,y):
    if weight[x]==weight[y]:
        return 0
    elif weight[x]>weight[y]:
        return -1
    else:
        return 1

def Sort(E,b):
    E.sort(b)
    return ContainerWrapper(A,E,"yellow")

#### Internal initialisation #################################

A.RegisterGraphInformer(MyMSTGraphInformer(G))



















