################################################################################
#
#       This is part of CATBox (Combinatorial Algorithm Toolbox)
#       version 1.0 from 2/25/10. You can find more information at
#       http://algorithmics.molgen.mpg.de/CATBox/
#
#	file:   PrimalDualKruskal.pro
#	author: Alexander Schliep (alexander@schliep.org)
#
#       Copyright (C) 1998-2010, Winfried Hochstaettler und Alexander Schliep
#	and (c) 2010 Springer Verlag.
#       
#       All rights reserved. Do not distribute. 
#	See http://schliep.org/CATBox for more information.
#




#
#       This file is version $Revision: 1.1 $ 
#                       from $Date: 2003/03/14 18:01:47 $
#             last change by $Author: schliep $.
#
#
################################################################################


#### Options #################################################

breakpoints = [1]
interactive = []
graphDisplays = 2
about = """<HTML>
<HEAD>
<TITLE>Minimal Spanning Tree - Kruskal (Primal / Dual)</TITLE>
</HEAD>
<BODY>

<H1>Minimal Spanning Tree - Kruskal (Primal / Dual)</H1>

This algorithm finds a minimal spanning tree. Edges in the input are
disregarded. Vertices in the input are interpreted as points in the
Euclidean plane and the resulting MST can contain edges not present
in the input.

<H3>Visualization (Main Graph)</H3>

<H5>Moat Colors</H5> <dl> <dt><colordef color="#FF0000"><colordef
color="#FF8800"><colordef color="#FFFF00">...</dt> <dd> Moats are
graphical representations of dual variables for individual vertices or
groups of vertices. The growing of moats visualizes increases in the
value of the corresponding dual variables.</dd>

</dl>


<H5>Edge Colors</H5>
<dl>

<dt><colordef color="#EEEEEE"></dt> <dd>Edges in this color are
part of the MST.</dd>

</dl>


<H3>Visualization (MST)</H3>

<H5>Vertex Colors</H5>

<dl>

<dt><colordef color="red"><colordef color="green"><colordef color="blue">...</dt> <dd>Vertex (and edge) colors designate the connected
components in the subgraph induced by the trees during the
computation. Therefore, each vertex has its own colors initially.</dd>

</dl>



<H5>Edge Colors</H5>

<dl>

<dt><colordef color="red"><colordef color="green"><colordef color="blue">...</dt> <dd>Edges
in the (partial) MST; color indicates the connected component.</dd>



</dl>




</BODY></HTML>
"""
#### Misc ####################################################

from math import sqrt

for e in G.Edges():
    G.DeleteEdge(e[0],e[1])

import copy
GS = copy.deepcopy(G)
self.OpenSecondaryGraph(GS, stripPath(self.graphFileName) + " (spanning)")
GSA = self.GUI.secondaryGraphDisplay
A.ShowGraph(G, stripPath(self.graphFileName))



class Moat:
    def __init__(self, A, pos, embed_radius, color):
        d = (embed_radius * A.zoomFactor ) / 100.0 #Canvas radius!
        self.d = d
        self.pos = pos
        m = A.canvas.create_oval(pos[0]-d, pos[1]-d,
                                 pos[0]+d, pos[1]+d, 
                                 fill=color, 
                                 tag="moats",
                                 width=0) 
        self.m = m
        try:
            A.canvas.lower(m,"edges")
            A.canvas.lower(m,"labels")
            A.canvas.lower(m,"moats")
        except:
            pass
        self.A = A

    def grow(self, new_embed_radius):
        d = (new_embed_radius * A.zoomFactor ) / 100.0 #Canvas radius!
        self.A.canvas.coords(self.m,
                             self.pos[0]-d, self.pos[1]-d,
                             self.pos[0]+d, self.pos[1]+d)


class MoatedPartition:
    """ A Moated partition is a collection of connected components
        where each vertex in a component has a moat around it.
        The moated partition keeps track of distances between vertices """
    def __init__(self, g, gs, gaAsMoats, gaAsGraph):
        self.g = g
        self.gs = gs
        self.gaAsMoats = gaAsMoats
        self.gaAsGraph = gaAsGraph
        
        self.moatRadius = {} # Radius of the moat per vertex
        for v in self.g.vertices:
            self.moatRadius[v] = 0.0

        self.mr = 0.0

        self.D = {} # Dictionary of distance d(u,v). Assume u < v because of symmetry
        for u in self.g.vertices:
            for v in self.g.vertices:
                if u < v:
                    tmp = (self.g.embedding[u].x - self.g.embedding[v].x)**2
                    tmp += (self.g.embedding[u].y - self.g.embedding[v].y)**2
                    self.D[(u,v)] = max(round(sqrt(tmp)), 0.0) # Enforce non-negative distances

        self.cm = ComponentMaker(self.g, self.gaAsGraph)
        self.components = []
        self.component = {} 
        for u in self.g.vertices:
            C = self.cm.NewComponent()
            C.AddVertex(u)
            self.components.append(C)
            self.component[u] = C

        self.colorList = ["#FF0000","#FF8800","#FFFF00","#00FF00","#00FFFF","#0088FF",
                          "#0000FF","#8800FF","#FF00FF"]
        self.colorIndex = -1
        self.InitMoats()
        

    def distance(self, u, v):
        if u < v:
            return self.D[(u,v)]
        else:
            return self.D[(v,u)]


    def findMinimalDistance(self):
        """ find the (u,v) having the minimal positive entry in self.D """
        def swapTupel(t):
            return (t[1],t[0])

        # items() gives (key, value)-List, swap yields (value, key), and max works on that
        (d, e) = min(filter(lambda t: t[0] > 0.0, map(swapTupel,self.D.items())))
        return (e,d)


    def InitMoats(self):
        self.gaAsMoats.DeleteDrawVertices()

    
    def NextColor(self):
        if self.colorIndex == len(self.colorList) - 1:
            self.colorIndex = - 1
        self.colorIndex = self.colorIndex + 1
        return self.colorList[self.colorIndex]
       
        
    def GrowMoats(self,dist):
        moatColor = self.NextColor()

        moats = []
        old_mr = self.mr
        self.mr += dist/2.0
        
        for C in self.components:
            for v in C.vertices:
                pt = self.gaAsMoats.G.GetEmbedding(v)  
                x,y = self.gaAsMoats.EmbeddingToCanvas(pt.x,pt.y)
                m = Moat(self.gaAsMoats, (x,y), old_mr, moatColor) 
                moats.append(m)

        stepwidth = 0.05
        steps = round((self.mr - old_mr)/stepwidth)
        delta = (self.mr - old_mr) / steps

        for i in xrange(1, steps+1):
            for m in moats:
                m.grow(old_mr + i*delta)
            self.gaAsMoats.update()

        self.tightEdges = self.UpdateExtraComponentDistances(dist)


    def UpdateComponents(self):
        for e in self.tightEdges: # self.tightEdges is set by GrowMoats. Call first!
            (u,v) = e
            if self.component[u] is not self.component[v]:
                self.gaAsMoats.AddEdge(u,v)
                srcComp = self.component[v]
                self.component[u].AddSubGraph(srcComp)
                self.component[u].AddEdge(u,v)
                for v in srcComp.vertices:
                    self.component[v] = self.component[u]
                self.components.remove(srcComp)

        self.UpdateIntraComponentDistances()


    def UpdateExtraComponentDistances(self, dist):
        tightEdges = []
        growingFactor = dist / 2
        for C in self.components:
            for D in self.components:
                if C is not D:
                    for u in C.vertices:
                        for v in D.vertices:
                            if u < v:
                                e = (u,v)
                            else:
                                e = (v,u)
                            self.D[e] = self.D[e] - growingFactor
                            if self.D[e] == 0.0:
                                tightEdges.append(e)
                                self.gaAsGraph.AddEdge(e[0],e[1])
                                    
        return tightEdges

    def UpdateIntraComponentDistances(self):
        for C in self.components:
            for u in C.vertices:
                for v in C.vertices:
                    if u is not v:
                        self.D[(u,v)] = 0.0
          
    
MP = MoatedPartition(G, GS, A, GSA)

nrOfComponents = lambda mp=MP: len(mp.components)
UpdateDuals = lambda mp=MP: mp.GrowMoats(mp.findMinimalDistance()[1])
ComputePartition = lambda mp=MP: MP.UpdateComponents()
