################################################################################
#
#       This is part of CATBox (Combinatorial Algorithm Toolbox)
#       version 1.0 from 2/25/10. You can find more information at
#       http://algorithmics.molgen.mpg.de/CATBox/
#
#	file:   WeightedMatching.pro
#	author: Winfried Hochstaettler
#               Alexander Schliep (alexander@schliep.org)
#
#       Copyright (C) 1998-2010, Winfried Hochstaettler und Alexander Schliep
#	and (c) 2010 Springer Verlag.
#       
#       All rights reserved. Do not distribute. 
#	See http://schliep.org/CATBox for more information.
#




#
#       This file is version $Revision: 1.29 $
#                       from $Date: 2004/09/20 16:19:44 $
#             last change by $Author: schliep $.
#
#
################################################################################

#
#
#
#
#### Options #################################################

breakpoints = [3]
interactive = []
graphDisplays = 2
about = """<HTML>
<HEAD>
<TITLE>Matching</TITLE>
</HEAD>
<BODY>

This algorithm finds a maximum weight matching.

</BODY></HTML>
"""


#from AnimatedDataStructures import AnimatedSubgraph
import copy
import math
import string
eps_prec = 0.01

#### Utility functions #########################################
def is_supervertex(v):
    return isinstance(v,tuple) or (isinstance(v,int) and v > G.Order())

def argmin(a):
    return a.index(min(a))

def argmin_min(a):
    m = min(a)
    return a.index(m), m

def UnmatchedVertices(G, M):
    result = set(G.Vertices())
    result.difference_update(set(M.Vertices()))
    return result

def dist_with_duals(G, u, v, w):
    d = dist(G, v, w) - u[v] - u[w]
    for key in u.duals[v]:
        if not key in u.duals[w]:
            d -= u[key]
    for key in u.duals[w]:
        if not key in u.duals[v]:
            d -= u[key]
    return d

def matched(M,v):
    return v in M.Vertices()

def matchingPartner(M,v):
    if v in M.Vertices():
        nbh = M.Neighborhood(v)
        #if len(nbh) > 1:
        #    print "matchingPartner(M,%d) is multiply matched" % v
        return nbh[0]

def mindist(V, W):
    eps = 99999.99
    tight_edges = []

    #V = expand_supervertices(V)
    #W = expand_supervertices(W)

    #print "mindist between sets", V, W

    # NOTE: We need to compare all un-ordered pairs v \in V, w \in W,
    # v \neq w As V and W can contain the same supervertex we have to
    # compare all the un-ordered pairs such that v and

    for vv in V:
        vv_tmp = vv
        if not is_supervertex(vv):
            vv = (vv,)
        else:
            vv = u.expand_supervertices(vv)
        for ww in W:
            if vv_tmp == ww:
                continue
            if not is_supervertex(ww):
                ww = (ww,)
            else:
                ww = u.expand_supervertices(ww)
            #print "after expand vv=", vv, "ww=", ww
            for v in vv:
                for w in ww:
                    if v is not w:
                        #print "dist", v, w, type(v), type(w), G.embedding.label
                        #print G, G.Vertices()
                        #print "dist", v, w, dist(G, v, w), u[v], u[w], "duals=", u.duals[v], u.duals[w]
                        d = dist_with_duals(G, u, v, w)
                        if d < eps - eps_prec:
                            if v < w:
                                tight_edges = [(v,w)]
                            else:
                                tight_edges = [(w,v)]
                            eps = d
                        elif (d - eps) < eps_prec:
                            if v < w:
                                tight_edges.append((v,w))
                            else:
                                tight_edges.append((w,v))

    tight_edges = list(set(tight_edges)) # remove duplicates
    return eps, tight_edges


def FindSuspectEdge(G, M):
    """

    
    """
    Q = Queue()
    pred = {}
    label = {}
    base = {}

    for v in UnmatchedVertices(G, M):
        pred[v] = v
        base[v] = v
        label[v] = 1
        Q.Append(v)

    #print "FindSuspectEdge: UnmatchedVertices", UnmatchedVertices(G, M)

    while Q.IsNotEmpty():
        v = Q.Top()
        for w in G.Neighborhood(v):
            if not pred.has_key(w):
                pred[w] = v
                base[w] = base[v]
                label[w] = 0
                
                if w in M.Vertices():
                    #print "FindSuspectEdge", v, w, M.Neighborhood(w)
                    u = M.Neighborhood(w)[0] # Unique since matching
                    pred[u] = w
                    base[u] = base[w]
                    label[u] = 1
                    Q.Append(u)
            elif pred[v] != w and label[w] != 0:
                #print "FindSuspectEdge", (v,w)
                return ((v,w), pred, base, label)
    return (None, pred, base, label) # Need to return for GallaiEdmondsDecomp
    

def FindAugmentingPath(G, M, pred, e):
    """Find an augmenting path containing e in the ShrinkableGraph G
       using the predecessor array pred
    """
    P = [e]
    v = e[0]
    while pred[v] != v:
        P.insert(0,(pred[v],v))
        v = pred[v]

    v = e[1]
    while pred[v] != v:
        P.append((v,pred[v]))
        v = pred[v]

    #print "FindAugmentingPath", P
    return P


class Bubble:
    """Graphical class representing bubbles. There are two types of bubbles: the ones
       around vertices and those around supervertices. In the latter case the minimal and
       initial bubble size is determined by the bubbles of the vertices it contains.
    """

    def __init__(self, G, A, u, key, vertices, offset_value, color = "red"):
        self.G = G
        self.A = A
        self.nr_animation_steps = 25 #250
        self.ovals = [] # Supervertices have several ovals
        self.vertices = vertices
        self.value = 0.0
        self.offset_value = offset_value
        
        for v in vertices:
            pos = self.A.VertexPosition(v)
            oval = self.A.canvas.create_oval(pos.x, pos.y, pos.x, pos.y, 
                                             fill=color, outline=color, 
                                             tag="bubbles", width=0)
            self.A.canvas.lower(oval,"vertices")
            self.A.canvas.lower(oval,"edges")
            self.A.canvas.lower(oval,"bubbles")

            self.A.canvas.tag_bind(oval, "<Any-Leave>", self.A.DefaultInfo)
            self.A.canvas.tag_bind(oval, "<Any-Enter>", \
                                   lambda event, x=key: u.DualInfo(event,x))
           
            self.ovals.append(oval)
            

    def blow(self, new_val):
        size = self.tocanvas(self.value)
        new_size = self.tocanvas(new_val)
        self.value = new_val

        delta_size = (new_size - size) / self.nr_animation_steps 
        while size < new_size:
            size = min(new_size, size + delta_size)
            for i, oval in enumerate(self.ovals):
                s = size + self.tocanvas(self.offset_value[i])
                pos = self.A.VertexPosition(self.vertices[i])
                self.A.canvas.coords(oval, pos.x-s, pos.y-s,
                                     pos.x+s, pos.y+s)
            self.A.update()

    def deflate(self, new_val):
        size = self.tocanvas(self.value) 
        new_size = self.tocanvas(new_val)
        self.value = new_val

        delta_size = (size - new_size) / self.nr_animation_steps 
        while size > new_size:
            size = max(new_size, size - delta_size)          
            for i, oval in enumerate(self.ovals):
                s = size + self.tocanvas(self.offset_value[i])                
                pos = self.A.VertexPosition(self.vertices[i])
                self.A.canvas.coords(oval, pos.x-s, pos.y-s,
                                     pos.x+s, pos.y+s)
            self.A.update()

    def __del__(self):
        for oval in self.ovals:
            self.A.canvas.delete(oval)
        

    def tocanvas(self, x):
        return (x * self.A.zoomFactor) / 100.0
        



class DualVars:
    """Dictionary of dual variables: NOTE: we use wmc for getting the depth """

    def __init__(self, G, A):
        self.G = G
        self.A = A
        self.vars = {} # List of variables: either vertices or
        # supervertices are keys
        self.bubbles = {} # Corresponding bubbles
        self.duals =  [ [] for i in xrange(G.Order() + 1)] # Which bubble dual variables are relevant
        self.vertices = {} # Dictionary of plain, expandend vertex lists
        self.colorList = ["#EE0000","#FF4400","#FF9900","#FFFF00",
                          "#DDDD99","#CCCC66","#CCCC33","#CCFF00","#33FF00",
                          "#99FF66","#66CC99","#66CCCC","#0099FF","#0033CC",
                          "#330099","#333399","#666699","#9999CC","#CCCCFF"]
        

    def expand_supervertices(self, sv):
        vertices = []
        for v in sv:
            if v > self.G.Order(): # v itself is a supervertex
                vertices += self.vertices[tuple(wmc.SG.blossom[v])] # XXX BAD global var and using its internals
            else:
                vertices.append(v)
        return vertices

    def compute_offsets(self, vertices):
        offsets = [0.0] * len(vertices)
        for i, v in enumerate(vertices):
            offsets[i] += self.vars[v]
            for key in self.duals[v]:
                offsets[i] += self.vars[key]
        #print "expand_supervertex", vertices, offsets
        return offsets

    def add_variable(self, key, value):
        #print "Adding dual", key
        if is_supervertex(key):
            depth = wmc.depth[key]
            self.vertices[key] = self.expand_supervertices(key)
            offset_values = self.compute_offsets(self.vertices[key])
            #print "Adding dual", key, " with vertices ", self.vertices[key] 
            for v in self.vertices[key]:
                self.duals[v].append(key)
        else:
            depth = 0
            self.vertices[key] = [key]
            offset_values = [0.0]
        color = self.colorList[depth % len(self.colorList)]
        self.vars[key] = 0.0
        #print "Adding dual", key, depth, color
        self.bubbles[key] = Bubble(G, self.A, u, key, self.vertices[key], offset_values, color)
        
    def del_variable(self, key):
        #Assume key is expanded
        for v in self.vertices[key]:
            self.duals[v].remove(key)
        del self.bubbles[key]
        del self.vars[key]
        del self.vertices[key]
        
    def __setitem__(self, key, value):
        if not self.vars.has_key(key):
            self.add_variable(key, value)
        if value > self.vars[key]:
            self.bubbles[key].blow(value)
        else:
            self.bubbles[key].deflate(value)
        self.vars[key] = value

    def has_key(self,key):
        return self.vars.has_key(key)

    def __getitem__(self, key):
        return self.vars[key]

    def __str__(self):
        return string.join(map(lambda t:" %s = %f" % t, self.vars.iteritems()), "\n")

    def DualInfo(self,event, key):
        self.A.UpdateInfo("Dual var u[%s] = %0.2f" % (key, self.vars[key]))
            

class ShrinkableGraph(Graph):

    def __init__(self, G, animator = None):
        """animator: Needed"""
        Graph.__init__(self)
        self.directed = 0
        self.supervertex = {}
        self.blossom = {}
        self.g_order = G.Order()
        self.true_edges = {}

        for v in G.Vertices():
            Graph.AddVertex(self)
        for e in G.Edges():
            Graph.AddEdge(self,e[0],e[1])

        if animator != None:
            self.labeling = copy.deepcopy(G.labeling)
            self.embedding = copy.deepcopy(G.embedding)
            self.edgeWeights = copy.deepcopy(G.edgeWeights)
            self.animator = animator
            self.graph_edit = animator
        else:
            self.graph_edit = self
            
    def addVertex(self, B=None):
        if B is None or self.animator is None:
            return Graph.AddVertex(self)
        else:
            # Compute position as center of blossom
            x = 0.0
            y = 0.0
            for v in B:
                x += self.embedding[v].x
                y += self.embedding[v].y
            v = Graph.AddVertex(self)
            return self.animator.AddVertex(x / len(B), y / len(B), v)

    def ShrinkBlossom(self, B):
        """For B the vertex set of an odd cycle in 'self', remove the
           vertices, add a new supervertex representing the cycle and
           return the supervertex
        """
        sv = self.addVertex(B)
        #print "ShrinkableGraph.ShrinkBlossom add supervertex", sv
        self.blossom[sv] = B

        # Add edges between sv and non-blossom neighbors of blossom
        for v in B:
            for w in wmc.SGV.Neighborhood(v):
                if not w in B and not self.QEdge(sv,w):
                    #print "ShriBlo adding", (sv,w), "with true edge", (v,w)
                    wmc.SG.AddEdge(sv,w)
                    self.graph_edit.AddEdge(sv,w)
                    if self.true_edges.has_key((sv,w)):
                        self.true_edges[(sv,w)].append((v,w))
                    else:
                        self.true_edges[(sv,w)] = [(v,w)]

            self.supervertex[v] = sv
               

        # Delete matching edges within blossom
        for v, w in wmc.SGM.Edges():
            if v in B and w in B:
                wmc.SGM.DeleteEdge(v,w)

            # Add matching edge to outside of blossom
            if (v in B and not w in B) or (not v in B and w in B):
                tmpv, tmpw = wmc.SGV.Edge(sv,w) 
                wmc.SGM.AddEdge(tmpv, tmpw)
                if v in B:
                    wmc.SGM.DeleteVertex(v)
                else:
                    wmc.SGM.DeleteVertex(w)

        for v in B:
            self.graph_edit.DeleteVertex(v)
        return sv

    def expand_supervertex(self, v, returnDepth = None):
        # Recursively expand to bottom
        terminate = 0
        depth = 0
        result = self.blossom[v][:]
        #print "expand_supervertex", v, result,
        while not terminate:
            terminate = 1            
            for i, w in enumerate(result):
                if self.is_supervertex(w):
                    result[i:i+1] = self.blossom[w]
                    #print result
                    terminate = 0
            depth += 1
        #print depth

        if returnDepth is None:
            return tuple(result)
        else:
            return (tuple(result), depth)

    def is_supervertex(self,v):
        return v > self.g_order

    def BlossomContaining(self, (v,w), pred):
        """Return the vertex set of the blossom containing edge e give the
           predecessor array pred
        """
        blossom = []
        
        p0 = [v]
        p1 = [w]

        for p in [p0, p1]:
            while pred[p[-1]] != p[-1]:
                p.append(pred[p[-1]])

        for i, v in enumerate(p0):
            if v in p1:
                break
        j = p1.index(v)

        p1slice = p1[0:j] # Need the first common just once
        p1slice.reverse()
        blossom = p0[0:i+1] + p1slice
        #print "ShrinkableGraph.BlossomContaining", blossom #p0, p1, blossom
        #print "pred=",pred
        return blossom

    def Augment(self, M, P):

        #print "Augment: Path", P
        # Unmatch first
        i = 0
        for (v,w) in P:
            if i % 2:
                (tv, tw) = self.findTrueEdge(v,w)
                #print "Augment deleting:", (tv,tw), "in G", (v,w), "in SG"                
                M.DeleteEdge(tv,tw)
                # Delete(v,w) in matching of SGV (SGM defined in Gallai-Edmonds: expose?)
                v,w = wmc.SGM.Edge(v,w)
                wmc.SGM.DeleteEdge(v,w)
            i += 1
        
        # match second
        i = 0
        for (v,w) in P:
            if i % 2 == 0:
                v,w = wmc.SG.Edge(v,w)
                tv, tw = self.findNextLevelEdge(v,w)
                ttv, ttw = self.findTrueEdge(v,w)
                #print "Augment: looking at", (v,w), "next=", (tv,tw)
                if matched(M,ttv):
                    self.HypoMatch(M,tv,(tv,tw))
                if matched(M,ttw):
                    self.HypoMatch(M,tw,(tv,tw))

                #print "Augment adding:", (tv,tw), "in G", (v,w), "in SG"
                ttv, ttw = self.findTrueEdge(v,w)
                if not M.QEdge(ttv,ttw):
                    M.AddEdge(ttv,ttw)
                wmc.SGM.AddEdge(v,w)
                #Add (v,w) to SGM

                # Expand blossoms along augmenting path
                # WH special add faked true edges to SG ...
                if self.is_supervertex(v) and self.is_supervertex(w):
                    tv, tw = self.true_edges[(v,w)][0]
                    if not self.QEdge(tv,w):
                        self.AddEdge(tv,w)
                    if not self.QEdge(v,tw):
                        self.AddEdge(v,tw)
                    if self.true_edges.has_key((tv,w)):
                        self.true_edges[(tv,w)].append((tv, tw))
                    else:
                        self.true_edges[(tv,w)] = [(tv, tw)]
                    if self.true_edges.has_key((v,tw)):
                        self.true_edges[(v,tw)].append((tv, tw))
                    else:
                        self.true_edges[(v,tw)] = [(tv, tw)]
                    #print "WH special:", (tv,w), (v,tw), (tv, tw)
                    
                    
                for vv in (v,w):
                    #print "Augment expanding considering", vv, self.is_supervertex(vv),
                    if self.is_supervertex(vv) and vv in wmc.sv_added:
                        #print self.blossom[vv]
                        wmc.Unshrink(self.blossom[vv])
                        # Deletion done in Unshrink
                        #for tmpv in self.blossom[vv]:
                        #    del(self.supervertex[tmpv])
                        del(self.blossom[vv])
                    #print
                   
            i += 1
            #print "return from Augment"


    def HypoMatch(self, M, v, entry_edge):
        """Hypomatch inside a blossom given the entry vertex v
           which should be part of a blossom"""
        if not self.supervertex.has_key(v):
            #print "HypoMatch", v, "is no supervertex"
            return
        sv = self.supervertex[v]
        b = self.blossom[sv]
        vi = b.index(v)
        #print "HypoMatch", v, b

        (tv1,tw1) = self.findTrueEdge(v, b[(vi + 1) % len(b)])
        (tv2,tw2) = self.findTrueEdge(v, b[(vi - 1) % len(b)])

        #print "Hypo",(tv1,tw1), M.QEdge(tv1,tw1), (tv2,tw2), M.QEdge(tv2,tw2)

        if M.QEdge(tv1,tw1): # walk forward in blossom
            for i in xrange(len(b)):
                wi = (vi + 1) % len(b)
                tv, tw = self.findNextLevelEdge(b[vi], b[wi])
                #print "walk forward blos:", (b[vi], b[wi]), "next=", (tv, tw)
                
                if i % 2 == 0:
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    try:
                        ttv, ttw = M.Edge(ttv,ttw)
                    except NoSuchEdgeError: # Found two consecutive unmatched edges
                        #print "Hypomatch break ",v
                        break
                    
                    M.DeleteEdge(ttv,ttw)
                else:
                    if self.true_edges.has_key((b[vi], b[wi])):
                        firstv, firstw = self.true_edges[(b[vi], b[wi])][0]
                        #print "Hypo true_edge", (firstv, firstw)
                    elif self.true_edges.has_key((b[wi], b[vi])):
                        firstw, firstv = self.true_edges[(b[wi], b[vi])][0]
                        #print "Hypo true_edge", (firstv, firstw)
                    else:
                        (firstv, firstw) = wmc.G.Edge(b[vi],b[wi])
                        #print "Hypo plain edge", (firstv, firstw)
                       

                    if is_supervertex(b[vi]):
                        self.HypoMatch(M,firstv,(firstv, firstw))                    
                    if is_supervertex(b[wi]):
                        self.HypoMatch(M,firstw,(firstv, firstw))
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    #print "walk forward adding:", (firstv, firstw), (ttv,ttw)
                    M.AddEdge(ttv,ttw)
                vi = (vi + 1) % len(b)
            #print

        elif M.QEdge(tv2,tw2): # walk backward in blossom
            tmp = range(len(b))
            tmp.reverse()
            for i in tmp:
                wi = (vi - 1) % len(b)
                tv, tw = self.findNextLevelEdge(b[vi], b[wi])
                #print "walk backward blos:", (b[vi], b[wi]), "next=", (tv, tw)

                if i % 2 == 0:
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    try:
                        ttv, ttw = M.Edge(ttv,ttw)
                    except NoSuchEdgeError: # Found two consecutive unmatched edges
                        #print "Hypomatch break ",v
                        break
                    
                    M.DeleteEdge(ttv,ttw)
                else:
                    if self.true_edges.has_key((b[vi], b[wi])):
                        firstv, firstw = self.true_edges[(b[vi], b[wi])][0]
                        #print "Hypo true_edge", (firstv, firstw)
                    elif self.true_edges.has_key((b[wi], b[vi])):
                        firstw, firstv = self.true_edges[(b[wi], b[vi])][0]
                        #print "Hypo true_edge", (firstv, firstw)
                    else:
                        (firstv, firstw) = wmc.G.Edge(b[vi],b[wi])
                        #print "Hypo plain edge", (firstv, firstw)
                   
                    if is_supervertex(b[vi]):
                        self.HypoMatch(M,firstv,(firstv, firstw))                    
                    if is_supervertex(b[wi]):
                        self.HypoMatch(M,firstw,(firstv, firstw))

                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    #print "walk backward adding:", (firstv, firstw), (ttv,ttw)
                    M.AddEdge(ttv,ttw)
                vi = (vi - 1) % len(b)

        #print "Hypo resolving super",v,"e=",entry_edge
        if self.is_supervertex(v):
            e = entry_edge
            while v in e:
                e = self.findNextLevelEdge(e[0],e[1])
                #print "Hypo resolving super",v,"e=",e,e[0] in self.blossom[v], e[1] in self.blossom[v]
            if e[0] in self.blossom[v]:
                self.HypoMatch(M,e[0],(e[0],e[1]))
            else:
                self.HypoMatch(M,e[1],(e[0],e[1]))

            
        #print "HypoMatch done with", v


    def findEdge(self,v,w):
        while self.supervertex.has_key(v):
            v = self.supervertex[v]
        while self.supervertex.has_key(w):
            w = self.supervertex[w]
        return (v,w)

    def findNextLevelEdge(self, v, w):
        """ Find edge one level into the blossom structure. Make sure
            that both v and w are resolved into blossoms if they are both
            super-vertices.
        """    
        e = self.Edge(v,w)
        if not self.true_edges.has_key(e):
            return e
        ##print "findNextLevelEdge", (v,w), self.true_edges[(v,w)]
        if self.is_supervertex(v):
            while v in e:
                e = self.true_edges[e][0]
        elif self.is_supervertex(w):
            while w in e:
                e = self.true_edges[e][0]
        elif self.is_supervertex(v) and self.is_supervertex(w):
            while v in e or w in e:
                e = self.true_edges[e][0]
        return e
            

    def findTrueEdge(self, v, w):
        v, w = self.Edge(v,w)
        if not self.true_edges.has_key((v,w)):
            return (v,w)
        edges = self.true_edges[(v,w)]
        #print "findTrueEdge", (v,w), edges, self.true_edges[self.Edge(v,w)]
        (true_v, true_w) = edges[0]
        if edges[0] != (v,w):
            while self.is_supervertex(true_v) or self.is_supervertex(true_w):
                try:
                    (true_v, true_w) = self.true_edges[(true_v,true_w)][0]
                except:
                    (true_v, true_w) = self.true_edges[(true_w,true_v)][0]
        return (true_v, true_w)
        

def dist(G,v,w):   
    return math.sqrt((G.embedding[v].x - G.embedding[w].x)**2 +
                     (G.embedding[v].y - G.embedding[w].y)**2)


class WeightedMatchingContext:
    """Utility class to hold all relevant data structures and
       methods necessary for computation
    """

    def __init__(self, G, A, u, gato_context):
        self.G = G
        self.GA = A
        for v in G.Vertices():
            u[v] = 0.0

        self.u = u # The dual variables
        self.gato = gato_context # Simply self in calling context
        for u, v in G.Edges():
            self.GA.DeleteEdge(u,v)        
        self.M = AnimatedSubGraph(self.G, self.GA)
        

        # The shrinkable graph collects all the original vertices and all edges
        # and all supervertices and all their edges so that we can selectively
        # show and hide things without having to worry about renumbering as
        # we would for Adding and Deleting vertices to a graph
        H = copy.deepcopy(G) # Temporary
        self.gato.OpenSecondaryGraph(H, stripPath(self.gato.graphFileName) + " (shrinking)")
        self.SGVA = self.gato.GUI.secondaryGraphDisplay
        self.SG = ShrinkableGraph(H, self.SGVA) 
        # ... SGV is what the current instance of SG looks like
        self.SGV = SubGraph(self.SG)
        for v in self.SG.Vertices():
            self.SGV.AddVertex(v)
        self.SGVA.ShowGraph(self.SGV, stripPath(self.gato.graphFileName) + " (shrinking)")

        # YYY Moved from  GallaiEdmondsDecomposition
        self.SGM = AnimatedSubGraph(self.SGV, self.SGVA)
        self.SGM.directed = 0
        
        self.iter = 1

        self.depth = {} # How many levels of shrunken supervertices are contained


    def GallaiEdmondsDecomposition(self, G):
        """ Compute the Gallai-Edmonds decomposition on (non-bipartite) graph G

            Even: the even-labeled vertices G (inner nodes
            Odd:  the odd-labeled vertices of G: a list of either vertices or list of vertices
                  If shrinking has been necessary, a supernode in SG is recursively expanded
                  to a flat list of all vertices which are contained in the blossom of the
                  supervertex
            Unlabeled: unlabeled vertices    
        """
        #print "#==== %d-th iteration =======================" % wmc.iter
        self.iter += 1
        #print "Before Matching in Shrinkable view SGV:", self.SGM.Edges()
        #print "Before Matching in original graph G:", self.M.Edges()

        SGV_order = self.SGV.Order() # Need for termination criterion

        self.sv_added = []

        # Find a maximal cardinality matching
        maximal = False
        while not maximal:
            (e, pred, base, label) = FindSuspectEdge(self.SGV, self.SGM)
            #print (e, pred, base, label)
            if e == None:
                maximal = True
            else:
                if base[e[0]] == base[e[1]]: # Found a blossom
                    #print e,"found blossom", e
                    B = self.SG.BlossomContaining(e, pred)
                    sv = self.SG.ShrinkBlossom(B)
                    self.sv_added.append(sv)
                    
                else:
                    P = FindAugmentingPath(self.SGV, self.SGM, pred, e)
                    #print "GallaiEdmondsDecomposition: Augmenting path found", P
                    self.SG.Augment(self.M, P)
                    

                    
        #print "Matching in Shrinkable view SGV:", self.SGM.Edges()
        #print "Matching in original graph G:", self.M.Edges()

        #print "GED", SGV_order, self.SGV.Order(), self.SGM.Size()
        if 2 * self.SGM.Size() < SGV_order: # Matching is not perfect
            Even = []
            Odd = []
            Unlabeled = []
            for v in self.SGV.Vertices():
                if label.has_key(v):
                    if label[v] % 2 == 0: # v even
                        Even.append(v)
                    else: # v odd
                        Odd.append(v)
                else:
                    Unlabeled.append(v)

            # COLOR A, B, C in shrunken graph
            self.SGVA.SetAllVerticesColor("red", vertices=Even)
            self.SGVA.SetAllVerticesColor(self.SGVA.g.cVertexDefault, vertices=Odd)
            self.SGVA.SetAllVerticesColor("blue", vertices=Unlabeled)

            # replace supervertices by lists of vertices
            sets = [Even, Odd, Unlabeled]
            for s in sets:
                for i, v in enumerate(s):
                    if self.SG.is_supervertex(v):
                        sv, depth = self.SG.expand_supervertex(v, returnDepth = 1)
                        self.depth[sv] = depth
                        s[i] = sv

            return (self.M, Even, Odd, Unlabeled)
        else:
            return (self.M, [], [], [])



    def AddTightEdge(self, v, w):
        """Add a new tight edge"""
        #print "wmc.AddTightEdge", (v, w)
        if not self.G.QEdge(v, w):
            self.GA.AddEdge(v,w)

            # Figure out what the edge (v,w) corresponds to in SG
            # original graph

            # Case 1: Neither v nor w are members of a supervertex
            #     
            if not self.SG.supervertex.has_key(v) and not self.SG.supervertex.has_key(w):
                if not self.SG.QEdge(v,w):
                    self.SG.AddEdge(v,w)
                if not self.SGV.QEdge(v,w):
                    self.SGVA.AddEdge(v,w)               

            oldvv = vv = v
            while self.SG.supervertex.has_key(vv):
                vv = self.SG.supervertex[vv]
                if not self.SG.QEdge(vv,w):
                    self.SG.AddEdge(vv,w)
                if vv in self.SGV.Vertices() and w in self.SGV.Vertices() and \
                       not self.SGV.QEdge(vv,w):
                    self.SGVA.AddEdge(vv,w)
                if self.SG.true_edges.has_key((vv,w)):
                    self.SG.true_edges[(vv,w)].append((oldvv,w))
                else:
                    self.SG.true_edges[(vv,w)] = [(oldvv,w)]
                #print "AddTight: adding true edge", (oldvv,w), "for", (vv,w)
                oldvv = vv

            oldww = ww = w
            while self.SG.supervertex.has_key(ww):
                ww = self.SG.supervertex[ww]
                if not self.SG.QEdge(vv,ww):
                    self.SG.AddEdge(vv,ww)
                if vv in self.SGV.Vertices() and ww in self.SGV.Vertices() and \
                       not self.SGV.QEdge(vv,ww):
                    self.SGVA.AddEdge(vv,ww)
                if self.SG.true_edges.has_key((vv,ww)):
                    self.SG.true_edges[(vv,ww)].append((vv,oldww))
                else:
                    self.SG.true_edges[(vv,ww)] = [(vv,oldww)]                   
                #print "AddTight: adding true edge", (vv,oldww), "for", (vv,ww)
                oldww = ww
        self.M.RaiseEdges()

    def Unshrink(self, sv):
        #print "Unshrink", sv
        svset = sv
        sv = self.SG.supervertex[sv[0]] # NOTE: this uniquely maps from
        # (v_1, ..., v_k) to v with v > G.Order()
        for v in self.SGV.Neighborhood(sv):
            if self.SGM.QEdge(sv,v):
                tmpv, tmpw = self.SGM.Edge(sv,v)
                self.SGM.DeleteEdge(tmpv,tmpw)
                #print "Unshrink SGM.DeleteEdge", (tmpv,tmpw)
            tmpv, tmpw = self.SG.Edge(sv,v)
            self.SGVA.DeleteEdge(tmpv, tmpw)
            
        self.SGVA.DeleteVertex(sv)
        for v in self.SG.expand_supervertex(sv):
            self.SGVA.AddVertex(self.SG.embedding[v].x, self.SG.embedding[v].y, v)

        #print "self.SGV.Vertices = ", self.SGV.Vertices()
        # Find all tight edges and add them
        for v in self.SGV.Vertices():
            for w in self.SGV.Vertices():
                if v < w:

                    #print (v,w), self.SG.QEdge(v,w)

                    if self.SG.is_supervertex(v):
                        tmpv = self.SG.blossom[v]
                    else:
                        tmpv = [v]
                    if self.SG.is_supervertex(w):
                        tmpw = self.SG.blossom[w]
                    else:
                        tmpw = [w]
                    d, t = mindist(tmpv,tmpw)

                    #print "d(%d,%d) = %d" % (v,w,d), dist(G, v, w), u[v], u[w]
                    #print "d(%d,%d) = %d" % (v,w,d)
                    if d < eps_prec:
                        if not self.SG.QEdge(v,w):
                            self.SG.AddEdge(v,w)
                        if not self.SGV.QEdge(v,w):
                            self.SGVA.AddEdge(v,w)

                        (tv,tw) = self.SG.findTrueEdge(v,w)
                        if self.M.QEdge(tv,tw) and not self.SGM.QEdge(v,w):
                            self.SGM.AddEdge(v,w)

        for v in svset:
            del[self.SG.supervertex[v]]


class WeightedMatchingInformer(GraphInformer):

    def __init__(self, G, wmc,u):
        GraphInformer.__init__(self,G)
        self.wmc = wmc
        self.u = u

    def EdgeInfo(self, v, w):
        l = self.G.edgeWeights[0][(v,w)]
        return  "Edge (%d,%d) has length %0.2f" % (v,w,l)

    def VertexInfo(self, v):
        return "Vertex %d, u[%d] = %0.2f" % (v,v,self.u[v])

    def DefaultInfo(self):
        return "Cardinality of matching is %d" % wmc.M.Size()


class ShrunkenInformer(GraphInformer):

    def __init__(self, G, wmc,u):
        GraphInformer.__init__(self,G)
        self.wmc = wmc
        self.u = u

    def VertexInfo(self, v):
        if self.wmc.SG.is_supervertex(v):
            return "Supervertex %d = %s" % (v,self.wmc.SG.expand_supervertex(v))
        else:
            return GraphInformer.VertexInfo(self, v)

    def EdgeInfo(self, v, w):
        tv,tw = self.wmc.SG.findTrueEdge(v,w)
        return "Edge (%d,%d) originally (%d,%d)" % (v,w,tv,tw)
        
    def DefaultInfo(self):
        deficiency = wmc.SGV.Order() - 2*wmc.SGM.Size()
        return "Number of unmatched vertices is %d" % deficiency


            
# =================== Initialisation =============================
u = DualVars(G, A)
wmc = WeightedMatchingContext(G, A, u, self)
Infinity = 999999.99

# =================== Syntactic sugar =============================

GallaiEdmondsDecomposition = lambda graph: wmc.GallaiEdmondsDecomposition(graph)
is_dualvar = lambda v: u.has_key(v)
AddTightEdge = lambda v,w,thewmc = wmc: thewmc.AddTightEdge(v,w)
def Unshrink(sv, thewmc=wmc, theu=u):
    thewmc.Unshrink(sv)
    theu.del_variable(sv)

A.RegisterGraphInformer(WeightedMatchingInformer(G,wmc,u))
wmc.SGVA.RegisterGraphInformer(ShrunkenInformer(G,wmc,u))

def RemoveIncidentUntightEdges(v, A, B, C, thewmc=wmc):
    nbh = thewmc.SGV.Neighborhood(v)
    for i, w in enumerate(nbh):
        if is_supervertex(w):
            nbh.remove(w)
            nbh += thewmc.SG.blossom[w]
    for w in thewmc.SGV.Neighborhood(v):
        if w in C or w in A:
            d = dist_with_duals(G,u,v,w)
            if d > eps_prec:
                vv, ww = thewmc.SGV.Edge(v,w)
                thewmc.SGVA.DeleteEdge(vv,ww)
                thewmc.GA.DeleteEdge(vv,ww)

     

    
