################################################################################
#
#       This is part of CATBox (Combinatorial Algorithm Toolbox)
#       version 1.0 from 2/25/10. You can find more information at
#       http://algorithmics.molgen.mpg.de/CATBox/
#
#	file:   NegativeCircuit.pro
#	author: Torsten Pattberg (pattberg@zpr.uni-koeln.de)
#               Alexander Schliep (alexander@schliep.org)
#
#       Copyright (C) 1998-2010, Winfried Hochstaettler und Alexander Schliep
#	and (c) 2010 Springer Verlag.
#       
#       All rights reserved. Do not distribute. 
#	See http://schliep.org/CATBox for more information.
#




#
#       This file is version $Revision: 1.3 $
#                       from $Date: 2004/03/22 14:08:49 $
#             last change by $Author: wh $.
#
#
################################################################################

#### Options #################################################

breakpoints = [5,9]
interactive = []
graphDisplays = 2
about = """<HTML>
<HEAD>
<TITLE>Minimum Cost Flow - Negative Cycle</TITLE>
</HEAD>
<BODY>
</BODY></HTML>
"""
self.NeededProperties({'EdgeWeights':2, 'VertexWeights':1})

#### Misc ####################################################

from math import *

ready = 0

#### Graph management ########################################

import copy

R  = copy.deepcopy(G)
self.OpenSecondaryGraph(R, stripPath(self.graphFileName) + " (residual network)")
RA = self.GUI.secondaryGraphDisplay
G.CalculateWidthFromWeight(0.7)
A.ShowGraph(G, stripPath(self.graphFileName))

#### Wrapper #################################################

class MyFlowWrapper(FlowWrapper):
    """
        Set flow to zero without changing the excess. Necessary when a new
        edge is added after initalizing the flow
    """

    def ForceZero(self,e):
        self.flow[e] = 0
        return

#### GraphInformer ###########################################

class MyResidualGraphInformer(ResidualGraphInformer):
    """ Also show the costs and some info about the cycle
    """
    def EdgeInfo(self,v,w):
        if not C or (v,w) not in C.Edges():
            return "Edge (%d,%d) - residual capacity: %d - cost: %d" % (v, w, self.res[(v,w)],cost[(v,w)])
        else:
            cyclecost = 0
            for e in C.Edges():
                    cyclecost = cyclecost + cost[e]
            return "Edge (%d,%d) - residual capacity: %d - cost: %d - Cycle cost: %d" % (v, w, self.res[(v,w)],cost[(v,w)],cyclecost)

#### Variables ###############################################

cost = G.edgeWeights[1]
flow = MyFlowWrapper(G,A,R,RA,G.edgeWeights[0],R.edgeWeights[0],G.vertexWeights[0])

pred = {}
dist = {}
mark = {}

NOW  = Queue()
NEXT = Queue()

D    = Stack()

#### Lambdas #################################################

C = AnimatedSubGraph(R,RA)

NewCycle              = lambda r=R, ra=RA: AnimatedSubGraph(r,ra,"red")

EstablishFeasibleFlow = lambda ra=RA,a=A: EstablishFeasibleFlow_(ra,a)
Vertices              = lambda r=R: r.vertices
Order                 = lambda g=G: g.Order()
Edges                 = lambda l, ra=RA: BlinkingContainerWrapper(ra,l.Edges(),"black")

cap                   = lambda e: flow.cap[e]
excess                = lambda v: flow.excess[v]
res                   = lambda e: flow.res[e]
ForwardEdge           = lambda u,v,g=G: g.QEdge(u,v)

#### Internal functions ######################################

def EstablishFeasibleFlow_(RA,A):
    A.AddVertex(3500, 3500)
    A.AddVertex(-20, -20)
    sink   = RA.AddVertex(3500, 3500)
    source = RA.AddVertex(-20, -20)

    flow.excess[source] = gInfinity
    flow.excess[sink]   = -gInfinity

    A.SetAllVerticesColor("grey")
    A.SetVertexAnnotation(source,"source","green")
    A.SetVertexAnnotation(sink,"sink","red")
    A.SetVertexColor(source,"white")
    A.SetVertexColor(sink,"white")
    RA.SetVertexColor(source,"white")
    RA.SetVertexColor(sink,"white")

    maxWeight = max(R.edgeWeights[0].label.values()) #XXX Using G internals
    #print R.edgeWeights[0].label
    #print "MaxWeight", maxWeight
    
    for v in Vertices():

        if (v != sink) and (v != source):
            if excess(v) > 0:
                # XXX Modifying G internals 
                G.edgeWidth[(source,v)]  = 0.7 * (1 + 35 * excess(v) / maxWeight)
                
                A.AddEdge(source, v)
                RA.AddEdge(source, v)

                flow.cap[(source,v)] = excess(v)
                flow.ForceZero((source,v))
                cost[(source,v)] = 0
                cost[(v,source)] = 0
                flow.res[(source,v)] = excess(v)
                flow.res[(v,source)] = 0
                A.SetVertexColor(v,"green")
                A.SetVertexAnnotation(v,excess(v),"green")
            elif excess(v) < 0:
                # XXX Modifying G internals 
                G.edgeWidth[(v,sink)] = 0.7 * (1 - 35 * excess(v) / maxWeight)

                A.AddEdge(v, sink)
                RA.AddEdge(v, sink)
                flow.cap[(v,sink)] = -excess(v)
                flow.ForceZero((v,sink))
                cost[(v,sink)] = 0
                cost[(sink,v)] = 0
                flow.res[(v,sink)] = -excess(v)
                flow.res[(sink,v)] = 0
                A.SetVertexColor(v,"red")
                A.SetVertexAnnotation(v,excess(v),"red")
                D.Push(v)

    Cmax = 0
    for e in R.Edges():
        if (cap(e) > Cmax):
            Cmax = cap(e)

    k = pow(2, floor(log(Cmax)/log(2)))

    while k >= 1:

        while not ready:

            P = FindPath(R,RA,source,sink,k)

            if P == None:
                break
            else:
                delta = MinimalRestCapacity2(P)

                for (u,v) in P.Edges():
                    if ForwardEdge(u,v):
                        flow[(u,v)] = flow[(u,v)] + delta
                    else:
                        flow[(v,u)] = flow[(v,u)] - delta
        k = k / 2

    for v in R.Neighborhood(source):
        if res((source,v)):
            #print "ERROR: No feasible flow found!"
            return False

    A.DeleteVertex(source)
    A.DeleteVertex(sink)
    RA.DeleteVertex(source)
    RA.DeleteVertex(sink)

    return True


def FindPath(R,A,s,t,k):

    P    = SubGraph(R)
    Q    = Queue()

    pred = {}

    for v in R.vertices:
        pred[v] = None

    Q.Append(s)

    while Q.IsNotEmpty():

        v = Q.Top()

        for w in R.Neighborhood(v):

            if pred[w] == None and w != s and res((v,w)) >= k:

                pred[w] = v
                Q.Append(w)

                if w == t:
                    while pred[w] != None:
                        P.AddEdge(pred[w],w)
                        w = pred[w]
                    return P

    return None



def ShowTree(R,RA):

    for v in R.vertices:
        RA.SetVertexAnnotation(v,dist[v],"black")
        try:
            RA.SetEdgeColor(pred[v],v,"green")
        except:
            None

    return None


def MinimalRestCapacity2(Graph):
    min = gInfinity
    for i in Graph.Edges():
        if res(i) < min:
            min = res(i)

    return min


def MinimalRestCapacity(Graph):
    min = gInfinity
    u   = 0
    v   = 0
    for i in Graph.Edges():
        if res(i) < min:
            min = res(i)
            u = i[0]
            v = i[1]

    RA.SetEdgeColor(u,v,"yellow")

    return min

#### Internal initialisation #################################

for e in G.Edges():
    cost[(e[1],e[0])] = -cost[e]

RA.SetAllVerticesColor("grey")

A.RegisterGraphInformer(FlowGraphInformer(A,flow))
RA.RegisterGraphInformer(MyResidualGraphInformer(R,flow))

#### functions ############################

def FindNegativeCycle():

    while D.IsNotEmpty():

        v = D.Pop()
        C = SearchCycleFrom(v)
        if not C == None:
            D.Push(v)
            return C

    return None


def SearchCycleFrom(v):

    NC = NewCycle()

    NOW.Clear()
    NEXT.Clear()

    for u in Vertices():
        dist[u] = gInfinity
        pred[u] = None
        mark[u] = False

    NOW.Append(v)
    dist[v] = 0
    counter = 0

    while not ready:

        if NOW.IsEmpty():

            if NEXT.IsEmpty():
                return None

            else:
                counter = counter + 1

                if counter > Order():
                    break

                while NEXT.IsNotEmpty():
                    u = NEXT.Top()
                    NOW.Append(u)

        else:
            u = NOW.Top()

            for w in R.Neighborhood(u):

                if (dist[w] > dist[u] + cost[(u,w)]):
                    dist[w] = dist[u] + cost[(u,w)]
                    pred[w] = u

                    if not (NOW.Contains(w) or NEXT.Contains(w)):
                        NEXT.Append(w)

    while not mark[w]:
        mark[w] = True
        w = pred[w]

    first_vertex = w

    while not (pred[w] == first_vertex):
        NC.AddEdge(pred[w],w)
        w = pred[w]
    NC.AddEdge(pred[w],w)

    return NC
