################################################################################
#
#       This is part of CATBox (Combinatorial Algorithm Toolbox)
#       version 1.0 from 2/25/10. You can find more information at
#       http://www.zpr.uni-koeln.de/CATBox
#
#	file:   WeightedMatching.pro
#	authors: Winfried Hochstaettler and Alexander Schliep
#
#       Copyright (C) 1998-2010, Winfried Hochstaettler und Alexander Schliep
#	and (c) 2010 Springer Verlag.
#       
#       All rights reserved. Do not distribute. 
#	See http://schliep.org/CATBox for more information.
#




#
#
################################################################################

#
#
#
#
#### Options #################################################

breakpoints = [84]
interactive = []
graphDisplays = 2
about = """<HTML>
<HEAD>
<TITLE>Matching</TITLE>
</HEAD>
<BODY>

This algorithm finds a maximum cardinality matching.

</BODY></HTML>
"""

import copy
import math
import string


#### Utility functions #########################################
def is_supervertex(v):
    return isinstance(v,tuple) or (isinstance(v,int) and v > G.Order())

def UnmatchedVertices(G, M):
    result = set(G.Vertices())
    result.difference_update(set(M.Vertices()))
    return result

def matched(M,v):
    return v in M.Vertices()

global LastOffQueue
LastOffQueue = None
global pathID
pathID = None

def FindAugmentingPath(G, M, pred, e):
    """Find an augmenting path containing e in the ShrinkableGraph G
       using the predecessor array pred
    """
    P = [e]
    v = e[0]
    while pred[v] != v:
        P.insert(0,(pred[v],v))
        v = pred[v]

    v = e[1]
    while pred[v] != v:
        P.append((v,pred[v]))
        v = pred[v]

    vertexList = [x for x,y in P] + [P[-1][1]]
    # Highlight augmenting path
    global pathID
    if pathID != None:
        cmc.SGVA.HidePath(pathID)
    pathID = cmc.SGVA.HighlightPath(vertexList, 'yellow')
    return P



class ShrinkableGraph(Graph):
    """ A graph in which we can identify two or more vertices.  That
        is, a blossom v_1, ..., v_k will be replaced by a supervertex
        v where all edges (v, w) are present for which an edge
        (v_i, w) exists.
    """
    def __init__(self, G, animator = None):
        """animator: Needed"""
        Graph.__init__(self)
        self.directed = 0
        self.supervertex = {}
        self.blossom = {}
        self.g_order = G.Order()
        self.true_edges = {}

        if animator != None:
            self.labeling = copy.deepcopy(G.labeling)
            self.embedding = copy.deepcopy(G.embedding)
            self.edgeWeights = copy.deepcopy(G.edgeWeights)
            self.animator = animator
            self.graph_edit = animator
        else:
            self.graph_edit = self

        for v in G.Vertices():
            Graph.AddVertex(self)
        for e in G.Edges():
            Graph.AddEdge(self,e[0],e[1])
            
    def addVertex(self, B=None):
        if B is None or self.animator is None:
            return Graph.AddVertex(self)
        else:
            # Compute position as center of blossom
            x = 0.0
            y = 0.0
            for v in B:
                x += self.embedding[v].x
                y += self.embedding[v].y
            v = Graph.AddVertex(self)
            return self.animator.AddVertex(x / len(B), y / len(B), v)

    def ShrinkBlossom(self, B):
        """For B the vertex set of an odd cycle in 'self', remove the
           vertices, add a new supervertex representing the cycle and
           return the supervertex
        """
        sv = self.addVertex(B)
        self.blossom[sv] = B

        for v in B:
            for w in cmc.SGV.Neighborhood(v):
                if not w in B and not self.QEdge(sv,w):
                    cmc.SG.AddEdge(sv,w)
                    self.graph_edit.AddEdge(sv,w)
                    if self.true_edges.has_key((sv,w)):
                        self.true_edges[(sv,w)].append((v,w))
                    else:
                        self.true_edges[(sv,w)] = [(v,w)]

            self.supervertex[v] = sv

        # Delete matching edges within blossom
        for v, w in cmc.SGM.Edges():
            if v in B and w in B:
                cmc.SGM.DeleteEdge(v,w)

            # Add matching edge to outside of blossom
            if (v in B and (not w in B)) or ((not v in B) and w in B):
                if v in B:
                    tmpv, tmpw = cmc.SGV.Edge(sv,w)
                else:
                    tmpv, tmpw = cmc.SGV.Edge(v,sv) 
                cmc.SGM.AddEdge(tmpv, tmpw)
                if v in B:
                    cmc.SGM.DeleteVertex(v)
                else:
                    cmc.SGM.DeleteVertex(w)                    

        for v in B:
            self.graph_edit.DeleteVertex(v)

        # Animation extras
        global pathID, lastOffQueue, blossomHighlightID
        if pathID != None:
            self.animator.HidePath(pathID)            
        if lastOffQueue is not None and not lastOffQueue in B:
            self.animator.SetVertexFrameWidth(lastOffQueue,0)
        lastOffQueue = None
        A.HideHighlight(blossomHighlightID)
        # No backlink to wmc use global
        cmc.sv_added.append(sv)
        return sv


    def expand_supervertex(self, v, returnDepth = None):
        """ Recursively expand a supervertex v to a flat
            list of vertices

            XXX Retarded implementation.
        """
        terminate = 0
        depth = 0
        result = self.blossom[v][:]
        while not terminate:
            terminate = 1            
            for i, w in enumerate(result):
                if self.is_supervertex(w):
                    result[i:i+1] = self.blossom[w]
                    terminate = 0
            depth += 1
        if returnDepth is None:
            return tuple(result)
        else:
            return (tuple(result), depth)

    def is_supervertex(self,v):
        return v > self.g_order

    def BlossomContaining(self, (v,w), pred):
        """Return the vertex set of the blossom containing edge e give the
           predecessor array pred
        """
        blossom = []
        
        p0 = [v]
        p1 = [w]

        for p in [p0, p1]:
            while pred[p[-1]] != p[-1]:
                p.append(pred[p[-1]])

        for i, v in enumerate(p0):
            if v in p1:
                break
        j = p1.index(v)

        p1slice = p1[0:j] # Need the first common just once
        p1slice.reverse()
        blossom = p0[0:i+1] + p1slice

        expanded_blossom = [v for v in blossom if not self.is_supervertex(v)]
        for v in [v for v in blossom if self.is_supervertex(v)]:
            expanded_blossom.extend(list(self.expand_supervertex(v)))
        global blossomHighlightID
        # XXX UGLY SG does not know about the non-shrunken graphs animator
        blossomHighlightID = A.HighlightVertices(expanded_blossom, 'yellow')
        return blossom

    def Augment(self, M, P):
        global pathID
        # Unmatch first
        i = 0
        for (v,w) in P:
            if i % 2:
                (tv, tw) = self.findTrueEdge(v,w)
                M.DeleteEdge(tv,tw)
                # Delete(v,w) in matching of SGV (SGM defined in Gallai-Edmonds: expose?)
                v,w = cmc.SGM.Edge(v,w)
                cmc.SGM.DeleteEdge(v,w)
            i += 1
        
        # match second
        i = 0
        for (v,w) in P:
            if i % 2 == 0:
                v,w = cmc.SG.Edge(v,w)
                tv, tw = self.findNextLevelEdge(v,w)
                ttv, ttw = self.findTrueEdge(v,w)
                if matched(M,ttv):
                    self.HypoMatch(M,ttv,(ttv,ttw))
                if matched(M,ttw):
                    self.HypoMatch(M,ttw,(ttv,ttw))

                ttv, ttw = self.findTrueEdge(v,w)
                if not M.QEdge(ttv,ttw):
                    M.AddEdge(ttv,ttw)
                cmc.SGM.AddEdge(v,w)
                #Add (v,w) to SGM

                # Expand blossoms along augmenting path
                # WH special add faked true edges to SG ...
                # Out of order expansion of blossoms might result
                # in edges without true_edges. Add them 
                if self.is_supervertex(v) and self.is_supervertex(w):
                    tv, tw = self.true_edges[(v,w)][0]
                    if not self.QEdge(tv,w):
                        self.AddEdge(tv,w)
                    if not self.QEdge(v,tw):
                        self.AddEdge(v,tw)
                   
            i += 1
        for v in [v for (v,w) in P] + [P[-1][1]]:
            if self.is_supervertex(v) and v in cmc.sv_added:
                if pathID != None:
                    cmc.SGVA.HidePath(pathID)
                global LastOffQueue
                if LastOffQueue != None:
                    cmc.SGVA.SetVertexFrameWidth(LastOffQueue,0)
                    LastOffQueue = None
                cmc.Unshrink(self.blossom[v])
                for tmpv in self.blossom[v]:
                    del(self.supervertex[tmpv])
                del(self.blossom[v])

        self.animator.HidePath(pathID)
        pathID = None


    def HypoMatch(self, M, v, entry_edge):
        """Hypomatch inside a blossom given the entry vertex v
           which should be part of a blossom"""
        if not self.supervertex.has_key(v):
            return
        sv = self.supervertex[v]
        b = self.blossom[sv]
        vi = b.index(v)

        (tv1,tw1) = self.findTrueEdge(v, b[(vi + 1) % len(b)])
        (tv2,tw2) = self.findTrueEdge(v, b[(vi - 1) % len(b)])

        if M.QEdge(tv1,tw1): # walk forward in blossom
            for i in xrange(len(b)):
                wi = (vi + 1) % len(b)
                tv, tw = self.findNextLevelEdge(b[vi], b[wi])
                if i % 2 == 0:
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    try:
                        ttv, ttw = M.Edge(ttv,ttw)
                    except NoSuchEdgeError: # Found two consecutive unmatched edges
                        break                    
                    M.DeleteEdge(ttv,ttw)
                else:
                    if self.true_edges.has_key((b[vi], b[wi])):
                        firstv, firstw = self.true_edges[(b[vi], b[wi])][0]
                    elif self.true_edges.has_key((b[wi], b[vi])):
                        firstw, firstv = self.true_edges[(b[wi], b[vi])][0]
                    else:
                        (firstv, firstw) = cmc.G.Edge(b[vi],b[wi])

                    if is_supervertex(b[vi]):
                        self.HypoMatch(M,firstv,(firstv, firstw))                    
                    if is_supervertex(b[wi]):
                        self.HypoMatch(M,firstw,(firstv, firstw))
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    M.AddEdge(ttv,ttw)
                vi = (vi + 1) % len(b)

        elif M.QEdge(tv2,tw2): # walk backward in blossom
            tmp = range(len(b))
            tmp.reverse()
            for i in tmp:
                wi = (vi - 1) % len(b)
                tv, tw = self.findNextLevelEdge(b[vi], b[wi])
                if i % 2 == 0:
                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    try:
                        ttv, ttw = M.Edge(ttv,ttw)
                    except NoSuchEdgeError: # Found two consecutive unmatched edges
                        break
                    M.DeleteEdge(ttv,ttw)
                else:
                    if self.true_edges.has_key((b[vi], b[wi])):
                        firstv, firstw = self.true_edges[(b[vi], b[wi])][0]
                    elif self.true_edges.has_key((b[wi], b[vi])):
                        firstw, firstv = self.true_edges[(b[wi], b[vi])][0]
                    else:
                        (firstv, firstw) = cmc.G.Edge(b[vi],b[wi])
                   
                    if is_supervertex(b[vi]):
                        self.HypoMatch(M,firstv,(firstv, firstw))                    
                    if is_supervertex(b[wi]):
                        self.HypoMatch(M,firstw,(firstv, firstw))

                    ttv, ttw = self.findTrueEdge(b[vi], b[wi])
                    M.AddEdge(ttv,ttw)
                vi = (vi - 1) % len(b)

        if self.is_supervertex(v):
            e = entry_edge
            while v in e:
                e = self.findNextLevelEdge(e[0],e[1])
            if e[0] in self.blossom[v]:
                self.HypoMatch(M,e[0],(e[0],e[1]))
            else:
                self.HypoMatch(M,e[1],(e[0],e[1]))


    def findEdge(self,v,w):
        while self.supervertex.has_key(v):
            v = self.supervertex[v]
        while self.supervertex.has_key(w):
            w = self.supervertex[w]
        return (v,w)

    def findNextLevelEdge(self, v, w):
        """ Find edge one level into the blossom structure. Make sure
            that both v and w are resolved into blossoms if they are both
            super-vertices.
        """    
        e = self.Edge(v,w)
        if not self.true_edges.has_key(e):
            return e
        if self.is_supervertex(v):
            while v in e:
                e = self.true_edges[e][0]
        elif self.is_supervertex(w):
            while w in e:
                e = self.true_edges[e][0]
        elif self.is_supervertex(v) and self.is_supervertex(w):
            while v in e or w in e:
                e = self.true_edges[e][0]
        return e
            

    def findTrueEdge(self, v, w):
        v, w = self.Edge(v,w)
        if not self.true_edges.has_key((v,w)):
            return (v,w)
        edges = self.true_edges[(v,w)]
        (true_v, true_w) = edges[0]
        if edges[0] != (v,w):
            while self.is_supervertex(true_v) or self.is_supervertex(true_w):
                try:
                    (true_v, true_w) = self.true_edges[(true_v,true_w)][0]
                except:
                    (true_v, true_w) = self.true_edges[(true_w,true_v)][0]
        return (true_v, true_w)
        

class AnimatedSubGraphEdgeOnly(AnimatedSubGraph):
    def AddVertex(self,v):
        try:
            SubGraph.AddVertex(self,v)
        except NoSuchVertexError:
            return


class CardinalityMatchingContext:
#class WeightedMatchingContext:
    """Utility class to hold all relevant data structures and
       methods necessary for computation
    """

    def __init__(self, G, A, gato_context):
        self.G = G
        self.GA = A

        self.gato = gato_context # Simply self in calling context
        self.M = AnimatedSubGraph(self.G, self.GA)
        
        # The shrinkable graph collects all the original vertices and all edges
        # and all supervertices and all their edges so that we can selectively
        # show and hide things without having to worry about renumbering as
        # we would for Adding and Deleting vertices to a graph
        H = copy.deepcopy(G) # Temporary
        self.gato.OpenSecondaryGraph(H, stripPath(self.gato.graphFileName) + " (shrinking)")
        self.SGVA = self.gato.GUI.secondaryGraphDisplay
        self.SG = ShrinkableGraph(H, self.SGVA) 
        # ... SGV is what the current instance of SG looks like
        self.SGV = SubGraph(self.SG)
        for v in self.SG.Vertices():
            self.SGV.AddVertex(v)
        for (u,v) in self.SG.Edges():
            self.SGV.AddEdge(u,v)
        self.SGVA.ShowGraph(self.SGV, stripPath(self.gato.graphFileName) + " (shrinking)")

        # YYY Moved from  GallaiEdmondsDecomposition
        self.SGM = AnimatedSubGraphEdgeOnly(self.SGV, self.SGVA)
        self.SGM.directed = 0
        
        self.iter = 1

        self.depth = {} # How many levels of shrunken supervertices are contained
        self.sv_added = []


    def GallaiEdmondsDecomposition(self, G):
        """ Compute the Gallai-Edmonds decomposition on (non-bipartite) graph G

            Even: the even-labeled vertices G (inner nodes
            Odd:  the odd-labeled vertices of G: a list of either vertices or list of vertices
                  If shrinking has been necessary, a supernode in SG is recursively expanded
                  to a flat list of all vertices which are contained in the blossom of the
                  supervertex
            Unlabeled: unlabeled vertices    
        """
        self.iter += 1
        SGV_order = self.SGV.Order() # Need for termination criterion
        self.sv_added = []

        # Find a maximal cardinality matching
        maximal = False
        while not maximal:
            (e, pred, base, label) = FindSuspectEdge(self.SGV, self.SGM)
            if e == None:
                maximal = True
            else:
                if base[e[0]] == base[e[1]]: # Found a blossom
                    B = self.SG.BlossomContaining(e, pred)
                    sv = self.SG.ShrinkBlossom(B)
                    self.sv_added.append(sv)
                    
                else:
                    P = FindAugmentingPath(self.SGV, self.SGM, pred, e)
                    self.SG.Augment(self.M, P)
                    
        if 2 * self.SGM.Size() < SGV_order: # Matching is not perfect
            Even = []
            Odd = []
            Unlabeled = []
            for v in self.SGV.Vertices():
                if label.has_key(v):
                    if label[v] % 2 == 0: # v even
                        Even.append(v)
                    else: # v odd
                        Odd.append(v)
                else:
                    Unlabeled.append(v)

            # COLOR A, B, C in shrunken graph
            self.SGVA.SetAllVerticesColor("red", vertices=Even)
            self.SGVA.SetAllVerticesColor(self.SGVA.cVertexDefault, vertices=Odd)
            self.SGVA.SetAllVerticesColor("blue", vertices=Unlabeled)

            # replace supervertices by lists of vertices
            sets = [Even, Odd, Unlabeled]
            for s in sets:
                for i, v in enumerate(s):
                    if self.SG.is_supervertex(v):
                        sv, depth = self.SG.expand_supervertex(v, returnDepth = 1)
                        self.depth[sv] = depth
                        s[i] = sv

            return (self.M, Even, Odd, Unlabeled)
        else:
            return (self.M, [], [], [])
            

    def Unshrink(self, sv):
        sv = self.SG.supervertex[sv[0]] # NOTE: this uniquely maps from
        for v in self.SGV.Neighborhood(sv):
            if self.SGM.QEdge(sv,v):
                tmpv, tmpw = self.SGM.Edge(sv,v)
                self.SGM.DeleteEdge(tmpv,tmpw)
            tmpv, tmpw = self.SG.Edge(sv,v)
            self.SGVA.DeleteEdge(tmpv, tmpw)
        self.SGVA.DeleteVertex(sv)
        for v in self.SG.expand_supervertex(sv):
            self.SGVA.AddVertex(self.SG.embedding[v].x, self.SG.embedding[v].y, v)

        for (u,v) in self.SG.Edges():
            if u in self.SGV.Vertices() and v in self.SGV.Vertices():
                if not self.SGV.QEdge(u,v):
                    self.SGVA.AddEdge(u,v)
        for (u,v) in self.M.Edges():
            if u in self.SGV.Vertices() and v in self.SGV.Vertices() and not self.SGM.QEdge(u,v):
                self.SGM.AddEdge(u,v)
       

class CardinalityMatchingInformer(GraphInformer):

    def __init__(self, G, wmc,u):
        GraphInformer.__init__(self,G)
        self.wmc = wmc
        self.u = u

    def EdgeInfo(self, v, w):
        l = self.G.edgeWeights[0][(v,w)]
        return  "Edge (%d,%d) has length %0.2f" % (v,w,l)

    def VertexInfo(self, v):
        return "Vertex %d, u[%d] = %0.2f" % (v,v,self.u[v])

    def DefaultInfo(self):
        return "Cardinality of matching is %d" % cmc.M.Size()


class ShrunkenInformer(GraphInformer):

    def __init__(self, G, wmc,u):
        GraphInformer.__init__(self,G)
        self.wmc = wmc
        self.u = u

    def VertexInfo(self, v):
        if self.cmc.SG.is_supervertex(v):
            return "Supervertex %d = %s" % (v,self.cmc.SG.expand_supervertex(v))
        else:
            return GraphInformer.VertexInfo(self, v)

    def EdgeInfo(self, v, w):
        tv,tw = self.cmc.SG.findTrueEdge(v,w)
        return "Edge (%d,%d) originally (%d,%d)" % (v,w,tv,tw)
        
    def DefaultInfo(self):
        deficiency = cmc.SGV.Order() - 2*cmc.SGM.Size()
        return "Number of unmatched vertices is %d" % deficiency


def ShowGallaiEdmondsDecomposition(label):
    global pathID
    if pathID != None:
        cmc.SGVA.HidePath(pathID)                
    
    for v in label.keys():
        if label[v] == 0:
            A.SetVertexColor(v,'red')
        else:
            if is_supervertex(v):
                B = cmc.SG.expand_supervertex(v)
                for w in B:
                    A.SetVertexColor(w,'orange')               
            else:
                A.SetVertexColor(v,'orange')


class SearchTreePredecessorLabeling(VertexLabeling):
    def __init__(self, theAnimator):
        VertexLabeling.__init__(self)
        self.A = theAnimator
        
    def __setitem__(self, v, pred):
        if pred == v:
            # Initializing
            c = self.A.g.Colors[v % len(self.A.g.Colors)]
        else:            
            c = self.A.GetVertexColor(pred)
        VertexLabeling.__setitem__(self, v, pred)
        self.A.SetVertexColor(v,c)
        if pred != v:
            (pred, v) = self.A.G.Edge(pred,v)
            self.A.SetEdgeColor(pred, v, c)
    


class TrackingVertexQueue(Queue):
    
    def __init__(self, theAnimator):
        Queue.__init__(self)
        self.Animator = theAnimator
        global lastOffQueue
        if lastOffQueue is not None:
            try:
                self.Animator.SetVertexFrameWidth(lastOffQueue,
                                                  self.Animator.g.VertexFrameWidth)
            except KeyError:
                pass
        lastOffQueue = None
                
    def Top(self):
        global lastOffQueue
        v = Queue.Top(self)
        if lastOffQueue is not None:
            self.Animator.SetVertexFrameWidth(lastOffQueue,
                                              self.Animator.g.VertexFrameWidth)
        self.Animator.SetVertexFrameWidth(v, 6)
        lastOffQueue = v 
        return v

    def HideTracked(self):
        global lastOffQueue
        if lastOffQueue is not None:
            self.Animator.SetVertexFrameWidth(lastOffQueue,
                                              self.Animator.g.VertexFrameWidth)        

        lastOffQueue = None 
       
    def Clear(self):
        global lastOffQueue
        Queue.Clear(self) 
        if lastOffQueue is not None:
            self.Animator.SetVertexFrameWidth(lsatOffQueue,
                                              self.Animator.g.VertexFrameWidth)
            lastOffQueue = None




            
# =================== Initialisation =============================
cmc = CardinalityMatchingContext(G, A, self)
lastOffQueue = None

SG = cmc.SGV
SGM = cmc.SGM
SGA = cmc.SGVA

BlossomContaining = lambda e, pred, G=cmc.SG: G.BlossomContaining(e, pred)
ShrinkBlossom = lambda B, G=cmc.SG: G.ShrinkBlossom(B)
Augment = lambda dummy, P, M=cmc.M, G=cmc.SG: G.Augment(M, P)

def Clear(a):
    a.SetAllVerticesColor('grey')
    a.SetAllEdgesColor('grey')
    a.ClearVertexAnnotations()
   
cmc.SGVA.Clear = lambda a=cmc.SGVA: Clear(a)

# =================== Syntactic sugar =============================

GallaiEdmondsDecomposition = lambda graph: cmc.GallaiEdmondsDecomposition(graph)


A.SetAllVerticesColor('grey')
cmc.SGVA.SetAllVerticesColor('grey')

#A.RegisterGraphInformer(WeightedMatchingInformer(G,wmc,u))
#cmc.SGVA.RegisterGraphInformer(ShrunkenInformer(G,wmc,u))

    
