{$M 65520,0,300000}

{$N+,E+}

Program TS_Image_Processor;

{             Authors:  TS Kelso }

{    Original Version:  1991 Jul 11 }

{    Current Revision:  1993 Feb 03 }

{ Program Information:  }



Uses

  CRT,DOS,Graph,Misc,TP030;                                       {*TS920416*}



const

  VGA320x200   = 0; (*     Standard VGA  *)

  SVGA640x400  = 1; (*  640x400x256 SVGA *)

  SVGA640x480  = 2; (*  640x480x256 SVGA *)

  SVGA800x600  = 3; (*  800x600x256 SVGA *)

  SVGA1024x768 = 4; (* 1024x768x256 SVGA *)

  numdim       = 2;                                               {*TS920416*}



type

  menu = array [1..26] of string[15];

  submenu = array [1..26,0..9] of string[12];

  line640 = array [0..639] of byte;

  line800 = array [0..799] of byte;

  line1024 = array [0..1023] of byte;



var

  AutoDetect  : pointer;

  GraphDriver : integer;  { The Graphics device driver }

  GraphMode   : integer;  { The Graphics mode value }



const

  samples = 799;  {Samples per line minus one}

  cx   : integer = 0;

  cy   : integer = 0;

  step : byte = 1;

  woffset = 5;

  wsize = 20;

  menu_active : boolean = true;

  submenu_active : boolean = true;

  windows = 3;

  overlays = 3;

  filetypes = 3;  {Palettes, Images, Overlays}

  workspaces = 4;

  help_menu : menu

    = ('Add Images     ','Blank Window   ','Copy Window    ','               ',

       'Examine Image  ','Filters        ','Group Images   ','Histograms     ',

       'Invert Image   ','Contour Plot   ','Classification ','Load Image     ',

       '               ','               ','Overlay        ','Palette        ',

       'Quit Program   ','Redisplay Image','Show All       ','               ',

       '               ','               ','Select Window  ','Cut Section    ',

       '               ','Zoom           ');

  filter_menu : menu

    = ('Average        ','Blur           ','Crispen        ','Line Detection ',

       '               ','FFT            ','               ','               ',

       '               ','               ','               ','Laplacian      ',

       'Median         ','Noise Removal  ','               ','               ',

       '               ','               ','Sobel          ','               ',

       '               ','               ','               ','Sharpen        ',

       '               ','Special        ');

  help_submenu : submenu

{A} = (('Average   ','AND       ','OR        ','XOR       ','Maximum   ',

        'Minimum   ','Difference','Subtract  ','',''),

{B}    ('','','','','','','','','',''),

{C}    ('','','','','','','','','',''),

{D}    ('','','','','','','','','',''),

{E}    ('Point     ','Line      ','','','','','','','',''),

{F}    ('','','','','','','','','',''),

{G}    ('','','','','','','','','',''),

{H}    ('Display   ','LinStretch','LogStretch','Equalize  ','','','','','',''),

{I}    ('','','','','','','','','',''),

{J}    ('Fixed 1   ','Fixed 2   ','Fixed 3   ','Fixed 4   ','Fixed 5   ',

        'Variable 1','Variable 2','Variable 3','Variable 4','Variable 5'),

{K}    ('Histogram ','1D        ','2D        ','','','','','','',''),

{L}    ('','','','','','','','','',''),

{M}    ('','','','','','','','','',''),

{N}    ('','','','','','','','','',''),

{O}    ('Average   ','AND       ','OR        ','XOR       ','Maximum   ',

        'Minimum   ','Difference','Subtract  ','Dis Last  ','Dis All   '),

{P}    ('Select    ','Customize ','','','','','','','',''),

{Q}    ('','','','','','','','','',''),

{R}    ('','','','','','','','','',''),

{S}    ('','','','','','','','','',''),

{T}    ('','','','','','','','','',''),

{U}    ('','','','','','','','','',''),

{V}    ('','','','','','','','','',''),

{W}    ('','','','','','','','','',''),

{X}    (' 10 x  10 ',' 20 x  20 ',' 50 x  50 ','100 x 100 ','200 x 200 ','250 x 250 ','','','',''),

{Y}    ('','','','','','','','','',''),

{Z}    ('Blow Up   ','N Neighbor','Bilinear  ','Cubic     ','','','','','',''));

  filter_submenu : submenu

{A} = (('3x3       ','5x5       ','','','','','','','',''),

{B}    ('','','','','','','','','',''),

{C}    ('High      ','Medium    ','Low       ','','','','','','',''),

{D}    ('Horizontal','Vertical  ','Diagonal 1','Diagonal 2','','','','','',''),

{E}    ('','','','','','','','','',''),

{F}    ('','','','','','','','','',''),

{G}    ('','','','','','','','','',''),

{H}    ('','','','','','','','','',''),

{I}    ('','','','','','','','','',''),

{J}    ('','','','','','','','','',''),

{K}    ('','','','','','','','','',''),

{L}    ('','','','','','','','','',''),

{M}    ('3x3       ','5x5       ','','','','','','','',''),

{N}    ('Primary   ','Secondary ','Median 3x3','Median 5x5','','','','','',''),

{O}    ('','','','','','','','','',''),

{P}    ('','','','','','','','','',''),

{Q}    ('','','','','','','','','',''),

{R}    ('','','','','','','','','',''),

{S}    ('Gx (1)    ','Gx (2)    ','Gy (1)    ','Gy (2)    ','Gx        ','Gy        ','Gx + Gy   ','','',''),

{T}    ('','','','','','','','','',''),

{U}    ('','','','','','','','','',''),

{V}    ('','','','','','','','','',''),

{W}    ('','','','','','','','','',''),

{X}    ('High      ','Medium    ','Low       ','','','','','','',''),

{Y}    ('','','','','','','','','',''),

{Z}    ('Avg/Crisp ','Bas Relief','','','','','','','',''));



type

  filenames = string[30];

  AllKeys   = set of #0..#127;

  Proc      = Procedure(x,y : integer);

  data = record

      pixel : array [0..samples] of byte;

      end; {record}

  hrecord = record

    data : array [0..255] of word;

    end; {record}



var

  nobeep                 : boolean;

  menuchoice             : char;

  image_drive,work_drive : string[2];

  hmin,hmax              : byte;

  worknumber             : integer;

  overlay_method         : array [1..windows,1..overlays] of char;

  histogram              : array [0..255] of longint;

  overlay_linked         : array [1..windows] of byte;

  file_linked,

  invert                 : array [1..windows] of boolean;

  ex,ey,

  xx,xy,

  fsize,xoff,yoff        : array [1..windows] of integer;

  osize                  : array [1..windows,1..overlays] of integer;

  first                  : array [1..filetypes] of boolean;

  names,fchoice          : array [1..filetypes] of integer;

  lh,lw                  : word;

  mselect,

  active,

  max_x,max_y,start      : integer;

  imgfilename            : array [1..windows] of string;

  ovlfilename            : array [1..windows,1..overlays] of string;

  work                   : array [1..workspaces] of pointer;

  image                  : pointer;

  display_menu           : menu;

  display_submenu        : submenu;

  dirinfo                : SearchRec;

  regs                   : Registers;

  ovlfile                : array [1..windows,1..overlays] of file of data;

  imgfile                : array [1..windows] of file of data;

  namelist               : array [1..filetypes,1..256] of string[15];

  image_directory,

  work_directory         : string;

  defaultname            : array [1..filetypes] of string;

  palfile                : text;

  palette,

  old_palette            : array [0..255,1..3] of byte;

  color_count            : hrecord;

  hfile                  : file of hrecord;



{$i at.trm}

{$i yes.sub}



{*** Video Routines **********************************************************}



{$F+}

Function SVGADetect: integer;

  begin

  SVGADetect := 0; {Place appropriate video mode here: 0,1,2 or 3}

  end; {Function Detect}

{$F-}



Procedure InitializeGraphics;

  var

    ErrorCode : integer;

  begin

  AutoDetect := @SVGADetect;

  GraphDriver := InstallUserDriver('SVGA256',AutoDetect);

  GraphDriver := DETECT;

  InitGraph(GraphDriver,GraphMode,'');

  ErrorCode := GraphResult;

  if ErrorCode <> grOk then

  begin

    Writeln('Graphics error: ', GraphErrorMsg(ErrorCode));

    Halt(1);

  end;

  ClearDevice; {* Required with SVGA256.BGI *}

end; { Procedure InitializeGraphics }



Procedure Initialize_Display;

  begin

  InitializeGraphics;

  SetGraphMode(mselect);

  max_x := GetMaxX;

  max_y := GetMaxY;

  cx := max_x div 2;

  cy := max_y div 2;

  start := (cx - 256) div 2;

  end; {Procedure Initialize_Display}



Procedure Save_Palette;

  begin

  regs.es := Seg(old_palette);

  regs.dx := Ofs(old_palette);

  regs.ah := $10;

  regs.al := $17;

  regs.bx := $0000;

  regs.cx := $0100;

  Intr($10,regs);

  end; {Procedure Save_Palette}



Procedure Set_Palette;

  begin

  Save_Palette;

  regs.es := Seg(palette);

  regs.dx := Ofs(palette);

{ Set Palette colors }

  regs.ah := $10;

  regs.al := $12;

  regs.bx := $0000;

  regs.cx := $0100;

  Intr($10,regs);

  end; {Procedure Set_Palette}



Procedure Change_Palette;

  begin

  end; {Procedure Change_Palette}



Procedure RestoreVideoMode;

  begin

  RestoreCRTmode;

  end; {Procedure RestoreVideoMode}



Procedure Restore_Palette;

  begin

  regs.es := Seg(old_palette);

  regs.dx := Ofs(old_palette);

  regs.ah := $10;

  regs.al := $12;

  regs.bx := $0000;

  regs.cx := $0100;

  Intr($10,regs);

  end; {Procedure Restore_Palette}



{*** Support Routines ********************************************************}



Procedure Beep(pitch : integer);

  var

    i : byte;

  begin

  for i := 1 to 3 do

    begin

    Sound(pitch);

    Delay(50);

    NoSound;

    Delay(50);

    end;

  end; {Procedure Beep}



Procedure Reverse_Video;

  begin

  TextColor(black);

  TextBackground(lightgray);

  end; {Procedure Reverse_Video}



Procedure Normal_Video;

  begin

  TextColor(lightgray);

  TextBackground(black);

  end; {Procedure Normal_Video}



Function Sign(arg : real) : real;

  begin

  if arg > 0 then

    Sign := 1.0

  else if arg < 0 then

    Sign := -1.0

  else

    Sign := 0.0;

  end; {Function Sign}



Function BMin(arg1,arg2 : byte) : byte;

  begin

  if arg1 > arg2 then

    BMin := arg2

  else

    BMin := arg1;

  end; {Function BMin}



Function BMax(arg1,arg2 : byte) : byte;

  begin

  if arg2 > arg1 then

    BMax := arg2

  else

    BMax := arg1;

  end; {Function BMax}



Function IMin(arg1,arg2 : integer) : integer;

  begin

  if arg1 > arg2 then

    IMin := arg2

  else

    IMin := arg1;

  end; {Function IMin}



Function IMax(arg1,arg2 : integer) : integer;

  begin

  if arg2 > arg1 then

    IMax := arg2

  else

    IMax := arg1;

  end; {Function IMax}



{*** Initialization Routines *************************************************}



Procedure Cursor_On;

  var

    regs : registers;

  begin

  with regs do

    begin

    ah := $01;

    ch := 0;

    cl := 7;

    end; {with}

  Intr($10,regs);

  end; {Procedure Cursor_On}



Procedure Cursor_Off;

  var

    regs : registers;

  begin

  with regs do

    begin

    ah := $01;

    ch := $20;

    cl := $00;

    end; {with}

  Intr($10,regs);

  end; {Procedure Cursor_Off}



Procedure Check_for_file(filename : filenames);

  begin

  if IOResult <> 0 then

    begin

    RestoreCRTmode;

    GotoXY(41,1);

    TextColor(Yellow);

    Write(filename,' missing!');

    TextColor(LightGray);

    GotoXY(1,24);

    Cursor_On;

    Halt;

    end; {if}

  end; {Procedure Check_for_file}



Procedure Initialize;

  var

    R,G,B   : byte;

    key     : char;

    i       : integer;

    cfgfile : text;

  begin

  Cursor_Off;

  TextColor(LightGray);

  TextBackground(Black);

  ClrScr;

  Writeln('ͻ');

  writeln('      TS-IP       ');

  writeln(' Image Processor  ');

  Writeln('͹');

  Writeln('    Written by    ');

  Writeln('     TS Kelso     ');

  Writeln('                  ');

  Writeln('  Copyright 1993  ');

  Writeln('    All Rights    ');

  Writeln('     Reserved     ');

  Writeln('͹');

  Writeln('   Version 2.80   ');

  Writeln('   1993 Feb 03    ');

  Writeln('ͼ');

  Writeln;

{ Program description }

  GotoXY(1,24);

  Write('<Press any key to continue>');

  repeat until keypressed;

  key := ReadKey;

{ Set program defaults }

  Assign(cfgfile,'TS-IP.CFG');

  Reset(cfgfile);

  Readln(cfgfile,mselect);

  Readln(cfgfile,image_drive);

  if image_drive = '  ' then

    image_drive := '';

  Readln(cfgfile,image_directory);

  image_directory := Copy(image_directory,1,Pos(' ',image_directory)-1);

  Readln(cfgfile,work_drive);

  if work_drive = '  ' then

    work_drive := '';

  Readln(cfgfile,work_directory);

  work_directory := Copy(work_directory,1,Pos(' ',work_directory)-1);

  Readln(cfgfile,defaultname[2]);

  defaultname[2] := Copy(defaultname[2],1,Pos(' ',defaultname[2])-1);

  Close(cfgfile);

  defaultname[1] := '';

  defaultname[3] := 'MAP-NA.IMG';

{ Set up video display }

  Initialize_Display;

  Save_Palette;

  Assign(palfile,defaultname[2]);

  Reset(palfile);

  for i := 0 to 255 do

    begin

    Readln(palfile,R,G,B);

    palette[i,1] := R;

    palette[i,2] := G;

    palette[i,3] := B;

    end; {for i}

  Close(palfile);

  Set_Palette;

  SetColor(255);

  Delay(800);

  Line(cx,0,cx,max_y);

  Line(cx+1,0,cx+1,max_y);

  Line(0,cy,max_x,cy);

  Line(0,cy+1,max_x,cy+1);

{ Constant Initialization }

  lh := TextHeight('');

  lw := TextWidth('');

  for i := 1 to filetypes do

    begin

    first[i] := true;

    fchoice[i] := 1;

    end; {for i}

  for i := 1 to windows do

    begin

    xoff[i] := 0;     {windows}

    yoff[i] := 0;

    ex[i] := cx div 2;  xx[i] := ex[i];

    ey[i] := cy div 2;  xy[i] := ey[i];

    file_linked[i] := false;

    overlay_linked[i] := 0;

    invert[i] := false;

    end; {for i}

  active := 1;

  worknumber := 0;

  nobeep := false;

  end; {Procedure Initialize}



Procedure DeInitialize;

  begin

  Restore_Palette;

  CloseGraph;

  Cursor_On;

  end; {Procedure DeInitialize}



Procedure Select_Input_File;

  begin

  end; {Procedure Select_Input_File}



{*** Main Routines ***********************************************************}



Procedure SetWindow(arg : integer);

  begin

  case arg of

    1 : begin

        SetViewPort(0,0,cx-1,cy-1,ClipOn);

        active := 1;

        end; {Set Window 1}

    2 : begin

        SetViewPort(cx+2,0,max_x,cy-1,ClipOn);

        active := 2;

        end; {Set Window 2}

    3 : begin

        SetViewPort(cx+2,cy+2,max_x,max_y,ClipOn);

        active := 3;

        end; {Set Window 3}

    4 : begin

        SetViewPort(0,cy+2,cx-1,max_y,ClipOn);

        {active := 4;}

        end; {Set Window 4}

    end; {case}

  end; {Procedure SetWindow}



Procedure WriteWindow;

  var

    pixel  : byte;

    ext    : char;

    i,j    : word;

    fn     : string;

    fo     : file of byte;

    ft,fb  : text;

  begin

  fn := 'WINDOW-@.RAW';

  ext := '@';

  repeat

    ext := Succ(ext);

    fn[8] := ext;

    Assign(fo,work_drive + work_directory + fn);

    {$i-} Reset(fo);

    Close(fo); {$i+}

  until IOResult <> 0;

{Generate batch file for GIF conversion}

  Assign(fb,work_drive + work_directory + Copy(fn,1,8) + '.BAT');

  Rewrite(fb);

  Writeln(fb,'RAWTOGIF ',Copy(fn,1,8),' ',cx,' ',cy,' ',work_drive);

  Close(fb);

(*Assign(ft,work_drive + work_directory + Copy(fn,1,8) + '.DIM');

  Rewrite(ft);

  Write(ft,'P5',^J,cx,' ',cy,^J,'255',^J);

  Close(ft);

*)Assign(fo,work_drive + work_directory + fn);

  Rewrite(fo);

  for j := 0 to cy-1 do

    for i := 0 to cx-1 do

      begin

      pixel := GetPixel(i,j);

      PutPixel(i,j,255);

      Write(fo,pixel);

      PutPixel(i,j,pixel);

      end; {for i}

  Close(fo);

  end; {Procedure WriteWindow}



Procedure WriteScreen;

  var

    pixel  : byte;

    ext    : char;

    i,j    : word;

    fn     : string;

    fo     : file of byte;

    ft,fb  : text;

    bu640  : line640;

    fo640  : file of line640;

    bu800  : line800;

    fo800  : file of line800;

    bu1024 : line1024;

    fo1024 : file of line1024;

  begin

  SetViewPort(0,0,max_x,max_y,ClipOn);

  fn := 'SCREEN-@.RAW';

  ext := '@';

  repeat

    ext := Succ(ext);

    fn[8] := ext;

    Assign(fo,work_drive + work_directory + fn);

    {$i-} Reset(fo);

    Close(fo); {$i+}

  until IOResult <> 0;

{Generate batch file for GIF conversion}

  Assign(fb,work_drive + work_directory + Copy(fn,1,8) + '.BAT');

  Rewrite(fb);

  Writeln(fb,'RAWTOGIF ',Copy(fn,1,8),' ',max_x+1,' ',max_y+1,' ',work_drive);

  Close(fb);

(*Assign(ft,work_drive + work_directory + Copy(fn,1,8) + '.DIM');

  Rewrite(ft);

  Write(ft,'P5',^J,max_x+1,' ',max_y+1,^J,'255',^J);

  Close(ft);

(*Assign(fo,work_drive + work_directory + fn);

  Rewrite(fo);

  for j := 0 to max_y do

    for i := 0 to max_x do

      begin

      pixel := GetPixel(i,j);

      PutPixel(i,j,255);

      Write(fo,pixel);

      PutPixel(i,j,pixel);

      end; {for i}

  Close(fo);*)

  case max_x of

     639 : begin

           Assign(fo640,work_drive + work_directory + fn);

           Rewrite(fo640);

           for j := 0 to max_y do

             begin

             for i := 0 to max_x do

               begin

               bu640[i] := GetPixel(i,j);

               PutPixel(i,j,255);

               PutPixel(i,j,bu640[i]);

               end; {for i}

             Write(fo640,bu640);

             end; {for j}

           Close(fo640);

           end; {640 pixel line}

     799 : begin

           Assign(fo800,work_drive + work_directory + fn);

           Rewrite(fo800);

           for j := 0 to max_y do

             begin

             for i := 0 to max_x do

               begin

               bu800[i] := GetPixel(i,j);

               PutPixel(i,j,255);

               PutPixel(i,j,bu800[i]);

               end; {for i}

             Write(fo800,bu800);

             end; {for j}

           Close(fo800);

           end; {800 pixel line}

    1023 : begin

           Assign(fo1024,work_drive + work_directory + fn);

           Rewrite(fo1024);

           for j := 0 to max_y do

             begin

             for i := 0 to max_x do

               begin

               bu1024[i] := GetPixel(i,j);

               PutPixel(i,j,255);

               PutPixel(i,j,bu1024[i]);

               end; {for i}

             Write(fo1024,bu1024);

             end; {for j}

           Close(fo1024);

           end; {1024 pixel line}

    end; {case}

  SetWindow(active);

  end; {Procedure WriteScreen}



Procedure ShowPalette(height : integer);

  var

    i,j : byte;

  begin

  SetWindow(4);

  SetWriteMode(NormalPut);

  for i := 0 to 255 do

    begin

    SetColor(i);

    Line(start+i,cy-1,start+i,cy-height);

    end; {for i}

  end; {Procedure ShowPalette}



Procedure SaveImageSection(x1,y1,x2,y2 : integer; window : byte);

  var

    size : word;

  begin

  SetWindow(window);

  size := ImageSize(x1,y1,x2,y2);

  GetMem(image,size);

  GetImage(x1,y1,x2,y2,image^);

  end; {Procedure SaveImageSection}



Procedure RestoreImageSection(x1,y1,x2,y2 : integer; window : byte);

  var

    size : word;

  begin

  SetWindow(window);

  PutImage(x1,y1,image^,NormalPut);

  size := ImageSize(x1,y1,x2,y2);

  FreeMem(image,size);

  end; {Procedure RestoreImageSection}



Procedure SaveWorkSection(section,x1,y1,x2,y2 : integer);

  var

    size : word;

  begin

  SetWindow(4);

  size := ImageSize(x1,y1,x2,y2);

  GetMem(work[section],size);

  GetImage(x1,y1,x2,y2,work[section]^);

  end; {Procedure SaveWorkSection}



Procedure RestoreWorkSection(section,x1,y1,x2,y2 : integer);

  var

    size : word;

  begin

  SetWindow(4);

  PutImage(x1,y1,work[section]^,NormalPut);

  size := ImageSize(x1,y1,x2,y2);

  FreeMem(work[section],size);

  end; {Procedure RestoreWorkSection}



Procedure FindFiles(name : string; filetype : integer);

  var

    searchinfo : SearchRec;

  begin

  FindFirst(name,Archive,searchinfo);

  while DOSError = 0 do

    begin

    names[filetype] := names[filetype] + 1;

    namelist[filetype,names[filetype]] := searchinfo.name;

    if first[filetype] and (searchinfo.name = defaultname[filetype]) then

      begin

      fchoice[filetype] := names[filetype];

      first[filetype] := false;

      end; {if}

    FindNext(searchinfo);

    end; {while}

  end; {Procedure FindFiles}



Procedure FileWindow(title : string; xpos,ypos,width,height,work : integer);

  var

    i          : word;

    barr,blank : string;

  begin

  xpos := xpos * lw;

  ypos := ypos * lh;

  SaveWorkSection(work,xpos,ypos,xpos+(width+4)*lw,ypos+(height+4)*lh);

  SetFillStyle(SolidFill,0);

  Bar(xpos,ypos,xpos+(width+4)*lw,ypos+(height+4)*lh);

  barr  := '';

  blank := '                                                                ';

  barr := Copy(barr,1,width+2);

  blank := Copy(blank,1,width);

  OutTextXY(xpos,ypos,  ''+barr+'');

  OutTextXY(xpos,ypos+lh,' '+title+' ');

  OutTextXY(xpos,ypos+2*lh,''+barr+'');

  for i := 1 to height do

    OutTextXY(xpos,ypos+(2+i)*lh,' '+blank+' ');

  OutTextXY(xpos,ypos+(height+3)*lh,''+barr+'');

  end; {Procedure FileWindow}



Procedure ListFiles(start : integer;

               var select : integer;

                 filetype,

   xpos,ypos,width,height : integer);

  var

    i,listlength : integer;

    blank,tbar   : string;

  begin

  SetWindow(4);

  xpos := xpos * lw;

  ypos := ypos * lh;

  tbar :=  Copy('',1,width);

  blank := Copy('                                                 ',1,width);

  listlength := IMin(height,names[filetype]-start+1);

  SetFillStyle(SolidFill,0);

  Bar(xpos+2*lw,ypos+3*lh,xpos+(width+2)*lw,ypos+(listlength+3)*lh);

  SetColor(255);

  for i := 1 to listlength do

    OutTextXY(xpos+2*lw,ypos+(2+i)*lh,Copy(namelist[filetype,start+i-1]+blank,1,width));

  OutTextXY(xpos+2*lw,ypos+(2+select-start+1)*lh,tbar);

  SetColor(0);

  OutTextXY(xpos+2*lw,ypos+(2+select-start+1)*lh,Copy(namelist[filetype,select],1,width));

  SetColor(255);

  end; {Procedure ListFiles}



Procedure SelectFile(title : string; var filename : string; filetype : integer);

  const

    width = 13;

    height = 10;

  var

    key           : char;

    xp,yp         : word;

    lchoice,start : integer;

  begin

  xp := ((cx - lw * (width + 4)) div 2) div lw;

  yp := 5;

  FileWindow(title,xp,yp,width,height,4);

  lchoice := 0;

  start := fchoice[filetype];

  repeat

    if fchoice[filetype] <> lchoice then

      ListFiles(start,fchoice[filetype],filetype,xp,yp,width,height);

    lchoice := fchoice[filetype];

    repeat

      key := ReadKey;

    until key in [#00,^M];

    if key = #00 then

      begin

      key := ReadKey;

      case key of

        Up : fchoice[filetype] := IMax(fchoice[filetype] - 1,1);

        Dn : fchoice[filetype] := IMin(fchoice[filetype] + 1,names[filetype]);

        end; {case}

      end; {if}

    if fchoice[filetype] < start then

      start := fchoice[filetype];

    if fchoice[filetype] > start+height-1 then

      start := start + 1;

  until key = ^M;

  filename := namelist[filetype,fchoice[filetype]];

  xp := xp * lw;

  yp := yp * lh;

  RestoreWorkSection(4,xp,yp,xp+(width+4)*lw,yp+(height+4)*lh);

  end; {Procedure SelectFile}



Procedure Center_Text(line : integer; content : string);

  begin

  SetTextJustify(CenterText,TopText);

  OutTextXY(cx div 2,line * lh,content);

  SetTextJustify(LeftText,TopText);

  end; {Procedure Center_Text}



Procedure Left_Text(line : integer; content : string);

  begin

  OutTextXY(0,line * lh,content);

  end; {Procedure Left_Text}



Procedure ShowWindows(arg : byte);

  var

    htw,hth,hsize : integer;

  begin

  SetWindow(4);

  SetFillStyle(SolidFill,255);

  Bar(cx-woffset-2*wsize,woffset,cx-woffset,2*wsize+woffset);

  SetFillStyle(SolidFill,0);

  htw := lh div 2;

  hth := lw div 2;

  hsize := wsize div 2;

  case arg of

    1 : begin

        Bar(cx-2*wsize-woffset+1,woffset+1,cx-woffset-wsize-1,wsize+woffset-1);

        OutTextXY(cx-wsize-hsize-woffset-htw,woffset+hsize-hth,'1');

        end; {Window 1}

    2 : begin

        Bar(cx-wsize-woffset+1,woffset+1,cx-woffset-1,wsize+woffset-1);

        OutTextXY(cx-hsize-woffset-htw,woffset+hsize-hth,'2');

        end; {Window 2}

    3 : begin

        Bar(cx-wsize-woffset+1,wsize+woffset+1,cx-woffset-1,2*wsize+woffset-1);

        OutTextXY(cx-hsize-woffset-htw,woffset+wsize+hsize-hth,'3');

        end; {Window 3}

    4 : begin

        Bar(cx-2*wsize-woffset+1,wsize+woffset+1,cx-woffset-wsize-1,2*wsize+woffset-1);

        OutTextXY(cx-wsize-hsize-woffset-htw,woffset+wsize+hsize-hth,'4');

        end; {Window 4}

    end; {case}

  end; {Procedure ShowWindows}



Procedure SelectWindow;

  var

    choice : char;

  begin

  SaveWorkSection(2,cx-woffset-2*wsize,woffset,cx-woffset,2*wsize+woffset);

  ShowWindows(active);

  repeat

    choice := ReadKey;

    if choice in ['1','2','3'] then

      begin

      case choice of

        '1' : active := 1;

        '2' : active := 2;

        '3' : active := 3;

        end; {case}

      ShowWindows(active);

      end; {if}

  until choice = ^M;

  RestoreWorkSection(2,cx-woffset-2*wsize,woffset,cx-woffset,2*wsize+woffset);

  SetWindow(active);

  end; {Procedure SelectWindow}



Procedure DisplayOverlay(level : byte);

  label

    interrupt;

  var

    i,j,ystart,xstart,pixel,oldpixel : word;

    line                             : data;

  begin

  nobeep := false;

  SetWindow(active);

  ystart := IMin(yoff[active],fsize[active]-cy);

  xstart := IMin(xoff[active],samples-cx);

  Reset(ovlfile[active,level]);

  Seek(ovlfile[active,level],ystart);

  if invert[active] then

    for j := 0 to cy-1 do

      begin

      Read(ovlfile[active,level],line);

      for i := 0 to cx-1 do

        begin

        pixel := line.pixel[cx-i+xstart-1];

        oldpixel := GetPixel(i,cy-j-1);

        case overlay_method[active,level] of

          '0' : pixel := (pixel + oldpixel) div 2;

          '1' : pixel := pixel AND oldpixel;

          '2' : pixel := pixel OR oldpixel;

          '3' : pixel := pixel XOR oldpixel;

          '4' : pixel := IMax(pixel,oldpixel);

          '5' : {if ((pixel = 0) or (oldpixel = 0)) then

                  pixel := IMax(pixel,oldpixel)

                else

                  pixel := IMin(pixel,oldpixel);}

                pixel := IMin(pixel,oldpixel);

          '6' : pixel := ((pixel - oldpixel)+256) div 2;

          '7' : pixel := 2 * pixel - oldpixel;

          end; {case}

        if pixel < 0 then pixel := 0;

        if pixel > 255 then pixel := 255;

        PutPixel(i,cy-j-1,pixel);

        end; {for i}

      if keypressed then

        begin

        nobeep := true;

        Goto interrupt;

        end; {if}

      end {for j}

  else

    for j := 0 to cy-1 do

      begin

      Read(ovlfile[active,level],line);

      for i := 0 to cx-1 do

        begin

        pixel := line.pixel[i+xstart];

        oldpixel := GetPixel(i,j);

        case overlay_method[active,level] of

          '0' : pixel := (pixel + oldpixel) div 2;

          '1' : pixel := pixel AND oldpixel;

          '2' : pixel := pixel OR oldpixel;

          '3' : pixel := pixel XOR oldpixel;

          '4' : pixel := IMax(pixel,oldpixel);

          '5' : {if ((pixel = 0) or (oldpixel = 0)) then

                  pixel := IMax(pixel,oldpixel)

                else

                  pixel := IMin(pixel,oldpixel);}

                pixel := IMin(pixel,oldpixel);

          '6' : pixel := ((pixel - oldpixel)+256) div 2;

          '7' : pixel := 2 * pixel - oldpixel;

          end; {case}

        if pixel < 0 then pixel := 0;

        if pixel > 255 then pixel := 255;

        PutPixel(i,j,pixel);

        end; {for i}

      if keypressed then

        begin

        nobeep := true;

        Goto interrupt;

        end; {if}

      end; {for j}

interrupt:

  Close(ovlfile[active,level]);

  end; {Procedure DisplayOverlay}



Procedure RedisplayImage;

  label

    interrupt;

  var

    i,j,xstart,ystart : integer;

    line              : data;

  begin

  nobeep := false;

  if file_linked[active] then

    begin

    SetWindow(active);

    ystart := IMin(yoff[active],fsize[active]-cy);

    Reset(imgfile[active]);

    Seek(imgfile[active],ystart);

    xstart := IMin(xoff[active],samples-cx);

    if invert[active] then

      for j := 0 to cy-1 do

        begin

        Read(imgfile[active],line);

        for i := 0 to cx-1 do

          PutPixel(i,cy-j-1,line.pixel[cx-i+xstart-1]);

        if keypressed then

          begin

          nobeep := true;

          Goto interrupt;

          end; {if}

        end {for j}

    else

      for j := 0 to cy-1 do

        begin

        Read(imgfile[active],line);

        for i := 0 to cx-1 do

          PutPixel(i,j,line.pixel[i+xstart]);

        if keypressed then

          begin

          nobeep := true;

          Goto interrupt;

          end; {if}

        end; {for j}

    end {if}

  else

    begin

    nobeep := true;

    Beep(500);

    end; {else}

interrupt:

  {$i-} Close(imgfile[active]); {$i+}

  end; {Procedure RedisplayImage}



Procedure RedisplayOverlays;

  var

    j : byte;

  begin

  for j := 1 to overlay_linked[active] do

    if not nobeep then

      DisplayOverlay(j);

  end; {Procedure RedisplayOverlays}



Procedure MoveImage;

  var

    direction : char;

    oldx,oldy : integer;

  begin

  SetWindow(active);

  direction := ReadKey;

  oldx := xoff[active];

  oldy := yoff[active];

  case direction of

    Dn : if invert[active] then

           yoff[active] := IMax(yoff[active]-(cy div 2),0)

         else

           yoff[active] := IMin(yoff[active]+(cy div 2),fsize[active]-cy);

    Up : if invert[active] then

           yoff[active] := IMin(yoff[active]+(cy div 2),fsize[active]-cy)

         else

           yoff[active] := IMax(yoff[active]-(cy div 2),0);

    Lt : if invert[active] then

           xoff[active] := IMin(xoff[active]+(cx div 2),samples-cx)

         else

           xoff[active] := IMax(xoff[active]-(cx div 2),0);

    Rt : if invert[active] then

           xoff[active] := IMax(xoff[active]-(cx div 2),0)

         else

           xoff[active] := IMin(xoff[active]+(cx div 2),samples-cx);

    Endd : yoff[active] := fsize[active] - cy;

    Home : yoff[active] := 0;

    end; {case}

  if direction in [Dn,Up,Lt,Rt,Endd,Home] then

    if (oldx <> xoff[active]) or (oldy <> yoff[active]) then

      begin

      RedisplayImage;

      if not nobeep then

        RedisplayOverlays;

      end; {if}

  end; {Procedure MoveImage}



Procedure ShowTitleBar(title : string;

               continue,wide : boolean);

  var

    i,

    title_length,

    title_start  : byte;

    top_line,

    title_line,

    middle_line,

    bottom_line  : string;

  begin

  title_length := Length(title);

  if wide then

    begin

    top_line    := 'ͻ';

    title_line  := '                                    ';

    bottom_line := 'ͼ';

    title_start := 19 - (title_length div 2);

    end {if}

  else

    begin

    top_line    := 'ͻ';

    title_line  := '               ';

    middle_line := '͹';

    bottom_line := 'ͼ';

    title_start := (17 - title_length) div 2;

    end; {else}

  for i := 1 to title_length do

    title_line[title_start+i] := title[i];

  if continue and wide then

    begin

    Center_Text(1,'ͻ');

    Center_Text(2,'               TS-IP                ');

    Center_Text(3,'          Image Processing          ');

    Center_Text(4,'              Software              ');

    Center_Text(5,'͹');

    end

  else

    Center_Text(5,top_line);

  Center_Text(6,title_line);

  Center_Text(7,bottom_line);

  end; {Procedure ShowTitleBar}



Procedure ShowMainMenuChoice(selection : char;

                             highlight : boolean);

  var

    select,line,position : byte;

  begin

  select := Ord(selection) - Ord('A');

  line := select div 2;

  position := select mod 2;

  case position of

    0 : position := (cx div 2) - 18 * lw + 1;

    1 : position := (cx + lw) div 2;

    end; {case}

  line := (line + 8) * lh;

  if highlight then

    begin

    SetFillStyle(SolidFill,255);

    Bar(position-1,line-1,position+lw*18+1,line+lh-1);

    SetColor(0);

    end {if}

  else

    begin

    SetFillStyle(SolidFill,0);

    Bar(position-1,line-1,position+lw*18+1,line+lh-1);

    end; {else}

  OutTextXY(position,line,selection+': '+display_menu[select+1]);

  SetColor(255);

  end; {Procedure ShowMainMenuChoice}



Procedure ShowSubMenuChoice(submenu,selection : char;

                                    highlight : boolean);

  var

    select1,select2,line,position : byte;

  begin

  select1 := Ord(submenu) - Ord('A') + 1;

  select2 := Ord(selection) - Ord('0');

  position := (cx - 15*lw) div 2;

  line := (select2 + 10) * lh;

  if highlight then

    begin

    SetFillStyle(SolidFill,255);

    Bar(position-1,line-1,position+15*lw+1,line+lh-1);

    SetColor(0);

    end {if}

  else

    begin

    SetFillStyle(SolidFill,0);

    Bar(position-1,line-1,position+15*lw+1,line+lh-1);

    end; {else}

  OutTextXY(position,line,selection+': '+display_submenu[select1,select2]);

  SetColor(255);

  end; {Procedure ShowSubMenuChoice}



Procedure ShowHelpMenu;

  var

    i : integer;

  begin

  SetWindow(4);

  ClearViewPort;

  ShowTitleBar('Menu',true,true);

  display_menu := help_menu;

  display_submenu := help_submenu;

  for i := 0 to 25 do

    ShowMainMenuChoice(Chr(i+Ord('A')),false);

  Center_Text(21,'?: Help Menu                        ');

  ShowPalette(10);

  menu_active := true;

  end; {Procedure ShowHelpMenu}



Procedure ShowFilterMenu;

  var

    i : integer;

  begin

  SetWindow(4);

  ClearViewPort;

  ShowTitleBar('Filters',false,true);

  display_menu := filter_menu;

  for i := 0 to 25 do

    ShowMainMenuChoice(Chr(i+Ord('A')),false);

  display_submenu := filter_submenu;

  end; {Procedure ShowHelpMenu}



Procedure Select_Image_File(auto : boolean);

  begin

  if not auto then

    begin

    names[1] := 0;

    FindFiles(image_drive+image_directory+'*.RSA',1);

    FindFiles(image_drive+image_directory+'*.RSB',1);

    FindFiles(image_drive+image_directory+'*.IMG',1);

    FindFiles(image_drive+image_directory+'*.CH*',1);

    if names[1] = 0 then

      Writeln('No files found!')

    else

      SelectFile(' Image Files ',imgfilename[active],1);

    xoff[active] := 0;

    yoff[active] := 0;

    end; {if}

  Assign(imgfile[active],image_drive+image_directory+imgfilename[active]);

  Reset(imgfile[active]);

  fsize[active] := FileSize(imgfile[active]);

  file_linked[active] := true;

  RedisplayImage;

  end; {Procedure Select_Image_File}



Procedure Select_Overlay_File(choice : char;

                               level : byte;

                                auto : boolean);

  begin

  if not auto then

    begin

    names[3] := 0;

    overlay_linked[active] := level;

    FindFiles(image_drive+image_directory+'*.RSA',3);

    FindFiles(image_drive+image_directory+'*.RSB',3);

    FindFiles(image_drive+image_directory+'*.IMG',3);

    if names[3] = 0 then

      Writeln('No files found!') {* Fix for graphics mode *}

    else

      SelectFile('Overlay Files',ovlfilename[active,level],3);

    end; {if}

  Assign(ovlfile[active,level],image_drive+image_directory+ovlfilename[active,level]);

{ Reset(ovlfile[active,level]);

  osize[active,level] := FileSize(ovlfile[active,level]); }

  overlay_method[active,level] := choice;

  DisplayOverlay(level);

  end; {Procedure Select_Overlay_File}



Procedure Select_Palette_File;

  var

    palfilename : string;

  begin

  names[2] := 0;

  FindFiles('*.PAL',2);

  if names[2] = 0 then

    Writeln('No *.PAL files found!')

  else

    SelectFile('Palette Files',palfilename,2);

  Assign(palfile,palfilename);

  Reset(palfile);

  end; {Procedure Select_Palette_File}



Procedure PaletteAdjust;

  var

    i,R,G,B : byte;

  begin

  Select_Palette_File;

  for i := 0 to 255 do

    begin

    Readln(palfile,R,G,B);

    palette[i,1] := R;

    palette[i,2] := G;

    palette[i,3] := B;

    end; {for i}

  Close(palfile);

  Set_Palette;

  end; {Procedure PaletteAdjust}



Procedure ShowHistogram;

  var

    k         : byte;

    i,j       : integer;

    value     : word;

  begin

  hmin := 255;

  hmax := 0;

  for k := 0 to 255 do

    histogram[k] := 0;

  SetWindow(active);

  for i := 0 to cx-1 do

    for j := 0 to cy-1 do

      begin

      value := GetPixel(i,j);

      if value < hmin then hmin := value;

      if value > hmax then hmax := value;

      histogram[value] := histogram[value] + 1;

      end; {for j}

  SetWindow(4);

  ClearViewPort;

  for k := 0 to 255 do

    begin

    SetColor(k);

    Line(start+k,cy-01,start+k,cy-(histogram[k] div 16)-01); {01 was 15}

    end; {for k}

  menu_active := false;

  submenu_active := false;

  end; {Procedure ShowHistogram}



Procedure ClearWindow;

  begin

  SetWindow(active);

  ClearViewPort;

  end; {Procedure ClearWindow}



Procedure Mark_Point(xarg,yarg,zarg,color : integer);

  var

    x1,x2,y1,y2 : integer;

  begin

  x1 := IMax(xarg - zarg,0);  x2 := IMin(xarg + zarg,cx-1);

  y1 := IMax(yarg - zarg,0);  y2 := IMin(yarg + zarg,cy-1);

  SaveImageSection(x1,y1,x2,y2,active);

  SetWriteMode(NormalPut);

  Line(x1,yarg,x2,yarg);

  Line(xarg,y1,xarg,y2);

  PutPixel(xarg,yarg,color);

  end; {Procedure Mark_Point}



Procedure Unmark_Point(xarg,yarg,zarg : integer);

  var

    x1,x2,y1,y2 : integer;

  begin

  x1 := IMax(xarg - zarg,0);  x2 := IMin(xarg + zarg,cx-1);

  y1 := IMax(yarg - zarg,0);  y2 := IMin(yarg + zarg,cy-1);

  RestoreImageSection(x1,y1,x2,y2,active);

  end; {Procedure Unmark_Point}



Procedure Profile;

  var

    choice          : char;

    win1,win2       : byte;

    i,j,change      : integer;

    img_ofs,img_seg : word;

  begin

  SetWindow(4);

  ClearViewPort;

  menu_active := false;

  submenu_active := false;

  ShowPalette(cy);

  SetColor(255);

  repeat

    SaveImageSection(ex[active],0,ex[active],cy-1,active);

    SetWriteMode(NormalPut);

    Line(ex[active],0,ex[active],cy-1);

    img_ofs := Ofs(image^);

    img_seg := Seg(image^);

    SetWindow(4);

{ SVGA256.BGI incorrectly implements GetImage; should have *three* words

  before data but only uses *two*.  If this is fixed, img_ofs+4 should be

  changed to img_ofs+6. }

    Moveto(start+Mem[img_seg:img_ofs+4],0);

    SetWriteMode(XORPut);

    for j := 1 to cy-1 do

      Lineto(start+Mem[img_seg:img_ofs+4+j],j);

    repeat

      choice := Upcase(ReadKey);

      if choice = #00 then

        begin

        choice := ReadKey;

        case choice of

          Lt : change := -1;

          Rt : change := +1;

          PgUp : step := IMin(64,step shl 1);

          PgDn : step := IMax(1,step shr 1);

          end; {case}

        end {if}

      else if choice = ESC then

        change := 0

      else if choice = 'W' then

        begin

        win1 := active;

        SelectWindow;

        win2 := active;

        active := win1;

        ex[win2] := ex[win1];

        SetWindow(4);

        Moveto(start+Mem[img_seg:img_ofs+4],0);

        for j := 1 to cy-1 do

          Lineto(start+Mem[img_seg:img_ofs+4+j],j);

        RestoreImageSection(ex[active],0,ex[active],cy-1,active);

        SetWindow(win2);

        end; {else}

    until choice in [ESC,Lt,Rt,'W'];

    if choice <> 'W' then

      begin

      Moveto(start+Mem[img_seg:img_ofs+4],0);

      for j := 1 to cy-1 do

        Lineto(start+Mem[img_seg:img_ofs+4+j],j);

      RestoreImageSection(ex[active],0,ex[active],cy-1,active);

      case change of

        -1 : ex[active] := IMax(ex[active]-step,0);

        +1 : ex[active] := IMin(ex[active]+step,cx-1);

        end; {case}

      end; {if}

  until choice = ESC;

  end; {Procedure Profile}



Procedure Examine;

  var

    choice          : char;

    win1,win2,pixel : byte;

    x,y,z           : string[3];

  Procedure Show_Coordinates;

    begin

    SetWindow(4);

    Str(ex[active]:3,x);

    Str(ey[active]:3,y);

    Str(pixel:3,z);

    SetColor(0);

    Left_Text(1,'    ');

    Left_Text(2,'    ');

    Left_Text(3,'    ');

    SetColor(255);

    Left_Text(1,'X = '+x);

    Left_Text(2,'Y = '+y);

    Left_Text(3,'Z = '+z);

    end; {Procedure Show_Coordinates}

  begin

  SaveWorkSection(1,0,0,8*lw,5*lh);

  SetFillStyle(SolidFill,0);

  Bar(0,0,8*lw,5*lh);

  SetWindow(active);

  pixel := GetPixel(ex[active],ey[active]);

  Mark_Point(ex[active],ey[active],3,pixel);

  Show_Coordinates;

  repeat

    choice := Upcase(ReadKey);

    if choice = #00 then

      begin

      choice := ReadKey;

      if choice in [Up,Dn,Rt,Lt] then

        begin

        Unmark_Point(ex[active],ey[active],3);

        case choice of

          Up : ey[active] := IMax(0,ey[active]-step);

          Dn : ey[active] := IMin(ey[active]+step,cy-1);

          Lt : ex[active] := IMax(0,ex[active]-step);

          Rt : ex[active] := IMin(ex[active]+step,cx-1);

          end; {case}

        pixel := GetPixel(ex[active],ey[active]);

        Mark_Point(ex[active],ey[active],3,pixel);

        Show_Coordinates;

        end {if}

      else if choice in [PgUp,PgDn] then

        case choice of

          PgUp : step := IMin(64,step shl 1);

          PgDn : step := IMax(1,step shr 1);

          end; {case}

      end {if}

    else if choice = 'W' then

      begin

      win1 := active;

      SelectWindow;

      win2 := active;

      active := win1;

      Unmark_Point(ex[active],ey[active],3);

      SetWindow(win2);

      ex[win2] := ex[win1];

      ey[win2] := ey[win1];

      pixel := GetPixel(ex[active],ey[active]);

      Mark_Point(ex[active],ey[active],3,pixel);

      Show_Coordinates;

      end;{else}

  until choice = ESC;

  Unmark_Point(ex[active],ey[active],3);

  RestoreWorkSection(1,0,0,8*lw,5*lh);

  end; {Procedure Examine}



Procedure DisplaySubMenu(choice : char;

                  var subchoice : char;

                        choices : byte);

  var

    select,i : byte;

    xp       : word;

  begin

  submenu_active := true;

  select := Ord(choice) - Ord('A');

  xp := ((cx - 19 * lw) div 2) div lw;

  FileWindow(display_menu[select+1],xp,7,15,choices,3);

  for i := 1 to choices do

    begin

    ShowSubMenuChoice(choice,Chr(i+Ord('0')-1),false);

    end; {for i}

  repeat

    subchoice := ReadKey;

  until subchoice in ['0'..Chr(Ord('0')+choices-1),ESC];

  if subchoice <> ESC then

    ShowSubMenuChoice(choice,subchoice,true);

  end; {Procedure DisplaySubMenu}



Procedure ResetSubMenu(choices : byte);

  var

    xp,yp,size : word;

  begin

  xp := ((cx - 19 * lw) div 2) div lw;

  xp := xp * lw;

  yp := 7*lh;

  if submenu_active then

    RestoreWorkSection(3,xp,yp,xp+19*lw,yp+(choices+4)*lh)

  else

    begin

    size := ImageSize(xp,yp,xp+19*lw,yp+(choices+4)*lh);

    FreeMem(work[3],size);

    end; {else}

  Delay(750);

  end; {Procedure ResetSubMenu}



Procedure PutZPixel(x,y,z : integer);

  begin

  x := 2*x;

  y := 2*y;

  PutPixel(x,y,z);

{ PutPixel(x+1,y,z);

  PutPixel(x,y+1,z);

  PutPixel(x+1,y+1,z); }

  end; {Procedure PutZPixel}



Procedure Mark_Zoom_Box(sx,sy : integer);

  begin

  SetWindow(active);

  SetWriteMode(XORPut);

  Rectangle(sx,sy,sx+cx div 2,sy+cy div 2);

  end; {Procedure Mark_Zoom_Box}



Procedure Select_Zoom_Block(var sx,sy : integer;

                           var choice : char);

  var

    lx,ly  : integer;

  begin

  lx := -1;

  ly := -1;

  repeat

    if (lx <> sx) or (ly <> sy) then

      Mark_Zoom_Box(sx,sy);

    lx := sx;

    ly := sy;

    choice := ReadKey;

    if choice = #00 then

      begin

      choice := ReadKey;

      if choice in [Up,Dn,Rt,Lt] then

        begin

        case choice of

          Up : sy := IMax(0,sy-step);

          Dn : sy := IMin(sy+step,cy div 2);

          Rt : sx := IMin(sx+step,cx div 2);

          Lt : sx := IMax(0,sx-step);

          end; {case}

        if (lx <> sx) or (ly <> sy) then

          Mark_Zoom_Box(lx,ly);

        end {if}

      else if choice in [PgUp,PgDn] then

        case choice of

          PgUp : step := IMin(64,step shl 1);

          PgDn : step := IMax(1,step shr 1);

          end; {case}

      end; {if}

  until choice in [^M,ESC];

  end; {Procedure Select_Zoom_Block}



Procedure Nearest_Neighbor;

  var

    i,j : word;

  begin

  for i := 0 to cx-1 do

    for j := 0 to cy-1 do

      if (i mod 2 = 1) or (j mod 2 = 1) then

        PutPixel(i,j,GetPixel(i - (i mod 2),j - (j mod 2)));

  end; {Procedure Nearest_Neighbor}



Procedure Bilinear;

  var

    p         : array [1..2,1..2] of byte;

    i,j,ri,rj : word;

  begin

  for i := 0 to cx do

    for j := 0 to cy do

      begin

      ri := i - (i mod 2);

      rj := j - (j mod 2);

      if (i mod 2 = 1) and (j mod 2 = 1) then

        begin

        p[1,1] := GetPixel(ri,rj);

        p[1,2] := GetPixel(ri+2,rj);

        p[2,1] := GetPixel(ri,rj+2);

        p[2,2] := GetPixel(ri+2,rj+2);

        PutPixel(i,j,(p[1,1]+p[1,2]+p[2,1]+p[2,2]) div 4);

        end {if}

      else if (i mod 2 = 1) then

        begin

        p[1,1] := GetPixel(ri,rj);

        p[1,2] := GetPixel(ri+2,rj);

        PutPixel(i,j,(p[1,1] + p[1,2]) div 2);

        end {else if}

      else if (j mod 2 = 1) then

        begin

        p[1,1] := GetPixel(ri,rj);

        p[2,1] := GetPixel(ri,rj+2);

        PutPixel(i,j,(p[1,1] + p[2,1]) div 2);

        end;

      end; {for j}

  end; {Procedure Bilinear}





Procedure Zoom;

  label

    interrupt;

  var

    hx,hy,

    start_x,

    start_y,

    i,j,pixel  : integer;

    win1,win2  : byte;

    choice,opt : char;

    image      : array [0..258,0..194] of byte;

  Procedure Cubic;

    var

      p           : array [1..4,1..4] of byte;

      i,j,i1,j1   : word;

      ri,rj,pixel : integer;

    begin

    for i := 0 to cx do

      for j := 0 to cy do

        if (i mod 2 = 1) or (j mod 2 = 1) then

          begin

          ri := i - (i mod 2) - 2;  ri := (ri div 2) + 1;

          rj := j - (j mod 2) - 2;  rj := (rj div 2) + 1;

          for i1 := 1 to 4 do

            for j1 := 1 to 4 do

              p[j1,i1] := image[ri+i1-1,rj+j1-1];

          if (i mod 2 = 1) and (j mod 2 = 1) then

            begin

            pixel := (p[1,1]+p[1,4]+p[4,1]+p[4,4])

                   - 5*(p[1,2]+p[1,3]+p[2,1]+p[2,4]+p[3,1]+p[3,4]+p[4,2]+p[4,3]

                        - 5*(p[2,2]+p[2,3]+p[3,2]+p[3,3]));

            pixel := pixel div 64;

            end {if}

          else if (i mod 2 = 1) then

            begin

            pixel := 5*(p[2,2]+p[2,3]) - p[2,1] - p[2,4];

            pixel := pixel div 8;

            end {else if}

          else

            begin

            pixel := 5*(p[2,2]+p[3,2]) - p[1,2] - p[4,2];

            pixel := pixel div 8;

            end; {else else}

          pixel := IMax(0,pixel);

          pixel := IMin(pixel,255);

          PutPixel(i,j,pixel); 

          end; {if}

    end; {Procedure Cubic}

  begin

  DisplaySubMenu('Z',opt,4);

  Delay(500);

  win1 := active;

  hx := cx div 2;

  hy := cy div 2;

  start_x := hx div 2;

  start_y := hy div 2;

  Select_Zoom_Block(start_x,start_y,choice);

  if choice <> ESC then

    begin

    SelectWindow;

    win2 := active;

    SetWindow(win1);

    Mark_Zoom_Box(start_x,start_y);

    for i := 0 to (cx div 2) + 3 do

      for j := 0 to (cy div 2) + 3 do

        image[i,j] := GetPixel(start_x+i-1,start_y+j-1);

    SetWindow(win2);

    for i := 0 to (cx div 2) do

      for j := 0 to (cy div 2) do

        begin

        PutZPixel(i,j,image[i+1,j+1]);

        if keypressed then

          begin

          choice := ReadKey;

          if choice = ESC then

            Goto interrupt;

          end; {if}

        end; {for j}

    SetWindow(win2);

    case opt of

      '1' : Nearest_Neighbor;

      '2' : Bilinear;

      '3' : Cubic;

      end; {case}

    end {if}

  else

    Mark_Zoom_Box(start_x,start_y);

interrupt:

  ResetSubMenu(4);

  SetWindow(win1);

  end; {Procedure Zoom}



Procedure SpecialFilter(arg : byte);

  label

    interrupt;

  const

    weight = 2;

  var

    ava,avb,

    diff,thresh,

    win1,win2,pixel : integer;

    i,j             : word;

    l,m,x,y         : shortint;

    k               : byte;

    order           : array [1..25] of byte;

    choice          : char;

  Procedure Sort(arg : byte);

    var

      changed : boolean;

      k,temp  : byte;

    begin

    repeat

      changed := false;

      for k := 1 to arg-1 do

        begin

        if order[k] > order[k+1] then

          begin

          changed := true;

          temp := order[k];

          order[k] := order[k+1];

          order[k+1] := temp;

          end; {if}

        end; {for k}

      for k := arg-1 downto 1 do

        begin

        if order[k] > order[k+1] then

          begin

          changed := true;

          temp := order[k];

          order[k] := order[k+1];

          order[k+1] := temp;

          end; {if}

        end; {for k}

    until not changed;

    end; {Procedure Sort}

  begin

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(win1);

  for i := 1 to cx-2 do

    for j := 1 to cy-2 do

      begin

      SetWindow(win1);

      case arg of

        1,

        2 : begin

            ava := (GetPixel(i-1,j-1)

                  + GetPixel(i-1,j+1)

                  + GetPixel(i+1,j-1)

                  + GetPixel(i+1,j+1)) div 4;

            avb := (GetPixel(i-1,j)

                  + GetPixel(i+1,j)

                  + GetPixel(i,j-1)

                  + GetPixel(i,j+1)) div 4;

            diff := Abs(ava - avb);

            thresh := diff * weight;

            pixel := GetPixel(i,j);

            if (Abs(pixel - ava) > thresh) or (Abs(pixel - avb) > thresh) then

              case arg of

                1 : pixel := avb;

                2 : pixel := ava;

                end; {case}

            end; {1,2}

        3,

        4 : begin

            k := 0;

            for l := -1 to 1 do

              for m := -1 to 1 do

                begin

                k := k + 1;

                order[k] := GetPixel(i-l,j-m);

                end; {for j}

            Sort(9);

            pixel := GetPixel(i,j);

            if arg = 3 then

              pixel := order[5]

            else if Abs(pixel - order[5]) > 30 then

              pixel := order[5];

            end; {3,4}

        5,

        6 : begin

            k := 0;

            for l := -2 to 2 do

              for m := -2 to 2 do

                begin

                k := k + 1;

                order[k] := GetPixel(i-l,j-m);

                end; {for j}

            Sort(25);

            pixel := GetPixel(i,j);

            if arg = 5 then

              pixel := order[13]

            else if Abs(pixel - order[13]) > 30 then

              pixel := order[13];

            end; {5,6}

        end; {case}

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

      SetWindow(win2);

      PutPixel(i,j,pixel);

      if keypressed then

        begin

        choice := ReadKey;

        if choice = ESC then

          Goto interrupt;

        end; {if}

      end; {for j}

interrupt:

  SetWindow(win1);

  end; {Procedure SpecialFilter}



Procedure Sobel(arg : byte);

  label

    interrupt;

  var

    pix1,pix2,

    pix3,pix4,

    win1,win2,pixel : integer;

    i,j             : word;

    x,y             : shortint;

    scale           : byte;

    choice          : char;

    mx,my           : array[-1..1,-1..1] of shortint;

  begin

  mx[-1,-1] := -1;  mx[ 0,-1] := -2;  mx[ 1,-1] := -1;

  mx[-1, 0] :=  0;  mx[ 0, 0] :=  0;  mx[ 1, 0] :=  0;

  mx[-1, 1] :=  1;  mx[ 0, 1] :=  2;  mx[ 1, 1] :=  1;

  my[-1,-1] := -1;  my[ 0,-1] :=  0;  my[ 1,-1] :=  1;

  my[-1, 0] := -2;  my[ 0, 0] :=  0;  my[ 1, 0] :=  2;

  my[-1, 1] := -1;  my[ 0, 1] :=  0;  my[ 1, 1] :=  1;

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(win1);

  for i := 1 to cx-2 do

    for j := 1 to cy-2 do

      begin

      SetWindow(win1);

      pixel := 0;

      case arg of

        1 : begin

            for x := -1 to 1 do

              for y := -1 to 1 do

                pixel := pixel + mx[x,y] * GetPixel(i+x,j+y);

            end; {Gx(1)}

        2 : begin

            for x := -1 to 1 do

              for y := -1 to 1 do

                pixel := pixel - mx[x,y] * GetPixel(i+x,j+y);

            end; {Gx(2)}

        3 : begin

            for x := -1 to 1 do

              for y := -1 to 1 do

                pixel := pixel + my[x,y] * GetPixel(i+x,j+y);

            end; {Gy(1)}

        4 : begin

            for x := -1 to 1 do

              for y := -1 to 1 do

                pixel := pixel - my[x,y] * GetPixel(i+x,j+y);

            end; {Gy(2)}

        5 : begin

            pix1 := 0; pix2 := 0;

            for x := -1 to 1 do

              for y := -1 to 1 do

                begin

                pixel := GetPixel(i+x,j+y);

                pix1 := pix1 + mx[x,y] * pixel;

                pix2 := pix2 - mx[x,y] * pixel;

                end; {for y}

            pixel := IMax(pix1,pix2);

            end; {Gx}

        6 : begin

            pix3 := 0; pix4 := 0;

            for x := -1 to 1 do

              for y := -1 to 1 do

                begin

                pixel := GetPixel(i+x,j+y);

                pix3 := pix3 + my[x,y] * pixel;

                pix4 := pix4 - my[x,y] * pixel;

                end; {for y}

            pixel := IMax(pix3,pix4);

            end; {Gy}

        7 : begin

            pix1 := 0; pix2 := 0; pix3 := 0; pix4 := 0;

            for x := -1 to 1 do

              for y := -1 to 1 do

                begin

                pixel := GetPixel(i+x,j+y);

                pix1 := pix1 + mx[x,y] * pixel;

                pix2 := pix2 - mx[x,y] * pixel;

                pix3 := pix3 + my[x,y] * pixel;

                pix4 := pix4 - my[x,y] * pixel;

                end; {for y}

            pix1 := IMax(pix1,pix2);

            pix3 := IMax(pix3,pix4);

            pixel := IMax(pix1,pix3);

            end; {Gx+Gy}

        end; {case}

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

      SetWindow(win2);

      PutPixel(i,j,pixel);

      if keypressed then

        begin

        choice := ReadKey;

        if choice = ESC then

          Goto interrupt;

        end; {if}

      end; {for j}

interrupt:

  SetWindow(win1);

  end; {Procedure Sobel}



Procedure Filter3x3(arg : byte);

  label

    interrupt;

  var

    win1,win2,pixel : integer;

    i,j             : word;

    x,y             : shortint;

    scale           : byte;

    choice          : char;

    m               : array[-1..1,-1..1] of shortint;

  begin

  case arg of

    1 : begin {Average}

        m[-1,-1] :=  1;  m[ 0,-1] :=  1;  m[ 1,-1] :=  1;

        m[-1, 0] :=  1;  m[ 0, 0] :=  1;  m[ 1, 0] :=  1;

        m[-1, 1] :=  1;  m[ 0, 1] :=  1;  m[ 1, 1] :=  1;

        scale := 9;

        end;

    2 : begin {Blur}

        m[-1,-1] :=  0;  m[ 0,-1] :=  1;  m[ 1,-1] :=  0;

        m[-1, 0] :=  1;  m[ 0, 0] :=  1;  m[ 1, 0] :=  1;

        m[-1, 1] :=  0;  m[ 0, 1] :=  1;  m[ 1, 1] :=  0;

        scale := 5;

        end;

    3 : begin {Crispen 1}

        m[-1,-1] :=  0;  m[ 0,-1] := -1;  m[ 1,-1] :=  0;

        m[-1, 0] := -1;  m[ 0, 0] :=  5;  m[ 1, 0] := -1;

        m[-1, 1] :=  0;  m[ 0, 1] := -1;  m[ 1, 1] :=  0;

        scale := 1;

        end;

    4 : begin {Crispen 2}

        m[-1,-1] :=  0;  m[ 0,-1] := -1;  m[ 1,-1] :=  0;

        m[-1, 0] := -1;  m[ 0, 0] :=  6;  m[ 1, 0] := -1;

        m[-1, 1] :=  0;  m[ 0, 1] := -1;  m[ 1, 1] :=  0;

        scale := 2;

        end;

    5 : begin {Crispen 3}

        m[-1,-1] :=  0;  m[ 0,-1] := -1;  m[ 1,-1] :=  0;

        m[-1, 0] := -1;  m[ 0, 0] :=  7;  m[ 1, 0] := -1;

        m[-1, 1] :=  0;  m[ 0, 1] := -1;  m[ 1, 1] :=  0;

        scale := 3;

        end;

    6 : begin {Line Detection Horizontal}

        m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] := -1;

        m[-1, 0] :=  2;  m[ 0, 0] :=  2;  m[ 1, 0] :=  2;

        m[-1, 1] := -1;  m[ 0, 1] := -1;  m[ 1, 1] := -1;

        scale := 1;

        end;

    7 : begin {Line Detection Vertical}

        m[-1,-1] := -1;  m[ 0,-1] :=  2;  m[ 1,-1] := -1;

        m[-1, 0] := -1;  m[ 0, 0] :=  2;  m[ 1, 0] := -1;

        m[-1, 1] := -1;  m[ 0, 1] :=  2;  m[ 1, 1] := -1;

        scale := 1;

        end;

    8 : begin {Line Detection Diagonal 1}

        m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] :=  2;

        m[-1, 0] := -1;  m[ 0, 0] :=  2;  m[ 1, 0] := -1;

        m[-1, 1] :=  2;  m[ 0, 1] := -1;  m[ 1, 1] := -1;

        scale := 1;

        end;

    9 : begin {Line Detection Diagonal 2}

        m[-1,-1] :=  2;  m[ 0,-1] := -1;  m[ 1,-1] := -1;

        m[-1, 0] := -1;  m[ 0, 0] :=  2;  m[ 1, 0] := -1;

        m[-1, 1] := -1;  m[ 0, 1] := -1;  m[ 1, 1] :=  2;

        scale := 1;

        end;

   10 : begin {Laplacian}

        m[-1,-1] :=  0;  m[ 0,-1] :=  1;  m[ 1,-1] :=  0;

        m[-1, 0] :=  1;  m[ 0, 0] := -4;  m[ 1, 0] :=  1;

        m[-1, 1] :=  0;  m[ 0, 1] :=  1;  m[ 1, 1] :=  0;

        scale := 1;

        end;

   14 : begin {Sharpen 1}

        m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] := -1;

        m[-1, 0] := -1;  m[ 0, 0] :=  9;  m[ 1, 0] := -1;

        m[-1, 1] := -1;  m[ 0, 1] := -1;  m[ 1, 1] := -1;

        scale := 1;

        end;

   15 : begin {Sharpen 2}

        m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] := -1;

        m[-1, 0] := -1;  m[ 0, 0] := 10;  m[ 1, 0] := -1;

        m[-1, 1] := -1;  m[ 0, 1] := -1;  m[ 1, 1] := -1;

        scale := 2;

        end;

   16 : begin {Sharpen 3}

        m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] := -1;

        m[-1, 0] := -1;  m[ 0, 0] := 11;  m[ 1, 0] := -1;

        m[-1, 1] := -1;  m[ 0, 1] := -1;  m[ 1, 1] := -1;

        scale := 3;

        end;

   17 : begin {Average/Crispen}

        m[-1,-1] :=  1;  m[ 0,-1] := -2;  m[ 1,-1] :=  1;

        m[-1, 0] := -2;  m[ 0, 0] :=  5;  m[ 1, 0] := -2;

        m[-1, 1] :=  1;  m[ 0, 1] := -2;  m[ 1, 1] :=  1;

        scale := 1;

        end;

    end; {case}

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(win1);

  for i := 1 to cx-2 do

    for j := 1 to cy-2 do

      begin

      SetWindow(win1);

      pixel := 0;

      for x := -1 to 1 do

        for y := -1 to 1 do

          pixel := pixel + m[x,y] * GetPixel(i+x,j+y);

      pixel := pixel div scale;

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

      SetWindow(win2);

      PutPixel(i,j,pixel);

      if keypressed then

        begin

        choice := ReadKey;

        if choice = ESC then

          Goto interrupt;

        end; {if}

      end; {for j}

interrupt:

  SetWindow(win1);

  end; {Procedure Filter3x3}



Procedure Filter5x5(arg : byte);

  label

    interrupt;

  var

    win1,win2 : byte;

    pixel     : integer;

    i,j       : word;

    x,y       : shortint;

    scale     : byte;

    choice    : char;

    m         : array[-2..2,-2..2] of shortint;

  begin

  case arg of

    1 : begin {Average}

        m[-2,-2] :=  1;  m[-1,-2] :=  1;  m[ 0,-2] :=  1;  m[ 1,-2] :=  1;  m[ 2,-2] := 1;

        m[-2,-1] :=  1;  m[-1,-1] :=  1;  m[ 0,-1] :=  1;  m[ 1,-1] :=  1;  m[ 2,-1] := 1;

        m[-2, 0] :=  1;  m[-1, 0] :=  1;  m[ 0, 0] :=  1;  m[ 1, 0] :=  1;  m[ 2, 0] := 1;

        m[-2, 1] :=  1;  m[-1, 1] :=  1;  m[ 0, 1] :=  1;  m[ 1, 1] :=  1;  m[ 2, 1] := 1;

        m[-2, 2] :=  1;  m[-1, 2] :=  1;  m[ 0, 2] :=  1;  m[ 1, 2] :=  1;  m[ 2, 2] := 1;

        scale := 25;

        end; {1}

    2 : begin {Bas relief}

        m[-2,-2] :=  0;  m[-1,-2] :=  0;  m[ 0,-2] := -1;  m[ 1,-2] :=  0;  m[ 2,-2] := 0;

        m[-2,-1] :=  0;  m[-1,-1] := -1;  m[ 0,-1] := -1;  m[ 1,-1] := -1;  m[ 2,-1] := 0;

        m[-2, 0] := -1;  m[-1, 0] := -1;  m[ 0, 0] :=  1;  m[ 1, 0] :=  1;  m[ 2, 0] := 1;

        m[-2, 1] :=  0;  m[-1, 1] :=  1;  m[ 0, 1] :=  1;  m[ 1, 1] :=  1;  m[ 2, 1] := 0;

        m[-2, 2] :=  0;  m[-1, 2] :=  0;  m[ 0, 2] :=  1;  m[ 1, 2] :=  0;  m[ 2, 2] := 0;

        scale := 1;

        end; {2}

    end; {case}

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(win1);

  for i := 2 to cx-3 do

    for j := 2 to cy-3 do

      begin

      SetWindow(win1);

      pixel := 0;

      for x := -2 to 2 do

        for y := -2 to 2 do

          pixel := pixel + m[x,y] * GetPixel(i+x,j+y);

      pixel := pixel div scale;

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

      SetWindow(win2);

      PutPixel(i,j,pixel);

      if keypressed then

        begin

        choice := ReadKey;

        if choice = ESC then

          Goto interrupt;

        end; {if}

      end; {for j}

interrupt:

  SetWindow(win1);

  end; {Procedure Filter5x5}



Procedure Filters;

  var

    choice1,choice2 : char;

  begin

  ShowFilterMenu;

  repeat

    choice1 := Upcase(ReadKey);

    if choice1 in ['A'..'Z'] then

      begin

      ShowMainMenuChoice(choice1,true);

      Delay(750);

      end; {if}

    case choice1 of

      'A' : begin

            DisplaySubMenu(choice1,choice2,2);

            case choice2 of

              '0' : Filter3x3(1);

              '1' : Filter5x5(1);

              end; {case}

            ResetSubMenu(2);

            end; {Average}

      'B' : begin

            Filter3x3(2);

            end; {Blur}

      'C' : begin

            DisplaySubMenu(choice1,choice2,3);

            case choice2 of

              '0' : Filter3x3(3);

              '1' : Filter3x3(4);

              '2' : Filter3x3(5);

              end; {case}

            ResetSubMenu(3);

            end; {Crispen}

      'D' : begin

            DisplaySubMenu(choice1,choice2,4);

            case choice2 of

              '0' : Filter3x3(6);

              '1' : Filter3x3(7);

              '2' : Filter3x3(8);

              '3' : Filter3x3(9);

              end; {case}

            ResetSubMenu(4);

            end; {Line Detection}

      'F' : begin

            end; {FFT/IFFT}

      'L' : begin

            Filter3x3(10);

            end; {Laplacian Operator}

      'M' : begin

            DisplaySubMenu(choice1,choice2,2);

            case choice2 of

              '0' : SpecialFilter(3);

              '1' : SpecialFilter(5);

              end; {case}

            ResetSubMenu(2);

            end; {Median Filter}

      'N' : begin

            DisplaySubMenu(choice1,choice2,4);

            case choice2 of

              '0' : SpecialFilter(1);

              '1' : SpecialFilter(2);

              '2' : SpecialFilter(4);

              '3' : SpecialFilter(6);

              end; {case}

            ResetSubMenu(4);

            end; {Noise Removal}

      'S' : begin

            DisplaySubMenu(choice1,choice2,7);

            case choice2 of

              '0' : Sobel(1);

              '1' : Sobel(2);

              '2' : Sobel(3);

              '3' : Sobel(4);

              '4' : Sobel(5);

              '5' : Sobel(6);

              '6' : Sobel(7);

              end; {case}

            ResetSubMenu(7);

            end; {Sobel Operators}

      'X' : begin

            DisplaySubMenu(choice1,choice2,3);

            case choice2 of

              '0' : Filter3x3(14);

              '1' : Filter3x3(15);

              '2' : Filter3x3(16);

              end; {case}

            ResetSubMenu(3);

            end; {Sharpen}

      'Z' : begin

            DisplaySubMenu(choice1,choice2,2);

            case choice2 of

              '0' : Filter3x3(17);

              '1' : Filter5x5(2);

              end; {case}

            ResetSubMenu(2);

            end; {Special}

      end; {case}

  until choice1 in ['A'..'Z',ESC];

  if choice1 in ['A'..'Z'] then

    begin

    SetWindow(4);

    ShowMainMenuChoice(choice1,false);

    Delay(750);

    end; {if}

  ShowHelpMenu;

  end; {Procedure Filters}



Procedure Combine_Images;

  label

    interrupt;

  var

    win1,win2 : byte;

    choice    : char;

    i,j       : integer;

    pixel     : integer;

  begin

  DisplaySubMenu('A',choice,8);

  if choice <> ESC then

    begin

  win1 := active;

  SelectWindow;

  win2 := active;

  for i := 0 to cx-1 do

    for j := 0 to cy-1 do

      begin

      SetWindow(win1);

      pixel := GetPixel(i,j);

      SetWindow(win2);

      case choice of

        '0' : pixel := (pixel + GetPixel(i,j)) div 2;

        '1' : pixel := pixel AND GetPixel(i,j);

        '2' : pixel := pixel OR GetPixel(i,j);

        '3' : pixel := pixel XOR GetPixel(i,j);

        '4' : pixel := IMax(pixel,GetPixel(i,j));

        '5' : pixel := IMin(pixel,GetPixel(i,j));

        '6' : pixel := ((GetPixel(i,j) - pixel)+256) div 2;

        '7' : pixel := 2 * GetPixel(i,j) - pixel;

        end; {case}

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

      PutPixel(i,j,pixel);

      if keypressed then

        Goto interrupt;

      end; {for j}

    end; {if}

interrupt:

  SetWindow(win1);

  ResetSubMenu(8);

  end; {Procedure Combine_Images}



Procedure CopyWindow;

  var

    win1,win2 : byte;

    i,j       : integer;

    pixel     : integer;

  begin

  win1 := active;

  SelectWindow;

  win2 := active;

  for i := 0 to cx-1 do

    begin

    SaveImageSection(i,0,i,cy-1,win1);

    RestoreImageSection(i,0,i,cy-1,win2);

    end; {for i}

  SetWindow(win1);

  end; {Procedure CopyWindow}



Procedure Stretch_Histogram(arg : byte);

  var

    choice      : char;

    y1,y2,

    hstep,range : byte;

    i,j         : word;

    f1,f2,f3    : real;

    pixel       : {longint}integer;

  Procedure Label_End(title : string);

    begin

    SetColor(0);

    Left_Text(1,'');

    SetColor(255);

    Left_Text(1,title);

    end; {Procedure Label_End}

  Procedure Show_Position(hpos,vpos : byte);

    var

      count,color,shift : string[4];

    begin

    SetWriteMode(XORPut);

    Line(start+hpos,0,start+hpos,cy-1);

    Str(hpos:4,color);

    Str(vpos:4,shift);

    Str(histogram[hpos]:4,count);

    SetColor(0);

    Left_Text(2,'');

    Left_Text(3,'');

    Left_Text(4,'');

    SetColor(255);

    Left_Text(2,'Color = '+color);

    Left_Text(3,'Count = '+count);

    Left_Text(4,'Shift = '+shift);

    end; {Procedure Show_Position}

  Procedure ShowScale(x1,y1,x2,y2 : byte);

    begin

    SetWriteMode(XORPut);

    if x1 > 0 then

      Line(start,cy-1,start+x1,cy-1);

    if y1 > 0 then

      Line(start+x1,cy-1,start+x1,cy-1-y1);

    Line(start+x1,cy-1-y1,start+x2,cy-1-y2);

    if y2 < 255 then

      Line(start+x2,cy-1-y2,start+x2,cy-256);

    if x2 < 255 then

      Line(start+x2,cy-256,start+255,cy-256);

    SetWriteMode(NormalPut);

    Rectangle(start-1,cy-256,start+256,cy-1);

    end; {Procedure ShowScale}

  Procedure Select_Position(var hpos,vpos : byte);

    begin

    repeat

      choice := ReadKey;

      if choice = #00 then

        begin

        choice := ReadKey;

        if choice in [Lt,Rt,Dn,Up] then

          begin

          Show_Position(hpos,vpos);

          case choice of

            Lt : hpos := IMax(0,hpos-hstep);

            Rt : hpos := IMin(hpos+hstep,255);

            Dn : vpos := IMax(0,vpos-hstep);

            Up : vpos := IMin(vpos+hstep,255);

            end; {case}

          Show_Position(hpos,vpos);

          end {if}

        else if choice in [PgUp,PgDn] then

          case choice of

            PgUp : hstep := IMin(64,hstep shl 1);

            PgDn : hstep := IMax(1,hstep shr 1);

            end; {case}

        end;

    until choice = ^M;

    Line(start+hpos,0,start+hpos,cy-1);

    end; {Procedure Select_Position}

  begin

  ShowHistogram;

  SetColor(255);

  SaveWorkSection(1,0,0,12*lw,5*lh);

  hstep := 1;

  y1 := 0;  y2 := 255;

  ShowScale(0,0,255,255);

  Label_End('Low End');

  Show_Position(hmin,y1);

  Select_Position(hmin,y1);

  ShowScale(0,0,255,255);

  ShowScale(hmin,y1,255,255);

  Label_End('High End');

  Show_Position(hmax,y2);

  Select_Position(hmax,y2);

  ShowScale(hmin,y1,255,255);

  ShowScale(hmin,y1,hmax,y2);

  RestoreWorkSection(1,0,0,12*lw,5*lh);

  SetWindow(active);

(*range := hmax - hmin;*)

  f1 := (y2 - y1)/(hmax - hmin);

  f2 := (y2 - y1)/Ln(10);

  f3 := 9/(hmax - hmin);

  for i := 0 to cx-1 do

    for j := 0 to cy-1 do

      begin

      pixel := GetPixel(i,j);

      if pixel < hmin then pixel := 0;

      if pixel > hmax then pixel := 255;

      if pixel in [hmin..hmax] then

         case arg of

           1 : pixel := y1 + Round(f1*(pixel - hmin));

           2 : pixel := y1 + Round(f2*Ln(f3*(pixel - hmin) + 1));

           end; {case}

      if pixel < 0 then pixel := 0;

      if pixel > 255 then pixel := 255;

(*    pixel := GetPixel(i,j);

      pixel := pixel - hmin;

      if pixel < 0 then pixel := 0;

      pixel := pixel shl 8;

      pixel := pixel div range;

      if pixel > 255 then pixel := 255; *)

      PutPixel(i,j,pixel);

      end; {for j}

  ShowHistogram;

  end; {Procedure Stretch_Histogram}



Procedure EqualizeHistogram;

  var

    last_nonzero,pixel  : byte;

    i,j                 : integer;

    size,interval,total : longint;

    map                 : array [0..255] of integer;

  begin

  ShowHistogram;

  size := (longint(cx) * longint(cy)) div 256;

  total := 0;

  for i := 0 to 255 do

    begin

    total := total + histogram[i];

    map[i] := (total div size) - 1;

    if map[i] < 0 then map[i] := 0;

    end; {for i}

  SetWindow(active);

  for i := 0 to cx-1 do

    for j := 0 to cy-1 do

      begin

      pixel := map[GetPixel(i,j)];

      PutPixel(i,j,pixel);

      end; {for j}

  ShowHistogram;

  end; {Procedure EqualizeHistogram}



Procedure EchoSelection(choice : char;

                    time_delay : integer;

                     load_menu : boolean);

  begin

  SetWindow(4);

  if not menu_active and load_menu then

    ShowHelpMenu;

  if menu_active then

    begin

    ShowMainMenuChoice(choice,true);

    Delay(time_delay);

    end; {if}

  SetWindow(active);

  end; {Procedure EchoSelection}



Procedure ResetSelection(choice : char;

                     time_delay : integer;

                      load_menu : boolean);

  begin

  SetWindow(4);

  if menu_active then

    begin

    ShowMainMenuChoice(choice,false);

    Delay(time_delay);

    end; {if}

  if load_menu then

    ShowHelpMenu;

  SetWindow(active);

  end; {Procedure ResetSelection}



Procedure Examines;

  var

    choice : char;

  begin

  DisplaySubMenu('E',choice,2);

  case choice of

    '0' : Examine;

    '1' : Profile;

    end; {case}

  ResetSubMenu(2);

  end; {Procedure Examines}



Procedure Histograms;

  var

    choice : char;

  begin

  DisplaySubMenu('H',choice,4);

  case choice of

    '0' : ShowHistogram;

    '1' : Stretch_Histogram(1);

    '2' : Stretch_Histogram(2);

    '3' : EqualizeHistogram;

    end; {case}

  ResetSubMenu(4);

  end; {Procedure Histograms}



Procedure Palettes;

  begin

  PaletteAdjust;

  end; {Procedure Palettes}



Procedure Overlay_Image;

  var

    choice : char;

  begin

  if file_linked[active] then

    begin

    DisplaySubMenu('O',choice,10);

    Delay(750);

    case choice of

      '0'..'7' : if overlay_linked[active] < overlays then

                   Select_Overlay_File(choice,overlay_linked[active]+1,false)

                 else

                   begin

                   nobeep := true;

                   Beep(500);

                   end; {else}

           '8' : begin

                 overlay_linked[active] := BMax(overlay_linked[active]-1,0);

                 RedisplayImage;

                 RedisplayOverlays;

                 end; {8}

           '9' : begin

                 overlay_linked[active] := 0;

                 RedisplayImage;

                 end;

      end; {case}

    ResetSubMenu(10);

    end {if}

  else

    begin

    nobeep := true;

    Beep(500);

    end; {else}

  end; {Procedure Overlay_Image}



Procedure Show_All;

  var

    i,j,win1 : byte;

  begin

  if file_linked[active] then

    begin

    win1 := active;

    if invert[win1] then

      case win1 of

        1 : begin

            xoff[2] := IMax(xoff[1] - cx,0);

            yoff[2] := yoff[1];

            xoff[3] := xoff[2];

            yoff[3] := IMax(yoff[1] - cy,0);

            end; {1}

        2 : begin

            xoff[1] := IMin(xoff[2] + cx,samples);

            yoff[1] := yoff[2];

            xoff[3] := xoff[2];

            yoff[3] := IMax(yoff[2] - cy,0);

            end; {2}

        3 : begin

            xoff[1] := IMin(xoff[3] + cx,samples);

            yoff[1] := IMin(yoff[3] + cy,fsize[3] - cy);

            xoff[2] := xoff[3];

            yoff[2] := yoff[1];

            end; {3}

        end {case}

    else

      case win1 of

        1 : begin

            xoff[2] := IMin(xoff[1] + cx,samples);

            yoff[2] := yoff[1];

            xoff[3] := xoff[2];

            yoff[3] := IMin(yoff[1] + cy,fsize[1] - cy);

            end; {1}

        2 : begin

            xoff[1] := IMax(xoff[2] - cx,0);

            yoff[1] := yoff[2];

            xoff[3] := xoff[2];

            yoff[3] := IMin(yoff[2] + cy,fsize[2] - cy);

            end; {2}

        3 : begin

            xoff[1] := IMax(xoff[3] - cx,0);

            yoff[1] := IMax(yoff[3] - cy,0);

            xoff[2] := xoff[3];

            yoff[2] := yoff[1];

            end; {3}

        end; {case}

    for i := 1 to windows do

      if i <> win1 then

        begin

        active := i;

        invert[active] := invert[win1];

        imgfilename[active] := imgfilename[win1];

        overlay_linked[active] := overlay_linked[win1];

        Select_Image_File(true);

        for j := 1 to overlay_linked[win1] do

          begin

          ovlfilename[active,j] := ovlfilename[win1,j];

          Select_Overlay_File(overlay_method[win1,j],j,true)

          end; {for j}

        end; {if}

    end; {if}

  active := win1;

  SetWindow(active);

  end; {Procedure Show_All}



Procedure Mark_Cut_Box(sx,sy : integer;

                          sz : byte);

  var

    sz2 : byte;

  begin

  sz2 := sz div 2;

  SetWindow(active);

  SetWriteMode(XORPut);

  Rectangle(sx-sz2,sy-sz2,sx+sz2-1,sy+sz2-1);

  end; {Procedure Mark_Cut_Box}



Procedure Select_Cut_Block(sz : byte;

                   var choice : char);

  var

    sz2   : byte;

    lx,ly : integer;

  begin

  sz2 := sz div 2;

  lx := -1;

  ly := -1;

  repeat

    if (lx <> xx[active]) or (ly <> xy[active]) then

      Mark_Cut_Box(xx[active],xy[active],sz);

    lx := xx[active];

    ly := xy[active];

    choice := ReadKey;

    if choice = #00 then

      begin

      choice := ReadKey;

      if choice in [Up,Dn,Rt,Lt] then

        begin

        case choice of

          Up : xy[active] := IMax(sz2,xy[active]-step);

          Dn : xy[active] := IMin(xy[active]+step,cy-sz2);

          Rt : xx[active] := IMin(xx[active]+step,cx-sz2);

          Lt : xx[active] := IMax(sz2,xx[active]-step);

          end; {case}

        if (lx <> xx[active]) or (ly <> xy[active]) then

          Mark_Cut_Box(lx,ly,sz);

        end {if}

      else if choice in [PgUp,PgDn] then

        case choice of

          PgUp : step := IMin(64,step shl 1);

          PgDn : step := IMax(1,step shr 1);

          end; {case}

      end; {if}

  until choice in [^M,ESC];

  end; {Procedure Select_Cut_Block}



Procedure Cut(arg : byte);

  var

    file_found     : boolean;

    pixel,arg2     : byte;

    i,j,xs,ys      : integer;

    choice         : char;

    fn,

    workfilenumber : string;

    workfile       : file of byte;

    ft,fb          : text;

  begin

  arg2 := arg div 2;

  xy[active] := IMax(arg2,xy[active]);

  xy[active] := IMin(xy[active],cy-arg2);

  xx[active] := IMin(xx[active],cx-arg2);

  xx[active] := IMax(arg2,xx[active]);

  Select_Cut_Block(arg,choice);

  if choice = ESC then

    Mark_Cut_Box(xx[active],xy[active],arg)

  else

    begin

    repeat

      Str((10000+worknumber):5,workfilenumber);

      Delete(workfilenumber,1,1);

      fn := 'WORK' + workfilenumber + '.RAW';

      Assign(workfile,work_drive + work_directory + fn);

      {$i-} Reset(workfile); {$i+}

      file_found := (IOResult = 0);

      if file_found then

        begin

        Close(workfile);

        worknumber := worknumber + 1;

        end; {if}

    until not file_found;

{Generate batch file for GIF conversion}

    Assign(fb,work_drive + work_directory + Copy(fn,1,8) + '.BAT');

    Rewrite(fb);

    Writeln(fb,'RAWTOGIF ',Copy(fn,1,8),' ',arg,' ',arg,' ',work_drive);

    Close(fb);

    Rewrite(workfile);

    xs := xx[active] - arg2;

    ys := xy[active] - arg2;

    Mark_Cut_Box(xx[active],xy[active],arg);

    for j := ys to ys + arg - 1 do

      for i := xs to xs + arg - 1 do

        begin

        pixel := GetPixel(i,j);

        PutPixel(i,j,255);

        Write(workfile,pixel);

        PutPixel(i,j,pixel);

        end; {for i}

    Close(workfile);

(*  Assign(ft,work_drive + work_directory + 'WORK' + workfilenumber + '.DIM');

    Rewrite(ft);

    Write(ft,'P5',^J,arg,' ',arg,^J,'255',^J);

    Close(ft);

*)  end; {else}

  end; {Procedure Cut}



Procedure CutImageSection;

  var

    choice : char;

    xsize  : byte;

  begin

  DisplaySubMenu('X',choice,6);

  case choice of

    '0' : xsize :=  10;

    '1' : xsize :=  20;

    '2' : xsize :=  50;

    '3' : xsize := 100;

    '4' : xsize := 200;

    '5' : xsize := 250;

    end; {case}

  Cut(xsize);

  ResetSubMenu(6);

  end; {Procedure CutImageSection}



Procedure GroupImages;

  var

    win1,win2,point,level : byte;

    name,ext              : string;

  begin

  point := Pos('.',imgfilename[active]);

  name := Copy(imgfilename[active],1,point-1);

  ext  := Copy(imgfilename[active],point+1,3);

  if (ext = 'RSA') or (ext = 'RSB') then

    begin

    win1 := active;

    SelectWindow;

    win2 := active;

    if ext = 'RSA' then

      ext := 'RSB'

    else

      ext := 'RSA';

    file_linked[win2] := true;

    imgfilename[win2] := name + '.' + ext;

    Assign(imgfile[win2],image_drive + image_directory + imgfilename[win2]);

    fsize[win2] := fsize[win1];

    xoff[win2] := xoff[win1];

    yoff[win2] := yoff[win1];

    invert[win2] := invert[win1];

    overlay_linked[win2] := overlay_linked[win1];

    for level := 1 to overlay_linked[win2] do

      begin

      ovlfilename[win2,level] := ovlfilename[win1,level];

      Assign(ovlfile[win2,level],ovlfilename[win2,level]);

      end; {for level}

    RedisplayImage;

    SetWindow(win1);

    end {if}

  else

    begin

    nobeep := true;

    Beep(500);

    end; {else}

  end; {Procedure GroupImages}



Procedure ShowClassification;

  label

    interrupt;

  var

    win1,win2,i : byte;

{   color_count : array [0..128,0..128] of word; }                {*TS920416*}

    color_count : arrdesp;                                        {*TS920416*}

  Procedure ContourHistogram;

    const

      z : array [1..5] of word = (5,55,105,155,205);

      c : array [1..5] of byte = (50,100,150,200,250);

    var

      k              : byte;

      p0,p1,p2,p3,p4 : word;

      i,j            : word;

      err            : integer;

    begin

    SetWindow(4);

    for i := 1 to 254 do

      for j := 1 to 254 do

        begin

        p0 := Round(HMatReadEl(color_count,i,j,err));

        p1 := Round(HMatReadEl(color_count,i,j-1,err));

        p2 := Round(HMatReadEl(color_count,i+1,j,err));

        p3 := Round(HMatReadEl(color_count,i,j+1,err));

        p4 := Round(HMatReadEl(color_count,i-1,j,err));

        for k := 1 to 5 do

          if z[k] in [p0..p1,p0..p2,p0..p3,p0..p4,p1..p0,p2..0,p3..0,p4..0] then

            PutPixel(start+i+1,259-j,c[k]);

        end; {for j}

    SetWindow(win1);

    end; {Procedure ContourHistogram}

  Procedure ClassifyQuadrant;

    var

      i,j,pix1,pix2 : word;

      err1,err2     : integer;

    begin

    for i := 0 to cx-1 do

      for j := 0 to cy-1 do

        begin

        SetWindow(win1);

        pix1 := GetPixel(i,j);

        PutPixel(i,j,255);

        SetWindow(win2);

        pix2 := GetPixel(i,j);

{       if (pix1 in [xlo..xhi]) and (pix2 in [ylo..yhi]) then }

        HMatWrtEl(color_count,pix1,pix2,HMatReadEl(color_count,pix1,pix2,err1)+1,err2);

{       color_count[pix1-xlo,pix2-ylo] := color_count[pix1-xlo,pix2-ylo] + 1; }

        SetWindow(win1);

        PutPixel(i,j,pix1);

        end; {for j}

    ContourHistogram;

    end; {Procedure ClassifyQuadrant}

  begin

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(4);

  ClearViewPort;

  for i := 0 to 255 do

    begin

    PutPixel(i+start+1,3,i);

    PutPixel(i+start+1,260,i);

    PutPixel(start,259-i,i);

    PutPixel(start+257,259-i,i);

    end; {for i}

  color_count := HMatDef(256,256);

  if color_count = nil then

    Halt;

  ClassifyQuadrant;

interrupt:

  HArrFree(color_count);

  SetWindow(win1);

  menu_active := false;

  submenu_active := false;

  end; {Procedure ShowClassification}



Procedure Classify_1D;

  var

    win1,win2          : byte;

    x,y,color,low,high : word;

  Procedure CheckNeighbors(argx,argy : word);

    var

      pixel : word;

    begin

    pixel := GetPixel(argx-1,argy);

    if pixel in [low..high] then

      begin

      SetWindow(win2);

      PutPixel(argx,argy,color);

      SetWindow(win1);

      CheckNeighbors(argx,argy);

      end; {if}

    end; {Procedure CheckNeighbors}

  begin

  win1 := active;

  SelectWindow;

  win2 := active;

  SetWindow(win1);

  Examine;

  SetWindow(win1);

  x := ex[win1];

  y := ey[win1];

  color := GetPixel(x,y);

  low := IMax(0,integer(color-25));

  high := IMin(255,color+25);

  CheckNeighbors(x,y);

  end; {Procedure Classify_1D}



Procedure Classify_2D;

  begin

  end; {Procedure Classify_2D}



Procedure Classify;

  var

    choice : char;

  begin

  DisplaySubMenu('K',choice,3);

  case choice of

    '0' : ShowClassification;

    '1' : Classify_1D;

    '2' : Classify_2D;

    end; {case}

  ResetSubMenu(3);

  end; {Procedure Classify}



Procedure ContourPlot;

  var

    win1,win2,k,slices,hstep : byte;

    choice                   : char;

    p0,p1,p2,p3,p4           : word;

    i,j                      : word;

    z                        : array [1..5] of byte;

  Procedure Label_End(title : string);

    begin

    SetColor(0);

    Left_Text(1,'');

    SetColor(255);

    Left_Text(1,title);

    end; {Procedure Label_End}

  Procedure Show_Position(hpos : byte; show_label : boolean);

    var

      color : string[4];

    begin

    SetWriteMode(XORPut);

    Line(start+hpos,0,start+hpos,cy-1);

    if show_label then

      begin

      Str(hpos:4,color);

      SetColor(0);

      Left_Text(2,'');

      SetColor(255);

      Left_Text(2,'Color = '+color);

      end; {if}

    end; {Procedure Show_Position}

  Procedure Select_Position(var hpos : byte);

    begin

    repeat

      choice := ReadKey;

      if choice = #00 then

        begin

        choice := ReadKey;

        if choice in [Lt,Rt] then

          begin

          Show_Position(hpos,true);

          case choice of

            Lt : hpos := IMax(0,hpos-hstep);

            Rt : hpos := IMin(hpos+hstep,255);

            end; {case}

          Show_Position(hpos,true);

          end {if}

        else if choice in [PgUp,PgDn] then

          case choice of

            PgUp : hstep := IMin(64,hstep shl 1);

            PgDn : hstep := IMax(1,hstep shr 1);

            end; {case}

        end;

    until choice = ^M;

    {Line(start+hpos,0,start+hpos,cy-1);}

    end; {Procedure Select_Position}

  begin

  DisplaySubMenu('J',choice,10);

  ShowHistogram;

  SetColor(255);

  case choice of

    '0','5' : begin

              slices := 1;

              z[1] := 125;

              end; {0,5}

    '1','6' : begin

              slices := 2;

              z[1] := 100; z[2] := 200;

              end; {1}

    '2','7' : begin

              slices := 3;

              z[1] := 75; z[2] := 150; z[3] := 225;

              end; {0}

    '3','8' : begin

              slices := 4;

              z[1] := 60; z[2] := 120; z[3] := 180; z[4] := 240;

              end; {1}

    '4','9' : begin

              slices := 5;

              z[1] := 50; z[2] := 100; z[3] := 150; z[4] := 200; z[5] := 250;

              end; {0}

    end; {case}

  case choice of

    '0'..'4' : begin

               for i := 1 to slices do

                 Show_Position(z[i],false);

               end; {0}

    '5'..'9' : begin

               SaveWorkSection(1,0,0,12*lw,3*lh);

               hstep := 1;

               for i := 1 to slices do

                 begin

                 Label_End('Break '+Chr(Ord('0')+i));

                 Show_Position(z[i],true);

                 Select_Position(z[i]);

                 end; {for i}

               RestoreWorkSection(1,0,0,12*lw,3*lh);

          end; {1}

    end; {case}

  ResetSubMenu(10);

  win1 := active;

  SelectWindow;

  win2 := active;

  for i := 1 to cx-2 do

    for j := 1 to cy-2 do

      begin

      SetWindow(win1);

      p0 := GetPixel(i,j);

      p1 := GetPixel(i,j-1);

      p2 := GetPixel(i+1,j);

      p3 := GetPixel(i,j+1);

      p4 := GetPixel(i-1,j);

      for k := 1 to slices do

        if z[k] in [p0..p1,p0..p2,p0..p3,p0..p4,p1..p0,p2..0,p3..0,p4..0] then

          begin

          SetWindow(win2);

          PutPixel(i,j,z[k]);

          end; {if}

      end; {for j}

  SetWindow(win1);

  end; {Procedure ContourPlot}



{*** Main Program ***********************************************************}



BEGIN



  Initialize;

  ShowHelpMenu;

  repeat

    menuchoice := Upcase(ReadKey);

    case menuchoice of

      #00 : MoveImage;

      '?',

      '/' : ShowHelpMenu;

      ',' : WriteWindow;

      '.' : WriteScreen;

      '1',

      '2',

      '3' : begin

            SaveWorkSection(2,cx-woffset-2*wsize,woffset,cx-woffset,2*wsize+woffset);

            ShowWindows(active);

            Delay(500);

            case menuchoice of

              '1' : active := 1;

              '2' : active := 2;

              '3' : active := 3;

              end; {case}

            ShowWindows(active);

            Delay(500);

            RestoreWorkSection(2,cx-woffset-2*wsize,woffset,cx-woffset,2*wsize+woffset);

            SetWindow(active);

            end; {Change Active Window}

      'A' : begin

            EchoSelection(menuchoice,0,false);

            Combine_Images;

            ResetSelection(menuchoice,0,false);

            end; {Combine Images}

      'B' : begin

            EchoSelection(menuchoice,750,false);

            ClearWindow;

            ResetSelection(menuchoice,0,false);

            end; {Clear Window}

      'C' : begin

            EchoSelection(menuchoice,0,false);

            CopyWindow;

            ResetSelection(menuchoice,0,false);

            end; {Copy Window}

      'E' : begin

            EchoSelection(menuchoice,0,false);

            Examines;

            ResetSelection(menuchoice,0,false);

            end; {Examine Image}

      'F' : begin

            EchoSelection(menuchoice,750,false);

            Filters;

            ResetSelection(menuchoice,0,false);

            end; {Filters}

      'G' : begin

            EchoSelection(menuchoice,0,false);

            GroupImages;

            ResetSelection(menuchoice,0,false);

            end; {Group Images}

      'H' : begin

            EchoSelection(menuchoice,500,false);

            Histograms;

            ResetSelection(menuchoice,0,false);

            end; {Histograms}

      'I' : begin

            EchoSelection(menuchoice,750,false);

            invert[active] := not invert[active];

            ResetSelection(menuchoice,0,false);

            end; {Invert}

      'J' : begin

            EchoSelection(menuchoice,0,false);

            ContourPlot;

            ResetSelection(menuchoice,0,false);

            end; {Contour Plot}

      'K' : begin

            EchoSelection(menuchoice,0,false);

            Classify;

            ResetSelection(menuchoice,0,false);

            end; {Classify}

      'L' : begin

            EchoSelection(menuchoice,500,false);

            Select_Image_File(false);

            ResetSelection(menuchoice,0,false);

            end; {Load Image}

      'O' : begin

            EchoSelection(menuchoice,500,false);

            Overlay_Image;

            ResetSelection(menuchoice,0,false);

            end; {Overlay Image}

      'P' : begin

            EchoSelection(menuchoice,500,false);

            Palettes;

            ResetSelection(menuchoice,0,false);

            end; {Palettes}

      'R' : begin

            EchoSelection(menuchoice,0,false);

            RedisplayImage;

            RedisplayOverlays;

            ResetSelection(menuchoice,0,false);

            end; {Redisplay Image}

      'S' : begin

            EchoSelection(menuchoice,0,false);

            Show_All;

            ResetSelection(menuchoice,0,false);

            end; {Show All Windows}

      'W' : begin

            EchoSelection(menuchoice,0,false);

            SelectWindow;

            ResetSelection(menuchoice,0,false);

            end; {Select Window}

      'X' : begin

            EchoSelection(menuchoice,0,false);

            CutImageSection;

            ResetSelection(menuchoice,0,false);

            end; {Cut Image Section}

      'Z' : begin

            EchoSelection(menuchoice,500,false);

            Zoom;

            ResetSelection(menuchoice,0,false);

            end; {Zoom Window}

      end; {case}

    if not nobeep then

      Beep(1000);

    nobeep := false;

  until menuchoice in ['Q'];



  DeInitialize;



END.



