!In this version all subroutines and functions that are used exclusively in this
! computation kernell are included in the section CONTAINS, 
!viz
!  REAL(8) FUNCTION Stimulus (offset)
!  REAL(8) FUNCTION ComposeStimulus (time)
!  SUBROUTINE Calculate_g
!end

subroutine RK4
	use Declare
	IMPLICIT NONE
    
	real(8), parameter	:: Kgs=9.6d-6
	real(8), parameter	:: kT = 300 * 1.3807d-23
	real(8), parameter	:: Yc(0:2) = (/-45d-9, 15d-9, 30d-9/)
    real(8), parameter  :: alphaNH=0.06d9
    real(8), parameter  :: cNH=6.d0   !!3.75d0

	real(8)				:: U_helicotrema	!Volume velocity	[m3/s]
	integer(4)			:: RK4step,iRK4
	real(8), dimension (0:n,4) :: M1234


  if (damping == ZWEIGLINEAR .OR. damping == ZWEIGNONLINEAR) CALL ZweigImpedance
  if (plotStimulus) stimulus0=stimulus(0)

  do RK4step=1,4
	select case (RK4step)
	case (1)
  		F0=Stimulus(0)
		Ytmp = Y
		Vtmp = V
	case (2)
		F0=Stimulus(1)
		Ytmp = Y + Vtmp * half_dt
		Vtmp = V + M1234(0:n,1) / 2.d0
	case (3)
		Ytmp = Y + Vtmp * half_dt
		Vtmp = V + M1234(0:n,2) / 2.d0
	case (4)
		F0=Stimulus(2)
		Ytmp = Y + Vtmp *dt
		Vtmp = V + M1234(0:n,3)
	end select

	CALL Calculate_g

	U_helicotrema = -stapesArea(parameterSet) * Vtmp(0) - SUM (Vtmp(1:n)) * dx * bm_width
	k(n) = -Asq * g(n) - phi * U_helicotrema
	DO i = n-1, 1, -1
		k(i) = -Asq * g(i) + k(i+1) * b(i+1)
	ENDDO
	k(0) = -Asq0 * (p0x * F0 + g(0) + r_Xtr0 * Ytmp(0)) +k(1) * b(1)
	q(0) = -k(0) * b(0)
	DO i = 1, n
		q(i) = (q(i-1) - k(i)) * b(i)
	ENDDO

	M1234(0,RK4step) = m0_RK4 * (q(0) - g(0) - p0x * F0 - r_Xtr0 * Ytmp(0)) * dt
	M1234(1:n,RK4step) = (q(1:n)-g(1:n)) * dt


  enddo !RK4step

	Y = Y + (V + (M1234(0:n,1)+M1234(0:n,2)+M1234(0:n,3)) / 6.d0) * dt
	V = V + (M1234(0:n,1) + 2.d0 * (M1234(0:n,2)+M1234(0:n,3)) + M1234(0:n,4)) / 6.d0

!!END MAIN RK4 ROUTINE
CONTAINS

  REAL(8) FUNCTION Stimulus (offset)
	! offset is numer of half_dt's from current time t, 
	!   e.g. offset = 1 means: time = t + 1 * half_dt
	! if useAudioFile T, function returns sample (2*kk+offset * half_dt) from resampled audiofile
	! audiofile must be resampled to 2 * ComputationalFrequency, so the sample
	! corresponds with t = t + offset * half_dt
	! if not useAudioFile, the function composes stimulus at t = t + offset * half_dt

	USE WaveReadModule

	IMPLICIT NONE 

	INTEGER, INTENT(IN) :: offset
	INTEGER index

	IF (useAudioFile) THEN
		index = 2 * kk + offset
		IF (index < ResampledWaveDataPoints) THEN
			Stimulus = audioFilePressure * DBLE(ResampledWaveData(index))
		ELSE
			Stimulus = 0.
		ENDIF
	ELSE
		Stimulus = ComposeStimulus(t + offset * half_dt)
	ENDIF

  END FUNCTION Stimulus

!!!!!!!!!!!!!!!!

  REAL(8) FUNCTION ComposeStimulus (time)
	! compose stimulus at t=time
    USE  Declare
	IMPLICIT NONE 
	REAL(8), INTENT(IN) :: time
	REAL(8) signal, signal1, signal2, signal3
	REAL(8) onsetWindow
	INTEGER pulseCounter /0/
	REAL(8) t_pulse

	signal = 0.d0
	IF (useSignal1) signal = signal1Pressure * COS(signal1AngularFrequency * time + phi1)
	IF (useSignal2) signal = signal + signal2Pressure * COS(signal2AngularFrequency * time + phi2)
	IF (useSignal3) THEN
		t_pulse = time - DBLE(pulseCounter) * signal3Period
		IF (t_pulse < signal3Duration) THEN
			signal = signal + signal3Pressure
		ELSEIF (t_pulse >= signal3Period) THEN
			pulseCounter = pulseCounter +1
		ENDIF
	ENDIF

	IF (time < onsetDuration) THEN
		onsetWindow = (1.d0 - COS(pi * time / onsetDuration)) / 2.d0
		ComposeStimulus = signal * onsetWindow
	ELSE
		ComposeStimulus = signal
	END IF  

  END FUNCTION ComposeStimulus

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! In linear case, g = ddivm * V +sdivm * y    (ddivm stands for d/m, etc)
! In nonlinear cases, the damping and/or the stiffness depend on velocity and position respectively.
! In the case of Zweig Impedance, the negative damping will be stabilized by a delayed feedback force
  SUBROUTINE Calculate_g
	IMPLICIT NONE
	REAL(8) tmp (n), tmpexp(n)

	REAL(8) absV (n)
	real(8) dtest  !! these 2 are used for the test-output only
	integer itest  !!  of damping and velocity

	g(0) = d_m_factor * Vtmp(0) + s_m_factor * Ytmp(0)

	SELECT CASE (damping)

	CASE (LINEAR)
		g(1:n) = Vtmp(1:n) * ddivm(1:n)

	CASE (NONLINEAR)
		tmp    = NLparmAlpha * Vtmp(1:n)  !!!paramete opgehoogd met factor 10, 22/09/2010
		g(1:n) = ddivm(1:n) * sinh(tmp) / NLparmAlpha
		IF (useActiveDamping) THEN
			WHERE (ABS (NLparmBeta * Vtmp(1:n)) < 1.d2)
				g(1:n) = g(1:n) - ddivm(1:n) * Vtmp(1:n) * 5.d0*gamma / COSH (NLparmBeta * Vtmp(1:n))
			ENDWHERE
		ENDIF
		
	CASE (NONLIN006)
		tmp = NLparmAlpha * Vtmp(1:n)
		IF (useActiveDamping) THEN
			g(1:n) = ddivm(1:n) * 1.d2*(1d0 - (0.999+gamma*0.01)/ COSH (tmp))*Vtmp(1:n)
		ELSE
		    g(1:n) = ddivm(1:n) * 1.d2*(1.d0 - 0.999 / cosh(tmp) )* Vtmp(1:n)
		ENDIF

	CASE (NONLIN007)
		tmp = NLparmAlpha * Vtmp(1:n)
		IF (useActiveDamping) THEN
			g(1:n) = ddivm(1:n) * 1.d2*(1d0 - (0.999+gamma*0.01)*exp(-0.5d0*tmp**2/3.d-1))*Vtmp(1:n)
		ELSE
			g(1:n) = ddivm(1:n) * 1.d2*(1d0 - 0.999*exp(-0.5d0*tmp**2/3.d-1))*Vtmp(1:n)
		ENDIF
    
    CASE (NONLINVDP)
        g(1:n) = Vtmp(1:n)*1.0 *( 1.d18*Ytmp(1:n)**2 - 1) * ddivmvdP(1:n) !!eps =0.05

    CASE (RAYLEIGH)
        g(1:n) = Vtmp(1:n)*1.0 *( 1.d12*Vtmp(1:n)**2 - 1) * ddivmvdP(1:n) !% the Rayleigh version 

    CASE (HOPF)
        g(1:n) = Vtmp(1:n)*1.0 *( 1.d12*Ytmp(1:n)**2) * ddivm(1:n) !* (s_m_factor/d_m_factor)**2  !this does not become negative, but approaches zero for zero velocity
	
    CASE (HOPV)
        g(1:n) = Vtmp(1:n)*1.0 *( 1.d12*Vtmp(1:n)**2) * ddivm(1:n) !* (s_m_factor/d_m_factor)**2  !this does not become negative, but approaches zero for zero velocity
	
	CASE (GOLDSTEIN)
		g(1:n) = Vtmp(1:n) * ddivm(1:n) * (1.d0 + 1.d12/3.5d0 * Vtmp(1:n) ** 2)

	CASE (ZWEIGLINEAR)
		g(1:n) = Vtmp(1:n) * ddivm(1:n)	* ZweigDampingAdaptation + &
						sdivm(1:n) * (Ytmp(1:n) + ZweigRhoAdapted * Yzweig(1:n))

	CASE (ZWEIGNONLINEAR)
		absV = ABS(Vtmp(1:n))
		tmp = ZweigBeta * absV
		g(1:n) = Vtmp(1:n) * ddivm(1:n) * (ZweigDampingAdaptation + ConstantQ * ZweigGamma * absV / (1.d0 + tmp)) + &
						sdivm(1:n) * (Ytmp(1:n) + ZweigRhoAdapted * (1.d0 - tmp / (1.d0 + tmp)) * Yzweig(1:n))
	ENDSELECT

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!Note 24 Jan 2009: In Furst and Goldstein 1982, the breakpoint for nonlinear action is set to 50 dB
!!this can be implemented by relating the nonlinear term to the linear resp at 50 dB
!!for some general parameters this gave Vmax=3.5x10^-6. The value can be checked in the linear version
!!Since in the linear case v=omega x y = 2 pi f x y, the related Ymax=3.5/(2 pi) x 10^(-9)
    if(storeTesting .AND. RK4step == 4) then
        itest=probes(1)                                                      !!!!
        dtest=g(itest)/Vtmp(itest)                                           !xtra, itest must have been defined
        call writeTestOutputFile (Vtmp(itest),tmp(itest),dtest,g(itest))     !xtra
    end if
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!    
    
	SELECT CASE (stiffness)

	CASE (LINEAR)
		g(1:n) = g(1:n) + sdivm(1:n) * Ytmp(1:n)

	CASE (NONLINEAR)
		DO i = 1, n
		    tmpexp(i)= exp(-alphaNH * Ytmp(i))
		    tmp(i)= cNH * tmpexp(i)/(1.d0 + tmpexp(i))**2 
		ENDDO
		g(1:n) = g(1:n) + sdivm(1:n) * ( 1.d0 - tmp) * Ytmp(1:n)

	CASE (GOLDSTEIN)
		g(1:n) = g(1:n) + sdivm(1:n) * Ytmp(1:n) / (1.d0 + 4.d0 * ABS(Ytmp(1:n)*pi/1.75d-9) ** 1.d-1)
	ENDSELECT
 
	RETURN
	END SUBROUTINE Calculate_g

!also contains the zweig-impedance
! Stores the Y-values of the last <delay> samples for each section in a
! circular buffer (so the last Y-value overwrites the Y-value of <delay> samples ago).
! The circular buffers for each section are part of one large array, where each circular
! buffer is stored consecutively. (so it's an array of circular buffers, where each circular
! buffer stores the displacements of the membrane of the last samples of its section)

! Determines <Yzweig> for every section by interpolating between ZweigSample1 and
! ZweigSample2. ZweigSample1 is the first sample before and
! ZweigSample2 is the first sample after the desired value of Y.
! The desired value, Yzweig,  is the value that Y had at exactly <ZweigFactor> times
! the resonantial period of that section ago.

SUBROUTINE ZweigImpedance
	USE declare
	IMPLICIT NONE

	INTEGER YbufferStart	! YbufferStart is first index in Ybuffer of current section
	INTEGER YbufferEnd		! YbufferEnd is last index in Ybuffer of current section

	! Ybuffer = |_____________|____________________|________________________|____....
	!                          ^          ^^      ^				  
	!                        Start    Sample1,2  End
	!                          <------delay(i)---->

	YbufferStart = 1
	DO i = 1, n
		YbufferEnd = YbufferStart + delay(i) - 1
		! Interpolation
		! ZweigSample1 is index of the sample before the desired value
		! ZweigSample2 is index of the sample after the desired value
		! delay_deviation is the time difference (measured in samples) between ZweigSample1 and the desired value
		! Yzweig is calculated by interpolating between the two samples
		Yzweig(i) = Ybuffer(ZweigSample1(i)) - delay_deviation(i) * (Ybuffer(ZweigSample1(i)) - Ybuffer(ZweigSample2(i)))
		! store new value of y (to be used after delay(i) samples)
		Ybuffer(ZweigSample1(i)) = Y(i)
		! shift indices, so next time the next samples will be used
		ZweigSample1(i) = ZweigSample2(i)
		IF (ZweigSample2(i) == YbufferEnd) THEN
			ZweigSample2(i) = YbufferStart
		ELSE
			ZweigSample2(i) = ZweigSample2(i) + 1
		ENDIF
		YbufferStart = YbufferEnd + 1	! values for next segment follow current section directly
	ENDDO

END SUBROUTINE ZweigImpedance

END SUBROUTINE RK4
