extensions [ array ]

globals [
  Year
  Speed ; How fast the sim is running per time step
  Weight_suitable_over_populated
  Weight_suitable_ok
; HATCHLINGS
  Init_hatchling_low
  Init_hatchling_high
  Hatchling_predation_rate_in_edges
  Hatchling_predation_rate_in_core
  Hatchling_predation_rate_outside_habitat
; JUVENILES
  Init_juvenile_low
  Init_juvenile_high
  Juveniles_predation_rate_in_core
  Juvenile_hunting_death_rate
  Juvenile_predation_rate_in_edges
  Fraction_of_Juveniles_migrating
  Juvenile_migration_steps
; ADULTS
  Init_adult_low
  Init_adult_high
  Adult_min_reproduction_rate
  Adult_max_reproduction_rate
  Fraction_of_Adults_migrating
  adult_predation_rate_in_edges
  adult_predation_rate_in_core
  adult_hunting_death_rate
  Territorialitycoefficient
  Max_adults_per_cell
  Max_juveniles_per_cell
  Adult_migration_steps
  Proportion_of_Breeding_Adults

  cell_size
  Years_as_juvenile
  Last_genetic_area
]

patches-own [
 Hatchlings
 Juveniles
 Adults
 adult_genetic_marker
 juvenile_genetic_marker
 in_habitat?
 in_core?
 in_edge?
 Adult_predation_rate
 Juvenile_predation_rate
 Adult_survival_rate
 Juvenile_survival_rate
 core_edge_map
 Cell_weight
 Cell_weight_basic
 MOVEOUT
 MOVEOUT_N
 MOVEOUT_NE
 MOVEOUT_E
 MOVEOUT_SE
 MOVEOUT_S
 MOVEOUT_SW
 MOVEOUT_W
 MOVEOUT_NW
 MOVEIN
 MOVEIN_N
 MOVEIN_NE
 MOVEIN_E
 MOVEIN_SE
 MOVEIN_S
 MOVEIN_SW
 MOVEIN_W
 MOVEIN_NW
 genetic_area
 genetic_marker_in
 tmp
]

to initialize-global-variables
  ; ---------------------------------
  ; Ability to cross the differnet land covers ( 1/0 )
  set Weight_suitable_ok 1                               ; Migration attractiveness in core/edge when under populated
  set Weight_suitable_over_populated 0.18                ; Migration attractiveness in core/edge when over populated
  set cell_size 90                                       ; In meters
  set adult_migration_steps 16                           ; In cells
  set juvenile_migration_steps 8                         ; In cells
  
  ;------------------------------
  ; Animal Biology
  set Year 0

  set Init_hatchling_low 0                               ; Min number of hatchlings initialized in core areas
  set Init_hatchling_high 0                              ; Max number of hatchlings initialized in core areas
  set Hatchling_predation_rate_in_edges 0.81             ; Percent of hatchlings in edges that are lost to predators
  set Hatchling_predation_rate_in_core 0.81              ; Percent of hatchlings in core that are lost to predators
  set Hatchling_predation_rate_outside_habitat 1         ; Percent of hatchlings everywhere else that are lost to predators

  set Init_juvenile_low 5                                ; Min number of juveniles initialized in core areas
  set Init_juvenile_high 10                              ; Max number of juveniles initialized in core areas
  set Juvenile_predation_rate_in_edges 0.2               ; Percentage of juveniles die each year in edges
  set Juveniles_predation_rate_in_core 0.1               ; Percentage of juveniles die each year in core
  set Juvenile_hunting_death_rate 0                      ; Percentage of juveniles killed by predators each year
  set Fraction_of_Juveniles_migrating 0.05               ; Percentage of juveniles that move each year even if not overpopulated
  set Years_as_juvenile 20                               ; years

  set Init_adult_low 1                                   ; Min number of adults initialized in core areas
  set Init_adult_high 3                                  ; Max number of adults initialized in core areas
  set Adult_min_reproduction_rate 4                      ; Min number of hatchlings per adult per year
  set Adult_max_reproduction_rate 8                      ; Max number of hatchlings per adult per year
  set Adult_predation_rate_in_edges 0.03                 ; Percent of adults lost each year to predation in the edge habitat
  set Adult_predation_rate_in_core 0.03                  ; Percent of adults lost each year to predation in the core habitat
  set Adult_hunting_death_rate 0.0                       ; Percent of adults lost each year to hunting
  set Fraction_of_Adults_migrating 0.05                  ; Percentage of adults that move each year even if not overpopulated
  set Proportion_of_Breeding_Adults 1                    ; % of adults that breed.  1 means all animals (no distinction between M and F)

  let Adults_per_homerange 3                             ; Adults per ha
  let Juveniles_per_homerange 12                         ; Adults per ha
  let Homerangesize 1                                    ; In hectares
  set Max_adults_per_cell Adults_per_homerange * ( (cell_size * cell_size / 10000) / Homerangesize )
  set Max_juveniles_per_cell Juveniles_per_homerange * ( (cell_size * cell_size / 10000) / Homerangesize )
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
to initialize-benning
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  initialize-global-variables
  initialize-maps
  initialize-patches
  initialize-edge-patches ; Ring the map with a no-mans-land
  setup-plot
  show-map
end

to initialize-benning-2
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  initialize-global-variables
  initialize-maps-2
  initialize-patches
  initialize-edge-patches ; Ring the map with a no-mans-land
  setup-plot
  show-map
end

to initialize-circles
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  initialize-global-variables
  initialize-circle-maps
  initialize-patches
  initialize-edge-patches ; Ring the map with a no-mans-land
  setup-plot
  show-map
end

to initialize-maps
  ;; Read these from maps
  read-arc-ascii-map "maps/habitat_90.asc"    ask patches [ set core_edge_map tmp ]        ; 1,2,3,4 habitat core and edge categories
  read-arc-ascii-map "maps/weight_90.asc"     ask patches [ set Cell_weight_basic tmp ]    ; Attractiveness map  (0-1)
  read-arc-ascii-map "maps/predation_90.asc"  ask patches [ set adult_predation_rate tmp ] ; Predation rate  (0-1)
  read-arc-ascii-map "maps/genetic_90.asc"    ask patches [ set genetic_area tmp ]         ; Predation rate  (0-1)
end

to initialize-maps-2
  ;; Read these from maps
  read-arc-ascii-map "maps/new_habitat_90.asc"    ask patches [ set core_edge_map tmp ]        ; 1,2,3,4 habitat core and edge categories
  read-arc-ascii-map "maps/weight_90.asc"     ask patches [ set Cell_weight_basic tmp ]    ; Attractiveness map  (0-1)
  read-arc-ascii-map "maps/new_predation_90.asc"  ask patches [ set adult_predation_rate tmp ] ; Predation rate  (0-1)
  read-arc-ascii-map "maps/new_genetic_90.asc"    ask patches [ set genetic_area tmp ]         ; Predation rate  (0-1)
end

to initialize-circle-maps
  set genetic_area_to_map 1
  ;; set background
  ask patches [
    set Cell_weight_basic 1
    set Adult_predation_rate .2
    set Juvenile_predation_rate 1
    set core_edge_map 10
    set genetic_area 0
  ]
  
  let middle-x ( max-pxcor + min-pxcor ) / 2
  let middle-y ( max-pycor + min-pycor ) / 2
  
  let center-x middle-x - (circle-1-size + separation) / 2
  ask patches with [ ( distancexy center-x middle-y ) <= ( circle-1-size / 2 ) ]
  [ set core_edge_map 2  set genetic_area 1 ]
  
  set center-x middle-x + (circle-2-size + separation) / 2
  ask patches with [ ( distancexy center-x middle-y ) <= ( circle-2-size / 2 ) ]
  [ set core_edge_map 2  set genetic_area 2 ]
end

to initialize-patches
  set Last_genetic_area max [ genetic_area ] of patches
  
  ask patches [
    set adult_genetic_marker array:from-list n-values (Last_genetic_area + 1) [0]
    set juvenile_genetic_marker array:from-list n-values (Last_genetic_area + 1) [0]
    array:set adult_genetic_marker genetic_area 1.0
    array:set juvenile_genetic_marker genetic_area 1.0
        
    let marker 1
    while [ marker <= Last_genetic_area ]
    [
      set marker marker + 1
    ]

    IFELSE ( core_edge_map = 1 OR core_edge_map = 2 )
    [ 
      set Adults     Init_adult_low     + RANDOM-FLOAT ( Init_adult_high     - Init_adult_low )
      set Juveniles  Init_juvenile_low  + RANDOM-FLOAT ( Init_juvenile_high  - Init_juvenile_low )
      set Hatchlings Init_hatchling_low + RANDOM-FLOAT ( Init_hatchling_high - Init_hatchling_low )
    ]
    [ 
      set Adults 0
      set Juveniles 0
      set Hatchlings 0
    ]

    IFELSE ( core_edge_map <= 4 AND core_edge_map > 0) [ set in_habitat? TRUE ] [ set in_habitat? FALSE ]
    IFELSE ( core_edge_map = 1 OR core_edge_map = 3 ) [ set in_edge? TRUE ] [ set in_edge? FALSE ]
    IFELSE ( core_edge_map = 2 OR core_edge_map = 4 ) [ set in_core? TRUE ] [ set in_core? FALSE ]
    
    IF ( in_core? )
    [ 
      set adult_predation_rate adult_predation_rate_in_core       ; pred rate in core
      set juvenile_predation_rate  Juveniles_predation_rate_in_core
    ]
    IF ( in_edge? )
    [ 
      set adult_predation_rate adult_predation_rate_in_edges      ; pred rate in edge
      set juvenile_predation_rate  juvenile_predation_rate_in_edges 
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to initialize-edge-patches
  let x min-pxcor
  while [ x <= max-pxcor ]
  [
    ask patch x max-pycor [ zero-out ]
    ask patch x min-pycor [ zero-out ]
    set x x + 1
  ]
  let y min-pycor
  while [ y <= max-pycor ]
  [
    ask patch max-pxcor y [ zero-out ]
    ask patch min-pxcor y [ zero-out ]
    set y y + 1
  ]
end

to zero-out             ; This puts a hard edge on the area, which avoids the need for special code for edges
  set core_edge_map 0
  set Cell_weight_basic 0
  set Adult_predation_rate 1.0
  set genetic_area 0
  set Adults 0
  set Juveniles 0
  set adult_genetic_marker array:from-list n-values (Last_genetic_area + 1) [0]
  set juvenile_genetic_marker array:from-list n-values (Last_genetic_area + 1) [0]
  set in_habitat? FALSE
  set in_core? FALSE
  set in_edge? FALSE
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to step
  set Year Year + 1
  let start-time get-time
  
  ;; Hunt hatchlings
  do-hatchling-hunting-predation
  
  ;; Hatchlings turn to juveniles and some juveniles mature to adults
  do-growth
  
  ;; Adults reproduce
  do-reproduction
  
  ;; Migrate
  do_migration
  
  ;;  Need to kill off all that are not in habitat to reset the year and move this to the migration step
  do-deaths  
  
  ;; Record results
  do-plot
  show-map
  
  set Speed get-time - start-time
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-deaths
  ; kill off all remaining tortoises not in habitat
  ask patches with [ NOT in_habitat? ] 
  [
    set Juveniles 0
    set Adults 0
    foreach n-values (Last_genetic_area + 1) [?] 
    [
      array:set adult_genetic_marker ? 0
      array:set juvenile_genetic_marker ? 0
    ]
  ]
  ; kill off all extra tortoises in habitats
  ask patches with [ in_habitat? ]
  [
    if Adults > Max_adults_per_cell [ set Adults Max_adults_per_cell ]
    if Juveniles > Max_juveniles_per_cell [ set Juveniles Max_juveniles_per_cell ]
  ]
end

to do-hatchling-hunting-predation
  ask patches [
    ; ---------------------------
    ;  Hatchlings predation and hunting death rates
    
    let hatchling_predation_rate Hatchling_predation_rate_outside_habitat
    IF ( in_core? )
      [ set hatchling_predation_rate Hatchling_predation_rate_in_core ]
    IF ( in_edge? )
      [ set hatchling_predation_rate Hatchling_predation_rate_in_edges ]

    set Hatchlings Hatchlings - ( hatchling_predation_rate  * Hatchlings )
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do_migration
  ask patches
  [
    ;; Adults migrate
    let Adult_death_rate Adult_predation_rate  + Adult_hunting_death_rate
    ifelse (1 <= Adult_death_rate) 
    [ set Adult_survival_rate 0 ]
    [ set Adult_survival_rate EXP(LN(1 - Adult_death_rate ) / Adult_migration_steps) ]
;    if SELF  = PATCH 4 3 [ print Adult_survival_rate ]
  ]
  
  let s 1
  while [ s <= adult_migration_steps ]
  [ 
    do-adult-migration 
    ask patches [ set Adults Adults * Adult_survival_rate ]
    set s s + 1
  ]

  ;; Juveniles migrate
  ask patches 
  [
    let Juvenile_death_rate juvenile_predation_rate  + Juvenile_hunting_death_rate
    ifelse (1 <= Juvenile_death_rate) 
    [ set Juvenile_survival_rate 0 ]
    [ set Juvenile_survival_rate EXP(LN(1 - juvenile_death_rate) / juvenile_migration_steps) ]
  ]
  
  set s 1
  while [ s <= juvenile_migration_steps ]
  [
    do-juvenile-migration
    ask patches [ set Juveniles Juveniles * Juvenile_survival_rate ]
    set s s + 1
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-reproduction
  ; ---------------------------
  ;  Adult reproduction
  ask patches
  [
    let Adult_reproduction_rate Adult_min_reproduction_rate + RANDOM-FLOAT ( Adult_max_reproduction_rate - Adult_max_reproduction_rate )

    IF ( in_habitat? )
    [ set hatchlings hatchlings + ( Adults * Adult_Reproduction_rate * Proportion_of_Breeding_Adults ) ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-adult-migration
  ; %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  ; pre-calculate cell weights, which neighboring cells will use
  ask patches 
  [
    IFELSE ( in_habitat? )
    [
      IFELSE ( Adults >= Max_adults_per_cell )
      [ set Cell_weight Weight_suitable_over_populated ]
      [ set Cell_weight Weight_suitable_ok ]
    ]
    [ set Cell_weight Cell_weight_basic ]
  ]

  ; %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  ; calculate the number of animals that must move out
  ask patches
  [
    IFELSE ( in_habitat? )
    [
      IFELSE ( Adults <= Max_adults_per_cell )
      [ set MOVEOUT Fraction_of_Adults_migrating * Adults ]
      [ set MOVEOUT ( Adults - Max_adults_per_cell ) ]
    ]
    ; outside habitat
    [ set MOVEOUT Adults ]
  ]

  calculate-move-out
  calculate-move-in
  
  ; Accept the animals moving in, remove animals moving out, and update the adult genetic_marker
  ask patches 
  [
    set Adults Adults - MOVEOUT    ; Outgoing animals must depart before incoming animals come in to keep the genetics over-moving

    let i 0                        ; Combine genetics with animals moving in
    while [ i <= Last_genetic_area ]
    [
      ifelse ( Adults + MOVEIN ) > 0
      [
        array:set adult_genetic_marker i ( (Adults * array:item adult_genetic_marker i) + (MOVEIN * array:item genetic_marker_in i) ) / ( Adults + MOVEIN )
      ]
      [ array:set adult_genetic_marker i 0 ]
      set i i + 1
    ]
   
    set Adults Adults + MOVEIN     ; Add the incoming animals
  ]
end

to do-juvenile-migration
  ; %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  ; pre-calculate cell weights, which neighboring cells will use
  ask patches 
  [
    IFELSE ( in_habitat? )
    [
      IFELSE ( Juveniles >= Max_juveniles_per_cell )
      [ set Cell_weight Weight_suitable_over_populated ]
      [ set Cell_weight Weight_suitable_ok ]
    ]
    [ set Cell_weight Cell_weight_basic ]
  ]

  ; %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  ; calculate the number of animals that must move out
  ask patches
  [
    IFELSE ( in_habitat? )
    [
      IFELSE ( Juveniles <= Max_juveniles_per_cell )
      [ set MOVEOUT Fraction_of_Juveniles_migrating * Juveniles ]
      [ set MOVEOUT ( Juveniles - Max_juveniles_per_cell ) ]
    ]
    ; outside habitat
    [ set MOVEOUT Juveniles ]
  ]

  calculate-move-out
  calculate-move-in
  
  ; Accept the animals moving in, remove animals moving out, and update the adult genetic_marker
  ask patches 
  [
    set Juveniles Juveniles - MOVEOUT    ; Outgoing animals must depart before incoming animals come in to keep the genetics over-moving

    let i 0                        ; Combine genetics with animals moving in
    while [ i <= Last_genetic_area ]
    [
      ifelse ( Juveniles + MOVEIN ) > 0
      [
        array:set Juvenile_genetic_marker i ( (Juveniles * array:item Juvenile_genetic_marker i) + (MOVEIN * array:item genetic_marker_in i) ) / ( Juveniles + MOVEIN )
      ]
      [ array:set Juvenile_genetic_marker i 0 ]
      set i i + 1
    ]
   
    set Juveniles Juveniles + MOVEIN     ; Add the incoming animals
  ]
end

to calculate-move-out
  ; distribute the animals that are moving out to neighboring cells
  ask patches
  [
    ; %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    ; Calculate the migration out of patches
    ;----------------------------------------
    ; Find the total cell weight in 8 neighboring cell
 
    let weight_N  [ Cell_weight ] of patch-at-heading-and-distance   0 1.4
    let weight_NE [ Cell_weight ] of patch-at-heading-and-distance  45 1.4
    let weight_E  [ Cell_weight ] of patch-at-heading-and-distance  90 1.4
    let weight_SE [ Cell_weight ] of patch-at-heading-and-distance 135 1.4
    let weight_S  [ Cell_weight ] of patch-at-heading-and-distance 180 1.4
    let weight_SW [ Cell_weight ] of patch-at-heading-and-distance 225 1.4
    let weight_W  [ Cell_weight ] of patch-at-heading-and-distance 270 1.4
    let weight_NW [ Cell_weight ] of patch-at-heading-and-distance 315 1.4

    let Total_Weight weight_N + weight_NE + weight_E + weight_SE + weight_S + weight_SW + weight_W + weight_NW

    IFELSE ( Total_Weight > 0 ) 
    [
      set MOVEOUT_N  MOVEOUT * ( weight_N  / Total_Weight )
      set MOVEOUT_NE MOVEOUT * ( weight_NE / Total_Weight )
      set MOVEOUT_E  MOVEOUT * ( weight_E  / Total_Weight )
      set MOVEOUT_SE MOVEOUT * ( weight_SE / Total_Weight )
      set MOVEOUT_S  MOVEOUT * ( weight_S  / Total_Weight )
      set MOVEOUT_SW MOVEOUT * ( weight_SW / Total_Weight )
      set MOVEOUT_W  MOVEOUT * ( weight_W  / Total_Weight )
      set MOVEOUT_NW MOVEOUT * ( weight_NW / Total_Weight )
    ]
    [
      set MOVEOUT_N  0
      set MOVEOUT_NE 0
      set MOVEOUT_E  0
      set MOVEOUT_SE 0
      set MOVEOUT_S  0
      set MOVEOUT_SW 0
      set MOVEOUT_W  0
      set MOVEOUT_NW 0
    ]
  ]
end

to calculate-move-in
  ; Accept the animals moving in and remove animals moving out
  ask patches 
  [
    set MOVEIN_N  [ MOVEOUT_S  ] of patch-at  0  1
    set MOVEIN_NE [ MOVEOUT_SW ] of patch-at  1  1
    set MOVEIN_E  [ MOVEOUT_W  ] of patch-at  1  0
    set MOVEIN_SE [ MOVEOUT_NW ] of patch-at  1 -1
    set MOVEIN_S  [ MOVEOUT_N  ] of patch-at  0 -1
    set MOVEIN_SW [ MOVEOUT_NE ] of patch-at -1 -1
    set MOVEIN_W  [ MOVEOUT_E  ] of patch-at -1  0
    set MOVEIN_NW [ MOVEOUT_SE ] of patch-at -1  1
    set MOVEIN MOVEIN_N + MOVEIN_NE + MOVEIN_E + MOVEIN_SE + MOVEIN_S + MOVEIN_SW + MOVEIN_W  + MOVEIN_NW
  ]

  ask patches
  [
    set genetic_marker_in array:from-list n-values (Last_genetic_area + 1) [0]
    if ( MOVEIN > 0 )
    [
      let i 0
      while [ i <= Last_genetic_area ]
      [
        array:set genetic_marker_in i
          ( MOVEIN_N  / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at  0  1 ) i +
          ( MOVEIN_NE / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at  1  1 ) i +
          ( MOVEIN_E  / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at  1  0 ) i +
          ( MOVEIN_SE / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at  1 -1 ) i +
          ( MOVEIN_S  / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at  0 -1 ) i +
          ( MOVEIN_SW / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at -1 -1 ) i +
          ( MOVEIN_W  / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at -1  0 ) i +
          ( MOVEIN_NW / MOVEIN ) * array:item ( [ adult_genetic_marker ] of patch-at -1  1 ) i
        set i i + 1
      ]
    ]
  ]
end



;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-growth    ;;;ADD AGE TO THIS - THE LOWER THE AVG JUV AGE, THE FEWER NEW ADULTS
  ask patches
  [
    ; Identify Juveniles -> Adults
    let new_adults Juveniles / Years_as_juvenile
    if ( Adults + new_adults ) > 0
    [
      ; Combine genetics for new adults with adults
      let i 0
      while [ i <= Last_genetic_area ]
      [
        array:set Adult_genetic_marker i 
           ( (Adults * array:item Adult_genetic_marker i) + (new_adults * array:item Juvenile_genetic_marker i) ) / ( Adults + new_adults )
        set i i + 1
      ]
      ; Update new Adults and Juveniles
      set Adults Adults + new_adults
      set Juveniles Juveniles - new_adults
    ]

    if ( Juveniles + Hatchlings ) > 0
    [
      ; Combine Hatchling genetics with juveniles (hatchling genetics come from adults)
      let i 0
      while [ i <= Last_genetic_area ]
      [
        array:set Juvenile_genetic_marker i 
           ( (Juveniles * array:item Juvenile_genetic_marker i) + (Hatchlings * array:item Adult_genetic_marker i) ) / ( Juveniles + Hatchlings )
        set i i + 1
      ]
      ; Update new Juveniles and zero out hatchlings
      set Juveniles Juveniles + Hatchlings
      set Hatchlings 0
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to print-report  
  print "Genetic Markers"
  let ga 1
  while [ ga <= Last_genetic_area ]
  [
    let gm 1
    type ga
    while [ gm <= Last_genetic_area ]
    [
      type ","
      type average-genetics ga gm
      set gm gm + 1
    ]
    print ""
    set ga ga + 1
  ]
  
  type "Num patches:"
  set ga 1
  while [ ga <= Last_genetic_area ]
  [
    type count patches with [ genetic_area = ga ]
    type ","
    set ga ga + 1
  ]
  print ""
  
  type "Adults:"
  set ga 1
  while [ ga <= Last_genetic_area ]
  [
    type sum [ adults ] of patches with [ genetic_area = ga ]
    type ","
    set ga ga + 1
  ]
  print ""
end

to-report average-genetics [ ga gm ]
   let tot count patches with [ genetic_area = ga ]
   if tot > 0
     [ report ( sum [ array:item adult_genetic_marker gm ] of patches with [ genetic_area = ga ] ) / ( tot ) ]
   report 0
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Plot procedures
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-plot
  set-current-plot "Population"
  set-current-plot-pen "Hatchlings/10" plot ( ( sum [ Hatchlings ] of patches ) / 10 )
  set-current-plot-pen "Juveniles" plot ( sum [  Juveniles  ] of patches )
  set-current-plot-pen "Adults" plot ( sum [ Adults ] of patches )
end

to setup-plot ;; set up plotting
  set-current-plot "Population"
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Utility procedures
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to-report GRAPH [ in coors ]
 let ncoors length coors
 let atcoor 0
 let coor item atcoor coors
 let x0 item 0 coor
 let y0 item 1 coor

 while [ atcoor < ncoors ]
 [
 set coor item atcoor coors
 let x1 item 0 coor
 let y1 item 1 coor
 ; type x0 type "," type y0 type " " type x1 type "," print y1
 if ( in <= x0 ) [ report y0 ]
 if ( in > x0 AND in <= x1 )
 [
  report y0 + ( y1 - y0 ) * ( ( in - x0) / ( x1 - x0 ) )
 ]
 set x0 x1
 set y0 y1
 set atcoor atcoor + 1
 ]
 report y0
end

to-report get-time
  let time date-and-time
  report( read-from-string substring time 0 2 ) * 3600 + 
        ( read-from-string substring time 3 5 ) * 60 + 
        ( read-from-string substring time 6 12 )
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Read maps
to read-arc-ascii-map [ map-name ]
 type "Reading map: "
 print map-name
 file-open map-name
 let line file-read-characters 6
 let map-cols file-read ; print map-cols
 set line file-read-characters 6
 let map-rows file-read ; print map-rows
 set line file-read-line ; print line
 set line file-read-line ; print line
 set line file-read-line ; print line
 set line file-read-line ; print line
 let skip map-north-offset     ; skip the first lines as selected by user
 let map-row 0
 while [ skip > 0 ] [ 
 set line file-read-line 
 set map-row map-row + 1
 set skip skip - 1 
 ]  ;
 let row max-pycor
 let val 0
 while [ map-row < map-rows AND row >= min-pycor ]
 [
 type "."
 let map-col 0
 set skip map-west-offset     ; skip the first lines as selected by user
 while [ skip > 0 ] [
  set val file-read
  set skip skip - 1
  set map-col map-col + 1
 ]
 let col min-pxcor
 while [ map-col < map-cols ]
 [
  set val file-read
  if (row >= min-pycor) AND (col <= max-pxcor)
  [
    ask patch col row [ set tmp val ]
  ]
  set map-col map-col + 1
  set col col + 1
 ]
 set map-row map-row + 1
 set row row - 1
 ]
 file-close
 print ""
end

to show-core-edge-map
 ask patches 
 [
 if (core_edge_map = 1) [ set pcolor approximate-rgb 255 0 0 ]
 if (core_edge_map = 2) [ set pcolor approximate-rgb 255 99 99 ]
 if (core_edge_map = 3) [ set pcolor approximate-rgb 0 255 0 ]
 if (core_edge_map = 4) [ set pcolor approximate-rgb 99 255 99 ]
 ]
end  

to show-edge-map
 ask patches 
 [
 if (core_edge_map = 1) [ set pcolor approximate-rgb 255 0 0 ]
 ]
end  

to show-map
  if ( map-display = "core_edge_map" )  [ show-core-edge-map ]
  if ( map-display = "Hatchlings" ) [ ask patches [ set tmp Hatchlings ] show-int-map ]
  if ( map-display = "Juveniles" )  [ ask patches [ set tmp Juveniles ] show-int-map ]
  if ( map-display = "Adults" )     [ ask patches [ set tmp Adults ] show-int-map ]
  if ( map-display = "Weight" )     [ ask patches [ set tmp Cell_weight_basic ] show-int-map ]
  if ( map-display = "Predation" )  [ ask patches [ set tmp Adult_predation_rate if ( tmp > 1) [ set tmp 1] ] show-int-map ]
  if ( map-display = "Genetic Areas")[ ask patches [ set tmp genetic_area ] show-int-map ]
  if ( map-display = "Genetic Diffusion" )    [ ask patches [ set tmp Adults * array:item adult_genetic_marker Genetic_area_to_map ] show-int-map ]
  if ( map-display = "Genetic Index" )    [ ask patches 
  [ 
    set tmp 0
    let i 1
    let tot 0
    while [ i <= Last_genetic_area ]
    [
      set tot tot + (adults * array:item adult_genetic_marker i)
      set i i + 1
    ]
    set i 1
    if ( tot > 0)
    [ 
    while [ i <= Last_genetic_area ]
    [
      let gen_equiv adults * (array:item adult_genetic_marker i)
      set tmp tmp + MIN( list 
        ( ( gen_equiv * ( tot / Last_genetic_area ) ) / tot )
        ( ( ( tot - gen_equiv ) / ( tot - ( tot / Last_genetic_area))))) /
        Last_genetic_area
      set i i + 1
    ]
    ]
  ] show-int-map ]
end

to show-int-map
  let the_max 0
  set the_max max [ tmp ] of patches
  ifelse ( the_max != 0 )
  [ ask patches 
    [ set pcolor (9.99 - 9.99 * ( tmp / the_max ) ) ]
  ]
  [ ask patches [ set pcolor 0 ] ]
;  show-edge-map   
end

to show-log-map
  let the_max 0
  set the_max max [ tmp ] of patches
  ifelse ( the_max != 0 )
  [ ask patches 
    [ 
      ifelse  tmp = 0 
      [ set pcolor 0 ]
      [
        let c ln ( 22026 * ( tmp / the_max ))
        if c < 0 [ set c 0 ]
        set pcolor c
      ]
    ] 
  ]
  [ ask patches [ set pcolor 0 ] ]
;  show-edge-map   
end
@#$#@#$#@
GRAPHICS-WINDOW
303
10
790
518
26
26
9.0
1
10
1
1
1
0
1
1
1
-26
26
-26
26
0
0
1
ticks
30.0

SLIDER
246
54
279
181
map-north-offset
map-north-offset
0
60
0
10
1
NIL
VERTICAL

SLIDER
200
54
233
180
map-west-offset
map-west-offset
0
60
0
10
1
NIL
VERTICAL

BUTTON
181
184
298
217
NIL
initialize-benning\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
4
424
252
469
map-display
map-display
"Hatchlings" "Juveniles" "Adults" "Sub_adults" "core_edge_map" "Weight" "Predation" "Genetic Areas" "Genetic Diffusion" "Genetic Index"
2

BUTTON
193
472
290
517
NIL
show-map
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
4
283
67
316
NIL
step
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
71
283
134
316
run steps
let i 0\nwhile [ i < steps ] [ step   set i i + 1 ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
133
320
183
365
Year
Year
0
1
11

PLOT
306
522
648
672
Population
Time
Pop
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Adults" 1.0 0 -16777216 true "" ""
"Juveniles" 1.0 0 -2674135 true "" ""
"Hatchlings/10" 1.0 0 -10899396 true "" ""

SLIDER
4
474
182
507
Genetic_area_to_map
Genetic_area_to_map
0
Last_genetic_area
1
1
1
NIL
HORIZONTAL

BUTTON
5
184
134
217
NIL
initialize-circles
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
55
43
180
circle-1-size
circle-1-size
0
50
8
2
1
cells
VERTICAL

SLIDER
49
54
82
182
circle-2-size
circle-2-size
0
50
8
2
1
cells
VERTICAL

SLIDER
89
54
122
181
separation
separation
0
50
1
1
1
cells
VERTICAL

BUTTON
3
560
111
593
NIL
print-report
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
160
283
300
316
steps
steps
1
100
20
1
1
steps
HORIZONTAL

MONITOR
5
320
124
365
Seconds per year
Speed
17
1
11

TEXTBOX
6
5
233
23
Step 1: Edit variables in the Procedures
11
0.0
0

TEXTBOX
5
38
242
56
Step 2: Initialize circles or \"Benning\" data
11
0.0
1

TEXTBOX
4
266
288
284
Step 3: Choose a number of steps to run and run
11
0.0
1

TEXTBOX
3
391
294
447
At any time, choose a map to display.  Will display on clicking \"show_map\" and at the end of each step.
11
0.0
1

TEXTBOX
2
530
299
556
Step 4: get distribution of genetics listed in the \"Command Center\"
11
0.0
1

BUTTON
181
219
298
252
NIL
initialize-benning-2
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This is the new FRAGGLE model parameterized for the Gopher Tortoise and associated with maps that represent an area to the east of Fort Benning, GA.

FRAGGLE was originally developed by J.P. Aurambout using Stella, with the resulting equations imported into the SME (Spatial Modeling Environment).  The Stella/SME environment is strictly grid cell based (NetLogo patches).  It does not make use of NetLogo agents (turtles), remaining strictily a cellular-automaton model.

## HOW DOES IT WORK?

Adults give birth to hatchlings that turn into juveniles that turn into adults.  This is accomplished with populations in each cell that diffuse into surrounding cells based on rules of the model. 

## HOW TO USE IT

Step 1: Edit the initialization procedures to set various variables that represent the animal of interest.  This is equivalent to editing the SME configuration file.  
Step 2: Click the "Initialize" button and then the "Run" button andlet the model run for as long as desired  
Step 3: Select a variable to display in the map.  This map will be shown each time step.  You can also click "show-map" to display the chose map immediately.

## INPUT MAPS

Three arc-ascii format maps are read into this model upon initialization.  They are found in the folder/directory called "maps" where this model exists.  They are listed below.

maps/habitat.asc  
This map was also developed by William Meyer, ERDC-CERL, based on published information and consultation with Gopher Tortoise experts.

This is the same as the land use map with the following additional categories  
1 - habitat ; Animals are initialized here

These categories are not used:  
2 - habitat core; Animals are initialized here  
3 - habitat edge  
4 - habitat core

maps/genetic.asc  
Gives a number for each grid cell that identifies it with a unique genetic area.

maps/predation.asc  
The annual probabiliy of predation to migrators for each cell.  (0 - 1.0)

maps/weight.asc  
The attractiveness of each cell to migrators (0 - 1.0)

## CREDITS AND REFERENCES

FRAGGLE was originally created and is documented in a dissertation and the literature by J.P. Aurambout.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
