extensions [ gis array ] ; Allows us to use arrays to store information during the simulation

globals [           ; Variables available to all objects
  total-habitats    ; patches are assigned a unique ID
  step              ; the step
  travel-success    ; keep track of how many travelers successfully traveled from one patch to another
  good-patches      ; list of all patches in good habitat 
]

patches-own [
  quality          ; -1 poor, 0 OK, 1 great
  tagged           ; used in the process of finding contiguous areas
  habitat-ID       ; pointer to the habitat that groups this patch with others
  lethality        ; probability of a traveller dying
  energy-cost      ; cost to traveler crossing
  crossed          ; number of times the patch was part of a successful crossing
]

breed [ travelers traveler ]

travelers-own [
  starting-habitat       ; last habitat that I remained in for at least a year
  current-habitat-home    ; the habitat that I've been occupying most recently
  total-steps             ; Number of migration steps
  starting-patch          ; Used to track where the traveler started each step
  patch-list              ; list of patches crossed in a migration attempt
  energy                  ; travel energy
]

breed [ habitats habitat ]

habitats-own [
  ID                ; my habitat ID
  extent            ; how many contiguous patches are associated
  edge-patches      ; the patches on my edge
]

to initialize
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks

  ; Establish coordinate transformation between GIS maps and Netlogo patches
  let dataset gis:load-dataset "maps/habitat.asc"                                            ; Read the boundary map
  gis:set-transformation (gis:envelope-of dataset) (list min-pxcor max-pxcor min-pycor max-pycor)  ; Establish transformation
  ; Print a warning to the user to ensure that the display area matches the map
  type "Make sure that the display area is setup for "
  type gis:width-of dataset
  type " columns and "
  type gis:height-of dataset
  print " rows."
  gis:apply-raster dataset quality

  ; Read in remaining raster maps into patch variables
  gis:apply-raster gis:load-dataset "maps/energy-cost.asc" energy-cost
  gis:apply-raster gis:load-dataset "maps/lethality.asc"   lethality
     
  ask patches [ set-color ]

  ; Identify how many contiguous cells there are in each cell's contiguous habitat
  set-extents
  
  ; Initialize habitat objects to keep track of migration success (and prevent cross-habitat breeding)
  initialize-habitats
    
  ; get list of all good patches - used to set new starting points
  set good-patches patches with [ quality = 1 ]
    
  initialize-travelers
end

to set-color
  ifelse ( -1 = quality )
  [ set pcolor 18 ]
  [
    ifelse ( 0 = quality )
    [ set pcolor 48 ]
    [ 
      if ( 1 = quality)
      [ set pcolor 68 ]
    ]
  ]
end

to initialize-habitats
  ; Initialize arrays for travel, migration, and geneflow success ID
  set travel-success    array:from-list n-values ( total-habitats * total-habitats ) [ 0 ]
  
  ask habitats
  [
    let tmp self
    set edge-patches patches with [ habitat-ID = tmp ]
    set edge-patches edge-patches with [ count neighbors with [ habitat-ID = tmp ] < 8 ]
    ask edge-patches [ set pcolor green ]
  ]
end
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; These three procedures clump like-cells together into numbered habitats                                   ;;
;;                                                                                                           ;;
to set-extents                                                                                               ;;
  print "Finding habitats"                                                                                   ;;
  ask patches [ set tagged FALSE]
  set total-habitats 0                                                                                       ;;
  let y min-pycor                                                                                            ;;
  while [ y <= max-pycor ]                                                                                   ;;
  [                                                                                                          ;;
    let x min-pxcor                                                                                          ;;
    while [ x <= max-pxcor ]                                                                                 ;;
    [                                                                                                        ;;
      ask patch x y [ find-extent ]                                                                          ;;
      set x x + 1                                                                                            ;;
    ]                                                                                                        ;;
    set y y + 1                                                                                              ;;
  ]                                                                                                          ;;
  ask patches [ if habitat-id = 0 [ set habitat-ID NOBODY ]]                                                 ;;                                                                                                         ;;
end                                                                                                          ;;
                                                                                                             ;;
to find-extent                                                                                               ;;
  if ( habitat-ID != 0 OR quality != 1 ) [ stop ]                                                            ;;
  set total-habitats total-habitats + 1                                                                      ;;
  sprout-habitats 1 [ set ID total-habitats ]                                                                ;;
  let the-habitat one-of habitats with [ ID = total-habitats ]                                               ;;
  tag-neighbors SELF quality                                                                                 ;;
  let the-count count patches with [ tagged = TRUE ]                                                         ;;
  ask the-habitat [ set extent the-count ]                                                                   ;;
  ask the-habitat [ set plabel ID set plabel-color black ]                                                   ;;
  ask patches with [ tagged = TRUE ] [ set habitat-ID the-habitat    set tagged FALSE ]                      ;;
end                                                                                                          ;;
                                                                                                             ;;
to tag-neighbors [ start-patch qual ]                                                                        ;;
  let tag-list (list start-patch)                                                                            ;;
  let at-patch 0                                                                                             ;;
  ask start-patch [ set tagged TRUE ]                                                                        ;;
  while [ ( length tag-list ) > 0 ]                                                                          ;;
  [                                                                                                          ;;
    let cur-patch first tag-list                                                                             ;;
    set tag-list butfirst tag-list                                                                           ;;
    ask cur-patch                                                                                            ;;
    [                                                                                                        ;;
       set tag-list lput ( neighbors4 with [ habitat-ID = 0 AND tagged = FALSE AND quality = qual ] ) tag-list
       ask neighbors4 with [ habitat-ID = 0 AND tagged = FALSE AND quality = qual ] [ set tagged TRUE ]      ;;
    ]                                                                                                        ;;
  ]                                                                                                          ;;
end                                                                                                          ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to initialize-travelers
  random-seed new-seed
  
  create-travelers #travelers
  [
    set shape "arrow"
    set size 3
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; traveler GO PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  tick
  ask travelers                                             ; start by assuming that each does not disperse/relocate
  [ 
    set starting-patch new-starting-patch
    set starting-habitat  [habitat-ID] of starting-patch
    set total-steps 0
    set patch-list []
    move-to starting-patch
    set pcolor black
    set energy initial-energy
    let my-neighbors neighbors with [ quality < 1 ]
    if count my-neighbors > 0
    [
    face one-of my-neighbors
    while [ energy > 0 ]
    [
      rt (random max-turn-angle)
      lt (random max-turn-angle)
      forward 1
      ifelse ( habitat-id = NOBODY )    
      [  ; still traveling
        set patch-list lput patch-here patch-list 
        set energy energy - energy-cost
        if (random-float 100) < lethality [ set energy 0 ]
      ]
      [  ; made it to another patch
        if ( starting-habitat != habitat-ID )
        [
          foreach patch-list
          [ 
            ask ? 
            [
              set crossed crossed + 1
              set pcolor black
            ]
          ]
          increment-travel-success starting-patch patch-here
        ]
        set energy 0
      ]
    ]
    ]
  ] 
end

to-report new-starting-patch
  let tmp [habitat-id] of one-of good-patches
  report one-of [ edge-patches ] of tmp 
end  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Travel success array management
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to increment-travel-success [ start-patch end-patch ]
  let index ( ( ( [ID] of [habitat-ID] of start-patch ) - 1 ) * total-habitats ) + ( [ID] of [habitat-ID] of end-patch - 1)
  array:set travel-success index  ( array:item travel-success index + 1 )
end

to-report get-travel-success [ start-habitat end-habitat  ]
  let index ( ( start-habitat - 1 ) * total-habitats ) + ( end-habitat - 1)
  report array:item travel-success index
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PLOTTING PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to show-paths
  let hi-val ln [crossed] of max-one-of patches [crossed]
  print hi-val
  ask patches with [crossed > 0]
  [
;    let tmp round (14 * crossed / hi-val)
;    set pcolor  ( 35 + (10 * tmp) ) mod 140
    let tmp  9.9 - ( 9.9 * ( ln crossed / hi-val ) )
    set pcolor tmp
  ]
end

to print-output
  output-print ""
  let row 0
  let x 0
  while [ row < total-habitats ]
  [
    let line ""
    let col 0
    while [ col < total-habitats ]
    [
      set line (word line array:item travel-success x " ")
      set x x + 1
      set col col + 1
    ]
    output-print line
    set row row + 1
  ]
end
    
  
@#$#@#$#@
GRAPHICS-WINDOW
324
11
1472
1214
-1
-1
0.6
1
10
1
1
1
0
0
0
1
0
1137
0
1171
1
1
1
ticks
30.0

BUTTON
17
182
141
215
NIL
initialize
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
144
29
299
62
#travelers
#travelers
1
500
500
1
1
NIL
HORIZONTAL

BUTTON
237
183
300
216
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
159
183
222
216
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
145
10
259
28
Traveler Setup
11
0.0
0

TEXTBOX
24
163
157
181
Model  Operation
11
0.0
0

SLIDER
145
64
301
97
initial-energy
initial-energy
0
10000
4000
1
1
NIL
HORIZONTAL

SLIDER
145
101
302
134
max-turn-angle
max-turn-angle
0
90
20
1
1
NIL
HORIZONTAL

BUTTON
195
223
300
256
NIL
show-paths
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
7
320
313
996
12

BUTTON
19
223
129
256
NIL
print-output
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This model illuminates the important areas between good habitat by sending out travelers from the edges of good habitat and finding out which patches participate in the most successful migrations.  In the initialization step, a random patchwork landscape is generated with green areas being good habitat where our critters live, yellow areas being relatively non-lethal habitat supporting migration, and red areas being more lethal for crossing.  At each tick the travelers choose a location on an edge of a green patch, face away from the patch, and begins a random-walk journey.  If that walk ends with arriving at a different green habitat, all patches involved in that crossing are told of this success and they increment a counter accordingly.

## HOW TO RUN

Set patch and traveler parameters.    
Hit "initialize".  
Hit "go" to start and then to stop.  
Hit "show-paths" to color-code the patches with respect to the participation in successful crossings  
Hit "print-output" to see the array of habitat-to-habitat crossing successes - the link strength among the network of habitats

## PATCH INITIALIZATION

This simple world consists of bad (red), neutral (yellow) and good (green) habitat.  Patchs are created by seeding an empty world with a user-specified number of "seed-patches" that are randomly established in these categories based on "seed-good" and "seed-bad" percentages.  These are then grown into the surrounding areas until all cells are assigned to one of the categories.  Like-category-cells are assigned a common patch value and a total of the combined area is calculated.  A fundamental question addressed by the model involves the number and success of migrating travelers among the good-quality patches.

Patch initialization sliders:  
* seed-patches - The number of patches that are randomly selected to initialize the growth of patches.  
* seed-good - the percent of those patches that will be "good" habitat  
* seed-bad - the percent that will be "bad" or most lethal land  
The remaining percentage will be "poor"  
* min-habitat-size - the smallest number of patches allowed in a green habitat

## TRAVELER INITIALIZATION

* "#travelers" - the total number of migrators that will attempt to move from one good patch to another  
* initial-energy - the amount of energy given to each traveler at the beginning of a journey  
* max-turn-angle - after every step of the journey, this is the maximum angle that the traveller will turn (right or left) before the next step.

## NOTES

The model runs MUCH faster with graphics turned off.  Play with the speed slider and the "view updates" check-box and watch the ticks fly by.  
-OR-  
Slow it down with the speed slider to watch it work in human-time

## WISH LIST

* Read in GIS maps of crossing energy cost and crossing lethality  
* Move habitat labels to the "center" of habitats  
* Use graphics to show link strength among habitats  
* Format the link array
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
