#################################################
#################################################
#								#
#                CHAPTER 2: 				#
#    A prior and a posteriori models for		#
#            infectious diseases			#
#								#
# last update: 21/08/2012				#
#################################################
#################################################

setwd("c:/book/chapter2")

# use library deSolve - if necessary install it: see code appendix
library(deSolve)

#################################################
## Section 2.2: An example: A basic model for 	#
##              HIV/AIDS				#
##								#
##  2.2.1 Mathematical model for HIS/AIDS		#
#################################################

### define input variables (vectors) for ode call
state <- c(S=9995,I=5,Y=0,A=0,Z=0)
times <- seq(0,25,by=0.01)
parameters <- c(N0=10000,mu=1/75,v1=1/8,v2=1/8,p=0.2,alpha=1)

### define input variable (function) for ode call
AIDS<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	B <- N0*mu
	lambda <- (I+Y)/N0
	dS = B-(mu+lambda)*S
	dI = p*lambda*S-(mu+v1)*I
	dY = (1-p)*lambda*S-(mu+v2)*Y
	dA = v1*I-(mu+alpha)*A
	dZ = v2*Y-mu*Z
	list(c(dS,dI,dY,dA,dZ))
	})
}

### numerically solve equation
require(deSolve)
out<-as.data.frame(
	ode(y=state,times=times,func=AIDS,parms=parameters)
	)

### output
head(out)

### FIGURE 2.3a
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(times,out$I+out$Y+out$A+out$Z,type="l",main=" ",
     xlab="time", ylab="predicted number")
lines(times,out$A,lty=8)
lines(times,out$Z,lty=3)

### FIGURE 2.3 (right)
plot(times,out$A,lty=8,type="l",main=" ",xlab="time", ylab="predicted number")

#################################################
## Section 2.2: An example: A basic model for 	#
##              HIV/AIDS				#
##								#
## 2.2.2 Statistical model for HIS/AIDS		#
#################################################

### DATA AIDS US
aids1.dat<-read.table('aids1.txt', header=FALSE, na.strings="NA", dec=".",  strip.white=TRUE)
cases<-aids1.dat$V2[1:18]
t.quater<-aids1.dat$V3[1:18]
fit.aidsUS<-glm(cases~t.quater,family=poisson(link="log"))
summary(fit.aidsUS)

### FIGURE 2.4 (upper left panel)
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=2,las=1,mgp=c(3, 0.5, 0))

plot(aids1.dat$V3,aids1.dat$V2,xlab="quarter (1=first quarter in 1982)",ylab="reported cases")
K<-exp(5.208901)
alpha<-0.181999
ti<-seq(from=0,to=18,length=100)
mu <- K*exp(alpha*ti)
lines(ti,mu)

### DATA AIDS IN UK
# Parameter estimates obtained using SAS
aids2.dat<-read.table('aids2.txt', header=FALSE, na.strings="NA", dec=".",  strip.white=TRUE)

### FIGURE 2.4 (upper right panel)
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(aids2.dat$V1,aids2.dat$V2,xlab="month (1=January 1982)"
    ,ylab="reported cases",xlim=c(0,60),ylim=c(0,40))
points(c(49:52),c(25,30,26,35))
K <- 0.7161
alpha <- 0.07512
ti<-seq(from=0,to=52,length=100)
mu <- K*exp(alpha*ti)
lines(ti,mu)

### DATA AIDS IN AUSTRALIA
# Parameter estimates obtained using SAS
aids3.dat<-read.table('aids3.txt', header=FALSE,na.strings="NA", dec=".",  strip.white=TRUE)

### FIGURE 2.4 (lower panel)
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=2,las=1,mgp=c(3, 0.5, 0))

plot(aids3.dat$V1,aids3.dat$V2,xlab="quarter (1=4'th quarter 1982)"
    ,ylab="reported cases",xlim=c(0,30),ylim=c(0,175))
K<-4.9239
alpha<-0.1394
ti<-seq(from=0,to=25,length=100)
mu <- K*exp(alpha*ti)
lines(ti,mu)


### FIGURE 2.5 (left panel)
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(aids3.dat$V1,aids3.dat$V2,xlab="quarter (1=4'th quarter 1982)"
    ,ylab="reported cases",xlim=c(0,30),ylim=c(0,175))

ti<-seq(from=0,to=25,length=100)
lines(ti,mu)

# model with log(t) and log link #
# Parameter estimates obtained using SAS
K2<-0.1884
alpha2<-2.0478
mu2 <- K2*(ti**alpha2)

# log logistic model #
K<-119.98
alpha<- 0.007814 
beta<- 0.3100   
mu3<-K*(alpha*exp(beta*ti)/(1+alpha*exp(beta*ti)))

# quadratic model  #
K<-0.6405
alpha<- 0.4207
beta<- -0.00853
mu4<-K*exp(alpha*ti+beta*(ti*ti))

# gompertz model  #
K<- 108.73
beta <- 0.2657 
alpha <- 0.01208
mu5 <- K*(1-exp(-alpha*exp(beta*ti)))

lines(ti,mu)
lines(ti,mu2,lty=2)
lines(ti,mu3,lty=3)
lines(ti,mu4,lty=4)
lines(ti,mu5,lty=5)
legend(0,175,c("Exp.(linear)","Exp.(quadratic)","Weibull","Logistic","Gompertz"),lty=c(1:5)) 
                

### FIGURE 2.5 (right panel)
### (prediction for 50 months)
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(aids3.dat$V1,aids3.dat$V2,xlab="quarter (1=4'th quarter 1982)"
    ,ylab="reported cases",xlim=c(0,45),ylim=c(0,1500))

K<-4.9239
alpha<-0.1394
ti<-seq(from=0,to=40,length=100)
mu <- K*exp(alpha*ti)

# model with log(t) and log link #
K2<-0.1884
alpha2<-2.0478
mu2 <- K2*(ti**alpha2)

# log logistic model #
K<-119.98
alpha<- 0.007814 
beta<- 0.3100   
mu3<-K*(alpha*exp(beta*ti)/(1+alpha*exp(beta*ti)))

# quadratic model  #
K<-0.6405
alpha<- 0.4207
beta<- -0.00853
mu4<-K*exp(alpha*ti+beta*(ti*ti))

# gompartz model  #
K<- 108.73
beta <- 0.2657 
alpha <- 0.01208
mu5 <- K*(1-exp(-alpha*exp(beta*ti)))


lines(ti,mu)
lines(ti,mu2,lty=2)
lines(ti,mu3,lty=3)
lines(ti,mu4,lty=4)
lines(ti,mu5,lty=5)
legend(0,1450,c("Exp.(linear)","Exp.(quadratic)","Weibull","Logistic","Gompertz"),lty=c(1:5)) 


