#################################################
#################################################
#								#
#                CHAPTER 3: 				#
#               The SIR Model	            	#
#								#
# last update: 21/08/2012				#
#################################################
#################################################

setwd("C:/book/chapter3.r")

library(deSolve)

#################################################
## Section 3.1: Introduction 			 	#
##								#
##  3.1.3 The basic model in R			#
#################################################

####################################
###  DYNAMICS IN THE SIR MODEL   ###
####################################
#State variables : 
#		X: Number susceptibles
#		Y: Number infected
#		Z: Number recovered
#Variables
#		mu: mortality and birth rate
#		beta: transmission rate
#		v: recovery rate 


SIR<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	dX <- N*mu*(1-p)-beta*Y*X - mu*X
	dY <- beta*Y*X - (nu+alpha+mu)*Y
	dZ <- N*mu*p+nu*Y -mu*Z
	list(c(dX, dY, dZ))
	})
}

### Define parameters
parameters <- c(mu=1/75,beta=0.0005,nu=1,N=5000,p=0,alpha=0)
state <- c(X=4999,Y=1,Z=0)
times<-seq(0,250,by=0.1)
require(deSolve)
res.scen1<-as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))

### FIGURE 3.2 (upper panels)
par(mfrow=c(1,3),cex.axis=1.3,cex.lab=1.5,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(res.scen1[,1],res.scen1[,2]/(res.scen1[,2]+res.scen1[,3]+res.scen1[,4]),xlab="time",ylab="proportion susceptible",type="l",lwd=2)
plot(res.scen1[,1],res.scen1[,3]/(res.scen1[,2]+res.scen1[,3]+res.scen1[,4]),xlab="time",ylab="proportion infected",type="l",lwd=2)
plot(res.scen1[,2],res.scen1[,3],xlab="number susceptible",ylab="number infected",type="l",lwd=2)

### Define parameters
parameters <- c(mu=1/75,beta=0.00025,nu=1,N=5000,p=0,alpha=0)
state <- c(X=4999,Y=1,Z=0)
times<-seq(0,250,by=0.1)
require(deSolve)
res.scen2<-as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))

### FIGURE 3.2 (lower panels)
par(mfrow=c(1,3),cex.axis=1.3,cex.lab=1.5,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(res.scen2[,1],res.scen2[,2]/(res.scen2[,2]+res.scen2[,3]+res.scen2[,4]),xlab="time",ylab="proportion susceptible",type="l",lwd=2)
plot(res.scen2[,1],res.scen2[,3]/(res.scen2[,2]+res.scen2[,3]+res.scen2[,4]),xlab="time",ylab="proportion infected",type="l",lwd=2)
plot(res.scen2[,2],res.scen2[,3],xlab="number susceptible",ylab="number infected",type="l",lwd=2)

#################################################
###  DYNAMICS IN THE SIR MODEL - EQUILIBRIUM  ###
#################################################

### Scenario 1
parameters <- c(mu=1/75,beta=0.0005,nu=1,N=5000,p=0,alpha=0)
state <- c(X=4999,Y=1,Z=0)
times<-seq(0,1000,by=0.1)
require(deSolve)
res.scen1<-as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
res.scen1[10001,]

### Scenario 2
parameters <- c(mu=1/75,beta=0.0005,nu=2,N=5000,p=0,alpha=0)
state <- c(X=4999,Y=1,Z=0)
times<-seq(0,1000,by=0.1)
require(deSolve)
res.scen2<-as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
res.scen2[10001,]

### FIGURE 3.3
windows(record=TRUE, width=7, height=3.5)
par(mfrow=c(1,3),cex.axis=1.3,cex.lab=1.5,lwd=3,las=1,mgp=c(3, 0.5, 0))

plot(res.scen1[,2],res.scen1[,3],xlab="number susceptible",ylab="number infected",type="l",lwd=2)
plot(res.scen2[,2],res.scen2[,3],xlab="number susceptible",ylab="number infected",type="l",lwd=2)
plot(res.scen1[,1],res.scen1[,2]/(res.scen1[,2]+res.scen1[,3]+res.scen1[,4]),xlab="time",ylab="proportion susceptible",type="l",lwd=2)
lines(res.scen1[,1],res.scen2[,2]/(res.scen2[,2]+res.scen2[,3]+res.scen2[,4]),lwd=2,lty=2)


#################################################
## Section 3.1: Introduction 			 	#
##								#
##  3.1.4 Vaccination in the basic model		#
#################################################


#############################
### CHANGE OF R_0 WITH NU ###
#############################

### FIGURE 3.4
par(mfrow=c(1,2),cex.axis=1.3,cex.lab=1.5,lwd=3,las=1,mgp=c(3, 0.5, 0))

par(mar=c(5.1,4.1,4.1,2.1))
rvec<-(0.0005*5000)/(seq(0.5,5,by=0.01)+1/75)
plot(seq(0.5,5,by=0.01)[rvec>=1],rvec[rvec>=1],type="l",xlab=expression(paste(nu," (year"^-1,")")),ylab=expression(R[0]))
points(1,(0.0005*5000)/(1+1/75))
text(1.5,(0.0005*5000)/(1+1/75),c(expression(paste(nu^-1,"=one year"))))
points(2,(0.0005*5000)/(2+1/75))
text(2.1,0.3+(0.0005*5000)/(2+1/75),c(expression(paste(nu^-1,"=6 months"))))
abline(h=1,lty=2)

par(mar=c(5.1,4.1,4.1,2.1))
sinfty<-1/((0.0005*5000)/(seq(0.5,5,by=0.01)+1/75))
iinfty<-1/75/(0.0005*5000)*((0.0005*5000)/(seq(0.5,5,by=0.01)+1/75)-1)
rinfty<-1-sinfty-iinfty
plot(sinfty[rvec>=1],rinfty[rvec>=1],xlab=expression(s(infinity)),ylab=expression(r(infinity)),type="l")
points(sinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==1],rinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==1])
text(0.20+sinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==1],rinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==1],c(expression(paste(nu^-1,"=one year"))))
points(sinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==2],rinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==2])
text(-0.18+sinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==2],rinfty[rvec>=1][seq(0.5,5,by=0.01)[rvec>=1]==2],c(expression(paste(nu^-1,"=6 months"))))



#######################################################
### Dynamic aspects of vaccination in the SIR Model ###
#######################################################

### define parameters
parameters <- c(mu=1/75,v=25,r0=15,v2=1/8,probii=0.8)
state <- c(y1=0.06666,y2=0.2)
times<-seq(0,20,by=0.01)

### model 
AM146<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	dy1 = mu*(1-probii) -(y2+mu)*y1
	dy2 =v*y2*(r0*y1-1) 
	list(c(dy1,dy2))
	}) 
}

require(deSolve)
out <- as.data.frame(ode(y=state,times=times,func=AM146,parms=parameters))
head(out)

fx1<- function(mu,lambda.t,x.t,probii)
  {
         fx.t <- mu*(1-probii) -(lambda.t+mu)*x.t
         return(fx.t)
  } 

gl1<- function(v,mu,r0,lambda.t,x.t)
  {
         gx.t <- (v)*lambda.t*(r0*x.t-1)
         return(gx.t)
  } 

rk4.1 <- function(step,nstep,L,lambda0,v,r0,x0,pp)
{
mu      <-1/L
x       <-c(x0,c(1:nstep)*0)
pi      <-c(0,c(1:nstep)*0)+pp
lambda  <-c(lambda0,c(1:nstep)*0)
timei <- c(1:(nstep+1))-1
x.star <- 1/r0
for(j in 1:nstep){
 mx1      <- fx1(mu,lambda[j],x[j],pi[j])
 ml1      <- gl1(v,mu,r0,lambda[j],x[j])
 mx2      <- fx1(mu,lambda[j]+(0.5*step)*ml1,x[j]+(0.5*step)*mx1,pi[j])
 ml2      <- gl1(v,mu,r0,lambda[j]+(0.5*step)*ml1,x[j]+(0.5*step)*mx1)
 mx3      <- fx1(mu,lambda[j]+(0.5*step)*ml2,x[j]+(0.5*step)*mx2,pi[j])
 ml3      <- gl1(v,mu,r0,lambda[j]+(0.5*step)*ml2,x[j]+(0.5*step)*mx2)
 mx4      <- fx1(mu,lambda[j]+(step)*ml3,x[j]+(step)*mx3,pi[j])
 ml4      <- gl1(v,mu,r0,lambda[j]+(step)*ml3,x[j]+(step)*mx3)
 x[j+1]     <- x[j]+(step/6)*(mx1+2*(mx2+mx3)+mx4)
 lambda[j+1]<- lambda[j]+(step/6)*(ml1+2*(ml2+ml3)+ml4)
}
#par(mar=c(5.1,4.1,4.1,2.1))
plot(timei*step,x,type="l",xlab="time",ylab="proportion susceptible")
lines(c(0,nstep)*step,c(x.star,x.star),lty=2)
mtext("time",side=1,line=3)
#par(mar=c(7.1,4.1,2.1,2.1))
plot(timei*step,lambda,type="l",xlab="time",ylab="lambda(t)")
lines(c(0,nstep)*step,c(mu*(r0-1),mu*(r0-1)),lty=2)
mtext("time",side=1,line=3)
}

### FIGURE 3.5
par(mfrow=c(2,1),lwd=3,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(4.8, 0.5, 0),mar=c(5.1,5.5,4.1,2.1))
rk4.1(0.01,20*(1/0.01),70,0.2,25,15,0.06666,0.0)

### FIGURE 3.6
par(mfrow=c(2,1),lwd=3,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))
rk4.1(0.01,20*(1/0.01),70,0.2,25,15,0.06666,0.4)

### FIGURE 3.7
par(mfrow=c(2,1),lwd=3,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))
rk4.1(0.01,20*(1/0.01),70,0.2,25,15,0.06666,0.8)

#################################################
## Section 3.1: Introduction 			 	#
##								#
##  3.1.5 The basic SIR model with vaccination	#
#################################################

SIR<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	dX <- N*mu*(1-p)-beta*Y*X - mu*X
	dY <- beta*Y*X - v*Y - mu*Y
	dZ <- v*Y -mu*Z+N*mu*p
	list(c(dX, dY, dZ))
	}) 
}

### define parameters
parameters <- c(mu=1/75,beta=0.0005,v=1)
state <- c(X=4999,Y=1,Z=0)
times<-seq(0,800,by=0.01)
N<-5000

require(deSolve)
p<-0
outp0 <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
p<-0.2
outp02 <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
p<-0.4
outp04 <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))

### FIGURE 3.8 - CHECK
par(mfrow=c(1,2),lwd=2,las=1,cex.axis=0.8,cex.lab=0.8,mgp=c(2, 0.45, 0),
mar=c(3.2, 2.8, 3.5, 1.5))

plot(times,outp0$X,type="l",main=" ", xlab="time", ylab="Number of susceptibles")
lines(times,outp02$X,lty=2)
lines(times,outp04$X,lty=3)
#legend(100,5000,c("p=0","p=0.2","p=0.4"),lty=c(1,4,2))

plot(outp0$X,outp0$Y,main=" ",type="l",xlab="Number of susceptibles",ylab="Number of infected")
lines(outp02$X,outp02$Y,lty=2)
lines(outp04$X,outp04$Y,lty=3)
#legend(4000,1100,c("p=0","p=0.4"),lty=c(1,4))



#################################################
## Section 3.1: Introduction 			 	#
##								#
##  3.1.5 The SIR model in endemic equilibrium	#
#################################################

#############################
### VACCINATION DATA      ###
#############################

#cases#                                                                                             
#http://www.who.int/immunization_monitoring/en/globalsummary/timeseries/tsincidencemea.htm          

#coverage#                                                                                          
#http://www.who.int/immunization_monitoring/en/globalsummary/timeseries/tscoveragemcv.htm           
#http://www.who.int/immunization_monitoring/data/data_subject/en/index.html                        

#BRASIL
BMpvac<-c(99,99,99,99,99,99,96,99,99,99,95,99,80,87,77,85,91,
          85,78,60,62,64,67,67,73,68,66,73,57)  
BMcases<-c(0,0,57,6,0,2,1,1,36,908,2781,52284,580,793,35,6037,
           7934,42537,61435,22853,26179,66059,129126,75993,80875,58259,39370,61281,99263) 

#Guatemala
GMcases<-c(0,0,0,0,0,0,0,0,0,0,1,8,1,23,204,17,97,209,
          8802,2413,182,400,1650,2272,3072,2762,
          3992,3472,2703) 
GMpvac<-c(96,93,95,95,95,94,92,90,86,83,79,74,69,83,
          66,71,59,49,68,60,55,24,55,23,24,9,12,8,23) 

ptime<-c(2008:1980)

### FIGURE 3.9
layout(matrix(c(1,1,2,2,3,3,4,4), 2,4, byrow = TRUE))
par(lwd=2,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))

plot(ptime,BMcases/1000,ylab="Number of cases (x1000)",xlab="time")
lines(ptime,BMcases/1000)
plot(ptime,BMpvac,ylab="Vaccination coverage",xlab="time",ylim=c(50,100))
lines(ptime,BMpvac)

plot(ptime,GMcases/1000,ylab="Number of cases (x1000)",xlab="time")
lines(ptime,GMcases/1000)
plot(ptime,GMpvac,ylab="Vaccination coverage",xlab="time",ylim=c(0,100))
lines(ptime,GMpvac)


#################################################
## Hep A data from Bulgraia                    ##
#################################################

hav<-read.table("HAV-BUL.dat",header=T)
grid<-hav$Age
pos<-hav$Pos
tot<-hav$Tot

### FIGURE 3.11
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))
plot(grid,pos/tot,cex=0.1*tot,pch=19,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(0,1))


#################################################
## Section 3.3: The time homogeneous SIR Model ##
##              and serological data           ##
#################################################


#################################################
## model: general model for illustraion        ##
#################################################

SIR<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	ds <- -lambda*s 
	di <- lambda*s - nu*i
	dr <- nu*i
	list(c(ds, di, dr))
	}) 
}

### define parameters
parameters <- c(lambda = 0.1, nu=36.5)
state <- c(s=0.999,i=0.001,r=0)
times<-seq(0,40,by=0.01)
require(deSolve)
out <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
head(out)

### FIGURE 3.12 (upper panel)
layout(matrix(c(1,1,2,2,3,3,4,4), 2,4, byrow = TRUE))
par(lwd=2,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))

plot(times,out$i ,type="l",xlab="age", ylab=" ")
plot(times,out$s ,type="l",xlab="age", ylab=" ")
lines(times,out$r,lty=2)

### define parameters
parameters <- c(lambda = 0.2, nu=36.5)
state <- c(s=0.999,i=0.001,r=0)
times<-seq(0,40,by=0.01)
require(deSolve)
out <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
head(out)

### FIGURE 3.12 (lower panel)
plot (times,out$i ,type="l",xlab="age", ylab=" ")
plot (times,out$s ,type="l",xlab="age", ylab=" ")
lines(times,out$r,lty=2)

### FIGURE 3.13                                  
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))
a<-seq(from=0,to=50,length=100)
pc<-0.89
mu<-1/75
pv<-c(0,0.1,0.3,0.5,0.75,0.89)
lambda<-1/9
R0<-(lambda+mu)/mu
R0
lambdaj<-mu*R0*(pc-pv)
xa1<-(1-pv[1])*exp(-lambdaj[1]*a)
plot(a,xa1,type="l",xlim=c(-2,50),xlab="age",ylab="proportion susceptible")
for(i in 1:length(lambdaj))
{
xa<-(1-pv[i])*exp(-lambdaj[i]*a)
lines(a,xa,lty=i)
text(0,xa[1]+0.02,paste("p=",pv[i]),cex=1)
}

par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))
r0<-c(15,10,5)
h<-c(-2,2,2)
L<-75
plot(c(0,0.95),c(0,75),pch=" ",ylab="average age at infection",xlab="proportion vaccinated")
ii<-c(1,6,4)
for(i in 1:3){
pc<-1-1/r0[i]
pp<-seq(from=0,to=pc,length=100)
A<-L/r0[i]
Ap<-A/(1-pp)
lines(pp,Ap,lty=ii[i])
lines(c(pc,pc),c(0,75),lty=2)
text(0.025,Ap[1]+h[i],paste(expression(Ro),"=",r0[i]),cex=1)
}

#################################################
## Hep A data from Bulgaria                    ##
#################################################

hav<-read.table("HAV-BUL.dat",header=T)
Age=grid=hav$Age
Pos=pos=hav$Pos
Tot=tot=hav$Tot
lAge=log(hav$Age)

glmfit=glm(cbind(Pos,Tot-Pos)~1,offset=lAge,family=binomial(link=cloglog))
summary(glmfit)

### FIGURE 3.14
par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=3,las=1,mgp=c(3, 0.5, 0))
plot(grid,pos/tot,cex=0.1*tot,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(0,1))

ti<-grid
beta<-0.0505
pii <- 1-exp(-beta*ti)
lines(ti,pii)


#################################################
## SIR for Hep A                               ##
## model: sir with foi=0.05 from the glm       ##
#################################################

SIR<-function(t,state,parameters)
	{
	with(as.list(c(state, parameters)),
	{
	ds <- -lambda*s 
	di <- lambda*s - nu*i
	dr <- nu*r
	list(c(ds, di, dr))
	}) 
}

## define parameters
parameters <- c(lambda = 0.05, nu=1/(14/365))
state <- c(s=0.99,i=0.01,r=0)
times<-seq(0,90,by=0.01)
require(deSolve)

out <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
head(out)

### FIGURE 3.14 (dashed line)
lines(times,out$Z ,type="l",main=" ", xlab="age", ylab=" ",lty=2)


###############################################
## Variability in serological data and in    ##
## deterministic model                       ##
###############################################

ti<-hav$Age+0.5
length(ti)
pii <- 1-exp(-beta*ti)
N<-10

### Resample serological data (size 20)
nsize<-20
posi<-matrix(0,length(pii),N)
for(j in 1:N){
for(i in 1:length(pii)){
posi[i,j]<-rbinom(1,nsize,pii[i])
}
}

### FIGURE 3.15 (upper left panel)
layout(matrix(c(1,1,2,2,3,3,4,4), 2,4, byrow = TRUE))
par(lwd=2,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))

#par(mfrow=c(1,1),cex.axis=1.2,cex.lab=1.2,lwd=2,las=1,mgp=c(3, 0.5, 0))
Pos<-posi[,j]
NEG<-nsize-Pos
plot(ti,posi[,1]/nsize,pch=" ",ylim=c(0,1),xlab="age",ylab="prevalence",lwd=2)
for(i in 1:N)
{
lines(ti,posi[,i]/nsize,lwd=1)
}
points(hav$Age,hav$Pos/hav$Tot,pch=3,col=1)

### Resample serological data (size 100)
nsize<-100
posi<-matrix(0,length(pii),N)
for(j in 1:N){
for(i in 1:length(pii)){
posi[i,j]<-rbinom(1,nsize,pii[i])
}
}

### FIGURE 3.15 (upper right panel)
Pos<-posi[,j]
NEG<-nsize-Pos

plot(ti,posi[,1]/nsize,pch=" ",ylim=c(0,1),xlab="age",ylab="prevalence",lwd=1)
for(i in 1:N)
{
lines(ti,posi[,i]/nsize,lwd=1)
}
points(hav$Age,hav$Pos/hav$Tot,pch=3,col=1)


### Deterministic model

lambda.i<-nu.i<-c(1:100)
lambda.i<-c(1:100)
zmat<-matrix(0,9001,100)
SIR<-function(t,state,parameters)
{
with(as.list(c(state, parameters)),
{
ds <- -lambda*s 
di <- lambda*s - nu*i
dr <- nu*i
list(c(ds, di, dr))
}) 
}
state <- c(s=0.999,i=0.001,r=0)
times<-seq(0,90,by=0.01)


### FIGURE 3.15 (lower left panel)

for(i in 1:10)
{
lambda.i[i]<-rnorm(1,0.0505,0.02)
#nu.i[i]<-runif(1,1/(21/365),1/(7/365))
parameters <- c(lambda = lambda.i[i], nu=1/(14/365))
#parameters <- c(lambda = lambda.i[i], nu=nu.i[i])
state <- c(s=0.999,i=0.001,r=0)
times<-seq(0,90,by=0.01)
require(deSolve)
out <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
zmat[,i]<-out$r
}
dim(zmat)
length(times)

plot(times,zmat[,1],ylim=c(0,1),type="n",xlab="age",ylab="prevalence")
for(i in 1:10)
{
lines(times,zmat[,i])
}

### FIGURE 3.15 (lower right panel)

for(i in 1:10)
{
lambda.i[i]<-rnorm(1,0.0505,0.01)
#nu.i[i]<-runif(1,1/(21/365),1/(7/365))
parameters <- c(lambda = lambda.i[i], nu=1/(14/365))
#parameters <- c(lambda = lambda.i[i], nu=nu.i[i])
state <- c(s=0.999,i=0.001,r=0)
times<-seq(0,90,by=0.01)
require(deSolve)
out <- as.data.frame(ode(y=state,times=times,func=SIR,parms=parameters))
zmat[,i]<-out$r
}
dim(zmat)
length(times)

plot(times,zmat[,1],ylim=c(0,1),type="n",xlab="age",ylab="prevalence")
for(i in 1:10)
{
lines(times,zmat[,i])
}

### average age at infection and basic reproduction number for HAV
library(car)
obj=coef(glmfit)
names(obj)<-"Intercept"
deltaMethod(obj,"1/exp((Intercept))",vcov.=vcov(glmfit))
deltaMethod(obj,"75*exp((Intercept))",vcov.=vcov(glmfit))

#################################################
## Section 3.4: Models with Maternal Antibodies##
##              and latent periods             ##
#################################################

### Data VZV-B19 Belgium

vzv.dat<-read.table('VZV-B19-BE.dat', header=TRUE)

dim(vzv.dat)

cbind(vzv.dat$age[vzv.dat$VZVmUIml>0 & vzv.dat$age <1 ],vzv.dat$VZVmUIml[vzv.dat$VZVmUIml>0 & vzv.dat$age <1])

ageb<-vzv.dat$age[vzv.dat$VZVmUIml>0 & vzv.dat$age <1.5 ]
agebm<-vzv.dat$age[vzv.dat$VZVmUIml>0 & vzv.dat$age <1.5 ]*12
antib<-log10(vzv.dat$VZVmUIml[vzv.dat$VZVmUIml>0 & vzv.dat$age <1.5 ])
ageb1<-vzv.dat$age[vzv.dat$VZVmUIml>0 & vzv.dat$age <1.5 ]
antib1<-log10(vzv.dat$VZVmUIml[vzv.dat$VZVmUIml>0 & vzv.dat$age <1.5])

### FIGURE 3.16
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(3, 0.5, 0))

plot(agebm,antib,xlim=c(0,18),xlab="age (months)",ylab="log(antibody)")

library(locfit)
alpha=c(0,6)
lpfit<-locfit(antib~agebm,alpha=alpha)
lpfit
lines(agebm[order(agebm)],fitted(lpfit)[order(agebm)],lty=1,lwd=2)


### MSEIR Model 
epi11<- function(last.age,d,lambda,sigma,ni)
{
        
        #d          <- (1/0.5)
        #lambda     <- (1/5) 
        #sigma      <- (34.76)
        #ni         <- (31.74)
        N0         <- 1000
        a          <- seq(from=1,to=last.age,length=500)
        la         <- 1
        N1a <- N0*la
        ia  <- exp(-d*a)
        kk1 <- (d/(d-lambda))
        kk2 <- (exp(-lambda*a)-exp(-d*a))
        xa  <- (d/(d-lambda))*(exp(-lambda*a)-exp(-d*a))
        #browser()
        ha  <- (
               (lambda*d)/(d-lambda)
               )*
               (
               ((exp(-sigma*a)-exp(-lambda*a))/(lambda-sigma))
              -((exp(-sigma*a)-exp(-d*a))/(d-sigma))
               )
        ya  <- (sigma*lambda*d)*
               (
               ((exp(-ni*a)-exp(-sigma*a))/((lambda-sigma)*(d-sigma)*(sigma-ni)))
              +((exp(-ni*a)-exp(-lambda*a))/((lambda-d)*(lambda-sigma)*(lambda-ni)))
              +((exp(-ni*a)-exp(-d*a))/((d-lambda)*(d-sigma)*(d-ni)))
               )  
        a  <- c(0,a)
        ia <- c(1,ia)
        xa <- c(0,xa)
        ha <- c(0,ha)
        ya <- c(0,ya)
        za  <- 1 - ia - xa - ha -ya   

plot(a,ia,type="l",xlab="Age",ylab="M(a)",pch=0.5,xlim=c(0,last.age))
#title("a:Proportion of host with maternal antibodies",adj=0,cex=0.35)
plot(a,xa,type="l",xlab="Age",ylab="S(a)",pch=0.5,xlim=c(0,last.age))
#title("b:Proportion of susceptibles",adj=0,cex=0.35)
plot(a,ha,type="l",xlab="Age",ylab="E(a)",pch=0.5,xlim=c(0,last.age))
#title("c:Proportion of host in the latent class",adj=0,cex=0.35)
plot(a,ya,type="l",xlab="Age",ylab="I(a)",pch=0.5,xlim=c(0,last.age))
#title("d:Proportion of infected",adj=0,cex=0.35)
plot(a,za,type="l",xlab="Age",ylab="R(a)",pch=0.5,xlim=c(0,last.age))
#title("e:Proportion host in the immune class",adj=0,cex=0.35)
plot(a,(ia+za+ya),type="l",xlab="Age",ylab="seroprevalence",pch=0.5,xlim=c(0,last.age))
#title("f:Proportion of sero-positive",adj=0,cex=0.35)
}

### FIGURE 3.18
par(mfrow=c(3,2),lwd=2,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))
epi11(40,1/0.5,0.2,26.07,36.5)


#################################################
## Section 3.5: Transmission with multiple     ##
##              subpopulations                 ##
#################################################

### Example: two interacting pop                     
### model: capasso                                      
### Note that the notation used here is different from the notation in the book

state <- c(Y1=0.8,Y2=0.2,Y3=0,Y4=0.8,Y5=0.2,Y6=0)
times<-seq(0,10000,by=0.01)

SIRtwo<-function(t,state,parameters)
{
	with(as.list(c(state, parameters)),
	{
	dY1 <- -(beta11*Y2+beta12*Y5)*Y1+mu-mu*Y1
	dY2 <- (beta11*Y2+beta12*Y5)*Y1-v1*Y2-mu*Y2
	dY3 <- v1*Y2 - mu*Y3
	dY4 <- -(beta21*Y2+beta22*Y5)*Y4+mu-mu*Y4
	dY5 <-  (beta21*Y2+beta22*Y5)*Y4-v2*Y5-mu*Y5
	dY6 <- v2*Y5-mu*Y6
	list(c(dY1,dY2,dY3,dY4,dY5,dY6))
}) 
}

### define parameters set 1 (beta12=0)
times<-seq(0,10000,by=0.5)
require(deSolve)

parameters <- c(beta11=0.05,beta12=0.00,beta21=0.00,beta22=0.05,v1=1/30,v2=1/30,mu=0.001)
out <- as.data.frame(ode(y=state,times=times,func=SIRtwo,parms=parameters))
outbeta120=out

### define parameters set 2 (beta12=0.025)
parameters <- c(beta11=0.05,beta12=0.025,beta21=0.025,beta22=0.05,v1=1/30,v2=1/30,mu=0.001)
out <- as.data.frame(ode(y=state,times=times,func=SIRtwo,parms=parameters))
outbeta12025=out

### define parameters set 3 (beta12=0.050)
parameters <- c(beta11=0.05,beta12=0.050,beta21=0.050,beta22=0.05,v1=1/30,v2=1/30,mu=0.001)
out <- as.data.frame(ode(y=state,times=times,func=SIRtwo,parms=parameters))
outbeta1205=out

### define parameters set 4 (beta12=0.075)
parameters <- c(beta11=0.05,beta12=0.075,beta21=0.075,beta22=0.05,v1=1/30,v2=1/30,mu=0.001)
out <- as.data.frame(ode(y=state,times=times,func=SIRtwo,parms=parameters))
outbeta12075=out

### FIGURE 3.21
layout(matrix(c(1,1,2,2,3,3,4,4), 2,4, byrow = TRUE))
par(lwd=2,las=1,cex.axis=1.1,cex.lab=1.1,mgp=c(3, 0.5, 0))

plot(outbeta120$Y1,outbeta120$Y2 ,type="l",xlab="susceptible proportion",ylab="infected proportion")
plot(outbeta1205$Y1,outbeta1205$Y2 ,type="l",xlab="susceptible proportion",ylab="infected proportion")
plot(times,outbeta120$Y2 ,type="l",xlab="time",ylab="infected proportion")
plot(times,outbeta1205$Y2 ,type="l",main=" ", xlab="time",ylab="infected proportion")

### FIGURE 3.22
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(3, 0.5, 0))

plot(outbeta120$Y1,log(outbeta120$Y2) ,type="l",xlim=c(0,1),ylim=c(-10,0),xlab="fraction susceptible individuals",ylab="log(fraction infected individuals)")
lines(outbeta12025$Y1,log(outbeta12025$Y2),lty=2)
lines(outbeta1205$Y1,log(outbeta1205$Y2),lty=3)
lines(outbeta12075$Y1,log(outbeta12075$Y2),lty=4)
#legend(0,-8,c("alpha=0","alpha=0.025","alpha=0.05","alpha=0.075"),lty=c(1:4))



