#################################################
#################################################
#								#
#                CHAPTER 6: 				#
#    Parametric Approaches to Model the		#
#     Prevalence and Force of Infection		#
#								#
# last update: 25/08/2012				#
#################################################
#################################################

setwd("c:/book/chapter6")

library(deSolve)

##################################################
## Section 6.1: Modelling the Force of 	 	 #
##              Infection: Historical perspective#
##								 #
##  6.1.1 Polynomial models                      #
##################################################


### HAV-BUL data

hav<-read.table("c:/book/chapter4/HAV-BUL.dat",header=T)
names(hav)

attach(hav)

model0<-glm(cbind(Pos,Tot-Pos)~Age,family=binomial(link="logit"),data=hav)

###  Muench's Model ###
#######################

model1<-glm(cbind(Tot-Pos,Pos)~-1+Age,family=binomial(link="log"))
summary(model1)
# Alternative R-code to fit Muench's model
# model2<-glm(cbind(Pos,Tot-Pos)~1,offset=log(Age),
#        family=binomial(link="cloglog"))
# summary(model2)
# exp(coef(model2))

###  Griffiths' Model ###
#########################
model3<-glm(cbind(Tot-Pos,Pos)~-1+Age+I(Age^2),
                               family=binomial(link="log"))
summary(model3)

### Grenfell and Anderson ###
#############################
model4<-glm(cbind(Tot-Pos,Pos)~-1+Age+I(Age^2)+I(Age^3),
                               family=binomial(link="log"))
summary(model4)


### FIGURE 6.1
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,Pos/Tot,cex=0.1*Tot,xlab="age",xlim=c(0,86),ylim=c(0,1),ylab="seroprevalence")
lines(Age,1-model1$fitted.values,lwd=2)
lines(Age,1-model3$fitted.values,lwd=2,lty=2)
lines(Age,1-model4$fitted.values,lwd=2,lty=3)

X<--matrix(rep(1,length(Age)))
lines(Age,5*X%*%model1$coefficients,lwd=2)
X<--cbind(rep(1,length(Age)),2*Age)
lines(Age,5*X%*%model3$coefficients,lwd=2,lty=2)
X<--cbind(rep(1,length(Age)),2*Age,3*Age^2)
lines(Age,5*X%*%model4$coefficients,lwd=2,lty=3)
axis(side=4,at=c(0.0,0.2,0.4),labels=c(0.00,0.04,0.08))
mtext(side=4,"force of infection", las=3,line=2)


##################################################
## Section 6.1: Modelling the Force of 	 	 #
##              Infection: Historical perspective#
##								 #
##  6.1.2 Nonlinear models                       #
##################################################

### Data  Rubella

rubella<-read.table("c:/book/chapter4/rubella-UK.dat",header=T)
names(rubella)
attach(rubella)

Tot<-Neg+Pos

### Farrington's Model  ###
###########################
farrington=function(alpha,beta,gamma)
{
p=1-exp((alpha/beta)*Age*exp(-beta*Age)
       +(1/beta)*((alpha/beta)-gamma)*(exp(-beta*Age)-1)-gamma*Age)
ll=Pos*log(p)+(Tot-Pos)*log(1-p)
#alternative definition of the log-likelihood
#ll=sum(log(dbinom(Pos,Tot,prob=p)))
return(-sum(ll))
}

library(stats4)
model5=mle(farrington,start=list(alpha=0.07,beta=0.1,gamma=0.03))
summary(model5)
AIC(model5)

### Farrington's model assuming gamma=0  ###
############################################
model6=mle(farrington,fixed=list(gamma=0),
                                  start=list(alpha=0.07,beta=0.1))
summary(model6)
AIC(model6)

### FIGURE 6.2
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,Pos/Tot,cex=0.018*Tot,xlab="age",xlim=c(0,45),ylim=c(0,1),ylab="seroprevalence")
alpha<-coef(model5)[1]
beta<-coef(model5)[2]
gamma<-coef(model5)[3]

p<-1-exp((alpha/beta)*Age*exp(-beta*Age)+(1/beta)*((alpha/beta)-gamma)*(exp(-beta*Age)-1)-gamma*Age)
lines(Age,p,lwd=2)
foi<-(alpha*Age-gamma)*exp(-beta*Age)+gamma
lines(Age,2*foi,lwd=2)

alpha<-coef(model5)[1]
beta<-coef(model5)[2]
gamma<-0

p<-1-exp((alpha/beta)*Age*exp(-beta*Age)+(1/beta)*((alpha/beta)-gamma)*(exp(-beta*Age)-1)-gamma*Age)
lines(Age,p,lwd=2,lty=2)
foi<-(alpha*Age-gamma)*exp(-beta*Age)+gamma
lines(Age,2*foi,lwd=2,lty=2)

axis(side=4,at=c(0.0,0.2,0.4),labels=c(0.00,0.1,0.2))
mtext(side=4,"force of infection", las=3,line=1.5)



rm(list=ls(all=TRUE))
detach(rubella)



### Other nonlinear models that can be linearized  ###
######################################################

### HepC data
hcvBE.dat<-read.table("hcv.txt",header=T)
head(hcvBE.dat)
attach(hcvBE.dat)
infected=infect
log.d=log(d)

infected<-hcvBE.dat$infect
log.d<-log(hcvBE.dat$d)
d<-(hcvBE.dat$d)
hcvfit=(glm(infected~log.d, family=binomial(link="cloglog")))
summary(hcvfit)


### FIGURE 6.3
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))


hcvBEgr.dat<-read.table('c:/book/chapter4/hcvdat.txt', header=FALSE,na.strings=".", dec=".",  strip.white=TRUE)
tgr=hcvBEgr.dat$V1
prop=hcvBEgr.dat$V4
tot=hcvBEgr.dat$V2
plot(tgr,prop,cex=0.1*tot,xlab="exposure time",xlim=c(0,25),ylim=c(0,1),ylab="seroprevalence")
b0=coef(hcvfit)[1]
b1=coef(hcvfit)[2]
fitted=predict(hcvfit,type="response")
fittedorig=1-exp(-exp(b0)*d^b1)
lines(sort(exp(log.d)),fittedorig[order(exp(log.d))],lwd=2)

foi=exp(b0)*b1*exp(log.d)^(b1-1)
foi.t=0.5*foi[exp(log.d)>=0.5]
t.t=exp(log.d)[exp(log.d)>=0.5]
lines(sort(t.t),foi.t[order(t.t)],lwd=2)
axis(side=4,at=c(0.0,0.1,0.2),labels=c(0.0,0.2,0.4))
mtext(side=4,"force of infection", las=3,line=1.5)


##################################################
## Section 6.2: Fractional Polynomial Models     #
##								 #
## 6.2.1 Motivating Example                      #
##################################################
rm(list=ls(all=TRUE))


### DATA HAV-BE
hav<-read.table("c:/book/chapter4/HAV-BE.dat",header=T)
attach(hav)
names(hav)
pos<-hav$Neg
neg<-hav$Pos
Tot<-hav$Tot

model.fp1<-glm(cbind(pos,Tot-pos) ~ Age + I(Age^3), family=binomial(link="logit"))
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^2) + I(Age^3), family=binomial(link="cloglog"))

### FIGURE 6.4
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,pos/Tot,cex=0.05*Tot,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(-0.2,1),yaxt="n")
lines(Age,model.fp1$fitted.values,lwd=2)
lines(Age,model.fp2$fitted.values,lwd=2,lty=2)

model.fp1.foi<-(model.fp1$coef[2]+3*model.fp1$coef[3]*Age^2)*model.fp1$fitted.values
model.fp2.foi<-(2*model.fp2$coef[2]*Age+3*model.fp2$coef[3]*Age^2)*exp(model.fp2$coef[1]+model.fp2$coef[2]*Age^2+model.fp2$coef[3]*Age^3)
lines(Age,4*model.fp1.foi,lwd=2)
lines(Age,4*model.fp2.foi,lwd=2,lty=2)
abline(0,0,lwd=2)
axis(side=4,at=c(-0.2,0.0,0.2,0.4),labels=c(-0.2,0.0,0.2,0.4))
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))

mtext(side=4,"force of infection", las=3,line=2)


##################################################
## Section 6.2: Fractional Polynomial Models     #
##								 #
## 6.2.5 Application to the data                 #
##################################################
rm(list=ls(all=TRUE))

search.fracpoly.one<-function(y,tot,x){
     pow1<-seq(-2,3,0.01)
     deviance<-deviance(glm(cbind(y,tot-y)~x, family="binomial"(link=logit)))
     power<-1
     mistake<-NULL

     for (i in 1: (length(pow1))){
          if(pow1[i]==0){term1<-log(x)} else{term1<-(x)^(pow1[i])}
          glm.try<-glm(cbind(y,tot-y)~term1, family="binomial"(link=logit))
          if(glm.try$converged==FALSE){mistake<-rbind(mistake, c(1,pow1[i]))}
          else{
               if(deviance(glm.try)<deviance){
                    deviance<-deviance(glm.try)
                    power<-pow1[i]
                                   }
               }
                                     }
     return(list(power=power, deviance=deviance, mistake=mistake))
}

search.fracpoly.two<-function(y,tot,x){
     pow<-seq(-2,3,0.1)
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=logit)))
     mistake<-NULL

     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          if(pow[i]==0){term1<-log(x)} else{term1<-(x)^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x)} 
          else if(pow[j]==0){term2<-log(x)} 
          else{term2<-(x)^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=logit))
          if(glm.try$converged==FALSE){mistake<-rbind(mistake, c(1,pow[i],pow[j]))}
          else{
               if(deviance(glm.try)<deviance){
                    deviance<-deviance(glm.try)
                    power<-c(pow[i],pow[j])
                                   }
               }
                                     
      }
      }
     return(list(power=power, deviance=deviance, mistake=mistake))
}


# same function but now with powergrid as parameter 
# mc=monotonicity constraint, assumes x is ordered
search.fracpoly.twoR<-function(y,tot,x,pow,mc){
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=logit)))
     mistake<-NULL

     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          if(pow[i]==0){term1<-log(x)} else{term1<-(x)^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x)} 
          else if(pow[j]==0){term2<-log(x)} 
          else{term2<-(x)^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=logit))
	    print(c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0)))
          if(glm.try$converged==FALSE){mistake<-rbind(mistake, c(1,pow[i],pow[j]))}
          else{
               if(deviance(glm.try)<deviance){
                    if (((mc)&&(sum(diff(predict(glm.try))<0)==0))|(!mc)){
			     #print(c(mc,(sum(diff(predict(glm.try))<0)==0)))
                       deviance<-deviance(glm.try)
                       power<-c(pow[i],pow[j])
                    } 
                                   }
               }
                                     
      }
      }
     return(list(power=power, deviance=deviance, mistake=mistake))
}


### Listing all fits
search.fracpoly.twoF<-function(y,tot,x,pow,mc){
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=logit)))
     mistake<-NULL
     res=matrix(NA,nrow=length(pow)*(length(pow)-1)/2+length(pow),ncol=4)
     k=1
     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          if(pow[i]==0){term1<-log(x)} else{term1<-(x)^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x)} 
          else if(pow[j]==0){term2<-log(x)} 
          else{term2<-(x)^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=logit))
          if(glm.try$converged!=FALSE){
             res[k,]=c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             } 
          else{
             res[k,]=c(pow[i],pow[j],NA,NA)
             }
          k=k+1
      }
      }
     res1=res[res[,4]==1,]
     res1=res1[order(res1[,3]),]
     res2=res[order(res[,3]),]
     return(list(pow.mon=res1,pow.all=res2))
}



### Listing all fits, symmetric in the powers
search.fracpoly.twoF2<-function(y,tot,x,pow,mc){
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=logit)))
     mistake<-NULL
     res=matrix(NA,nrow=length(pow)*length(pow),ncol=4)
     k=1
     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          if(pow[i]==0){term1<-log(x)} else{term1<-(x)^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x)} 
          else if(pow[j]==0){term2<-log(x)} 
          else{term2<-(x)^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=logit))
          if(glm.try$converged!=FALSE){
             res[k,]=c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             k=k+1
             }
             } 
          else{
             res[k,]=c(pow[i],pow[j],NA,NA)
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],NA,NA)
             k=k+1
             }
             }
      }
      }
     res1=res[res[,4]==1,]
     res1=res1[order(res1[,3]),]
     res2=res[order(res[,3]),]
     return(list(pow.mon=res1,pow.all=res2))
}


### Listing all fits, symmetric in the powers, 
### but now checking monotonicity over a fine grid of x-values
search.fracpoly.twoF2v2<-function(y,tot,x,pow,mc){
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=logit)))
     mistake<-NULL
     res=matrix(NA,nrow=length(pow)*length(pow),ncol=4)
     k=1
     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          xgrid=seq(min(x),max(x),by=0.01)
          if(pow[i]==0){term1<-log(x);term1grid=log(xgrid)} else{term1<-(x)^(pow[i]);term1grid=xgrid^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x);term2grid=term1grid*log(xgrid)} 
          else if(pow[j]==0){term2<-log(x);term2grid=log(xgrid)} 
          else{term2<-(x)^(pow[j]);term2grid=xgrid^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=logit))
          if(glm.try$converged!=FALSE){
             res[k,]=c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try,newdata=data.frame(term1=term1grid,term2=term2grid)))<0)==0))
#             res[k,]=c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],deviance(glm.try),(sum(diff(predict(glm.try,newdata=data.frame(term1=term1grid,term2=term2grid)))<0)==0))
             k=k+1
             }
             } 
          else{
             res[k,]=c(pow[i],pow[j],NA,NA)
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],NA,NA)
             k=k+1
             }
             }
      }
      }
     res1=res[res[,4]==1,]
     res1=res1[order(res1[,3]),]
     res2=res[order(res[,3]),]
     return(list(pow.mon=res1,pow.all=res2))
}


foi.num<-function(x,p)
{
grid<-sort(unique(x))
pgrid<-(p[order(x)])[duplicated(sort(x))==F]
dp<-diff(pgrid)/diff(grid)
foi<-approx((grid[-1]+grid[-length(grid)])/2,dp,grid[c(-1,-length(grid))])$y/(1-pgrid[c(-1,-length(grid))])
return(list(grid=grid[c(-1,-length(grid))],foi=foi))
}


### HAV - Bulgaria
###################

hav<-read.table("c:/book/chapter4/HAV-BUL.dat",header=T)
names(hav)
#suc=hav$Pos
#fail=hav$Tot-hav$Pos
#age=hav$Age
#mfp(cbind(suc,fail)~fp(age, df = 4, select = 0.3),family = binomial(link="logit"), alpha=0.3)


search.fracpoly.one(y=hav$Pos,tot=hav$Tot,x=hav$Age)
search.fracpoly.two(y=hav$Pos,tot=hav$Tot,x=hav$Age)
search.fracpoly.twoR(y=hav$Pos,tot=hav$Tot,x=hav$Age,seq(-2,3,0.1),T)

Age<-hav$Age
pos<-hav$Pos
Tot<-hav$Tot
model.fp1<-glm(cbind(pos,Tot-pos) ~ Age, family=binomial(link="logit"))
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^1.9) + I(Age^2), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age^1.6) + I(Age^2.1), family=binomial(link="logit"))

plot(Age,pos/Tot,cex=0.05*Tot,pch=19,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(0,1))

### FIGURE 6.5 (right panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,pos/Tot,cex=0.1*Tot,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(-0.2,1),yaxt="n")
lines(Age,model.fp1$fitted.values,lwd=2)
lines(Age,model.fp2$fitted.values,lwd=2,lty=2)
lines(Age,model.fp2.m$fitted.values,lwd=2,lty=3)

model.fp1.foi<-c(foi.num(Age,model.fp1$fitted.values))
model.fp2.foi<-foi.num(Age,model.fp2$fitted.values)
model.fp2.m.foi<-foi.num(Age,model.fp2.m$fitted.values)

lines(model.fp1.foi$grid,4*model.fp1.foi$foi,lwd=2)
lines(model.fp2.foi$grid,4*model.fp2.foi$foi,lwd=2,lty=2)
lines(model.fp2.m.foi$grid,4*model.fp2.m.foi$foi,lwd=2,lty=3)
abline(0,0,lwd=2)
axis(side=4,at=c(0.0,0.2,0.4),labels=c(0.00,0.05,0.10))
axis(side=2,at=seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=2)


### summary results
model.fp2.m.foi$foi[model.fp2.m.foi$foi==max(model.fp2.m.foi$foi)]
hav$Age[model.fp2.m.foi$foi==max(model.fp2.m.foi$foi)]
model.fp1$deviance-model.fp2.m$deviance

model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^1.9) + I(Age^2), family=binomial(link="logit"))
summary(model.fp2)

library(stats4)
expit=function(u) 1/(1+exp(-u))
ll=function(b0,b1,b2) -sum(log(dbinom(pos,Tot,prob=expit(b0+b1*Age^(1.9)+b2*Age^(2.0)))))
fit=mle(ll,start=list(b0=-1.09,b1=0.02,b2=-0.01))
summary(fit)


ll3=function(b) -sum(log(dbinom(pos,Tot,prob=expit(b[1]+b[2]*Age^(1.9)+b[3]*Age^(2.0)))))
fit=optim(c(1.09,0.02,-0.01),ll3)
summary(fit)

ll=function(b0max=-1.09,b1max=0.02,b2max=-0.01) -sum(log(dbinom(pos,Tot,prob=expit(b0max+b1max*Age^(1.9)+b2max*Age^(2.0)))))
fit=mle(ll)
summary(fit)

expit=function(u) 1/(1+exp(-u))
ll=function(b0max=-1.09474,b1max=0.02623,p1max=1.9,b2max=-0.01613,p2max=2.0) -sum(log(dbinom(pos,Tot,prob=expit(b0max+b1max*Age^(p1max)+b2max*Age^(p2max)))))
fit=mle(ll)
fit=mle(ll,method="CG")
summary(fit)

ll5=function(b) -sum(log(dbinom(pos,Tot,prob=expit(b[1]+b[2]*Age^(b[3])+b[4]*Age^(b[5])))))
fit=optim(c(-1,0.02,1.9,-0.01,2.0),ll5)

### HAV - Belgium
##################
hav<-read.table("c:/book/chapter4/HAV-BE.dat",header=T)
names(hav)
Age<-hav$Age
Pos<-hav$Neg
Neg<-hav$Pos
Tot<-hav$Tot
search.fracpoly.one(y=hav$Neg,tot=hav$Tot,x=hav$Age)
Agep=hav$Age^(0.42)
fitp1=glm(cbind(hav$Neg,hav$Tot-hav$Neg)~Agep,family=binomial(link="logit"))
fitlin=glm(cbind(hav$Neg,hav$Tot-hav$Neg)~hav$Age,family=binomial(link="logit"))
fitlin$deviance-fitp1$deviance
search.fracpoly.two(y=hav$Neg,tot=hav$Tot,x=hav$Age)
search.fracpoly.twoR(y=hav$Neg,tot=hav$Tot,x=hav$Age,seq(-2,3,0.1),T)
fpres=search.fracpoly.twoF(y=hav$Neg,tot=hav$Tot,x=hav$Age,seq(-2,3,0.1),T)
#edit(fpres$pow.mon)
#edit(fpres$pow.all)
fpres=search.fracpoly.twoF2v2(y=hav$Neg,tot=hav$Tot,x=hav$Age,seq(-2,3,0.1),T)
#edit(fpres$pow.mon)
#edit(fpres$pow.all)

fpres=search.fracpoly.twoF2(y=hav$Neg,tot=hav$Tot,x=hav$Age,seq(-2,3,0.1),T)
edit(fpres$pow.mon)
edit(fpres$pow.all)
mat=fpres$pow.all
mat[(mat[,3]>700),]=NA

library(scatterplot3d)

model.fp1<-glm(cbind(pos,Tot-pos) ~ Age, family=binomial(link="logit"))
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^1.9) + I(Age^2), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age) + I(Age^1.6), family=binomial(link="logit"))


### FIGURE 6.6
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(3, 0.5, 0))

s3d=scatterplot3d(mat[,1],mat[,2],mat[,3],pch=4,type="p",xlab="power 1",ylab=" ",zlab="deviance",color="grey")
text(5.5,1,"power 2")
#s3d$points3d(rep(mat[1,1],2),rep(mat[1,2],2),c(0,mat[1,3]),type="l",pch=16,col="green",lwd=3)
#s3d$points3d(mat[1,1],mat[1,2],0,pch=16,col="grey",lwd=3)
#s3d$points3d(mat[,1],mat[,2],rep(mat[1,3],length(mat[,3])),pch=16,col="grey")
s3d$points3d(mat[1,1],mat[1,2],mat[1,3],pch=16,col="black",lwd=3)
s3d$points3d(mat[1,1],mat[1,2],0,pch=16,col="black",lwd=3)
s3d$points3d(rep(mat[1,1],length(seq(-2,mat[1,2],0.1))),seq(-2,mat[1,2],0.1),rep(0,length(seq(-2,mat[1,2],0.1))),pch=16,col="black",type="l",lwd=3)
s3d$points3d(seq(mat[1,1],3,0.1),rep(mat[1,2],length(seq(mat[1,1],3,0.1))),rep(0,length(seq(mat[1,1],3,0.1))),pch=16,col="black",type="l",lwd=3)
s3d$points3d(rep(mat[1,1],2),rep(mat[1,2],2),c(0,mat[1,3]),type="l",pch=16,col="black",lwd=3)
sel=(mat[,4]==1)
s3d$points3d(mat[sel,1],mat[sel,2],mat[sel,3],pch=4,type="p",col="black")
s3d$points3d(mat[sel,1][1],mat[sel,2][1],mat[sel,3][1],pch=16,col="grey",lwd=3)
s3d$points3d(mat[sel,1][1],mat[sel,2][1],0,pch=16,col="grey",lwd=3)
s3d$points3d(rep(mat[sel,1][1],length(seq(-2,mat[sel,2][1],0.1))),seq(-2,mat[sel,2][1],0.1),rep(0,length(seq(-2,mat[sel,2][1],0.1))),pch=16,col="grey",type="l",lwd=3)
s3d$points3d(seq(mat[sel,1][1],3,0.1),rep(mat[sel,2][1],length(seq(mat[sel,1][1],3,0.1))),rep(0,length(seq(mat[sel,1][1],3,0.1))),pch=16,col="grey",type="l",lwd=3)
s3d$points3d(rep(mat[sel,1][1],2),rep(mat[sel,2][1],2),c(0,mat[sel,3][1]),type="l",pch=16,col="grey",lwd=3)

### FIGURE 6.5 (left panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(hav$Age,hav$Neg/hav$Tot,cex=0.03*hav$Tot,xlab="age",ylab="seroprevalence",xlim=c(0,86),ylim=c(-0.2,1),yaxt="n")

model.fp1<-glm(cbind(hav$Neg,hav$Tot-hav$Neg) ~ I(hav$Age^0.42), family=binomial(link="logit"))
model.fp2<-glm(cbind(hav$Neg,hav$Tot-hav$Neg) ~ I(hav$Age^1.9)+I(hav$Age^2.0), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(hav$Neg,hav$Tot-hav$Neg) ~ I(hav$Age^1.0)+I(hav$Age^1.6), family=binomial(link="logit"))

lines(hav$Age,model.fp1$fitted.values,lwd=2)
lines(hav$Age,model.fp2$fitted.values,lwd=2,lty=2)
lines(hav$Age,model.fp2.m$fitted.values,lwd=2,lty=3)

model.fp1.foi<-c(foi.num(hav$Age,model.fp1$fitted.values))
model.fp2.foi<-foi.num(hav$Age,model.fp2$fitted.values)
model.fp2.m.foi<-foi.num(hav$Age,model.fp2.m$fitted.values)

lines(model.fp1.foi$grid,4*model.fp1.foi$foi,lwd=2)
lines(model.fp2.foi$grid,4*model.fp2.foi$foi,lwd=2,lty=2)
lines(model.fp2.m.foi$grid,4*model.fp2.m.foi$foi,lwd=2,lty=3)

#abline(0,0)
#axis(side=4,at=c(-0.2,0.0,0.2,0.4),labels=c(-0.05,0.00,0.05,0.10))
#axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))

abline(0,0,lwd=2)
axis(side=4,at=c(0.0,0.2,0.4),labels=c(0.00,0.05,0.10))
axis(side=2,at=seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=2)

### summary results

model.fp2.m.foi$foi[model.fp2.m.foi$foi==max(model.fp2.m.foi$foi)]
hav$Age[model.fp2.m.foi$foi==max(model.fp2.m.foi$foi)]



##################################################
## Section 6.2: Fractional Polynomial Models     #
##								 #
## 6.2.6 Influence of the link function          #
##################################################

### data VZV-BE

parvovirus<-read.table("c:/book/chapter4/VZV-B19-BE.dat",header=T)
subset<-(parvovirus$age>0.5)&(parvovirus$age<76)&(!is.na(parvovirus$age))&!is.na(parvovirus$VZVres)
parvovirus<-parvovirus[subset,]
y<-parvovirus$VZVres[order(parvovirus$age)]
a<-parvovirus$age[order(parvovirus$age)]

grid<-sort(unique(round(a)))
neg<-table(y,round(a))[1,]
pos<-table(y,round(a))[2,]
tot<-neg+pos

search.fracpoly.one(y=pos,tot=tot,x=grid)
search.fracpoly.two(y=pos,tot=tot,x=grid)
fpres=search.fracpoly.twoF2(y=pos,tot=tot,x=grid,seq(-2,3,0.1),T)
edit(fpres$pow.mon)
edit(fpres$pow.all)

fpres=search.fracpoly.twoF2v2(y=pos,tot=tot,x=grid,seq(-2,3,0.1),T)
edit(fpres$pow.mon)
edit(fpres$pow.all)
head(fpres$pow.mon)


Age<-grid
pos<-pos
Tot<-tot
model.fp1<-glm(cbind(pos,Tot-pos) ~ I(Age^0.1), family=binomial(link="logit"))
model.lin=glm(cbind(pos,Tot-pos) ~ Age, family=binomial(link="logit"))
model.lin$deviance-model.fp1$deviance
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^(-1.2)) + I(Age^(-1.1)), family=binomial(link="logit"))
#model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age^(-0.8)) + I(Age^(-0.5)), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age^(-0.6)) +  I(Age^(-0.2)), family=binomial(link="logit"))

age.grid=seq(min(Age),max(Age),by=0.01)
fp1.fitted=predict(model.fp1,newdata=data.frame(Age=age.grid),type="response")
fp2.fitted=predict(model.fp2,newdata=data.frame(Age=age.grid),type="response")
fp2.m.fitted=predict(model.fp2.m,newdata=data.frame(Age=age.grid),type="response")

### FIGURE 6.7
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,pos/Tot,cex=0.02*Tot,xlab="age",ylab="seroprevalence",ylim=c(-0.2,1),yaxt="n")
lines(age.grid,fp1.fitted,lwd=2)
lines(age.grid,fp2.fitted,lwd=2,lty=2)
lines(age.grid,fp2.m.fitted,lwd=2,lty=3)

model.fp1.foi<-foi.num(age.grid,fp1.fitted)
model.fp2.foi<-foi.num(age.grid,fp2.fitted)
model.fp2.foi.m<-foi.num(age.grid,fp2.m.fitted)

lines(model.fp1.foi$grid,model.fp1.foi$foi/2,lwd=2)
lines(model.fp2.foi$grid,model.fp2.foi$foi/2,lwd=2,lty=2)
lines(model.fp2.foi.m$grid,model.fp2.foi.m$foi/2,lwd=2,lty=3)
axis(side=4,at=c(-0.2,0.0,0.2,0.4),labels=c(-0.2,0.0,0.2,0.4)*2)
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))
abline(h=0,lwd=2)
mtext(side=4,"force of infection", las=3,line=1.5)


### Data  Rubella
##################
rubella<-read.table("c:/book/chapter4/rubella-UK.dat",header=T)
rubella$Tot<-rubella$Pos+rubella$Neg
names(rubella)
search.fracpoly.one(y=rubella$Pos,tot=rubella$Tot,x=rubella$Age)
search.fracpoly.two(y=rubella$Pos,tot=rubella$Tot,x=rubella$Age)
fpres=search.fracpoly.twoF(y=rubella$Pos,tot=rubella$Tot,x=rubella$Age,seq(-2,3,0.1),T)
fpres=search.fracpoly.twoF2v2(y=rubella$Pos,tot=rubella$Tot,x=rubella$Age,seq(-2,3,0.1),T)
edit(fpres$pow.mon)
edit(fpres$pow.all)


Age<-rubella$Age
pos<-rubella$Pos
Tot<-rubella$Tot
model.fp1<-glm(cbind(pos,Tot-pos) ~ I(Age^0.05), family=binomial(link="logit"))
model.lin<-glm(cbind(pos,Tot-pos) ~ Age, family=binomial(link="logit"))
gain=model.lin$deviance-model.fp1$deviance
gain
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^(-0.9)) + I(Age^(-0.9)*log(Age)), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age^(-0.9)) + I(Age^(-0.4)), family=binomial(link="logit"))


age.grid=seq(min(Age),max(Age),by=0.01)
fp1.fitted=predict(model.fp1,newdata=data.frame(Age=age.grid),type="response")
fp2.fitted=predict(model.fp2,newdata=data.frame(Age=age.grid),type="response")
head(fp2.fitted)
fp2.m.fitted=predict(model.fp2.m,newdata=data.frame(Age=age.grid),type="response")


### FIGURE 6.8 (left panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,pos/Tot,cex=0.018*Tot,xlab="age",ylab="seroprevalence",ylim=c(-0.2,1),yaxt="n")
lines(age.grid,fp1.fitted,lwd=2)
lines(age.grid,fp2.fitted,lwd=2,lty=2)
lines(age.grid,fp2.m.fitted,lwd=2,lty=3)

model.fp1.foi<-foi.num(age.grid,fp1.fitted)
model.fp2.foi<-foi.num(age.grid,fp2.fitted)
model.fp2.m.foi<-foi.num(age.grid,fp2.m.fitted)

lines(model.fp1.foi$grid,model.fp1.foi$foi,lwd=2)
lines(model.fp2.foi$grid,model.fp2.foi$foi,lwd=2,lty=2)
lines(model.fp2.m.foi$grid,model.fp2.m.foi$foi,lwd=2,lty=3)
abline(h=0,lwd=2)
axis(side=4,at=c(-0.2,0.0,0.2),labels=c(-0.2,0.0,0.2))
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=1.5)

### Data Mumps
###############
mumps<-read.table("c:/book/chapter4/MUMPSUK.dat",header=T)

search.fracpoly.one(y=mumps$pos,tot=mumps$ntot,x=mumps$age)
search.fracpoly.two(y=mumps$pos,tot=mumps$ntot,x=mumps$age)
fpres=search.fracpoly.twoF(y=mumps$pos,tot=mumps$ntot,x=mumps$age,seq(-2,3,0.1),T)
fpres=search.fracpoly.twoF2v2(y=mumps$pos,tot=mumps$ntot,x=mumps$age,seq(-2,3,0.1),T)
edit(fpres$pow.mon)
edit(fpres$pow.all)

Age<-mumps$age
pos<-mumps$pos
Tot<-mumps$ntot
model.fp1<-glm(cbind(pos,Tot-pos) ~ I(Age^(-0.17)), family=binomial(link="logit"))
model.lin=glm(cbind(pos,Tot-pos) ~ Age, family=binomial(link="logit"))
model.lin$deviance-model.fp1$deviance
model.fp2<-glm(cbind(pos,Tot-pos) ~ I(Age^(-2)) + I(Age^(-0.8)), family=binomial(link="logit"))
model.fp2.m<-glm(cbind(pos,Tot-pos) ~ I(Age^(-1)) + I(Age^(-1)*log(Age)), family=binomial(link="logit"))

age.grid=seq(min(Age),max(Age),by=0.01)
fp1.fitted=predict(model.fp1,newdata=data.frame(Age=age.grid),type="response")
fp2.fitted=predict(model.fp2,newdata=data.frame(Age=age.grid),type="response")
head(fp2.fitted)
fp2.m.fitted=predict(model.fp2.m,newdata=data.frame(Age=age.grid),type="response")


### FIGURE 6.8 (right panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,pos/Tot,cex=0.008*Tot,xlab="age",ylab="seroprevalence",ylim=c(-0.2,1),yaxt="n")
lines(age.grid,fp1.fitted,lwd=2)
lines(age.grid,fp2.fitted,lwd=2,lty=2)
lines(age.grid,fp2.m.fitted,lwd=2,lty=3)

model.fp1.foi<-foi.num(age.grid,fp1.fitted)
model.fp2.foi<-foi.num(age.grid,fp2.fitted)
model.fp2.m.foi<-foi.num(age.grid,fp2.m.fitted)
abline(h=-0,lwd=2)

lines(model.fp1.foi$grid,model.fp1.foi$foi/2,lwd=2)
lines(model.fp2.foi$grid,model.fp2.foi$foi/2,lwd=2,lty=2)
lines(model.fp2.m.foi$grid,model.fp2.m.foi$foi/2,lwd=2,lty=3)

axis(side=4,at=c(-0.2,0.0,0.2),labels=c(-0.2,0.0,0.2)/2)
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=1.6)

### summary results
edit(cbind(model.fp2.m.foi$grid,model.fp2.m.foi$foi))


### ANALYSES WITH COMPLEMENTARY LOG-LOG LINK
############################################
rm(list=ls(all=TRUE))

search.fracpoly.one.cll<-function(y,tot,x){
     pow1<-seq(-2,3,0.01)
     deviance<-deviance(glm(cbind(y,tot-y)~x, family="binomial"(link=cloglog)))
     power<-1
     mistake<-NULL

     for (i in 1: (length(pow1))){
          if(pow1[i]==0){term1<-log(x)} else{term1<-(x)^(pow1[i])}
          glm.try<-glm(cbind(y,tot-y)~term1, family="binomial"(link=cloglog))
          if(glm.try$converged==FALSE){mistake<-rbind(mistake, c(1,pow1[i]))}
          else{
               if(deviance(glm.try)<deviance){
                    deviance<-deviance(glm.try)
                    power<-pow1[i]
                                   }
               }
                                     }
     return(list(power=power, deviance=deviance, mistake=mistake))
}


### Listing all fits and symmetric in the powers
search.fracpoly.twoF2.cll<-function(y,tot,x,pow,mc){
     deviance<-deviance(glm(cbind(y,tot-y)~x+I(x^2), family="binomial"(link=cloglog)))
     mistake<-NULL
     res=matrix(NA,nrow=length(pow)*length(pow),ncol=4)
     k=1
     for (i in 1: (length(pow))){
     for (j in i: (length(pow))){
          if(pow[i]==0){term1<-log(x)} else{term1<-(x)^(pow[i])}
          if(pow[j]==pow[i]){term2<-term1*log(x)} 
          else if(pow[j]==0){term2<-log(x)} 
          else{term2<-(x)^(pow[j])}
          glm.try<-glm(cbind(y,tot-y)~term1+term2, family="binomial"(link=cloglog))
          if(glm.try$converged!=FALSE){
             res[k,]=c(pow[i],pow[j],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],deviance(glm.try),(sum(diff(predict(glm.try))<0)==0))
             k=k+1
             }
             } 
          else{
             res[k,]=c(pow[i],pow[j],NA,NA)
             k=k+1
             if (i!=j){
             res[k,]=c(pow[j],pow[i],NA,NA)
             k=k+1
             }
             }
      }
      }
     res1=res[res[,4]==1,]
     res1=res1[order(res1[,3]),]
     res2=res[order(res[,3]),]
     return(list(pow.mon=res1,pow.all=res2))
}



foi.num<-function(x,p)
{
grid<-sort(unique(x))
pgrid<-(p[order(x)])[duplicated(sort(x))==F]
dp<-diff(pgrid)/diff(grid)
foi<-approx((grid[-1]+grid[-length(grid)])/2,dp,grid[c(-1,-length(grid))])$y/(1-pgrid[c(-1,-length(grid))])
return(list(grid=grid[c(-1,-length(grid))],foi=foi))
}


# HAV - Belgium
###############

hav<-read.table("c:/book/chapter4/HAV-BE.dat",header=T)
names(hav)

Age=hav$Age
Pos=hav$Neg
Tot=hav$Tot
Neg=hav$Pos

search.fracpoly.one(y=Pos,tot=Tot,x=Age)
model.fp1=glm(cbind(Pos,Tot-Pos)~I(Age^(0.42)),family=binomial(link="logit"))

search.fracpoly.one.cll(y=Pos,tot=Tot,x=Age)
model.fp1.cll=glm(cbind(Pos,Tot-Pos)~I(Age^(-0.04)),family=binomial(link="cloglog"))


fpres=search.fracpoly.twoF2(y=Pos,tot=Tot,x=Age,seq(-2,3,0.1),T)
edit(fpres$pow.mon) # 1.0 1.6 93.45
edit(fpres$pow.all) # 1.9 2.0 79.58

fpres=search.fracpoly.twoF2.cll(y=Pos,tot=Tot,x=Age,seq(-2,3,0.1),T)
edit(fpres$pow.mon) #0.5 1.1 105.9586
edit(fpres$pow.all) #1.5 1.6 81.6033

model.fp2=glm(cbind(Pos,Tot-Pos)~I(Age^(1.9))+I(Age^(2.0)),family=binomial(link="logit"))
model.fp2.m=glm(cbind(Pos,Tot-Pos)~I(Age^(1.0))+I(Age^(1.6)),family=binomial(link="logit"))

model.fp2.cll=glm(cbind(Pos,Tot-Pos)~I(Age^(1.5))+I(Age^(1.6)),family=binomial(link="cloglog"))
model.fp2.m.cll=glm(cbind(Pos,Tot-Pos)~I(Age^(0.5))+I(Age^(1.1)),family=binomial(link="cloglog"))

model.fp1.clllog=glm(cbind(Pos,Tot-Pos)~I(log(Age)),family=binomial(link="cloglog"))
model.fp1.clllogB=glm(cbind(Pos,Tot-Pos)~1,offset=log(Age),family=binomial(link="cloglog"))




age.grid=seq(min(Age),max(Age),by=0.01)
fp1.fitted=predict(model.fp1,newdata=data.frame(Age=age.grid),type="response")
fp1.cll.fitted=predict(model.fp1.cll,newdata=data.frame(Age=age.grid),type="response")
fp2.fitted=predict(model.fp2,newdata=data.frame(Age=age.grid),type="response")
fp2.m.fitted=predict(model.fp2.m,newdata=data.frame(Age=age.grid),type="response")
fp2.cll.fitted=predict(model.fp2.cll,newdata=data.frame(Age=age.grid),type="response")
fp2.m.cll.fitted=predict(model.fp2.m.cll,newdata=data.frame(Age=age.grid),type="response")



### FIGURE 6.9 (left panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,Pos/Tot,cex=0.03*Tot,xlab="age",ylab="seroprevalence",ylim=c(-0.2,1),yaxt="n")
lines(age.grid,fp1.fitted,lwd=2)
lines(age.grid,fp1.cll.fitted,lwd=2,lty=2)
model.fp1.foi<-foi.num(age.grid,fp1.fitted)
model.fp1.cll.foi<-foi.num(age.grid,fp1.cll.fitted)
abline(h=0,lwd=2)
lines(model.fp1.foi$grid,4*model.fp1.foi$foi,lwd=2)
lines(model.fp1.cll.foi$grid,4*model.fp1.cll.foi$foi,lwd=2,lty=2)
axis(side=4,at=c(-0.2,0.0,0.2,0.4),labels=c(-0.05,0.00,0.05,0.1))
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=1.9)


### FIGURE 6.9 (right panel)
windows(record=TRUE, width=5, height=5)
par(las=1,cex.axis=1.1,cex.lab=1.1,lwd=3,mgp=c(2, 0.5, 0),mar=c(4.1,4.1,4.1,3))

plot(Age,Pos/Tot,cex=0.03*Tot,xlab="age",ylab="seroprevalence",ylim=c(-0.2,1),yaxt="n")
lines(age.grid,fp2.fitted,lwd=2)
lines(age.grid,fp2.m.fitted,lwd=2)
lines(age.grid,fp2.cll.fitted,lwd=2,lty=2)
lines(age.grid,fp2.m.cll.fitted,lwd=2,lty=2)

model.fp2.foi<-foi.num(age.grid,fp2.fitted)
model.fp2.m.foi<-foi.num(age.grid,fp2.m.fitted)
model.fp2.cll.foi<-foi.num(age.grid,fp2.cll.fitted)
model.fp2.m.cll.foi<-foi.num(age.grid,fp2.m.cll.fitted)
abline(h=0,lwd=2)
lines(model.fp2.foi$grid,4*model.fp2.foi$foi,lwd=2)
lines(model.fp2.m.foi$grid,4*model.fp2.m.foi$foi,lwd=2)
lines(model.fp2.cll.foi$grid,4*model.fp2.cll.foi$foi,lwd=2,lty=2)
lines(model.fp2.m.cll.foi$grid,4*model.fp2.m.cll.foi$foi,lwd=2,lty=2)

axis(side=4,at=c(-0.2,0.0,0.2,0.4),labels=c(-0.05,0.00,0.05,0.1))
axis(side=2,at= seq(0,1,0.2),labels=seq(0,1,0.2))
mtext(side=4,"force of infection", las=3,line=2)


edit(cbind(model.fp2.m.foi$grid,model.fp2.m.foi$foi))




