%% Tutorial 1

% Calculate the maximum of a spectrum of an AR[2]-process and compare it to the frequency obtained from the coefficients a_1 and a_2. 
% Compare the results to the spectral estimate (based on an AR spectrum).


clear all; close all;

% Set the parameters of an AR[2] and simulate the process

T=10
tau=5

a1=2*cos(2*pi/T)*exp(-1/tau);
a2=-exp(-2/tau);

x=zeros(2000,1);

for i=3:2000
    x(i)=a1*x(i-1)+a2*x(i-2)+randn(1);
end

x=x(1001:2000); % delete transient behaviour

% The frequency of the process should be f=1/T or omega 2*pi/T



% Begin Maximum Likelihood Estimation of the parameters

A=zeros(length(x)-2,3);

for i=1:length(x)-2
  for j=1:3
    A(i,j)=x(i+3-j);
  end
end

b=A(:,1);
A=A(:,2:3);

R=A'*A;
  
ar=zeros(2,1);
ar=inv(R)*A'*b

e=b-A*ar;
S=e'*e/(length(x)-2)

% End Maximum Likelihood Estimation of the parameters


% Caluclate the spectrum based on the AR coefficients
[spc,f]=specAR(x,ar,S,1);

% Plot the spectrum
semilogy(f,spc)
xlabel('Frequency');
ylabel('log Spectrum');

fmax=f(find(spc==max(spc))) % Maximum of the spectrum as frequency f=1/T, compare with T=10 in the beginning

Tmax=1/fmax


% The analytic derivation gives
%
% abs(1-a1*exp(i*omega)-a2*exp(2*i*omega)) should be minimal
%
% this leads to
%
% cos(2*pi/Tmax)=cosh(1/tau)*cos(2*pi/T)
%

Tmaxana=2*pi/acos(cosh(1/tau)*cos(2*pi/T))

% Note that due to the parameter tau in the model, the frequency of the
% maximum of the spectrum is not exactly the frequency 1/T


%% Tutorial 2
% 2. Use the Kalman filter to track a parameter of the model x(t)=a(t) x(t-1) + noise contaminated with observational noise, i.e. y(t)=x(t)+noise.

clear all; close all;

% Set all the initial condictions etc. Please note that due to the
% nonlinearity we need to use the extended Kalman filter here
x(:,1) = [0;0.5];          
x_(:,1) = [0;0.4];     

P = [0.01 0;            
     0     0.01];
 
sigmav = 1;           
sigmaw = 1;           
Q = sigmav*sigmav;      
R = sigmaw*sigmaw;      

G = [1;0];              
H = [ 1 0];             
W = 1;                  

steps = 1000;  
deltat=1;

% As it is simulated data, we can do the simulation as well as the fitting
% in one loop. It also demonstrates that the Kalman filter is an online
% algorithm

for i =2:steps          
  x(:,i) = [x(2,i-1)*x(1,i-1) + randn*sigmav ; x(2,i-1) ];
  % The state space model is augmented. 
  
  if i==500
      x(2,i)=0.48;  % After 500 interations the parameter a changes
  end
  
  z(i) = x(1,i) + randn*sigmaw; % only x(t) is observed, not a, but contaminated with observational noise
  
  
  x_(:,i) = [x_(2,i-1)*x_(1,i-1); x_(2,i-1)]; % best guess about the hidden state
  z_(i) = x_(1,i); % best guess about the observation

  
  F = [x(2,i) x(1,i);
       0        1  ]; % Due to the nonlineariy, first order derivatives need to be calculated.
   
  P = F*P*F' + G*Q*G';% Prediction covariance
  S = H*P*H'+R; % Innovation Covariance
  K = (P*H')/S; % Kalman gain
  x_(:,i) = x_(:,i) + K * (z(i)-z_(i)); % Udate step
  P = (eye(2)-K*H)*P;% Update of the covariance matrix
  
  sigmaP(:,i)=sqrt(diag(P)); %Contains the uncertainties in the estimation
end

figure(1)
plot(x(1,:),'-k');                  %plot the true x(t)
plot(z,'.r');                       %plot the observations 
plot(x_(1,:),'-b');                 %plot the Kalman filter prediction 
plot(x_(1,:)+2*sigmaP(1,:),'-b');   %Confidence intervals
plot(x_(1,:)-2*sigmaP(1,:),'-b');


figure(2)
plot(x(2,:),'-b');                  %Plot the time course of a


%% Tutorial 3

% 3. Estimate the coherence, partial coherence, and partial directed coherence for the following set of equations:
%    x_1 (t) =epsilon_1 (t)
%    x_2 (t) =epsilon_2 (t)
%    x_3 (t) = x_1 (t-1) + x_2 (t-1)

% Simulate the system

x=randn(2000,3);
for i=2:2000
    x(i,3)=x(i-1,1)+x(i-1,2);
end

x=x(1001:2000,:); % delete transient behaviour

sig_level = 0.05; % alpha level for the critical values (statistic)

N=length(x(:,1));

%estimate smoothed periodogramm berechnen
[smp_full nu] = compute_SMP(x,length(x(1,:)),1,0.04,'none');

%keep only the positive frequencies
smp = smp_full(:,:,N/2+1:end);

CS = NaN(length(x(1,:)),length(x(1,:)),N/2);
for iomega= 1:N/2
  CS(:, :, iomega)= compute_partialized(smp(:, :, iomega)); % estimate also the partial spectral information
end
COH = abs(CS);

%Signifikanzlevel
sig_level = sqrt(1-sig_level^(2/(nu-2)));


% estimate multivariate AR model
obj = arfit_init(1);
obj = arfit_update(obj, x);
out = arfit_coeff(obj)

[pdc_est,f]=pdc(x(:,1),out.A,1); %Calculate PDC

% plot results
figure(1)
for i=1:3
    for j=1:3
      subplot(3,3,(i-1)*3+j)
      plot(f,squeeze(COH(i,j,:)))
      hold on
      plot(f,squeeze(sig_level*ones(length(COH(1,1,:)))))
      if i<j  
        ylabel('Coherence');
      elseif i>j
        ylabel('Partial Coherence')
      end
      xlabel('Frequency');
      axis([0 0.5 0 1])
    end
end


figure(2)
for i=1:3
    for j=1:3
      subplot(3,3,(i-1)*3+j)
      plot(f,squeeze(pdc_est(i,j,:)))
      ylabel(sprintf('PDC %d -> %d',j,i));
      xlabel('Frequency');
      axis([0 0.5 0 1])
    end
end


%%Tutorial 4

% 4. Estimate the order of an AR[2] process using the partial autocovariance function. Compare the results to the one obtained for an MA[2] process. 

% Simulate the data
T=10
tau=5

a1=2*cos(2*pi/T)*exp(-1/tau);
a2=-exp(-2/tau);

x=zeros(2000,1);

for i=3:2000
    x(i)=a1*x(i-1)+a2*x(i-2)+randn(1);
end

x=x(1001:2000); % delete transient behaviour


N = length(x);
x = x - mean(x);

k=10;

acorr = acf(x,k);  % Caluclate ACF
[pacorr] = pacf(acorr); % Calculate PACF

% confidence intervals
s = 1./sqrt(repmat(N,1,k)-1-(1:k));
alpha = 0.05;
cu = norminv(alpha/2).*s;
cl = -cu;        

% plot results
figure
subplot(2,1,1)
plot(acorr(2:end)/var(x)/N)
hold on
plot(cu,'r')
plot(cl,'r')
xlabel('Lag');
ylabel('ACF');
subplot(2,1,2)
plot(pacorr)
hold on
plot(cu,'r')
plot(cl,'r')
xlabel('Lag');
ylabel('PACF');

% the same for the MA process
a1=1
a2=0.4

eta=randn(2000,1);
x=zeros(2000,1);

for i=3:2000
    x(i)=a1*eta(i-1)+a2*eta(i-2)+eta(i);
end

x=x(1001:2000);


N = length(x);
x = x - mean(x);

k=10;

acorr = acf(x,k);
[pacorr] = pacf(acorr); 

% confidence intervals
s = 1./sqrt(repmat(N,1,k)-1-(1:k));
alpha = 0.05;
cu = norminv(alpha/2).*s;
cl = -cu;        

%plot results
figure
subplot(2,1,1)
plot(acorr(2:end)/var(x)/N)
hold on
plot(cu,'r')
plot(cl,'r')
xlabel('Lag');
ylabel('ACF');
subplot(2,1,2)
plot(pacorr)
hold on
plot(cu,'r')
plot(cl,'r')
xlabel('Lag');
ylabel('PACF');



