function  [SPECresults, varargout]= compute_SMP(parameters,varargin)
% Computation of the Smoothed Periodogram (SMP) of a multivariate process with time series Xts and point process Xpp components. 
%
%% usage without FDMa toolbox
%function [smp nu nu_taper nu_smoother] = compute_SMP(data,Dimension,Fs,SMPh0Hz,tapertype,taperparameter)
% Fs: sampling frequency in Hz (optional, default 1)
% SMPh0Hz: halfwidth of smoothing window in Hz (optional, default 8 frequency bins, set to 'default' for using default if other optional parameters are needed)
% tapertype is optional (Options are: 'none','raisedcosine','triangular')
% tapertype 'raisedcosine' is used as default
% taperparameter is needed for tapertype 'raisedcosine' only (default: 1/32 relative length of roll-off)
%
%% usage within the FDMa toolbox
%function [SPECresults, smp]= compute_SMP(parameters)
%
% Computes the Smoothed Periodogram of the multivariate signal X. 
% Tapering and smoothing is defined in here. 
% Also performs Wilson factorization of the Smoothed Periodogram. 
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Check usage of FDMa 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if isstruct(parameters)
    using_FDMa = 1;
    SPECresults.timing= cputime;
else
    using_FDMa = 0;
    analyze.SMP = 1;
    if nargin == 1
        error('At least one input argument is missing. Input should contain: data, Dimension, optional paramters');
    else
        D = varargin{1};
        Dts = D;
        %Check Dimension
        [dim1 dim2] = size(parameters);
        if dim1 == D
            N = dim2;
            Xts = parameters';
        elseif dim2 == D
            N = dim1;
            Xts = parameters;
        else
            error('Dimensionparameter (%d) and dimensions of data (%d x %d) are not consistent.',D,dim1,dim2);
        end
        if nargin > 2
            Fs = varargin{2};
        else
            Fs = 1;
        end;
        if nargin > 3
            analyze.SMPh0Hz = varargin{3};
            analyze.tapertype = 'raisedcosine';
            analyze.taperparameter = 1/32;% relative length of roll-off
            set_h_to_8 = 0;
        else
            analyze.SMPh0Hz = 'default';
            analyze.tapertype = 'raisedcosine';
            analyze.taperparameter = 1/32;% relative length of roll-off
            set_h_to_8 = 1;
        end
        if strcmp(analyze.SMPh0Hz,'default')
            set_h_to_8 = 1;
        end
        if nargin > 4
            analyze.tapertype = varargin{4};
            if nargin > 5
                analyze.taperparameter = varargin{5};
            else
                analyze.taperparameter = [];
            end
        end
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Extract parameters 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if using_FDMa
    vars= genvarname(fieldnames(parameters));
    for i= 1:length(vars)
       eval([vars{i} '= parameters.(vars{i});'])
    end
    
    shortname= 'SMP';
    compute_SPEC_description
end

if(analyze.SMP)
   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Taper Design 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   [SPECresults.taperwin, SPECresults.nutaper]= compute_taper((0:N-1)/Fs, analyze.tapertype, analyze.taperparameter, N, Fs);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Smoother Design
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(using_FDMa && analyze.SMPh0Hz)
        h= floor(analyze.SMPh0Hz/Fs*N);
   elseif(using_FDMa && ~analyze.SMPh0Hz)
      if(isempty(analyze.f0))
         h= 8;
      else
         h= min(ceil(N/32), ceil(min(diff([0; analyze.f0(:)]))/2/Fs*N));
      end
   elseif(~using_FDMa && set_h_to_8)
        h = 8;
   else
        h= floor(analyze.SMPh0Hz/Fs*N);
   end

   SPECresults.SMPh0Hz= h*Fs/N;
   SPECresults.h= h;
   [SPECresults.smootherwin, SPECresults.nusmoother]= compute_smoother('WIDTH', h);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Smoothed Periodogram of Xts and Xpp (pos and neg Frequencies)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   SMP= zeros(D, D, N);
   Xfft= fftshift(fft(Xts.*(SPECresults.taperwin*ones(1, Dts)), [], 1), 1);
   if using_FDMa
       for d= 1:Dpp
          [ppftpp, omega]= ppft(Xpp{d}, 2*N, 2*analyze.Nfreq, 2*Fs, analyze.tapertype, analyze.taperparameter);
          ppftpp= ppftpp(N/2+1:3*N/2);
          if(sum(abs(ppftpp).^2))
             ppftpp= ppftpp/sqrt(sum(abs(ppftpp).^2))*N;
          end
          Xfft= [Xfft ppftpp];
       end
   end   
   for d1= 1:D
      for d2= 1:D
         WXd1d2= conv(SPECresults.smootherwin, Xfft(:, d1).*conj(Xfft(:, d2))/N);
         WXd1d2= WXd1d2((length(SPECresults.smootherwin)+1)/2:end-(length(SPECresults.smootherwin)-1)/2);
         SMP(d1, d2, :)= WXd1d2;
      end
   end
   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if(using_FDMa && analyze.specsub>1)% Spectral Subsampling CHANGE!!!
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      smp= zeros(D, D, 2*analyze.Nfreq);
      for d1= 1:D
         for d2= 1:D
            smp(d1, d2, :)= SMP(d1, d2, 1:analyze.specsub:end);
         end
      end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   else%if(analyze.specsub>1)% Spectral Subsampling 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      smp= SMP;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   end%if(analyze.specsub>1)% Spectral Subsampling 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Degrees of Freedom with Smoothing and Tapering
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   SPECresults.nu= SPECresults.nusmoother*SPECresults.nutaper;
   SPECresults.message= sprintf('%4.2f Hz (%d, nu=%5.4f)', 2*SPECresults.SMPh0Hz, SPECresults.h, SPECresults.nu);
   

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% SMP and BAP post processing
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   if using_FDMa
       spec= smp;
       compute_SMP_BAP_postproc
   end
   
else
   SPECresults.SPEC= -inf;
   SPECresults.specmax= -inf;
   SPECresults.message= '';
end%if(analyze.SMP)

if using_FDMa
    SPECresults.timing= cputime-SPECresults.timing;
    varargout{1} = smp;
else
    varargout{1} = SPECresults.nu;
    varargout{2} = SPECresults.nutaper;
    varargout{3} = SPECresults.nusmoother;    
    SPECresults = smp;
end
