#######################################################
### Calculation of the SST-ratio following the SST
##  standard model for a simple example
#######################################################

## library for multivariate normal distributions
library(mvtnorm)     
time.start.user <-  proc.time()[["user.self"]]

#######################################################
### Input data
#######################################################

## repeatable random numbers
set.seed(1)       

## number of Monte Carlo simulations
N <-  100000
## safety level
alpha <-  0.01
## risk free spot rates
return.spot.chf <- c(0.00155, 0.00013, -0.00011) 
return.stock.chf <- 0.05

# cost of capital for market value margin
coc <- 0.06  
## mortality of the insured
q <-c(0.02,0.02,0.02)

## insured sum, survival to the end of period
IS <- c(0, 0,1000)  

## nominal values, i.e. cashflow
## -- assumption: no default risk
assets.bond.gov.chf <- c(0, 300, 500)

## value at the beginning of period 1
assets.stock.chf <- 300

## expected increase per period of value
stock.chf.incr <- 0.05 

### standard deviations provides by FINMA (2012)
SD <- c(  0.0060292643195745         # spot.chf[1]
        , 0.00605990680656078        # spot.chf[2]
        , 0.00633152336734695        # spot.chf[3]
        , 0.150517788474455          # stock.chf
        , 0.05                       # q
        )

### correlation matrix provided by FINMA (2012)
###                 spot.chf[1], spot.chf[2]
###                 , spot.chf[3], stock.chf, q
Corr <- matrix( c(  1.000000000, 0.721560179
                  , 0.545556472, 0.402239567, 0
                  , 0.721560179, 1.000000000
                  , 0.953194547, 0.433388849, 0
                  , 0.545556472, 0.953194547
                  , 1.000000000, 0.413041139, 0
                  , 0.402239567, 0.433388849
                  , 0.413041139, 1.000000000, 0
                  , 0.000000000, 0.000000000
                  , 0.000000000, 0.000000000, 1
                  ), ncol = 5)

### Scen: Data frame that describes the scenarios
##        to be added.
###       The following columns are necessary:
###       -- Target:      0 or 1 depending on whether 
###                       the particular scenario is
###                       included
###       -- Probability: Probability for this scenario
###       -- Delta_RTK:   Effect of this scenario on the
###                       risk bearing capital RTK)
Scen <-
    list( Name = c( "SZ01: Equity drop -60%"
            ,       "SZ03: Stock mkt crash (87)"
            ,       "SZ04: Nikkei crash (89/90)"
            ,       "SZ05: Europ. FX crisis (92)"
            ,       "SZ06: US interest crisis (94)"
            ,       "SZ07: Russian crisis/LTCM (98)"
            ,       "SZ08: Stock mkt crash (00/01)"
            ,       "SZ11: Financial crisis (08)"
            )
         , Target = c( 1, 1, 1, 1, 1, 1, 1, 1 )
         ##                SZ01   SZ03   SZ04   SZ05
         ##    SZ06   SZ07   SZ08   SZ11
         , Probability = c(0.001, 0.001, 0.001, 0.001
             , 0.001, 0.001, 0.001, 0.001)
         , Delta.x = rbind(
             ##  #spot.1   #spot.2  #spot.3   #stock #q
             c(  0.00000,  0.00000, 0.00000, -0.6000, 0)
             ,c(-0.00155, -0.00013, 0.00000, -0.2323, 0) 
             ,c( 0.01563,  0.01098, 0.01177, -0.2643, 0) 
             ,c(-0.00155, -0.00013, 0.00000, -0.0580, 0)
             ,c( 0.01109,  0.01406, 0.01509, -0.1852, 0)
             ,c(-0.00155, -0.00013, 0.00000, -0.2841, 0)
             ,c(-0.00155, -0.00013, 0.00000, -0.3567, 0) 
             ,c(-0.00155, -0.00013, 0.00000, -0.3881, 0)
             )
         , Delta_RTK = c()
         )

#######################################################
### Functions
#######################################################

### Expected Shortfall for an (unsorted) vector of
### random outcomes
### -- In this implementation the expected shortfall 
###    is taken with respect to the left tail. (In the 
###    literature it is often taken with respect to
###    the right tail.)
ES <- function(x,perc){
    VaR <- quantile(x,perc,names=FALSE)
    sum(x[x<VaR])/(length(x)*perc)
}

### survival to the end of the period
Survival <- function(q) cumprod(1-q)               

### discount to the beginning of the first period
Discount <-function(s)  1/(1+s)^(1:length(s))       

### valuation of government bonds
bond.gov.value <-  function(ZB,date,s){
    T <- length(s)
    if (length(s) != length(ZB)){
        return(NA)
    }else{ 
        return(  sum(Discount(s)[date:T]*ZB[date:T])
               / if (date == 1) 1
               else Discount(s)[date-1]
               )
    }
}

# risk bearing capital at the beginning of period "year"
value.assets <- function(x,year){
return(  bond.gov.value(  assets.bond.gov.chf
                        , year
                        , x[i.spot.chf]
                        )
       + (1+stock.chf.incr)^(year-1)
       * assets.stock.chf
       * x[i.stock.chf]
       )
}

### valuation of insurance polices
### -- only survival benefit
### -- observe that we do not condition on the 
###    policyholder's being alive at time year.
value.liabilities <- function(x,year){
    T <- length(IS)
    return(  sum(Discount(x[i.spot.chf])[year:T]
                 * Survival(x[i.q]*q)[year:T]
                 * IS[year:T]
                 )
           / if (year == 1) 1
           else Discount(x[i.spot.chf])[year-1]
           )
}

RTK <- function(x, year){
    return(  value.assets(x,year)
           - value.liabilities(x,year) )
}


### positive sensitivity for component i of the
### risk factors x
x.up <- function(x,h,additive,i){
    if (additive[i]) x[i] <- x[i]+h[i]
    else x[i] <- x[i] * (1+h[i])
    return(x)
}
### negative sensitivity for component i of the
### risk factors x
x.down <- function(x,h,additive,i){
    if (additive[i]) x[i] <- x[i]-h[i]
    else x[i] <- x[i] * (1-h[i])
    return(x)
}

### Correction for the fact that some sensitivities
### are multiplicative
Delta <- function(x,h,additive){
    Delta.x <- NULL
    for (i in 1:length(x)){
         if (additive[i]) Delta.x[i] <- h[i]
         else Delta.x[i] <- x[i] * h[i]
     }
    return(Delta.x)
}

quad.approx <- function(x, D.RTK, DD.RTK){
    return(t(x) %*% D.RTK + 0.5*t(x) %*% DD.RTK %*% x)
}

rand.Delta.RTK <- function(N, risk.index){
    Cov <- (  SD[risk.index] %*% t(SD[risk.index])
            * Corr[risk.index, risk.index]
            )
    return( apply(  rmvnorm( N
                            , rep(0,length(risk.index))
                            , Cov
                            )
                  , 1
                  , quad.approx
                  , D.RTK = delta[risk.index]
                  , DD.RTK =
                  Gamma[risk.index,risk.index]
                  )
           )
}

### Aggr.Scen: Aggregate Scenarios to randomly 
###            generated values rand
### -- We use the same approximation as in the shift
###     method, namely, two different scenarios cannot
###     happen in the same year.  
Aggr.Scen <- function(rand,Scen){
    ## rand: random values for the distribution
    ##       without added scenarios
    ## Scen: Data frame that describes the scenarios
    ##       to be added.
    N.Scen <- length(Scen$Delta_RTK)
    N.rand <- length(rand)
    k <- 0
    for (j in 1:N.Scen){
        N.Adjust <- floor(  N.rand
                          * Scen$Target[j]
                          * Scen$Probability[j]
                          )
        if (N.Adjust>0){
            for (l in 1:N.Adjust){
                rand[k+l] <-
                    (  rand[k+l]
                     + min(0, Scen$Delta_RTK[j])
                     )
            }
            k <- k+N.Adjust
        }
    }
    return(rand)
} 

#######################################################
### Calculations
#######################################################

### Risk factors, sensitivities, and indicators whether
### the sensitivity is additive
x <- NULL
h <- NULL
additive <- NULL
i.spot.chf <- (length(x)+1):( length(x)
                             +length(return.spot.chf))
h <- c(h, rep(0.01,length(i.spot.chf)))
additive <- c(additive, rep(TRUE,length(i.spot.chf)))
x <- c(x,return.spot.chf)
i.stock.chf <-  (length(x)+1):(length(x)+1)
h <- c(h,0.1)
additive <- c(additive, FALSE)
x <- c(x,1)
i.q <- (length(x)+1):(length(x)+1)
h <- c(h, 0.1)
additive <- c(additive, FALSE)
x <- c(x,1)

### expected risk bearing capital at the end of period 1
### and RTK at the beginning of the projection
RTK.end <- RTK(x,2)
RTK.start <- RTK(x,1)

### positive and negative linear sensitivities of the
### risk bearing capital
d.RTK.plus <- NULL
d.RTK.minus <- NULL
for (i in 1:length(x)){
  d.RTK.plus[i] <- RTK(x.up(x,h,additive,i),2)
  d.RTK.minus[i] <- RTK(x.down(x,h,additive,i),2)
}

### positive and negative quadratic sensitivities of the
### risk bearing capital at the end of period 2 
dd.RTK.plus.plus <-
    matrix(  rep(NA,length(x)*length(x))
           , ncol=length(x)
           )
dd.RTK.plus.minus <-
    matrix(  rep(NA,length(x)*length(x))
           , ncol=length(x)
           )
dd.RTK.minus.plus <-
    matrix(  rep(NA,length(x)*length(x))
           , ncol=length(x)
           )
dd.RTK.minus.minus <-
    matrix(  rep(NA,length(x)*length(x))
           , ncol=length(x)
           )
for (i in 1:length(x)){
    for (j in 1:length(x)){
        dd.RTK.plus.plus[i,j] <-
            RTK(x.up(  x.up( x, h, additive, i)
                     , h , additive, j)
                , 2)
        dd.RTK.plus.minus[i,j] <-
            RTK(x.up(  x.down( x, h, additive, i)
                     , h, additive, j)
                , 2)
        dd.RTK.minus.plus[i,j] <-
            RTK(x.down(  x.up( x, h, additive, i)
                       , h, additive, j)
                , 2)
        dd.RTK.minus.minus[i,j] <-
            RTK(x.down(  x.down( x, h, additive, i)
                       , h, additive, j),2)
    }
}

### numerical derivative and second derivative of the
## risk bearing capital
### at the end of period 2 
Delta.x <- Delta(x,h,additive)
delta <- (d.RTK.plus-d.RTK.minus)/(2*Delta.x)
Gamma <- (  dd.RTK.plus.plus - dd.RTK.plus.minus
          + dd.RTK.minus.minus - dd.RTK.minus.plus
          ) / (4 * Delta.x %*% t(Delta.x))


rand.market <-
    rand.Delta.RTK(N,c(i.spot.chf,i.stock.chf))
rand.insurance <-
    rand.Delta.RTK(N,c(i.q,i.q))
rand.market.insurance <-
    rand.Delta.RTK(N,c(i.spot.chf,i.stock.chf, i.q))

for (j in 1:length(Scen$Target)){
    Scen$Delta_RTK[j] <-
        quad.approx(Scen$Delta.x[j,],delta,Gamma)
}
rand.market.insurance.scen <-
    Aggr.Scen(rand.market.insurance, Scen)

risk.capital <-
    list(  insurance = ES(rand.insurance, alpha)
         , market = ES(rand.market, alpha)
         , market.insurance =
         ES(rand.market.insurance, alpha)
         , market.insurance.scen =
         ES(rand.market.insurance.scen, alpha)
         )

### market value margin for non-hedgeable risks
### -- we assume that the following risks are
###    non-hedgeable:
###    -- both risk from extreme scenarios and
###    -- biometric risk is non-hedgeable
### -- we assume that the following risks are hedgeable:
###    -- investment risks for bonds of
###       duration <= 15 years
###    -- investment risk for stocks
### -- for simplicity we approximate future risk capital
###    using the market value of liabilities. 
###    This is an approximation.
cost.of.capital <- 0
for (i in 1:length(IS)){
 cost.of.capital[i] <-
     (  coc
      * value.liabilities(x,i)
      / value.liabilities(x,1)
      * (  (risk.capital$market.insurance.scen
            - risk.capital$market.insurance)
         + risk.capital$insurance)
      )
                        
}
MVM <- sum(Discount(return.spot.chf) * cost.of.capital)

### observe that the function RTK already incorporates
### expected gains
target.capital <- (  risk.capital$market.insurance.scen
                   - return.spot.chf[1] * RTK.start
                   + MVM
                   ) / (1+return.spot.chf[1])

SST.ratio <- -RTK.start/target.capital

#######################################################
### Output
#######################################################
 
Results <- list()

## placeholder for the 1st entry in Results
Results$calc_start<- numeric(0)

## placeholder for the 2nd entry in Results
Results$calc_time_in_s <- numeric(0)   

Results$number.simulations <- N
Results$alpha <- alpha
Results$RTK.start <- RTK.start
Results$RTK.end <- RTK.end
Results$ES.insurance <- risk.capital$insurance
Results$ES.market <- risk.capital$market
Results$ES.market.insurance <-
    risk.capital$market.insurance
Results$ES.market.insurance.scen <-
    risk.capital$market.insurance.scen
Results$MVM <- MVM
Results$target.capital <- target.capital
Results$SST.ratio <- SST.ratio
time.end.user <-  proc.time()[["user.self"]]

Results$calc_start <-
    format(Sys.time(), "%Y-%m-%d_%H:%M:%S") 
Results$calc_time_in_s <-
    time.end.user-time.start.user

print(t(data.frame(Results)))                                 
