-- F. Urbano and F. Cagnacci (eds.), Spatial Database for GPS Wildlife Tracking Data
-- DOI: 10.1007/978-3-319-03743-1_4, Springer International Publishing Switzerland 2014

-- Dump of the PostgreSQL database created by the code presented in Chapters 2,3,4,5,6,7,8,9.

-- Dumped from database version 9.2.4
-- Dumped by pg_dump version 9.2.2

SET statement_timeout = 0;
SET client_encoding = 'UTF8';
SET standard_conforming_strings = on;
SET check_function_bodies = false;
SET client_min_messages = warning;

--
-- TOC entry 10 (class 2615 OID 109217)
-- Name: analysis; Type: SCHEMA; Schema: -; Owner: -
--

CREATE SCHEMA analysis;


--
-- TOC entry 3368 (class 0 OID 0)
-- Dependencies: 10
-- Name: SCHEMA analysis; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON SCHEMA analysis IS 'Schema that stores key layers for analysis.';


--
-- TOC entry 11 (class 2615 OID 109242)
-- Name: env_data; Type: SCHEMA; Schema: -; Owner: -
--

CREATE SCHEMA env_data;


--
-- TOC entry 3369 (class 0 OID 0)
-- Dependencies: 11
-- Name: SCHEMA env_data; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON SCHEMA env_data IS 'Schema that stores environmental ancillary information.';


--
-- TOC entry 8 (class 2615 OID 107965)
-- Name: lu_tables; Type: SCHEMA; Schema: -; Owner: -
--

CREATE SCHEMA lu_tables;


--
-- TOC entry 3370 (class 0 OID 0)
-- Dependencies: 8
-- Name: SCHEMA lu_tables; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON SCHEMA lu_tables IS 'Schema that stores look up tables.';


--
-- TOC entry 7 (class 2615 OID 107905)
-- Name: main; Type: SCHEMA; Schema: -; Owner: -
--

CREATE SCHEMA main;


--
-- TOC entry 3371 (class 0 OID 0)
-- Dependencies: 7
-- Name: SCHEMA main; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON SCHEMA main IS 'Schema that stores all the GPS tracking core data.';


--
-- TOC entry 9 (class 2615 OID 108047)
-- Name: tools; Type: SCHEMA; Schema: -; Owner: -
--

CREATE SCHEMA tools;


--
-- TOC entry 3373 (class 0 OID 0)
-- Dependencies: 9
-- Name: SCHEMA tools; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON SCHEMA tools IS 'Schema that hosts all the functions and ancillary tools used for the database.';


--
-- TOC entry 231 (class 3079 OID 11727)
-- Name: plpgsql; Type: EXTENSION; Schema: -; Owner: -
--

CREATE EXTENSION IF NOT EXISTS plpgsql WITH SCHEMA pg_catalog;


--
-- TOC entry 3374 (class 0 OID 0)
-- Dependencies: 231
-- Name: EXTENSION plpgsql; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON EXTENSION plpgsql IS 'PL/pgSQL procedural language';


--
-- TOC entry 232 (class 3079 OID 108095)
-- Name: postgis; Type: EXTENSION; Schema: -; Owner: -
--

CREATE EXTENSION IF NOT EXISTS postgis WITH SCHEMA public;


--
-- TOC entry 3375 (class 0 OID 0)
-- Dependencies: 232
-- Name: EXTENSION postgis; Type: COMMENT; Schema: -; Owner: -
--

COMMENT ON EXTENSION postgis IS 'PostGIS geometry, geography, and raster spatial types and functions';


SET search_path = tools, pg_catalog;

--
-- TOC entry 1659 (class 1247 OID 117501)
-- Name: bursts_report; Type: TYPE; Schema: tools; Owner: -
--

CREATE TYPE bursts_report AS (
	animals_id integer,
	starting_time timestamp with time zone,
	ending_time timestamp with time zone,
	num_locations integer,
	num_locations_null integer,
	interval_step integer
);


--
-- TOC entry 1670 (class 1247 OID 117530)
-- Name: geom_parameters; Type: TYPE; Schema: tools; Owner: -
--

CREATE TYPE geom_parameters AS (
	animals_id integer,
	acquisition_time timestamp with time zone,
	acquisition_time_t_1 timestamp with time zone,
	acquisition_time_t_2 timestamp with time zone,
	deltat_t_1 integer,
	deltat_t_2 integer,
	deltat_start integer,
	dist_t_1 integer,
	dist_start integer,
	speed_mh_t_1 numeric(8,2),
	abs_angle_t_1 numeric(7,5),
	rel_angle_t_2 numeric(7,5)
);


--
-- TOC entry 1673 (class 1247 OID 117534)
-- Name: grid_element; Type: TYPE; Schema: tools; Owner: -
--

CREATE TYPE grid_element AS (
	cell_id integer,
	geom public.geometry
);


--
-- TOC entry 1647 (class 1247 OID 117450)
-- Name: locations_set; Type: TYPE; Schema: tools; Owner: -
--

CREATE TYPE locations_set AS (
	animals_id integer,
	acquisition_time timestamp with time zone,
	geom public.geometry(Point,4326)
);


--
-- TOC entry 248 (class 1255 OID 108087)
-- Name: acquisition_time_update(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION acquisition_time_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$BEGIN
  NEW.acquisition_time = ((NEW.utc_date + NEW.utc_time) at time zone 'UTC');
  RETURN NEW;
END;$$;


--
-- TOC entry 3376 (class 0 OID 0)
-- Dependencies: 248
-- Name: FUNCTION acquisition_time_update(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION acquisition_time_update() IS 'When a record is inserted, the acquisition_time is composed from utc_date and utc_time.';


--
-- TOC entry 1156 (class 1255 OID 117541)
-- Name: age_class(integer, timestamp with time zone); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION age_class(animal_id integer, acquisition_time timestamp with time zone) RETURNS integer
    LANGUAGE plpgsql
    AS $$
DECLARE
  animal_age_class_code_capture integer;
  add_year integer;
  animal_date_capture date;
BEGIN
-- Retrieve the age class at first capture
animal_age_class_code_capture = (SELECT age_class_code FROM main.animals WHERE animals_id = animal_id);

-- If the animal is already an adult any location will be adult
IF animal_age_class_code_capture = 3 THEN
  RETURN 3;
END IF;

-- In case the animal at capture was not an adult, the function checks if the capture was before or after April. 
-- In the second case, the age class will increase the April of the next year.
animal_date_capture = (SELECT age_class_code FROM main.animals WHERE animals_id = animal_id);

IF EXTRACT(month FROM animal_date_capture) > 3 THEN
  add_year = 1;
ELSE 
  add_year = 0;
END IF;

-- If the animal was an yearling at capture, I check if it went through an age class increase.
IF animal_age_class_code_capture = 2 THEN
  IF acquisition_time > ((extract(year FROM animal_date_capture) + add_year)|| '/4/1')::date THEN
    RETURN 3;
  ELSE
    RETURN 2;
  END IF;
END IF;

-- If the animal was a fawn at capture, I check if it went through two and then one age class increase.
IF animal_age_class_code_capture = 1 THEN
  IF acquisition_time > ((extract(year FROM animal_date_capture) + add_year + 1)|| '/4/1')::date THEN
    RETURN 3;
  ELSEIF acquisition_time > ((extract(year FROM animal_date_capture) + add_year)|| '/4/1')::date THEN
    RETURN 2;
  ELSE
    RETURN 1;
  END IF;
END IF;

END;
 $$;


--
-- TOC entry 3377 (class 0 OID 0)
-- Dependencies: 1156
-- Name: FUNCTION age_class(animal_id integer, acquisition_time timestamp with time zone); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION age_class(animal_id integer, acquisition_time timestamp with time zone) IS 'This function returns the age class at the acquisition time of a location. It has two input parameters: the id of the animal and the timestamp. According to the age class at first capture, the function increases the class by 1 every time the animal goes through a defined day of the year (1st April).';


--
-- TOC entry 1155 (class 1255 OID 117535)
-- Name: create_grid(public.geometry, integer); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION create_grid(locations_collection public.geometry, xysize integer) RETURNS SETOF grid_element
    LANGUAGE sql
    AS $_$

WITH spatial_object AS
  (SELECT
    ST_Xmin(ST_Transform($1,tools.srid_utm(ST_X(ST_Centroid($1)), ST_Y(ST_Centroid($1)))))::integer AS xmin,
    ST_Ymin(ST_Transform($1,tools.srid_utm(ST_X(ST_Centroid($1)), ST_Y(ST_Centroid($1)))))::integer AS ymin,
    ST_Xmax(ST_Transform($1,tools.srid_utm(ST_X(ST_Centroid($1)), ST_Y(ST_Centroid($1)))))::integer AS xmax,
    ST_Ymax(ST_Transform($1,tools.srid_utm(ST_X(ST_Centroid($1)), ST_Y(ST_Centroid($1)))))::integer AS ymax,
    tools.srid_utm(ST_X(ST_Centroid($1)), ST_Y(ST_Centroid($1))) AS sridset)
  SELECT 
    (ROW_NUMBER() OVER ())::integer, 
    ST_Translate(cell, i , j)
  FROM 
    generate_series(
      ((((SELECT xmin FROM spatial_object) - $2/2)/100)::integer)*100, 
      (SELECT xmax FROM spatial_object) + $2, $2) AS i,
    generate_series(
      ((((SELECT ymin FROM spatial_object) - $2/2)/100)::integer)*100, 
      (SELECT ymax FROM spatial_object) + $2, $2) AS j, 
    spatial_object,
    (SELECT 
      ST_setsrid(ST_GeomFROMText('POLYGON((0 0, 0 '||$2||', '||$2||' '||$2||', '||$2||' 0,0 0))'),
      (SELECT sridset FROM spatial_object)) AS cell) AS foo;
$_$;


--
-- TOC entry 3378 (class 0 OID 0)
-- Dependencies: 1155
-- Name: FUNCTION create_grid(locations_collection public.geometry, xysize integer); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION create_grid(locations_collection public.geometry, xysize integer) IS 'Function that creates a vector grid with a given resolution that contains a given geometry.';


--
-- TOC entry 1149 (class 1255 OID 117502)
-- Name: detect_bursts(integer, integer); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION detect_bursts(animal integer, buffer integer DEFAULT 600) RETURNS SETOF bursts_report
    LANGUAGE plpgsql
    AS $$
DECLARE
  location_set tools.locations_set%rowtype;
  cursor_var tools.bursts_report%rowtype;
  starting_time timestamp with time zone;
  ending_time timestamp with time zone;
  location_time timestamp with time zone;
  time_prev timestamp with time zone;
  start_burst timestamp with time zone;
  end_burst timestamp with time zone;
  delta_time integer;
  ref_delta_time integer;
  ref_delta_time_round integer;
  n integer;
  n_null integer;
BEGIN

SELECT min(acquisition_time) 
FROM main.view_locations_set
WHERE view_locations_set.animals_id = animal
INTO starting_time;

SELECT max(acquisition_time) 
FROM main.view_locations_set
WHERE view_locations_set.animals_id = animal
INTO ending_time;

time_prev = NULL;
ref_delta_time = NULL;
n = 1;
n_null = 0;

FOR location_set IN EXECUTE 
  'SELECT animals_id, acquisition_time, geom 
  FROM main.view_locations_set 
  WHERE animals_id = '''|| animal ||''' ORDER BY acquisition_time'
LOOP
  location_time = location_set.acquisition_time;
  IF time_prev IS NULL THEN
    time_prev = location_time;
    start_burst = location_time;
  ELSE
    delta_time = (extract( epoch FROM (location_time - time_prev)))::integer;
    IF ref_delta_time IS NULL THEN
      ref_delta_time = delta_time;
      time_prev = location_time;
      end_burst = location_time;
    ELSIF abs(delta_time - ref_delta_time) < (buffer) THEN
      end_burst = location_time;
      time_prev = location_time;
      n = n + 1;
      IF location_set.geom IS NULL then
        n_null = n_null + 1;
      END IF;
    ELSE
      ref_delta_time_round = (ref_delta_time/buffer::double precision)::integer * buffer;
      IF ref_delta_time_round = 0 THEN
        ref_delta_time_round = (((extract( epoch FROM (end_burst - start_burst)))::integer/n)/60.0)::integer * 60;
      END IF;
    RETURN QUERY SELECT animal, start_burst, end_burst, n, n_null, ref_delta_time_round;
    ref_delta_time = delta_time;
    time_prev = location_time;
    start_burst = end_burst;
    end_burst = location_time;
    n = 1;
    n_null = 0;
    END IF;
  END IF;
END LOOP;

ref_delta_time_round = (ref_delta_time/buffer::double precision)::integer * buffer;
IF ref_delta_time_round = 0 THEN
  ref_delta_time_round = ((extract( epoch FROM end_burst - start_burst))::integer/n)::integer;
END IF;
RETURN QUERY SELECT animal, start_burst, end_burst, n , n_null, ref_delta_time_round;
RETURN;
END;
$$;


--
-- TOC entry 3379 (class 0 OID 0)
-- Dependencies: 1149
-- Name: FUNCTION detect_bursts(animal integer, buffer integer); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION detect_bursts(animal integer, buffer integer) IS 'This function gives the "bursts" for a defined animal. Bursts are groups of consecutive locations with the same frequency (or time interval). It gets an animal id and a buffer (in seconds) as input parameters and returns a table with the (supposed) schedule of location frequencies. The output table has the fields: "animals_id", "starting_time", "ending_time", "num_locations", "num_locations_null", and "interval_step" (in seconds, approximated according to multiples of the buffer value). A relocation is considered to have a different interval step if the time gap is bigger or smaller than the defined buffer (the buffer takes into account the fact that small changes can occur because of the delay in receiving the GPS signal). The default value for the buffer is 600 (10 minutes). The function is directly computed on "main.view_locations_set" (locations_set structure) and on the whole data set of the selected animal.';


--
-- TOC entry 1154 (class 1255 OID 117531)
-- Name: geom_parameters(integer, integer, double precision, character varying, timestamp with time zone, timestamp with time zone); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION geom_parameters(animal integer, time_interval integer DEFAULT 10800, buffer double precision DEFAULT 600, locations_set_name character varying DEFAULT 'main.view_locations_set'::character varying, starting_time timestamp with time zone DEFAULT NULL::timestamp with time zone, ending_time timestamp with time zone DEFAULT NULL::timestamp with time zone) RETURNS SETOF geom_parameters
    LANGUAGE plpgsql
    AS $$
DECLARE
  cursor_var tools.geom_parameters%rowtype;
  check_animal boolean;
  var_name character varying;
BEGIN
EXECUTE 
  'SELECT ' || animal || ' IN 
    (SELECT animals_id FROM main.animals)' INTO check_animal;

IF NOT check_animal THEN
  RAISE EXCEPTION 'This animal is not in the data set...';
END IF;

IF starting_time IS NULL THEN
  SELECT min(acquisition_time) 
  FROM main.view_locations_set
  WHERE view_locations_set.animals_id = animal
  INTO starting_time;
END IF;

IF ending_time IS NULL THEN
  SELECT max(acquisition_time) 
  FROM main.view_locations_set
  WHERE view_locations_set.animals_id = animal
  INTO ending_time;
END IF;

IF NOT locations_set_name = 'main.view_locations_set' THEN
  SELECT nextval('tools.unique_id_seq') into var_name;
  EXECUTE 
    'CREATE TEMPORARY TABLE temp_table_temp_table_geoparameters_'|| var_name ||' AS 
      SELECT animals_id, acquisition_time, geom 
      FROM ' || locations_set_name || ' 
      WHERE animals_id = '|| animal;
  locations_set_name = 'temp_table_temp_table_geoparameters_'|| var_name;
END IF;

FOR cursor_var IN EXECUTE 
  'SELECT 
    animals_id, 
    acquisition_time, 
    acquisition_time_t_1, 
    acquisition_time_t_2, 
    deltaT_t_1, 
    deltaT_t_2, 
    deltaT_start, 
    dist_t_1, 
    dist_start, 
    speed_Mh_t_1, 
    abs_angle_t_1, 
    CASE WHEN (deltaT_t_2 < ' || time_interval * 2 + buffer || ' and deltaT_t_2 > ' || time_interval * 2 - buffer || ') THEN 
      rel_angle_t_2
    ELSE 
      NULL
    END
  FROM 
    (SELECT 
      animals_id, 
      acquisition_time, 
      lead(acquisition_time,-1) OVER (PARTITION BY animals_id ORDER BY acquisition_time) AS acquisition_time_t_1,
      lead(acquisition_time,-2) OVER (PARTITION BY animals_id ORDER BY acquisition_time) AS acquisition_time_t_2,
      rank() OVER (PARTITION BY animals_id ORDER BY acquisition_time), 
      (extract(epoch FROM acquisition_time) - lead(extract(epoch FROM acquisition_time), -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::integer AS deltat_t_1,
      (extract(epoch FROM acquisition_time) - lead(extract(epoch FROM acquisition_time), -2) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::integer AS deltat_t_2,
      (extract(epoch FROM acquisition_time) - first_value(extract(epoch FROM acquisition_time)) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::integer AS deltat_start,
      (ST_Distance_Spheroid(geom, lead(geom, -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time), ''SPHEROID["WGS 84",6378137,298.257223563]''))::integer AS dist_t_1,
      ST_Distance_Spheroid(geom, first_value(geom) OVER (PARTITION BY animals_id ORDER BY acquisition_time), ''SPHEROID["WGS 84",6378137,298.257223563]'')::integer AS dist_start,
      (ST_Distance_Spheroid(geom, lead(geom, -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time), ''SPHEROID["WGS 84",6378137,298.257223563]'')/(extract(epoch FROM acquisition_time) - lead(extract(epoch FROM acquisition_time), -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time))*60*60)::numeric(8,2) AS speed_Mh_t_1,
      ST_Azimuth(geom::geography, (lead(geom, -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::geography) AS abs_angle_t_1,
      ST_Azimuth(geom::geography, (lead(geom, -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::geography) - ST_Azimuth((lead(geom, -1) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::geography, (lead(geom, -2) OVER (PARTITION BY animals_id ORDER BY acquisition_time))::geography) AS rel_angle_t_2
    FROM 
      '|| locations_set_name ||'
    WHERE 
      animals_id = ' || animal ||' AND 
      geom IS NOT NULL AND 
      acquisition_time >= ''' || starting_time || ''' AND 
      acquisition_time <= ''' || ending_time || ''') a
  WHERE 
    deltaT_t_1 <' || time_interval + buffer || ' AND 
    deltaT_t_1 > '|| time_interval - buffer
LOOP
RETURN NEXT cursor_var;
END LOOP;

IF NOT locations_set_name = 'main.view_locations_set' THEN
  EXECUTE 'drop table ' || locations_set_name;
END IF;
RETURN;
END;
$$;


--
-- TOC entry 3380 (class 0 OID 0)
-- Dependencies: 1154
-- Name: FUNCTION geom_parameters(animal integer, time_interval integer, buffer double precision, locations_set_name character varying, starting_time timestamp with time zone, ending_time timestamp with time zone); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION geom_parameters(animal integer, time_interval integer, buffer double precision, locations_set_name character varying, starting_time timestamp with time zone, ending_time timestamp with time zone) IS 'This function returns a table with the geometrical parameters of the data set (reference: previous location): time gap with the previous point, time gap with the previous-previous point, distance to the previous point, speed of the last step, distance from the first point of the data set, absolute angle (previous location), relative angle (previous and previous-previous location). The input parameters are the animal id, the time gap and the buffer. The time gap select just locations that have the previous point at a defined time interval (with a buffer tolerance). All the other points are not taken into consideration. A "locations_set" class is accepted as input table. It is also possible to specify the starting and ending acquisition time of the time series. The output is a table with the structure "geom_parameters".';


--
-- TOC entry 249 (class 1255 OID 108089)
-- Name: gps_data2gps_data_animals(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION gps_data2gps_data_animals() RETURNS trigger
    LANGUAGE plpgsql
    AS $$ begin
INSERT INTO main.gps_data_animals (
  animals_id, gps_sensors_id, acquisition_time, longitude, latitude)
SELECT 
  gps_sensors_animals.animals_id, gps_sensors_animals.gps_sensors_id, NEW.acquisition_time, NEW.longitude, NEW.latitude
FROM 
  main.gps_sensors_animals, main.gps_sensors
WHERE 
  NEW.gps_sensors_code = gps_sensors.gps_sensors_code AND 
  gps_sensors.gps_sensors_id = gps_sensors_animals.gps_sensors_id AND
  (
    (NEW.acquisition_time >= gps_sensors_animals.start_time AND 
     NEW.acquisition_time <= gps_sensors_animals.end_time)
    OR 
    (NEW.acquisition_time >= gps_sensors_animals.start_time AND 
     gps_sensors_animals.end_time IS NULL)
  );
RETURN NULL;
END
$$;


--
-- TOC entry 3381 (class 0 OID 0)
-- Dependencies: 249
-- Name: FUNCTION gps_data2gps_data_animals(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION gps_data2gps_data_animals() IS 'Automatic upload data from gps_data to gps_data_animals.';


--
-- TOC entry 251 (class 1255 OID 108093)
-- Name: gps_sensors_animals2gps_data_animals(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION gps_sensors_animals2gps_data_animals() RETURNS trigger
    LANGUAGE plpgsql
    AS $$ begin
IF TG_OP = 'DELETE' THEN
  DELETE FROM 
    main.gps_data_animals 
  WHERE 
    animals_id = OLD.animals_id AND
    gps_sensors_id = OLD.gps_sensors_id AND
    acquisition_time >= OLD.start_time AND
    (acquisition_time <= OLD.end_time OR OLD.end_time IS NULL);
  RETURN NULL;
END IF;

IF TG_OP = 'INSERT' THEN
  INSERT INTO 
    main.gps_data_animals (gps_sensors_id, animals_id, acquisition_time, longitude, latitude)
  SELECT 
    NEW.gps_sensors_id, NEW.animals_id, gps_data.acquisition_time, gps_data.longitude, gps_data.latitude
  FROM 
    main.gps_data, main.gps_sensors
  WHERE 
    NEW.gps_sensors_id = gps_sensors.gps_sensors_id AND
    gps_data.gps_sensors_code = gps_sensors.gps_sensors_code AND
    gps_data.acquisition_time >= NEW.start_time AND
    (gps_data.acquisition_time <= NEW.end_time OR NEW.end_time IS NULL);
  RETURN NULL;
END IF;

IF TG_OP = 'UPDATE' THEN
  DELETE FROM 
    main.gps_data_animals 
  WHERE
    gps_data_animals_id IN (
      SELECT 
        d.gps_data_animals_id 
      FROM
        (SELECT 
          gps_data_animals_id, gps_sensors_id, animals_id, acquisition_time 
        FROM 
          main.gps_data_animals
        WHERE 
          gps_sensors_id = OLD.gps_sensors_id AND
          animals_id = OLD.animals_id AND
          acquisition_time >= OLD.start_time AND
          (acquisition_time <= OLD.end_time OR OLD.end_time IS NULL)
        ) d
      LEFT OUTER JOIN
        (SELECT 
          gps_data_animals_id, gps_sensors_id, animals_id, acquisition_time 
        FROM 
          main.gps_data_animals
        WHERE 
          gps_sensors_id = NEW.gps_sensors_id AND
          animals_id = NEW.animals_id AND
          acquisition_time >= NEW.start_time AND
          (acquisition_time <= NEW.end_time OR NEW.end_time IS NULL) 
        ) e
      ON 
        (d.gps_data_animals_id = e.gps_data_animals_id)
      WHERE e.gps_data_animals_id IS NULL);
  INSERT INTO 
    main.gps_data_animals (gps_sensors_id, animals_id, acquisition_time, longitude, latitude) 
  SELECT 
    u.gps_sensors_id, u.animals_id, u.acquisition_time, u.longitude, u.latitude 
  FROM
    (SELECT 
      NEW.gps_sensors_id AS gps_sensors_id, NEW.animals_id AS animals_id, gps_data.acquisition_time AS acquisition_time, gps_data.longitude AS longitude, gps_data.latitude AS latitude
    FROM 
      main.gps_data, main.gps_sensors
    WHERE 
      NEW.gps_sensors_id = gps_sensors.gps_sensors_id AND 
      gps_data.gps_sensors_code = gps_sensors.gps_sensors_code AND
      gps_data.acquisition_time >= NEW.start_time AND
      (acquisition_time <= NEW.end_time OR NEW.end_time IS NULL)
    ) u
  LEFT OUTER JOIN
    (SELECT 
      gps_data_animals_id, gps_sensors_id, animals_id, acquisition_time 
    FROM 
      main.gps_data_animals
    WHERE 
      gps_sensors_id = OLD.gps_sensors_id AND
      animals_id = OLD.animals_id AND
      acquisition_time >= OLD.start_time AND
      (acquisition_time <= OLD.end_time OR OLD.end_time IS NULL)
    ) w
  ON 
    (u.gps_sensors_id = w.gps_sensors_id AND 
    u.animals_id = w.animals_id AND 
    u.acquisition_time = w.acquisition_time )
  WHERE 
    w.gps_data_animals_id IS NULL;
  RETURN NULL;
END IF;

END;$$;


--
-- TOC entry 3382 (class 0 OID 0)
-- Dependencies: 251
-- Name: FUNCTION gps_sensors_animals2gps_data_animals(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION gps_sensors_animals2gps_data_animals() IS 'When a record in gps_sensors_animals is deleted OR updated OR inserted, this function synchronizes this information with gps_data_animals.';


--
-- TOC entry 250 (class 1255 OID 108091)
-- Name: gps_sensors_animals_consistency_check(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION gps_sensors_animals_consistency_check() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
  deletex integer;
BEGIN
SELECT 
  gps_sensors_animals_id 
INTO 
  deletex 
FROM 
  main.gps_sensors_animals b
WHERE
  (NEW.animals_id = b.animals_id OR NEW.gps_sensors_id = b.gps_sensors_id)
  AND
  (
  (NEW.start_time > b.start_time AND NEW.start_time < b.end_time)
  OR
  (NEW.start_time > b.start_time AND b.end_time IS NULL)
  OR
  (NEW.end_time > b.start_time AND NEW.end_time < b.end_time)
  OR
  (NEW.start_time < b.start_time AND NEW.end_time > b.end_time)
  OR
  (NEW.start_time < b.start_time AND NEW.end_time IS NULL )
  OR
  (NEW.end_time > b.start_time AND b.end_time IS NULL)
);
IF deletex IS not NULL THEN
  IF TG_OP = 'INSERT' THEN
    RAISE EXCEPTION 'This row is not inserted: Animal-sensor association not valid: (the same animal would wear two different GPS sensors at the same time or the same GPS sensor would be deployed on two animals at the same time).';
    RETURN NULL;
  END IF;
  IF TG_OP = 'UPDATE' THEN
    IF deletex != OLD.gps_sensors_animals_id THEN
      RAISE EXCEPTION 'This row is not updated: Animal-sensor association not valid (the same animal would wear two different GPS sensors at the same time or the same GPS sensor would be deployed on two animals at the same time).';
      RETURN NULL;
    END IF;
  END IF;
END IF;
RETURN NEW;
END;
$$;


--
-- TOC entry 3383 (class 0 OID 0)
-- Dependencies: 250
-- Name: FUNCTION gps_sensors_animals_consistency_check(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION gps_sensors_animals_consistency_check() IS 'Check if a modified or insert row in gps_sensors_animals is valid (no impossible time range overlaps of deployments).';


--
-- TOC entry 1153 (class 1255 OID 117484)
-- Name: interpolate(integer, character varying, integer); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION interpolate(animal integer, locations_set_name character varying DEFAULT 'main.view_locations_set'::character varying, limit_gap integer DEFAULT 172800) RETURNS SETOF locations_set
    LANGUAGE plpgsql
    AS $_$
DECLARE
  location_set tools.locations_set%rowtype;
  starting_point record;
  ending_point record;
  time_distance_tot integer;
  perc_start double precision;
  x_point double precision;
  y_point double precision;
  var_name character varying;
BEGIN
IF NOT locations_set_name = 'main.view_locations_set' THEN

-- I need a unique name for my temporary table
  SELECT nextval('tools.unique_id_seq') 
  INTO var_name;
  EXECUTE 
    'CREATE TEMPORARY TABLE 
      temp_table_regularize_'|| var_name ||' AS SELECT animals_id,
      acquisition_time, 
      geom 
    FROM 
      ' || locations_set_name || ' 
    WHERE 
      animals_id = '|| animal;
  locations_set_name = 'temp_table_regularize_'|| var_name;
END IF;

-- I loop though all the elements of my data set
FOR location_set IN EXECUTE 
  'SELECT * FROM ' || locations_set_name || ' WHERE animals_id = ' || animal
LOOP

-- If the record has a NULL geometry values, I look for the previous and next valid locations and interpolate the coordinates between them
  IF location_set.geom IS NULL THEN

-- I get the geometry and timestamp of the next valid location
    EXECUTE 
      'SELECT 
        ST_X(geom) AS x_end, 
        ST_Y(geom) AS y_end, 
        extract(epoch FROM acquisition_time) AS ending_time, 
        extract(epoch FROM $$' ||location_set.acquisition_time || '$$::timestamp with time zone) AS ref_time 
      FROM 
        ' || locations_set_name || ' 
      WHERE 
        animals_id = ' || animal || ' AND 
        geom IS NOT NULL AND 
        acquisition_time > timestamp with time zone $$' ||location_set.acquisition_time || '$$ 
      ORDER BY acquisition_time 
      LIMIT 1'
    INTO ending_point;

-- I get the geometry and timestamp of the previous valid location
    EXECUTE
      'SELECT 
        ST_X(geom) AS x_start, 
        ST_Y(geom) AS y_start, 
        extract(epoch FROM acquisition_time) AS starting_time, 
        extract(epoch FROM $$' ||location_set.acquisition_time || '$$::timestamp with time zone) AS ref_time 
      FROM 
        ' || locations_set_name || ' 
      WHERE 
        animals_id = ' || animal || ' AND 
        geom IS NOT NULL AND 
        acquisition_time < timestamp with time zone $$' ||location_set.acquisition_time || '$$ 
      ORDER BY acquisition_time DESC 
      LIMIT 1'
    INTO starting_point;

-- If both previous and next locations exist, I calculate the interpolated point, weighting the two points according to the temporal distance to the location with NULL geometry. The interpolated geometry is calculated considering lat long as a Cartesian reference. if needed, this approach can be improved casting geometry as geography and intersecting the line between previous and next locations with the buffer (from the previous location) at the given distance.
    IF (starting_point.x_start IS NOT NULL AND ending_point.x_end IS NOT NULL) THEN
      time_distance_tot = (ending_point.ending_time - starting_point.starting_time);
      IF time_distance_tot <= limit_gap THEN
        perc_start = (starting_point.ref_time - starting_point.starting_time)/time_distance_tot;
        x_point = starting_point.x_start + (ending_point.x_end - starting_point.x_start) * perc_start;
        y_point = starting_point.y_start + (ending_point.y_end - starting_point.y_start) * perc_start;
        SELECT ST_SetSRID(ST_MakePoint(x_point, y_point),4326) 
        INTO location_set.geom;
      END IF;
    END IF;
  END IF;
RETURN NEXT location_set;
END LOOP;

-- If I created the temporary table, I delete it here
IF NOT locations_set_name = 'main.view_locations_set' THEN
  EXECUTE 'drop table ' || locations_set_name;
END IF;
return;
END;
$_$;


--
-- TOC entry 3384 (class 0 OID 0)
-- Dependencies: 1153
-- Name: FUNCTION interpolate(animal integer, locations_set_name character varying, limit_gap integer); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION interpolate(animal integer, locations_set_name character varying, limit_gap integer) IS 'This function accepts as input an animals_id and a locations_set (by default, the main.view_locations_set). It checks for all locations with NULL geometry. If these locations have a previous and next valid locations (according to the gps_validity_code) with a gap smaller than the defined threshold (default is 2 days), a new geometry is calculated interpolating their geometry.';


--
-- TOC entry 1151 (class 1255 OID 117472)
-- Name: make_traj(character varying, character varying); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION make_traj(locations_set_query character varying DEFAULT 'main.view_locations_set'::character varying, description character varying DEFAULT 'Standard trajectory'::character varying) RETURNS integer
    LANGUAGE plpgsql
    AS $$
DECLARE
  locations_set_query_string character varying;
BEGIN
  locations_set_query_string = (SELECT replace(locations_set_query, '''',''''''));
  EXECUTE
    'INSERT INTO analysis.trajectories (animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, original_data_set, geom) 
      SELECT sel_subquery.animals_id, min(acquisition_time), max(acquisition_time), ''' ||description|| ''', current_user, count(*), ST_length2d_spheroid(ST_MakeLine(sel_subquery.geom), ''SPHEROID("WGS84",6378137,298.257223563)''::spheroid), '''|| locations_set_query_string ||''', ST_MakeLine(sel_subquery.geom) AS geom
      FROM 
        (SELECT * 
        FROM ('||locations_set_query||') a 
        WHERE a.geom IS NOT NULL
        ORDER BY a.animals_id, a.acquisition_time) sel_subquery
        GROUP BY sel_subquery.animals_id;';
  raise notice 'Operation correctly performed. Record inserted into analysis.trajectories';
  RETURN 1; 
END;
$$;


--
-- TOC entry 3385 (class 0 OID 0)
-- Dependencies: 1151
-- Name: FUNCTION make_traj(locations_set_query character varying, description character varying); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION make_traj(locations_set_query character varying, description character varying) IS 'This function produces a trajectory from a locations_set object (animals_id, acquisition_time, geom) into the table analysis.trajectories. Two parameters are accepted: the first is the SQL code that generates the locations_set object, the second is a string that is used to comment the trajectory. A trajectory will be created for each animal in the data set. If you need to include a single quote ('') in the SQL that select the locations (for example, when you want to define a timestamp), you have to use two single quotes to escape the special character: ''''. An example: SELECT tools.make_traj(''SELECT * FROM main.view_locations_set WHERE acquisition_time > ''''2006-01-01''''::timestamp and animals_id in (3,4)'').';


--
-- TOC entry 1147 (class 1255 OID 117522)
-- Name: mcp_perc(integer, double precision, character varying, character varying, timestamp with time zone, timestamp with time zone); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION mcp_perc(animal integer, perc double precision DEFAULT 1, description character varying DEFAULT 'Standard analysis'::character varying, locations_set_name character varying DEFAULT 'main.view_locations_set'::character varying, starting_time timestamp with time zone DEFAULT NULL::timestamp with time zone, ending_time timestamp with time zone DEFAULT NULL::timestamp with time zone) RETURNS integer
    LANGUAGE plpgsql
    AS $_$
DECLARE
  hr record;
  var_name character varying;
  locations_set_name_input character varying;
BEGIN
locations_set_name_input = locations_set_name;

IF NOT locations_set_name = 'main.view_locations_set' THEN
  SELECT nextval('tools.unique_id_seq') INTO var_name;
  EXECUTE 
    'CREATE TEMPORARY TABLE temp_table_mcp_perc_'|| var_name ||' AS 
      SELECT * 
      FROM ' || locations_set_name || ' 
      WHERE animals_id = '|| animal;
  locations_set_name = 'temp_table_mcp_perc_'|| var_name;
END IF;

IF perc <= 0 OR perc > 1 THEN
  RAISE EXCEPTION 'INVALID PARAMETER: the percentage of the selected (closest to the data set centroid) points must be a value > 0 and <= 1';
END IF;

IF starting_time IS NULL THEN
  EXECUTE 
    'SELECT min(acquisition_time) 
    FROM '|| locations_set_name ||'
    WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL '
  INTO starting_time;
END IF;

IF ending_time IS NULL THEN
  EXECUTE 
    'SELECT max(acquisition_time) 
    FROM '|| locations_set_name ||'
    WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL '
  INTO ending_time;
END IF;

EXECUTE
  'SELECT 
    animals_id, 
    min(acquisition_time) AS start_time, 
    max(acquisition_time) AS end_time, 
    count(animals_id) AS num_locations, 
    ST_Area(geography(ST_ConvexHull(ST_Collect(a.geom)))) AS area, 
    (ST_ConvexHull(ST_Collect(a.geom))).ST_Multi AS geom
  FROM 
    (SELECT '|| locations_set_name ||'.animals_id, '|| locations_set_name ||'.geom, acquisition_time, ST_Distance('|| locations_set_name ||'.geom, 
      (SELECT ST_Centroid(ST_collect('|| locations_set_name ||'.geom))
      FROM '|| locations_set_name ||'
      WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL AND '|| locations_set_name ||'.acquisition_time >= $$' || starting_time ||'$$::timestamp with time zone AND '|| locations_set_name ||'.acquisition_time <= $$' || ending_time || '$$::timestamp with time zone 
      GROUP BY '|| locations_set_name ||'.animals_id)) AS dist
    FROM '|| locations_set_name ||'
    WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL AND '|| locations_set_name ||'.acquisition_time >= $$' || starting_time ||'$$::timestamp with time zone and '|| locations_set_name ||'.acquisition_time <= $$' || ending_time || '$$::timestamp with time zone 
    ORDER BY 
      ST_Distance('|| locations_set_name ||'.geom, 
     (SELECT ST_Centroid(ST_Collect('|| locations_set_name ||'.geom))
     FROM '|| locations_set_name ||'
     WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL AND '|| locations_set_name ||'.acquisition_time >= $$' || starting_time ||'$$::timestamp with time zone and '|| locations_set_name ||'.acquisition_time <= $$' || ending_time || '$$::timestamp with time zone 
     GROUP BY '|| locations_set_name ||'.animals_id))
     LIMIT (((SELECT count('|| locations_set_name ||'.animals_id) AS count
  FROM '|| locations_set_name ||'
  WHERE '|| locations_set_name ||'.animals_id = ' || animal || ' AND '|| locations_set_name ||'.geom IS NOT NULL AND '|| locations_set_name ||'.acquisition_time >= $$' || starting_time ||'$$::timestamp with time zone AND '|| locations_set_name ||'.acquisition_time <= $$' || ending_time || '$$::timestamp with time zone 
))::numeric * ' || perc || ')::integer) a
  GROUP BY a.animals_id;'
  INTO hr;
  IF hr.num_locations < 3 or hr.num_locations IS NULL THEN
    RAISE NOTICE 'INVALID SELECTION: less then 3 points or no points at all match the given criteria. The animal % will be skipped.', animal;
RETURN 0;
END IF;
INSERT INTO analysis.home_ranges_mcp (animals_id, start_time, end_time, 
percentage, description, ref_user, num_locations, 
area, geom, original_data_set)
values (animal, starting_time, ending_time , perc , description, current_user, hr.num_locations, hr.area/1000000.00000, hr.geom, locations_set_name_input);
IF NOT locations_set_name = 'main.view_locations_set' THEN
EXECUTE 'drop table ' || locations_set_name;
END IF;
RAISE NOTICE 'Operation correctly performed. Record inserted into analysis.home_ranges % ', animal;
RETURN 1; 
END;
$_$;


--
-- TOC entry 3386 (class 0 OID 0)
-- Dependencies: 1147
-- Name: FUNCTION mcp_perc(animal integer, perc double precision, description character varying, locations_set_name character varying, starting_time timestamp with time zone, ending_time timestamp with time zone); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION mcp_perc(animal integer, perc double precision, description character varying, locations_set_name character varying, starting_time timestamp with time zone, ending_time timestamp with time zone) IS 'This function applies the MCP (Minimum Convex Polygon) algorithm (also called convex hull) to a set of locations. The input parameters are the animal id (each analysis is related to a single individual), the percentage of locations considered, a "locations_set" object (the default is "main.view_locations_set"). An additional parameter can be added: a description that will be included in the table home_ranges_mcp, where the result of the analysis is stored. The parameter "percentage" defines how many locations are included in the analysis: if for example 90% is specified (as 0.9), the 10% of locations farthest from the centroid of the data set will be excluded. If no parameters are specified, percentage of 100% and the complete data set (from the first to the last location) are considered. The function, once computed the MCP and stored the result in home_range_mcp, does not return anything. Few errors trapping are added (no points selected, percentage out of range). Note that this function works with a fixed centroid, computed at the beginning, so the distance is calculated on this basis for all the selection process.';


--
-- TOC entry 1148 (class 1255 OID 109379)
-- Name: ndvi_acquisition_range_update(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION ndvi_acquisition_range_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  NEW.acquisition_range = daterange(
    (substring(NEW.filename FROM 12 FOR 4) || '-' || 
      substring(NEW.filename FROM 17 FOR 2) || '-' || 
      substring(NEW.filename FROM 20 FOR 2))::date,
    LEAST((substring(NEW.filename FROM 12 FOR 4) || '-' || 
        substring(NEW.filename FROM 17 FOR 2) || '-' || 
        substring(NEW.filename FROM 20 FOR 2))::date + 16,
      (substring(NEW.filename FROM 12 FOR 4)::integer + 1 
        || '-' || '01' || '-' || '01')::date));
RETURN NEW;
END;
$$;


--
-- TOC entry 3387 (class 0 OID 0)
-- Dependencies: 1148
-- Name: FUNCTION ndvi_acquisition_range_update(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION ndvi_acquisition_range_update() IS 'This function is raised whenever a new record is inserted into the MODIS NDVI time series table in order to define the date range. The acquisition_range value is derived from the original filename (that has the structure MODIS_NDVI_YYYY_MM_DD.tif)';


--
-- TOC entry 1141 (class 1255 OID 109382)
-- Name: ndvi_intersection_update(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION ndvi_intersection_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  UPDATE main.gps_data_animals
  SET ndvi_modis = 
    (SELECT ST_Value(NEW.rast, geom)
      FROM env_data.ndvi_modis 
      WHERE ST_Intersects(geom, NEW.rast)
      AND NEW.acquisition_range @> NEW.acquisition_time::date)
  WHERE ST_Intersects(geom, NEW.rast) AND
    NEW.acquisition_range @> acquisition_time::date AND
    ndvi.modis IS NULL;
END;
$$;


--
-- TOC entry 3388 (class 0 OID 0)
-- Dependencies: 1141
-- Name: FUNCTION ndvi_intersection_update(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION ndvi_intersection_update() IS 'When new NDVI data is added, the ndvi_modis field of main.gps_data_animals is updated.';


--
-- TOC entry 1150 (class 1255 OID 109215)
-- Name: new_gps_data_animals(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION new_gps_data_animals() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE 
thegeom geometry;
BEGIN
IF NEW.longitude IS NOT NULL AND NEW.latitude IS NOT NULL THEN
  thegeom = ST_setsrid(ST_MakePoint(NEW.longitude, NEW.latitude), 4326);
  NEW.geom =thegeom;
  NEW.gps_validity_code = 1;
    IF NOT EXISTS (SELECT study_area FROM env_data.study_area WHERE ST_intersects(ST_Buffer(thegeom,0.1), study_area.geom)) THEN
      NEW.gps_validity_code = 11;
    END IF;
  NEW.pro_com = (SELECT pro_com::integer FROM env_data.adm_boundaries WHERE ST_intersects(geom,thegeom)); 
  NEW.corine_land_cover_code = (SELECT ST_Value(rast, ST_Transform(thegeom, 3035)) FROM env_data.corine_land_cover WHERE ST_Intersects(ST_Transform(thegeom,3035), rast));
  NEW.altitude_srtm = (SELECT ST_Value(rast,thegeom) FROM env_data.srtm_dem WHERE ST_intersects(thegeom, rast));
  NEW.station_id = (SELECT station_id::integer FROM env_data.meteo_stations ORDER BY ST_Distance_Spheroid(thegeom, geom, 'SPHEROID["WGS 84",6378137,298.257223563]') LIMIT 1);
  NEW.roads_dist = (SELECT ST_Distance(thegeom::geography, geom::geography)::integer FROM env_data.roads ORDER BY ST_Distance(thegeom::geography, geom::geography) LIMIT 1);
  NEW.ndvi_modis = (SELECT ST_Value(rast, thegeom)FROM env_data_ts.ndvi_modis WHERE ST_Intersects(thegeom, rast) 
AND EXTRACT(year FROM acquisition_date) = EXTRACT(year FROM NEW.acquisition_time)
AND EXTRACT(month FROM acquisition_date) = EXTRACT(month FROM NEW.acquisition_time)
and EXTRACT(day FROM acquisition_date) = CASE
WHEN EXTRACT(day FROM NEW.acquisition_time) < 11 THEN 1
WHEN EXTRACT(day FROM NEW.acquisition_time) < 21 THEN 11
ELSE 21
END);
ELSE 
NEW.gps_validity_code = 0;
END IF;
RETURN NEW;
END;$$;


--
-- TOC entry 3389 (class 0 OID 0)
-- Dependencies: 1150
-- Name: FUNCTION new_gps_data_animals(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION new_gps_data_animals() IS 'When called by the trigger insert_gps_locations (raised whenever a new GPS position is uploaded into gps_data_animals) this function gets the new longitude and latitude values and sets the field geom accordingly, computing a set of derived environmental information calculated intersection the GPS position with the environmental ancillary layers. GPS positions outside the study area are tagged as outliers.';


--
-- TOC entry 1157 (class 1255 OID 117542)
-- Name: randompoints(public.geometry, integer, numeric); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION randompoints(geom public.geometry, num_points integer, seed numeric DEFAULT NULL::numeric) RETURNS SETOF public.geometry
    LANGUAGE plpgsql
    AS $$ 
DECLARE 
  pt geometry; 
  xmin float8; 
  xmax float8; 
  ymin float8; 
  ymax float8; 
  xrange float8; 
  yrange float8; 
  srid int; 
  count integer := 0; 
  bcontains boolean := FALSE; 
  gtype text; 
BEGIN 
SELECT ST_GeometryType(geom) 
INTO gtype; 

IF ( gtype != 'ST_Polygon' ) AND ( gtype != 'ST_MultiPolygon' ) THEN 
  RAISE EXCEPTION 'Attempting to get random point in a non polygon geometry'; 
END IF; 

SELECT ST_XMin(geom), ST_XMax(geom), ST_YMin(geom), ST_YMax(geom), ST_SRID(geom) 
INTO xmin, xmax, ymin, ymax, srid; 

SELECT xmax - xmin, ymax - ymin 
INTO xrange, yrange; 

IF seed IS NOT NULL THEN 
  PERFORM setseed(seed); 
END IF; 

WHILE count < num_points LOOP 
  SELECT 
    ST_SetSRID(ST_MakePoint(
      xmin + xrange * random(), 
      ymin + yrange * random()),
      srid) 
  INTO pt; 

  SELECT ST_Contains(geom, pt) 
  INTO bcontains; 

  IF bcontains THEN 
    count := count + 1; 
    RETURN NEXT pt; 
  END IF; 
END LOOP; 
RETURN; 
END; 
$$;


--
-- TOC entry 3390 (class 0 OID 0)
-- Dependencies: 1157
-- Name: FUNCTION randompoints(geom public.geometry, num_points integer, seed numeric); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION randompoints(geom public.geometry, num_points integer, seed numeric) IS 'This function generates a set of random points into a given polygon (or multipolygon). The number of points and the polygon must be provided as input. A third optional parameter can define the seed, and thus generate consistent (random) set of points.';


--
-- TOC entry 1152 (class 1255 OID 117481)
-- Name: regularize(integer, integer, double precision, timestamp with time zone, timestamp with time zone); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION regularize(animal integer, time_interval integer DEFAULT 10800, buffer double precision DEFAULT 600, starting_time timestamp with time zone DEFAULT NULL::timestamp with time zone, ending_time timestamp with time zone DEFAULT NULL::timestamp with time zone) RETURNS SETOF locations_set
    LANGUAGE plpgsql
    AS $$
DECLARE
  location_set tools.locations_set%rowtype;
  cursor_var record;
  interval_length integer;
  check_animal boolean;
BEGIN
-- Error trapping: if the buffer is > 0.5 * time interval, I could take 2 times the same locations, therefore an exception is raised
IF buffer > 0.5 * time_interval THEN
  RAISE EXCEPTION 'With a buffer (%) > 0.5 * time interval (%), you could get twice the same location, please reduce buffer or increase time interval.', buffer, time_interval;
END IF;

-- If the starting date is not set, the minimum, valid timestamp of the data set is taken
IF starting_time IS NULL THEN
  SELECT 
    min(acquisition_time) 
  FROM 
    main.view_locations_set
  WHERE 
    view_locations_set.animals_id = animal
  INTO starting_time;
END IF;

-- If the ending date is not set, the maximum, valid timestamp of the data set is taken
IF ending_time IS NULL THEN
  SELECT max(acquisition_time) 
  FROM main.view_locations_set
  WHERE view_locations_set.animals_id = animal
  INTO ending_time;
END IF;

-- I define the interval time (number of seconds between the starting and ending time)
SELECT extract(epoch FROM (ending_time - starting_time))::integer + buffer
INTO interval_length;

-- I create a "virtual" set of records with regular time interval (from starting_time to ending_time, with a step equal to the interval length, then I go through all the elements of the virtual set and I check if a real record exist in main.view_locations_set which has an acquisition_time closer then the defined buffer. If more then 1 record exists in the buffer range, then I take the "closest".
FOR location_set IN 
  SELECT 
    animal, 
    (starting_time + generate_series (0, interval_length, time_interval) * interval '1 second'), 
    NULL::geometry
LOOP
  SELECT geom, acquisition_time
  FROM main.view_locations_set 
  WHERE 
    animals_id = animal AND 
    (acquisition_time < (location_set.acquisition_time + interval '1 second' * buffer) AND 
    acquisition_time > (location_set.acquisition_time - interval '1 second' * buffer)) 
  ORDER BY 
    abs(extract (epoch FROM (acquisition_time - location_set.acquisition_time))) 
  LIMIT 1 
  INTO cursor_var;

-- If I have a record in main.view_locations_set, i get the values from there, otherwise I keep my "virtual" record
  IF cursor_var.acquisition_time IS NOT NULL THEN
    location_set.acquisition_time = cursor_var.acquisition_time;
    location_set.geom = cursor_var.geom;
  END IF;
  RETURN NEXT location_set;
END LOOP;
RETURN;
END;
$$;


--
-- TOC entry 3391 (class 0 OID 0)
-- Dependencies: 1152
-- Name: FUNCTION regularize(animal integer, time_interval integer, buffer double precision, starting_time timestamp with time zone, ending_time timestamp with time zone); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION regularize(animal integer, time_interval integer, buffer double precision, starting_time timestamp with time zone, ending_time timestamp with time zone) IS 'This function creates a complete, regular time series of locations from main.view_locations_set using an individual id, a time interval (in seconds), a buffer time (in seconds, which corresponds to the  accepted delay of GPS recording), a starting time (if no values is defined, the first record of the animal data set is taken), an ending time (if no values is defined, the last record of the animal data set is taken). The function checks at every time step if exists a "real" record (with or with coordinates) in the main.view_locations_set table (which is the "locations_set" object of the "main.gps_data_animals" table): if any real data exist (inside a defined time interval buffer from the reference timestamp generated by the function) in main.view_locations_set, the real record is used, otherwise a "virtual" record is created (with empty geometry). The output is a table with the structure "location_set" (animals_id integer, acquisition_time timestamp with time zone, geom geometry).';


--
-- TOC entry 1146 (class 1255 OID 109210)
-- Name: srid_utm(double precision, double precision); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION srid_utm(longitude double precision, latitude double precision) RETURNS integer
    LANGUAGE plpgsql STRICT
    AS $$
DECLARE
  srid integer;
  lon float;
  lat float;
BEGIN
  lat := latitude;
  lon := longitude;

IF ((lon > 360 or lon < -360) or (lat > 90 or lat < -90)) THEN 
  RAISE EXCEPTION 'Longitude and latitude is not in a valid format (-360 to 360; -90 to 90)';
ELSEIF (longitude < -180)THEN 
  lon := 360 + lon;
ELSEIF (longitude > 180)THEN 
  lon := 180 - lon;
END IF;

IF latitude >= 0 THEN 
  srid := 32600 + floor((lon+186)/6); 
ELSE
  srid := 32700 + floor((lon+186)/6); 
END IF;

RETURN srid;
END;
$$;


--
-- TOC entry 3392 (class 0 OID 0)
-- Dependencies: 1146
-- Name: FUNCTION srid_utm(longitude double precision, latitude double precision); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION srid_utm(longitude double precision, latitude double precision) IS 'Function that returns the SRID code of the UTM zone where a point (in geographic coordinates) is located. For polygons or line, it can be used giving ST_x(ST_Centroid(the_geom)) and ST_y(ST_Centroid(the_geom)) as parameters. This function is typically used be used with ST_Transform to project elements with no prior knowledge of their position.';


--
-- TOC entry 233 (class 1255 OID 108049)
-- Name: test_add(integer, integer); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION test_add(integer, integer) RETURNS integer
    LANGUAGE sql IMMUTABLE STRICT
    AS $_$SELECT $1 + $2;$_$;


--
-- TOC entry 247 (class 1255 OID 108057)
-- Name: timestamp_last_update(); Type: FUNCTION; Schema: tools; Owner: -
--

CREATE FUNCTION timestamp_last_update() RETURNS trigger
    LANGUAGE plpgsql
    AS $$BEGIN
IF NEW IS DISTINCT FROM OLD THEN
  NEW.update_timestamp = now();
END IF;
RETURN NEW;
END;$$;


--
-- TOC entry 3393 (class 0 OID 0)
-- Dependencies: 247
-- Name: FUNCTION timestamp_last_update(); Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON FUNCTION timestamp_last_update() IS 'When a record is updated, the update_timestamp is set to the current time.';


SET search_path = analysis, pg_catalog;

SET default_tablespace = '';

SET default_with_oids = false;

--
-- TOC entry 224 (class 1259 OID 117505)
-- Name: home_ranges_mcp; Type: TABLE; Schema: analysis; Owner: -; Tablespace: 
--

CREATE TABLE home_ranges_mcp (
    home_ranges_mcp_id integer NOT NULL,
    animals_id integer NOT NULL,
    start_time timestamp with time zone NOT NULL,
    end_time timestamp with time zone NOT NULL,
    description character varying,
    ref_user character varying,
    num_locations integer,
    area numeric(13,5),
    geom public.geometry(MultiPolygon,4326),
    percentage double precision,
    insert_timestamp timestamp with time zone DEFAULT timezone('UTC'::text, ('now'::text)::timestamp(0) with time zone),
    original_data_set character varying
);


--
-- TOC entry 3394 (class 0 OID 0)
-- Dependencies: 224
-- Name: TABLE home_ranges_mcp; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON TABLE home_ranges_mcp IS 'Table that stores the home range polygons derived from MCP. The area is computed in hectars.';


--
-- TOC entry 223 (class 1259 OID 117503)
-- Name: home_ranges_mcp_home_ranges_mcp_id_seq; Type: SEQUENCE; Schema: analysis; Owner: -
--

CREATE SEQUENCE home_ranges_mcp_home_ranges_mcp_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3395 (class 0 OID 0)
-- Dependencies: 223
-- Name: home_ranges_mcp_home_ranges_mcp_id_seq; Type: SEQUENCE OWNED BY; Schema: analysis; Owner: -
--

ALTER SEQUENCE home_ranges_mcp_home_ranges_mcp_id_seq OWNED BY home_ranges_mcp.home_ranges_mcp_id;


--
-- TOC entry 230 (class 1259 OID 117547)
-- Name: test_randompoints; Type: TABLE; Schema: analysis; Owner: -; Tablespace: 
--

CREATE TABLE test_randompoints (
    id bigint NOT NULL,
    geom public.geometry(Point,4326)
);


--
-- TOC entry 3396 (class 0 OID 0)
-- Dependencies: 230
-- Name: TABLE test_randompoints; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON TABLE test_randompoints IS 'This table is a test that permanently stores 100 random points into the boundaries of the first polygon stored in the home_ranges_mcp table.';


--
-- TOC entry 220 (class 1259 OID 117457)
-- Name: trajectories; Type: TABLE; Schema: analysis; Owner: -; Tablespace: 
--

CREATE TABLE trajectories (
    trajectories_id integer NOT NULL,
    animals_id integer NOT NULL,
    start_time timestamp with time zone NOT NULL,
    end_time timestamp with time zone NOT NULL,
    description character varying,
    ref_user character varying,
    num_locations integer,
    length_2d integer,
    insert_timestamp timestamp with time zone DEFAULT now(),
    original_data_set character varying,
    geom public.geometry(LineString,4326)
);


--
-- TOC entry 3397 (class 0 OID 0)
-- Dependencies: 220
-- Name: TABLE trajectories; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON TABLE trajectories IS 'Table that stores the trajectories derived from a set of selected locations. Each trajectory is related to a single animal. This table is populated by the function tools.make_traj. Each element is described by a number of attributes: the starting date and the ending date of the location set, a general description (that can be used to "tag" each record with specific identifiers), the user who did the analysis, the number of locations (or vertex of the lines) that produced the analysis, the length of the line, and the SQL that generated the dataset.';


--
-- TOC entry 219 (class 1259 OID 117455)
-- Name: trajectories_trajectories_id_seq; Type: SEQUENCE; Schema: analysis; Owner: -
--

CREATE SEQUENCE trajectories_trajectories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3398 (class 0 OID 0)
-- Dependencies: 219
-- Name: trajectories_trajectories_id_seq; Type: SEQUENCE OWNED BY; Schema: analysis; Owner: -
--

ALTER SEQUENCE trajectories_trajectories_id_seq OWNED BY trajectories.trajectories_id;


SET search_path = main, pg_catalog;

--
-- TOC entry 184 (class 1259 OID 108028)
-- Name: gps_data_animals; Type: TABLE; Schema: main; Owner: -; Tablespace: 
--

CREATE TABLE gps_data_animals (
    gps_data_animals_id integer NOT NULL,
    gps_sensors_id integer,
    animals_id integer,
    acquisition_time timestamp with time zone,
    longitude double precision,
    latitude double precision,
    insert_timestamp timestamp with time zone DEFAULT now(),
    update_timestamp timestamp with time zone DEFAULT now(),
    geom public.geometry(Point,4326),
    pro_com integer,
    corine_land_cover_code integer,
    altitude_srtm integer,
    station_id integer,
    roads_dist integer,
    ndvi_modis integer,
    gps_validity_code integer
);


--
-- TOC entry 3399 (class 0 OID 0)
-- Dependencies: 184
-- Name: TABLE gps_data_animals; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON TABLE gps_data_animals IS 'GPS sensors data associated to animals wearing the sensor.';


SET search_path = analysis, pg_catalog;

--
-- TOC entry 200 (class 1259 OID 109227)
-- Name: view_convex_hulls; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_convex_hulls AS
    SELECT gps_data_animals.animals_id, (public.st_convexhull(public.st_collect(gps_data_animals.geom)))::public.geometry(Polygon,4326) AS geom FROM main.gps_data_animals WHERE (gps_data_animals.gps_validity_code = 1) GROUP BY gps_data_animals.animals_id ORDER BY gps_data_animals.animals_id;


--
-- TOC entry 3400 (class 0 OID 0)
-- Dependencies: 200
-- Name: VIEW view_convex_hulls; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_convex_hulls IS 'GPS locations - Minimum convex polygons.';


SET search_path = lu_tables, pg_catalog;

--
-- TOC entry 180 (class 1259 OID 107975)
-- Name: lu_age_class; Type: TABLE; Schema: lu_tables; Owner: -; Tablespace: 
--

CREATE TABLE lu_age_class (
    age_class_code integer NOT NULL,
    age_class_description character varying
);


--
-- TOC entry 3401 (class 0 OID 0)
-- Dependencies: 180
-- Name: TABLE lu_age_class; Type: COMMENT; Schema: lu_tables; Owner: -
--

COMMENT ON TABLE lu_age_class IS 'Look up table for age classes.';


--
-- TOC entry 179 (class 1259 OID 107967)
-- Name: lu_species; Type: TABLE; Schema: lu_tables; Owner: -; Tablespace: 
--

CREATE TABLE lu_species (
    species_code integer NOT NULL,
    species_description character varying
);


--
-- TOC entry 3402 (class 0 OID 0)
-- Dependencies: 179
-- Name: TABLE lu_species; Type: COMMENT; Schema: lu_tables; Owner: -
--

COMMENT ON TABLE lu_species IS 'Look up table for species.';


SET search_path = main, pg_catalog;

--
-- TOC entry 178 (class 1259 OID 107957)
-- Name: animals; Type: TABLE; Schema: main; Owner: -; Tablespace: 
--

CREATE TABLE animals (
    animals_id integer NOT NULL,
    animals_code character varying(20) NOT NULL,
    name character varying(40),
    sex character(1),
    age_class_code integer,
    species_code integer,
    note character varying,
    insert_timestamp timestamp with time zone DEFAULT now(),
    update_timestamp timestamp with time zone DEFAULT now(),
    CONSTRAINT sex_check CHECK (((sex = 'm'::bpchar) OR (sex = 'f'::bpchar)))
);


--
-- TOC entry 3403 (class 0 OID 0)
-- Dependencies: 178
-- Name: TABLE animals; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON TABLE animals IS 'Animals catalog with the main information on individuals.';


SET search_path = analysis, pg_catalog;

--
-- TOC entry 198 (class 1259 OID 109219)
-- Name: view_gps_locations; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_gps_locations AS
    SELECT gps_data_animals.gps_data_animals_id, gps_data_animals.animals_id, animals.name, timezone('UTC'::text, gps_data_animals.acquisition_time) AS time_utc, animals.sex, lu_age_class.age_class_description, lu_species.species_description, gps_data_animals.geom FROM main.gps_data_animals, main.animals, lu_tables.lu_age_class, lu_tables.lu_species WHERE ((((gps_data_animals.animals_id = animals.animals_id) AND (animals.age_class_code = lu_age_class.age_class_code)) AND (animals.species_code = lu_species.species_code)) AND (gps_data_animals.gps_validity_code = 1));


--
-- TOC entry 3404 (class 0 OID 0)
-- Dependencies: 198
-- Name: VIEW view_gps_locations; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_gps_locations IS 'GPS locations.';


--
-- TOC entry 225 (class 1259 OID 117524)
-- Name: view_locations_buffer; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_locations_buffer AS
    SELECT gps_data_animals.animals_id, (public.st_union(public.st_buffer(gps_data_animals.geom, (0.001)::double precision)))::public.geometry(MultiPolygon,4326) AS geom FROM main.gps_data_animals WHERE (gps_data_animals.gps_validity_code = 1) GROUP BY gps_data_animals.animals_id ORDER BY gps_data_animals.animals_id;


--
-- TOC entry 3405 (class 0 OID 0)
-- Dependencies: 225
-- Name: VIEW view_locations_buffer; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_locations_buffer IS 'GPS locations - Buffer (dissolved) of 0.001 degrees.';


--
-- TOC entry 228 (class 1259 OID 117536)
-- Name: view_probability_grid_traj; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_probability_grid_traj AS
    WITH setx AS (SELECT gps_data_animals.gps_data_animals_id, gps_data_animals.animals_id, public.st_makeline(gps_data_animals.geom, lead(gps_data_animals.geom, (-1)) OVER (PARTITION BY gps_data_animals.animals_id ORDER BY gps_data_animals.acquisition_time)) AS geom, public.st_length((public.st_makeline(gps_data_animals.geom, lead(gps_data_animals.geom, (-1)) OVER (PARTITION BY gps_data_animals.animals_id ORDER BY gps_data_animals.acquisition_time)))::public.geography) AS line_length, CASE WHEN ((date_part('epoch'::text, gps_data_animals.acquisition_time) - date_part('epoch'::text, lead(gps_data_animals.acquisition_time, (-1)) OVER (PARTITION BY gps_data_animals.animals_id ORDER BY gps_data_animals.acquisition_time))) < (((60 * 60) * 24))::double precision) THEN (date_part('epoch'::text, gps_data_animals.acquisition_time) - date_part('epoch'::text, lead(gps_data_animals.acquisition_time, (-1)) OVER (PARTITION BY gps_data_animals.animals_id ORDER BY gps_data_animals.acquisition_time))) ELSE (0)::double precision END AS time_spent FROM main.gps_data_animals WHERE ((gps_data_animals.gps_validity_code = 1) AND (gps_data_animals.animals_id = 1)) ORDER BY gps_data_animals.acquisition_time), gridx AS (SELECT setx.animals_id, tools.create_grid(public.st_collect(setx.geom), 100) AS cell FROM setx GROUP BY setx.animals_id) SELECT ((a.animals_id * 10000) + a.cell_id) AS id, a.animals_id, a.cell_id, (public.st_transform(a.geom, 4326))::public.geometry(Polygon,4326) AS geom, (((sum(a.segment_time_spent) / (60)::double precision) / (60)::double precision))::integer AS hours_spent FROM (SELECT gridx.animals_id, (gridx.cell).cell_id AS cell_id, CASE setx.line_length WHEN 0 THEN setx.time_spent ELSE ((setx.time_spent * public.st_length(public.st_intersection(public.st_transform(setx.geom, public.st_srid((SELECT (gridx.cell).geom AS geom FROM gridx LIMIT 1))), (gridx.cell).geom))) / setx.line_length) END AS segment_time_spent, (gridx.cell).geom AS geom FROM gridx, setx WHERE ((public.st_intersects(public.st_transform(setx.geom, public.st_srid((SELECT (gridx.cell).geom AS geom FROM gridx LIMIT 1))), (gridx.cell).geom) AND (setx.time_spent > (0)::double precision)) AND (setx.animals_id = gridx.animals_id))) a GROUP BY a.animals_id, a.cell_id, a.geom HAVING (sum(a.segment_time_spent) > (0)::double precision);


--
-- TOC entry 3406 (class 0 OID 0)
-- Dependencies: 228
-- Name: VIEW view_probability_grid_traj; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_probability_grid_traj IS 'This view presents the SQL code to calculate the time spent by an animal on every cell of a grid with a defined resolution, which correspond to a probability surface. Trajectory (segments between locations) is considered. Each segment represents the time spent between the two locations. This view calls the function "tools.reate_grid". This is a view with pure SQL, but this tool can be coded into a function that using temporary tables ad some other optimized approaches, can speed up the processing time. In this case, just animals 1 is returned.';


SET search_path = env_data, pg_catalog;

--
-- TOC entry 204 (class 1259 OID 109258)
-- Name: study_area; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE study_area (
    gid integer NOT NULL,
    study_area numeric(10,0),
    study_name character varying(50),
    geom public.geometry(MultiPolygon,4326)
);


--
-- TOC entry 3407 (class 0 OID 0)
-- Dependencies: 204
-- Name: TABLE study_area; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE study_area IS 'Layer (polygons) of the boundaries of the study area.';


SET search_path = analysis, pg_catalog;

--
-- TOC entry 229 (class 1259 OID 117543)
-- Name: view_test_randompoints; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_test_randompoints AS
    SELECT row_number() OVER () AS id, (a.geom)::public.geometry(Point,4326) AS geom FROM (SELECT tools.randompoints((SELECT study_area.geom FROM env_data.study_area), 100) AS geom) a;


--
-- TOC entry 3408 (class 0 OID 0)
-- Dependencies: 229
-- Name: VIEW view_test_randompoints; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_test_randompoints IS 'This view is a test that shows 100 random points (generated every time that the view is called) into the boundaries of the first polygon stored in the home_ranges_mcp table.';


--
-- TOC entry 199 (class 1259 OID 109223)
-- Name: view_trajectories; Type: VIEW; Schema: analysis; Owner: -
--

CREATE VIEW view_trajectories AS
    SELECT sel_subquery.animals_id, (public.st_makeline(sel_subquery.geom))::public.geometry(LineString,4326) AS geom FROM (SELECT gps_data_animals.animals_id, gps_data_animals.geom, gps_data_animals.acquisition_time FROM main.gps_data_animals WHERE (gps_data_animals.gps_validity_code = 1) ORDER BY gps_data_animals.animals_id, gps_data_animals.acquisition_time) sel_subquery GROUP BY sel_subquery.animals_id;


--
-- TOC entry 3409 (class 0 OID 0)
-- Dependencies: 199
-- Name: VIEW view_trajectories; Type: COMMENT; Schema: analysis; Owner: -
--

COMMENT ON VIEW view_trajectories IS 'GPS locations - Trajectories.';


SET search_path = env_data, pg_catalog;

--
-- TOC entry 208 (class 1259 OID 109283)
-- Name: adm_boundaries; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE adm_boundaries (
    gid integer NOT NULL,
    cod_reg integer,
    cod_pro integer,
    pro_com integer,
    nome_com character varying(58),
    geom public.geometry(MultiPolygon,4326)
);


--
-- TOC entry 3410 (class 0 OID 0)
-- Dependencies: 208
-- Name: TABLE adm_boundaries; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE adm_boundaries IS 'Layer (polygons) of administrative boundaries (comuni).';


--
-- TOC entry 207 (class 1259 OID 109281)
-- Name: adm_boundaries_gid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE adm_boundaries_gid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3411 (class 0 OID 0)
-- Dependencies: 207
-- Name: adm_boundaries_gid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE adm_boundaries_gid_seq OWNED BY adm_boundaries.gid;


--
-- TOC entry 212 (class 1259 OID 109323)
-- Name: corine_land_cover; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE corine_land_cover (
    rid integer NOT NULL,
    rast public.raster
);


--
-- TOC entry 3412 (class 0 OID 0)
-- Dependencies: 212
-- Name: TABLE corine_land_cover; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE corine_land_cover IS 'Layer (raster) of land cover (from Corine project).';


--
-- TOC entry 213 (class 1259 OID 109346)
-- Name: corine_land_cover_legend; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE corine_land_cover_legend (
    grid_code integer NOT NULL,
    clc_l3_code character(3),
    label1 character varying,
    label2 character varying,
    label3 character varying
);


--
-- TOC entry 3413 (class 0 OID 0)
-- Dependencies: 213
-- Name: TABLE corine_land_cover_legend; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE corine_land_cover_legend IS 'Legend of Corine land cover, associating the numeric code to the three nested levels.';


--
-- TOC entry 211 (class 1259 OID 109321)
-- Name: corine_land_cover_rid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE corine_land_cover_rid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3414 (class 0 OID 0)
-- Dependencies: 211
-- Name: corine_land_cover_rid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE corine_land_cover_rid_seq OWNED BY corine_land_cover.rid;


--
-- TOC entry 202 (class 1259 OID 109246)
-- Name: meteo_stations; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE meteo_stations (
    gid integer NOT NULL,
    station_id numeric(10,0),
    geom public.geometry(Point,4326)
);


--
-- TOC entry 3415 (class 0 OID 0)
-- Dependencies: 202
-- Name: TABLE meteo_stations; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE meteo_stations IS 'Layer (points) of meteo stations.';


--
-- TOC entry 201 (class 1259 OID 109244)
-- Name: meteo_stations_gid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE meteo_stations_gid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3416 (class 0 OID 0)
-- Dependencies: 201
-- Name: meteo_stations_gid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE meteo_stations_gid_seq OWNED BY meteo_stations.gid;


--
-- TOC entry 215 (class 1259 OID 109357)
-- Name: ndvi_modis; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE ndvi_modis (
    rid integer NOT NULL,
    rast public.raster,
    filename text,
    acquisition_range daterange
);


--
-- TOC entry 3417 (class 0 OID 0)
-- Dependencies: 215
-- Name: TABLE ndvi_modis; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE ndvi_modis IS 'Table that stores values of smoothed MODIS NDVI (16-day periods).';


--
-- TOC entry 214 (class 1259 OID 109355)
-- Name: ndvi_modis_rid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE ndvi_modis_rid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3418 (class 0 OID 0)
-- Dependencies: 214
-- Name: ndvi_modis_rid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE ndvi_modis_rid_seq OWNED BY ndvi_modis.rid;


--
-- TOC entry 206 (class 1259 OID 109270)
-- Name: roads; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE roads (
    gid integer NOT NULL,
    highway character varying(80),
    geom public.geometry(MultiLineString,4326)
);


--
-- TOC entry 3419 (class 0 OID 0)
-- Dependencies: 206
-- Name: TABLE roads; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE roads IS 'Layer (lines) of roads network.';


--
-- TOC entry 205 (class 1259 OID 109268)
-- Name: roads_gid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE roads_gid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3420 (class 0 OID 0)
-- Dependencies: 205
-- Name: roads_gid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE roads_gid_seq OWNED BY roads.gid;


--
-- TOC entry 210 (class 1259 OID 109300)
-- Name: srtm_dem; Type: TABLE; Schema: env_data; Owner: -; Tablespace: 
--

CREATE TABLE srtm_dem (
    rid integer NOT NULL,
    rast public.raster
);


--
-- TOC entry 3421 (class 0 OID 0)
-- Dependencies: 210
-- Name: TABLE srtm_dem; Type: COMMENT; Schema: env_data; Owner: -
--

COMMENT ON TABLE srtm_dem IS 'Layer (raster) of digital elevation model (from SRTM project).';


--
-- TOC entry 209 (class 1259 OID 109298)
-- Name: srtm_dem_rid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE srtm_dem_rid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3422 (class 0 OID 0)
-- Dependencies: 209
-- Name: srtm_dem_rid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE srtm_dem_rid_seq OWNED BY srtm_dem.rid;


--
-- TOC entry 203 (class 1259 OID 109256)
-- Name: study_area_gid_seq; Type: SEQUENCE; Schema: env_data; Owner: -
--

CREATE SEQUENCE study_area_gid_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3423 (class 0 OID 0)
-- Dependencies: 203
-- Name: study_area_gid_seq; Type: SEQUENCE OWNED BY; Schema: env_data; Owner: -
--

ALTER SEQUENCE study_area_gid_seq OWNED BY study_area.gid;


SET search_path = lu_tables, pg_catalog;

--
-- TOC entry 216 (class 1259 OID 117434)
-- Name: lu_gps_validity; Type: TABLE; Schema: lu_tables; Owner: -; Tablespace: 
--

CREATE TABLE lu_gps_validity (
    gps_validity_code integer NOT NULL,
    gps_validity_description character varying
);


--
-- TOC entry 3424 (class 0 OID 0)
-- Dependencies: 216
-- Name: TABLE lu_gps_validity; Type: COMMENT; Schema: lu_tables; Owner: -
--

COMMENT ON TABLE lu_gps_validity IS 'Look up table for GPS positions validity codes.';


SET search_path = main, pg_catalog;

--
-- TOC entry 176 (class 1259 OID 107908)
-- Name: gps_data; Type: TABLE; Schema: main; Owner: -; Tablespace: 
--

CREATE TABLE gps_data (
    gps_data_id integer NOT NULL,
    gps_sensors_code character varying,
    line_no integer,
    utc_date date,
    utc_time time without time zone,
    lmt_date date,
    lmt_time time without time zone,
    ecef_x integer,
    ecef_y integer,
    ecef_z integer,
    latitude double precision,
    longitude double precision,
    height double precision,
    dop double precision,
    nav character varying(2),
    validated character varying(3),
    sats_used integer,
    ch01_sat_id integer,
    ch01_sat_cnr integer,
    ch02_sat_id integer,
    ch02_sat_cnr integer,
    ch03_sat_id integer,
    ch03_sat_cnr integer,
    ch04_sat_id integer,
    ch04_sat_cnr integer,
    ch05_sat_id integer,
    ch05_sat_cnr integer,
    ch06_sat_id integer,
    ch06_sat_cnr integer,
    ch07_sat_id integer,
    ch07_sat_cnr integer,
    ch08_sat_id integer,
    ch08_sat_cnr integer,
    ch09_sat_id integer,
    ch09_sat_cnr integer,
    ch10_sat_id integer,
    ch10_sat_cnr integer,
    ch11_sat_id integer,
    ch11_sat_cnr integer,
    ch12_sat_id integer,
    ch12_sat_cnr integer,
    main_vol double precision,
    bu_vol double precision,
    temp double precision,
    easting integer,
    northing integer,
    remarks character varying,
    insert_timestamp timestamp with time zone DEFAULT now(),
    acquisition_time timestamp with time zone
);


--
-- TOC entry 3425 (class 0 OID 0)
-- Dependencies: 176
-- Name: TABLE gps_data; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON TABLE gps_data IS 'Table that stores raw data as they come from the sensors (plus the ID of the sensor).';


--
-- TOC entry 183 (class 1259 OID 108026)
-- Name: gps_data_animals_gps_data_animals_id_seq; Type: SEQUENCE; Schema: main; Owner: -
--

CREATE SEQUENCE gps_data_animals_gps_data_animals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3426 (class 0 OID 0)
-- Dependencies: 183
-- Name: gps_data_animals_gps_data_animals_id_seq; Type: SEQUENCE OWNED BY; Schema: main; Owner: -
--

ALTER SEQUENCE gps_data_animals_gps_data_animals_id_seq OWNED BY gps_data_animals.gps_data_animals_id;


--
-- TOC entry 175 (class 1259 OID 107906)
-- Name: gps_data_gps_data_id_seq; Type: SEQUENCE; Schema: main; Owner: -
--

CREATE SEQUENCE gps_data_gps_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3427 (class 0 OID 0)
-- Dependencies: 175
-- Name: gps_data_gps_data_id_seq; Type: SEQUENCE OWNED BY; Schema: main; Owner: -
--

ALTER SEQUENCE gps_data_gps_data_id_seq OWNED BY gps_data.gps_data_id;


--
-- TOC entry 177 (class 1259 OID 107932)
-- Name: gps_sensors; Type: TABLE; Schema: main; Owner: -; Tablespace: 
--

CREATE TABLE gps_sensors (
    gps_sensors_id integer NOT NULL,
    gps_sensors_code character varying NOT NULL,
    purchase_date date,
    frequency double precision,
    vendor character varying,
    model character varying,
    sim character varying,
    insert_timestamp timestamp with time zone DEFAULT now(),
    update_timestamp timestamp with time zone DEFAULT now(),
    CONSTRAINT purchase_date_check CHECK ((purchase_date > '2000-01-01'::date))
);


--
-- TOC entry 3428 (class 0 OID 0)
-- Dependencies: 177
-- Name: TABLE gps_sensors; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON TABLE gps_sensors IS 'GPS sensors catalog.';


--
-- TOC entry 182 (class 1259 OID 108005)
-- Name: gps_sensors_animals; Type: TABLE; Schema: main; Owner: -; Tablespace: 
--

CREATE TABLE gps_sensors_animals (
    gps_sensors_animals_id integer NOT NULL,
    animals_id integer NOT NULL,
    gps_sensors_id integer NOT NULL,
    start_time timestamp with time zone NOT NULL,
    end_time timestamp with time zone,
    notes character varying,
    insert_timestamp timestamp with time zone DEFAULT now(),
    update_timestamp timestamp with time zone DEFAULT now(),
    CONSTRAINT time_interval_check CHECK ((end_time > start_time))
);


--
-- TOC entry 3429 (class 0 OID 0)
-- Dependencies: 182
-- Name: TABLE gps_sensors_animals; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON TABLE gps_sensors_animals IS 'Table that stores information of deployments of sensors on animals.';


--
-- TOC entry 181 (class 1259 OID 108003)
-- Name: gps_sensors_animals_gps_sensors_animals_id_seq; Type: SEQUENCE; Schema: main; Owner: -
--

CREATE SEQUENCE gps_sensors_animals_gps_sensors_animals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3430 (class 0 OID 0)
-- Dependencies: 181
-- Name: gps_sensors_animals_gps_sensors_animals_id_seq; Type: SEQUENCE OWNED BY; Schema: main; Owner: -
--

ALTER SEQUENCE gps_sensors_animals_gps_sensors_animals_id_seq OWNED BY gps_sensors_animals.gps_sensors_animals_id;


--
-- TOC entry 218 (class 1259 OID 117451)
-- Name: view_locations_set; Type: VIEW; Schema: main; Owner: -
--

CREATE VIEW view_locations_set AS
    SELECT gps_data_animals.animals_id, gps_data_animals.acquisition_time, CASE WHEN (gps_data_animals.gps_validity_code = 1) THEN gps_data_animals.geom ELSE NULL::public.geometry END AS geom FROM gps_data_animals WHERE (gps_data_animals.gps_validity_code <> 21) ORDER BY gps_data_animals.animals_id, gps_data_animals.acquisition_time;


--
-- TOC entry 3431 (class 0 OID 0)
-- Dependencies: 218
-- Name: VIEW view_locations_set; Type: COMMENT; Schema: main; Owner: -
--

COMMENT ON VIEW view_locations_set IS 'View that stores the core information of the set of GPS positions (id of the animal, the acquisition time and the geometry), where non valid records are represented with empty geometry.';


SET search_path = tools, pg_catalog;

--
-- TOC entry 221 (class 1259 OID 117482)
-- Name: unique_id_seq; Type: SEQUENCE; Schema: tools; Owner: -
--

CREATE SEQUENCE unique_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


--
-- TOC entry 3432 (class 0 OID 0)
-- Dependencies: 221
-- Name: SEQUENCE unique_id_seq; Type: COMMENT; Schema: tools; Owner: -
--

COMMENT ON SEQUENCE unique_id_seq IS 'Sequence used to generate unique number for routines that need it (e.g. functions that need to generate temporary tables with unique names).';


SET search_path = analysis, pg_catalog;

--
-- TOC entry 3251 (class 2604 OID 117508)
-- Name: home_ranges_mcp_id; Type: DEFAULT; Schema: analysis; Owner: -
--

ALTER TABLE ONLY home_ranges_mcp ALTER COLUMN home_ranges_mcp_id SET DEFAULT nextval('home_ranges_mcp_home_ranges_mcp_id_seq'::regclass);


--
-- TOC entry 3249 (class 2604 OID 117460)
-- Name: trajectories_id; Type: DEFAULT; Schema: analysis; Owner: -
--

ALTER TABLE ONLY trajectories ALTER COLUMN trajectories_id SET DEFAULT nextval('trajectories_trajectories_id_seq'::regclass);


SET search_path = env_data, pg_catalog;

--
-- TOC entry 3212 (class 2604 OID 109286)
-- Name: gid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY adm_boundaries ALTER COLUMN gid SET DEFAULT nextval('adm_boundaries_gid_seq'::regclass);


--
-- TOC entry 3225 (class 2604 OID 109326)
-- Name: rid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY corine_land_cover ALTER COLUMN rid SET DEFAULT nextval('corine_land_cover_rid_seq'::regclass);


--
-- TOC entry 3209 (class 2604 OID 109249)
-- Name: gid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY meteo_stations ALTER COLUMN gid SET DEFAULT nextval('meteo_stations_gid_seq'::regclass);


--
-- TOC entry 3237 (class 2604 OID 109360)
-- Name: rid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY ndvi_modis ALTER COLUMN rid SET DEFAULT nextval('ndvi_modis_rid_seq'::regclass);


--
-- TOC entry 3211 (class 2604 OID 109273)
-- Name: gid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY roads ALTER COLUMN gid SET DEFAULT nextval('roads_gid_seq'::regclass);


--
-- TOC entry 3213 (class 2604 OID 109303)
-- Name: rid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY srtm_dem ALTER COLUMN rid SET DEFAULT nextval('srtm_dem_rid_seq'::regclass);


--
-- TOC entry 3210 (class 2604 OID 109261)
-- Name: gid; Type: DEFAULT; Schema: env_data; Owner: -
--

ALTER TABLE ONLY study_area ALTER COLUMN gid SET DEFAULT nextval('study_area_gid_seq'::regclass);


SET search_path = main, pg_catalog;

--
-- TOC entry 3193 (class 2604 OID 107911)
-- Name: gps_data_id; Type: DEFAULT; Schema: main; Owner: -
--

ALTER TABLE ONLY gps_data ALTER COLUMN gps_data_id SET DEFAULT nextval('gps_data_gps_data_id_seq'::regclass);


--
-- TOC entry 3205 (class 2604 OID 108031)
-- Name: gps_data_animals_id; Type: DEFAULT; Schema: main; Owner: -
--

ALTER TABLE ONLY gps_data_animals ALTER COLUMN gps_data_animals_id SET DEFAULT nextval('gps_data_animals_gps_data_animals_id_seq'::regclass);


--
-- TOC entry 3201 (class 2604 OID 108008)
-- Name: gps_sensors_animals_id; Type: DEFAULT; Schema: main; Owner: -
--

ALTER TABLE ONLY gps_sensors_animals ALTER COLUMN gps_sensors_animals_id SET DEFAULT nextval('gps_sensors_animals_gps_sensors_animals_id_seq'::regclass);


SET search_path = analysis, pg_catalog;

--
-- TOC entry 3361 (class 0 OID 117505)
-- Dependencies: 224
-- Data for Name: home_ranges_mcp; Type: TABLE DATA; Schema: analysis; Owner: -
--

INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (1, 1, '2005-10-18 22:00:54+02', '2006-10-29 13:00:49+01', 'test 0.1', 'postgres', 165, 0.91037, '0106000020E6100000010000000103000000010000000A0000005C4B1295571C2640C99EF3098F004740B537537CD71726401F436C55B7004740F18288D4B417264011D77B3BD3004740FAE70E07AE172640A3224E27D900474075649B0AA71726408EFFA7870D0147405677D1FCD61726401245ED235C014740B6AEE29EF8172640A03369537501474066AEC21CF32026408173EBDA83014740C6B1E4E088202640E13C4107020147405C4B1295571C2640C99EF3098F004740', 0.10000000000000001, '2014-01-07 03:26:14+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (2, 1, '2005-10-18 22:00:54+02', '2006-10-29 13:00:49+01', 'test 0.5', 'postgres', 824, 3.12442, '0106000020E6100000010000000103000000010000000F000000791E81E4421E26400B6F206E98FF46402F08F6A974152640CA4D79196A00474006989E550B152640EAB69FE7AA004740C1368710FC1426404AFC9B61B40047405902846ADE1426408577B988EF00474038D1F8CFE41426401863BBD639014740705B00BFEB142640496F13494F0147400BB77C24251526402F84413E8D0147405D4D43F927152640CEB74FD88F01474099092BBA9A152640EE6F1AEC9701474048CCA266FE1F26407CDC6DCD0C0247408BE88D4935222640BA13ECBFCE014740973C9E961F242640A6A036058D0147404BB846BA551E26404533AA679CFF4640791E81E4421E26400B6F206E98FF4640', 0.5, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (3, 1, '2005-10-18 22:00:54+02', '2006-10-29 13:00:49+01', 'test 0.75', 'postgres', 1235, 4.52416, '0106000020E61000000100000001030000000100000017000000791E81E4421E26400B6F206E98FF46404E51E4FF4413264093E8C0CD3D004740163C3BCF8E122640CCBF4CCA93004740C878944A781226409EC9A3C0A7004740A67AD74576122640AD78D965AE00474005AEE1D869122640361081D9F3004740BC2429441B132640AE2589CA2B0147401E0C8684391326403EC10F18350147401CF1BFF0A5132640730866964E014740E5F27A8BE2142640D56FCBEC9801474053019C830C1526403673486AA101474048CCA266FE1F26407CDC6DCD0C024740472E49AF172426409C0B6D4AC3014740E3D011AFA1262640D766086C7301474011A0F023C82626408B6D52D158014740BE6BD097DE262640AB05F69848014740C7C26BF2EF262640C1CAFCFE2801474055D33FD2F126264000E72043220147403CE75DABF32626405F9E29CF17014740FCA71B28F02626400BCD1A61F60047400AAAFC21EC262640F2F16492EC0047404A7F8A3E7A26264069835D5ECA004740791E81E4421E26400B6F206E98FF4640', 0.75, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (4, 1, '2005-10-18 22:00:54+02', '2006-10-29 13:00:49+01', 'test 1', 'postgres', 1647, 8.08596, '0106000020E6100000010000000103000000010000000E000000791E81E4421E26400B6F206E98FF464034B33B495F0D2640911216702A0047409786866AEF082640E4EB7C89C20047409628D604AC082640CE77A686EC004740AF175465290F264017E4784F2F024740CCA5A77E940F2640F7FAA4243502474048CCA266FE1F26407CDC6DCD0C024740114EB0A481282640AE22ED90AC01474089B3226AA22B2640E9616875720147403354200E6D2B2640F570B8454F0147404B3904334B2B2640C550F3FA380147407D676AC82E2B26407F315BB22A01474016B0D3FEBD2A2640524E0F6503014740791E81E4421E26400B6F206E98FF4640', 1, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (5, 1, '2006-01-01 00:00:00+01', '2006-01-10 00:00:00+01', 'test start and end', 'postgres', 37, 0.18170, '0106000020E6100000010000000103000000010000000500000006E4011BC627264023F77475C7004740B024D4676225264049CF9959F00047406BA1BF2BDD242640098D60E3FA004740114EB0A481282640AE22ED90AC01474006E4011BC627264023F77475C7004740', 1, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (6, 1, '2005-10-18 22:00:54+02', '2006-10-29 13:00:49+01', 'test all animals at 0.9', 'postgres', 1482, 5.25487, '0106000020E6100000010000000103000000010000000D000000791E81E4421E26400B6F206E98FF46404E51E4FF4413264093E8C0CD3D00474085BB69E9C0102640359B221BA300474049A2F20A9F1026407CC0F2D6AF004740E5F27A8BE2142640D56FCBEC9801474053019C830C1526403673486AA101474048CCA266FE1F26407CDC6DCD0C024740114EB0A481282640AE22ED90AC014740FCDEA63FFB29264049F196506601474009A9DBD9572A2640CE5C96653E0147402461DF4E222A264025027A86150147405805C5EAEA29264061E35531F0004740791E81E4421E26400B6F206E98FF4640', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (7, 2, '2005-03-20 17:03:14+01', '2006-05-27 18:02:25+02', 'test all animals at 0.9', 'postgres', 1975, 9.03271, '0106000020E6100000010000000103000000010000001200000082A3891CC720264099A9FF626CFE4640931CB0ABC91F2640A72A23E472FE46407E22AA4B211F2640635A400D84FE46401B9B1DA9BE0B264059C57652BA004740C2C073EFE1062640936A44D554024740924FD9FA33052640A583F57F0E034740FB952941DA0A264095F661180B044740636765564A0E2640C2E3367579034740E9876C7B16232640BA30D28BDAFF46403FB1A94E62232640AB5B3D27BDFF46402F8617E9EA232640720635327EFF46409B536E47DD232640D6BCF44075FF464025404D2D5B232640016FDCBD37FF4640B80F2F3EBB222640CD881C6CFBFE4640B3A492F073222640E6F8B2FEE0FE464058EDF54949222640EE34E37ED1FE46404940964FA02126403260240F9FFE464082A3891CC720264099A9FF626CFE4640', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (8, 3, '2005-10-23 22:00:53+02', '2006-10-28 14:01:18+02', 'test all animals at 0.9', 'postgres', 1643, 8.93319, '0106000020E61000000100000001030000000100000010000000C1255BB8071C264048A7AE7C960047405456D3F544172640D2C2651536014740F3B116FABB162640DBCB6C46BC01474026D98FB9D716264082A6CA9CD30147401616DC0F782C2640F977C773C7064740399270C6D52D26401BD1E231A806474003E89C44292E2640F0E65F819E064740EA33B10AE52E2640CB147310740647408D7C5EF1D42F26403DFD56A137064740BD81131B3D302640466A90381A064740352493F881302640E242797A0006474025D7039C943026400A99D0C9F705474009F02774A8302640FBDAE9BDE705474004FB54BA602D264002637D03930347408F47BAB07F202640B50309E5D8004740C1255BB8071C264048A7AE7C96004740', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (9, 4, '2005-10-21 22:00:47+02', '2007-02-09 09:11:24+01', 'test all animals at 0.9', 'postgres', 2377, 9.74893, '0106000020E6100000010000000103000000010000001100000023B9A1CE81022640F28AF1AB83FE4640D8F6D104E5FD254029C709B88CFE4640B6C4252257FD254061D394AEAAFE4640C0C3FEFC08FD25402BACAF09C4FE4640E5492DEF05FD254006FEA666C5FE4640ADA23F34F3FC254097B3D2FFCDFE46405F155E38C6FC254050752387E3FE464055B7C4DB94FC2540C05E61C1FDFE4640A0408E9C3BFC2540AAAE8DC23FFF4640931F4CE548FE25408C1766FC56004740602B572E0A0D2640BADD2637E50247400D2950D54E1626402C802903070347407A0FE1A991172640EDBB22F8DF024740D5E8D500A51926403E986FD811024740C79459CEEF1926402651D43EC2014740BA9F53909F192640CAD2985A0001474023B9A1CE81022640F28AF1AB83FE4640', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (10, 5, '2006-11-13 01:02:24+01', '2008-03-15 09:01:37+01', 'test all animals at 0.9', 'postgres', 2426, 6.57880, '0106000020E6100000010000000103000000010000000D0000006F754A51781A2640BD981C2DBDFF46402BFE4A427D06264059AA66310C0047406A45E570F7FF25406968A8F68E004740DF18028063FF2540C7968A32C0004740779C476A46FF2540ABAA3C26F7004740FFD2FDF73BFF254069519FE40E0147409E6230DA3EFF2540EF5F0FDC3701474082CEFF0648FF254042E32E675B014740D88E988F5AFF25406ABE4A3E76014740A038807EDF0326408ACE8D44C301474029C297D3AF162640BC0E304ED7014740C43EA65AB01D26408901B7932D0147406F754A51781A2640BD981C2DBDFF4640', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');
INSERT INTO home_ranges_mcp (home_ranges_mcp_id, animals_id, start_time, end_time, description, ref_user, num_locations, area, geom, percentage, insert_timestamp, original_data_set) VALUES (11, 6, '2005-04-04 08:01:41+02', '2005-05-06 01:01:47+02', 'test all animals at 0.9', 'postgres', 250, 0.13362, '0106000020E61000000100000001030000000100000011000000ACD50DCAEA202640E85894C8F4074740A601CD316A202640A0A696ADF5074740DF86BDF506202640578C0464F9074740259012BBB61F2640BCFEC98404084740A24C593CAB1F2640DFFC868906084740359BC761301F264011F0B5B12308474001E4DFC2151F2640E69E64613E08474026B2C524121F26409E375F364B0847407336D373C11F26404B06802A6E08474095E3045C46202640C9C9C4AD82084740F34DE4935F202640B772405C83084740E3FA777DE62026404B18721184084740318969954F2126405CB3F0506F0847404EB10F57732126407A0BDA9A630847402EBD47478321264048E757CE3908474057CAD70A3F2126406424D63D0D084740ACD50DCAEA202640E85894C8F4074740', 0.90000000000000002, '2014-01-07 03:26:21+01', 'main.view_locations_set');


--
-- TOC entry 3433 (class 0 OID 0)
-- Dependencies: 223
-- Name: home_ranges_mcp_home_ranges_mcp_id_seq; Type: SEQUENCE SET; Schema: analysis; Owner: -
--

SELECT pg_catalog.setval('home_ranges_mcp_home_ranges_mcp_id_seq', 11, true);


--
-- TOC entry 3362 (class 0 OID 117547)
-- Dependencies: 230
-- Data for Name: test_randompoints; Type: TABLE DATA; Schema: analysis; Owner: -
--

INSERT INTO test_randompoints (id, geom) VALUES (1, '0101000020E610000052AA71A2AFEA2540B10F647E0F024740');
INSERT INTO test_randompoints (id, geom) VALUES (2, '0101000020E6100000150D8B89BF1026408A975C6042FB4640');
INSERT INTO test_randompoints (id, geom) VALUES (3, '0101000020E61000009058810F4C232640B1732336CF004740');
INSERT INTO test_randompoints (id, geom) VALUES (4, '0101000020E61000003A5EB4D24E1C264004BD696A140F4740');
INSERT INTO test_randompoints (id, geom) VALUES (5, '0101000020E6100000B9FF15DF3DF22540C4CD756DC0FE4640');
INSERT INTO test_randompoints (id, geom) VALUES (6, '0101000020E610000038E6BDA7431D2640108DFF9159FB4640');
INSERT INTO test_randompoints (id, geom) VALUES (7, '0101000020E6100000ED7A2B4FFC182640D81FFEE34E0F4740');
INSERT INTO test_randompoints (id, geom) VALUES (8, '0101000020E6100000F19895906FFA25405C05F25EF3094740');
INSERT INTO test_randompoints (id, geom) VALUES (9, '0101000020E6100000AB87DA3BA1EA25407FEE8C0897044740');
INSERT INTO test_randompoints (id, geom) VALUES (10, '0101000020E6100000C4D56DDCC3F025402293F38F460A4740');
INSERT INTO test_randompoints (id, geom) VALUES (11, '0101000020E6100000A73FA9FA54072640073561DA88094740');
INSERT INTO test_randompoints (id, geom) VALUES (12, '0101000020E61000002BC53972C3162640649CC86EA60D4740');
INSERT INTO test_randompoints (id, geom) VALUES (13, '0101000020E6100000BCF94BAA1D1F264063EA9056AAFB4640');
INSERT INTO test_randompoints (id, geom) VALUES (14, '0101000020E610000089662503FB132640F94DCBB7B0074740');
INSERT INTO test_randompoints (id, geom) VALUES (15, '0101000020E61000008FCF5248BC2826405E08D91EAD044740');
INSERT INTO test_randompoints (id, geom) VALUES (16, '0101000020E610000021F107B8B2E8254091CA306ED7FD4640');
INSERT INTO test_randompoints (id, geom) VALUES (17, '0101000020E610000060D28895BBE32540AA2B397312014740');
INSERT INTO test_randompoints (id, geom) VALUES (18, '0101000020E61000006BCB483A2AF92540D98FEE2E74064740');
INSERT INTO test_randompoints (id, geom) VALUES (19, '0101000020E6100000387D8E91D52C26409DFC382E3B0B4740');
INSERT INTO test_randompoints (id, geom) VALUES (20, '0101000020E6100000ACD670C164142640DDB9956E8F0A4740');
INSERT INTO test_randompoints (id, geom) VALUES (21, '0101000020E61000009CC8D98BFA162640BC5E28D379024740');
INSERT INTO test_randompoints (id, geom) VALUES (22, '0101000020E610000072D002A80C182640F86567A0E5074740');
INSERT INTO test_randompoints (id, geom) VALUES (23, '0101000020E6100000E79743B12E172640C9E0ECBA840D4740');
INSERT INTO test_randompoints (id, geom) VALUES (24, '0101000020E6100000F06BC2E8A6252640727149AAF60E4740');
INSERT INTO test_randompoints (id, geom) VALUES (25, '0101000020E61000002097417FFC1A2640232B86C5E0044740');
INSERT INTO test_randompoints (id, geom) VALUES (26, '0101000020E6100000F06F9EBFDE232640E87EBE47F6044740');
INSERT INTO test_randompoints (id, geom) VALUES (27, '0101000020E6100000151E975AFAF325408A86AB0896094740');
INSERT INTO test_randompoints (id, geom) VALUES (28, '0101000020E61000009CF7C04C08E52540B8084C36AA004740');
INSERT INTO test_randompoints (id, geom) VALUES (29, '0101000020E6100000A0F7347DB21B2640CBCABCC3CBFD4640');
INSERT INTO test_randompoints (id, geom) VALUES (30, '0101000020E6100000960411FA540A26404A68717CDF0B4740');
INSERT INTO test_randompoints (id, geom) VALUES (31, '0101000020E6100000BF3DF51977EE2540B5D76730DD084740');
INSERT INTO test_randompoints (id, geom) VALUES (32, '0101000020E6100000B597363274EA254001AD15B131FB4640');
INSERT INTO test_randompoints (id, geom) VALUES (33, '0101000020E6100000F92E069DD60B2640CE122142570E4740');
INSERT INTO test_randompoints (id, geom) VALUES (34, '0101000020E61000007FC5547E63E82540124E973642FC4640');
INSERT INTO test_randompoints (id, geom) VALUES (35, '0101000020E61000007E41DDA650122640E5C3047B19074740');
INSERT INTO test_randompoints (id, geom) VALUES (36, '0101000020E61000001373D708C52B2640630FA13EA6004740');
INSERT INTO test_randompoints (id, geom) VALUES (37, '0101000020E610000049A6143F4B1026400EE1AED3890B4740');
INSERT INTO test_randompoints (id, geom) VALUES (38, '0101000020E610000098AD8E335D272640F768CC45540E4740');
INSERT INTO test_randompoints (id, geom) VALUES (39, '0101000020E610000029EF6CFDA2F6254052ED6670ED024740');
INSERT INTO test_randompoints (id, geom) VALUES (40, '0101000020E610000045336D33492026409B35BF13E80E4740');
INSERT INTO test_randompoints (id, geom) VALUES (41, '0101000020E610000037D4524C5DFB2540FBC41EA2C9034740');
INSERT INTO test_randompoints (id, geom) VALUES (42, '0101000020E6100000B6E73CB457142640F0AFEABCF0094740');
INSERT INTO test_randompoints (id, geom) VALUES (43, '0101000020E61000007C4250F9F9072640592D65C608094740');
INSERT INTO test_randompoints (id, geom) VALUES (44, '0101000020E6100000DE043E29BBEB25404E8627FACEFC4640');
INSERT INTO test_randompoints (id, geom) VALUES (45, '0101000020E61000001DAE1ADCFB2E2640291BF654C9024740');
INSERT INTO test_randompoints (id, geom) VALUES (46, '0101000020E6100000CA17243650FE254088C56108AFFA4640');
INSERT INTO test_randompoints (id, geom) VALUES (47, '0101000020E6100000F1E9616E7E2C2640B235C5E28E074740');
INSERT INTO test_randompoints (id, geom) VALUES (48, '0101000020E61000004A6361407A02264094472EB768064740');
INSERT INTO test_randompoints (id, geom) VALUES (49, '0101000020E6100000F944EB58E4EF2540D0A49BFEDEFE4640');
INSERT INTO test_randompoints (id, geom) VALUES (50, '0101000020E61000002F3C636C0E00264051721094C4034740');
INSERT INTO test_randompoints (id, geom) VALUES (51, '0101000020E6100000D0EF601837152640952310ADB4084740');
INSERT INTO test_randompoints (id, geom) VALUES (52, '0101000020E61000008A0E7579AA0E2640AA5ED233E5074740');
INSERT INTO test_randompoints (id, geom) VALUES (53, '0101000020E61000004EE5786D4C2426403D0746010F0D4740');
INSERT INTO test_randompoints (id, geom) VALUES (54, '0101000020E6100000C4E6C244001D26400E1065B51B0C4740');
INSERT INTO test_randompoints (id, geom) VALUES (55, '0101000020E61000009C51B779F6F12540C0BCDC42BDFA4640');
INSERT INTO test_randompoints (id, geom) VALUES (56, '0101000020E61000006C24F41F11FD2540E89608181F014740');
INSERT INTO test_randompoints (id, geom) VALUES (57, '0101000020E6100000AFC5FE6AC10E2640896157F5F0014740');
INSERT INTO test_randompoints (id, geom) VALUES (58, '0101000020E6100000BD7BF5FF21242640E80914FFF2FA4640');
INSERT INTO test_randompoints (id, geom) VALUES (59, '0101000020E61000009C87BAD1E50426406477704E3B074740');
INSERT INTO test_randompoints (id, geom) VALUES (60, '0101000020E610000003916F599217264033F61B3F99FD4640');
INSERT INTO test_randompoints (id, geom) VALUES (61, '0101000020E6100000857BAFEB9EF72540B2EF5208ED024740');
INSERT INTO test_randompoints (id, geom) VALUES (62, '0101000020E6100000C6082E9002202640F09DB5E9330E4740');
INSERT INTO test_randompoints (id, geom) VALUES (63, '0101000020E610000024EFC9E34610264005640772A70B4740');
INSERT INTO test_randompoints (id, geom) VALUES (64, '0101000020E6100000E609A239D50D2640E2A71BDB2E084740');
INSERT INTO test_randompoints (id, geom) VALUES (65, '0101000020E610000041B2A2AE08EB2540F4CCC7BB18024740');
INSERT INTO test_randompoints (id, geom) VALUES (66, '0101000020E61000002C7E37A10C032640E02599C82CFF4640');
INSERT INTO test_randompoints (id, geom) VALUES (67, '0101000020E6100000CDE949A07F2226401B9DF8286E064740');
INSERT INTO test_randompoints (id, geom) VALUES (68, '0101000020E6100000B587AF538F2726407CEBB2208C014740');
INSERT INTO test_randompoints (id, geom) VALUES (69, '0101000020E61000009A85077DD31B26403A28A65E58044740');
INSERT INTO test_randompoints (id, geom) VALUES (70, '0101000020E6100000B4B751AC1D17264004C0F26B7CFB4640');
INSERT INTO test_randompoints (id, geom) VALUES (71, '0101000020E6100000FBFA59B9C6F4254043AF7F0B41FD4640');
INSERT INTO test_randompoints (id, geom) VALUES (72, '0101000020E61000007D1DB73342192640B42CAEA69D004740');
INSERT INTO test_randompoints (id, geom) VALUES (73, '0101000020E6100000A8B5C8623FF72540381E0305B6044740');
INSERT INTO test_randompoints (id, geom) VALUES (74, '0101000020E61000009929DDA71EF92540C618A66CD9014740');
INSERT INTO test_randompoints (id, geom) VALUES (75, '0101000020E61000001D36D9AF72FA2540E6E635EFDB024740');
INSERT INTO test_randompoints (id, geom) VALUES (76, '0101000020E6100000BF69E56FA4232640213B0BA3E2094740');
INSERT INTO test_randompoints (id, geom) VALUES (77, '0101000020E61000000D3F5969ECEF2540D65CB20AA0014740');
INSERT INTO test_randompoints (id, geom) VALUES (78, '0101000020E61000002935EECBD50E264057B0C58618004740');
INSERT INTO test_randompoints (id, geom) VALUES (79, '0101000020E6100000A7FD82F8C1E5254002910B471C034740');
INSERT INTO test_randompoints (id, geom) VALUES (80, '0101000020E61000000B41C08C102B2640C136C292E00B4740');
INSERT INTO test_randompoints (id, geom) VALUES (81, '0101000020E6100000795E89003C192640FD242B36E3044740');
INSERT INTO test_randompoints (id, geom) VALUES (82, '0101000020E610000086F369DC5DF22540D8A7071455014740');
INSERT INTO test_randompoints (id, geom) VALUES (83, '0101000020E610000008936CCA380F2640D6A825E6AF0C4740');
INSERT INTO test_randompoints (id, geom) VALUES (84, '0101000020E610000044589FCA0BE625405BF0300D2F004740');
INSERT INTO test_randompoints (id, geom) VALUES (85, '0101000020E61000000422F9714711264091B696056FFC4640');
INSERT INTO test_randompoints (id, geom) VALUES (86, '0101000020E6100000AB1FBE6FEB0B2640E423327CCE064740');
INSERT INTO test_randompoints (id, geom) VALUES (87, '0101000020E61000002F07E4D9B62826401E1D75487B074740');
INSERT INTO test_randompoints (id, geom) VALUES (88, '0101000020E610000045A390AF5A00264050751573A5034740');
INSERT INTO test_randompoints (id, geom) VALUES (89, '0101000020E610000023BC3CDBEE1B264000572F28BF014740');
INSERT INTO test_randompoints (id, geom) VALUES (90, '0101000020E61000007EB73B7B600F26406440B851C5064740');
INSERT INTO test_randompoints (id, geom) VALUES (91, '0101000020E6100000E01AC3F69A2B2640CB87356F050B4740');
INSERT INTO test_randompoints (id, geom) VALUES (92, '0101000020E6100000B09F78E8DB0C26405B581E765B094740');
INSERT INTO test_randompoints (id, geom) VALUES (93, '0101000020E610000081D39E8EC1E32540088BC76C7E034740');
INSERT INTO test_randompoints (id, geom) VALUES (94, '0101000020E6100000D307F74F57F425402396BC04C6094740');
INSERT INTO test_randompoints (id, geom) VALUES (95, '0101000020E610000005BE6552D5E02540890A6EF72D064740');
INSERT INTO test_randompoints (id, geom) VALUES (96, '0101000020E6100000F644F1FE18F125407C46F9B41B084740');
INSERT INTO test_randompoints (id, geom) VALUES (97, '0101000020E61000003C70E25579102640EF15737417094740');
INSERT INTO test_randompoints (id, geom) VALUES (98, '0101000020E6100000EC3E02985828264088E0AA6E7DFB4640');
INSERT INTO test_randompoints (id, geom) VALUES (99, '0101000020E6100000C9DC29572E182640F9EFF30B0DFE4640');
INSERT INTO test_randompoints (id, geom) VALUES (100, '0101000020E610000088934329231B26409459DAE6D5024740');


--
-- TOC entry 3358 (class 0 OID 117457)
-- Dependencies: 220
-- Data for Name: trajectories; Type: TABLE DATA; Schema: analysis; Owner: -
--

INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (1, 3, '2006-01-01 01:01:24+01', '2006-10-28 14:01:18+02', 'First test', 'postgres', 1426, 288928, '2014-01-07 04:24:25.665+01', 'SELECT * FROM main.view_locations_set WHERE acquisition_time > ''2006-01-01''::timestamp AND animals_id = 3', '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');
INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (2, 1, '2005-10-18 22:00:54+02', '2005-12-31 21:01:54+01', 'Second test', 'postgres', 332, 70043, '2014-01-07 04:24:31.643+01', 'SELECT animals_id, acquisition_time, geom FROM main.gps_data_animals WHERE gps_validity_code = 1 AND acquisition_time < ''2006-01-01''::timestamp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
INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (3, 2, '2005-03-20 17:03:14+01', '2005-12-31 21:00:55+01', 'Second test', 'postgres', 1614, 307836, '2014-01-07 04:24:31.643+01', 'SELECT animals_id, acquisition_time, geom FROM main.gps_data_animals WHERE gps_validity_code = 1 AND acquisition_time < ''2006-01-01''::timestamp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
INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (4, 3, '2005-10-23 22:00:53+02', '2005-12-31 21:01:41+01', 'Second test', 'postgres', 400, 73284, '2014-01-07 04:24:31.643+01', 'SELECT animals_id, acquisition_time, geom FROM main.gps_data_animals WHERE gps_validity_code = 1 AND acquisition_time < ''2006-01-01''::timestamp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
INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (5, 4, '2005-10-21 22:00:47+02', '2005-12-31 21:01:42+01', 'Second test', 'postgres', 424, 72499, '2014-01-07 04:24:31.643+01', 'SELECT animals_id, acquisition_time, geom FROM main.gps_data_animals WHERE gps_validity_code = 1 AND acquisition_time < ''2006-01-01''::timestamp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
INSERT INTO trajectories (trajectories_id, animals_id, start_time, end_time, description, ref_user, num_locations, length_2d, insert_timestamp, original_data_set, geom) VALUES (6, 6, '2005-04-04 08:01:41+02', '2005-05-06 01:01:47+02', 'Second test', 'postgres', 278, 40602, '2014-01-07 04:24:31.643+01', 'SELECT animals_id, acquisition_time, geom FROM main.gps_data_animals WHERE gps_validity_code = 1 AND acquisition_time < ''2006-01-01''::timestamp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


--
-- TOC entry 3434 (class 0 OID 0)
-- Dependencies: 219
-- Name: trajectories_trajectories_id_seq; Type: SEQUENCE SET; Schema: analysis; Owner: -
--

SELECT pg_catalog.setval('trajectories_trajectories_id_seq', 6, true);


SET search_path = env_data, pg_catalog;

--
-- TOC entry 3348 (class 0 OID 109283)
-- Dependencies: 208
-- Data for Name: adm_boundaries; Type: TABLE DATA; Schema: env_data; Owner: -
--

INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (1, 4, 22, 22192, 'Terlago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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (2, 4, 22, 22205, 'Trento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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (3, 4, 22, 22215, 'Vezzano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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (5, 4, 22, 22003, 'Aldeno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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (6, 4, 22, 22031, 'Calavino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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (4, 4, 22, 22222, 'Villa Lagarina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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (7, 4, 22, 22053, 'Cavedine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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (8, 4, 22, 22058, 'Cimone', '0106000020E6100000010000000103000000010000002201000036E0B2194F1B2640BFAF80C28EFF4640BBFFCB56801B2640F253817E8DFF4640F8F9336F0A1C2640C36048A892FF46407C469237931C264037C649A480FF4640DAE48613981C26407EC800107EFF4640C45A7067DB1C26403A1757CF74FF464071854D25281D26405B99789B75FF4640E6AEE0F9611D2640D86057DA79FF464040243E97BD1D26405FEE66127EFF4640B8CAA9AF2F1E2640417CA0AD80FF4640D4A295AE6D1E2640AA9BF62B7BFF4640C57DE573E31E26402A1125B670FF46409A3A729E851F264096D024C260FF4640F3A6D253EB1F26408D28D9A156FF464011740C13272026403E98D8AE50FF46409FB602EFDC20264008B6C87740FF4640D3D6FA2594212640C92868692EFF464030111647492226404FD0CB751DFF464073E693491D2326403A64334B09FF46407D25E0CB0E24264028013143F3FE46409C23405827242640F48E062DF0FE46406D284745412426400F4AFCFBEBFE46403B5DFC0CC5242640ABFBB55FD8FE46400D74DA9F4B2526401B782F45C4FE4640963554E8D92526405F1E620EAFFE464053C5B33555262640C5025AB79AFE4640340A22BB9C262640D41BF11691FE4640DBF980FAEC262640D8A4497E78FE4640A4B302D70A272640F9CC5D1E6AFE4640F17CA78D3B272640C084025556FE464067107A0B6D272640F728765D45FE4640A315B1E2C82726405E3DC7A21FFE4640D03306D4D6272640D31A33D71AFE4640C76FF4A3E7272640B802D52018FE464078EE910CF72726401FE0DD6A15FE464021B59C4315282640C1A812CC13FE46400BF07D31952826404D8B7F890EFE464056CD9266A22826407C440DF40DFE464014FA76F6F7282640CE4EF82B0AFE46404EB9AB8A03292640B646F5A809FE46406632F5F586292640C2DB4F6A03FE4640D053EEA0242A26404BE5BDCF01FE46408C01DC64952A2640FEE4C6A1EFFD46403D11DFCCE52A2640DFB74FA0E4FD4640667BFC4B312B2640F507093ED9FD46407D228DE85E2B26400E6865F8D1FD464074638BC1432B2640D748651AB9FD464034A960F1372B26404CED5AACACFD4640BCFC04FF242B2640B279C774A1FD4640C660D238102B26406D3BF5409CFD4640FB0D217FFB2A26400EF16B5096FD4640FE4094EBE52A26401E360A7692FD4640F6ACC728072B2640DFC4A7BD86FD46403BC78BE51F2B2640CAB8109E7BFD46406513B0F62F2B2640EC82918B73FD4640919622CB322B2640AA44915B69FD464099C59688302B2640504E9F5E61FD46402AAE39E0F52A264028F47FAE4FFD464017245D6DD22A264026D75FBA48FD4640053E0C4AA72A26401EF109E640FD4640184689EE772A26401A67F6EF38FD4640686DEB02532A2640548D3ED732FD464044677200562A2640426BFD6330FD46407A580A6E822A26404350F7DE2FFD46409A8BE925A92A2640E7F3D27130FD464039C74A8DBD2A2640BEE16EDB30FD464098C6EEE2D52A2640B6152A812BFD4640569DFE65E92A2640E8CDA78B24FD4640F5BBF4B2012B264064674AAF1FFD464020A16BEA312B2640344FB05D1BFD46400D7BC226642B2640F818056718FD4640281E0638A52B2640F208A8FA14FD46407449104DE62B2640D2A6534F11FD464012AC56172A2C26404DAD4A3E0FFD46404602FFAE322C264055E7EE4B0DFD46407F02C8565D2C2640042637DC07FD4640CCF47C04732C2640D3E2EDAA05FD4640D3665FC9B12C2640B395B151FFFC464073033C89C12C2640BA2B7ECAFBFC4640837ABA73D42C2640D649E03BF3FC46409B03B036EA2C264035526450EAFC4640CB061DFCF52C26408BEF3337E3FC4640F51AC0ABF82C264012B50DFEDAFC4640739F1B2AEB2C26404399C59CD2FC4640338D23F8E12C2640CEB05D03C9FC4640232003F4BE2C26407CE5BCFDC5FC4640A953444EA52C264091441BEBBFFC46403DF8F6CC8D2C2640878D5B7BB9FC464005A144E1872C264021D38473B3FC4640B24849BF792C2640A53CC2F6A9FC4640AD94E5AC542C26401458ED13A6FC4640898EBC36282C2640A70ECC16A7FC46403EA84842FA2B264019D91E53A9FC4640B45527B10D2C2640C5247E78A3FC46405C1096B0132C264009A552539EFC4640F727B67B042C26401C4B041E9AFC4640BE43C17EC42B264036B6F04298FC46404F39BDBD932B2640178DF23E9AFC464064CB07BC712B264046A506E09CFC4640D41417B5512B2640DCB48079A1FC46404DD4235C2C2B264055807E84A1FC464001EEADBFD92A2640E54A00F999FC46405F9D3A8A242A2640BB7283969AFC46400498BA2E6629264052D637F296FC4640B3FD3F2311292640966BD5159AFC46406DD22E48BE282640B097FB0AA0FC4640D8B709C68B282640457FF15BA5FC46403EBBBE445228264065BB6027ACFC4640DD6C32A33028264009DAF635B1FC46405AC6AC120A282640D06F54E7B5FC4640743D6C36F22726404E1EB221B8FC46406FB13717C12726402E0973D5BCFC4640779B62257827264046598BFDC4FC4640AF06BF29172726400A518A1EC9FC46400E264FFA17272640E977496EE6FC4640825FF3F50E2726402221FC5C04FD4640153830D2B926264039AE0F5004FD464002E09399A52626402216447B00FD46408D26D3D19C26264089216ED1FEFC4640BF9638B57B2626409A4BDD5FFDFC4640C429BA2619262640917BA67EFBFC4640758957411A26264000C90431D7FC4640E163CAF338262640C5EECDB8CDFC46405B48C9C545262640776DCCD7B9FC4640BA324A713A2626404C7FD046B7FC464069A9BE392D2626402962F0E889FC46408900AEB2412626406E8041F572FC46404F5533A08625264037260FCB42FC46409E302CBE6B25264001AD526B3DFC46404CD32DAA51252640A407089A3BFC4640319D6396242526409D6EB3CC39FC46405DB71E9B0C25264029ADA69C37FC46406AC41B1BF824264048B2861234FC4640E2A29F0AD62426405706696E29FC4640FCB327D974242640E5647FF40BFC4640A9DEBC512A242640338232DBF3FB4640471916DFA923264078E91CA7C9FB464025CCAE1151232640A11E496EABFB4640BC9512C32D232640B3CBEA8DA6FB46400318BF27F8222640EC34A872A1FB4640B0F6B050CA222640B2FB61319DFB4640649B85F5912226407362366B9AFB4640DC3ACEA23822264048EE071B9BFB4640A66D5E3CE121264057B92AEC94FB4640EE9D5EDD97212640D10646228DFB46404B0278E1432126405F1E442583FB46402037FE53FD20264024B8137A7BFB4640B99C3955D22026409545177278FB4640403CB053992026405359ED8277FB464023C8A0EF65202640568DCD1476FB4640736B501D4420264048C71D0674FB4640509F3EE027202640099E295E70FB4640DABD2FD4D51F26409AF44CC15FFB46407C74D4539F1F264010FEBA2A53FB46403955D198511F26403C2E751B46FB4640CC5BB283071F2640FCF8B0D33DFB4640973000FAA61E2640F9D090A637FB46401F167A5A3A1E264002A4DA932CFB464045E437E9D61D26405EC3FEF722FB4640ADA5F490881D2640DDED8DF918FB46404289C7DE251D2640D26D35D610FB464030AF599ACB1C2640F4216F3308FB46402EED7677751C2640428ED420FCFA464022DFCDDF391C2640B2DB38E2F2FA46403125908BCE1B26403ED8E9EDC3FA46402DCD9EC17F1B2640C49AFC27C2FA4640F3FCC4E3541B2640C8400B0FBFFA46408788F7792B1B2640A02DF91EBAFA464095D399C0071B264041F2229CB2FA46408F53E193E71A2640238567C0A9FA46409C68D3C7C81A26401D93623EA2FA46408187EE97A31A2640791739109DFA4640535C41DB251A264079EB8A4C92FA46409D51A8779619264088436EDA89FA464076DB2E0CE1182640036E859B82FA46403F0C6EA07E182640231025A17DFA464007BA19D4591826400492391286FA4640156787704C18264059314BCC86FA464087DB912A201826403C0A33D283FA4640435F8B7DF41726403BEC368285FA4640E1874B09C617264074842C9985FA46408F1AC5A582172640BBAD0DF17DFA4640E77F97A134172640B011FB3879FA464009261268E216264027836D0274FA4640E593910BC216264014E6685F73FA4640F38A97ECAC162640B27F649A73FA4640726F0EF89B16264028A0F66976FA46404F153AB89216264000395A947AFA464039E2E447851626408AAF89647DFA46409DDE14677B1626405442225E7EFA464041F4910703162640CFFD54437FFA464070936D59F61526407846B4BE7FFA4640740B0E3EE8152640347D57F680FA4640101272ABE415264092A11B2B83FA4640C978060CED1526406A0BC71E86FA464040E811DA17162640D0D24BAC8BFA4640E34F4676451626402B27C79891FA464075BC380378162640A668FDE896FA46405BCC4755C0162640D24B42539EFA4640250563521A172640876540AFA2FA4640917E501D65172640C59869F3B8FA4640797C59FE71172640A9607011CEFA46403AAADDE4D41726400BA417BEDCFA4640B9AB05F9471826402415BD0AECFA464033A98EA3B918264020FA41B5FBFA4640074766DCEE182640101A776707FB46401AAAEBAA30192640C90CEBE916FB4640391EDD1255192640EAF33D681FFB46406C0A2F6E701926400882569923FB4640578B8ABB84192640AD18D04628FB46404A2970DA8F192640A5EF238B2EFB46402FD6E51F911926404207766F34FB4640E82EB81882192640F49695FA3EFB46403A24BFC35D1926404CF15EDA51FB464075E7A0864D1926406DAF33BF5AFB4640C2DE34B3331926406A002AE568FB4640C561F56A14192640861A074076FB4640BA83185A0A192640FFDE7E727FFB4640973FDF1A03192640626BAB6688FB46400A88B327F218264042381B28A9FB4640B68E0991D818264059AECD20D2FB4640966ABE89C4182640EDA36F4BFEFB4640DC0C5741C4182640130A7DCEFFFB4640454AB4EABE182640A6D3213C1CFC464088423797B4182640522D8C7D44FC464055F141C2B1182640D1E121864FFC46409E14CF94671826402AF4CDA35AFC46403C87EEFEF2172640C74864B86CFC464065594BB87817264087B3D25F81FC4640CE43E84EE416264029A8027299FC464066D91D0D781626409039A770B0FC4640AA67ED2D191626405E6E00F4C6FC46400D9CEF18C715264091A43D52D8FC4640E457FF5A8215264029F8594BEBFC4640D3DF9D050A15264084C1E17306FD4640197365248414264028F2167928FD46405F8F5C580A142640364B747845FD4640C64296BBE4132640500421F651FD4640CBC74AFBC2132640506366BE5EFD4640ACC54AD6B5132640A8EFDAB863FD46400F78318A87132640CBC25EC779FD46403E5EAC6B74132640F790B76D7EFD46402A9AB8FE1913264059AF10C28AFD4640966A86F8AD1226408AF7B7F396FD46403E51CDD057122640E862FC51A1FD4640A38BE0167C12264037AE8A29C1FD4640AF83AC0BDC122640009A2D9003FE4640B9887A242C132640C0B34AA536FE46401F2D30832A132640B27EF7E649FE4640A989BAE479132640A898973F7CFE464045B59563BE132640F26BFE7AA1FE4640924A8781F7132640A507A9D0C1FE46408800EBC61414264096CC273AD2FE46405AC02EFB27142640DF8C392FDAFE4640DC64D8753A14264070100568E1FE46404987CA968C1426404B8B2599F4FE4640490B5BEE211526405A0D039915FF4640901243CC91152640211699A22FFF464090416E00F715264006C6316345FF46406E9B31E04916264000C383ED59FF4640CE11D8F6A016264003C5AF186EFF46408124324BC2162640D0176DC277FF46409755B170D7162640628FD0D878FF4640105B4ED5F2162640EDD33F9E76FF46407770A4690B1726401EA9A98374FF46402A08908E1B17264067057CA972FF4640C0F2C48E4A17264079053C7E6CFF46404FAB277B791726404967686163FF4640E6D38CC9A91726408827DF2959FF464093000C84D4172640D42A7B2851FF46407791F2F4E31726409EABDC6D4FFF4640B0D5493CF6172640D367E2AD4EFF4640253B4C6906182640A663346769FF4640D527EA0F4218264000B17BD67DFF4640E2CB227559182640A723DA5184FF4640BD4FFE917A1826409939268585FF4640BE73D1199818264056FB5D4584FF4640AC9A43C0A4182640DCA06B8683FF464080E7AA6CE61826400D1126D08BFF464083C4E127C3192640FC4314D49BFF4640D8D9ECF9CD1926401ECA65E0A5FF4640EF3B0E93DB19264037967B73AEFF4640300D4B083A1A2640FDCAC2C5B3FF4640FD5A7A67611A264048F45406B2FF46401B2CD327851A26408966BA5AACFF4640688E331EAF1A26402D3B85E5A1FF46407331ED0BD01A2640D609231B9CFF4640A77C142FE71A2640FD21E58798FF464043212551181B26404256465793FF464036E0B2194F1B2640BFAF80C28EFF4640');
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (9, 4, 22, 22079, 'Dro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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (10, 4, 22, 22091, 'Garniga Terme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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (11, 4, 22, 22101, 'Lasino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
INSERT INTO adm_boundaries (gid, cod_reg, cod_pro, pro_com, nome_com, geom) VALUES (12, 4, 22, 22132, 'Padergnone', '0106000020E610000001000000010300000001000000DE0000004237701762FC25403ACBFA5FD0084740540280667EFC2540243AC2F1C60847407D2EE3D59CFC25405791AFE6BC0847401742BD7BACFC254017098ABFB10847403F28F1C3C1FC25405CE5D4D7A6084740C2ACD93BF5FC2540A761F7A7A408474075ACC01416FD2540534EA5F4A2084740FB9098A52DFD25409CFD42BCA108474001CEFF5152FD25409E48094C9F084740F1B97DC966FD2540DAFCABF99C08474059C1FF8D7AFD2540C11A370A9A0847405CF6FD8EAFFD25402E262E0B8E08474081EDED69F3FD2540D56240E27D084740FEA8F6D932FE2540A151308F7C084740FB57612E5DFE2540AB46E4A6780847406B38DF9B95FE25404F2C6B42740847404B3BA849C1FE2540DBE2120375084740600F0F5DFAFE25406AE7684775084740D09532EB1CFF25403D05C915730847402301C67933FF254011841E7B72084740E2523F9C78FF2540EB3712FF720847400B34ADC5CCFF2540254AE5176E084740CDBEDC81160026403434E03954084740E209D644B4002640BC3D29006C08474003BAC7000B0126405935091E6208474016137BDD83012640BBC71B415408474038C7E9A40D0226409078808C46084740F6EDC9DF7B0226408CA86566390847404C7AAAC0E802264017D466632B08474041E65602AA022640957013340C08474070120E8DFF02264015552481EA0747407EEC2DDB240326404FEBA761E607474040C888822104264042972575CA0747404BE3F21B1F0426401AE0D550C40747400EBE83585B0426402D413386AB074740937C91C536052640D98F4E548B074740FC9AD8EB5805264085B05CFB860747401B60A5388A052640294FE5A17D074740A16AC173C80526403A3087DF7007474017C8C882160626409A02937B5F0747404B0FB1EC68062640EFC40BF341074740C37DC0619A0626400A5D19944007474040F9DB86CF0626409CBB2AE23C07474073B70EC0F80626408A869B2E380747403D434D20C50726405A1B39251B074740B7A9F81D57082640AAF92F36060747406D7DE14A290926401A30BD0AF106474007874AF6F709264047AADE7EDB064740CCC324E23F0A264072D2DBC6C60647403D7424917B0A26400EBA2EB7B506474021C69D2EE80A2640AF23A503AB06474041D793CE700B2640335E97249B0647405BBF2A4B760B2640DAA46C819A064740CDFB72BE810B264005F551B1990647402CA7A2557B0B264051F423BF95064740C3D5C6C8F90A2640CA1894BE5106474042C94994F20A264080F809E25206474018469215E90A2640F4D332BA540647407344D604CC0A26406EB6E06E6106474056ACD709BB0A26406159AF1F77064740363EDEA98B0A2640495C4FDB8D064740995A96ED6D0A26409A62F0119A06474008433B4A530A2640F652DB1DA206474011953771420A2640D42B2169AB064740CEF11856110A26406D76740FB20647406A6E5BB4E309264026A23374B906474052DB3B22D4092640B4770648BA064740740FE4B1160926402919E893B6064740C3245955BF082640BFD6CA28CA0647402E7C48B0360826403A76167BBD0647400C6285161C082640917C579ABA064740B7A6E796DD07264031B1B5CAB6064740A220E12FE0072640D98889ADAF064740A6DF38F9CF0726405A76A1E69A0647408966943BBD0726405BDE8AD281064740E48F9B5D71072640DB313BDF6806474062AC347A2E07264078692EC352064740189A516BA2062640CE774F39590647403CB995FFE1052640F83C586662064740AB611D343D0526401DC409AB69064740B68AAE48C60426407EFA914B7006474051C1B7E635042640F67E831878064740871B2FFBB20326400E3565F17E064740FB89614221032640FD0AB4A285064740E6D29C7F8D022640F62927388B064740637C274809022640A2282C5890064740CDA392196D012640EEA8B94B95064740D12B1C7CCF002640C832C79C9A06474057ED9A8AC00026404A9B7B8B9C0647405DEF0E8CBB00264001F5D494A7064740A977FBF74E0026409248D847A70647402A2C705B060026407CB53A14A7064740FC54C9C878FF25409D94B016A80647403D298A29DDFE25405A294EB6A70647404AE101DC4EFE25406EC58B76A70647404BE1D061B7FD25400F4588D1A7064740F7D212F145FD25408D24E073A8064740302E0AF8D7FC2540F82DB0CEAA064740BD20095770FC25407387B72AAD064740C50CF94B5BFC25406E5BE1BFC106474054182BD04FFC2540C29B56F9C806474028F65D8E48FC2540C83B6A14D0064740BF30D7114BFC2540E076FF61DA0647407C31F22C29FC2540F8F589EDDC0647400B78562487FB2540217B29C8DC06474054B1E0C7D2FA2540E1914ADDDC064740EFC4DC4534FA254099892699DC06474057EC058CC0F92540AC5ED605DD0647404959F872B5F92540DBBF41F9DC064740061BABF757F92540E45D4313ED064740339ACE2B71F92540E6361B5FF2064740719D598B6DF92540CCD23ECDF50647405B7FDDB552F9254077ADB1DCF7064740423E026DBEF82540DF092B020307474000C9C620AFF82540A9E3EA3FFC0647403DF96D50AFF825404ACB691EF6064740E99115A1A7F825405D2273E7F3064740C8632E2785F82540F916BC66F20647404AAA7F8D73F82540D3687B04F2064740B664720C4DF82540AD3A86E3E90647403E4C53BF22F825405856C2D4EA0647407F669445F9F72540AAF18ED9E706474036E605F8B9F725408C32AE5BE406474080A0396782F7254005BF3D5DE10647408224E02D4AF725403E63C6E5DC06474015DCA5E3A4F6254059E0A9EAD30647401F2F9F808AF6254096D72570DE0647402470D88A6FF62540E6BA7D8BE406474028F88BF75FF62540C8A9EC1AE7064740A30C9C4B48F625404AA0BFF0F4064740216F1C5542F62540F05CA19AFF0647402EFD636141F625404C3547B207074740CF029E072CF625408D8C15CA0E074740478D675AFCF52540423A49321E074740604AA8DFDCF525403A3777FC2D074740AF44AF23D6F525403D1221DB3A074740D5B37BEBD0F525406DF3D22B44074740F9B625F3B0F52540059012D94C07474091C5F11C73F52540F15051925E074740A2D9B32A44F52540C697A0FF6C074740A20525350EF525401DD706707A0747406A6FAB7614F52540BDF37A4786074740FCA6918815F5254055D6354D8807474010F3007EB3F42540F79DD8A48B0747401BED87BC9BF42540B9AB53778C0747405B1AA02F6AF42540BECBE0F48C074740679871D463F4254076549E0A8D074740B002646E4AF42540E02F121F8E074740A480B6F94DF42540C146E87B900747402F5B314578F425408F74D8089A074740E2657DB293F4254031D587B99D07474070A46427B6F425401EE23266A2074740BB319C45ECF4254081B7BFFCAB0747400BB8D42C0BF52540292E3274B00747405794635D1DF72540403D077CF207474018A313532CF7254058BCBA55F407474033E78D3E25F7254052ED51A9F6074740F1EDA6412CF7254059CEF81DF9074740D8B275CE3DF72540BC1822E8FD07474026EA6BAC55F725405617D88904084740464AD68866F725400ADCF5190808474075559F2873F72540FAE201660C08474095840FAC7DF7254033E7C55310084740914361378FF725408354197C150847404A73CB5E9FF725404A0FA289190847402BE0E71AAEF72540FA7AF5D51D084740764AD822BCF725406E264F80220847408B8EA6EBD1F7254016A612AE26084740E2BABA4FE6F725403EDBCDFF290847404B18B5F6F2F72540FC7134362C084740669AC599FFF72540416B1D872F0847402993F2DF05F8254044806E6F34084740247E5A430EF82540DB80065839084740621A9DD313F8254083BEC0BD3E0847405AC4626925F82540B8D7151441084740FD2950B037F825409271E92644084740803AE79048F825407C477C9C46084740EBAD0FAB5DF82540604ECB7049084740FA3741E674F825400B9FB32A4B0847402903AFAE99F8254069CC71CE4F084740B648888FB5F82540D1374D5557084740F754D87CECF825406F54ECCE59084740898ECB9508F92540367BFC70600847401F2709AE24F92540C7B69B3267084740AC530EDD3BF925406761D07A6C08474040EC296960F925406715007C730847401ECCA01F87F92540B9222EB076084740574FEFCFADF925402B99F99B7B084740D39B762DCEF92540B5B2A42D7F08474066CFA4AFDDF925407E5276F07E0847401DB27F6BF6F92540E1D3CD8D7908474035D509DB0FFA25407F9A116A74084740B238E92538FA25402D2E8DDD6A08474021EA827143FA25401F2E11666908474028FB28EE52FA25400A3C8FA16A0847404093EAE15EFA254046198BB86C084740298FA3FE67FA254029F06B28700847401EF691D071FA254095EE0979730847404950AC0D7DFA2540B13E394C76084740BB3010D98BFA25408DA1728C7608474039B2D449A0FA2540DDB2633076084740D6A37DDABDFA2540A79FDC8778084740103B2DF4D2FA2540EE9F1D5C7B0847404310D61CE3FA25402C322AEC7E08474035F2DDFAF3FA25403E4C357C820847404E1D6DF406FB254042D6EB2B86084740357E734401FB25400974E5178A0847404F1C0D9BFFFA2540076AB8958C0847400E995CACFDFA2540F655D67C8F0847405A7A883722FB2540CC79BABC960847403C2D051C41FB2540BE76A0749F084740CD441A625FFB2540BC38708DA108474010534CDA7AFB25404FB8E86BA2084740CD410D98A1FB25405585E7A9A30847407DA687A4BBFB2540FF446083A5084740A9D1FCDAC6FB2540F913692DAA084740D5C9690BD9FB2540ADA4CE5CB408474054762FA3ECFB25407E9385EABE0847404210D1FCFBFB2540D9313192CA08474002F49AA914FC254064A816B9C9084740370BDE4853FC2540FE5185FBD00847404237701762FC25403ACBFA5FD0084740');


--
-- TOC entry 3435 (class 0 OID 0)
-- Dependencies: 207
-- Name: adm_boundaries_gid_seq; Type: SEQUENCE SET; Schema: env_data; Owner: -
--

SELECT pg_catalog.setval('adm_boundaries_gid_seq', 12, true);


--
-- TOC entry 3352 (class 0 OID 109323)
-- Dependencies: 212
-- Data for Name: corine_land_cover; Type: TABLE DATA; Schema: env_data; Owner: -
--

INSERT INTO corine_land_cover (rid, rast) VALUES (1, '0100000100000000000000594000000000000059C00000000010C15041000000000088434100000000000000000000000000000000DB0B00001400140044001F20201A1A1A1A1A1A1A1A1A18181818181818191F201A1A1A1A1A1A1A1A1A1A1A181818181818192020201A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A1A2020201A1A1A1A1A1A1A1A1A191919191919191A2020201A1A1A1A1A1A1A1A19191919191919191A20201A1A1A1A1A1A1A191919191919191919191A1A1A1A1A1A1A1A1A1A19191919191919191A191A1A1A1A1A1A1A1A1A1A1D191919191919191A1A1A1A1A1A1A1A1A1A1A1D1D1D1D19191919191A1A1A1A1A1A1A1A1A1A1A1D1D1D1D1D1D1919191919191A1A1A1A1A1A1A1A1A1A1D1D1D1D1D1D191919191A1A1A1D1D1A1A1A1A1A1A1D1D1D1D1D1D1919191D1D1D1D1D1A1A1A1A1A1D1D1D1D1D171D1919191D1D1D1D1D1D1D1D1A1D1D1D1D1D1717171919191D1D1D1D1D1D1D1D1D1D181D1D1D1717171719171D1D1D1D1D1D1D1D181818181D1D1717171717171D1D1D1D1D1D1D1D181818181D181D1D171717171D1D1D1D1D1D1D1D1818181818181D171718171720201D1D1D1D1D1D1818181818181D1D181818182020202020201D1D1818181818181D1D1D181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (2, '0100000100000000000000594000000000000059C00000000004C35041000000000088434100000000000000000000000000000000DB0B0000140014004400191919191919191919191919191919191919191819191919191919191919191919191919191919121919191919191919191919191919191919121212191919191919191919191219191919191202121219191919191919121912121212121212020212121A191919191919191212121212121212020202121A191919191919191212120202020202020202121A1A1919191919191912121202020202020212181A1A19191919191919121212120202020202121219191919191919191919121202020202121212121919191919191919191912121202020212122929191919191919191919191212121212292929292919191919191919191919121212122929292929291919191919171719191912122929292929292918171717191717171719192929292929292929291817171717171717171717292929292929292929181717171717171717172929292929292929292918171717171717172929292929292929292929291817171717172929292929292929292929292929191717171729292929292929292929292929291919');
INSERT INTO corine_land_cover (rid, rast) VALUES (3, '0100000100000000000000594000000000000059C000000000F8C45041000000000088434100000000000000000000000000000000DB0B00001400140044001218181212121212121818181818181818181818121212121212121218181818181818181818181812121212121212181818181818181818181818181212121218181818181818181818181818181818121218181818181818181818181818181818181812181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818191818181818181818181818181818181818181919191818181818181818181818181818181818191919181818181818181818181818181818181B');
INSERT INTO corine_land_cover (rid, rast) VALUES (4, '0100000100000000000000594000000000000059C000000000ECC65041000000000088434100000000000000000000000000000000DB0B000014001400440018181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818121818181818181818181818181818181818181812181818181818181818181818181818181818181B1818181818181818181818181818181818181B1B18181818181818181818181818181818181B1B1B1818181818181818181818181818181B1B1B1B1B18181818181818181818181818181B1B1B1B1B1B181818181818181818181818181B1B1B1B1B1B1B181818181818181818181818181B1B1B1B1B1B1B181818181818181818181818181B1B1B1B1B1B1B1818181818181B1B1B1818181B1B1B1B1B1B1B1B18181818181B1B1B1B1B1B1B1B1B1B1B1B1B1B1B18181B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B');
INSERT INTO corine_land_cover (rid, rast) VALUES (5, '0100000100000000000000594000000000000059C000000000E0C85041000000000088434100000000000000000000000000000000DB0B0000140014004400181818181818181B1B1B1B1B1B1B1B1B1B1B1B1B1818181818181B1B1B1B1B1B1B1B1B1B1B1B1B1B18181B1818181B1B1B1B1B1B1B1B1B1B1B1B1B1B181B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B121B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1B1B1B12121B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1F1B1B12121B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1F1F1F121B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1F1F1F121B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1F1F1B121B1B1B1B1B1B1B1B1B1B1B1B1B1B1F1F1B1F1B121212121B1B1B1B1B1B1B1B1B1B1B1F1F1B1B1B12121B1B1B1B1B1B1B1B1B1B1B1B1F1F1B1B1B1B12121B1B1B1B1B1B1B1B1B1B1B1F1F1F1B1B1B1B12121B1B1B121B1B1B1B1B1F1F1F1F1B1B1B1B1B1212121B1B12121B1B1F1F1F1B1B1B1B1B1B1B1B121B121B1B1212121B1B1F1F1B1B1B1B1B1B1B1B1B1B121212121212121B1F1F1B1B1B1B1B1B1B171B1B1B1212121B1B1B1B1F1F1B1B1B1B171717171B1B1B121B1B1B1B1B1F1F1B1B1B1B1B17171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (6, '0100000100000000000000594000000000000059C000000000D4CA5041000000000088434100000000000000000000000000000000DB0B00001400140044001D1D1F1F17171717171717171717171D1D1010101B1F1F1F1F1F1717171717171717171D1D1010101B1B1F1F1F1F1F171717171717171D1D1D1D10101B1B1F1F1F1F171717171717171D1D1D1D1D10101B1F1F1F1F1F1F1F17171717171D1D1D1D1D10101B1F1F1F1F1F1F1717171717171D1D1D1D1D10101F1F1F1F1F1F1F17171717171D1D171D1D1D10101F1F1F1F1F1F1F17171717171717171D1D1D10101B1B1B1F1F1F17171717171717171D1D1D1010101B1B1B1F1F1F1F1F1717171717171D1D1D1010101B1B1F1F17171F1F1717171717171D1D101010101B1B1F17171717171717171717171D1D101010101B1B1F17171717171717171717171717171010101B1F1717171717171717171717171717171710101B1F1F1717171717171217171717171717171D1D1B1F1F17171717171212171717171717171D1D1D1F1F1717171717121212121717171717171D1D1D1F171717171717121212121717171717171D1D1D1F17171717171717121212171719171717171D1D1717171717171717121217171919191917191D1D');
INSERT INTO corine_land_cover (rid, rast) VALUES (7, '0100000100000000000000594000000000000059C000000000C8CC5041000000000088434100000000000000000000000000000000DB0B0000140014004400101010101010101010101010030303030303030310101010101010101010101003030303030303031010101010101010101010100303030303030F0F1010101010101010101010100303030303030F0F101010101010101010101003030303030F0F0F0F101010101010101010100303030F0F0F0F0F0F0F1010101010101010100303030F0F0F0F0F0F0F0F10101010101010101003030F0F0F0F0F0F0F0F0F101010101010101010030F0F0F0F0F0F0F0F0F0F101010101010101010020F0F0F0F0F0F0F0F0F0F1010101010101010100202020F0F0F0F0F0F0F0F1010101010101010100202020F0F0F0F0F020202101010101010101010020202020202020202020210101010101010100202020202020202020202021D1D1D10101010100202020202020202020202021D1D1D101010101002020202020202020202020F1D1D1D1D101010100202020202020202020215151D1D1D1D100202020202020202020215151515151D1D1D1D1D0202020202020202151515151515151D1D1D1D1D020202150202021515151515151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (8, '0100000100000000000000594000000000000059C000000000BCCE5041000000000088434100000000000000000000000000000000DB0B000014001400440003141414141414141402191919171715191919191414141414141414020202191919191519191919021414141414141402020219191915151919191902021414140214020202021919191915191919190F020214140202020202021919191915191919190F0F0202020202020202020215191519191919190F0F0F0F020202020202020215151919191919150F0F0F0F020F02020202020215151919191915150F0F0F0F0F0F02020202020202151502191915150F0F020F0F020202020202020202020219191515020F0F0202020202020202020202021919191515020F0F0202020202020202020202021915191519020F0F0F0F0F0202020202020202151515151919020F0F0F0F0F0202020202020215151515151919020F0F0F0F1502020202021419151515151515150F0F1515151414140202141419151515151915151515151514141414141414141915151919191919151514141414141403030319191515191919191915141414141414140303031915151519191919191514141414140303030319191515151515191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (9, '0100000100000000000000594000000000000059C000000000B0D05041000000000088434100000000000000000000000000000000DB0B000014001400440019191919070707071919190707190707071919191919190707071519151519191919191919191919191919151515151915191919191919191919191919191515151515151919191919191919191919191515151515151919191919191919191919191919151515151519191919191919191919151919191915151515151915151919191919191515181819191515151519191519191919191515181815181819151515191919151919191515151515181818181915151919151915191919151515151515181818191519191915151502021919151515151518181819191915021515150215151515151515151818181815150202151515021515151518151515181818181515021515151502151515151815151518181818151502151515020202151515181515151515151815150215150202020215151518151518151518181515020215150202021515151818181818151518190202021515020202151515181818181818151519020202020214141515151818181818181818181402021414141414151515151818181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (10, '0100000100000000000000594000000000000059C00000000010C15041000000001884434100000000000000000000000000000000DB0B00001400140044002020202020201D18181818181818181D18181818202020202020201818181818181818181818181820202020202020201818181D181818181818181820202020202020202018181D1D1D1D1D18181818202020202020202020201D1D1D181D1D18181818202020202020202020201D1D1D181D1D18181818202020201D1D1D1D1D1D1D1D18181D1D181818182020201D1D1D1D1D1D1D1D1D18181D1D181818182020201D1D201D1D1D1D1D1D18181D18181818182020202020201D1D1D1D1D1818181D1818181718202020202020201D1D1D1D1D1818181818171718202020202020201D1D1D1D1D18181818171717181A202020202020202020181818181818171717171A1A1A20202020202018201818181818171717171A1A1A1A1A2020202018181818181818171717171A1A1A1A1A1A20202018181818181818171717171A1A1A1A1A1A20202018181818181817171717171A1A1A1A1A1A20202020181818181817171717171A1A1A1A1A1A20202020181818181717171717171A1A1A1A1A1A1A20201717181818171717171729');
INSERT INTO corine_land_cover (rid, rast) VALUES (11, '0100000100000000000000594000000000000059C00000000004C35041000000001884434100000000000000000000000000000000DB0B000014001400440017171819191919191929292929292929292919191818181919191919191929292929292929291919181818191919191919192929292929292929191918181717191919191919192929292929292919191817171717171919191929292929292929291919181817171719191929292929292929292919191918171717171717192929292929292929291919191817171717171729292929292929292929191919181717171717292929292929292929292919191918171717172929292929292929292929291919191717171717292929292929292929292919191919171717171729292929292929292929291919191917171717292929292929292929292919191919191717172929292929292929292929291919191919172929292929292929292929292919191919191917292929292929292929292929291919191919192929292929292929292929292919191919191919292929292929292929292929191919191919191929292929292929292929291919191919191919192929292929292929291919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (12, '0100000100000000000000594000000000000059C000000000F8C45041000000001884434100000000000000000000000000000000DB0B0000140014004400191919191818181818181818181818181818181B1919191918181818181818181818181818181B1B19191818181818181818181818181818181B1B1B19191818181818181818181818181818181B1B1B191919181818181818181818181818181B1B1B1B1818181819191918181818181818181B1B1B1B1B18181819191919191818181818181B1B1B1B1B1B19191919191919191818181818181B1B1B1B1B1B1919191919191919181818181B1B1B1B1B1B1B1B19191919191919191818181B1B1B1B1B1B1B1B1B19191919191919191818181B1B1B1B1B1B1A1B1B191919191919191919191B1B1B1B1B1A1A1A1B1B191919191919191919191B1B1B1B1A1A1A1A1A1B191919191919191919191B1B1B1B1A1A1A1A1B1B191919191919191919191B1B1B1B1B1B1B1A1B1B19191919191919191B1B1B1B1B1B1B1B1B1B1A1B1919191919191919191B1B1B1B1D1D1D1B1B1A1B19191919191919181B1B1B1B1B1D1D1D1D1B1A1A19191919191918181B1D1B1B1B1B1D1D1D1D1B1A191919191918181B1B1D1D1D1B1B1B1B1D1D1D1A');
INSERT INTO corine_land_cover (rid, rast) VALUES (13, '0100000100000000000000594000000000000059C000000000ECC65041000000001884434100000000000000000000000000000000DB0B00001400140044001B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B201B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B20201B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B20201B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B20201B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B201B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B201B1B1B1B1B1B1B1B1B1B1B1B1B1B2020202020201B1B1B1B1B1B1A1A1A1A1B1B1B202020202020201B1B1B1B1B1B1A1A1A1A1A1A1A201A2020202020201B1B1B1B1B1B1A1A1A1A1A1A1A1A2020202020201B1B1B1B1B1B1B1B1A1A1A1A1A1A2020202020201B1B1B1B1B1B1B1B1A1A1A1A1A1A2020202020201B1B1B1A1A1B1B1A1A1A1A1A1A1A2020202020201B1B1A1A1A1A1A1A1A1A1A1A1A1A2020202020191B1A1A1A1A1A1A1A1A20201A20202020202020191B1A1A1A1A1A1A1A1A20202020202020202019191A1A1A1A1A1A1A1A1A20202020202020201919191A1A1A1A1A1A1A1A1A202020202020201919191D1A1A1A1A1A1A1A1A20202020202020191919191D1A1A1A1A1A1A1A2020202020202020191919191D');
INSERT INTO corine_land_cover (rid, rast) VALUES (14, '0100000100000000000000594000000000000059C000000000E0C85041000000001884434100000000000000000000000000000000DB0B00001400140044001B1B1B1B1B1B1D1D1B1F1F1B1B171B1B171717171B1B1B1D1D1D1D1D1F1F1B1B1717171B171717171B1B1B1D1D1D1D171D1F1B1717171717171717171B1B1D1D1D1D1D171717171717171717171717171B1D1D17171D17171717171717171717171717171D1D1D17171717171717171717171717171717171B1D1D17171717171717171717171717171717171B1D1D19191919171717171717171717171717121B1B1919191919191717171717171717171712121919191919191919191717171717171717121212191919191919191919171717171717171712121219191919191919191919171717171717121212121919191919191919191919171717171212121212191919191919191919191917171717121212121219191919191919191919191717171212121212121D191919191919191919171717121212121212151D1D1919191717171717171717121215121212121D1D1917171717171717171712121215151515151D1D1D17171717171717121212121215151515151D1D1D1717171717171712121212121515151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (15, '0100000100000000000000594000000000000059C000000000D4CA5041000000001884434100000000000000000000000000000000DB0B0000140014004400171717171717121712171717191919191919191D171717171717121212121719191919191919191D171717171717121217121219191919191919191D171717171712121217121219191919191919191D1717171712121217171719191919191919191919171712121212171717171819191919191919191912121212121217171718181819191919191919191212121217171717171818191919191919191919121212121717171718181919191919191919191912121212171717181819191919191919191919191212121217171818181919191919191919191919121212121717181818191919191919191919191912151717171718181819191919191919191919191515171717171818191919191919191919191919151517171517181919191919191919191919171715151715181819191919191919191919191717171515151818181819191919191919191917171717151518181818181919191919191919171717171715181818181818181819191919191917171717171518181818181818191919191719171717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (16, '0100000100000000000000594000000000000059C000000000C8CC5041000000001884434100000000000000000000000000000000DB0B00001400140044001D1D1D1D1D1D02021502151515151515151514141D1D1D1D1D1D02021515151515151515151414141D1D1D1D1D1D02021515151515151514141414141D1D1D1D1D1D15151515151515151514141414141D1D1D1D1D1D1D151515151515030303141414141D1D1D1D1D1D1D151515151503030303141414141D1D1D1D1D1D1D15151515150303141414030303191D1D1D1D1D1D1D151515030303031403030303191D1D1D1D1D1D1D151515150303030303030303191D1D1D1D1D1D1D1D151515030303030303030319191D1D1D1D1D03030303150303030303030303191D1D1D1D1D1D03030303151503030303030303191D1D1D1D1D1D1D030303031503030303030303171D171D1D1D1D1D03030303151503030303030317171D1D1D1D1D1D030303030315030303030303171D1D1D1D1D1D1D030303030303030303030303171D1D1D1D1D1D1D030303030303030303030303171D1D1D1D1D1D1D030303030303030303031010171D1D1D1D1D1D1D0303030303030303031010101D1D1D1D1D1D1D1D030303030303030303101010');
INSERT INTO corine_land_cover (rid, rast) VALUES (17, '0100000100000000000000594000000000000059C000000000BCCE5041000000001884434100000000000000000000000000000000DB0B000014001400440014141414030303030303191915020202021919191414141414030303030319151502020202191919141414141403030303191915150202021919191914141414030303030319191502020219191919191414140303030303191919020202021919191914030314140303030319191919020202191919191903030314030303031919191515020202021515190303031403030303151519151502020202151515030303030303030315151915151502021515151903030303030303031519191919151502151515190303030303030303151919191515151515151519030303030303031515151919151515151919191903030202020215151515191515151515191919190303020202021515151515151515151519191919030202020202151515151515151515151519141403101010020215151515151515151515151519141010101002020215151515151515151515191914101010100202021515151515151515151919191910101010020202021515151515151515191919191010101002020202021515151515151519191818');
INSERT INTO corine_land_cover (rid, rast) VALUES (18, '0100000100000000000000594000000000000059C000000000B0D05041000000001884434100000000000000000000000000000000DB0B000014001400440019140202141414141515151818181818181818181919021414141414151518181818181818181818191402141414141415181818181818181818181814140202141414141518181818181818181818181414141402021415151818181818181818181818141414140202020214151818181818181818181814141414141414141415181818181818181818181414141414141414181818181818181818181818141414141414141818181818181818181818181814141414141414181814141418151818181818181414141414141414141414141818151818181818191914141414141414141414141815151515181819191914141414181818181818181815151518181919191414141418181818181818181515181818141414141414181818181818181818151515151814141414181818181818181818181515151515181414141418181818181818181818151515151818191914181818181818181818181818181518181819181818181818181815181515151515181818181818181818181818151515151515151518181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (19, '0100000100000000000000594000000000000059C00000000010C15041000000003080434100000000000000000000000000000000DB0B00001400140044001A1A1A1A1A1A1A202017171717171717171717291A1A1A1A1A1A1A201717171717171717171717171A1A1A1A1A1A20201717171717171717171717171A1A1A1A1A2020171717171717171717171717171A1A1A1A1A2020171717171717171717171712121A1A1A1A1A2020171717171717171717171712121A1A1A1A1A1D1D171717171717171712171712171A1A1A1A1A1D1D171717171717171712121212191A1A1A1A1A1D1D171717171717171712121212191A1A1A1A1A1D1D1D1717171717171212121219191A1A1A1A1D1D1D1D1D17171717171212121219191A1A1A1A1D1D1D1D1D1D171717171712121919191A1A1A1D1D1D1D1D1D1D17171D1D1712191919191A1A1A1D1D1D1D1D1D1D17171D1D1212191919191A1A1A1A1D191919191D171D1D1D1212191919191A1A1A1A191919191D1D1D1D1D1D1D12191919191A1A1A1A1919191D1D1D1D1D1D1D1212191919191A1A191919191D1D1D1D1D1D1D121219191919191919191919191D1D1D1D1D1D1D121919191919191919191919191D1D1D1D191D1D12191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (20, '0100000100000000000000594000000000000059C00000000004C35041000000003080434100000000000000000000000000000000DB0B00001400140044002929292929291717171919191919191919191919292929292917171717191919191919191919191929292929171717171919191919191919191918182929291717171719191919191919191919191818121717171717171919191919191919181818181817171717171719191919191918181818181818181717171717191919191919181818181818181818171717171719191919191918181818181818181817171717191919191919191818181818181818181919191919191919191919181818181818181818191919191919191919191918181818181818181819191919191919191919191818181818181818181919191919191919191919181818181818181818191919191919191919191919181818181818181D191919191919191919191918181818181818181D1919191919191919191918181818181818181A1A1919191919191919191918181818181818181A1A19191919191919191918181818181818181A1A1A191919191919191919181818181818181A1A1A1A191919191919191918181818181818181A1A1A1A');
INSERT INTO corine_land_cover (rid, rast) VALUES (21, '0100000100000000000000594000000000000059C000000000F8C45041000000003080434100000000000000000000000000000000DB0B0000140014004400191919191818181B1B1D1D1D1D1D1B1B1B1D1D1A191919191818181B1B1D1D1D1D1D1D1D1D1D1D1A1819191818181B1B1B1D1D1D1D1D1D1D1D1D1D1A18181818181B1B1B1B1D1D1D1D1D1D1D1D1D1A1A181818181B1B1B1B1B1D1D1D1D1D1D1D1A1D1A1A181818181B1B1B1B1D1D1D1D1D1D1D1D1A1A1A1A1818181B1B1B1D1D1D1D1D1D1D1D1D1D1D1D1A1A1818181B1B1B1B1D1D1D1D1D1D1D1D1D1D1D1A1A18181B1B1B1B1B1B1D1D1D1D1D1D1D1D1D19191918181B1B1D1B1B1D1D1D1D1D1D1D1D1D19191919181B1B1B1D1D1D1D1D1D1D1D1D1D1D1D19191919181D1B1B1B1D1D1D1D1D1D1D1D1D1D1D191919191D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D191919191D1D1D1D1A1A1A1A1A1A1A1A1D1D1D19191917171D1D1D1D1A1A1A1A1A1A1A1D1A1A1A19171717171A1D1D1A1A1A1A1A1A1A1A1A1A1A1A17171717191A1A1A1A1A1A1A1A1A1A1A1A1A1A1717171717191A1A1A1A1A1A1A1A1A1A1A1A1A1A1A17171717191A1A1A1A1A1A1A1A1A1A1A1A17171717171717191A1A1A1A1A1A1A1A1A1A1A1A1717171717171919');
INSERT INTO corine_land_cover (rid, rast) VALUES (22, '0100000100000000000000594000000000000059C000000000ECC65041000000003080434100000000000000000000000000000000DB0B00001400140044001A1A1A1A1A1A202020202020202019191919191D1A1A1A1A1A1A20202020202020191919191D191D1A1A1A1A1A20202020201919191919191D1D1D1D1A1A1A1A1A1A2020202019191919191D1D1D1D1D1A1A1A1A1A1A202020191919191919191D1D1D1D1A1A1A1A1A1A202019191919191919191D1D1D1D1A1A1A1A1A20202019191919191919191D1D1D1D1A1A1A1A1A19191919191919191919191D1D1D19191919191919191919191919191919191D19191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919141419191919191919191919191919191919191914141919191919191919191919191919191919141414191919191919191919191919191919191914141419191919191919191919191919191919191414141919191919191919191919191919191915141414');
INSERT INTO corine_land_cover (rid, rast) VALUES (23, '0100000100000000000000594000000000000059C000000000E0C85041000000003080434100000000000000000000000000000000DB0B00001400140044001D1D1D17171717171212121212121215151515151D1D1D19191717171212121212121212151515151D1D1D19191917171712121212121215151515151D1D1D19191717171712121212121215151515151D1D191919121212121219121212121512121515191919191212121212191912121212121212191819191912121219191919191919191919191919181919191212191919191919191919191919191919191919191919191919191919191919191919191919141919191919191919191919191919191919191914191914191919191919191919191919191919141419191419191919141419191919191919141414191914191919191914141414141414141414141419191414191419141414141414140202021414141414141414141414141818181414020202020214141414141414141818181818181414140202021414141414181818181818181818141414140202141414141818181818181818181818141414020214141418181818181818181818191817020202021414181818181818181818181919191702020214');
INSERT INTO corine_land_cover (rid, rast) VALUES (24, '0100000100000000000000594000000000000059C000000000D4CA5041000000003080434100000000000000000000000000000000DB0B0000140014004400151818181818181819171717171717171D1D1D1D15181818181818191917171717171719191D1D1D1818181818181819171717171D17171919191D1D18181818181818171717171D1D1919191919191718181818181717171D1D1D1D1D1D1D1919191717181818181817171D1D1D1D1D1D1D1D191919171718181818171717171D1D1D1D1D1D1D1919191717181818191917171D1D1D1D1D1D1D191919191717191919191917171D1D1D1D1D1D191919191917171919191919191D1D1D1D1D1D191919191919171719191919191D1D1D1D1D1D191919191919191717191919191D2929291D1D19191D1919191919171714191914141D29291D1D1D19191919191919171714141414141D29291D1D1D19191919191917171702141414141D2929291D1919191919191917171702021414141429291D1D1919191919191717171702021414141414291D1919191919191717171717020214141414142919191919191717171717171702021414141414191919191919171717171717171414141414142919191919191717171717171714');
INSERT INTO corine_land_cover (rid, rast) VALUES (25, '0100000100000000000000594000000000000059C000000000C8CC5041000000003080434100000000000000000000000000000000DB0B00001400140044001D1D1D17171D1D1D0303030303030303030310101D1D1717171D1D1D0303030303030303030310101D171717171D1D1D03030303030303030303031017171717171D1D1D1D03030303030303030303031717171717171D1D1D031D030303030303030303171717171717191D1D1D1D1D0303030303030303171717171719191D191D1D1703030303030303031717171719191919191919170303030303030303171717171919191919191917171010100303030317171719191919191919191717101010030303031717171919191919191919171710101010031010171719191919191919191917171010101010101017171919191919191919191717171010101010101719191919191919191919191717101010101010171719191919191919191919171710101010101017171919191919191919191917171010101010101717191919191919191919191717101010101010171919191919191919191919171710101010101014191919191919191919191917171010101010101419191919191919191919191717101010101010');
INSERT INTO corine_land_cover (rid, rast) VALUES (26, '0100000100000000000000594000000000000059C000000000BCCE5041000000003080434100000000000000000000000000000000DB0B000014001400440010101010020202020202151515151515191919181010101010020202020215151515151919191918101010100202020202020215151519191919191903101010020202020202020202191919191919190303030302020202020202020219191919191919030303030202020202020202021919191919191903030303020202020202020202021919191919190303030202021402020202020202021919191919030202141414140202020202020202191919191902020202021414020202020202020202191919190202020202141402020202020202020219191919020202021414020202020202020202021919191902020214141402020202020202020202021919191002020202141402020202020202020202191919100202021414140202020202020202020202191510101402020214141402020202020202020219151010101402021414140202020202020202151915101010141414141402020202020202020215151510101414141414020202020202020202020215151010101403020203030202020202020202021515');
INSERT INTO corine_land_cover (rid, rast) VALUES (27, '0100000100000000000000594000000000000059C000000000B0D05041000000003080434100000000000000000000000000000000DB0B00001400140044001818181818181815151515151515151515151518181818181818181515151515151515151515151518181818181818181818181515151515151515181818181818181818181818151515151515151518181818181818181815151515151518151518181518181818181818181818181818181818151818151818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181818181918181818181818181818181818181818181818191818181818181818181818181818181818181D191818181818181818181818181818181818181D191918181818181818181818181818181818181D1919191919191818181818181818181818181D1D1919191919191919191818181818181818181D1D151919191919191919191818181819191D1D1D191515191919191919191919181818191919191D1D151515150219191919191919191919191919191D151515150202141414181818181918181818181D1515151502020214141418181819191818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (28, '0100000100000000000000594000000000000059C00000000010C1504100000000487C434100000000000000000000000000000000DB0B000014001400440019191919191D1D1D1D1D191D1212191919191919191919191D1D1D1D1D1D1912121919191919191919191D1D1D1D1D1D19191919121219191919191919191D1D1D1D1D1D19191919121919191919191919191D1D1D1D1D1D1919191919191919191919191D1D1D1D1D1D191D1919191919191919191919191D1D1D1D1D1D191D1D19191919191919191919191D1D1D1D1D1919191919191919191919191919191D1D1D1D1D1919191919191919191919191919191D1D1D1D1D191919191919191919191919191919191D1D1D19191919191912121219191919191919191D1D191717191919191212121919191919191919191917171719191912121219191919191919191919191717171919121212121919191919191919181817171717191912121212191919191919191918171717171919121212121919191919191919191817171717191912121219191919191919191919171717171719191212191919191919191919191917171717191912121219191919191919191919191717171719191919191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (29, '0100000100000000000000594000000000000059C00000000004C3504100000000487C434100000000000000000000000000000000DB0B00001400140044001919191919191918181818181818181A1A1A1A1A19191919191919191818181818181A1A1A1A1A1A19191919191919191818181818181A1A1A1A1A1A19191919191919191818181818181A1A1A1A1A1A19191919191919191818181818181A1A1A1A1A1A19191919191919191818181818181A1A1A1A1A1A19191919191919181818181818181A1A1A1A1A1A191919191919191818181818181A1A1A1A1A1A1A191919191919191819181818181A1A1A1A1A1A1A1919191919191919191918181A1A1A1A1A1A1A1A19191919191919191918181A1A1A1A1A1A1A1A1A19191919191919191818181A1A1A1A1A1A1A17171919191919191918181A1A1A1A1A1A1A171717171919191919191918181A1A1A1A1A1A1A171717171919191919191818191A1A1A1A1A1717171717171919191919191819191A1A1A1A1A1717171717171919191919191919191A1A1A1A171717171717171919191919191919191A1A1A1A1917171717171719191919191919191919191919191917171717171919191919191919191919191919191717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (30, '0100000100000000000000594000000000000059C000000000F8C4504100000000487C434100000000000000000000000000000000DB0B00001400140044001A1A1A1A1A1A1A1A1A17171717171717171719191A1A1A1A1A1A1A1A1717171717171717171719191A1A1A1A1A1A1A171717171717171717171919191A1A1A1A1A1A17171717171717171717191919191A1A1A1A1A1717171717171717171919191919191A1A1A1A1A1717171717171717171919191919191A1A1A1A171717171717171717191919191915151A1A1A17171717171717171717191919191515151A1A1717171717171717171719191919151515191A1717171717171717171717191919191515191917171717171717171717171D1D1D191519151919171717171719191917171D1D1D15151515151515171717171719191919191D1D1515151515151515171717171919191919191D1D1515151515151515171717191919191919191D1D1515151515151515171719191919191919191D1D1515151502150202171719191919191919191D1D15151515020202021717191919191919191D1D1D15151515151502021717191919191919191D1D151515151515151515171719191919191D1D1D1D151515151515151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (31, '0100000100000000000000594000000000000059C000000000ECC6504100000000487C434100000000000000000000000000000000DB0B000014001400440019191919191919191919191919191915151419191919191919191919191919191919151515151919191919191919191919191919191915151519191919191919191919191919191919191515191919191919191919191919191919191915151919191919151919191919191515151919151515191919191915191919191915151515151515151919191919191919191919191515151519191515191919191919191919191515151515191919151519191919191919191919151515151519191915191919191919191515191919151515151919191919191919191919151515151515151515191919191919191919191915151515151515151919191919191919191919191515151502151519191919191919191919191919151515150215151919191919191919191919191915151502020215191919191919191919191919190202150202021919191919191515151919191919150202020202191915151515151515191919191915151502020215151919191515151519191919151502020202151519191919191515151915151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (32, '0100000100000000000000594000000000000059C000000000E0C8504100000000487C434100000000000000000000000000000000DB0B000014001400440014141818181818181818181819191717021414141414141818181818181818191917171714141414191414181818181818181819191717171417141419141415191819181919191919171714171414141914151519191919191919191717171417141414191415151919191919191919171717171414141419151515191919191919191717171717141414141919151515191919191917171717141414141414191915151519191919191717171414141414131319191515151919191919171717141414131313131915151515191919191717171414141313131313191915151519191919171717141414131313131319191515151919191717171414141413131313131919151515151919171714141414131313131313191915151515191917171414141313131313131319151515151519191717141414131313131313131915151515151515151313131313131313131313151515151515151515151313131313131313131315151515151515151515151313131313131313131515151515151919151515131313131313131315');
INSERT INTO corine_land_cover (rid, rast) VALUES (33, '0100000100000000000000594000000000000059C000000000D4CA504100000000487C434100000000000000000000000000000000DB0B000014001400440014141414292929191919191919171717171717141414141429292919191919191917171717171414141414292929191919191919191717171414141414141414291919191919191902171702141414141414141419191919191919190202020202141402141414141919191919191919020202020202020214141419191919191919191902020202141414141414141919191919191919020202021414141414141414191919191914141402020214141414141413141919191919141414141414021414141414141414191919141414141414141414021414141414141419191914141414141414020202141414141414191919191414141414141514020214141515151319191919141414141515151515151415151515131919191914141415151515151515151515151519191913191915151515151515151515151515151313131319191515151502021515151515151515131313191915151515150202021515151515151519191919191515021515020202151515151515151919191915151502020202020215151515151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (34, '0100000100000000000000594000000000000059C000000000C8CC504100000000487C434100000000000000000000000000000000DB0B000014001400440014191419191919191919191917171010101010101414141919191919191919191917171010101010141919191919191919191919191717101010101014191919191919191919191919191710101010101419191919191919191919191917171010101010141419191919191919191919191717171010101014141919191919191919191919191717171010101414191919191919191919191917171717171010141919191919191919191917171717171717101014151919191919191717171717171717171717101515191919191717171717171717171717171717151919151515171717171717171717171717171715151515171717171717171717171717171717171515151517171717171717171717171717171717151515151919191919191919191717171717171715191919191919191919191919191717171717171519191919191919191919191717171717171717191919191919191919191717171717171717171515191919191919191917171717171717171717151519191919191919191917171717171717171515');
INSERT INTO corine_land_cover (rid, rast) VALUES (35, '0100000100000000000000594000000000000059C000000000BCCE504100000000487C434100000000000000000000000000000000DB0B000014001400440010100303030303030303020202020202020215151010030303030303030302020202020202020215101003030303030303030302020202020202021510101003030303030303030202020202020202151010101003030303030303020202020202020215101010100303030303030202020202020202021510101010100303030302021010020202020202151010101010100303020202101010020202020202101010101010101010101010020202020202020210101010101010101010100202020202020202021710100210101010100202020202020202020202171502021010101002021010020202020202020215150202101010101010101002020202020202021715020202101010101010101002020202020202171502020210101010101010100202020202020215151502021010101010101010020202020202021515151502101010101010101002020202020101151515150202021010020202020202020201010115151515150202101002020202020202010101011515151515020202020202020202020201010101');
INSERT INTO corine_land_cover (rid, rast) VALUES (36, '0100000100000000000000594000000000000059C000000000B0D0504100000000487C434100000000000000000000000000000000DB0B0000140014004400151515151502020214141418181819191818181D1515150202020202141414141818181919181D1D1515150202020202141414141414141819181D1D1515151502020202141414141414141418181D1D1515151502020214141414141414141414181D1D15150202020202141414141414141414141D1D1D15150202020202141414141414141414141D1D1D1515020202020214141414141414141414141D1D151502020202021414141414141414141414141D15150202020214141414141414141414141414141515150202021414141414141414141414141414151515020202141414141414141414020214020215150202020214141414140214141402021414021515020202021414141414020202020202141414151502020202141414141402020202021414141415150202020214141414141402020202141414141515020202021414141414020202020214141414010202020202021414021402020202020214141401010202020202020202020202020202020202020101020202020202020202020202020202020202');
INSERT INTO corine_land_cover (rid, rast) VALUES (37, '0100000100000000000000594000000000000059C00000000010C15041000000006078434100000000000000000000000000000000DB0B0000140014004400171717191919191919191919191919191919191917171719171919191919191919191919191919191717171719191919191919191919191919191919171717171919191919191919191919191919191917171719191919191919191919191919191919191717191919191919191919191919191919151919171919191919191919191919191919191515191919191919191919191919191919191919191515191719191919191919191919191919191919191519171919191919191919191919191919191919151919191919191919191919191919191919191915191D1D1919191919191919191919191919191D1519171D191D1919191919191919191919191D1D15191D1D1D1D1D1D1D1D1D1D19191919191D1D1D151519191919191D1D1D1D1D1D1D1D1D1D1D1D151515191919191919191D1D1D1D1D1D1D1D1D151515151919191919191919191919191919191919151515191917171919191919191919191919191915151519191717191917191919191917191919191915151919171917171719191919191717191919191515');
INSERT INTO corine_land_cover (rid, rast) VALUES (38, '0100000100000000000000594000000000000059C00000000004C35041000000006078434100000000000000000000000000000000DB0B00001400140044001919191919191919191919191919191719171719191919191919191919191919191919191919191919191917191919191919191919191919191919191919191917191919191919191919191919191919191919171719191919191919191919191919191919191917171717191919191919191919191912191919191717171719191919171919191919121219171717171717171919191717171919191912121919191717171717191919171717191212121212121917171717171719191717171719121212121217151517171717151717171717191912121212121715151515151515171717171717191212121717171515151515151717171717171919191717171717151515151515171717171719191919171717171715151517151517171717171719191917171717171515151517171717171717171717171717191717151515151518171818171717171717171719191D151515151818181818171717171717171719191915151818181818181817171717171717171919191818181818181818181717171717171717191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (39, '0100000100000000000000594000000000000059C000000000F8C45041000000006078434100000000000000000000000000000000DB0B0000140014004400171919191919191D1D1D1D1515151515151515151719191919191919191D1D1515151515151515151717191919191919191D1D1515151515151515151917171919191919191D1D15151515151515151519171717191919191D1D1D1515151515151515151917171717191919191D1D151515151515151515191917171719191919191515151515151515151519171717191919191919151515151515151515151217171719191D1919191515151515151515151517171717171D1D1D19191515151515151515151517171717171D1D1D191915292915151515151519171717171D1D1D1D191529292929151515151519171D1D1D1D1D1D1D192929292929151515151519171D1D1D1D1D1D19192929292915151515151519171D1D1D1D1919192929292915291515151515191D1D1D1D191919192929292929151515151515191D1D1D1919191919292929291515151515151519191919191919191929292915151515151515151919191919191919191515151515151515151515191919191919191919151515151515151515151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (40, '0100000100000000000000594000000000000059C000000000ECC65041000000006078434100000000000000000000000000000000DB0B000014001400440015020202151519191919191919191515151919190202021515151919191919191919191919191919021515151515191919191919191919191919191915151519191919191919191919191919191919191515151919191919191919191919191919191919151515191919191919191919191919191919191915151919191919191919191919191919191919191515191919191919191919191919191919191919151519191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191519191919191919191919191919191919191915191919191919191919191919191919191919191519191919191919191919191919191919191919151919191919191919191919191919191919191915191919191919191919191919191919191919191919191919191919191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (41, '0100000100000000000000594000000000000059C000000000E0C85041000000006078434100000000000000000000000000000000DB0B000014001400440019151915151919191515131313191919191919151515191919191919191313131319191919151515191919191919191919131313131919151515151519191919191919191913131319191915151519191919191919191919191313131919151515151919191919191919191919131313191919151515191519191919191919191913131919191515151915151919191919191919191313131919151519191515191919191919191919131319191915151919151519191919191919191919191919191915191915151919191919191919191919191919191919151515191919191919191919191919191919151515151919191919191919191919191919191515151515191919191919191919191919191915151515151919191919191919191919191919191515151919191919191919191919191919191915151519191919191919191919191919191919191919191919191919191919191919191919191919191519191919191919191919191919191919191919151919191919191919191919191919191919191515191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (42, '0100000100000000000000594000000000000059C000000000D4CA5041000000006078434100000000000000000000000000000000DB0B000014001400440019151915151515020202020202151515151515151515151515151302020202020202151515151515151515151515130202020202020202151515131515151515131313020202020202020202131313131515151513131302020202020202021313131313151515131313020202020202130202131313131915151513131302021313020213021313191919191515131313131313131302131302131313191919151313131313131313131313020213131313131915131313131313131313130202021313131319191313131919131313131313020213131313191919191919191913131313131313131313181818181919191919191313131313131313181818181819191919191913131313131318181818181819191918191919131313131818181818181819191818181819191913131818181818181818181818181818191919191818181818181818181812121818181818191919181818181818181812121212181818181819191918181818181812121212121818181818181919191818181818181212121212121818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (43, '0100000100000000000000594000000000000059C000000000C8CC5041000000006078434100000000000000000000000000000000DB0B000014001400440015191919191919191919171717191917171515151519191919191919191517171919191915151515151519191915151515151719191919151515151515151515191515151515191919191915151919151919191515191919191919191919191919191915191919191919191919191919191919191917191919191919191915151919191919191919171719191919191919191515191919191919191717171919191919191915151519191919191917171717191919191915151515151919191919191717171719191919191915151515151519191919171717170202191819191515151515151519191917171717020218181919191515151515151919191717171702021819191919191515151519191917171717020202191919191919151515151919191717171717020219181919191915151519191919171717171919021802021902191919191919191917171719191902180202021919191919191919171717191919191902020219191919191919191717171919191919190202021919191919191919171718181819191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (44, '0100000100000000000000594000000000000059C000000000BCCE5041000000006078434100000000000000000000000000000000DB0B00001400140044001515151515020202021919020202020101010101151515151502020219191919190202010101010115191919151902191919191919020201010101011919191919191902021919191902020101010101191919191919191902191919020201010101010119191919191D1D1D1D02020202020101010101011919191919191D1D021D020202020101010101011919191919171D1D0202020202010101010101011919191919171D1D1D02020202010101010101011919191919171D1D1D1D020202010101010101010202020217171D1D1D1D020202010101010101010202020217171D1D1D1D02020202020201010101020202021717171D141D020202020202020101010202020217171D141414020202020202020101010202020217171D1414140202020202020201010102020202171714141402020202020202020101010202020217171714140202020202020202010101191902171717171414020202020202020101010119191717171717141402020202020202010101011917171717171714140202020202020201010101');
INSERT INTO corine_land_cover (rid, rast) VALUES (45, '0100000100000000000000594000000000000059C000000000B0D05041000000006078434100000000000000000000000000000000DB0B00001400140044000101020202020202020202020202020202020202010101020202020202020202020202020202020201010102020202020202020202020202020215150101010102020202020202020202020202151515010101010202020202020202020202151515151501010101020202020202020202151515151515150101010202020202020202151515151515150F0F010101020202020202020215150F0F0F0F0F0F0F010101020202020202020215020F0F0F0F0F0F0F010101010101020202020202020F0F0F0F0F0F0F010101010101010102020202020F0F0F0F0F0F140101010101010101010202020F0F0F0F0F0F0F140101010101010101020202020F0F0F0F0F141414010101010101010202020202020F0F0F0F1414140101010101010101020202020F0F0F0F0F14141401010101010101010202020F0F0F0F0F0F1414140101010101010101020202020F0F0F0F0F14141401010101010101010202020F0F0F0F0F0F141414010101010101010102020F0F0F0F141414141414010101010101010202020F0F0F0F1414140F0F0F');
INSERT INTO corine_land_cover (rid, rast) VALUES (46, '0100000100000000000000594000000000000059C00000000010C15041000000007874434100000000000000000000000000000000DB0B000014001400440019191719171717191919191917171719191918181919171717171919191919191717191919191918191917171717171919191917171719191919191919191717171717171919171717171919191919191917171717171717191917171717171919191919191917171717171719191917171717191919191919191917171717171919191717191919191919191919191917171719191917171919191919191919191919191717171919191717171919191919191919191917171717191919171717171719191919191919191717171919191919171717171719191919191919171717191919191919171717171919191919191917171719171719191919191917171702021919191917171717171719191919171717171702191919191717171717171719191917171717171719191919191717171717171717171717171717171919191917171717171717171717171717171717191919191717171717171717171717171717171719191919171717171717171717171717171717171919191919171717171717171717171717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (47, '0100000100000000000000594000000000000059C00000000004C35041000000007874434100000000000000000000000000000000DB0B000014001400440018181818181818181817171717191919191919191818181818181817171717171919191919191919181818181818181717171717191919191919191918181818181817171717171919191919292929191818181818181D1D171717171919292929292929191D1D1818181D1D1D1717171929292929292929191D1D18181D1D1D1D1D171717292929292929291D1D1D1D1D1D1D1D171D171714291429292929291919191D1D1D1D1D1717171714142929292929291919191D1D1D1D1717171714141429292929292902191D1D1D1D1D17170214141414142929292915021D1D1D1D1D1D0202020214141414142929151502021D1D1D0202020202141414141414151515150202020202020202020214141414141415151515170202020202020202141414141414141515151517171717020202020214141414141407151515151717171714141414141414141414140715151515171717171414141414141414141407070707070717171717141714141414141414141407070707071717171417171414141414141414140707070707');
INSERT INTO corine_land_cover (rid, rast) VALUES (48, '0100000100000000000000594000000000000059C000000000F8C45041000000007874434100000000000000000000000000000000DB0B000014001400440019192919192929191515151515151515151515151929292929291919151515151515151515151919292929291919151515151515151515151519191929292919151515151515151515151515191919192919191515151515151515151515151919191919291915292915151515151515151519191919191929292929151515151515151515151915151919192929291515151515151515151519191515191919291515151515151515151515151915151919191915151515151515151515151515151515151518181515151515151515151515151502021515151818151515151515151919151502020202151518181815151515151515191502020202020215151818181515151515151919020202020202021515181818151515151919191402020202020215151818181815151519191914141414140202021518181818181515071919191414141414140215151818181818070719191919141414141414151515181818181807071919191414141414141515151518181818180707191919141414141414151518181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (49, '0100000100000000000000594000000000000059C000000000ECC65041000000007874434100000000000000000000000000000000DB0B000014001400440019191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191918181819191919191919191919191919191919191818181819191919191919191919191919191919181818181919191919191919191919191919191918181818181919191919191919191919191919191818181818191919191919191919191919191919181818181818181919191919191919191919191918181818181818181819191919191919191919191818181818181818181919191919191919191919181818181818181818191919191919191919191918181818181818181919191919191919191915151818181818181819191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (50, '0100000100000000000000594000000000000059C000000000E0C85041000000007874434100000000000000000000000000000000DB0B000014001400440019191919191919191919191915191919191919191919191919191919191919151519191919191919191919191919191919151515151519191919191919191919191919191915151515151919191919191919191919191919191915151519191919191919191915191919191919191919191919191919191919151519191919191919191919191919191919181915151919191919191919191917171919191918191519191919191919191919191717191919191819151919191919191919191919171719191918181915151919191917171917171917191919181818191515191919191717191717171719191818181819151519191919171717171717191919181818181915191919191917171717191919191918181818191519191919191717171719191919181818181819151919191917171717171919191918191818181915191919151717171717191919191919181818151519191515171717171719191919191818181815151515151517171717171919191919181818181515151515151717171719191919181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (51, '0100000100000000000000594000000000000059C000000000D4CA5041000000007874434100000000000000000000000000000000DB0B000014001400440019191818181818181818121212121818181818181919181818181818181212121818181818181818181818181818181818121818181818181818181819181818181818181818181818181818181818021818181818181818181818181818181818181802181818181818181818181818181818181818180218181818181818181818181818181818181818021818181818181818181818181818181818180202181818181818181818181818181818020202020218181818181818181818181818181802020218181818181818181818181818181818020202181818181818181818181818181818181802151518121818181818181818181818181818181515121218181818181818181818181818181515151512121212181818181818181818181815151515151212121218181818181818181818151515151515121212121818181818181818181815151515151212121212181818181818181818181515151512121212121A181818181818181818151515151212191912121A181818181818181818151515121219191912121A');
INSERT INTO corine_land_cover (rid, rast) VALUES (52, '0100000100000000000000594000000000000059C000000000C8CC5041000000007874434100000000000000000000000000000000DB0B0000140014004400020219191919191919191718181818181919191902020219191918181918181818181819191919190202021818181818181818181818191919191919021818181818181818181818181819191919191902181818181818181818181818191919191919190202181818181818181818181819191919191919020218181818181818181818191919191717171702021818181818181818181919191917171717170202181818181818181819191919171717171717181818181818181818191919191917171717171718181818181818181819191919171717171717171818181818181818181817171717171717172020181818181818181818171717171717171717201718181812181818181A1717171717171717202017181212121218181A2020202017171717202020171212121A1A1A1A20202020202017171720201717121A1A171717202020202020201720202017171D1A1A17171720202020202020202020201717171D1A1A17171720201717202020202020171717171D1A171717201717172020202020201717171D1D1D');
INSERT INTO corine_land_cover (rid, rast) VALUES (53, '0100000100000000000000594000000000000059C000000000BCCE5041000000007874434100000000000000000000000000000000DB0B000014001400440019191919171714141402020202020201010101011917171717171714140202020202020201010101191717171717171414020202020202020101010119171717171717141402020202020202010101021717171717171714140202020202020201020202171717171717141414020202020202020202020217201717171714141402020202020203030202021720201717141414140202020202030303030202172020171414141414140202020303030314140220201717171414141414020202020303031414142017171714141414141402020203030303141414171717171714141414141402020303030314141417171717191414141414140202020303030303141D1D1D19191414141414140202020303030303031D1D1919191414141414141402030303030303031D191919191914141414141402030303030303031D191919191914141414140202030303030303031D191919191402020202020203030303030303031D191914141414020202020203030303030303031D19191414141402020202020203030303030303');
INSERT INTO corine_land_cover (rid, rast) VALUES (54, '0100000100000000000000594000000000000059C000000000B0D05041000000007874434100000000000000000000000000000000DB0B0000140014004400010101010101010202020F0F0F141414140F0F0F010101020202020202020F0F0214141414140F0F020202020202020202020F0202141414140F0F0F0202020202020202020202020202141414140F0F0202020202020202020202020202141414140F0F020202020202020202020202021414140202020202020202020202020215150214141414020202020202020202020202151515021414141402020215020202020202020215151502021414140202020202020202020202021515150202141414140202021402020202020202151515020214140202020215141402020202020215151502021414020202020214141402020202151515020202140202020202021414140202020202151515020214020202020202141414140202020215150202020202020202020214141414020202021515020202020202020202020314141402020202151515020202020202021515031414141402020202151502020202020215151503031414141402020202151502020202021515150303030314140202020202150202020202151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (55, '0100000100000000000000594000000000000059C00000000010C15041000000009070434100000000000000000000000000000000DB0B0000140014004400191919191717171717171717171717171717171719191919171717171717171717171717171717171919191919191917171717171717171717171717191919191919191917191717171717171717170719191919191919191919191917171717171717071919191919191919191919191717171717170707191919191919191919191917170707070707070719191919191912191919190707070707070707071919191919121212191907070707070707070707191919191912121212191D0707070707071515071818191912121212121D1D0707070707070715151818181912121212121D1D0707070707070707071819191912121212121D1D07070707070707070719191219121212121D1D1D1D1D0707070707070719191212121212121D1D1D1D1D1D07071D1D07071919121212191919191D1D1D1D1D1D1D1D1D19151919191919191919191D1D1D1D1D1D1D1D1919151919191919191919191D1D1D1D1D1D1D1D1919151919191919191919191D1D1D1D1D1D1D1D19191519191919191919191D1D1D1D1D1D1D1D1D1D1919');
INSERT INTO corine_land_cover (rid, rast) VALUES (56, '0100000100000000000000594000000000000059C00000000004C35041000000009070434100000000000000000000000000000000DB0B000014001400440017171717171414141414141414141407070707071717171717141414141414141414140707070707071717171414141414141414141414140707071407171714141414141414141414141414141414140717141414141414141414141414141414141419070714141414141414141414141414141414141915150714141414141414141414141414141419191515151414141414141414141414141414141919151515151414141414141414141414141414191907151515141414141414141414141414141919191515151514141414141414141414141414191919071515151414141414141414141414141419191907071515141414141414141414141414141919190715151515141414141414141414141419191919071515151515141414141414141414141919191915151515151414141414141414141419191919191515151514141414141414141414141919191919151515151514141414141414141414191915151515151515151414141414141414141419151515191515151515141414141414141414141515151519');
INSERT INTO corine_land_cover (rid, rast) VALUES (57, '0100000100000000000000594000000000000059C000000000F8C45041000000009070434100000000000000000000000000000000DB0B000014001400440007071919191414141414141515181818181818181919191914191914141414181818181818181818191919141419191414141418181818181818181819191919191919191414141414141418181818181919191919191914141414141414141418181818191919191919191414141414141414141818181819191919191919141414141414141414181818181919191919191915141414141414141418181818191919191919191515141414141414140218181819191919191915151514141414141402021818181919191919151515151414141414140202181818191919191915151515141414141414021818181819191919191515151514141414140202171818181919191919151515151414141402020217181818191919191915151515141414140202021717181819191919191515151514140202020214171717181919191919151515141402020202021417171718191919191515151514141402020214171717171719191919151515151414140202141417171717171919191919151515141414020214141717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (58, '0100000100000000000000594000000000000059C000000000ECC65041000000009070434100000000000000000000000000000000DB0B000014001400440018181818181818191919191919191919191919191818181818181819191919191919191919191919181818181818181919191919191919191919191918181818181818191919191919191919191919191818181818181919191919191919191919191919181818181818181919191919191919191919191918181818181818191919191919191919191919191818181818181819191919191919191919191919181818181818191919191919191919191919191818181818181818191919191919191919191918181818181818181819191919181818191919191918181818181818181919191918181818191919191818181818181818191919191818181818191919181818181818181819191919181818181818181818181818181818181919191818181818181818181818181818181818191919181818181818181818181818181818181819191918181818181818181818171818181817181819191818181818181818181817171717171718181818181818181818181818181717171717171818181818181818181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (59, '0100000100000000000000594000000000000059C000000000E0C85041000000009070434100000000000000000000000000000000DB0B000014001400440019151515151517171717191919191818181818181515151515151517171919191919191818181918151515151515151717191919191919181818191915151518181515171717191919191918181819191515151818181517171719191919191818191919151515181818151717171919191918181919191915151518181815151717191919191919191919121515151818151515171919191919191919191912151518181815151517191919191919191912121818151818181515171719191919191212121218181818181815151518171919191919191212121818181818181515151817171919191912121212181818181818181818181717191919191912121212181818181818181818181719191919191212121212181818181818181817171719191919121212121218181818181D1D1817171719191919121212121218181818181D1D18171717191818181212121212181818181D1D1D18171712121818181212121212181D1D181D1D1D18181712121818181812121212181D1D1D1D1D1818181812121218181812121212');
INSERT INTO corine_land_cover (rid, rast) VALUES (60, '0100000100000000000000594000000000000059C000000000D4CA5041000000009070434100000000000000000000000000000000DB0B00001400140044001818181818181818151515151212191912121A1A181818181818181515151212121919191A1A1A1A18181818181815151518121919191A1A1A171A171818181818181818181812191919191A1A171717191818181818181818181D191919191F171717171818181818181818181D1D191919191F171717171818181818181818181D1D1919191F1F1B1B17171218181818181818181D191919191F1F1B1717171818181818181818181D1919191F1B1B1B17171718181818181818181D1D19191F1F1B1B1B17171718181818181818181D1D19191F1F1F1F1B1717171818181D181D1D1D1D1D19191F1F1F1B1B1717171818181D1D1D1D1D1D19191F1F1B1B1B17171717181818181D1D1D1D1D1D191D1F1F1B1B1717181818181818181D1D1D1D1D1D1F1F1F1F1B1718181818181818181D1D1D1D1D1D1F1F1F1F1B1B181818121818181818181D1D1D1D1B1F1F1F1B18181818121818181818181D1D1D1D1B1F1F1F1818181818121218181818181D1D1D1D1B1B1F1B181818181812121818181818181D1D1D18181B181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (61, '0100000100000000000000594000000000000059C000000000C8CC5041000000009070434100000000000000000000000000000000DB0B000014001400440017171717202017171717202017171717171D1D1D171720172017172020202017171717171D1D1D1D171720202017171720201717171717171D1D1D1D172020202020201717171717171717171D1919191720202020202017171717171717191919191919171717202017171717171719191919191919191917171717171717171717171919191919191919171717171717171717171717171717191717171717171717171717171717171717171717171717171917171717171717171717171717171717191919191717171717171717171717171717171717191919171717171717171717171717171717171719191917171717171717171717171717171717171719191717171717171717171717171717171717171719171717171717171717171717171717171717171918171717171717171717171717171717171717171817171717171717171717171717171717171717181817171717171717171717171717171717171718181817171717171717171717171717171717171818181717171717171719191919191717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (62, '0100000100000000000000594000000000000059C000000000BCCE5041000000009070434100000000000000000000000000000000DB0B00001400140044001D191914141402020202020202030303030314031D191914140202020202020202030303031414031D1919141919190202020202020303030303141419191919191919190202021414140303030314141919191919151515190202141414141414141414191915151515151515151514141414141414141419191915151515151515151414141414141414141919191515151515151515141414141414141414191915151515151515151514141414141414141419191519191915151515151414141414141414141919191919191915151515141414141414101010191919151919191915151414141414101010101019191915151519151515141414141410101010101919191515151919151519141414141410101010191919191515151919191919141414141010101019191919151515191919191914141414141010101919191915151519151519191414141414101010171919191519151515151919141414141010101017171915151919191919191914101010101010101717191519141414141419191010101010101014');
INSERT INTO corine_land_cover (rid, rast) VALUES (63, '0100000100000000000000594000000000000059C000000000B0D05041000000009070434100000000000000000000000000000000DB0B000014001400440003030303141402020202151515151502020215020303030314020202021515151717150202020202030303031402020215151517171717020202020203030303020202021515171717171717020202020303140314140202151517171717171702020202030314141414020215151717171717171702021814031414141414021515171717171717170202181403141414141402151517151717171717020218141414141414020215151517171717171702021814141414141402021515151517171717170202181014141414100202151515151717171717021919101010101010101015151515151717171702191910101010101010101515151515171717171719191010101010101010151515151717171717171919100606101010101015151515171717171717191910060610101010101515151517171717191919191006061010101010151515171717171719191919100606061010101010151717171717191919190F06060606061010101017171717171719191919190606060606101010101717171717171719191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (64, '0100000100000000000000594000000000000059C00000000010C1504100000000A86C434100000000000000000000000000000000DB0B00001400140044001D171D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D19191D17171D1D1D1D1D17171D1D1D1D1D1D1D1919151D1717171D1D1D1D1D17171D1D1D1D1D1D1915151D1717171D1D17171717171D1D1D1D1D191915151D171717171D1717171717171D1D1D19191919151D1717171717171717171717171D1D19191919151D171717171717171717171719191D19191919191D171717171717171717171719191919191919191D171717171717171717171717191919191919151D17171717171717171717171719191919191915171717171717171717171717171919191919151517171717171717171717171717191515151515151717171717171717171717171719151515151519171717171717171717171719191515151515191917171717171717171717171915151515151519191D1D1717171717171717171915151515151515191D171717171717171717171915151515151515151D171717171717171717171919191515151919191D1D1717171717171717171919191915191919191D1D171717171717171717191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (65, '0100000100000000000000594000000000000059C00000000004C3504100000000A86C434100000000000000000000000000000000DB0B000014001400440015151515151414141414141414141415151515191515151515141414141414141414151515151515151515141414141414141414141415151515151515151515141414141414141414141515151515151515151515141414141414141414151515151515151515151515141414141414141415151515151815151515151414141414141414141515151515181515151414141414141414141415151515151518151514141414141414141414141515151515181815141414141414141414141414151515151518181514141419141414141414141515151515151818141414191914141414141414151515151518181814141919191414141414141515151515151518181919191919191414191414151515151515151818191919191919191919141415151515151515181819191919191919191929291515151515151515181919191919191919192929151515151515151518191919191919191929292915151515151515181819191919191919292929291515151515151518181919191919192929292929151515191915181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (66, '0100000100000000000000594000000000000059C000000000F8C4504100000000A86C434100000000000000000000000000000000DB0B000014001400440019191919151515151514140202141417171717171519191915151515151414020214141717171717151919191915191915140202141414141717171715191919191519191514141414141414171717171819191919191919151514141414141717171717181919191919191915151414141414171717171718191919191519191515141414141417171717171819191915151919151514141414171717171717181919151515191915151414141414171714171718191915151515151515141414141414141417171818181515151515151514141414141417171717181815151515151515151414141414171717171718181515151515151514141414141417171717171818151515151515151414141414141717171717181515151515151515141414141414171717171718151515151515151414141414141419191717171815151515151515141414141414141419171717181815151515151514141414141414191717171718181515151515151414141414141419171717171818181515151515141414141414141919171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (67, '0100000100000000000000594000000000000059C000000000ECC6504100000000A86C434100000000000000000000000000000000DB0B000014001400440017171717171718181818181818181818181818181717171717171818181818181818181818181818171717171717181818181818181818181818181817171717171717181818181818181818181818181717171717171718181819191918191D1D181818171717171717171818191919191919191D1D1D181717171717191919191919191919191919191D1D1717171919191919191919191919191919191D1D171719171919191919191919191919191D1D1D1D17171717191919191919191919191919191D1D1D1717171919191919191919191919191919191D1D171719191919191919191919191919191919191917171919191919191919191919191919191919191719191919191919191919191919191919191919171719191919191919191919191919191919191917171919191919191919191919191919191919191717191919191919191919191919191919191919171717191919191919191919191919191919191917171717191919191919191919191919191919191717171717191919191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (68, '0100000100000000000000594000000000000059C000000000E0C8504100000000A86C434100000000000000000000000000000000DB0B000014001400440018181D1D1D1D1D1818181212121818181212121218181D1D1D1D1D1D18121212121818181212121218181D1D1D1D1D1818121212121818181212121218181D1D1D121218181212121818181812121212181D1D1D1D1212181212121818181818121212121D1D1D1D121212181812121212181812121212121D1D1D1D121212181812121212121212121212121D1D1D1D121212181812121212121212121212121D1D1D1D1D1212121812121212121212121212121D1D1D1D1D1212121812121212121212121212121D1D1D1D1D1212121812121212121212121212121D1D1D1D1D121212121212121212121212121212191D1D1D1D121218121212121219121212121212191D1D1D1D12121212121212191912121212121219191D1D1D12121212121219191912121212121219191D1D1D121212121219191919121212191212191D1D1D1D121212191919191919121219191919191D1D1D1D12121919191919191912121919191919191D1D1D12121919191919191919191919191919191D1D1D121919191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (69, '0100000100000000000000594000000000000059C000000000D4CA504100000000A86C434100000000000000000000000000000000DB0B000014001400440012121218181818181818181818181818181818181212121218181818181818181818181818171718121212121218181818181818181818181717171812121212121818181818181818181817171717181212121212121812121812181818171717171717121212121212121212121212181717171717171712121212121212121212121217171717171717171212121212121212121212121217171717171919121212121212121212121212121717171717191912121212121212121212121212171717171719191212121212121212121219121217171717171919121212121212121919191919171717171717191912121212121219191219191917171717171719191212121212121219121919171717171717191919121919191212121212191917171717191919191919191919121212121919171717171719191919191919191912121212191917171717191919191919191919191912121219191717171719191919191919191919191912121919171719191919191919191919191919191912191917171719191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (70, '0100000100000000000000594000000000000059C000000000C8CC504100000000A86C434100000000000000000000000000000000DB0B000014001400440018181717171717171717171919191917171717171817171717171719191919191919171717171717181717171717171919191919191717171717171717171717171719191919191919191717171717171717171717191919191919191919171717171717171717171719191919191919191919191717171717171717171719191919191919191917171717171717171717171915191515191915171717171717191717171717151515151515151517171717171719171717171717151515151515151515171717171917171717171715171715151517171517171717191917171717171717171517171717171717171719191717171717171717171717171717171717171919171717171717171717191919191717171717191717171717171717171919191915171717171717171717171717191919191919151519191919191717171717191919191919191919151519191919191919191919191919191919191915191919191919191919191919191919191902021515191919191919191919191919191919190202021515151919');
INSERT INTO corine_land_cover (rid, rast) VALUES (71, '0100000100000000000000594000000000000059C000000000BCCE504100000000A86C434100000000000000000000000000000000DB0B0000140014004400171719191914140214020214141010101010141417171919171714020202141414141410101414141717171717170202021414141414141414141414171717171702020202141414141414141414141417171717170202020214141414141414141414141717171717140202021414141414141414141414171717171414020202141414141414141414141417171717171414020214141414141414141414141717171717141402141414141414141414141414171717171717141414141414141414141414141417171717171717141414141414141414141414141717171717171714141414141414141414141414171717171717171714141414141414141414141417171717171717171414141414141410101414141717171719171919141414141414101010141414191919191919191910101014101010101014141419191919191919191010101010101010101010141919191D1D1D19191010101010101010101010141919191D1D1D1D10101010101010101010101014191D1D1D1D1D1D10101010101010101010101414');
INSERT INTO corine_land_cover (rid, rast) VALUES (72, '0100000100000000000000594000000000000059C000000000B0D0504100000000A86C434100000000000000000000000000000000DB0B000014001400440006060606061010101017171717171717191919190606060606101010101717171717171719191919060606060610101017171717171717191919190F0606060610101010171717171717171919190F0F06060614141010101017170217171919190F0F19141414141410101017170202021919190F1919191414141414141002020202020219190F0F190F0F14141414141414020202020202190F0F0F190F0F141414020202140202020202190F0F0F0F19190F141414020202020202020219190F0F0F0F19190F141414020202020202021919190F0F0F0F19190F141414020202020202021919190F0F0F0F190F0F1414140202020202020219190F0F0F0F0F0F0F0F14140202020202020202190F0F0F0F0F0F0F0F0F14020202020202020202190F0F0F0F0F0F0F19191402020202020202020F0F0F0F0F0F0F0F19191914020202020202020F0F0F0F0F0F0F0F0F1919191402020202020202020F0F0F0F19190F0F0F0F1914020202020202020F0F0F150F0F0F190F0F19191414020202020215150F0F1515150F1919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (73, '0100000100000000000000594000000000000059C00000000010C1504100000000C068434100000000000000000000000000000000DB0B00001400140044001D20202017171717171717191919191919191919202020201717171717171719191919191919191920202020171717171719191919191919191919192020202017171717191919191919191919191919202020201717171719191919191919191919191920201F1F1717171719191920192020201919191920201F1F171717171919192020202020191919191F1F1F1F181818181919192020202019191919191F1F1F18181818181519192020202019191919191F1F1818181818191519192020202019191919191F1F181818181819151919202020201919191919181818181818191915151920202020202020191918181818181819191515192020202020202020191818181818181818151519202020202020201919181818181818181818151920202020202020202918181818181818181815202020202020202020201818181818181818181520202020202020202020181818181818181818152020202020202020202018181818181818181815152020202020202020201818181818181818181515201919202020202020');
INSERT INTO corine_land_cover (rid, rast) VALUES (74, '0100000100000000000000594000000000000059C00000000004C3504100000000C068434100000000000000000000000000000000DB0B000014001400440019191919191929292929291515151919151818181919191919192929292915151515191915181818191919191929292929291515151519191515181819191919192929292929151519151919151515151919191929292929291515151919191915151515191919192929292929151515191919191515151519191919292929292915151919191919151515151919192929292929291515191919191915151515191919292929292929191919191919191515151519192929292929292919191919191919191919191919292929292929191919191919191919191919192929292929291519191919191919191919191919292929292915151919191919191919191919191929151515151515191919191719191919191919291515151515151519191917171919191919191929151515151515191919171719191919191919192015151515151519191917171917171919191919202020201515151917171719191719191919191920202020151515171717171917191919191919192020202015151515171717171719191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (75, '0100000100000000000000594000000000000059C000000000F8C4504100000000C068434100000000000000000000000000000000DB0B000014001400440015181515151515151414141414141419191917171515151515151514141414141414141818191919151515151515151414141414141418181818181815151515151514141414141414141418181818181515151515151514141414141414181818181818151515151515151402020214141818181818181815151502021515140202021414181818181818181515020202141514020214141818181818181818151502020214140202021414181818181818181815150202020202021414141818181818181818181919020202020214141414141418181818181818191919020202021414141414141818181818181819190202020202141414141418181818181818181919190202020202141414141818181818181919191919020202021414141419191818181819191919191902021402141414141919191918191919191919191414141414141414191919191919191919191919141414141414141414141919191919191919191919141414141414141414141419191919191919191914141414141414141414141919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (76, '0100000100000000000000594000000000000059C000000000ECC6504100000000C068434100000000000000000000000000000000DB0B00001400140044001919191919191919191919191919191919191919191919191919191919191919191919191919191918181818191919191919191919191919191919191818181818191919191919191919191919191919181818181818181919191919191919191919191918181818181818181818191919191919191919191818181819191919191919191919191919191919181818181919191919191919191919191919191918181818191919191919191919191919191919191818181819191919191919191919191919191919181818181919191919191919191919191919191918181818191919191919191919191919191919181818181819191919191919191919191919191919181818181919191919191919191919191919191A191918191919191919191919191919191919191A19191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919202020');
INSERT INTO corine_land_cover (rid, rast) VALUES (77, '0100000100000000000000594000000000000059C000000000E0C8504100000000C068434100000000000000000000000000000000DB0B000014001400440019191D1D1D12191919191919191919191919191919191D1D1D1A1919191919191919191919191919191919191A1A1919191919191919191919191818191919191A1A19191A1919191919191919181818191919191A1A1A1A1A191919181818181818181819191919191A1A1A1A1A19191818181818181818191919191A1A1A1A1A1A1A1A18181818181818181919191A1A1A1A1A1A1A1A1A18181818181818181919191A1A1A1A1A1A1A1A1F18181818181818201819191A1A1A1A1A1A1A1A1F18181818181818201818181A1A1A1A1A1A1A1A1F1818181818181820181818181818181A1A1A1F1F1F18181818182020181818181818181818181F1F1F18181818182020181818181818181818181F1F1F181818182020201A1818181818181818181F1F1F181818202020201A1A18181818181818181F1F1F181818202020201A1A1A1A18181818181F1F1F1F2020202020202019191A1A1A1A1A1A1A1F1F1F2020202020202020192020201A1A1A1A1A1F1F202020202020202020202020201A1A1A1A1A1A1F2020202020201D1D20');
INSERT INTO corine_land_cover (rid, rast) VALUES (78, '0100000100000000000000594000000000000059C000000000D4CA504100000000C068434100000000000000000000000000000000DB0B0000140014004400191919191919191919171717171919191919191919191919191919191717171717191919191919191818181919191917171717191919191919191919181818181918181818171919191919191919191918181818181818181819191919191919191919191818181818181818181919191919191919191919181818182018181818191919191919191919191720181820202018181819191919191919191919172020202020201D1D1D191D1D1D191919191917172020202020201D1D1D1D1D1D1D1919191917171720202020201D1D1D1D1D1D1D191917171717171720202020201D1D1D1D1D1D1D191917171717171720202020201D1D1D1D1D1D1D191717171717171720202020201D1D1D1D1D1D1D191717171717171720192020201D1D1D1D1D1D19191717171717171720191920202019191919191919171717171717172020191920201919191919191717171717171717202019191919191919191917171717171717171720191919191919191917171717171717171717172020191919191919191717171717171717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (79, '0100000100000000000000594000000000000059C000000000C8CC504100000000C068434100000000000000000000000000000000DB0B000014001400440019191919191919191919190202020215151519191919191919191917171919020202020215191919191919191919171719191902021502021519191D191919191917171719191902021502021519191D1919191917171717191919190215151519191D1D1919171717171715191919190215151519191D1D1717171717171515191919190215151519191D1D1717171717151519191919190215151519191D1D1717171717151919191919190215151919191D1D17171717151519191919191915151519191D1D1D17171717151519191919191515151919191D1D1D171717151515191919151515151519191D1D1D1D171515151519191915151515151519191D1D1D1D1715151515191915151515151519191D1D1D1D1D1717171715151515151515151519191D1D17171D17171717151515151515151515191D1D1717171D17171717171515151715151515191D1D17171D1D17171717171517151717151515191D1D171D1D1D171715171717171717171515171D1717171717141717151517171717171715171717171717171717');
INSERT INTO corine_land_cover (rid, rast) VALUES (80, '0100000100000000000000594000000000000059C000000000BCCE504100000000C068434100000000000000000000000000000000DB0B0000140014004400191D1D1D1D1D1D101010101010101010101414141D1D1D1D1D1D10101010101010101014141414141D1D1D1D1D1D10101010101010141414141414141D1D1D1D1D1010101010101014141414101010141D1D1D1D1D1010101010141414101010101010141D1D1D1D101010141414141414141010101010141D1D1D1D101414141414101014101010101010141D1D1D10101414141010101010101010101010141D1D1D10101014141410101010101010101010141D1D1010101014141414101010101010101010141D1D1010101014141414141010101010101010141D1D1010101010101414101010101010101014141D1D1010101010101010101010101010101014141D1D1010101010101010101010101010101414141D101010101010101010101010101010101414141D101010101010101010101010101010141414141D10101010101010101010101010101414141414141410101010101010101010101010141414141414141414141010101010101010101414141414141414141414101010101014101014141414141414');
INSERT INTO corine_land_cover (rid, rast) VALUES (81, '0100000100000000000000594000000000000059C000000000B0D0504100000000C068434100000000000000000000000000000000DB0B000014001400440014140202020215151515151515191919191919191414020202151515151515151919191919191919141402020215151515151919191919191919191914020202151515151515151515191919191919191402021515151515151515151519191919191919140202151915151519191515191919191919191914021414191515151919191919191919191919191402141414191919191919191717191919191919141414141414141919191717171717171719191914141414141417171717171717171717171919191414141414141717171717171717171717191917141414141417171717171717171717191919191714141414141717171717171717171719191919191414141414171717171717171717171919191919141414141717171717171717171717171919191914141717171717171717171717171919191919191414141717171717171717171717191919191919141919171717171717171717171717191919191919191919171717171717171719191719191917171919191717171717171719191917171919191717');
INSERT INTO corine_land_cover (rid, rast) VALUES (82, '0100000100000000000000594000000000000059C00000000010C1504100000000D864434100000000000000000000000000000000DB0B000014001400440018181818181818181815151919191919202020202018181818181818151515191919191919202020202018181818181515151519191919191920202020201818181818151515151919191919192020202020181818151515151515151919191920202020202018181815151515192019191919202020202020181818181815151920202019191919202020202018181818181819191920202019192020202020201818181818181919191920202020202020202020181818181818191919191920202020202020171818181919191819191919191920202020202017181819191919191919192020192020202020201718181919191919141919202020202020202017172018191919191914141919202020202020191917181819191919191414191919202020202019171718181919191919141419192019202019191717171818141414141414191919191920201917171717181814141414141419191914141919191717171518181414141414141919191414191917171715151814141414141419191414141414171717151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (83, '0100000100000000000000594000000000000059C00000000004C3504100000000D864434100000000000000000000000000000000DB0B0000140014004400202020201515151517171717171919191915191920202020151515171717171719191919191515192020201717151517171719191919191915151519201717171715171717171919191919191915151520201717171517171717171919191919191919192020201715171717171717191919191919191919201717171517171717171719191919191919191920171717171717171717171919191919191919192017171717171717171719191919191919191919171717171717171717171919191919191919191917171717171717171719191919191919191919191717171717171717171919191D1D191919191919171717171717171D1D1D19191D1D191919191919171717151D1D1D1D1D1D1D1D1D1D1D191915191917171715151D1D1D1D1D1D1D1D1919191915151517171715151D1D1D1D1D1D1D1D1D191919151515171717151D1D1D1D1D1D1D1D1D15191515151515151715151D1D1D1D1D1D1D1D1D1D15151515151515151515151D1D1D1D1D1D151D1D1D1515151515151515151515151515151515151D1D1D15151515');
INSERT INTO corine_land_cover (rid, rast) VALUES (84, '0100000100000000000000594000000000000059C000000000F8C4504100000000D864434100000000000000000000000000000000DB0B000014001400440019191519141414141414141419141419191919191919151919141414141414141419191919191919151915191914141414141414141419191919191915191515151414141414141414141419191919191919191515141414141414141414141414141917191919151515141414020214141414141419191919191915151514141414020202021414191919191919191515151414140202020202141414141919191919151515141414020202020214141419191919191915151502020202021414141414141419191915151515151502020214021414141919191919191515151515020202140202141419191919191919151515151502141414141414141414191919191515151515151414141414141414141919191919151515151514141414141414191919191919191915151515141414141414141919191919141919191515151514141414141414191419191414141919151515141414141414141414141914141414191915151414141414141414141414141414141919191414141414141414141414141414191919141919');
INSERT INTO corine_land_cover (rid, rast) VALUES (85, '0100000100000000000000594000000000000059C000000000ECC6504100000000D864434100000000000000000000000000000000DB0B000014001400440019191919191919191919191919191919191920201919191919191919191919191919191919192020191919191919191919191919191919192020202019171717171717171719171717171920202020201717171717171717171717171720202020202019191919191917171717171717172020202020191919191919191717171717171717202020202019191919191917171717171717171720202020171919191919191717171717171717172020171718191919191717171717171717171717171717181818181917171919191917171717171717181818181818191919191919191717171717171818181818181819191919191919171717171818181818181818181919191919191918181818181818181818181818191919191919191818181818181818181818181819191919191918181818181818181818181818181919191919191918181818181818181818181818191919191919191918181818181818181818181819191919191919191918181818181818181818181919191818181818191918181818181818181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (86, '0100000100000000000000594000000000000059C000000000E0C8504100000000D864434100000000000000000000000000000000DB0B0000140014004400202020201A1A1A1A1A1A202020201D1D1D1D1D2020202020201A1A1A1A1A1A1A2020201D1D1D1D202019201A1A1A1A1A1A1A1A1A1A1A1D1D1D1D1D1D1919191A1A1A1A1A1A1A1A1A1A1A1A1A1D1D1D1D1919191A1A1A1A1A1A1A1A1A1A1A1A1A1A20201D19191919191A1A1A1A1A1A1A201A1A1A2020201719191919191A1A1A1A1A1A20202020202020171719191919191A1A1A1A1A1A202020202020171717191919191A1A1A202020202020171717171717171818181919192020202020201717171717171717181818191920202020202017171717171717171718181819192020202017171717171717171717171818181920202020171717171717171717171717181819192020201717171717171717171717171719191919202020171717171717171717171717171919191D1D201D171717171717171717171717171818191D1D1D1717171717171717171717171717181D1D1D1D1717171717171717171717171717191D1D1D1D171717171717171717171717171717171D1D1D1D17171717171717171919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (87, '0100000100000000000000594000000000000059C000000000D4CA504100000000D864434100000000000000000000000000000000DB0B0000140014004400202019191917191917171717171717171715171720191919191717171717171717171715151517171D191919191717171717171717151515151517171D191917171717171717171715151515151515151D17171717171717171717171515151515151515171717171717171717171717151515151515151517171717171717171717171715151515151515151717171717171717171715151515151515151515171717171717171717171515151515151515151517171717171717171915151515151515151515151717171717171719191519151515151515191915171717171719191915191915151519191919191917171919191919191519191515191919191919181717171919191919151519151919191919191918171717191919191915191915191915151919181817171919191919191515151515181818191818191919191919191919151515151818181918181819191919191919191915151518181818181919191919191919191919181515181818181819191919191919191919191918151818181818181D1D191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (88, '0100000100000000000000594000000000000059C000000000C8CC504100000000D864434100000000000000000000000000000000DB0B000014001400440017151515171717171717171717171717171717141515151517171717171717171717171717170214151515151517171717171717171717171702021415151515171717171717171717171717170202021515151517171717171517171717171717020202151515151717171717151717171717170202020215151515151717171515171717171717020202021515151515151717151717171717170202020202151515151515151515171717171702020202020215151515151919191517171717170202020202021515181818191919191717171717020202020202181818181919191919191717171717020202020218181818191919191917171717171702020202141818181919191919191717171717171010020214191919191919191917171717171717101010021419191919191919191717171717171710101002021919191919191917171717171717171010100202191919191919171717171717171717101010020219191919191717171717171717171717101010101917191917171717171717171717171717101010');
INSERT INTO corine_land_cover (rid, rast) VALUES (89, '0100000100000000000000594000000000000059C000000000BCCE504100000000D864434100000000000000000000000000000000DB0B000014001400440014141414141414101014141414141414141414141414141414141414141414141414141414141419141414141414141414141414141414141419191914141414141414141414141414141414141919191414141414141414141414141414141414191919141414141414141414141414141414141919191902141414141414141414141414141414191919190214141414141414141414141414141419191919020214141414141414141414141414141919191902021414141414141414141414141419191919190202141414141414141414141414141919191919140214141414141414141414141414191919191914141414141414141414141414141419191919191414141414141414141414141414141919191919141414141414141414141414141414191919191914141414141414141414141414141419191919190214141414141414141414141414141919191919101010141414141414141414141414191919191910101010141414141414141414141419191919191010101010101414141414141414191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (90, '0100000100000000000000594000000000000059C000000000B0D0504100000000D864434100000000000000000000000000000000DB0B00001400140044001919191717171717171919191917171919191717191917171717171919191919191717191919171719171717191919191919191919171719191717171919191919191D1D1919191919171719191917171919191919191D1D1919191917171719191717171919191919191D1D1D1919171717171919171717191919191919191D1D1917171917171917171717191919191919191D1D1D1919171717171717171719191919191D1D1D1D1919171919171717171717191919191D1D1D1D1D19191919191717171717171919191D1D1D1D1D1D191919191917171717171719191D1D1D1D1D1D1D1919191919171717171718191D1D1D1D1D1D1D1D1919191919191717171818191D1D1D1D1D1D1D1D1D19191919191717171818191D1D1D1D1D1D1D1D1D191919191919171918181D1D1D1D1D1D1919191D191919191919191818181D1D1D1D1D1D1919191919191919191918181818191D1D1919191919191919191919191918181818191D1D19191919191919191919191919181818181D1D1D1D1D1D1919191919191919191919181818');
INSERT INTO corine_land_cover (rid, rast) VALUES (91, '0100000100000000000000594000000000000059C00000000010C1504100000000F060434100000000000000000000000000000000DB0B000014001400440018141414141414191914141414141717171717151414141414141414141414141417171717171717141414141414141414141414171717171717171714020202141414141414141417171717171717191402020214141414141414141717171717191919020202021414141414141417171717171919191902020214141414141414171717171719191919190202141414141414141717171717191919191919021414141414141414171717171719191919191914141414141414141717171717191919191919191414141414141417171717171919191919191919141414141414171717171719191919191919191914141414141717171717171919191919191919191414141417171717171719191919191919191919141414171717171717191919191919191919191914141717171717171919191919191919191919191417171717171717191919191919191919191919171717171717171719191919191919191919191917171717171717191919191919191919191919191717171717171919191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (92, '0100000100000000000000594000000000000059C00000000004C3504100000000F060434100000000000000000000000000000000DB0B000014001400440017151515151515151515151515151D1D15151515171515151515151515151515151515151D1D1514151515151515151515151515151515151D14141419191919151915151515151515141414141414141919191919191915151515151414141414141414191919191519151515151514141414141414141419191919151515151515141414141914141414141919191915151515151919191919191414141414191919191915151519191919191919141414141419191919191915151919191919191914141414141919191919191919191919191919191414141919191919191919191919171719191919191919191919191919191919191917171919191919191919191919191919191919191717191919191919191919191919191919191919171719191919191919191919191919191919191917171919191919191919191919191919191919191717191919191919191919191919191919191919171719191919191919191919191919191919191917171919191919191919191919191919191919171717191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (93, '0100000100000000000000594000000000000059C000000000F8C4504100000000F060434100000000000000000000000000000000DB0B000014001400440014141414141414141414141414141414141419191414141414141414141414191919141919191419141414141414141414141919191919191919141414141414141414141414191919191919191914141414141414141414141919191919191919191919141414141414141419191919191919191919191914141914141414191919191919191919191919191919191914141419191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191919191717171919191919191919191919191919191919171717171719191919191919191919191919191717171717171919191919191919191919191919171717171717191919191919191919191919191917171717171719191919191919191919191919171717171717171919191919191919191919191717171717171719191919191919191919191919191917171717171919191919191919191919191919171717171717171919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (94, '0100000100000000000000594000000000000059C000000000ECC6504100000000F060434100000000000000000000000000000000DB0B0000140014004400191919191818181818181818181818181818181D1919191918181818181818181818181818181D1D19191919191919191818181818181818181D1D1719191919191919191918181818181818181D1717191919191919191919191818181818181D1D17171919191919191919191919181818181D1717171719191919191919191919191919181D1D17171717191919191919191919191919191D1D1D17171717191919191919191919191919191D1D1D171717171919191919191919191919191D1D1717171717171919191919191919191919191D1D17171717171719191919191919191919191D1D1D17171717171719191919191919191919191D1D1D171717171718191919191919191919191D1D1D1D1D1717181818191919191919191919191D1D1D1D1D1717181818191918181818181919191D1D1D1D1D17171818181919181818181818191D1D1D1D1D1D17171818181919181818181818181D1D1D1D1D1D171718181819181818181818181D1D1D1D1D1D1717171718181918181818181D1D1D1D1D1D1D1D171718181815');
INSERT INTO corine_land_cover (rid, rast) VALUES (95, '0100000100000000000000594000000000000059C000000000E0C8504100000000F060434100000000000000000000000000000000DB0B00001400140044001D171D1D1717171717171719191919181919191917171D1717171717171919191819181818191919171717171717171719191918181818181819191917171717171717171919181818181515151515151717171717171717181818181818151515151515171717171717171718181818181815151515151517171717171717181815181815151515151515151717171717171718151515151515151515151515171717171717181815151515151515151515181817171717171818181515151515151515151818181717171718181818151515151515151818181818181717181818151515151515151518181818181818181818181815151515151818181818181818181515151815151515151515181818181818181818181515151518181515151818181818181819191918181818181818181518181818181819191919191818181818181815151818181819191919191919181818181818151518191819191919191919191915151515151515181919191919191919191919191515151515181819191919191919191919191919');
INSERT INTO corine_land_cover (rid, rast) VALUES (96, '0100000100000000000000594000000000000059C000000000D4CA504100000000F060434100000000000000000000000000000000DB0B000014001400440015191919191918181818181818181D1D1D1D1919151919191918181818181818181D1D1D1D1D18171515191918181818181818181D1D1D1D1D18181715151519181818181818181D1D1D1D1818181717151518181818181818181D1D1D1D1818181717171518181818181818181D1D1D18181818171717171518181818181D1D1D1D1D18181818181718171718181818181D1D1D1D18181818181818181817171818181D1D1D1D1D18181818181818181817171718181D1D1D1D1D18181818181819191818171717181D1D1D1D1D1D181818181919181818171717191D1D181818181818181819191818171717171717181818181818181818191919191717171717171718181818181818191919191917171717171719191818181818191919191917171717171717171919191918191919191919191919191717171719191919191919191919191919191919171717171919191919191919191919191917171717171717191919191919191919191919171717171717171919191D1919191919191919171717171717171719191D1D');
INSERT INTO corine_land_cover (rid, rast) VALUES (97, '0100000100000000000000594000000000000059C000000000C8CC504100000000F060434100000000000000000000000000000000DB0B0000140014004400171719171717171717171717171717171710101017191917171717171717171717171717171010101919171717171717171717171719191910101010191717171717171717171717171919191910101017171717171717171717171719191919191010101717171717171717171717171919191919191010171717171717191919171919191919191919101017171717171919191919191919191919191910101717171919191919191919191919191919191010191919191919191919191919191919191D1D1010191919191919191919191919191919191D1D1010191919191919191919191919191919191D1D1010171919191919191919191919191919191D1D1D10191919191D19191919191919191D19191D1D1D10191919191D1D1D1D1D1D1D1D1D1D191D1D1D1D101919191D1D1D1D1D1D1D1D1D1D1D1D1D1D1D1D101919191D1D1D1D1D1D1D1D1D1D1D171D1D1D1D1019191D1D1D1D1D1D1D1D1D1D1D1717171D1D1D101D1D1D1D1D1D1D1D1D1D1D1D1D1717171D1D1D101D1D1D1D1D1D1D1D1D1D171D1D171717171D1D10');
INSERT INTO corine_land_cover (rid, rast) VALUES (98, '0100000100000000000000594000000000000059C000000000BCCE504100000000F060434100000000000000000000000000000000DB0B0000140014004400101010101010101414141414141419191919191910101010101010101014141414191919191919191010101010101010101414141419191919191919101010101010101010101414141919191919191910101010101010101010141414191919191919191010101010101010101014141419191919191917101010101010101010101414191919191919191710101010101010101010141419191919191917171010101010101010101010141919191717171717101010101010101010101014191917171717171710101010101010101010101419191717171717171010101010101010101010141917171717171719101010101010101010101419191717171719191910101010101010101010100317171717191919191010101010101010101003031719191919191919101010101010101010100303191919191919191910101010101010101003030303191919191919191010101010101010100303030319191919191D1D101010101010101003030303030319191D1D1D1D10101010101010100203030303031919191D1D1D');
INSERT INTO corine_land_cover (rid, rast) VALUES (99, '0100000100000000000000594000000000000059C000000000B0D0504100000000F060434100000000000000000000000000000000DB0B000014001400440019191D1D19191919191919191919191919181818191D1D1D1D1D19191D1D1D191919191918181818191D1D1D1D1D19191D1D19191919191919181818191D1D1D1D1D1D1D1D1919191919191919181818171D1D1D1D1D1D19191919191919191919181818171D1D1D1D1D1D19191919191919191919181818171D1D1D1D191919191919191919191918181718171919191919191919191919191919191818171817191919191919191919191919191919181717181919191919191919191919191919191917171717191919191919191919191919191917171717171719191919191919191919191919191717171717171919191919191919191919191919171717171717191919191919191919191919191919171717171719191919191919191919191919191717171717171919191D1D1D1D1D1919191919191717191717171919191D1D1D1D1D1919191919171719191919171D19191D1D1D1D1D1D1D191717171919191919191D1D1D1D1D1D1D1D1D17171717191919191919191D1D1D1D1D1D1D1D1D1717171717191919191918');


--
-- TOC entry 3353 (class 0 OID 109346)
-- Dependencies: 213
-- Data for Name: corine_land_cover_legend; Type: TABLE DATA; Schema: env_data; Owner: -
--

INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (1, '111', 'Artificial surfaces', 'Urban fabric', 'Continuous urban fabric');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (2, '112', 'Artificial surfaces', 'Urban fabric', 'Discontinuous urban fabric');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (3, '121', 'Artificial surfaces', 'Industrial, commercial and transport units', 'Industrial or commercial units');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (4, '122', 'Artificial surfaces', 'Industrial, commercial and transport units', 'Road and rail networks and associated land');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (5, '123', 'Artificial surfaces', 'Industrial, commercial and transport units', 'Port areas');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (6, '124', 'Artificial surfaces', 'Industrial, commercial and transport units', 'Airports');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (7, '131', 'Artificial surfaces', 'Mine, dump and construction sites', 'Mineral extraction sites');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (8, '132', 'Artificial surfaces', 'Mine, dump and construction sites', 'Dump sites');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (9, '133', 'Artificial surfaces', 'Mine, dump and construction sites', 'Construction sites');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (10, '141', 'Artificial surfaces', 'Artificial, non-agricultural vegetated areas', 'Green urban areas');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (11, '142', 'Artificial surfaces', 'Artificial, non-agricultural vegetated areas', 'Sport and leisure facilities');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (12, '211', 'Agricultural areas', 'Arable land', 'Non-irrigated arable land');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (13, '212', 'Agricultural areas', 'Arable land', 'Permanently irrigated land');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (14, '213', 'Agricultural areas', 'Arable land', 'Rice fields');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (15, '221', 'Agricultural areas', 'Permanent crops', 'Vineyards');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (16, '222', 'Agricultural areas', 'Permanent crops', 'Fruit trees and berry plantations');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (17, '223', 'Agricultural areas', 'Permanent crops', 'Olive groves');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (18, '231', 'Agricultural areas', 'Pastures', 'Pastures');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (19, '241', 'Agricultural areas', 'Heterogeneous agricultural areas', 'Annual crops associated with permanent crops');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (20, '242', 'Agricultural areas', 'Heterogeneous agricultural areas', 'Complex cultivation patterns');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (21, '243', 'Agricultural areas', 'Heterogeneous agricultural areas', 'Land principally occupied by agriculture, with significant areas of natural vegetation');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (22, '244', 'Agricultural areas', 'Heterogeneous agricultural areas', 'Agro-forestry areas');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (23, '311', 'Forest and semi natural areas', 'Forests', 'Broad-leaved forest');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (24, '312', 'Forest and semi natural areas', 'Forests', 'Coniferous forest');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (25, '313', 'Forest and semi natural areas', 'Forests', 'Mixed forest');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (26, '321', 'Forest and semi natural areas', 'Scrub and/or herbaceous vegetation associations', 'Natural grasslands');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (27, '322', 'Forest and semi natural areas', 'Scrub and/or herbaceous vegetation associations', 'Moors and heathland');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (28, '323', 'Forest and semi natural areas', 'Scrub and/or herbaceous vegetation associations', 'Sclerophyllous vegetation');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (29, '324', 'Forest and semi natural areas', 'Scrub and/or herbaceous vegetation associations', 'Transitional woodland-shrub');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (30, '331', 'Forest and semi natural areas', 'Open spaces with little or no vegetation', 'Beaches, dunes, sands');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (31, '332', 'Forest and semi natural areas', 'Open spaces with little or no vegetation', 'Bare rocks');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (32, '333', 'Forest and semi natural areas', 'Open spaces with little or no vegetation', 'Sparsely vegetated areas');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (33, '334', 'Forest and semi natural areas', 'Open spaces with little or no vegetation', 'Burnt areas');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (34, '335', 'Forest and semi natural areas', 'Open spaces with little or no vegetation', 'Glaciers and perpetual snow');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (35, '411', 'Wetlands', 'Inland wetlands', 'Inland marshes');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (36, '412', 'Wetlands', 'Inland wetlands', 'Peat bogs');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (37, '421', 'Wetlands', 'Maritime wetlands', 'Salt marshes');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (38, '422', 'Wetlands', 'Maritime wetlands', 'Salines');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (39, '423', 'Wetlands', 'Maritime wetlands', 'Intertidal flats');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (40, '511', 'Water bodies', 'Inland waters', 'Water courses');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (41, '512', 'Water bodies', 'Inland waters', 'Water bodies');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (42, '521', 'Water bodies', 'Marine waters', 'Coastal lagoons');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (43, '522', 'Water bodies', 'Marine waters', 'Estuaries');
INSERT INTO corine_land_cover_legend (grid_code, clc_l3_code, label1, label2, label3) VALUES (44, '523', 'Water bodies', 'Marine waters', 'Sea and ocean');


--
-- TOC entry 3436 (class 0 OID 0)
-- Dependencies: 211
-- Name: corine_land_cover_rid_seq; Type: SEQUENCE SET; Schema: env_data; Owner: -
--

SELECT pg_catalog.setval('corine_land_cover_rid_seq', 99, true);


--
-- TOC entry 3342 (class 0 OID 109246)
-- Dependencies: 202
-- Data for Name: meteo_stations; Type: TABLE DATA; Schema: env_data; Owner: -
--

INSERT INTO meteo_stations (gid, station_id, geom) VALUES (1, 1, '0101000020E6100000468CA9D042F625400FBD49F714FE4640');
INSERT INTO meteo_stations (gid, station_id, geom) VALUES (2, 2, '0101000020E6100000A642426721E8254099B385B72F024740');
INSERT INTO meteo_stations (gid, station_id, geom) VALUES (3, 3, '0101000020E6100000C66D21DC95172640DB3937D6370B4740');
INSERT INTO meteo_stations (gid, station_id, geom) VALUES (4, 4, '0101000020E6100000E80A40B3B5402640899E9CAECB024740');
INSERT INTO meteo_stations (gid, station_id, geom) VALUES (5, 5, '0101000020E6100000EE3AAB2D771C2640C133EA86C2014740');
INSERT INTO meteo_stations (gid, station_id, geom) VALUES (6, 6, '0101000020E6100000C89ACC2F7B3226408604A809BB134740');


--
-- TOC entry 3437 (class 0 OID 0)
-- Dependencies: 201
-- Name: meteo_stations_gid_seq; Type: SEQUENCE SET; Schema: env_data; Owner: -
--

SELECT pg_catalog.setval('meteo_stations_gid_seq', 6, true);


--
-- TOC entry 3355 (class 0 OID 109357)
-- Dependencies: 215
-- Data for Name: ndvi_modis; Type: TABLE DATA; Schema: env_data; Owner: -
--

INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (1, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540989537D1F812474000000000000000000000000000000000E6100000140014004548F45C015C015C015C0119009FFF9FFF0F000C005D005D005D005D005D005D0085FF85FF54005400540086007E004E003A003A003A00C9FFC9FF0F00B7FFB7FF28FF5D005D006DFF32FF32FF32FF2AFF54005D005D0053004A004A0039003C003C00370080FF6AFF6AFFB8FFBAFFBAFF86FF20FF20FF1FFF42FF5000500050005E005E005E006B005700570051004E004E0061004E004E0069009CFFE1FFE1FFFEFF5900590059006900780078007900930093006B003900080008001F003F003F00DDFFA4FFA4FFE1FF7F00590059000000850085007E007E00640064003900CCFFCCFFD8FF17001700D2FFCCFFCCFFFBFF160116019AFF9AFF1600160016009500C200C2008700580058006C0055002AFF2AFF0DFFCBFFCBFF2E012E015D005D0076008A008A001600C200C200C200D500700170013001D800D800D8008D018D01C600EC00D100760076007600760076006E02090109010101B800B800C10075007500750058014401790079002F013D003D001E011E011E016E02920092001001E2002E012E011BFFF3FFF3FF46FF2800E700E7003E01220022005C00E103E103E1036E02AF01AF013B02E700E70033FF66FE66FE3DFFA8FEE700B400B4005B0105010501E103E103E103E103320032000104510002000200BDFF9FFF9FFF5BFFCD00BE00BE00AB04AB04AB04CF086903AE03AE031C0567086708DA00BDFFBDFF7400410041002A00A2020302030298012709E503E503C504220A220A1B0BAA0AAA0AEF039C002B002B00B702EC00EC007C00B300C6FFC6FFBF03270927099003380D380DF70EF00F2D062D06BC00D701D701A9019E019E01D200B3008F018F01EF007102C900C90021048E0C8E0CFA0C2D062D06CB059D059D050802E00189000EFE0EFE7CFE56025602D40258015801EB04920892087F0D5010BC0ABC0A8303DA02DA0226FFDAFF9600960098FFF501F501E3018001EB04EB04810832123212E310CC11CC112005C302C30257005100EB00EB005B00020002000200C203AA06AA06AB0881088108821282128212E60C6701AB00AB008300CA01470147016300F9FFF9FF57037C047C04CF05CE08680F680F680FE60CE60C60014A004A00FC01', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (2, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540989537D1F812474000000000000000000000000000000000E6100000140014004548F454009E00080108014401490149013401740174010101BA01CA02CA02CB01DE06DE067D1328102810540054002F002F00C800C500C5001A020A0266026602D6038E028E02E907B308B3087D13C40CC40C42FFB4FFEAFFF900F90088001A021A021202550255022704CA05CA05F6041F0C7F157F15BB0AB404EAFFEAFFEBFF0D010D01C3009E029E02C5021E03360736072D068A098A091A0C1A0C1A0C92090309C200C200C200AF00E200E2009E029E029E022107D00BD00B63073E0A3E0A4F10701094109410A713C200C200C2004701EF00EF00C200210721072107550E550E2C0F0C130C130C13A610CA0FCA0FD0121A0127012701A9011A011A011A01A909A909A909C40D251725170C13F213F2134F0F8F128F12BF128D01FB01DC01DC0126012601260184FFA909A90925172517251725172517251725173613E71FE71F4401FB01FC00FC0055010A010A01FE014802490249024302FD06FD062517921F921FCB130722072228009BFF8D002A002A00F1011A021A023C0235023502CD01B604B6044D045E0FE70DE70DAF19ED1A37003700BAFFECFFECFFB5FFF500F5003502AE02AE02B0024104D908D9080B0C5E0F5E0FAF19941BCDFFCDFF9701F700F700CBFFF5001E001E00CC03CC03CC03D908D908D908D90861146114611461146B00C700C700560126012601DFFFF300F300D200CC03CC03340F340F340F340FCD12611461146114F300F400F400BD009800980084019D009D009D001101890D890D890D890D890D2316CD12CD12140F730086FF86FF0F00E8002E012E01F600DD00DD0064023E043E043E04890D2A132A132A13770D770D89009200DDFFDDFFE900940194011C02E401E40133033E04DE04DE045D0B560F560FCA10170D170DDAFF69010D010D01E200ECFFB700B7005D013A033A03500391059105C909750E750E0E12CE0FA10C99FF99FF4D00FB00FB004900BA00BA003901500350039606C106CE13CE130010CD10CD10140FCF19D9FFD9FF4D009B009B004802C400210121018605110811081209CE13CE13CE13CE13CE13CE13CF19FC01FC0100019E00370237023702ED00ED000604340534056E0CD519D519D519CE13CE13CE13831A', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (3, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540989537D1F812474000000000000000000000000000000000E6100000140014004548F42E0DE811E8111519D418221A221AAE1A911691160C16FB11FB11531A531A531A531A8525A91CA91C5D09010C921892182614211621160114551155110010E510E5101512AC16531A531A402134233423B4047F095A0F5A0F2614720F720FE50E530D130D130DCD1176237623702593269326EE156C1A6C1A0309E308DE0CA513A5131313361036109C0E060D060D7617762376238C242E22291829186C1A621C11161116231AA116A1163D1891119111A40ECF1157125712501E86268626E72528252825EF240D2170167016EE15231A981B981BE80ECA09CA09EC11A911A9117416FD1EFD1EBF1F6E246F1F6F1F651E5216EE15EE15EE152E142E14CA09CA09CA09BA18BA18BA18581A2D242F242F24E31DBF19BF191D18521699149914320A320A320A950595053E173E17BA18581A581ABB1C2F242F242821532053206E23E515BB13D915D9153615C414C4140B2268166816F41668166816142575216D206D20AB20E91FE91FED1A1116541754178C158C158C15E7065925501E501E6816142514251425DD24DD24DD24A61FA61F941B3D13B713B7136C101F111C161C165925AE12AE124C1DB223B223B2237625A91BA91BA91BA91B311031106912350F350F1F0E610C610C7D1274127412D1148620652265229523C019C019B01D2117DC0DDC0DE40E940E940E800BA10B6A186A181C0DD114D114CB243E1A3E1A3E1A851285122B11A00E470F1E0D1E0D8B0CF409F409650CF312F3126A18CB24CB24CB24881EF21AF21A870F910F910FEE0B020CED09ED09392139213921C713F312F312F3126D241F1E1F1E2A15CB24CB24F60BF20EF20E2509E50C3E0B3E0B120BB8066C0A6C0A470FCE1FCE1FBF231F1E1F1E70177017AA0BAA0BF60B78077807A10C640B240B240BF411F411F4115D14CE1FCE1FCE1FCE1FCE1FCE1F980B980B980B980BC804C804CF199F0DDC13DC13DC13C71B611661168C13CE1FCE1FCE1FCE1FCE1FBC09620B620BAD015903E201D311D311B315891889180818F61CF61CAC258C138C138C138C139C089C081B050F020F023A01B4014C184C18D718B81AB81A0818C11A0E240E24AC258C138C133B08BC07BC07F004CF02CF02E4009B01', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (4, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540989537D1F812474000000000000000000000000000000000E6100000140014004548F483191F1B1F1BDB1FED20ED20D41F65246524A2230C1EBB1DBB1D2617D810D810611246084608A605361931263126EF265F1D6E226E22CB19F816F8163120132013201320F20FF20FDF073605F902F9025F15CD1C5A1F5A1FF5216E226E22FD1D891B891B1718AB14AB14F116741393049304CF02B502B502621CFB23312431243E23571E571E571E511E6F226F22B4179F0F9F0F260B6403640333020A020A020D2105218A243E233E23A01E2B212B215B1A6F226F226F2253135313C1094F04470147014B01310242184218FD15F61BF61B6617B215B215CD1BFF0C0B0D0B0D480BDB05DB05E404EB01EB018001AD01E611E6119A149A14480F480FDF0D540954094506CC06CC06DE076F036F036D042702F001F001CD02201B171217120910B70DB70DF5069A079A0797021F041F043903DF01E302E30221053C043C04F304C4192A112A11B30710071007C50359046E026E028E02280228021E0140014001210521052105090DC4191D17370B370B6B052F032F030302860186011C01FE00FE009C009800F805F805E80BE80BE80BA91BB0141904190433040C030C03F8013301C200C200E3005E005E00AE00F205F205020F081108112117B014F702F702CC06A1019E009E00DA00680068008C00DDFFDDFF91008D087E0D7E0DB00DB00D9A089A085904F903F903E1009C009C007A00520052006400D4FFF400F400AF015B0E5B0E5B0E5B0E3B033B0306028B008B007B007E008600860075001C011C012500C700C700ED013D023D020F035B0E8B05680268022601BE00BE0061007C007C00F0001C011C015700CB00C101C1016A01930393036D09780712051205B8018A018A01A2006800700170014D014D014D018503850385038503D403D4036D091C034B044B045904690219021902E100200120011F041F041F0446094609340634062406AC0EAC0EE20119022502250286027502750219027D027D0272021F04AE04AE044609EB0DEB0DEB0DA211A211B401CA010A020A026002530249024902B40650025002F701780F780F780F780F780FEB0D98147C12210221023702450445049204760276029407B406B406B4069514780F780F780F07130713D914BE12', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (5, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F4A605A605F0023501E800E800E8000A050A050A0578057805F90CF90CF90C6E139613961396134313BF02770277024D01F200F2005C011C011C0118010205E509E509E5097A107A10BE11B710B710B710B902B901B9011101F200F200A2003101940194010D04EA0CEA0CE509F612F61285115E135E135E131A033C02C001C001A901950195016304630463041509B80EB80E2710F612811281120914FE12FE12310287027F027F028A025902590281048104C50FC50FB80E660E660E5C11631063105911D314D314AD017F029A0232033203CF04CF04CF04AA037D047D04BC11A313A313A313F813401340139C12BC16E402E402A803CA05CA05870ECF04CF04B10B7B08160C160C1016101610161C14401340135514C3109C049C04DF09DF09DF09870E870EF70AF70AF70A510D510D0814081408143B131C142D152D152913B311B60BB60BDF095912591259121E0E1E0EBB0C260F260F7E147E143B133B133113551C551C401D4011E213E213E213DE15DE15DE1505106E0C6E0CBE0F651265128B13121212122419901A901AB6190811571457145714DE15DE15DE15DE15BA14BA14BA144A114A118B13AF1239143914F2190E1A0E1AB00D2B1357145714B513B513B513B513D310BA14BA144A11771677160718A420A4206B1FBB1EBB1E5B0E2B132B132B13F512F512F512F512D310D310D3105D0FA20FA20F07180421C71FC71FCB1E97232A152A152A15F412F412F412F512F5120C120C120C12F314C9125B1F5B1F2B1F2B1E2B1E331DEE1E2A0C2A0C2A0CF412F412F412F412491149114911F314F3141216691B691BF91BB01CB01CDE1E7E246D096A0C6A0C380E631863187C117C117C114911F014F01431174922BD1EBD1EA019A019A019CD17AC0E1E0B1E0B380C760D760DAF127C117C117C113A17F31DF31D2C20751C751CB217B217B2177B121F11771077107710FD0CAF12AF12AF12AF12AF12E315451E451E2C20A516A713A713A713451245127C12C110C110C11010106A116A116A116A116A1172137B17E81AE81A4F1391129112841384138413BE1200110510051005103316331633169F159F159F157B17201620167A1246124612C510C5104E13', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (6, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640989537D1F812474000000000000000000000000000000000E6100000140014004548F4AA14AA14EB138110811046102A122A122A128F101D0D1D0DAE0AE107E1078B0C88138813C8148B12C212C2126C10BF0E53115311061155115511800DF406F406350C350C350C3C10E614181218129912AA10390E390E4B0C5B105B101911D410D410D90B020DC812C812C812BC11BC11701170117011730CC80E0B0F0B0F16113E143E143E14D90DAE0DAE0D6C1193119311A312A312A3126D0F2B0DF70AF70AD3148B0E2D0E2D0E6513DC12DC12EE0F400E400E0E0F210F210F971097100B0F0B0F980CE008E008BC16BB13630E630E450EDC11DC11F40FAD0E0E0F0E0F210F090B090BC809AF0FAF0FBF11160B160BC310B11060169A0F9A0F5513C410C4106209B708B7081306E305E305C708C90EBF11BF11390D5A0CC312C312101423182318B511700E700E3A08C4099F0A9F0A670C040A040AF90EF90EF90EB20F660CCB18CB189D12231823189210FF0D750C750C2B09080C080CF50BB30DB30DF90EF90ED210D210E6082616CB18CB18B314851485141B10980998099809AB09AB0900107312310F310F310F4D0C4D0CF606361D3B193B193B1953155315760E3C0E3C0E3C0E050DDA0EDA0E8611781078107810CC09CC099E07591CA61B06180618B715F40FF40FF70B3F0C3F0CF90B400C400C5F0EA60E720A720AB70A9C0B9C0B9723191D501950199514310F310F500BF50EF50EF50EAB0B3B073B07FE031C071C07F90BF60FF60FEE1EA21C9B19D019D019310F8F0C8F0C540CC50CC50CAB0BFD08FD086A061709C80CC80C560F4A13A21CA21CC518D019D019740FFC09FC09540CA60BA60B3207AC04790879081E0E690D690DFE0DD10F49114911F410D818D8183212E907E907E907EA088F068F068F0679087908220C2A0F2A0F6E1129131E11871187111512A411A411640FC508C508EA088F068F061D09470AAD11AD118C100C0F0C0FBC118712AD11AD111512900F900FD209D209E007E007200A63096309B610AD11AD11DC12A70DA70D6A094E13AD116B126B12E2116D0C6D0C6D0CE007E0078507C806C806F30B6A1364156415960EFA07FA074E1311145C145C142914D60ED60ED110C809C809CA0A9007F307F307840E840E840E700C7E097E09', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (7, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640989537D1F812474000000000000000000000000000000000E6100000140014004548F48B12C6101211CA11CA1101124D144D14DB0E6A0E6A0ED20E68106810CE0FCB12DB0CDB0C7D159922F210F210E812E812E812D112E214E214B513701389118911A81107110711561C0119011992173A1C4A0D4A0D40120C110C110C114F12B513B513C4124E164E16511A421842184F1D8A1BF41AF41A021B250D0A110A117E12E612E612E612671267125114B415FB17FB17B61A681B681B6712F717F7175E1D680AE60AE60AB70FC511C511E6120C141D121D125612E712E7121416580E580E9F0EF717F717561E380AF40E7D0D7D0D2712B110B110B110DE10DE1059102D102D105B11A70E270F270FD8123B233B235A0C0511051105112712911291125511551139123912CA0E1E111E114D0FCF0CCF0C8417511A511A660C230A1214121412145E10101110111B12E80CE80CE80CE610E61022165C17F514F5144218CA1BEE0CEE0C3E133E133E134411ED10ED101F1041111C101C10101359165916DC20CB24CB243D24BF1CEE0CEE0C250F9E0F9E0F681177132B0D2B0DD410E312E312E119621A621A621A7E1B761B761B2C1A010EC30FC30F520E0D0F0D0FF311D310D310EE0D160E160ED019651E76237623F21E2B1D2B1D3B14C21050115011890E5D115D113E13B9134B124B12380DD616D61631200A210A214F1E911A911ADA22EE15EE15C213C213D4103E133E135F1431123112890F411641161F1E7821D11FD11FD11F631A631A4A132A133D123D124512FA11FA11A412CA12F911F91117168F168F168F165A1A5A1A171EAC1BAC1BD10FB511C1110A130A131B121B121B127114081308136B19541B541B1A19F815ED16ED16B216FD13571257125712BD0EBD0EA2111B121B12490D081308131819AA1AD619D619D6194E144E14481A481ACC13CC134C114C114C11E412CD11CD11CD115A15301430143212281B281B9012731673162F1D4B1B8B0F26132613261323172317E315E315E3152E16E314E3149110390DEB0EEB0E550D52165216FB268B0FF410F410691269126912F8153D233D233D233F17B215B21540134E104E105F0DBE10BE10B8134209FB0CC012C012C712F815F8153D233D233D2306205A1A5A1A92123013DE0BDE0B980ED710D710', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (8, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F4992299221A121A120C1FC80FC80F1414BD0BC80FC80F8B0AAD06AD060708480B480B9C0AA208A2083A1C3A1CA5144A124A129D1E831783178317D215D2150D0D70067006AE06C106CF0ACF0AD10715089C209C208D20FC1BFC1BE024D81AD81A61146114230E230EEA0C9E059E059506130A130A130A350A111911194317DD171D191D19E31AA518A518501950195019EA0CB906B906AD06AD06050B050B470A641954205420111F111F111FC016391C391C391CCE172E0D2E0DC80A9806980637064A084A086109091E54205420DA1D111F111F481C4815551E551E551E5E105E10D90FAA0FAA0F720A110A110A980C091E091EDA1DDA1D471BF219F219781450145014291484118411431110143B153B151B19C51CC51CCA1BE11DE11DE11D3C1E801980199D15D918D918D918BB162F142F14ED25D720D7203F1DB91EB91EBF1CB41A4920CB1ECB1E380F160E160E7E10D420D420D420D01CD01CF322631B881F881F2E21F126C013C013BC20D720D720350F5E0B5E0B28103A216E216E213021E622E622771D741674165216FD21661166117F1B41254125BD15881131143114922192219221EE2298169816771D3A1B9B1D9B1D5618DA22A51EA51EB11C2D1B2D1BAC11A610A61074181D1E1D1E9D1E7D1A37143714E719391139117A0E1F15E517E5171E17D118D1182A13AA12F811F811F512E21CE21C7D1AF91AF91AB71C850F850F0D0E3F16E615461846184618AA12AA12A814BE16BE16591CBE1DBE1D541DFA1311121112900F480E480EFD131D15151315134618A318A3186C14D812892089201B1FBE1DBE1D2C110B150B150B15BA0EBA0E481A2413B716930F930FEC12901490143413BF15BF152619480F480F480F0B15522352237615281E4B1B4B1B1417AE11AE11AD14341334133413DD0DBB12BB121910880D880D6510121C121CF41A9821EF1AEF1A791429102910E70D770D6A0D6A0D4A0DEF0DEF0DF80C250C250C230E27142714291376158618591759173E151B101B106111031B031B720ED30DD30D710CEF0BEF0BEF0B2913E61BE61B3C0C7D1859175917FC1511121112EA15A014F911F9112D0E280E280E000D9C0C9C0CE60BC40CC40CB40C', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (9, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640989537D1F812474000000000000000000000000000000000E6100000140014004548F49309830C830C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F415087D0FD70C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4350A900AD70C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F469096909470C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F461096109EE0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4500F9211921148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F488169116911648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C51CF523F52348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F126F126742048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FD21CD18F32048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EC1EEC1E912648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EC1EEC1E502048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B3142222222248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45B0DDE0FDE0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42B0C2C0C341348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4281E281EEB1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49821281E8F1248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4660E660EC00D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49A0C9A0CBA0C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A60C920D920D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (10, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F49F002302230241019902D401D4017303CB05CB0578094C0F4C0FA30BDE0CDE0CD20135021D011D01C300C101FC01FC01AD02FA02FA025E01E106E1069C0D1016150D150D650CEF09EF092609BE06BE067800AE0025012501E7005903590334015105FD0FFD0F3B173B173B176D12F50CF50C5A06BE06BE068B00E0002601CC00CC00C101C101C10143077E087E080604F211F211B30E27119D059D05A7024208CD00CD00A501B300B3009D029D029D02D108F009DD08DD081E0CE70FE70F0F0E55095509450A450A58025802DB03C102730373036602380A380AD411CB0BCB0B310D49124912510E4A0B7E0F7E0FA611DF00AA03AA036F0419061906D70443054305D4116B0A7E0D7E0D491249124912CA0F7E0F7E0F3B12E202B903B9037F050C040C044A055B0517111711150FB610B6100D140D140D148210301165186518A001EE02FD06FD061B071B071B078B0BF70DF70D8C13CF19CF199E1671143C133C133C13231823182D003704F605F6059B0D9B0D9B0D950FF70D8C138C13CF199E169E166B116B116B11AF13EA14EA144D0053005605560556059B0D950F950FE70D110F110FBC1075187518E71BE71B7D157D15EA144E19990099000C01A201A2011507700C700CDF106612E510E510B6131A1A1A1AE2177D157D157D1518171F001F009900E802E802BA036A0AFB12FB12271BE414E414B6131A1A1A1A6117CB13CB13D812D8120B01360036004FFF41014101C30236123612BF14E414E4144F1303193415341593133812381238120702DB02DB02310369FF69FFC7006204CB15CB15CB15CB15CB15551AAA18AA1893133812381203101406000810081008FB04350835089A050E0B0E0B490DE10FE10F3F197F17431943196412A511A5116C082309E209E2090409FE09FE09E90BD60BD60B920BB20B9E149E147F177F177F175E0FD110D110D4050E0AB60AB60A1F0C510E510E510E3510AF10AF1027122B152B15081AFD15FD15FD15DB167512FD09FD09100D420C420CB612FB11FB11FB11AF10AF10A61671152A162A162912291229122B135C13050D050D100D9A0E9A0E9115431365136513D318FB16FB16A6166B116B11B610B610B610B411FF13', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (11, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD254023E87981A30D474000000000000000000000000000000000E6100000140014004548F44401060106011A013702370237029800980063041D054C0B4C0BD519D519D519D519871987199116D304D403D40314020B040B040B04FB015E045E042705A30AA30A7D177D177D17D51987194C194C19BE0621056D046D046D040B040B040B048E048E04AB067D177D177D177D177D177D176718681B681B42086606FD04FD043D03FF02180618062B0660076007EC16EC16EC16EC167D177D176718E4182B1B450A6B03330833083308FE07D008D0088E08240B240BC119EC16EC16EC16EC1629192919FB193F1AFE11FE11B510661166116611DA14DA148115E10E33183318FF1899199919C1192919291959160317D918D918B317891B89168916CA17B819B81903190B180B18331808190819F018FA17BC16BC16C618761C5A185A187B1B891B891BEE1CDF19DF19031903194D1E4D1E4D1E251C251CEC1727192719F518491A451B451B2B1B2C1A2C1AA71DDF19DF19DF195F1B2A1C2A1C2A1C2E1D2E1D231C511AAD17AD17A0194819621A621A621A2C1B2C1BB119B119B119481CC71AC71AF71D2E1D131C131CCB1C351A351A4E19B01805190519041A9D199D190E162B14481C481C781878187818541F1E1D1E1D581C361B361B18171F141F14491349139B13A515A515B414A714A714A714F913F9139E1E541F621D621DAD1E811E6F116F116F11951095100A13F714F714F7147B15ED14ED14A0169E1E9E1E9E1E651D651DDD255A1FE60EE60ED90C301030107910681368136813B718691B691B4416171C171C171CE019DD25DD25DD25370F031003108C0B350F350F9314931493140717691B691BDB1817165C165C164E18F81EF81E431B011123102310F00D3E113E113E11CD1307170717D6158A218A217419EB18EB184E184D194D19F11875122310B310B310DD14DD14DD145A1617141714F41AB918B9189E1BDD16EB18EB182E173016301675124F1142114211C6140A160A160A1657165B135B13431371197119DD1695199519E31ACF15CF155C13EB12751375137B131A150A160A163B15881388139916611561153B19551B551B811981198C155B135B138613CC12CC12391258115811581146104610651061151D1A1D1AC51A631863183C118C15', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (12, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E3254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4C61AC61A2718FB18951B951BA51D2E1A2E1AAC25A215A215AD0538046E026E024501340134016E066E186E186E18C918341F341F591B2E1A2E1A6F0A8D074B064B064403F701F7012201400140016E06D819AB19AB19501D391ECA15CA153915AB08AB08F20774067406F2038D018D01D500CB00A602A6022B1B311B3B213B211E1DCF14CF14CF146F066F066905120532053205B5011B011B01FD00260126013F1A3F1ADB1DDB1D43230B10FC0EFC0EDD0E200520053903C002C0022F01340134018301FE0106020317371A9218021D021D82123E173E175910591059103D02E701F900F9009203BD03BD03F70116023A213A214820F619F619020E3E173E173E173E17C104C104EB01C503C5037A067A067A060D011503161D161D331BF6198D0C8D0C3E173E173E173E1758015801A2FFFB04FB04E8087A066E016E01490A0514331B331B431143114311FD16FD16FD16FD16DE01C201C2017606A20DA20DE808DE01DE01490A9716871887184E1C761A55055505FD16FD16FD165B02900590057F0BA20DA20DA20DDB03820582059720C618F116F116761A93069306220C2D042D04FD16FD16FD168905F8118D038D031D04ED08ED08811EEF1B3617361789178F068F06D90C44020716071607165A065A0698075E065E064F0D7F0C7F0C5A1F1C1AEC17DB17DB17450E450E450ECB0907160716D5096F0E6F0EEE0863054F0D4F0D230D0F0C8B1C8B1CEC17F718F7184A11450E450E030A3E13D411D411700F9F0F9F0F160C700B700B9A0ACA0D50155015F718F718F7181D121D122C112C119B14F210F210ED0FD910D910790D2F0D260D260DC515A816450F450FFC1326152615F5145A1B5A1B9B148D1A8D1A3E1A0D181F131F138C0DF40FF40FC515C716F514F5147114F117F117F21A5A1B5A1B5A1BFD1B64186418C0186B126B12DB12371537159C17F517F51422152215A91508170817411C46164616031A031A031A3E144816AC11AC11FE159C179C178C158212CE12CE12FF123C113C119815A618DE11DE11DE11FB11FB117E12B70FB70FAD0FD10ED10E8C150C13A913A913B91274121A131A133111AE11AE11BD12BD12BD127E12870FF50EF50ECC0DD00E', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (13, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4AA01AA01A6020903090345040207A113A113A113650A650A9514951495149514951495140013C31021022102C8033B048F048F043D07CC0DCC0DA113A113A113A11396139514951495147315731573153803EC06EC06EC06EC06EC066308CA10CA10CC0DD30DD30DD30D96139613961396135514551473153E0348084808C207AE089B0C9B0C30107C107C109211401240128213D713D71314135514551455140602480841084108080F7B0F7B0F8A15C715C71578129D15A612A612A6124E144E144E14E019E01916029F0322092209D10D460E08160816CD15BB15BB1534163913391339134E144E144E14771B771B15035102710AC211C2111113B715B7153516D116D116F216481539133913FF14FF14FF142116211692059205AA03D911D911ED1110121012D9161D16ED17ED17481548154815DA16DA16DA16C1162116490A490A490A490A6215621505131B171B1708182B192B194415441544157D177D17C116C116C116490A490A490AA90D461646164616461646166215D918BA18BA1844157D177D177D17731573157315410CDB0DDB0DA90D4616461646164616461646163B160B180B180B189D189D18F317731573157315EF0BDC0D4B104B104B10A90DA90DDF0D461646163B163B163B163B163B16BF16BF16BF16531553150F0C3D124B104B104B104B104B104B10530DBF15BF1567153B163B161D151D151D15D913E0125315CA0D7511BA11531253124B104B104B10AE0F06170617401F591659165916FB15F513F513E012E012AD11AD11E310F710F710E3108E128E12B813481B401F401F3F10241424140019001900197D11BE12C515C5154B14EC15E310E310EC1AEC1AEC1AEC1AEC1EEC1E3A1D3A1D3A1D0019351935193519CB189C174B144B144B144B144B14C41CFC1BFC1BEC1AEC1EEC1EEC1E1A231A231A23D61FD61FD61FCB189C179C179C17941894189418C41CC41C061C061C061CD21FD21F1A231A231A231A2305130513F51FF80E3E16941894189418C21DC21DC21DC21DC21D061C502150215021FA21422442243C13551B551BD00EDC14AC19AC19AC19AB15AB15C21DFE23FE23FE23C92050215021A91FC625C625C625C625C625', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (14, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4F912F912331277147714FB163316331643169F159F158C173015261426143310BA11BA117013D413731573158B1277147714C5158A1614141414311431143114E214EF0FEF0F5E0F25122512701370137315D813D813D813B512B512011414141414171317131713DB12A50DAE0DAE0D25127014701470145514E716E716E716BB12BB12A412AF0FAF0FC910E011E011E0117B11A70CA70CB313D313D31360142A16C61BC61BE716C513AA13AA132212B30FB30FBC10BC10BC105713020D020DA614CF14FD12FD12771B631CC61BC61B981663166316DE11A010A010B911C31157135713A1112D142D142612E212E21221165D18331933199A16B8168C128C1213137B137B137D12C30FC30FEC0E2C0C2C0CE7106F13AF137F187F18D118D215D215EC168C128C1213137B137B13B911DD0FDF13DF13510F15111511D310A712FD17FD176F18CE14CE14BE14FF11FF1143109910560F560F1011B714B714D412A00FA00F340F0E0E88158815F716961473137313DB124D104D106D0F7A107A1007129D119D114811850E850E850E7D105F14401440147F12501250128B12721172119911EE0F07120712AD12BA13BA133E123E123E12D313531500150015A412A811DD11DD116C0EA00FA00FEE0F401340136613BD14BD14DD12DD12871487145315381538153815F712F712F7127E0EE10FE10FF90E611579127912E8131A121A121A1287148714E0121113111311132C128611861186110C11300E300ED9133B133B1342124C0E4C0E1A12E2122915BE12C91137142C122C128711871187118712690F690F1015EA12EA120812060E060E060EAF0FF014F813F8133714371437140714071407148712DB13DB13DB13021302130213070E070E070E5F0F9E0FCB18CB186A186A18DE16DE16EA13EA13EA13B611B611B611901002130213310B650BEE0DEE0D8C0FF51FF51FF51F5B1ADE16DE162D12A110A110800FB611901090109010170B170BEB0B710C710CC60D551B9F1B9F1BF419F419F4198312040E800F800FD10DD10DD10D880C9D0A9D0A7E0C0D0C170E170EEC199F1B052205228B159A0D9A0D040E040E040E110F110F110FA60BEE07E70FE70F260F950D950D', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (15, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F00040723264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4D413B113251825182518F015A30FA30FA30F2C0F2C0F32071B061B06930675077507F00D4310DE09451445141B1A1B1A1B1AF015B813B81373100F0D0F0D9710971023072307E307F70EF70E4310BD0A62186218051B1A191A191A193D1250105010501097109710A112A112A112E40DAC0DAC0D4910820D60142F172F17721A721A721AFA14F910F91092105C0F5C0F9E13ED10ED10ED10AC0D49104910250AFD122F172F1748184818481849173213F111F1110C1010121012D210ED10ED107C1103120312DB10E2122B152B153E15B516231923191416141614169612751275120813BF12BF12BF128A1012091209AF132B15041604164B1823192319501A501A501AE014E20E081308136814AC0DAC0DF406E305E305A712AB13311631161F178B16041B041B8A188A188A18140F091109117412A513A513090AE3057007200F200F73126E186E18F518411B411B76197619761980103311DD12DD1260126012601223079B06E90CE90CDF13E019E0197F1A25197515751575157A157A15FA11B910B9107110C211C211040B2E0A6A0D6A0D45122F1640134013A514F714F714A113A113A1133E12A910A9104011AC12AC12AC1243120B151F151F156813F510F510F5101C141C1416123213221422148E1391109110090F930D930D930D49177B187B18CC14051E051E051E391435133513AA14AA14AA14C4139110911070101015101510152915BF25AB18AB185E1F942194210A22EE10EE102F12721372137213ED10840F840F4713C410C410F014AA158D1B8D1B7E1CE21E28152815CC17DE13DE13D51272137213B9113B103B10380F170D930F9E0F6E0ECB17591A591AFB14B613B613B613C214C2147A14B911B911B911FA10CA0ECA0EAE0F0B129A0E9A0E121512151215AE11CE10CE109B0E870EE50FE50F240ED20BD20BA410D00BD00B3F101810EE0FEE0F7118030B9411941190103D133D13971076117611A90B2E0B2E0BA410040E091109113E0B2A12B321B321E40C030B030B320A370E370EE1111F12240E240E330B00110011040EF811F811060C7B0AB321B321190E190E92069206FB0BC910C910C414240E240ECE0B48094809440C4311B20EB20E', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (16, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C38264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4DE090610810F810F2411431943194319D01B4B1E4B1E1317531453142019141214120B1616171617BD0AE80E17111711CC103F143F143F146A19241B241B9717F418F4180919E418E4188A1816175119160A160A410ECC10CC103F12F514F514F514EC16EC16DE1EBE1C6A1B6A1BB6201A191A191A19980C9A0A9A0A49108B0D8B0DFF12B014CB17CB17CB17911A911A5F1A45194519F518F314F3149618DB0EA10F6A106A10110EFE10FE10E11211131113CB1763166316CB18821AD218D2187516F314F3141B125A0B7E097E09B90AFE10FE10E1121113D615D6153516351635162116EA15EA15DE17A317A317E815DD056F076F07ED077A0BA50BA50BD712E211E2114010AB12AB12AB126117231723175719C417C4177007D60721062106D709580E580E131039113911620E78106A136A136A13031403140314C417C4179B063E096F076F072909AF0E3B113B112213F80DF80D4E0F90129012F1104011401103149A0B6C0AA60CA60C870929092909C20F731073101712C010C010C210011301130113A011401140114114C713C40CC40CB70777067706910A3F103F103F10D7107E117E111812EA12EA12A011A011A0118D12DD12530E530ECA07B307B309B309EF0E7D0F7D0F4D0FFB11FB1118120E11FF10FF10FF1065116511D413E90FB60EB60ED209CC0ACC0A630B55125512B70E70129913991383101D111D11901065116511D413E90FE40BE40BBA093808750B750B55124E124E12E610DF13DF13A111691169119611E4112C122C12930F8A0C440844087607A908A9083E0F701170117011180FAD12AD12B40DB40DB40DAB0FAB0FAB0F0B120C0F300A300AD309A908610B610B3B0DB510B510B510AD12AD129C0E020E020EE80D880F031018100C0F7D0DD808D8080208EE08EE08330E111211121112A112AA0FAA0F2F11250F250F2914A9118C098C098508D808D8085E0962076207160A0D0F2A112A112A114F0D4F0D3310EF13EF1329141E11BA09BA0917072E071407140733067A0A7A0AFC0ADB11DB11DB112F102F10940ED7132D132D13440E350C180918094308FE07FE079507EC09EC098F09420D1D0D1D0D1D0D240F240F8212361236124D0E', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (17, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F45119A4196B136B137F13B40EB40EA014AC14AC14610E24122412530DC80CDA0ADA0AAA0B1F0B1F0B51195C1150115011391009100910D80E7F0DC80EC80E400E581258129E0D0C100C10E10FE10FE10F980CC50E501150115011890E220E220EE10E280D280D8C0D6B0D6B0DB9110A114A144A14200D4E0DC110C110C110C10BC10B360CAE08AE08CB080C0E0C0EFB16650D2E0F2E0F511242184218E6154A0DF020F020F0205C115C115B09DF07B40CB40CD70D13171317940EA10FA10F1A15590D590D88114211931DB126B126B126851E851E2E1C6A166A16A20EEB11EB116312B812E815E815110DA40DA40D3C11D515A318A318A318C317C3172E1C9F1F9F1F9F1F6E0D9F0E9F0E330D6C0F6C0F3F16600E600E840E0D12B910B91014122213B20FB20F921292129212E60E9F0C9F0C490D890C2A112A11EC172E112E116C0A920F920F920F221341144114D211D211D211A90E9F0CE10CE10C820B2A112A114616E613E613C713A4111410141014103112CD0FCD0F700E5C0E5C0EFA0D7A0B7A0B2D0CF40AF40A870CFD0DFD0DC911C911BE10BE10BE10800F3C0C3C0C700E700E700E71101211100B100B100BC809C8095F0D320D22112211DC0FDA0EDA0ED810D9101D0D1D0D1D0DC013C013C01368096809AA080F0B0F0B7C0B470B6813DB12DB12DB12BF0EBF0ED910C10EC10EAB0D881F881F881F2F1715131513180A440A440AD0087D102A112A11DB12BF0EBF0EC40B440C560C560C2F1E061D061D2F17B813B813B00E2B0D2B0D290A031057115711CA0E9112A70FA70F360C670D670DBA11BF1ABF1AE91D581558151414670FA016A016031034123E103E103E102C102C10D9105D0F5D0F6A186A1834133413C613F213F213B114AF17AF17A9111913FB10FB10FB10A90F90109010DC12FE13FE13FE13C414C41451142A142A149E14F113061A571057106B1245114511451190109010A111BD12BD120C130C1351145114C41104110411F113F5146B106B104C1260116011BB0FD31081128112E413030B030B7B117B117B11FF12C615C615E70F240D970D970D970DB90C73117311990EF513F513F513621062102D167C1C7C1C7C1C1410841084108410', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (18, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C00663264023E87981A30D474000000000000000000000000000000000E6100000140014004548F45F0CCE0FCE0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4830B6B10E61048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44E0DAF0FCE1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44A0DD10D2F1348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4421142113D1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43C113C111A1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F90C300D300D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F00D870B870B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EA11C010C01048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FD0DDF0D7E0D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4320DA50CB70B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42D0A2D0AE70748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45A0B5A0B850748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47207230A230A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DC0ADC0ADC0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47B0DDC0ADC0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4061A6314781148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F514EE16EE1648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4460E460EE20D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F473107310B51948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (19, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4EB14EB14721C721C921892182D1765136513D318021802183B177A137E127E125713271227122418F217F217F217131E641A641A311AD11CD11C231D8B1A2F172F17FF1419141914F916D614D614721760160B1A0B1A0B1A4C1F9C189C18EC1A891C891C561B561B561BB710C314C314F215B5174519451996156A190B1A0B1A0C18B510B510C511891C891C7D1DD913ED10ED10DF10EC11EC11D011021302139F150616D715D7151412F412EC11EC110C1304120412D410801080105B13801080101B126911111399126914691452115211CA13EC11EC117A11B411B4112614E314E314CD11D210DB11DB11B8112D15A713A713A71352115211F412231423144C12B011261426147011701170115E1165126512DC122D158D128D128D123C133A123A123915BC11BC117C11061106110212261026108E12D113BD14BD14E4152C119B129B12F813FF0FFF0FF010D110D110CB120211510F510F5611FC10FC1069101C151C15DA14120F9C109C100514F612F612B8130A14431643169614F613F613AB11C213C213F4131C157E1A7E1A760E9A109A109A10A31231153115CD18B816B816011308140814FD135F128C128C12E2156A196A197A166A106A106A102012AB15AB15181AEC15C015C01595125F125F125F128C128C1209151A1F1A1F751646167014701470144211671667164A1407100710000D2C112C112C111B11F31DF31D1E21BF1E6A0E6A0E200D200D200D550EB211B2117D138C1005150515FD12501450142611FA1CFA1C311EF11D3E0E3E0E7D107D107D10B11035122A152A154F12A115A1155A159B149B142716FA22FA22FA228C1D850C4010401048114811481135122B142B140B130B140B144B158C126A176A179A1DF921F921F9211D133C143C1494154811481196150B13A40FA40F471262146214ED14571757176E1C931C931C431F0B18DE122D112D11941539153915C5132C102C10170E44134413B41999196E1C6E1C931C431F431F241C6911981098102D11A116A1169C12311017111711261510161016DF18E81AE81AE81D091C091CE8164611520F520F1A10C912F412F4129C1270127012F514D015D015221A4B1B4B1BF51F151A571C', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (20, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F41817181749145A135A133912160E510E510E0C15A610A610F40F3E123E128F15ED14ED14A6106611D218CB14CB14D5145E115E11EC1D53175317531778147814B7104F11BF16BF16FA1708100810081099185B175B17F51AF51AF51AFE16F919F919F9197814E912E9129F156B136B13A71656135613A612C41576177617541A541A521C521C291CEF16EF1600167B147B147B11F611F611A716A016581B581B11131E1403200320A315BB11BB11BB11EF16EF16B315B3154E114E11D31B791479143816931593152D150320032003204116DF11C010C010C0103D0F3D0F0A133E103E10D31BB512B512A9121F1373122D15D616D616BE19BE19DF11C010C010C010C512C5129B12B112361F361F6A1BE515E515A5104113871787172718261526151C121A131A131A131A13C314C314C314C314C3145716501650165D1441135A1A5A1A1522AA139D0F9D0F1A131A131A131A13C314C314C314C314C314AE18781798159815A610F41907260726AA139D0F9D0FAE10AE10AE10AE100B140B140B140B14851885188518981598154B17051C201E201ED611F0121F111F1112119E129E121516A315A3150B14E017E017A514C4172A152A15B617201E08180818F9170F130F130F137D117D113914F013171517150219C012C012671567156715BF1E432307190719681CE813E813C013051013161316A616021902190219ED1AED1AED1AED1AF115F11D68168E198E198E194119CC12CC1202117A167A160F14F311F311BC181620162016201620F115DF13DF136C12EB13EB13411989228922CD0F5018AE12AE122F10391439145010A010A0101620B4128E148E14CA10680D910591058F26620E620EEC1590119011B70AE707E707BA077F0D9C169C169C16431F081908190213612261220D21A618A6187105A6053D103D106504A605A6053602D80AD80AB612431FF522F522F52246184618CA12770E880A880A32103D103D101F03FB02FB020601560356036708441EFC196B146B14FC13D00FD00F2F0EBC0FBC0F6B096B096B0992059205EB03EB03521052105210571C65195A125A127A11FC0DFC0D160DED0B260C260CF00AF90EF90EF90EA006A006EB17EB17EB17', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (21, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4981198117716771577157412FF12FF12841359165916D719F713F713F713321032103210E90DE90DFC10FC107716A816A8161115CE167B197B193F1AD719D719121D121D121DD017D412D412A210A210A6122C1C2C1C1A1D661766179D15E218E218301C681B681BDD17DD17D017D017D017D712D7127013581BE21FE21F1A1DA419A4199A1C261E301C301C891CDD17DD17DD17211821182118931693167013AF15AF15AF15B11CBD18F21BF21B261E3A1F3A1F25185B115B11DB147B1C7B1C7B1C9316801380137312AF15971697166018171A171AD71BEE15EE15ED053A090012001212157B1C7B1CE215E217E2174113671267126712F318EF1BEF1BEF1B96120C140C14BE090B120B121215D514D514D5145B14E2179717971797179918991841120A150A150A15EB10EB10901596144F154F15DF14DF14DF145B14B8122414241424148E128E123211001432143214F113F113F113961494159415411341134113A312B7124D164D164D16041328102810C9133C133C13071230133013C2139414EC14EC14EC1412131213B1101517BA13BA13BA1394139413A014DC12DC12741015119B0C9B0CBC1317161716171657145714B1104D144D144D14BA130113911291123D1374107410070AE30AE30A9C15ED14ED14ED145714B013B013F1154D14B313B313B313C315C315FA15B013B013E10BEA10B212B212A117C514C514481548154815F115F1156D156D156D15E2160A170A17B013BB12BB12A712300F300FDC14C514C514E412E412D816B412B4122914BB15BB15BB1581168116FD14E215E2156215CD10CD10DC14DC14D712D712E4125D129C169C163C13701470147014BF15BF1582144114181218125F150D160D160D16D712D712D7124012E012E0123C137014E914E9148C1487148714F814951495145F15581658160D160113011301132C130A0CFC13FC136E166E166E16DD129B139B1365139514A718A718EA155F165F162F13B70FB70F2C13090E121512156814A312A312A312DE0FEA13EA13CA18051705177214D615D61585128512B80EB80E9F13C910BD14BD1490142B122B122B12EA13EA13BC1653155315D216231588138813221122112211', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (22, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4E90D8415801880188018AB15F722F722F722FE23FE23C9200716071611163D13C625C625C625EF1E591259121914801880188D180D210D210D21361D061D061DC21150105010F8138E138E130522052259125912EE12BD15BD153016AF1E251E251EBC1D361D361DA214A214A214F8138A148A148A147B167013971497144F1C4F1C4F1CAF1EB61BB61B7F1B0A1A0A1AA214B012B012B012B00F0D100D107B1680135117511700194F1C4F1C6B19FE1BAD1BAD1B0A1AC60FC60F441044104410DF0EFD0DFD0DFD0DE21751170019001900199F159F151516A51AA51A1A1B8E148E1416121612DF0EDF0E6C08EF08EF08E2179F147314731473145B1A5B1A15167217721772174E19161216120B0D6C0C6C0C750C220B220BB812B812A315A315B51A5B1AF418F4180D17E117E1174E19A113A113DA0EDA0EDA0E5B0D66106610B712B712A315B51AB51AB51A99189918E419E117E117E117BF12451145113B0E5B0D5B0D7A09230EB712B712EB16FD1EFD1ECB1CCB1CE419E4195617911691169116451145113B0E3B0E3B0EC5072609A516A516A5164F18AC1FAC1FCB1CF01CF01CF9139A159A159116D70FE909E9094406120512054D07C31AC31AC31A4F18AC1FAC1FAC1F891989197C139A159A159A15BC10BB09BB09B109390439048F03D816C31AC31A241AFE1A001B001B001B7C137C138415841584155312890C890CA0045903AF02AF02D816571C571C571C331D401D401D001B051A051A491BC21853125312C60CB304B3042602330433045D12481748174817331D401D401D401D4919231B231BC218C218C218BE0EB306B30685042402FF0740122E134817C41AC41A591B2A182A18BF18231B231BA119C318BB17BB17970E850485045202CA032E132E13B01CB01CB01CDB1C441D441D8318F51AE819E819701CBB17BB17830D2707270727070604D615D6152D182D187A197A19F31CF31CF31C7D1C7D1C7D1C701C701C701C7F132F0AD805D805D8050B135F195F192D187A197A19F31CF31CF31C7D1C7D1C4B264B264B2643174317D4068F048F04510479115F195F195F19091BFE1CFE1CFE1C821C821C821CD218D218F11143174317D406D50229032903', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (23, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4EF1E9D179B199B1954137C0E7C0E430D430D09120912110F640D640D730E1E0F1E0F1D23DD0EDD0E0522FA1B841CBC14BC148D188D188D18551109120912A716A716A716C3196A186A186A186A18E01A052205226F256F256F256C1F6C1F6C1F55115511861F861F861F7A0D7A0D0809450F450F1E0D3B127B167B16E3102F0F2F0F641A6C1F0F120F125021502150217B21B20EB20E170F6F099F0E9F0E2615AC0CC906C9067506ED09ED097812641464148B24502150214A1B3E15EA09EA09E61083128312FE154F0F6E116E11AA07CB0DCB0D7B127B12D91FD91F6026C813C813671A0D210D210D218714871476104F0F321E321E321E3F1691169116C31CC31CC31C9720221422144A234A23EA11EA112C192C192C196610C117C117C117371691169116FC21D125771877185B1AED18ED188410C00DC00DDE0B2C192C19230E840DC117C117E314E3149B229B22691B7A1C7A1C7A1C28182818CD15230EDE0BDE0B4704D90CD20AD20A1D0FE314E3146B109C109C10D01AA724A724D025D025170817088306C106C106A80325054D074D07320A760C760CE312290FE71FE71FA91BB713B713B70FBB0DBB0DF90867036703C804A80386046F0B6F0BC00D490F490FD50EC115C1153A13B713B7136B0FB40E851085106C11780178017801EC026F0B6F0B0C1533163316761E761E761E761E1511D60ED60ECA09B704B704B704C604C6046C11C409C00AC00AC410C410001F001F761E761E761EDA08B10CB10CD907B704B704B7047207C904C904FF07580E04120412C4103B093B093B098003800350023B073B073B073B073B073B073B07710B710BCA0310050412041204129E0589018901EA011A021A0242012200220038013B073B073B07300A710B8A048A04E906190419041302A100A1003501170417046203230076FF76FFE201B604B604F408060DAA08AA08AA08C604C6042602380070017001DE01E802E8027E00BBFFBBFFE000280128019C094F04040404040C071306E305E3051901BB00BB00DC01B905B905E802460146014601CB02CB02CB02DE0601039C049C046307C502C5028F02A400A40051038E058E058E0567059D069D069507E506E5062206', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (24, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4D11F0A1A0A1A0A1AA512500B500B8C0C88148814C0140411420C420CF00A4011401140116E136E13E01A5C110A1A0A1A5308B411B4111711FB13E715E7150710C00CC00C210FF511F5116310031103113B12D10D650BDA0CDA0CB815541854180E143019301936100A120A12360E3511BE0FBE0FC310C310530F530F540E010F010FC11B461846180B25912371157115ED1B66166616BB10BB10BB10FC10E310761076100C132C15C11BC11BC11B0B250B25692585178517411667156715C611321032103210E3109D120C130C13D116F825F825F825E71DE71D311C0119E411E411E411831083100F1132103210C70E7B0F9F069F06CF10FA25FA25CF10C512311C311C2617261726172617041104110F110F1161106110D50E130E970C970CCC148B128B128B128B128B12A410261726171412AD11A711A711901103120312D90C130E970C970C970C6119611961196119320F320F551082128212AD11CD12CD120B123C163C1625052505560D3B0D3B0D3F0FAF0FAF0FE610E610E6100E100A100A108212EE123914391414183C16A504A504A804D906D906110A890F890F800B500C540F540FD60FCA11CA116614E014E014BD0F0015A504A5044205D906D906AF0B610CE40FE40FE40FA70FA70F3E10451145119A125E13BD0FBD0F500CF00773067306280A2F0C2F0C1F131F131F13E40F6B116B113E105112F811F811EF10F80CF80C550C210A890789077707CE0CCE0C5E0C1F131F131F1373108B108B10991010101010FA0E9D119D119D11210AFA0523082308880C8F0C8F0C7A0E0C150C150C15E913E9132814B91306140614F7116B116B11710B7D08760676061813181318135E17A0150C150C15D3111B161B16B2167B137B13C81142124212060D42088E168E16EE0DCE11B015B015A015A015A0151115491749173E17A8138C148C1414139514D006D006D40FD10CD10C800F531053105F14B018B018B0189418FC17FC177D13DF13DF13FE139514DE06DE06F50D160916098F08F012601160115015181418143619131A131A2D1554155415B71129112206E30DE30DE30D93109310790F3E143E14BC14181418141814B816A213A2130B1314141414A011', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (25, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F40311A80CA80C11091109220822084D0BEE09EE090D0A980F980F9814240F240F9711B4126011601103110A0A3A093A093A09C308C3084D0BA308A3085B0DED0E941194119814A912A912771263116311C3105E096809680942094209420993088207330C330C400C9411941155144E134E1350156311B910E310CF0AC409E607E6074209F808F808210B640B640B59092115211520185C165C165C16A510F612690D690DF80AD50CD50C270B270B270B380D960A840A840A1412CC13CC13DC13DC13DC1341128E12650E650E450ED50C4F084F084F085C0D5C0DFD08840A840A2C0FB20FB20FF811240FEA12EA1284106F0D8C0C8C0CD108A20AA20AF309F309F309EB0CF00E0C0F0C0FE21455115511240F240F240F0F0CF410D509D5098E0ADD103B0D3B0D3B0DEB0CEB0C420A470D470D710E0D140D14420EEE0CEE0CEE0C3C16330EA10DA10DE90EC112C112500A180818086309200F790B790B0D14181018108710ED0EED0E3C16B512AD0AAD0A670BAE0EAE0E140B500A82098209420E880F880FAA11E011E01121128D0DD21000159612980713091309EE08100F100FC30AC30AC30ABA0F880F880FCF0E00103A133A132913D2107C0B7C0B5E0692099209EE085A0C5A0C9C0D110D110D110D4A121B101B10F210181518155B13E7127C0B7C0BDF09060A3C0D3C0DA6061F0A1F0A0A0E0A0E0A0E320F16111611F81174105B135B134C18101110111011BF0903100310180B720A720A630EF310F310F310F90E5C135C13680EAE12AE12CA1B2D11F112F112D4130E140E148A0F9409640F640FE110881488148814BA13BA137D12B00ED117D117EF118511F112F1123F15B014B0144012640F640F700F1012101210120514D612D6127E0F571A571A95141413E915E9154015461246120511580D660D660D0318D714D71405142B152B15F00E211621169514C613E414E414E4144D13B40DB40D7A11F912F91227136A126A12AB15AB15B014B0148F18AA1649124912AA14151815181518C612C612BF0E111477117711A3122D122D122D12B612B612BF13AA162A122A128514D616D6161518AE11261026104311BF16BF165F15601460142C122C12371037101714', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (26, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F45F0D5C0B5C0BC20C410D410DC618F721F721DD0F22222222222283181C251C259C129912991288138A108F0D8F0D1B122E15821F821F5718211421145319E419E4198D18FD19FD198515161316131613B9102411D212D21269155B255B258F248F248F24BE1EB11E5417541754173C163C16541554155415F6129612971497144D1B1B201B201B202822381A381A0814BF11BF119214A610A6108C135415C7138E1296125B1C5B1C5B1C4F1CB313B313E814E814E814B010800E320A320A300EE50EE50EC0111711FF13FF130C1F0C1F0C1F8A1CB313B313950FB5106D0F6D0FED0F0C0C0C0C100B080F080FF210E2113718371890219021E31BE31B10106B136B136B13E80DE80D400F1B0E1B0EBC0AC80C720E720E760CDF129A1B9A1B9A1BE31BE31BF41233103310F710C913B711B7118B0F5A0D5A0DD50C6B106B10D30F5A117A197A197A19CC1519101910AE102C102C102C105F135F13180EBE0DBE0DBC0CA10CED0DED0DD210E4177A197A19721A42184218FB100A170A17E6155F13E810E8103F0E7E0D7E0DA30B460E460ED210BC15BC15BC1548182C132C13970D80184F224F227E187E187E18420D3F0E3F0E4A0CF90D880EE712FC1AC31D9812981294107810781048184F224F2276189D259D25D10C940C460B460B970D0612501D501D3F1B9E129E12340F070F070F4818481852165216011B3E143E143E14340D340DC00E4A19CC19CC19BC19E81C8511851199114C114C11DB0FC71BC71BBE235925592559256F1084158415A316AC1B181818186D1A2A1E2A1EEF10B014B014070ED9136D166D160216BB12BB1254114B124B128D15281C75177517A9192A1E0B110B11D414070E070E971431193119A7150A120A12C80E4212411241123D177517FB18FB183B1776127612F612F50CF50C1D1B802520182018A115D70ED70ED813D613D613AA166A161214121412146A136A13DC14DC14E81AE81A80254F144F146414D013D013DC1274119815AA161E14121412141214540FC60CC60C550DB819B819F819F116F116F1163412C311C3111310ED14E512E512C711D412D41216123C123C129214DC135E165E161B160E160E1637147315731553118910', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (27, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F48813C515C51548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EF14A026A02648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CD14991B991B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C713F114221F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41711BE188C1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44A134A13A31348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C312C312051348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D710D710961248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41D126A116A1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B5110E1B0E1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4880EE618202448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40612191E581948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44A197C1C7C1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41E181E185B1848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45D105D10A31548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47214A315A31548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CD14CF14CF1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4981597153F1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4ED14D010041248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48910A514E01548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (28, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4E510E510D310770F770FE912F412F4124A144A144A145B14D617B51AB51AD11DC51DC51D5A1743195A0C5A0C0A0C300C300C68115516321332131D131D131D13D6171716171615195E1E5E1E04194319680E680E160C140B681168114F13581358134D12E411E4110B15FD194C1C4C1C1F21A91FA91F68206A0CD20DD20D4F0EB00FB00F0C15001500155B11D311B410B410B410DE15DE15941DA91FA91F7919610CD20DD20D4F0E190F9D139D133713B411B4113D0F3510351098173C1D3C1D381EFA1D76267626690FEB0B610E610E8A0E5D115D113E12CC11CC112A100112001800183C1D0A1A0A1ABA21D31FD31F0B12EB0BEC0CEC0C810DDF0F43134313CC11841284129113B313B3130A1AA31BA31B341FD31FA616FF14A811511151115111DB1094139413EB123515351585123015382138213D15341F341F341F8114D810D810B011200D200D270D94139413681A4B15DF1ADF1A2E17811881183D15BC16BC16BC161E12561356135A0F100E860E860E66138C238C235025012501259F258F188F184C21D11C43194319BA0DB5138C148C14760C060F060FD511C518C518C8210A249D239D233C219A209A20D11C431943196113E90FE90FE90F200C231123111812A722EF24EF24D219D219D2193C211B1E1B1EC81DD614F015F01580161F12BE0EBE0E3317331733172E206F1F6F1F6B2122172217981C4219331833182D0CA90BA90B6B1813186A166A16B822282328239622F01D41214121331D841F841F6D1AC41AC41AD016F80DF80D4915CE1786194A204A205924761C761CC81553225322611A611A611A201C7E1DD016D016D016CF0FCE17CE1764181E1E1E1E59242717271780144D18F019F0196B1CB218B2189E1B711D711D711D501AAD16AD16AD16901A901ACF1844214E134E13A417EF1CEF1C071DA919A9194119CC18A116A116501A98189818981899195A255A25241D9B169B1617197C1C7C1C131D161A4C144C14ED15A116A116A116D218BD24BD24BD24D91BD91B85148B1517191719A315A315A315A71740174017ED15521352134F123A19511A1C231C23042504250425A711F211F2110512DB12DB120E180E18231823186B12E313E313', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (29, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F443199818461546154E153E0F000D000D9B0D590F590F590FF90EF90EF90EF90E70177017701770124E1D4E1DA0190B1B0B1B4714DB13DB13BA10111211121B11041288138813311635173517811381139B209B20DA1FD126D1263718ED16A014A0145D15021702170B189D179D172E165A1A5A1A481581134B1CD81BD81B131DCE16CE16FD166C166C16F01848174817E619E6192E162E16A9159114911448157626AB19AB19381A9114911406159015001800184817F31BF31BF31B551655165B159114911404155119DD17DD17A311A3116314631463148E158E153A17CF19CF19CF19ED11BD11BD11B414D215D215A6161A18001300139C120811081108110811081118173019ED11ED11ED11FB10FB1057130514051481148114AC0EAC0E9C12FD10FD10FD10F511E217E217E217A917A917A9174E124E1295121A1232158114811495108B0A8B0A8B0AFC0BFC0BAB18AB18AB188B188B188B188B180A131D141D147E127410790879089509920792072A0BE407CD0ECD0ECD0ED416D416D416331633164213A714A714EF13A512440E440EA3095D092A0B2A0B630B371037103710D416D416D4162C142C142C142513CC13CC13A512D614691269129E10EB0FEB0FEB0F561A561A3710EA14EA14EA145C125C125C129510D413D4134C154411441144114411311531153115561A561A561A1618161816186811151215126C15D41349134913250E710C3A0D3A0D981331153115451C451C451C891716186811681168112114211421148D148D14CF0FCF0FA015A01598139813981369206920891789178917C711C711C7118212821221146418C718501A6A10A015A015A015A0157F1E7F1E6920AE1BAE1BAE1B6A146A146A1482128212821255153517501A501A121512151215B01B7F1E7F1E7F1EFD1DFD1DFD1D6A146A146A14AC10E311E311E3112615A116A116A413A413B01BB01BB01B191A191A191A401840184018B016B016B016AC0F1A0E1A0E1A0E3316A413A413A413EF18EF18EF1841164116191A051605160516791479147914AC0FD40DD40DD40DCE15CF18CF18C719C719C719C7194116411641163F133F133F130310791479149C0F0D0F35123512', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (30, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F47012C910CB12CB122713691269128514E6144E144E140C14B417B41795145C155C15F10F5B0F5B0F8113D211921428142814541385148514851469166916CD154C154C157316161666126612B90EB90EE613E613FE126A146A14AD12E312E312BA13BA13171517152917A515A515A116B910B91099129912B813B8137514A415A4157D141E12571257127E141619161929174C154C15A0167813F113F11334120415BD15BD1555157D147D14AA12851585151916911A911AF01862140A140A14C413611361130614D215121412143215E016E016CE13CE13F818F818911A911A911A961621142114B8139B149B14EA11F116F116321532155F154E134E13041657145714401DE417E417E417B1146D146D142A12EA11EA113215DB13211621168B10E20CE20CE91391149D1B9D1BDA18931393139313311531152A12AC10AC107410F811F811F811160C6D0C3D143D1491144C1B4C1B9B1D9F169F169F16941115101510B10F9A1420132013760CE50BE50BD70D80148014C315E519E5193118C415DF12DF12DF1215101510C90FF913B714B7148B0ED610D610D61080140114011462154D134D13ED11481348132212D012D012DF108B130816600B600B9E0D600D600D5C1406140614EA112F112F11ED112D13C812C812DA10DF10DF10CE14AA1024102410D8090D150D150D157C139D149D146815661166114B0F9A0D9A0D410DA015A015A114DC134F0E4F0E311569140C110C110C11DC17DC172D19651265123F0DA10DE110E1106B156B156B15C71800173115311599190E120E125616561656162D19B0185F115F11C70E701270125B125B125B1235174517D715D7157E155B0FC514C514801AAC1AAC1A2D1952155215D010701270122F139113F115261545176516E617E617B310E011E011801A801A801A1B19721836143614DC139513951398129B14AF13AF136516581B581B0110BC19BC19191A11171117111740164C164C16010F8D128D125D13B0124C0E4C0E3E140D14AE12AE12BC197E157E1521144817481748174C16EB13EB13EF116A136A13AC133512181A181ADD107A137A13181A7017701789165D15B714B7142515301230123B12D013D0135B12', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (31, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F488106317DA1ADA1ADA1A46184618851D821C821C821CDC17F710F710F71083068306760376037603B90ED6163A193A19D41A4618461853185318C914C914DC179A0F9A0F9A0F64056405D8027603880199129F17C919741974191019F516F5169E16C914C914C914990D990D990D6405220222027D0176019F179F178E1A621B621B421AD518D518E117A617621462140113CF09CF09CA07B602B602C100D00067136713B618711AA71AA71A101AE817E8177017C416C4166B13C80AC80A77059602800180018C012B12EF17EF17701CD41BD41BAD1BA419A419E91AC416AB14AB144D10F007F007DC031E021E02A5016113C116C116A01C691B691B5D1A3B1BE91AE91A4C18B717B7177D12100A100A170751035302530233135916881B881BB31AA619A619AD19A318A318A318131813183314690EC809C8096204670267029A1447150B1B0B1B191ACF19CF194219D218D218D218A01753165316F110091009104804E103E103F913E713051BF918F918C518A019A019F317D218D2187B194516451670120910C606C60637049E00F917F9172F192C182C183A183F193F19F3173D1842164216E215C913C9137D0A470A470A8C0109057E197E197E19C317C31754175016871B871B331AC616C616C01567136713EE0EEE0E8C018C01CB0DCF17CF17CF17CC16B915B9152015CE18CE18DC17F314F314A816101874107410EE0E8C078C07CB0D721A721A721A4619DF15DF152015CE182515251532141A151A15A314741074107410E905E90581019817461746174617DF152D1A2D1A7214EB13EB13D7129D0F9D0F5A0FDC0811061106E90508010801F115F1153C173C173C17EA17EA17FA140D153B103B10040EB00BB00BC9074D044D04C600CF00CF009B149B14A316A316A31607194F194F19E416F913F9134E0D050D050D5908AF037E017E015B0157006815681568152B142B1407198C178C175B10191019104E0D8D0BC108C108B5042C062C0623039801D014D014B11366136613F716BA166C136C131910080D080D230B100E100E100E2C062C06E40283036215221622161E141E141E142C15F611F6119911FA0EFA0E340DC60C100E100EDB09FD03FD032105', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (32, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4C501B103B10338026A028F028F0271001E031E03240439083908FD091C0D1C0D330F9A119A119A1188014D027A017A013802BF02BF023100CCFFCCFF6A00390899099909E60EE60EE60E9A119A119A11760109027A017A01380538055A005A001500DFFFDFFFCD001A051A05E907E60EE60EB512B512B512D000090288003D003D00A2005A005A002D007B007B0005010501D000D00023016C0B6C0BA40C390C8C018C014800FDFFFDFF8FFF8FFF8FFFC8FF6400ED00ED000B01100110011001B708B7083D0A5D13A501A5017300ECFF39003900E3FF160016001100ED00ED006601DA05DA054A07B708B708B7085D135302600160016001600160016001600160016A01000316041604D403DA05DA05971597159715C119F4008402840284027E02700470048707870787077E06910791073E04EB04EB042F112F11E816E81672008402C702C7029906700470041008A413A413A413A413D108D108D108FE09FE092F11A415A4159E000905090509059906990699069906B507A413A413A413F90EF90EC411F90EF90EB30A881988190905090509050905090509052A052A057A06BF07BF07740FCA11CA11041A42192E112E11E0086907CB0DCB0DCB0D090509050905090509057F028103A003A0031407E807E807CF1700080008E207EF06CB0DCB0DCB0DCB0DCB0DCB0D09059C019C017F02870387039C03F706F7069306E207480A480AE2077100CB0DCB0DCB0DCB0DCB0D6202F900F9003C010C03980298022703E408E4081E0D1E0D1E0D1E0D14010801080142008E008E008E007C01B702B7020401150415040C0861086108960A1E0D1E0D1E0D1401140103FF03FF44FF220022004501EC02EC0295007C097C097C09B308140A140AF20EF20EF20E570057002100210094003101310135025D02380038007C09560A560A7D0DCE0CCE0C6414AB11AB11980150012901B8FFB8FF29018104810486041A021A027D0909120912840CC30B64146414AB11AB11480348037D011C011C011501210821086F086002AD09AD092E0A5E0B5E0B3A0B18121812C310C310F803F803F004B802B802B3027001E101E1010706610561051B0AAA0DAA0D230D181275107510C310', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (33, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F44B0BE20CE20CE20C4A0E4A0EEB0D3011DE1ADE1ACF147B147B14F612A912A91237121414141433119A11FD12FD12FD12D115B210B21070130C140C14041544144414C011C011721172114C12DC10DC10B5128614FD12FD12B415AA1BAA1BE918851A851A13111C11601260121113691369133B0E1F121F12390C861486148614B417081608160816BB191E161E1669158D118D111613D111D111AC0E2814DF131D161D167511621262124510FD0EFD0E9D111F121F1252140D15161316136A13091309138D138D13F615F61575115813581358139A15671467145612A013A0130C13141414142D12C710C71074118D13871987191B1687129A159A159A15671467141A1497159715CD137D13B412B4127311B90EB90E7F14C119F217F2177A17ED15ED15ED15FD17FD17820F1112D217D217011AA516A516670F490F490F7813B51C2F162F167A172816C013C013C0133F0F3F0F1112581D581D581DA112A112AE137C10331633168819A8109015901505165F155F155F15CD0FCD0F3A117412FF11FF11FF11AE13AE135C12AF14AF146907F409101010100912CB14E90FE90FCD0FCE11CE118F13421142114211CB12CB12441319159015EF06B009EB0A401040102810B913B913E90F9C119C11D2130911091109115713C712C7125D144914DC06DC061409D00BD00B2F0DB913B913921392131C151C154013031103117E14AE15AE154315A0130B080B08D5099E0BBD0FBD0F3C10BC12BC12BC12F415F4150B103B103B103B10DA153E153E159714690D6911691140103A133A137811891189110A14AF110B100B10141314131413BD16B815B815F315F20E69116911B1100A11511051105311531153115B149E0F9E0F210ED316D31694141D158A158A15AB115411B411B4117811521052103F113F113F115B14CF10A010A0109915AB13AB13D8135C155C15AB1154118D128D120811C611C6116811CC112E112E115213A313A3136313CD12CD12901664152C15C310AC0FA810081108112B1197119711CC11CC11CC117F126A126A1242169714C413C413D8137512AC0FAC0F63128710871064121E141E1467126712F810F810AD0F5A135A13A7135B135B139014170F', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (34, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4781267146714D9160D140D14A81177117711C012BF16BF163613EF12DC11DC1136125E155E152D145D1192119211D9160D140D14B2116E1014111411F516F111F11168107110711078173015301530152B123413ED12ED12FA12B710B710B710DB10DB103A11CB11CB11C00E0110781778177A167A167A16DF131315CB15CB15CE13FF12FF12D411D411DC13DC13920C130D130D4C10CF15CF15D0148B158B158D130016FE14FE148F1211115E0F5E0FFF1394129412A10C141114114C101A144515451590157E13B913B913A5155D115D112912A90EA90EA90E770A770A640C3210691069104D0F2C152C154A156C140E130E13AA115C135C134810C80A9D0E9D0E9D0E481048103210361036104C0CCD11CD11E6136C14171314141414201238103810240AA80BA80BFB0D8F0E8F0EA70F0813750B750B1C221E111E11630F1515041604169615BA10BA10230D3C0EA60BA60BA013A013A013B81136163616B9131E111E112A21AF14BA15BA15BD13BC12A90FA90FEC0BF90CF90CC50E271027106310A6232E252E2543262A212A2190159015EF12EF122812AA0EAA0EA80BE10FE10F9512CE11E612E612181B2E252E254326C819C819491449143E133E132812C50C470B470B4110DC1ADC1AF4101F151F159C1B00190019BA20E520D322FE14FE14FE149D139D13C90D300C300C6912211121112B16881F50175017C01EE520E520E520E520BE14BE14BF13BF13BF13B10D5F0D5F0D5F0D451BAC16AC16691AAE15AE159A1C3715371537153715501350136F156F15450B450BDA0B55155515BD1B2F1E2F1E691AB1118F108F108F106F076F07D40E5813DB14DB14A90D850D850DE60D241324130A1A921BA216A216F516F516F51614255D075D07D40E5813D810D810600A800E08150815EC1CAA23AA23F51671197119FF19F516D823D8238E238E238E232C15410C6E0B6E0BE20E08150815D41989218921DD18DD187C247C246425A826A8268D128D128D1275129812D30ED30E2810501701120112952446214621A819CB1BCB1B0826A826A8262B128D128D12170F98126D0F641364135017BA1EBA1EBC1E4A224A227B13CB1B1D1A1D1A1D1ACB1BCB1BFE16FE16', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (35, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4701370138813F2107C137C1386129A149A1457144D144D14E81144144414091657151D121D124913841216191619E91B26192619D7171F101F10AC0F560F5413541355135C125C121B129411941137132E17D818D818E91B591D591D4E1E2E17F913F913F913BE10BE100713A012A0121813B912BA0FBA0F2216D81890179017E51690189018A419D814D814D81456195619E60FCE0DDB0CDB0CF30C3F0C3F0C7E13221601170117C816251625164D1793196D1A6D1A7E19531553155C1485188518410AC308C3086C147E131115A214A214FE128B148B14F316D01AD01A6A1C53155315621B8518011301133C0E7B1523142314231456135613CB12AF11AF1189168E1780158015A614621B621B071CE613E613E6136015A914A914B714B714B7147E10261188138813DB10961096108F1198129812AE0F520D140D140D510C2A21A51CA51CAF16270E270EFA0FA70EA70E210E810D810DE610FA0EED0EED0EC51096089608E803EA17EA17EA171F18EA17EA174E10140FF10DF10DF00CB40DB40D150EDD0DDD0DA90D27062706D203541C071A071A071A071A071A071A071A801480149F0B660E660E010CAE0B9E069E06610628012801D322D322D322D322D322071A071A071A071AF315F315A810980C980CA50A6E056E054506D306D306E520D322D322D322D322D322D322D322561A561A561A9D1B060B060B6015BC085407540754078D06371537153715D322D322D322D322D3222111BB17931993190514051405141614881788172B15720FD40ED40ED40E5D075D07C708D322B114B114D91765136513931AC214C214811AA91663166316720FD40ED40ED40E5D075D075D0791081F121F126D17BA12BA12151615164C174C17F519B118B11837188E23941094108D20C31DC31DD61A781051125112E31244124412C21CC21CC21CB120B914B9147F138D1294108D208D208E26B422B422B422F423F423F423331D331DDB18B6120A1A0A1AB120862286228D128D124A0F4A0F490DCA0BCA0BB422F423F423F423221B3F1D3F1D3F1DAD19AD19C8218C1D8C1DFE169C07960B960B490DCA0BCA0BCA0BCA0BED08ED084A24BE14BE14600F600F241A241A11261126', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (36, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F407150715131648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FB13FB13E91748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43714A212A21248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44613D313D31348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49D12B913E01348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47B15CC108E1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F460150D16FF1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AE06AE06DB0548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F430053005670548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F3041404140448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F105B305B30548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47706C5055D0648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48D06FF02BA0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4720F7E066F0348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D007D007330548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4371837182A1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45019E21FE21F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F486226622662248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4911E911E881A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41126761C661C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (37, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4251E7C207A1C7A1CC7197F117F1143124E0BB50DB50DF0112019201920194A164A166B1209170917251E251EE820E8203F206213F512F512CD099B0A9B0A9A0E47134713C1105F135F135F13EF170517DE1FDE1FE8201C141C14B11287128712CA091E0D1E0DC30D640D640D640D05150515051535171413DE1FDE1FDE1F931E931EB1128712150E150EC60E050F050F930E1412141271150A140A140A14DB12301E301E7F1C6D25621462142112E50EE50E700E120E120E6E0E4C14711571154914491449149A12301E16171617AE215B1F5B1FBD18E50EE50E880C4C10161416140216B816B81672189C139C13D114361976197619FF17CA1BA61EA61E090E090E090EB20D021502153015AB17AB1772187D1452175217D911B915FE15FE15E719BA10BA10D90FCE0DCE0D2F159917C115C1154A13BB14BB14BB14D519D519752158209F189F185118211B47174717A810611661165A1734163416B4133B143B14A318A3189E1A4E1F58205820981498144A1B1D171D17A416861786171315E513F112F112AA159A199A199A198D1888208820361B361B361B9C11641064109E1B831A7B177B17AB1321142114AA15101A101A101A7C170213021370126C125C125C12E510BE12BE12BE127B177B17BF15C514C514A916BC16BC16BC16EE152B147B117B11B5139B139B13831292129212DE1220118A138A130A159C189C185617CD15CD15CD153D12F211F211191334113411D1110F110E130E136515CA15CA150A15121912198017381438143814E91D3719A718A7181D179C119C111914D213D213D213E216D417D4174F1A22192219FB188A158A151814681A8515851595145D165D168F146915D213D2138817341734179B149A179A17A81763171312FD14681A5C12AF11AF111D1597179717691569156915C5152E162E165C147D16191719170D158314621262123912AC13AC132B16971897182F172F1719171917C815E913E91335145B165B165B16191466136613661326152615AB177017801680160415041504155415E913E9132F1738193819381985150313FA12FA12FA12BD15BD157017EF15EF150613061306136415DD1568186818E819CB15CB15CB15', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (38, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F47516D117A817A817A817031A031A031A031A031A87153F133F1303100310230E230EEE0BD414D41405179D16A817A817A817A817A81723172317871587158715151115111511110E110EEA0B6013601314136E169B129B129B127D137D137D13D613D613D613D613451145111F10110E8E148E148E14001966126612EE15EE15EE15E2157D137D13FF13FF13CF12CF1245111F101F101F10361936193619001908120812D9170A180A18E215E215B713B713B713CF12CF12CF12B012B012B0126C1436193619BC15D114181918190A18FC15FC15251980168016CB13BD13BD13BD133815381538156C14691269126912291A181918197815A014A0148114401322142214381338133813361138153815E415D716D716D716291A291A291A4B1582144A154A150E1398139813321238133813D812531453145314BD16D716D7169E1A1E141F121F121F1297129712A01498133212321210114D144D14051605160516BD16751C751C8D18D015561256120610610F5F135F13A014A812A812791C791C791C85151615A517A517551A551ACF13CF13EA101311131113115F135F132311DD14DD141818791C85158515D911441244120B1BD71BD213D21311107711771113115E11BB15BB15D31718181818FF16FF16FF168419442044200B1B451E9F169F1603157711791379130C12C316C316FE17191719171917551B551B551B442044204420FF1CA2161915191548145C145C147D15871B871BD21AD21AFC1DFC1D551B551B551B1D22821B821B821BB7123C153C150115AF15371937191421BD1CBD1CD218FC1DFC1DFC1D1B1B1B1B251A3A1E821B821B131266133A153A15501AB01BB01BD41E8D1F8D1F291E291EE816E816F415F415F4153A1E8515851583141E11CA13CA13501ACB1AB11DB11D8D1F47134713471369146914F415F415F4159F15571457141914301223167C197C19011CB11DB11DED1B7B157B157B15691443134313F414F414F414E5232316CF13CF135D16F91BF91BF91B351D351D771D7515FC15FC15B5121A171A17C620C620C620E523B118CE14CE148314701CF91BF91B301D051D051DAD1CC41CC41C3817812081205312F616B618B6180E0E', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (39, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F403190319031911127A13FD14FD148419F117F1175015B714B714B7147D127D12D111DF0E8A128A12CF14131C131C131CDF1470147014DB1771187118A7171414B411B411B411600F600FB8102910291000195A16C415C415C41580168016B016C417A717A71756161016101687118C0F8C0F29101214741300192C12931293129312DC130E150E15C01577177717D51410161016A011A0117B117B11A915351616151615141080148014801471147114C015311621142114640F2C102C101E121E121E1230138912BD11BD11601258128E128E128E1212141214DA1342134213E110CE12CE12DE12A714211221124213BD11561256127A1011111111571257125712E4144E14E412E412CE12DE12DE12211521152115D716710F9B159B159B1533163316F613F6134013401318135C125C12BD10BD10BD10CD1C8D1488138813751C371D9B159B153316C915C9157C127C127C1270125C125C12C91094175E185E180614880F880F551A1613A112A1121F1645164516CE15D612BC13BC132917DF1CDF1CF51204140414581607140714D71BD71B921F1F161F161F160B160B16A913D515D51597172E152E156116941A941A941AE7176E0A451E451E2C1C502150212719011A011A5D25531EA31EA31E0816E012E0128D18901590150C133912B419B4192C1C5021502150211D175D255D250A24801380132216D817D8178618210D3A0F3A0F640CB419C71FC71FC71FB11AB11A5B191E1A1E1A1C25A613A6131E162E174317431789138A158A15E111DA196B146B141F157D177D172C19FC1E941E941E5A267A197A19B61A2A192A198614FF12FF12560F8515431243126B142A16571657164C1A3B183B184A19052205227A1AEF1789168916F812A10CA10C5714E70F210F210F210FC216C2166B1519175D155D15EC122D162D16E20EF812F812F8126A0D6A0D2316B20F4E0F4E0F5811851A7D187D18DB172B122B12301139103910B70A07120712071207123F0698169816BC0B661066103418D418D418061E061E061E7413670B8B098B09A913041204120412FE099F0F9F0FBC0B6E0C6E0C6610D210E013E0134C14E616E616F108F108F108F1082A0A2A0A0412040B', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (40, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F47F14A516A5169B19EF17EF172C15BF11991199119113C210C210C210F010F0109D0B7906790629037313E514E514CC17EF17401540157615171417148414181118115A115A110B0E0B0E7D07E609E6097413C618951695167215ED14ED14501650165016B41507106022602260228A0D8A0D6D0C2908290835163516951695162018BD16401F401F401F3418341852136022602260225C135C135C131E08A80989121113BF13602360231525CC14CC1471147114711452135C1DFA23FA23F60DF60DF60D620CFB11111311134B17B018B0189217A714A714AE197114401F401F242224222422670B5C0D5C0D7D0EB610D716D7164C194B17171A171AC51890169016CF1990269026F70B24226B0C6B0CDF0D930F930FA310D7164C194C194C19091D091D381772187218CF19CF196E0B6E0B290CAA0CAA0CA30E1A101A10A3104410FA11FA113B155E19C30FC30FE21E3C233C237512480D480D3C0FC20FC20FC20F1A107211721193194412FE1CFE1C4D201F111F11D6103C233C233C23A812A812A812F80DC20FC20FB511E112E1126E0A840E650D650DD21682148214B211B50FA010A010A010441144115812261526152615791479143912640C640CD216D216F50A3B103B10A30AA50CA50CA010BD10BD100A10651057185718AC10F90E640C640C640C0F0A0F0A430A3B103B10A20A230CD20CD20CA40D2A142A142A143A113A11C1114B158A108A108A10A10A360F360F1F0A0B0B0B0B5F0E5F0E5F0E7F0EA215A2152A14A318F018F0184316560F8A108A1038093C0E3C0E190FC10BC10B590F590FD60ED60EA215630E630E410F8E0F8E0F4316780A340834081B06D10FD10FD10FDC0EDC0EDC0E021814141414E916670F670FCC11661200120012B20829074A074A076708670767072F0DDC0EDC0E7A1716111611091C8E178E178E17A215681268123F063F06580758073A071C061C062F0D2F0DA214A214161163186318F719981B981B981B1315C615FE09FE0958073B083B08C90AC90AC90ADB0C5E145E14A21270177017A819771A981B981B8D18E315040B040B620C620C620C1C0F1C0F1C0F1C0F5E145A135A139A1587158715B219771A771A02187A16', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (41, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4880652065206A702A902A902B202E201E201E201E201E2012305C706BA10BA10BA10751075107510E6098D068D065C0584048404B3023E04E201E201E2017B057B058D0595089508D811CC0CCC0CCC0CF2079D04A504A504B204A40BA40B3E043E043E043E04BB0CBB0CDD0D4C0D311F311F021FFE17FE17A809E4099A079A07A80BA40BA40BF00CA40BA40BA40B860BB90DB90D311F311F311F021F421B421BFB11FB110F160F16A80BC114C114C114770E060C060C060C361136116E0F5A172817281771188224B610B610FB110F160F160F16C114C114C114BD17BD17BD1725156D156D155A175A175A17CA1AC71FB610B610F412201620160F160313241424146C15BD17BD17BD17E310E3106E1F99139913AE11370EEA14EA14EA14341CFF18FF189C10591359135213431343137918F8116D126D12F70BA90AA90A650A0E180E180E180F1872137213420B111211121112350F461046101C14991199114A12641164114511B515DC0DDC0D27107213970F970F6B0DED0BED0BE60BB80BB80B170C1E119E109E100218CF10CF107914F10C09100910860FB60EB60E0C11DB0DDB0DF30AE50A510A510AAE088F098F09B00BFE0CFE0CF90EE80B260F260FB70E1E2203150315D4119E0F9E0FFB0EA10AA10A190A150A150AC10B880BC00B4B154B15340FFF14FF145312D30ED30EBF1221102110B710960D480D480DE90C270C270CFE0BF30B4B154B1543134A154A15DF14CA0FBF12BF12781207100710D70E740F740FD00DDB0CDB0C7A0DD30A431643160C137416F418F4188414A30FA30FC60F3D103D107912BB11A312A31210148B138B13440D5213521352137C165A195A195F164613461356105B1394129412BB1107150715701419131913A812AC1483148314B9168B155F165F169215A012A0125313F511F511741274127412EF131B1423152315C615E514B916B9168B158B158B15EC123C113C119811F51162126212B112B112B112831223152315E3158B15AD19AD1934143414071207124411A511A5112C11DF11DF11DF1159135913A212911339117A1600140B16BF14BF14BF14F413F4138D12A511A5112D11EF10EF10EF106812A212A212F8113512', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (42, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4890F890F6312D111E713E7131E149D129D129D129D109D108F0B5A135A13A7150E143B133B1334116008580D580D6210E411E41128134211421142118F0B860B860BFD11581558153413591559159511271960076007400FE411E41142106F11330C330C330C860B860B1012CC13CC13341308146A126A12DA214C1AA819A8193B1C2C122C127811AB10AB10AB106C0B6C0B66119F14EB15EB158413401240128224D310FE22FE22AE12F610F610990FD30F461346134613EB13EB139F149F149F145D146C122012C71F2119E80EBF11BF11F610720E720E4E0FB613B6137412741274128512FD126C126C126C127112D415D4157F0DBF11BF1104114D0F4D0F9A113311EE12EE12EE12531353137A1375117511E61102199D0F9D0FD40CA310A3108D103811B210B210C11078137813EE1260146014E013F111F111F11145189A0E7F0C7F0C4D0EE110E1104812911091109A11F913F9137813911482128212D311041204124518170EA50CA50C540F841084100216EC1596139613F913F913F913B413BE12BE12CF11CB11100E100E170E530E07130713A112341034100C12961396132F112F112F11D312CE12B512B512E00EC00EC00EC00B0D0D1C0C1C0C1E0C5D0D5D0D0C1279121012101210123B123B124A1258125812450F07100710F30B190CD90BD90B1C0B9E0EBD0EBD0E8210D810D810D810D111D111281251111C0F1C0F1B0F0512FA0BFA0BE30B880A880A9E0E39103910AA10EA0EEA0EEA0EB411DC11DC118D0EF20BF20B3D0C2812450D450D170CB40DB40D200D6310230F230F6610C610C610FC100F110F115C0E440BE10AE10A560E9911C012C012B40D190E190E6610190F190F2D10460F460F4E0FBA0F7B0E7B0E750E7B0E7B0E6210DE18C012C012E60F8E0F8E0F66118B11BD11BD11A010460F460F4B0F9F0D9F0D9C0D6C0F6C0F681029163112311231122E110A130A13DC11011201120E12161216121E12C00FC80DC80D290F4E104E103911AC11AC11AC112E11D713D713B112011201127B11B5113F103F105D1045104510F50D410F410F3512A811A811A8116412721269126912DE116811681176114B124B124B12F210290E290E1E0E2C0E', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (43, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4DB10DB10D70D4514451470178A22EA1AEA1A9D23E415E4152A121D1A1D1A1D1A780B780BE50B110989108910370E1711441644167A1A7A1A7A1AB914B713B713F612AF0F1D1A1D1A4C07AE05AE0508033B12F70EF70EC60FE315E315B11E651D651D5C0F030EB40CB40C0E0C740A740A6808C60BC60B6A044511330F330F35101612EE1AEE1AB30B190B190BEE0D940B940B1609EB08EB08D208AA1872127212201220126E0D6E0DFC0FE20DE20D520D3A0C3A0C820ED40D950E950ECF0E1C0A1C0A701CF119F1197112BE0EA50EA50E861229189A119A11450F1A101A105D103A0E3A0EB21181128112101C4120652102190219B20ED90DD90D5A11F111F1112E12DD0EDD0E7910240F361336137D12CA21CA2136231923451845184318361E361ED20FD20FD20F2E125710980A980A060F310F310FC613C41DC41DCE16CD194518451846195E19C412C412C610C610C6107411771377134810EA0CEA0CDB133F187E107E108C19E010531D531D8D1B0C140C14051369176917F41678163F133F13510ED915D9153517D10DD10D4C0CAA0F79127912411191134D154D152D1D0A170A170A17E117E1173611731673167316E51CE10FE10F5613C414C414C414911363176317AE19E114E114F311E11766136613861386138613B016960F960F0512C414C414C41482149B149B149B14C6171910191037173F1A3F1A3F1A34153415B016D10C5E0D28121F124314821482149B142817281792179A179A17ED1A3C1C3C1C151A0A1ACF15CF15E11C411B5E0E5E0EF712311631161416F415F41598159A15DE15DE15AD1AAD1AAD1A0A1AE412E412EF1BE11CA30EA30E92121815A916A9161417B215B215BC15DE15DE15691A691A691A6A18A71231193119A20E68106E106E10E71081168116D4166019601927142714FF19FF19A816A816A816C1121410141082114E10C114C1147B11B518B518B51829229818981898184C1D4C1DD21BD21BD21BED115215521552156B104C17D317D317601D001B001B302230223022A91F4C1D4C1DB013B013481548154815521552152C0E70153E173E178C1D6B1D6B1DC41F0A26C71CC71C8E1E261626165B16481548154C1532133213', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (44, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4ED02ED02F901C500C500B202590859085908590859084E0BDC11620F620F620FDD14DD140D120E14BA01BA01F8007B007B00DC0060FF4FFF4FFF9AFFE600E600FB0053085308DC112D0C2D0C85120B13C401590259028C0184018401C6FF19FE19FEC6FDB500B50001FE5D005C015C01A70D0B130B130B134E17CC0ACC0A230501020102AF0103015C015C01E4FF1CFF1CFFF3FE44FF44FF88FE64026402630986162E122E122E121F0870037003C601260126012801C400C40017FF64FFC0FEC0FE13024202420265219E1C9E1C9E1C250A6603660366030B020B025301E70064006400EFFFB0FFB0FF54FF37023702192353209E1C9E1C380D650580048004E8018D028D028801770177013C00C9FFC9FF9FFFBD0010FF53205320861B881A881A650B800480043302DA01DA018D02C10192009200EC0031003100BEFF18FE8C198C197E1E7E1E7E1EEA0E420C810281025803D201D2010802D501D5016801680168015F006800431743177E1E7E1E20182018510A880688061703DE02DE02D801BD01C801C801C8019B009B00CB00A7151F1E1F1E1F1E05180518F70D380938091506850572037203DE02B602B6020D01F600F6004202060DC315C3154117371B051805181B0AC207C207B3046D056D05E102E2022C022C023B02E902E9025E0D5D1DBA1DBA1D371B8B0E8B0EB70B3E073E0741039902F102F1029D0231043104E503A907A907411BB61DBA1DBA1DCD21B219B219B21905110B040B04CB027A027A02D802310431043104F808F808E11C221F221F4B174B174216B219B2190511EC0EEC0ECF063305CF03CF03A002E502E5020005F8087513751309164B174B1742164216871487148714A608A6080507380438043B032503250300054305251425149C101613321A321A321A27182718740F580F580F060DE80964046404DD03DD03DD03DD032514251425140617A518A518A518271827187D127D12060D060D82056404640431058C028C0294050B170B170B17EB18A518A518A5180E15150F150FAA0B620A620A820582058205F504F00494059405C01195125F105F109414941494145D0F370E370EAA0B130A6B046B046B04B604B604F50454025402', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (45, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F40E14761C561748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CC26CC26CC2648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40B130B13C12048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4370CC120C12048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4370C960C960C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44D04DE02CA0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F410FF9501370248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F418FEFCFE0CFF48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F494009400190048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4850185014E0148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44B02E403E40348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F436030909090948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A9070311031148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F8080311031148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F808B312B31248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4940C940C250C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A70BA70B940C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A70B6B156B1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F494050E070E0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42B02FB05FB0548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (46, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4AA125113511351137C147C148C16EF158C138C138C13D816D8168816DC16DC162E19E214E214CB15C8112614261426147C14361436146215031503158F17D816D816E7162218B816B816D714621362134315F917F917F9173714501550152D1843195B185B188F1763176317291680158015D714EF16EF16521600154214421442141E14AB17AB17C2195B185B185B1844174417FD13971312151215941A5B1B81168116FA15FA15FA158813C215C21508199018901858185818AA15AA158515DA15DA156B15501A811681165115511551151216F817081908193B180517051705178E138E138515101510151D17D819A817A817A0195B15C514C51407171D171D17FF175419541923182318871587158A13451845189F19D319461946194619251B251B6D12601660162A1BF319401A401A231833153315C41039103910DF180B16B515B515441824181E1B1E1B4A1DA21BA21B251C4A174A173B123B12C612C612E712B711B7116212E51230123012A213A215A215A916A818A818B115C4125B105B105B10A10FA10F21110C140C14A90F0E110B110B1141134113831383139710BC10BC105A0F1A0F1A0F1A0F830F830FE6123A1379163C0E6D127913351335135F109A109A10980EBC10BC10771021123B153B153B15E612E61291141D14D009D009791377147714CE10CE10CE107B0E370E5F115F114212751275128315AC12AC1210142B11860D860D7F0E3A12D311D311690E7B0E7B0E7E12B211B2110F1102130213D611D611D212D2127C1022136A126A122E0F8D0E8D0EA714941394131E13A410AD11AD116C1303130313801380138013B2134215BC11BC11660C3D123D12A714A7140F0D0F0D030F4C124C126C134B144B14D61351145114511466145411E90CE90C75138F138F1353112E092E0976103F13B611B61194143E143E148F148F148F14D611530D790D790D46124E0F4E0FC30A3308F409F4093F1307150715D212ED13ED132F15F113F113EB0E640EC30FF20FF20F9F0F9F0F9F0FA70DA70DA70D851580168016E6149F14DF14DF14D7139912B411B411E2117411741120166D0F6D0F390BD315D315D3153E16D017D0179F14A213A213DE11FE11', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (47, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD25404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4E01579167916701C701C701CFC1D331A331A4B1CC41CA624A6248120EC17EC170619B618B61899132716441A441AE51CDC1EDC1ECF1D811C4B1C4B1C1218BC1BBC1B4716FD16FD16E31602190C100C10391C6F1EAD1DAD1DDC1E1C171C17811C811C811C7F193422D714D714BA13E215E2150219581758175B1B5B1B171C171C4A199F149F141515C525C525C525AE1A5E245E240418D116D116FD1858178A11501A0219071BB114B114D2166D156D155D19C525C525AE1A39143914541FDD163B123B12F0187B11E817E817F515F515F515401F091C091CFC1BA81B38173817CE185F185F18E20D7E0C7E0C0A0CA20790199019661720152015BD145517D715D715871E151B151BF410070F070F3C0E5210741074108F0A391AAD18AD18CD15E515E51557141B141B143D13B813B81300116810171117119711921192114610BD15401D401DA51CC718C71869154C156E156E155013001100111913D413D4131A13B313241324135A1B0B19C318C318BE18C617C617861434143414C714981398131913BF12BF12BF126D139715971579169C161A181A18BE18E116E1161214EF13EF13EF131E13041704178717161416141C143F163F161D146111D715D7157A17361825182518F91AF91AF91A1E13A417A417AA17B41479177917EA1EEC144A104A10D7155B175B177018861B861B861B541554159415B31580158015AB15A523A523511B1415881188118216121612168F1753208D1D8D1D5A18121C121C0C176D156D15B314D515D515A714431332123212FC131216C71DC71DCD22CD22CD22CB1A661B661B531F9317B314B314E419A714A714D9129514C315C3152815AB19AB19A1133D1E0E160E167514F023F0232C1DBA1BBA1BE41990119011C20D0C14971F971F971F501B07150715B01488198819D122FF1CFF1CFF1CB61E110D110D870C1E0C1E0CF1131513271627168819851A851AB0140B1B0B1B241FA719A719A719761A321D321D5C0FA00BA00B9912F015E216E2160D1A841BEC1EEC1E0E1221162116D41512161216A01A321D321D080F080F130EFE11F0152C1744184418E5186D246D246621C612C612D013D417A01AA01A6F14230C230CF20BFC1A', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (48, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E325404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4A80CA80CFD07FD071C0B1C0BC00CFE0FFE0F170F4013401379052F092F092F09A00A62076207B008EE09790779076F05F105F105C00C7008700810079C04EB05EB056603CF08CF088A0F18091809B008090A090A090A79083705A106A106751075107510670AE106E1064A03340A120D120D19065B055B058A11090AA40BA40B0F06E206E206670A75107510670A530552055205A106830B830B79045F035F037B117B1124072407D609C509020A020ADB06870C870CFC0A990E990E990E74067406670691091F07A20714075E07C701C701C201A006A006DB06710B710B400C6908DC0DDC0DDC0D080608069109B80A1007100703055302530211029B009B002D00CC00880388036109DC0DDC0DDC0D420C420CD31972092F0E2F0E2F0EE40BE803E80388018D018D01AC0073FF73FF9F00AF07AF07E70BC31BD319D319D3197F11360F360FE40BEA05EA057A0686058605AB000301CE00CE00010701070107520F520F520F2117D5120B100B102B0B1507770677068405040504052802A902A902FB05100A100AEB0DEB0D520F520FE910AE0FF208F208BD08110311037C04F704F7048001C702FB05FB05E309EB0DEB0DEB0DEB0DEB0DEC14F70EF406F4061A07BD02BD020B014E00D004D0042C0488048804720C260F260F260F4F0DB51314150A0FFB05810681065A020D020D020D023F023F021608AC06AC06290E5111260F260FCE0F7D060B0F0B0F0707B00BB00B980CD10AD10A880E1B10590F590FA50C290E290EE50C51115111D50E9002D912D9129E0C1D0C1D0CD60A6D0BE50DE50D9D1234133413770F250E250EE50C4F08AA03AA03290262099E0C9E0CB40A70097009F8095A0A5A0A9D12F7129D129D12F40EAB0AAB0A0D04A602A6024D02770947084708A209660D660DF10D130E120C120C60134E0F4E0F740C530953095806BF010E020E029809FE09E20AE20AE20AC70DC70D1E11231123112817181418143B0C2A0AA001A001740174017401130EF10ACE0BCE0BCE0B501250125012510FDC10DC100F0CA00AA00A0D09AD04AD04AD044F064F06FC1A3511A910A91018116010A20FA20F520CE50BE50BCA066A096A09BC0AAD04AD04AD043608E70C', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (49, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4B008B008A20CA20CA20C56101C0F0E150E152F142F142F14EB1465146514B2197F18361636161D14B008DE09DE09560F561056105610E314E31416155F135F1393133916051B051B57175717571797183F09510C510CE2134A104A104A10B01853155315B013681468144913CE11CE11A114A114A11433194506510CE213E213E2135C185C185C18971197119711091309133912D5119A119A119A116B126B121F07080E8B158B158B155C185C18C0166B116B116B1147114A104A10D511AC11AC11AC116B126B12B80A950D8B158B158B159213511551156B1147114711C50FD20FD20F4E0F7910A512A512AD1271129E0A9E0ABF13BF13BF13BA0E351535154E12CF0FCF0F2D0FE80ED20FD20FCB10A112A112E0126B12361836187F187F187F18220D4E1528142814420FD50CD50C9C1018111811CB105C125C12F1119E11881788177F187F183B133B137D1386148614420F3F0F3F0FB3105711761076105F112D0F2D0F70114B174B174B174B17931493147D13351435143514E513111211125C12F610F610F610B80EB80E5712B513B513B5134B171D15331533158A17FC14FC14FC147412741275122E122E122E12E4100A120A12B513B513B513B513111258175817A1141B131B131B139E1435123512E510A910A910A9100A120A127D06C2088C0E8C0E11123D138111811123129E149E14A816B014B014A312FF10FF10FF10E2104312900257024213C412C412BF142C112C11D510BE13BE13BE136612201120113912360D360D360DB20F5E055E051C1326132613B3132C11ED10ED10F810F810F8105F10501150114E128410841084103312950295023211A212520D520DD112D80FD80F6011601160114A1207114E124E12311231123112AE133506D313D3137C107C107C109B12E70FE70F19122913291329132A11D811D811BD106312631263125505A311A3117C10741774177417FB0E59125912021302130213DB13EF10EF10AD0F071107110711A709A31104190419F614F614F614FB0E8F118F11C6122A13DB13DB1304138B108B10201187128712E70C790FD217D217D217D316D316A80F4D11C612C6124913EF13EF13621262126212DF1387128712', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (50, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F428142814451306160616061624142414771305126A126A1236119A109A109A10961196113F12D912AE18AE18DD12E514E514E514EA14C513C513361300120012C012C012C0129A100212C613C6137813AE18ED14ED142C13CF11CF11B413C513C51334146214F913F913F9135B125B12FA11C010C0108A1102160A130A13A3111E12AE13AE133114641364134A13B915B9151C131C131C133614BB113F103F10411479129F119F11D210801180116E12730F730F5E11F2156914691469143B123B125A111A0F1A0F7112FD13AB14AB14D21088128812780EE20D07120712F2153E143E143E14D00FD00F6C0E1A0F930C6B120814B313481048104611270E270E0E0FF410F41041134F104F109F0F9F0F3D123D12E90FFC0E9E129E123F13771277126911451245125C13A61282138213E70FC511C511C5113D123D12B3135112A011A0114612BF0CBF0C320E6E12C313C3131C15A312A31214114C134C13C612AC1102130213FC12A0110C0F0C0FFF08820D820D4712731473147514F410A211A2117112D011D01174105C0E5C0E36100F10330E330E360ABC0DBC0DBD1269127A117A11B212130E130E1F10750D750D6E0EDC0A620D620DCC0F3710B10AB10ABC0D32103210C810A713A713C310331033105A0D5A0DA10FA10F070F981198114312F812BF0EBF0E780D940E940ED110FF12A514A5144219FA12FA12BA14BA14BA144D1698119811B20F0A104A0F4A0FF20F780DD00ED00E3D132E152E15421998169816D314C614C614C614B814E115EF0DEF0DEC0F9710971062100B0E0B0E11102813A21AA21A1A16AD13AD136C14C614C614A3105D120D120D12EC0F4A114A11CF107F0E7F0E7F0E2813CA19CA19331693149314CA15F813F813F81359161312C110C1103E10261026104F0F4F0F4F0FDF117D147D142917BC16CA15CA153E134213421342131312071207124E0FED0FED0F1F100D0E0D0E0D0E191211161116C316D115D1155C0FEC13EC135010DA0E8A0F8A0F5D0E870D8A0D8A0D210E480E480E480E59145914C316E016B011B011C312EA13EA138712190EBF0DBF0D600C710B710BB20C2F142F142F145F1681148114FC12B80FB80F0C1257135713', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (51, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F0004072326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4F914F914501350135013991174127412BB0DC40DC40DD40D23104812481248128C108C108B106B0F1F111F11F512F512F512AD0FB00F780B780B650CB10DB10D230F230F230F230FAA10AA10B810D3111F111F11991199112C102C10480E590B590B6C0B6C0B6C0B650D640E420F420F510E340E340E77119710B90FB90FB90F790E790E790EB10DB10DB30DB00D540D540D5B0D5B0D5B0DB90FB40FB40F8816B00E4210421042104D104D104D10FA0FB70DB70DBA0DCA0ECA0ECF0E280FD20FD20FCB0F1A131A13930C650EDE0CDE0C490F4D104D10C210C210C210C210BE0EE30EE30E5B0E710E710E4E0ED511D511FC0E2710DE0CDE0C490F490FD412D412D412C210C210DF0EDF0EDF0E7919E215E2155F0ED511C61451129D13D411D411D411D411D412D412D412D412D412EF10EF10791979197919AE16AE160A1A6B17011401149112D411D411D411041304130413B710EF10EF10EF10A726A726A726702670267026321FDF12DF12F7127E0E1B131B131B1304130413B710B710B7108C17FE10A726A7265E12702670262C1CE0127E0E7E0E7E0E530D530D1B131B131B13FA1AFA1AFE10FE10FE107C1F7C1F7C1F7C1F7C1F2C1CA90AB909B90978087808101510153B25FA1AFA1AFA1A981B981B601A231323137C1FA012A012A012E0144B1470177017A51C7B1A7B1A3B253B253B254B1EB117CE16CE160411041104113E119F119F11E115B91396149614A51CA51CA51CDB1ADB1A4B1E4B1E4B1E6C0B6C0B3B113B113B113B113B1106115D123716560F341A341A6E186E186E1848231A1D1A1D1A1D7A163B113B113B113B113B113B11720ECA14CA147911A610A6106419CD12CD12BC1645141A1D1A1D7A167A167A16D91781138113D20C620B0A110A117911370F52165216B31101110111691A070F070F070FB90EB90E771076115A115A115A1150107D107D101310671167118B118B118B11BA11BA11BA11BA110111620C620C2112A91AA91AA91A1E111E111E11F71316171617860DCD0BA80AA80ABA11BA11BA111014101410141014610F04150415C313DD11DD11DD110F12860D860D2E0CAF0AAF0AE20A1B0E130F130F101449184918491849184918', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (52, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C3826404F3283424EF8464000000000000000000000000000000000E6100000140014004548F46B0F7616FD16E519E519E5193D1A3D1A0A26DD19DD193719E516E516DA1614170C170C174113BF11D416D416C51724192419E519D218D218D018691827182718F516FE16FE16E318E918E918C514CE11781778170719651965196C19A5171E171E175118491849182A173A173A177B17821EE013E0134A1BBA16EB16EB16231A86198619E617DB17DB17BF18C118C118F8184625AD16AD16821E52165216072127131A131A135A1A511A511AF01BE61BBB1ABB1ABF18FC1AFC1A46254625462562266226871887186813C314BC14BC14BC141E151E15151C561B561B5D1BA11DA11D881D4625F51FF51F622687188718C61406178916891659153D153D152615B217981C981C611D8F1D8F1DBA13BF13BF139B1AB11FB11F6B17F417361736173E175B16DE13DE139C17B217B217CA16E416E4166413BE139C1A9C1AA11A56124E174E17E92487148714A416B113B113FC1345144514BF11BF168C0F8C0FB10FBF13BF13AB1190102C1C2C1CE924E924E924E924AA17EE16EE164C13D511D511B311681168115F11B90FB70FB70F230C2C1C92199219921992199219AA17AA17AA17EB14EB14EB148E126B1162116211810F760F760F7E0CA0122F102F109219A524A524BF19AB1631203120EB14461346134613461346132312F50EF50E2D0B7E0E7E0E7E0E0610A524A524A5243120312031203120B316B316B31607130713071307138D0D8D0D061106115C125C127E1945264526531F1B231B233212B316B316B3166F136F136F137812AE0EAE0E720E2E1009110911F50F7A1E49194919F50C34093409060D811B811B811B8215510D510D470E3D10BC0DBC0DD218B717B7172518481848184919E30BE30B010A811B811B811B3C204C1D4C1DFB0AC111931293122018BB18BB18CA172918471D471DAE15EC13EC131B13220F220FE51F4C1D4C1D971CE522440F440F57135713F519F519F51949254925121D3D253D253D25BD123D253D257823071E071EE522280EE90EE90E0D13F519F519F519001F1C221C22051C3D253D255D0D5D185D188A0F0E0D0E0D91124918F50DF50DF602F602F803F8030C115E205E20F624FD24FD249818B117BE09BE09E70D1F101F10', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (53, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4BF11A713A713A713A414290CAE09AE09C108C108C108130A150315036B049D059D051D05AF034A02CE11EB11EB11621862180611BF0BBF0B670704070407550439033C033C03C103B507B507B5078104311631166D0E211421143414EE0BEE0B1706990514051405F003AE03AE03A807020B020BAB0BF10831163116FD107F17C512C512BE0FA909A90985064A044A04D703AE02AE02E10D5115F814F814FE0E0721F21BF21B47166E106E10BE0FCB11CB11F1088F08940394030E04C60EC60E88179F169F16F008471347134713DD160918841284123210BA0FBA0F7C0949164916380B14121412C81B4616021502157D1B5B1BA018A01828191D181D185019641764174C17171749164916041BC81BC81BC81B2E182E1856124D12B815B815B21545134513FA1822192D192D191717A813A813041B5F105F100D0D6A0D051990100F11280B781478149213F311F3111012181118110C160C160C1659135F0D470F470F0907D7040B0C0B0C58078B098B098B0945074507CB0FCB0F26102610010DF406F406C1054C0F4C0FF3013F026E046E04B603AA03E106E106EC05EC0DEC0DEC0D850F850F93098E0A8E0AD70A41056002600282029808D707D707430B720472044905C308C308C3083A099B0D9B0DFD0B290B290BFB036D026D028202B908100B100BAE0B410B410B790BCB0A3B093B096F093A093A099509600A600A04079601BA00BA00B908C60AC209C2092F0C8D0E8D0EA50CC10CC10CAA0AEA0A260926099509D705D7059705190919093D10A910C00EC00E1E10D510D5107E14340EF10EF10E3A08B208B208220647034703A507D116930DC11163153A14CF11CF11C00F7B0F7B0F0811720D720D7F0B7A0B7A0B05052A03A502A502D116041AAC14AC14772277227722FA1A46104610F810FA1F210E210E210EE30FE30F96111E0D1E0D4208580F7E1A7E1A772277227722601B721BB81FB81FFA1F872587258524391A391A0E20AA13610C610C910E1A161A161A161A16C51AC51A3920392039205820872593249324C42051205120151DF116F11658214A0EE409E409361B361B361B361B32212D172D1710242B132B132B13721E721ED214F116C917C917', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (54, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C0066326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F44A029702770848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F481048104B60348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E708E708A00748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E207E2075D0848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D0032F072F0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48809700F700F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40519C418C41848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40519DE1A7D1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D7046907331748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F301F3011E0848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BA01BA01AF0248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D101D101A20248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D0015904590448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42F068710871048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4930D66232E1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4041A5C25AA1F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4580F71206C2248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AA21AA21730648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4582158211E0D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C917B817B81748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_01.tif', '[2005-01-01,2005-01-17)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (55, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540989537D1F812474000000000000000000000000000000000E6100000140014004548F425012501BB00BB002A0087FF87FFF7FFF4FF450045003B00DEFFDEFFE5FF59FF59FFEBFFB9FFB9FF9C009A006F00630063006600A5FFA5FF1A00B0FFB0FF1AFFCBFFCBFFEAFE06FF01FF01FF08FF7C008C008C0079006D006D005F005B005B0056001AFF45FF45FFC7FFF2FFF2FF19FFD3FED3FECCFE58FF7D007D00730082007E007E008B007200720080008000800080006900690084008DFFD9FFD9FF1B007D004F004F00AD00A100A100A300B800B8008E0021000000000021004F004F00DAFF94FF94FFD9FFC5004A004A00F5FFBB00BB00AD00AD007E007E002100BCFFBCFFC6FF530053005300C2FEC2FE5300C60079007CFF7CFF31003B003B00BA00D700D700B20079007900A200780064FF64FF4DFF1F001F00800180015E005E004C00760076003B00A400A400A4002501FA01FA016901E500E500E500BD01BD01D1004C01FB00520052007000B000B000AC013B013B013A01360136014401FF00FF00FF0077015601BE00BE00AF01420042008E018E018E01BB014401440157013101340234020AFFF5FFF5FF20005E0032013201AF01A800A8008E01CE02ED02ED021F031F031F03F00102010201F3FEABFEABFE21FFC2FEDB00DB00DB00FB016E016E014503D702D7024B037B007B004804A1005E005E001600160016001600680168016801B004860286028806EC04330533050F07B109B1096002BAFFBAFFF300F300F300DA00B004B004B004B0043407340734079A05E50CE50C8B0CEB0CEB0C0F06F80032023202D5024901490156035603560356033407340734070804861086108610D30FB709B70953012B032B035C0244024402DC015603560356035603480CF000F000E302740B740BB111DB0ADB0AB70726072607C502740242015AFD5AFDFAFD6E056E05480CD202D202FB024C084C08C20A450CE90AE90AAF05AF05AF0533001900460246025F01F702F702F702BD02660366032908AC10AC101B0E910A910AD303AF05AF05D6009900000200025F017200060006003405EA05EA058D0C1C0C1C0CBB1209110911E3097A021F011F01D700D002F601F60101014F004F00170546064606A907F308640D640D2110460B460B5C01A900A9001602', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (56, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540989537D1F812474000000000000000000000000000000000E6100000140014004548F47C00C70078017801BA01D101D101BF0133023302E601A202AD03AD039E023E093E09EF12E211E2117C007C00900090000A0103010301E002C00208030803C5047E037E03540BEE0DEE0D94121C101C1058FF9EFFE0FF690169011801E002E002C402320332034C055F055F055009CA102F112F11F2096806DFFFDFFFE3FF85018501FA00040304031703CC04F00AF00AE907140E140EF610F610F610BA0CEB0963006300B500D900810181013A02E402E402640544134413F80A6F0E6F0E7E157E1518161816F316CF0078017801BC0129012901BF00FD06FD06060722132213F70F4913441444147E15851485140617AD01A501A5015C02980198019801EE055F085F08FF11601360136914001D001DF014DB13DB134F15BD016902AF02AF029D019D019D0121FFAD05AD05B50C380E380E1414001D451C451C451CE11AE11A560176025E015E01CF0152015201A9022503BC02BC023E03530A530A0515E71CE71CDD19492149215E007CFFF700DF00DF00AF02D802D80226038E038E033703430643061F06DA0FFC11FC11D01B6F1CB600B6005B00630063002100110211021103C503C503B5037F05860A860A4C10D310D310B219DA1AAB00AB00B402640264024B001102410041009003710471041607C30AC30ACA0C8E128E12F9153316BF00FC00FC00A501BD01BD01C4000D020D020D0223042304210C3A10BC11BC11D6189E149E147B1674017601760148015F015F014302D40092009200CF01870587051A0AB810B810421D91159115D113F600870087007700B901E201E2018601FA00FA00170365036503AE03AD0ED618D6181A18D811D81142012401A000A0004D01E202E2022703AB03AB032104FB0364066406300F0E150E1517156B116B1119000B024C014C010401A3004C014C01FB01700470048504CA06CA06460E3014301444187C15F0142B002B0036002E012E018F00DF00DF00C801850485042109570991129112AF150317031747147D15120012003400B700B7001A021001C901C90141079B0A9B0A3A0BBC14BC14FB1461156115E71682160A030A03C500BC00DD01DD01E60196019601E905AD07AD077511251925199A17591759175917C317', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (57, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540989537D1F812474000000000000000000000000000000000E6100000140014004548F43F0DD117D1170C186118471A471AE019411841188415EB11EB110F230F230F230F23CD1C75217521390A740E1B181B1823151C141C14C013EE16EE165B0FA60EA60EB315E3190F230F230C23E621E6216806FB088D148D14D013D314D314EE0EEE0EEE0EEE0E672586168616001E982298220B1CF41FF41FEB0911085211271B271BD0168D0F8D0FEE0EEE0EEE0E67256725672567257F212F1F2F1F8B1CBF1CF316F316EF19B81FB81F271B1C141C142A13B51EB51EB51EB51E6725672517233D243D2465258324061706176618B81FB81FB81FD314070D070DEF1EB51EB51EB51ECD1FCD1F07260726E51EE51E8324061650185018C316F215F215940D070D070DEF1EEF1EEF1EEF1E3F24072607260726751F751F751FF016381538150F0A0F0A0F0AC907A207F81BF81BEF1EEF1EEF1E891FC025C025C0257D237D23B4214819F91657205720081C081C081C9C23F81BF81BF81BF81BF81BA525C025C025C025B625B625B6256F1C6D152F192F19BF1B7E147E142A069C239C239C23F81BA525A525A525A525A5259822B625B625DA1A2C157E157E151315CB22CB22CB22CB22CB22CB22CB229C239C239C2397225D225D22741B741B32163216AE14FC14FC14B31588138813CB22CB22CB22CB22CB22CB22CB22BC204F204F204F20741B91129112B7148D148D143915B315BD18BD187A15601860182321CB22CB22CB22F019F019F0194F205B144D104D107410C813C813AB18371937193719981F981F2F247C1C9E179E177917F019F019F0196410AD10AD100313F817F817F5172018131913199622C71FC71FCA18F21AF21A791779177917F509471442144214EA13B818231623163E18E41CE41C0021E916E9161717F413791779177917F307F307F0149E141F141F14B818B818B8180A1C14211421AE213E18461B461B3813381338133813F307F3077D15EB14DB19DB19DB19301BB818B818BB14DC1ADC1A6B17851D851D4511381338137E020402A701541554154E1744184418C516CB1ACB1AE8200F150F150F15C00C230A230A451169036903A701A701271727171618F218F21831179F1A192019206E225E195E19ED0AF008F008B608A404A4041C013201', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (58, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540989537D1F812474000000000000000000000000000000000E6100000140014004548F490215F215F21CC1F761F761FEA1EFB1DFB1DD822B61F6F1D6F1DE724CC0ECC0E3210B00CB00C9104E81DCE23CE23D923AB22F823F823831D1E261E266825ED1CED1C7C1CDF0DDF0DD307570A0C040C047E1D31233123312331233C233C23A01D1E261E26E01DF01DF01D251E700E85068506D204B303B303BF1C3425832583252823032203224D266526CA20CA205F1948154815A4097C077C079B030703070383240523202228232823D920BF20BF20451ACA20CA20CA2032153215B70B5508390239024102C20283248324571EA61DA61D301DCD1DCD1DDF1A0F1227122712D60E100A100A180AB302B302AA023D02D417D417A61DA61DBD15BD155A13780A780A050A820F820F7C093805380596072A03BB02BB02D0036D19381738173817920F920FA409EF0AEF0AD505B905B905ED0313038304830413046306630685054D1938173817270AAA09AA0976059F04CC03CC030D0441034103CA01AA01AA01FC041E071E07700A02180218970A970AC2084A044A042B026D026D02620107010701C500F00065056505910AF008F008741B741B3C073C0759060C040C041502C701FF00FF0028017D007D000201620562058B0C790B790B741B741B8D078D074D0AAE010C010C018301A000A0008B00DFFFDFFFF000E107180C180C090ACB094C074C076E01EA05EA056201D000D000C300740074009800C4FF65016501D504870C870C3A0A490A19031903D900A300A3009F009E00AA00AA00CD00A501A50140012E012E01EB0446064606EE03BD0AC605EE02EE02C301FB00FB009500AC00AC005B01A501A50187004B01C601C601350169096909530FAD06A403A4032F02DC01DC0107018F00470247026A01730173013003AE04AE045904AE09AE09530FC902A203A20380035302EB01EB01F600A501A5015603AD03AD03A207A207B70CB70C750CC410C410A701B2028F028F028A02580258021B02A204A204DF0201047B0A7B0A0E0D40104010D91075117511A701B001C001C001FB03C40437033703FC048F028F02B702FB11FB11101310131013811292144612EB01EB010804A305A305BE090A060A06E810600A600A600A7614361236128612031403147C14AD12', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (59, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F49D049D0462046201260126012601290929092909CE08CE08F90BDC0CDC0C11124A14EF14EF14950FEB0324042404850122012201CA027E017E01CD012909550B550B250C06110611A8113F113F11D40EBB036C026C022C011C011C011B014401690169011103D107D1075E0C08130813BB128E12901190119103AF026702670253022F022F02CD03CD03CD038305790879080F0FC612B513B5137914C911C911C2025103440344038403330333039D039D031009100989083C103C10CB1094119411DB0FA915A9153D02E203BD032C042C048605EF06EF065D069F0A9F0AF7096C146C144715B9149F149F14A915341612041204390480088008B00C67086708C110070D290E290EF2146B156B15291299129912F6129112E004E004800781088108790D15119A0D9A0D960D7A107A102013F313F3131810C812A112A112D911BD0CB20FB20FB90BCA0FCA0FA8126C136C131D116D136D13EF13EB142B112B114711EC17EC17AB1AF50CB512B512B5125A145A146A146E159D109D106F14E715E7154D1AF318F3184B191E191E19EC19A10BCB12821382134514BE14BE146815AB12AB12EF13C613C613121BDE1A7F1B7F1B891B25192519CB09FF11F312F3122F120712071293134E129E129E124B12F922F922F922C41EC41E091DFF1BFF1B490AFF114312431231128C11BE11BE11F90FD90FD90FA30E16131613F922F9226C1C6C1C3D1CA41CDB13DB131114D811D8115011D910D9106410C210C210141294129C1C9C1CEF1CE21BE21BFC1B6D1C8A0F8A0F900F78127812F7111410440F440FD50F65126512BF15021C021CBD19C51BC51B311C141FCB0D07100710081238143814B90F610F610F4810791279124F164D1A3F1C3F1CB1197C1A7C1AC2172010E80EE80E810F660E660EFA0F2310B110B1109F15BE1ABE1AA11B24192419DB16B116B1167313B0102A10630F630FF30CEB0EEB0EB210C110C1103715B419B419BE1F1E1878147814F513661366134612D4107C0F7C0F630F140E140E670FF510F5105D17B01696199619F613C213C2136813AE14AE14AD12B0124F164F164F161E13A010A01030148B188B18EE170C170C173414821382138D1301125413', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (60, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640989537D1F812474000000000000000000000000000000000E6100000140014004548F46E136E138312511051105E0E1D101D104210E30D8E0B8E0B80088F088F085C0E65116511E5123A11A511A5113F11130E081008104C0FA40FA40F120ECB07CB07A70A090D090D510DB2119F109F10A10F990F570E570EA50DF90EF90E07107C0F7C0F0A0B400CD710D710810F8A0F8A0F6D0E600D600D780BEA10DB0EDB0E330F8511851185111E0D0C0D0C0DA50E110F110FD90F82108210870D080C870A870A4015C30E3E0E3E0E4E1109110911EA0E5E0D5E0DBC0D610E610E0C0E4E0E6F0D6F0D310BD809D8093416AF11E50DE50D060ED60FD60F2B0EE30A0C0D0C0D9F0B280A280A7A09110E110E2910170C170C9112AE111C13BC0EBC0E9D10620F620FA7082E072E07BE0514071407CF07B00C291029103E0CEF0A6A136A138911311331136A108C0D8C0D6507F90610091009780A3D093D096C0CEF0EEF0EA30D620C601860180D16A914A9145E10660D650A650A0A0881098109200CFE0BFE0B6A0D5C0F270F270FA509EC1921162116F71272127212390FA60AA60A5309990B990B0C0F210F520E520E8B0E2F0B2F0BD207681A3D173D1769160F130F13FD0E320D870C870C190D8C0F8C0F4410F60FF60FB90E99099909DF08AF1AD31907170717E11200100010C80B630B630BAD0AB60BB60BD60C7B0D8C0A8C0A5B096C0B6C0BA41CBB1B6D196D193A13B20FB20F560C730C360D360D7B0A00070007DC042E062E061E0AFD0DFD0D6D1C471BB2187F167F164E0FD00BD00B300BE90AE90A580A930793071407D907B40BB40B660D1411261B261B471871187118DD0F7B0A7B0ACA0A700A700AEF067D0553095309300C800B800BA50C8D0F84148414D6146E156E155D109C0A9208920815097A067A0693064D0A4D0AD40BE20CE20C810D3811EB11FE10FE103A1123102310CD0DD208D20854072F072F07D5098B0D3C0F3C0F1F0FE90DE90D4D107D116F116F11491100110011390B6B09F207F207DB0894089408E20E3C103C102411AF0CAF0C9D0C4A127D13AC13AC13B911000E000E8C0CCD09CD09FE09EF09EF09CB0C7E11841284127E0DB208B2085413221302140214E313E10FE10F9F15AB0BAB0B7B0A9B06B108B108750C310C310C1D0C2E092E09', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (61, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640989537D1F812474000000000000000000000000000000000E6100000140014004548F43A116C126C12A111A1117610E50FE50F940ECF0ECF0E6D0D980F980F5B161C11B20FB20F1F134E1E2C0F2C0F8A117F117F11071102130213CB11A6114E114E1161147C157C15EB1984188418B7151F1B710C710CEB0F601167106710BE11931193114C12B814B814B0155A1C5A1C3B1C2B18B717B717B317A90C14111411141147114711B4109F109F106512F21216171617DB1855165516B612301630165118E409120C120C4F0E5D0E5D0EDA100F119F119F11FB10EF13EF13BE12301130116311DF14DF14911A0D0A5B0BA30EA30E290F7810781020102B102B10460FBC0EBC0EA311FB120613061325189B1F9B1FEF0A50102C112C111E117D117D117B112A10191019101A10041104115C12970E970E8A14FC1DFC1D620C1D0A4D11901090103B10BD11BD11850F430F430FCA0C820F820F1A142E1B4B124B123C1330191F0C1F0CD710D011D011D011D810D8106E0E7B0F550F550FC611A917A9176B1F62216221E61D32170E0D0E0DC10ED60DD60DEE0F4912210D210DC40E441144114815531853188118E3184D1B4D1BDF1A000E420E420EF80C610E610ED60F830F830FE70EF30EF30E08168F1D9B1E9B1E451DAB18AB189A1A2A0FF70FF70FEB0BB60DB60D3710F1104B104B106B0D54125412751C231F231F381D331B331B5A1C3613CF12A411A411500F901090105211A60FA60F8411141314131E1A0C1D491E491EE11C0D190D191411471272117211D30EA510A510411255118D108D10F212FB12FB12FB134C174C176D1DD819D8198D0F5F115E1143124312A511771277121D12571257128317A619A6193818C315D014D01465164F16121212128B107F0F7F0F311148114811EB0F62126212081DFE1AFE1AFE1AB2177F137F131C184518B111B1113A10E411E411601175111C111C11381527132713791979197919A712AB15AB158A1B211D38103D113D11F211E714E714AF149C149C142116D914D9148610AD0F910E910E680F0A170A172223320D8B0F8B0F4B11AA13AA13CF14161B611D611D54168D138D13E20F260F260F160D871487140D1A1109020EE111E111FB12C815C815C41E722072201F1C6916691604127211880E880E0E1220162016', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (62, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F44E1EB21E991899185A17DA14DA14A811B90ECF0FCF0FB40971087108A70A510C510C080C080B080B1F1B821B0D19E617E617481A8D168D167A178B158B15BD118B078B07B3091D0AFF0CFF0C570A6C0A19191919FC19BB16BB16F51DAD18AD181013371331123112A70AA308A308B309610B610B180BA00CE717E717D0174016B916B916D51792169216CD16F316F316710CC509C509A308E1084C0E4C0ECA0CE217FC1BFC1BD91CC81AC81A1B1621192119D8193D1762116211F011D509D5091309290A290AF70B7F1C841B841B391C521D521DD9181F15C91BC91B0B1B071707172D139C0F9C0F9C0F9C0F9C0F1F10FC1D5B1A871A871A9E1D15191519A51AA51AA51AF119F119F1197019B719781978198417F21AF21A30197C19B618B6189E1D091909192415C017AA1AAA1A0221831A831AB322501F501F08210821082132177317001C5E215E213B1280108010BD13172017200221F01EF01ED21F401F0821082108210321FB12FB126D17FC1DFC1D0C11F80CF80C3811B022AD20AD209D2050215021F01E401F401F0F1C4920EA14EA145218B920B920491DAA16CC15CC156A1D95209520AB1FE31EE31EF41AC91EA81BA81B651E661B2E192E19D818491949190F16B414B414AF18111C111C311C211A281A281A881A59175917F812EB16B915B9154A154E184E180A193B1707170717EA18371C371CE41A95199519971ABE14BE143C12F2147A144E164E160B170A130A1351159F169F16411B7A1D7A1D5D1CC3161D171D170414EB12EB124F167B14101310131C18021B021BB01A6418A81DA81DF61C3B183B182713B914B914C916C312C31245188017D31545124512B918811B811B5718451645163F170F130F139E113A19481A481A0D1D0D1D3B1A3B1A1C16DF14DF1484149E189E18E116EF11E711E711DC10721072109E15781E781E901CF21EEF1DEF1D651A9E149E14BD10D20F54115411BA10CD11CD116D109A0E9A0E4F12371A371A9117D01A77181917191794143E113E11DD16981898186B129F119F11230F620FA80FA80FA8141E1A1E1A7E0E71176B166B166515451245126517001774147414F811EB11EB11C3101C101C10710D7B0F7B0FC70F', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (63, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640989537D1F812474000000000000000000000000000000000E6100000140014004548F4660B8710871048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46C0A8710871048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A00C190C9E1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45D0C5D0C840F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48C0B8C0BBE0D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B00F3A103A1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47A13C519C51948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EA1D851EBC2048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40321E520282048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4492015203A2048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4821E821E622048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46C1B6C1BD21E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41F15C71EC71E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F311E813E81348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42411FE0E871848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40D1DFE1C891748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F21E341B151848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C011C0116F1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AA0FAA0FDC0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B90F4311431148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (64, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F41C01D202D202EA012F031A031A0325051F071F07480907100710320EBF0DBF0D290219049F029F0227011D0216021602440391039103F5019E049E04040C69139E0F9E0F820DC10CC10CA80CE508E508BA001C01360136015E01D903D903AC01D603800D800D0712FB13FB13FC14A80FA80FA30A0C09A407C300CC00120128012801E001E001E001A1041109110916062F102F10931AD1174B0A4B0A1906800A22012201F601F700F7008A02DE02DE02590AAD09240924098A0C4B174B17DB12B10EB10E7C0A330A7C037C0329060205C104C104F3020D0B0D0B41103C0D3C0D570E361736170F12A011441944194419A20204050405B706E408E40823072B092B094110F80C58105810EE122F172F17A011441944194419E40227052705EF068505850586082A08BD13BD13AE14871587150914F812F8122F172F17C01BC01BDF0249043E093E092906DA08DA08FE0E9E119E116A149318931835150B12D414D4142F17C01BC01B2E01370698079807C40D910F910F22100D10971597158C17D417D41730127014701450148E1A8E1A5E018C013706F506F506C010271127112D11491149118C120D170D175F19FC18441544150A178E1A63016301D701220222021407E70DE70D1813DE11BE11BE11CE157F197F196C19F612F6129413E116C700C700C701A104A1042D07820FD113D1133A174F144F14B8136E186E18FD18291229126812B512CD01E7FFE7FF21FFD201D2010B04550F550FD6125B155B15311420172418241899122B112B11900F04025C035C0317040C000C001A024506C411C411A614F216F2163719EC18EC18C513931193110412F007970AAD0AAD0AE505D109D1097E067D0E7D0EA21173157315C417561853175317920E04120412430B300C460D460D0B0BCB0CCB0C600F890F890F690F9F0FE516E516B41967166716AB0E221322130D08F80C5E0E5E0E760F83102E122E126B10A90FA90F3D11871387135E18DC15DC15C71386132213350C350CCD113B103B1024121D121D12601183108310CA148A15E715E7152012871087107913791356115611FA11BE13BE130C150315151215128F1437163716C715AF13AF13790FCA0ECA0EE1147913', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (65, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD254023E87981A30D474000000000000000000000000000000000E6100000140014004548F44E03F300F300110102020202DD021B011B0187068C07CB0FCB0F3A15C218C218C2185917591759175006A704A704C10161036E046E04E3027F067F06AC07B10EB10E0E14C218C218C218C218C218C218A4070807DC05DC05DC05140514050E06230A230A470A7214291429147D1A7D1A7D1A7D1A19191919800A67070B050B052B0443046D076D07230A7A0A7A0AF7146712671237167D1A7D1A7D1A7D1AF01B330A9805670A670A670A0F0A690869085F115F115F11E6139612961237163716771677167D1AEF195D155D1525165B0F5B0F5812C415C4151216EE12D412D4120713B41AB41AA116B016B01637162B1644194419B219D61C181718172218F617F6175F1535163516231AB41AB41AB41AB41AB41AB41A1A1A441CB219B219D61CD61CD61CD61C0C1A0C1ACB162717AD1AAD1ACC1ACB1ACB1AB41AB41AB41AEB1CC01BC01BC01BCA1A4A1A4A1AD61C0C1AFA18FA185A1B391C391C4B1A8A1A8A1A511AE119611761178E1A901AC01BC01B651ACA1ACA1AAB17AB17AB17CF1B811B811B991DAA1CAA1FAA1F7F1A5D175D178E1A8E1A8E1A8E1A6B1A221922199319361747194719C218DD18DD18731EAA1FAA1FAA1F23212321E1169E158615C815C81551160E160E1653175D175D173C17E418E418FC1BBA1EF11CF11C6C212321E912E912B512D216D216D216D216D21653175317D317D317E31FFC1BFC1B481EF31CF31CA721CB1C040F040F040FD216D216D216E516FC16FC16E919361B361BE31FE31FE31FE31FE31FA721A7210D1E040F040F040F040F461546154615D216D2163419BA19BA19D7196F19E31FE31FE31F3A1E3A1EBA1C04120412041204124615461546154615081B081B6F1B491B491B6F196F196F196F1961186118BA1C221304126A126A12BB1657175717BB16081B081B971B081B081BB61979197A197A1961186118611822132213221322137513BB16BB16BB16911B911B911B911B93199319791979197919871A871A871A79137913221322133613C714ED14ED14C714911B911B911B301A301A301AD51AD51A871A871A871A791379131212E411E41190133C133C138C138C138C131217301A301A301AF21BE019E019F718F718', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (66, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E3254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4301A301A2B1A981AFC1AFC1A7F1B7F127F12D81C7B0F7B0F1707C80749044904B801B601B6016002CA18641A641A72193C1C3C1CEF196A116A118A093F0983068306E00557035703B601B601B6011E04E71A311A311A161A871B841884184D13A808A8081C0AB406B4060C075E025E021701FD00A902A902F01B5C1B2A1E2A1E1B1C361936191813FB04FB047306C40504060406DF02710271026901ED01ED01EF193E1A65226522D21E72160D0E0D0EB8048E038E03A9020D020D02430207020702A603AB0271022B163E1AD41F911A911AE116CE0ACE0A5508A805A805C903D2016D016D013C06D107D107A802B702331D331D491E8E198E196311671367139A1047118F078F07F401E104E1040409B60AB60A34046307EB1CEB1C9C23641B051005103A1390109010D20F9F029F020901660766070D0BCE0A73057305790D1E166C196C197414501250126413DB10DB1001116A037603760397044D064D06DD0543054305CC0C7C16E416E41665192A19D108D1089011AC12AC12B104E905E905A2088F068F06D3098C09B808B8082321A21BF016F016BD189218921892188F088F080313D214D214E40B240907050705970A770B770B232123212321232194189218921892187D03271127117D14F009F009E709DB0CDB0CCA10AC0FAC0FCB1C9C1CE418E418E418E418E418E4180D0EA914A914E90E37113711540C000BCA10CA100B10860F9C1C9C1C8A1A8A1A8A1AE418E418E4186B0EFC1380178017AD120F130F13330F170E170E930D0212BA1CBA1C8A1A8A1A8A1A8A1A8A1AAA16AA16E117DE15DE15A913DE15DE15FD117011A411A411C414BA1CDC14DC14C3164C174C178A1A331933199817141A141ADD183016FE12FE121D126F166F166F166118F115F11526144B184B18CB19641A871A871ACC1A0D170D178D188D188D188D186F166F166F166118F115F115F11527163B163B161719441944196D186D186D1800158D188D188D182915AF14AF14871A261726172617F115F115F1154914FF1449144914001500150015001500150015031441144114F71826172617261726172617ED16ED164914671267125D123612361236120015B713B7133B123712', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (67, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4AF01AF015D0665086508B0087B0CE812E8128F12E810E81081149614961481138A148A14151354141C021C02B90AA008C308C308170C7C137C137C13211421149F141F13B113B1133814F713F71325149207190C190C190C320C320CE70B7C137C137C137C137C137C139313441444142913A612A612E61207070F0D0F0D420D650CF810F8101A14A413A413A4137C137C137C133C143C143C143C14A612A6127102EA0CEA0CEA0C1C11E010E0100B141D151D15A4130214BC15BC153C143C143C143C147F147F14B7023606C70DC70D140F3611121612161216121612164014BC15BC15BC15BC15BC15BC15D815071A6307D5027A0D4C114C119C13091509151216121612161216071607160716BC15BC15BC15BC15AE18DA09DA098808291129112913E112E112061506151F151F150716071607160716071607169C167618F00CF00CFD0C1E0D84138413F1126F126F1232154615461506150A160A16181616169C169C169C16970D430D430D840F121412140713231223121612F412C314C3140A162416241618160916091609162510EA11EA11D50F121395139513BF121A131A1304135A135A133415CF15CF153216091609160916560FEB11861186112D113E113E1193115A125A1267141016A714A7144415571557154E154E154E15860FFE10DA0FDA0F2D112D112D112D1136111E151E1593147E147E143112701270124E154E154E1502120B126B16541154115411541154115E11311531159B18C514C514131450144B134B134B135A13C414C414C414321732179D16BC15BC158D164019551A551A40124D134D13C51531233123D113D018C414C414C4140A132D122D125F258C208C20661AAF1BAF1B771A311931199A14B317A916A9163123A4166F166F16F615B613B6135F255F255F255F25EE1B971B971B6E1FB31FB31F5D1D9B1D9B1D831BBB16CC16CC16611789178917B41BB41B5F255F25191BA51CA51CEB1FCE20CE204020B219B2193B1E41148D173F193F192419B41BB41BB41BF71BF71B191B691D691D6F1DC11DE51EE51E491A321F321F37129617EE1AEE1A9D1B671A671A7D1AEA1BEA1BEA1B2E1B421B421BB01BC71FC71F5521A622A622', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (68, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4161216124F164F164F16BA13231523150616581958195819AF15CD14CD14B911BE11BE119312721463146314E214E214E21410144615EF16EF16D6131614161450167E0F7E0FC20F04120412C612D313E214E214E214E214AE13AE13AE137C137C13CF11531153115210490EFE0FFE0F4014321432142B155A10DA10DA106A13D311D311C10F3D0E3D0EB10FE610E610E6101510A00EA00EA913DA14DA14BF14071AFD1AFD1A7D18B91526112611CF109B0D9B0DEF0EC90EC90E200FA610A61041137213C013C013071A071A1F1A1F1A071A901490147710A60EA60E520F410E890E890EC00FE412E412C51123122312AE18AE18071A071A071A0115C212C212C212CC0ECC0E5B0E910D910DFC0D320C320CB40FC711F317761876187618761876181B15C212C212C212CB13CB13CB13CB13671367137710B40EB40EEF0E6D11C916C916F9169C139C1345129F119F119F11B112CB13CB13CB13D313D3139F13141214126D116D119C169C169C163C133413341334139F119F11B112B112B112B112021402143D1214125B105B105B10091609160916F412341334133413DA13DA13DA13DA13B112B11202140214021402149C149C14DF104E1509160916A412A412A412A4129C12DA13DA13DA131714171417140214021402149C149C149C144E15181418141814A412A412A4129C129C129C129C1217141714171417140F130F130F139C149C145A1318141814181418148014801480149C129C129C12FC12FC12FC12FC120F130F130F130F13871DD018C217C217C217C2178014801480148014121412141214FC12FC12FC129B119B119B119B11871D441A441AC217C217C21724152415241524151214121412141214E90EE90EC80D9B119B11C211C211441A441A441A2C182C182C18241524152415FA10FA10FA10FA10130F130F000E1A0EDC0FDC0F0C10E91D3A1B3A1B4D18361836182C1871107110250FFA107B0F7B0F0C0F5E0D5E0DD910B60FB60F4D10021ECC20CC20CC20CC20CC202C188C10A310A310D00FDD0DDD0D570E9B0D9B0DD910D910411141112C21E01DCC20CC20CC20B110B1104E1085108510170F7D0E7D0EB00D380CED12ED127D1229112911', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (69, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F00040723264023E87981A30D474000000000000000000000000000000000E6100000140014004548F472148314AD16AD1660162C14DE10DE109311040D040D7406AA0AAA0A6706A907A907AF0C460D900909140914541C541C541C541C891489140C12550E550E1E0E160EF407F4072F0A530F530FDB0E420AD016D016C81C541C541C541C98162016201620166E0F6E0F1B111B111B11B80D080D080D620EEA0BC81CC81CC81CC81CC81CC81C421A201620162016B10FB10F5F112110AD0FAD0F8E0BCF0ECF0E2A0BDA16DA16DA16C81CC81CC81C421A421A421A421AB10FF20FF20F720E000F000FD00F14101410590FDA16DA16DA16DA16DA16901D901D421A421A421A5A12EE0EEE0E5410450F450F610FBA0E030B030BF317F317F317F317901D901D901D901D901D901D4016A2113D103D106D11680C680C8B08EC05EC056D11F317F317F3170B1B0B1B0B1B0B1B901D821582151C15A30FA30F890FBD0FBD0F9D09790619086D116D11B611751D751D0B1B0B1B0B1B3C18F417F417B217550F02100210EB10C10EC10EBD0722086A106A10751D751D751D751D751D241824180E18B217B21751117B0E7B0E2D0F131013103B0DAB0CB910B9100215A0155D175D17751D581758175817581758177C1170107010A00F600E4D0F4D0FA80F9C14231E231E0815021502150215891789178917581767136713DB1166106610390E7C0E7C0E6D0C9C14231E231E231E3F1FAD18AD1862188917891789171B131B13F6115D105D10530F1712C911C911871D5D1E231E231E231E12211221071EDC16DC1689178B139D109D10EF0FC40EC40E0610500F500F871D871D871D871D051C441C6315631547174B154B15E714E714E7147011AE0FAE0F410E8A0C590CC211EF113D17051C051C051C051C051C051C60156015E714E714E011E011AE10C50DC50D8D0E8214001200124B174B174B174B174B174B173B120C122A122A120012D30DD30D33116E0C6E0C330F0E0F061006100717810B4B174B1716102511251166104F114F11710DAA0DAA0D89100C10671067106A0D7E18A81BA81B170E8C0B8C0BEB09B410B410D812D011CD10CD10100E8310831050104A114A118F0DF50B001C001C410F490D33073307970CA511A51122143E103E10D90EF90BF90B3E1030119B0F9B0F', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (70, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C38264023E87981A30D474000000000000000000000000000000000E6100000140014004548F490096B0EE60FE60F9B14B816B816DA17EF20B01DB01D5618B417B417331B301930197516C214C214420AB10DF50DF50D4E10CF15F917F9172B1C6F1A6F1A2022F21DF21D1B1FBE23BE23BE2399162316920A920A3A0D800F800FB011F114F114831657195719A71DFD1A9C1A9C1AA01DD219D2196417B20E5A0C5A0C900ECC0DCC0D11123013DC14DC14AD174B1C4B1C5B1DF81AF81ABA18F613F61397168812E90CEF0CEF0C780C39103910FA13601760178E1762186218B8156C1946184618DA14B113B1132513EA0B060C060C2F0DD40FD40FB3100F111C141C1435161F151F158415D216D216F7165F175F17F214F5068B078B07A708F90E7B0E7B0ECE10EC0EEC0EA90FCC10CC107F151F15341934193419A615A615190819087B087B08E90B430C430C1A0F3E0F3E0F4E0DD50EFC11FC11D4128A148A14FE131915191522086F0943094309BE0BE70D141014109B106F0D6F0D150D10111011B00F6B116B11A2120C0EBE0B820B820B390BEC0BEC0B030EB40EB40E6910151015106C0F8A10CD10CD10421076127612D413BC128F0C8F0CF1084C084C08820AED0D1D101D101610D211D211C611C511C511D50F8A108A1079124012500C500C5F0ADA097E087E08670D470F470FAE0F51115111B411B811101210120C12D311D311D113F90C190B190BF709060A060A9F0E7A0F7A0F760FFD10921192117011311131110B12D311D3118912A90C7C0A7C0AE2080B0AAF0EAF0E5A0F7F0F7F0FC11081128112F010C30FC30FD70F600F89128912590CAE0AE107E107D0072A0B2A0B9E0C141014102D104D104D104D10360FA50FA50FEB0E161116118214B20F0D0A0D0AFC072A0B2A0B2A0B2D0D7A0F7A0F2E109B109B10110F2B0F2B0F88138813C60E0E0F330F1C0C1B081B08ED07F609F6090D0DFE10FE10FE10430F810E810EA20F88138813881388135F0E5F0E5008160816089B084B074B07050A480FBE0FBE0FBE0F5D0E5D0E2D10C411C4115812690F5E095E09C008DA07B506B506DE06E609E609900C731073107310690E690E03104E123D123D12670FEB0B61096109E107820782078A070B090B09E409F30C420D420DA90D0E0E0E0E4C11B110B1107F0E', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (71, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F442186B18B612B6128A12C710C710F813831383139F129F119F1112116510C20DC20DC80EE00DE00D2316AC15AC15AC153A1194119411261381101710171031124A114A11AD118D0F8D0F550F0F0F0F0FB20EDB12831183111C11CD12221222122E13801080100711F110F110B61072104B114B11A310FF102D102D1083117B0E7B0EE80EAC09AC09C60B59115911F115C1101F101F10E510C714C7146613F510521A521A2F1CB610B610A5093C0AEE0FEE0F4011CB14CB14CF116D116D119811F110F110B310B5102C19332533253325B018B0183B18D114D1145511B112B112B6117710E010E0107310C010C010B8105815371737174C17FF16FF1600194F184F184F189511090E090E1D0E9B109B10A411621262129B0FCC109B119B115B11D513801080101B1286128612901119101910FC0DE50D051005103212B210B210BE0B240E3B0F3B0F2F11711171113C10291029102110EC0F030D030DFD0A960F960F7D11B511B511BC12F710060F060FA90E1F10AD0FAD0F0A110A110A11C40F6A0C6A0C650FF20CF20C0910230FDF0FBD11BD11490F3E0F3E0F0C0E1B0E1B0E850E0A110A114C102210E10DE10DE10D3E0B3E0B0F11C310DE10DE10600F600F600F1F10A5113D113D11EF0FFD13FD13FD13AD0BAD0B710A680D680D4E0EF50D1F10AC10AC10AC10910F910FA5112313231309131916191697169614CD1BCD1B0C0D380D380DCC0A1A0F0B100B102E10E50DE50DA70B410FD40FD40F51177B187B189C14CD1BCD1B4C13C711C711910C4B0D210F210FDB0D3E0ED70DD70D6B0FC80FC80FC61705190519A61EC91DC91D441D0F133E153E15C60EC10FD50ED50E3E0F990E990E780F7E0F7E0F7A157A15FD14FD14F31C3D1D3D1DBE1D7617761788136A12E011E011A90F7E0E7C127C127C12AB11AB11FD14FD14FD14A216A216A216A2162813E8166A126A126A126A126A12EE11A60FA60F5C115C115C118011741129122912A216A216A2163313B513BD10BD10BD1044124412AF0E1A0E7C0E7C0ED210F80CF80C31126C126C120C121A131A13A3144210980C980CC00C800D69106910150EFE10FE108610C416C416F721F72180178017C514C414C414D114', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (72, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C00663264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4680F9014901448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D70D8815541648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FF10CC14D41748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F5105211981448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F427112711861448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F497109710351048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42A106310631048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F4115C0E5C0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A0108B0F0F0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DF0FC0111F1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C3103710520E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DB0CDB0C820A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F70DF70DDE0948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49706C10CC10C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45E086408640848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F4119F099F0948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E8161716A80D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B5137E15F81148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F426122612720D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F459155915571748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (73, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F42C142C148D1AAE19F117F117EA172A142A146B178A178A17CD15ED116F116F11200FE114E1144A153217C81AC81AE41E0A1B0A1BF418B21BB21BB51C331A331A331A6A13071307131A14061606160616FA15181B181BD91CD21B4C194C19E119C81BC81B4E1A331A331A4111661366134D140616A515A51539166519A319A319B319291629162916A41BA41B321B5A11410E410E6A0F670F670F2E115A125A121A165A17141714179B16CF12CF12CF12CF12291629165A115A115A115A115C135C132E11641159157B117113B311911491143614CF12CF12CF12CF12CF124113411341135A115C135C135C13A013AD1655115511A510911491149114361436143614361441134113411341134113CC12CC12CC129313C019481048104210331008100810E312361436143614A411A411A411A411A41113129012D814D814761A820E5D105D10EA1041104110B110610F610F5210E60FA411A411A41113121312B6129C139C13B318180EF20EF20EDF0F61106110FB12D71217141714C7139A129A129A1221142114C712131638163816F00C3D0E270F270F2511291429146E178D168D1692139A129A1221142114B313B3137C1D4D194D19A614561410111011A1116614661451177515E412E4128B111D171D1721145A135A137C1DE120E1209A118914A3127D127D122D12E612E6124212920F920F4B0F1D171D171D1738137C1D7C1DE120E120110D110DA60CF20CF20C90106F0F6F0F1710D0102D112D11E910AF11AF119C16181818186A1B521B060F060F9E0EAE0EAE0E9010D8108A118A11241354115411C411001500150015021BCF1DCF1D031C930E940E940E160F5E105E10CF104011401124132413241324135F171B151B153923392339239022B410FC10FC10DA10581058109211D1103F103F103F105F175F175F175F175F179B1F9B1F9B1F9022C8123A12D90FD90FB51201130113C2123F103F105E1083198319831983199B1F9B1F9B1FE021E021BE18C5128A0E8A0E901213131313C212C212C212C2128319831983196120612061206120E021E0215314CB102B0E2B0E0E0E1D0FD711D711C212C212C212DF15DF15DF15931D612061208C1E8C1E8C1E', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (74, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F433153315E114F313F313581358138C138C138C138C138C139215921592159514F718F718F718F718D115A314A314A814271627163217461246124F12AF11AF113B147C132D152D157415741574157415A61ACC18CC18361C0F1C0F1C6116D51AD51AD51AB9113B143B143B143B143B145F17C117C117DB1524178D1B8D1B1B1C8D1B8D1B8D1BD51AD51AD51AD51A001400140014971297125F1755174C184C1859157C1AD61DD61D8D1BDC17DC17DC17C816C8162A164416B812B8127318DC12DC12551755175517AD16E51CE51CE51CD519E9143A153A15481348134813B71DDD14DD145117EA13EA137D12C2129213C019C019C019071907194E153D153D154F12041D041D9A1CBA1CE519E5198D189B149B14CC135213761A761A131C8D1B8D1B25157F157F155615D714141614162619D015D015FF157E147E14B0135611761A761A331F331FD011D0112216201620163B15871587159515AE16AE160A173016C514C5142E14C417B71FB71F331F221222128212D112D112A3144115661666168A1688158815AE174D164D163117BF1AB319B319BC1640178E148E14871490169016CB1C581C581C0A145E155E157116E716C715C715E120E1202A1F2A1F1A1FE117E117E117CE15CE150D1AA2199C1A9C1A09186B166B16191790169016E120E1206420642064206617661729174C137F1D7F1D7F1D89188918AD1BB41BB41B431BFF18E914521B411BE622E622E622E6221216121610157F1D7F1D7F1D4B184B18F41DD71DD71DD71D85181F164F194F19E622E622E622E622141F141F28132521411C411C5315F41DF41DA816F816F816F9181F16131A131AFC15B613FB09FB097C24601560155B1C651A651AFB0E7B0C7B0C730B9611571257128116BD1AC421C4213317AC21AC216C1A931E931EFB08FB08931E931EB407540854082604F20AF20A7511E021A81EA81E9316531753175317F11106110611931E931E931E6C0658045804690153045304FA0AE021F918961896189618710F710F7C1119161916061106110611C707C707B606B606F60E751075108C1E4C18961896189618650F650FD60FF30F06110611E30B680E680E680E3C093C097513C115C115', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (75, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F443134313D01326172617ED16ED16ED166B15A416A4165D1849159F139F138114C615C6159413941343134313301512171217071724171A191A196519181A181A751B6D1B6D1BC31B9D189D189B149C13BC1489178917B11A431943191719831A831A951B311B311BA91A181B111C111C84189B149B141515271CD41ED41E7F1A41194119011DD41DEF1CEF1C191B601A601A9516CF16CF16551759155915BE12891B891B891B651A081B7D1C7D1CCC1DAD1DAD1D9A182D162D163517121CB219B219E814B712B7129213E213891B891B8118011B011BD71A3619361920162016E517E5173C18381A381A8513051405145213E8130E180E18A119261B8419841936196F0E6F0EE509D110D110AE135F115F110313F912CA13FA14FA140217F116F1164C14481448149313D611D61147153E14CD13CD13F812521352135F12351194119411BF12D813D8131511DC116E106E1035123B123B12F31226142614C512E811E8118E0F0E0F2315231539137511C410C4106D1043104310CF10FB10FB109D101713B913B91364134D124D12B90EE515E012E0120B121512151224129E119E11F90D360EB00DB00D950F81148114081494139413A7130F15D813D8131B121B123A123A123A123B0E3B0EA70DC20DC20D2210091409146513481381158115E914B0141B121B121B12E912E912D7125E105E108B0F1910071107118412181418146C14651465141F16C6145313531376140515D614D6142C131C131C13A810761176112212A313A313BC1288130A171F16C6147213991399134914D014D014171347134713CB127F117F112E12C6134B124B127F122A13651565159912A713A713E213FC13FC1380128012F311F311AF13AA14AA14B014061206123512EA12751275127D10E613AB12AB124B120E130E139313851385131A14CF14CF1420149A12DB11DB119613B10BDC12DC1263148C128C122412361336135413F215DD16DD16A116F513F513B112010F010FC5116A0D9B139B133113E010E010DD105510E712E71260160C160C167915B113B11326114610FC0EFC0EF2113D109F109F10FE116911691104124D134D135915DA13DA139A141F149F129F12701017101710', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (76, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F494130618F8192D1C2D1C5B19AE1EAE1E071C041C041CF91A141A141A04177617981F981F04212F2141154115DE1AC81CC81C8C1C661D661D931DE8195F1A5F1A4717471747174717FB16FB16241E571F181718175317EC1AEC1AD019BC1DF21BF21BAC1B3E193E1968186A176A174717D214D2145A1787195417A618A618FB1C9E1B9E1BA91B2C1A2C1A621A8018801868182A13A912A912C9126115611507156D16D117D1176519C81AC81A0719501AEA1AEA1AF318B314B3140511051105114310710F710F3F0F1915F71665196519A2174316431641172A192A19C319E014E01438128D11C10FC10F760B710C710CCA133F15A917A917BA19E919E91941179117701770175E18981298121011071007108E0F7B0E7B0E351196146D186D18161AE919B017B017F018F018F018F01833133313C01212101210B30F9F0E7516651665164418381C381C381C381C381C22190D190D190D19F0182C122C125810121012105C0C9E1342174217F118AF1DAF1DB51CD41CD119D1190D190D190D190D192C122C12410F090E090ECF0A660C2218E81AE81ADA1C7B1E7B1EBD1DC51BC51B1416A815A8152316DA0EF30CF30CC90808070807E407501A0D1C0D1C391D2C1E2C1E2C1EDA1ADA1AEA139814981498141311BF0CBF0CBC0AD105D1052F05B318811B811B031B9B1C9C1C9C1C071940164016CA168A148A14B313B40FB40F3806D304AD03AD030A17321BE41BE41B321D321D321D321D86188618A3183318D713D713940F580658065803CF05CF052A130417161A161A501C311C311C311C2024DD18DD18511814191419CD10CB08CB088D057503940BEA129A14601A311C311C311C311C311C202420242024202420241B151B154A1303080308A4034E054C154C15451C2D1C2D1CB91C151C151C9B239B239B239B239B23511651169411820A820AB206D405D115D1157D17E31ADE1BDE1BF81B761F761F9B239B239B239B23441A441A7913A30D5809580958098E150D180D184019FA19FA194D1B761F761F761F761FBD20BD206D206E186E188807580958091F064A1407180718BE1FBE1FBE1FBE1FFA1FFA1FFA1FFA1FFA1FFA1F1C184D194D19C107FA04E004E004', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (77, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F42F21CD1D611D611DFF14D711D711791079107C117C11860FE20FE20FFD1252135213701E52135213571F1A263523741C741C1A15C815C8158A158A158A157015261426141716F61E911491143E1B3420FE1EFE1EA924A924A924A924A924A92416161616E11AE11AFE19E20DE20D0B0B9C0D9C0DD70D5118D314D314A316EB13EB13A924141CED17ED17E71C391D391D8D17791279124212241224122412B21D0511120812082F080B0C0B0CD013E81BE81B9C21FF1FFF1F2320C41E420F420FF813E014E014631B051157175717D6098E158E1524192419381D381DAA25E41CE41CD223782078205223B91BB91B6F175710711DEC1AEC1A9A1EEC1EEC1E801D532253220E21AB1BAB1B7A245D22F61AF61A951B481A481A75164217B317B317921EEC1EEC1EB5216025AE20AE203223E022E022B6174C134C13AA11511251129E136012D616D616791C791CE620E620B8209B229B220E240F180F18461B4013AA11AA1161067F18220C220CB20E791C791CAC161F171F17EB207725772521248A24DA0BDA0B1B0AF609F6091D081D08810C810C840D5F115F11E01AC514B321B3219624A721A721E4159A149A149F0B1D081D081D081D088D07990C990C0914CB14CB141E14A01EA01EAD15961896184C15B81AB81AB81AE80ADD01DD01DD015006C70AC70AA416A61EA61EB021DE19701A701AFE164C134C13E410E904E904AE03BE05BE05630C940B77177717A61BA61EA219A2193E15AE14AE14810F2E102E10EC100B04E703E7036D0820082008940B4618C71BC71B6A0EF009F009A40AEB05EB051F07BB03F504F504320499059905AC04801080104E05AF07700E700E640D4F09A602A6025B0345064506E40284008400ED0024042404E407FE0D80100F070F07A8077E057E05F302FB00FB008F01DE04DE049805AD00CAFFCAFF2202D307D307C60CA90F4C074C07A607BB05BB05F202BB00EE01EE01CD02CD02CD02E90076007600F000B201B2019E115205DD05DD05370BE408E408E4088901320132012203680868087B05FC01340134015D044D044D04C3048B04B805B805E40866036603CB02E000E00048064806E905E9057A07CC08CC08890AD109D109B005', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (78, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F44F1C881A1717171753114E0D4E0DA50FC114C1141F140311670F670F4B0E4D104D10B011861186113420B319FF18FF18660BA510A510A513641494159415AB10FD0CFD0C460EC910C910F910C611D80F51189A152A1058105810BD150D170D17881492179217210F720F720F7D11BC1045134513A712F30F2B152B152B152B152B15591A441844188E1BDE1FDA17DA172917141414144511DD10DD10F110550FED12ED12FE13981B761D761D181DA81FA81F7C20F317F3177916AC12AC120811510F140E140EC00D0114C913C913F71AC322C3229C220E1E0E1E7F1FE120AF14AF14EA1020102010FF0F180F180FEC0D481AEE09EE09861106230623981F9017701F701F301B521B521B8213C00FC00F1E103F104E0F4E0F7F187F187F187F186D216D216D21361636163616BA16BF17BF173A117B10231223127F10001100117F187F187F187F187F187F187F185517551759135913A21258115811881065116511B6101A131A131D081D087F187F187F187F187F187F187F18D813D81334128F0F8F0F04111F126B126B129D1415131D081D081D08260826087D0D7F187F18F00D990FA210A210B910161116117113A012A0124112E4131D081D081D08F808F8081C0F4410B90FB90FDF10111111113D11AE10AE1034136E11030F030F9B0A440A440A440A3A0D3D0E3D0ECA11CA11CA11F4114811481194119D11DC10DC10E70F0A0D0A0DAA0A890DBA0ABA0AAA0AD30ED30E2510B515D312D3126712C811C811D011860F860F3D0E590F590FEE0F890D5B0B480D480D5C107E107E10B2138116811629166514651484143E1357125712B9100E100E108010B30AFE0CFE0C701B701B701BC21B97165E145E148E13B215B215B6152F122F128C10C310C310A90F800B771477140C10701B701B701B701B701B701BBE14AF15AF154C165B13BB13BB139D117E119109910997105E105E106411391039100D13701B701B8E16BB16A216A216DF129B129B1212125012F408F4086B0D440A440A950BD41105110511FD12B713B713441719181918CD1481128112261226122704080808089016901690168E10C714C714D713391239123D134515FB14FB14C011491249122612', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (79, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4FD0F180B180BE1099F0804080408960A600960091D0AB50DB50D0C11CC0ECC0E781035112F0F2F0FD80F5909BD07BD07D608D608D6085F0A4B094B09C40B050EE610E610681214111411BF10D30FD30FF30F7808EA08EA08A80879097909700923095B0C5B0CD60B631063100D13AD12AD121613A8101610550F060A1809BB08BB08B7094A0A4A0A5909F109F109E20933133313CD14EB13831283128B0F1311B40CB40C5C09500A500A7C09C50AC50A490B0F0AB50AB50A0A114C124C126B119611961182101E11B60CB60CA70C350A96099609E708920B920BA80C250B250B720EE90FE90F52100F0F961196114210030D9C0A9C0AFD090A0A0A0A5E09E00AE00A6F0BE90BAE0EAE0ED71186108610730F3A0F3A0FA50EE40D110A110AEE0A5A0E1F0D1F0D2E0A450B450B6B09070C070CDD0E20122012C30E380E710F710F7C10720CC60BC60B600D8C108C10D60BDC07DC07B20DA10CC50CC50CF2109C0F9C0FCA0F321032101513630F7F0B7F0B840BD70BD70B670E0E0AB20DB20D240E4F0E4F0E250F121112118710DF0E3B0FE413DA0E350A350A350A410AF50DF50DCA09B20DB20D950E3E0E3E0E4B0EE60FBD11BD11AD12B4103E0A3E0AF607010901095208320B320B020C980CB40CB40CFA109C0E9C0E7D10F512F512CD129412B40BB40B2608C0099C0B9C0BD408C308C308150DE30DE30D730E790F790F4011661137133713A3173B10810C810C600B0C0E0C0E960B0D0B0D0B860CD80EAD0FAD0F7D0E12121212C70F731273129A19E40F6B116B11A111931193119D0E5D0BA20CA20CF61072117211D311FB13FB138712EF0FD619D6193C100D10F611F6111D1342124212520FCF0ECF0E5A14F610F610F610FB1312141214E310611B611B7E110C1046134613F212CD11CD11880FCF0E5A145A14CD145A145A14D3143D143D14221184178417501230108F10641264129811C60FC60FC80F8B128B125A1462146214D314E4149D149D149B165B169C119C1101127A157A15C31289118911C50F8B128B128B127C147C147C14D3147D127D12B314A115261226122612E613E6137A158911AA12AA12AA12BD14BD147C147C147C147C140B12901290129F12', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (80, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4A20C750B750B1E0EC310C3103C15341B341BF5156B19F721F721F721F721F7216D196D196D19CF14780F590D590D96107B139A189A18D317971C971C2F18F11FF11F7924C41AC41A6D196D196D196D191610BD10DD11DD11A313851D851D1F1D461F461F6824FA22C21BC21BD21DE016E016AE1EBE1BBE1B1311AE111C131C13EC186A1B6A1B6A1B391CE719E719741CC617C6170F1AAA15AA15BE1BBE1BBE1B1E11F911E916FA19FA19941BD515D515CC15F915F9150C16A116C80DC80DE31103140314DE189614AE12AE12421AA21DA21DD31B9316931650143016C115C1153D17BA0FBA0F160E491449141D181D180E160E16A11EA11EFD19FD195211FD13FD13B015C115C1151E1415121512EE0DF61049144914F20FC012E917E9179319FD19FD19E11373107310F1153E15A214A21469141D111D11F610521752172316CD11B816B8169B177B16FD11FD112E11951095102912EC12EC12FF11241124117A10F210731273123B0FE0142B172B175F171617161797114B144B143A14B614970F970F7D123B113B11220F1F121F12B410B41352175217AE16201420145D0F0615AA1BAA1B5515BD13BD1324127D127D12EE0FFE113E1394129218551BBB16BB1659141F141F143316D21BD21B1C18BC1BBC1B42116512C90FC90F1D1137164A1C4A1CC11C3D183D180314BC13BC13B017C11C3B1B3B1B921A8D1A8D1A7712651265127312F022F51BF51B4619D01A8B168B16CF1734173417DA14C11CC11C611EAF1DAF1DEC1F5E1CFA1FFA1FFA1F081D4E194E19D217BF18BF18C016F21CF21C0812D31CC11CC11CFC1CD31CD31C281E17201720FA1FB41CA018A018A818551B43174317F21C421242123D1642244224841FD31CD31CFC1DAB20A920A9209B18C4188C188C18D617B117B11789190E120E12A01D1524EB23EB230124D51CD51CD020CD20CD205B163B16A916A916C916891989198919AC162E1D2E1DEB22C123C123EB23D423D423751D751DD220A11566145B1510151015B6134C0F4C0F9910DB25DB25DB25DB25DB25DB254323921D921DA21DA21D1F131F13E313A314A3140E18B818B8188A1DAF1D97229722DB25DB25DB250225B924B924EA20CD1D', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (81, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4E313C218C21848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AE1D7C1F7C1F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AE1DAE1DAE1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BE1BA81CCB1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49614AA15571D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EC1AEC1ADE1A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FC19FC19A21948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F423162316A21948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DC189D169D1648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DC18B217B21748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43E130021332248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43716D925F21B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F022F022792148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F022F022792148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46E156E15FA1F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41720FA1FFA1F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CD20D020D02048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D220D020731648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A21DA21D2A1848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CD1DCD1DCD1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (82, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4A310A310B50E740E740E250FD711D711F211981498149814DF15931D931D931D2F1B2F1B7E1A8C1E940D940D740E840E840E74104611F710F7109814D814D8142815F719F719F7190F1D0F1D7E1AAB20740D740D740D740D10121012FF10741074106110611061108512F7190F1A0F1AFF1E7A1B7A1B0A1BDF0CDF0CDF0C740D101210125E1210121012A1126110601260128F12F91DF91DF91D231B231B941C4B0DDF0CDF0CDF0C8A10E310E3101012A112A112A112531353134D14F91DF91DF91DF91D5F215F21F80CDA0CFE0CFE0C8A108A108A1098129812981298125313BA15BA154C1D4C1D4C1D891D531E531ED50F600DE50FE50FE50F5E10461646169812781578157815781578154C1D8E1D8E1D291AEA1D85199810CA0FDB0FE50FE50FE50F461646164616241324137815B618451A451A6C18CF1ACF1A9A1D36198C0F8C0FAF0F211021104810101910194616DF16051A051AA217751875189E18EB16EB1663188415BC10BC10370E420FFC11FC11791AB91CB91C1E2190229022A724A724A724F11C9E183B173B1784153B11F80FF80F130FB213B2132416B81DB81DE91E9022B523B523B523651E651EDA19A219A2193B17B3114A0F4A0F060F251825181B16301DB91DB91DCC1ECC1ECC1EB52314201420BD1E4F1B5A1C5A1C0C15891145114511C917F917F917331E6B1F6B1FB31CF01BF01BF01B4919081708176219BD11BD113916D015CB17CB17C1194F234F230623371DB320B320171C661C661C9919BC17BC17BD17A213A213501A4915511DF81AF81AB81B51195119E01A3C213C214F1CFA1AFA1AEB1AB11D501D501D0115D913FC15FC15E919BD1DBD1D9E1C331933198719F71C281D281D2D1B491B491BA01C891D891D691CFB1A6B156B154314171D171D51236123FC17FC1720190B1C0B1C3F1B371B371BCF1AA81C5F1C5F1C261C5815901590158E1C7F237F232B24031903191818D31CD31C8F1BCF1ACF1ACF1ACF1ACF1ACF1AE9171419FC1DFC1DFB1CB522B522BE165314F917F917A31793159315AA16BD16BD162515CF1ACF1AE9175616D617351F351F9F1FCD17CD174011EF0FEF0F8E176A136A1332176217C016C016B71453155315', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (83, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F48C1E3B183B183B183B186E14540F540F0D0FE30EE30EB410B410B410CC0F7910CD15CD154615A711AB20AB20AB203B183B18C4155E155E15B01393139313B410D710CF12CF123112B015B015F3116910AB20AB20AB208E218E211A1C4B18AB13AB13DB13371537151915AA15AA152214461846183E140A133E1CDA19DA19DA191A1C1A1C561A4B184B18D7162515251583167517E117E117E6194414441444140020DA19DA191513E511E5113B12C51387168716FB1559185918F71796179617F5193E143E143A14A718A718A718D914D914D411D4119E13D714D714F415B616B6161C176117591759178213D613D6138519A718DA18DA187A15D914D914D9140B140B145B155A16DD12DD1283145A145A144914491449143619B80F561056101A10D91412171217EA17E216E216D516A817A8174416391439141F11D511E3135D0F5D0F1F17B80FB80FB80FE30FE30F951631173117CD18DA18241824184518551355132412FA11600A600A000DCC0ACC0A070D780AC60EC60E94131B171B17E616DA17DA17DA17A314A314A2120612530F530F1E0DB20B860D860D0E0E6B106B10F11148154815291682149D139D13A41361136113911373134F174F1751149A0F9A0FB510E718E7181C130917BE14BE14271227122712BE12D012D012BB13BD0FFB0EFB0E0F0E631032153215BE17CC19CC191A1AA217A217F813D813D81314128D14DB13DB13FC13C90F18111811C11160136013F6187C1C7C1C14196615E913E913D813A011A011981481148114D9132013E310E310BF16C318C318341E131F9E199E1923161A111A1190100A130A132C134418B116FB1A49144614871487149B1A061D061DB51E8B1C8B1CDA190E130E130E1381120A130A136B1379150D1C0D1CF017871687166D1B991D991D721DA01CBF1BBF1B961397129712930FC80FC80F7D11D512A01BA01B3319D617091809189619E318E318B21A3119311935146815681568156F0F8F0F8F0F4410E917E917E917E917A01BA01BFF150E170E17CE187C173C173C172C152C152C159C10D410D41094119114E917E917EF175E165E16C01894165014501469133C113C11DC112C152C1558125D135F135F13', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (84, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4A711860F131013109B11741174119311CB130F130F1357132B152B1528137C137C133F0FFB0EFB0E6910A30F0612FB11FB11CC11A311A3118514771577151A14DA13DA1370152115C810C810040E1F10A60FA60FE510CC11CC115A11411141111B127A12451445140A1634143414F5145B105B1087118B1295129512E011F811F811BC110E1183108310E212A816A8167D1692139213B81479125E125E12BB11BA14181318135413081308130011CA11CA1156161319131942185413C112C1128412F20FF20FD90FBA144E134E131C12591259127111BA1132183218FF18EA18EA18801441134113D5120D110D11520FDF15F615441444146E13E310E3104A12221522151A196F186F184716AD130A130A134C10880E880EE31308147B147B145D105D105D10E11252131C181C185F18941794179D11421242127D0F620D620DFA114E11AF10AF10CB0FAA0F5A135A13E81346174617CC1AF115F1150714FB0EE50EE50E1A0D7B0E3A123A12EF11CE0FCE0FFD1046134613D3123D143D143A16E81205120512750F990E990EE90DA80E8A158A153214D910D9106C111F132115211521154F154F1570103D123D123711ED0FED0F2E0E8811341490139013901310111011D8130715071521152115211513121312BD10BD10910EC30EC30E1E126813681368136813AE13AE13D0141F150B150B150715F711F71113123F0C3F0CA50B1F111F113E16E7136813681368138513B614B614231549164916C6163F153F15F711F60B210F210F0011A213A213B116FF14171417149416FD15FD15CA14231623163D175917F711F711330EE90FE90FBD116214621479158E13491349137A13AE14981598157F1B271B271B1A1933153315280F4F104F1074111013E214D51299136113D813D813FF131613161345180B190B198019D117E111E111B6109310931016117914171217124F14951595152C1416181618AE18D4188B178B176117ED14ED147C0E331033104E110C14771277124913E81568156815A118AE18AE18AE1883158315CF161B1492109210940FE110E1107113D116B918B91818161A161A16C719D519D519C7194A153C153C150D141E101E1065102F112F115410', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (85, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F47012251625182518C319FC18FC18FA1FFA1FFA1FFA1FFA1F7A137A134611290629067E05350535051F103115BD17BD174C19891789178917A218FF17FF17D117941094100F0FC805C8054F044203D8028B12CF182D1A261A261A5218301830187316731673167316B110B110CE0D7F05A103A10369035C0356175617CE188D1A8D1A0E1957185718AE18AF16731673166513060E060EC907AA04AA041E035C03451345137417141A031A031AB819701C701C701C701C701C7F14890E890EB40771051E031E031E032513BE16BE166B1A181B181B051BD818D8183B19C3191D161D162C14F208F2089F055B045B041E03EB11011601167A1B411A411A251A941A4A1D4A1D4A1D4A1D4A1D32194A0C4A0CAC065B045B045B043912CF18A81AA81AE619EC18EC18F0182D1D2D1D2D1D2D1D2D1D3219B712DB07DB0779075B045B047B0EC614691A691AB7190C190C19C5182D1D2D1D2D1D2D1D701B701B5415B712B712790779077907A80EB7120119381838183318421842188E1E8E1E8E1E8E1E701B701B1917FF0D541154117907C801FF17FF179A18091B091B091B091B091B83179F1F9F1F9F1F9F1F19171917730EB60BB60B8503B503E317E317601939193919B916091BFD18FD18C41D9F1F9F1F9F1F3519351964114D11850385036B0A78164A184A1875185C155C157C18A81AA81AC41DC41DC41D3A1F3A1F641364134D11E309E3099D08EA17A417A4172917F816F81681168817691D691D691D3A1F3A1F3A1F9C149C14FB0D140414044B020F16871642164216B315E516E516881788178817E21905120512E60DEF0C92079207B104BA01BA01E214E21445164516B916021502156715F114E014E014B912720E720EEF0B7C057C056901AB01AB0179146714E515E515DA16E515DB16DB1647158A148A1423110B110B110D0ADF04F501F501C6012601161C161C161C65166516C316251625162516F612F6125B11990E130A130AD90548084808C6050702161C161C161C161C161C161CFF158E148E14B81011111111A00DD30ED30ED30EB107B107D4065109651C161C161C161C161C161C1915E713E7132F134B124B124511F50ED30ED30E4E0DD406D4065109', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (86, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4FC02BA03BA037603CA0294029402C6004806480648069E099E09730E691169118816440F60126012D802D802130213027202470447046C000100010006014B08730E730E7810881688169914991499145C035C032202220201049904CC00CC004C00BDFFBDFFB701E607E607880EFD0EFD0EB514B514A8125C035C030D01AC00AC00FE00FE00FE005200DA00DA005101FC00020102017301720F720FAA1060121E031E03B200F300F300E5FFE5FFE5FF2800B300450145013901B601B601D40191099109960C94161E031E03B200B200620062007900BC00BC00A50040014001FF01AC09AC09CE09210A8C0A8C0A00175B04DB01DB0156012C012C0184019D019D01D4016A03800580053506770677064B10481048109317DB01F801F80163027502DB0CDB0CDB0CDB0CDB0CA10966096609350644074407550F3B1420172017C8019902120612061206DB0CDB0CDB0CDD0EDD0E90119011880C880CBB0B1D0D1D0DB711801F801FC801F403C004C0040507960A960A960A560BAD11AD11AD11C10EC10EA010240C240CB30CF524F524B5037902C004270427047604960A960A960A0C0D0C0D950FC510C510E91514150A110A11870AB809EB08EB08EB08FC03FC0399039903990320033C04D704D7044A06790A790A8814DD0BDD0B3C0A7607DA04DA04CF081804AB07AB07AA03190219027602BE03BE03EF04790A790A790ABB09A90EA90E0E084001050805083E07CB08CB086C08270127016901AE040D050D053706E20FE20F3E0FC30EC30E9A0D4002A301A3017C009A009A009A00D504D504D5048302250B250B950E950E950E5E125311EB0FEB0F4002400222FF22FF31FF62006200EE04D504D504FB004C0B4C0B950E790D411041107A11BF12BF12260126012F002F00EC01EF01EF01EE04EE049D009D008A0F60166016B6133C123C12D912F011F01107020702D201F300F300EC01ED07ED07ED071F031F0377148517851770135A1161136113F0112511040504057A02920192018F01AF0FAF0FAF0F5403671167110412E211E2119B10F811F8119B11D10F5109510951091105110553057A027A027A02ED09DE09DE09E211E211E2116F1168109A0F9A0FD70F', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (87, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F41E12B90DB90DEA0D9710971004100414E817E8177213CA12CA121A129712971298112F122F1298114C135A125A12E0118815661466146513A417A417D212C112C11271119711D811D811D81198119811A81284147F197F19B5162C1F2C1FFF1CFF1CFF1C6C125D13F511F511D91125122512D811D811D811601200167F197F197F19351B481748179E18741574158D14A211A2116312721172117011BA11D8119E169E16AC1475147514710F331033108F140B120B1276139F1398129812ED12891189116F1140116718671830131B131B13B413DA13841284124112041304130213461246129B110111011101112F1114191419BF168B14C313C313B013771377135C1329172917291719132412241294103E103E102A1131190924092417151D151D153215641564152D0EA312AE1BAE1BAE1B241624162416241224122412801F09240924901F901F621662163012DE0EDE0E8C10BE16BE16BE16191119118E132412EC12EC12F5249914E116E1162517D615D6159115B10EB10E4A104811AA12AA12A2108E138E138E138E138E13B809110D281628163816F31340134013970E8A108A1013125211521122113013301330138E13741376073E0C950D60106010CA1106130613EE11B210B2100413EB12221122113013301330133013031323092309480BE90DE90D25113714371401141A12EB13EB13E213EB12EB12EB1223132313E71203131F0A1F0A4B0C150EAE10AE10E510611161119812EE13EE133F143F143F14EB124913C012C012E412A71163126312EF0F4F124F1218117D107D105E12BC123F143F143F143F143F143F14B812B8120C13F412FD11FD115010A110D30ED30E8310B310B310EE12BC12BC12BC12041304139912A112DB12DB129E121E11A910A9105610C10FC10F0A106F106F10BC12BC12BC12BC12BC12041204123C133212321225113611D211D2117E11F410F4103F102410E710E710E11144114411F711F711F711981455129712D10F430FC10F1B101B107B0F7C107C108310BE11BE11AF11281128113012571163126312F7115910290F290F95100A100A10B41170127012D01014114A104A10B70EA40FA40F321163116311EB11300E', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (88, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F49811AD12AD12E713F512F512BE106F116F11AA12321332131F132D131F131F13C415C415C415C415981146134613DB12CE11CE11BE106F116F116F115C12841184111F131F131F134214F313F313C415D811FB11CE11CE11CE11CE11CE11E00FE00FE00F6F118411841184110E0F241424143A1370177017D8119112CF12CF12FB11CE11CE111F10E00FA110A110EA0E84118411AC0EB212B2120914AE14AE144011DE11F211F211FB110110F10EF10E1B0FA60FA60F490F420F420F630F6A12DE13DE13301483151C111C114C11850F850F2210960E960E880DE40FE40FE40FE40F790F790F8D126C136C13DD137713031103110E10E210E2109B0FB50CB70EB70E270FE40FE40FE40F5D0F5D0F4B0E66116611701284122B112D122D122910920F920F5E0DE60EE60E720FEC11EC110C130C13BE0DBE0D0C19DB12DB122B128B12651465147B12AB0FAB0F5E0D5E0D5E0D5E0D94120C130C138F158F158F158F1534163416BD1BEF1140144014251225121B0E1B0ED70D94129412941288138813FA17621A2B192B19451A571D571D74139413251125115510F50DF50DD70DE215E215E2152016FA17FA17FA171A221A22D21AB317B31703130313381138115510551055105510E21584168416E215C719C719BE19D116D116D4185F19DA19031303130313851085105510551055107D167D167D16BB1F861ABE19BE194C1D4C1D4C1DC81ADB19C012C012F110B010B0106310FB11FB11FB11EE16BB1FBB1FBB1FBB1FBB1F7C1B4C1D4C1D7E145714C012C012F510C6109A0F9A0FFB110D120D1266145A165A16CA17EE16B212B212F411650A650AF81036128B128B12D2111D101D10631005110511FC148915211E211EB4151D161D166B1BF30CF30C8B103612F50FF50FD00E310FEB15EB157713A524A524A524A524A524A524B915901D901DCA23251F251F9712C80FC80FC80FEB15EB15EB154B16A524A524A524A524A524A524A524921D921DDE16731373135910F00FC80FC80FEC114813581658163F1E071D071D9A1BC21BC21B4519501A501ADA14B7147915300ECE0FC80FDB11DB11431A932093209320931E931E3618981AC119C119AE1B8A1D8A1D9A14DF16', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (89, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F40E130E13BD13661231143114FA188A1D8A1D8C1D741D741D741D2C222C2202250C22F821F821E5217E16541654167C17BA1ABA1AE019F815F815CD14A3145D1D5D1D5F1D7B1B7B1B881A76217621E521B1171918191805195D1C5D1C4F1C9917B315B315B315B115B115F8197B1B7B1B941A981A8E1A8E1A17154A17371837181A19491949195819FF15FF15F31515181518DD11331219101910CC0FC20EC20E83154015A115A115291B291B291B4E17FD18B418B4185F184B164B1632145B165B169B0EF10DF10D7713E2127E14291B291B171377147714F0179B199B19681A691769175119401752135213960F7120601260124A13C112C1120F12AD12AD12B615A717F715F7151816CA18CA187219E113E113A9129020111311132F132F132F13121108119A129A123A147B157B15C4170B150B1574127110AE0DAE0D340DBD1B481C481C2B147E127E123C0E560F560F71117116711671163B11A812A81268160B0B0B0BE6071D17BE1ABE1A451B5C155C15B010311113111311A210121112113C11F810F810D11077057705A006641B3D1A5C185C18EA16341734173C213C213C21A210C010C010A90D830D860486046704CC01CC01DA19561C241C241C831BEE18EE18EE181717121612167611860F860F5F0CCE03CE033A040C060C06DB19B216FD1AFD1AFD1A541D541D541DB926B926B9264B24770C770CA411600BEC06EC06EC06B9057F127F124113A816A8161C1E1C1E1C1E25159C204B244B249320932093207E18A11EA11EA4119B09870D870DFB0F4B0B4B0B6C0C551C4B1A4B1A9C209C209C2014230C220C220C22A11EA11EA11EB30C701273127312090A5D0E5D0E750D4B1A4B1AE7249C209C20142314230C220C220425ED22ED22ED22CA1D53135313501B1A1C1A1CA31A91159B159B1556185F175F17C11757185718771FF919F9194C154311B40F90179017911A921E921E3D1DDD1DDD1DDD1D141914195618A61A26202620AF1F771F771F79157915EF0EEF0E54101B101B100D145D14D11ED11EE619761B761B211CD91AD91A3A20D21BD21BDF16A30C130E130E1B101B101B101B101B101B101B1072143E1A3E1AE619E619311B311B4121BD22', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (90, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F446224622081E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C121C1219C2148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CB1C6521652148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A71AA71AA71A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B51AB11AB31A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F471205C205A1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F490209020912048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C007C0079E0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DF06DF06D60648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D6056105610548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41D04F104F10448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EB05D903160548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B9058C04AE0648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49B09F304370448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40F070F07090648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4750F750F9D0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43418B614B61448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EE1EF815F81548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D91E871EAE1748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BD22BD227D1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (91, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4261BB81C412541250D22881388134B11D40A221022104313361A361A44186B156B15501787168716B2190E1F891F891F891FA212CE14CE14EA0B4F0D4F0DC9106C156C155D154A1502180218B8151316C91BC91B9F1D361436148A140A120A12E30B781078101D11631163116311971452175217E715DE13811B811B721C8B1A8B1A2512EE11D611D6112312D012D0125A12D212D2124415B614B61440135D13341A341A4E1FBF1FB313B3130A11F90EF90EF911D011D0112012311331153115E2126C136C136713CE1B831E831EE621381F381FE414E10DE10DB00EA1100A130A134F14FD15FD15EC13391439144014C11AAC20AC20D71F111E3E183E1864116B116B11AD12C214C214BB132F162F16BE1510143A163A165312671973187318C71921152115641164116411E2141716601560152D130714071482148A188A181120F01D2518251863166E1F661C661C5F10BD14BD14B216A215A215C613071407141D1669180119B91E0A1ED91CAB16AB16661C1C1F1C1F661C311931193119B214531353136414761676165A193E18FD1EFD1EF819F918F9189C181E171E17371731193119311931193119311934159E189E186D193E18471947197C162E1B0E1B0E1B211857185718BA16C815C815311931193119EA15AF159E189E184315021C5C165C16C01D8B1D8B1D5C1BAB1AAB1A7B17A61601150115E814401740176B156B156B158916AB1A151A151AB119FE18FE18371A76171B171B17891641154115E5149E169E167015371543144314D71A551A031A031A3B18F617F61790166917691790159816FC16FC16891508170817D516581358130E1A0E1A0E1A0E1A8C1782178217B1149316AA15AA15B216D814D8140413CE15CE151B176814C314C8190E1A0E1A4F174F1796154F174F17F21642154215FC143A143A148F133F152E152E1540143B14C819C819C819D417D417D516EE16EE164F17A315371537154D14DD13DD138B13551455145315061441134113E614421642168517E91776167616F7131F151F1542151F151F15CB1642177216721609156D1253135313E614F516F516F5166D156D1548138C158C158C158615EB16EB16D417CD14CD143014', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (92, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F49715321643164316B515DE18DE18251943184318F515AF12AF127D0F9C0FC90FC90F0910D112D112131621163C153C15EB14411D411D411DC815F515F515EE139C119C112B10F60EF60E011024152415DE1315162C15F513F513441944194419C4187A127A12541280118011A10F431055135513CC169218A715A71539152914291467142215221522151213141214124212D512D5120A1385168516CD17C618241324138F167D167D166116CD152215221522150D120D126813681368134C132715561556156714C616071807184A171F151F154417221522152215E512E51264147A13711671164B14061406142214F61802170217DD149A129A12DA124F12741274127412641464148B168B168B16AB158E148E147214D0188E158E153F144F13BD12BD12AC1247134713CF12CF13CF13361A361A98149814401514161416011915142A142A14A712AE11AE111212431371137113D516ED1CED1C2415961496142D1E771877183E183B1415141514CC107110821482145915F216F2168D20E21EE21E3F156E162D1E2D1EEE1D2D1E231323137411FC12FC1267125B155B15C0161E1F1E1F7B1D6D1FD415D41505161E1A1E1A8D20242242134213B6115412541236143C168D188D181323E717E7177F15801480143515871B871B641BB11DDA15DA159613F7121E151E15F616B11EB11E1323A217A217181575176A186A18B51B2B1C2B1C641B6E1484148414E013FA13FA130C1E831D831D4A1585177C187C1845176A186A18981C241B241B241B43140515051522142214F21CF21C6A1F581C581C3D18E11AE11AA11A1C171C17241B741B15211521C314C314BB14BB14811F7D1E7D1E2F1E9C1A9C1A731B621A67186718ED19351C351C511C541C541C3B146A16D814D814811F811F52215221E1169C169C1643185A175A1746173C173C17B3185B1CA61206142616AA145C1E5C1E5C1E63216321DD1553145314F813161446154615AE18DA18DA18AB1AC91FA113A113BE15951C951C991E632163213915481AA715A71515146F156F15CD19001A001AC91FC91FF713F713F7131D24991E991E991EE81DE81DEA1AA01AA01ACC1A4B1C4B1CB214031AE41DE41DDA10', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (93, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F404188E188E182D18E31651195119CD19131A131A02164A154A154A15711071105E0FE50D7A107A10641518197B177B177617691969197119C619C6199B164A154A154A157110711071107110CD13CD13921828163B163B163D1610171017DE17C619B616B6167D167D167D167110711071107110CD13CD13C618F414E91496159615461749174917D217651665167D160E130E133A123A12FD14FD14CD13D713361436140E15B715B715CE15D215D2159014161503150315AE12911291123A12FD14FD14FD14FD14A913A913AC13D412BA13BA13BC15381438143C14EF13EF131813911291129112EB16EB16EB16FD147D1388138813E212EA12EA120913F414F414D814011477137713791361156115CF15EB16EB163115FB13221422149E1398139813DD13C814C314C314281482148214AB13CD15CD154F16AE187C137C13171A4118BE14BE14DB1343144314B714B714B714ED16AE14AE14E8151B1A371A371A1619971597152D1E6E18251725179B18CE19CE19F3140E17951C951C151DF920F920AC1BF11BF11BF720F720F720242277223E1C6F186F18641AC31AC31AEC1C441F441F681E8B1B8B1BF51DC322C322C322F720580FC022C022F62268236823AC231A1C1A1CC31D3E1D97219721B11D27192719631F0F1D0F1D11192E18CE1ECE1E241F2821282191259125B126B126FC1D7B197B19DD1D811F811FA420601480148014EA11B31D7F217F21972295249524D823DA22DA227A1DBB19BB19D91D232023202320232023202320D218ED1C101A101A8E1C4B214B216F2432259F199F199F1EE822E8225F255F255F25232023202320D218BA14C216C216571CA51EC51EC51E711E171E171EE822E822E822E822E822F221F221D01A32123212A6125313521252127B125D1E5D1E3E1C891B441944192C16E822E8225314D01AD01AD01A1F141F14C91FBD122F132F136E1470188C1E8C1E691B221822182C162C162C164F0C52205220522052203E09C91FC91F6C102C112C115E20391F391F081C5D195D19EF17470E000E000E52205220522052204A10531353136C106C106C109B0FD10FF315F3155717EF17EF17000E000E000E000E000E000E52204A10', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (94, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4651C651C651C651C161C161C19150A140D140D14B814821082106E1185118511501007090709830B651C651C651C651C651C9E149E145815C814C814D413841384133F14501398139813AA0BE90BE90BCD134D1683168316D715A214A2140B160B160B16A915D0146D176D17471D95129512E50FE50FE50FD7134A164A164A169016D2150A1A0A1AE9194517451745136D176D176C1CB116B1169416770E770EFD1485144A16681D681D431E6A166A162C1597149714E613BE13A422A4222C125E125E12F010F010851485141815C017C017311749154915EB17F214B419B419531C201520156B1095109510F010F0102714271403186516CB17CB174B17BB15BB154F1AF01EF01E1811BD15C210C2100211581258125812EA160F160F167717531E531E5321E823E8232D192D19C110C110071172117211BE115812581258126A1529172917141DDA25DB15DB15E823DF1EDF1E7C163B123B125A10A011A011F81173131C161C16F720BC173C253C25461BD022D02211175E205E20731B4D11711171110713731373138C1312151215580FE514071307134815D022D0222F1D3411E112E1129E129E129E129E12081308137815171701172E18EA129A13DA15DA15C50E79197919890E760E760E3912391239123912BE15D018D01805153012821382139A13DE0DDE0D490E791979193A0E620E391239123912471347135515491649160517C818D218D218D218E30D720F720F1A0DA30EA30E740E110F110F6A13A714A714FF1607178C188C188E1BD218D218D218B10D220F220FA916D20FD20FCF0FDF0FA213A213A714E211E211161292139213BF14980E1D0B1D0B100A500C500C211DE90CB10DB10D0A14151715177E1670137013B916BA1311121112410CB2097B097B09BF0B3F0B3F0B900C140F140F1814F715F715A51AF518851685167A160C130C133E09450A450945091409A807A807930E7010AC12AC122B134A184A18FF1CFF1CFF1CFF1C3415D0134A104A104509BF0ABF0ABA0DF80CF80C0E1113151315EF13991799176419FF1CFF1CFF1CF81985184A104A104A104A104A10C710831083105C10131513151315131564196419BE196419641907188518', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (95, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4830B830B830B3F0569056905BF05D201D201F103380238026409FF09A311A3111510AC0FAC0F1A0FBD0E830B830BBC08BC08BC08C0050003D201D201D201C807C807DA07D30CD30C66108B0E8B0E2D0CE50FD607FB07FB070B08630A630A2A042A042A042A04D610D6102B12871137193719241853265326770E650ED50AD50AC00F4B0E4B0E1010DD078E0D8E0D2A0EE711E711011ABA19BA196118A61AA61AF010610FEA13EA13C00FD912D912D912670FDD0DDD0DDD0DD710D7102B125F13331C331C4818461F5F105F10F211DF14DF142A15C913C913F6130715071507150D14D214D214FE14681C681C7419F31BE610E610A914B218B2189D18C612941394136114CF15CF156B150E190E198618681C681C11119412EB12A217A217311BBA15BA15BD1271127112B811AE11AE112B17271827182718030CE80CE80CC70BB317BE18BE18FF158D118D11A30C690FDE0FDE0F96102718271827182718271827182718271883177B1549114911D714A314A314A314B1104A0F4A0F800F6F0F6F0F590F27182718271831182617261701179C0EA314A314A314A314A3148915A314A314C60EC60EC60EC60E050EF10DF10DE50E091109113012930E071307138413E51E021902195C18751575153115EE10EE10C40D180E180EDC0E2B0F9D0F8A168A1676137A177A175D194513451369151F161F166312D111411141113611311031102F106B0F161C161CF71AF71AF71A8613CD1414151415831558165816B0117C117C11D311AC10AC103610A50EF71AF71AF71AF71A081908198C1665156515AC1559165916451638163013301370132D132D13ED10D417D417D417D4170819081908196B136B138C162415BF15BF155711FC13FC13B013B612B61270120B1387148714D4175D175D175D1766140D140D1475137513751387115412541286149C149D149D14D0135E15DA16DA16111571157115B5117D147D140C127513751375137513611261126D148515851585182315181618165E169114851185117211E810E810981039113911751375137513751399116D1085182A148616801680166B15941294120511FB10FB10CA1061105F105F100511A30FA30F54107610', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (96, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4A40EA40EEF10AB109A129A12B612BD11BD115811E70FE70FFF0BD511D511BF117811EF11EF112010170B420E420EDB0FE210E2100F12071107113F101B0CB30AB30A631042114211E61120132013050F5326150D150D391539153915C10F60107F0D7F0DEB0B860B860BE80F9E109E101A1181117510751053267E26D517D517411736113611DA10640F640F640F3B0F3B0FA90FAA10C513C513951167116711461F6613D31DD31DAB15F611F61116114E0F251025103D10F411F411B410991299125612FC0F6711F31BB51BF81170127012B811F810F8104510E910E9105B107D107D10C2103112C911C91194109F10D913D9138811671167111312171117117D10451079107910E710C010C0100D12521052105210971193129312E6109C109C1004109110FE0FFE0FDD1034113411DD104C114C1138119610561056100812A612290F290F221037103710D11108110811E6103A113A1159127411D310D3109F1056105610FB117D10C50FC50F6210601060104A149314D712D712DE12D712D712FF12AD12AD127A117A117A117A1143104311BA13BA136411AB10AB10F111EE12EE12D712D712D712D71284117A117A117A117A117A119D0F5C11530F530F4E0FCD0FCD0FB210A1118D138D13D21124112411BD104A104A10A110BE0FBE0F6B0F870FB90FB90F3A0D8F0E970E970E2F109A119A112C11C610C61074111310180F180F180F2710F50FF50FE30F0F0E0F0E790E7B0F7B0F0510EE0FEE0FC00F3F104511451197124B0F4B0F3D0CB30F0311031157107C107C10060DDE0F330F330F18102A102A10E110EC0FEC0FE70F370E000B000B4F0C201144124412DC0EA10DA10DD90FBE0EBE0E990FAE0FAE0FC30FB7106E0E6E0E430D950C950C250EA916141514159812F20FF20FCD10BB10FF10FF10BB12BB12BB123811590E590E1B0CAD0CAD0CF90E2315901390138313C21082118211E91019111911EC12BB12BB12AD107E0F900F900F470ED50DD50D6D10791084108410A41078127812A0111E121E1267118510B00FB00F580F350F350F5B0EBC0DBC0D76107710C210C2101C123812161216121611E710E7100E109C0F9C0FCD0FD6108D108D10050DA80C', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (97, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F48E0E8E0E3E0E7D127D121613F81961236123B01D121B121BF415741874182A19D711D71172109E0DAB0EAB0ECD0D4814331333135F18B417B417DF1A17191719CD16AF12141814181B095C075C07180587102D0E2D0E0C11C114C114A81732143214660EE70D6A0F6A0FE80BA40DA40D950DB40AB40AD306A00E0F0E0F0ECE0FA01129172917000E5D0D5D0DDA11800F800FF50A1B0C1B0C950D1310801780176711671167116711BE129E119E115C10600E600E6D12BF11D114D1140B150F0A0F0ACE1C682368239F106711F911F9118811F112D512D5120913171517155915361336135E18D517D517611E931E30209711511278126712671267125413541378153B123B12751520144F1A4F1A1618CD20CD2083217B25771577157615021502151B13671267125414B812B50EB50E4813931393137A19911C911CED1DB420951595159F16AB16D818D818BF12E213E2130B1444144414F11332103210F01BF01B470E470E97183D1327142714041438143814DE13B516B516A4163E17F414F4144F12A21FA21FA21F45144514C50BA11283138313061213128D138D1326198D178D172D164D154D154316041704173917AA1BFC14FC14BB10F911C712C7123D15A915A915A716FB18FB181F178218A918A918721498159815F218B214B2142710F3110C130C13E2132B142B1435151A17C113C1132718271827187E18ED15ED159F16C8112512B30F830F5915B312B3124013531553156115E316E3162F1C2F1C2F1C2F1C5219BA15BA15041C571C630D630D0411B414B4140C133A143A14C713C314761976192F1C2F1C2F1CDF19461946194F1B2A1C110D110D9C0F801799189918FE1572147214EF1676197619FD1976197619791946199A189A189A18450F1D161D164716C116C116561609190919EF164317761976197619191619166914701670169A183D0F651165112919291929192919951AAE1AAE1A6418831B831B0E188E188E186914701670167016BC0E0E13BD18BD184318041A041A3725372537256D1DD719D7194317711871187118941694169416A80C8F0F7D1B7D1BA51C811B811BD51D8421B019B019131A2E172E172E172E172E17941694169416', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (98, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F474087408240324032403690451095109540954095409F00DEA19EA19EA19EA19E619E619E6193D19610261023E019E019E016904F602A200A200EA00700370035702EA19EA19EA19EA19EA1945169B136B02B702B70213021A021A020C00DEFDDEFD6EFD700370030E016C01B902B9022E0D36153615DE258017FA0FFA0F5C06AA03AA03EE010E0168026802680283FF83FFA100A100A100AD06AD06AD06AA0A8C1C4B124B12EC10B906810481046A023B013B015E011E011E013C00A10026FF26FF4E054E054E0530208C1C091B091B300C3A043A04B9039F029F02A5014A01740074000F0033003300E4014E054E057B257B250E240E24E70D6006920492045C02AF02AF028B01720172017400F7FFF7FFE5FFE4010BFF0E240E240E24C914C9147C0AD605D605CB02690269029B027301CD00CD0029017C007C00C4FF01FEBB1ABB1A7A23191D191DA512B311280328034103FB01FB019B0281028102850188018801C900C900D225D225D2255922D518D518BB0DED08ED08C603F702F70269026A020D020D02F401E300E300170137164821482148214C194C19EC12BF0BBF0B62082C05F703F703E703AA03AA039501690169014902C11048214821482148214D164D16620B390B390BC00532053205E903EE03C202C202FC021D041D042512841AAA1AAA1A091AD312D312370DD709D70927048103410441048E03C403C40367061A071A07571C661CD21AD21AD41D6A1B3D1A3D1A3D1A5D055D05E50363036303ED03AD03AD03AD03F206EE0A2A1CBF1C9F1E9F1E9F1E3E1D3D1A3D1A3D1AE20EE20EE308E30897059705E402080308038A0473089A189A189F1E9F1E9F1E3E1D3E1D291529152915510A510AE308B704B704B704B704B70483047D079A189A1879147914041904193E1D8F188F18A514A514A514A5140D0A42064206B704B704B704F40670169B169B167914FD17FD17D818A418A4183415A514A514A514C308C308C308C308C605C6055A0A70169B169B169C16BA18BA18BA18AB16AB16AB16AB16F10DF10D6B0A6B0A6B0AC308C308620762079416441844184418C714C515C5151A15AB16AB16AB16FC0DC909C909C9099B089B08C3087C057C05', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (99, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F43D19DB1EDB1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DC1ADC1A082148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DE25DE25EC2348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43A0D2B212B2148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47F080419041948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A606A606520C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40BFF7000D40148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F401FE28FFE3FE48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48C018C01AE0048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F435023502C00148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F449031004100448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E1059908990848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EE0A5F12870F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EE0A5F125F1248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F473086916371348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4020D020DE00D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48B0C8B0C020D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48B0C2E112E1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46207460E460E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CB07570C570C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (100, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4481253135313C2133B143B147B152C159B139B139815B416B416AB167915791560176C146C145E148212A913D513D5131C13C013C013AF14011501152116F715F715E71622168E158E1522130D150D15E414A616A615A615361389148914C2176018CF17CF17BE16731673162614DE13DE139713B717B717131632157C137C1376138A13311731178918C117C11785178D168D1663143B146313631375192120A416A416AA14A414A414301333153315F117A917A91742164416821482147A1449144914F3163F1BF116F116B5148614861412166517491849189A1747164716EE1564136413C513631363137E162A1865186518831713175C145C14B116E916E916B9173C183C18A31781166D146D149712081708170F190418311831180418E219E2198F17CD18CD18AC196F1902190219F41743144314A5122E112E116F15871495169516CA1677189D199D193B1BC11AC11AA619DC15DC1541141310DE0FDE0FE40FC70FC70FCD0F2311C810C8100012A214A2149213E613E6135212BD12EE11EE11340F260F260F860F56105610EB0EEE0EF60FF60FAC101F12E811E81141127E0F7E0FC10EA60CA60CCE0C460E460E89105011A312FD0BAA0F14109A109A1018106C0F6C0F420F870F870FCE0EE60F6112611261129F109F103612E511AE0BAE0B6110A112A112A1127C0E7C0E450CF70C4710471072107C107C100312051105114A11A4120F0B0F0B860DA20E3B0F3B0FEF0E80108010BC0EF60FF60FF30FD910D910A110AD10B010B010A4129710971097106C0D340E340E780FC910C91095117B0E900F900F041122112211101110111011FC12CE1097109710170DDB10DB107C115B10090D090D1A0EF610F6107C111C121C126A119312FC12FC1205115D10A30EA30E9A0FCB11CB11240FD70AD70A660C8E1101100110BC1108120812D612F812F812560F060FA30EA30E3C0FF20CF20C130C0C0A3E0B3E0BBC113211321184106E126E122C138B136E18300EAC0E640E640E640E7D0EC60CC60C800D910D910D5D13521352132611E9122013201350147A18941294128D0F310F310F2912050E050E8C0D980FDC12DC124D12B112B112D412441444149514F419', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (101, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD25404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4E712D717D7173A21B516B516841692159215FB1B3120311E311E4B1CBD1CBD1CEE1E4D1A4D1AC619FE25FE25FE253A210E1C0E1C801BAF1ACE1BCE1B9D20B322B322291CEB1EEB1EBE1D981D4413441321206E1D451D451DFD1C7E167E164E18EF18EF1825193D1BCF19CF19BC1BBC1BBC1B6D1D0D130D132120C91EC91EC91EBA17FB13FB13BA1CDD1DE61FE61F1C20E11BE11BA01EA01EA01EA01E181BA2143F1B28196419091409146315BA1CBA1CBA1C76207620D71FDE20DE20D11FA01EA01EA01E861E1216541754173F17BB15BB152D198B1C8B1C6C1B8C19DE1FDE1FDE20A61FA61FDB0E971797173E1261092B182B183F179A179A17AC178B1CF413F413E9173D183D181E18410F410FA10EDB0E391139113911FA17D61AD61AD61A96199619EA16C615C615C615451845181E186A146A146A146A154011401136102D15A31CA31C551C332033202219C615C615C615501650165016CD186A146A146A156A156B146B146A15951659185918332022192219231723172317E31750165016CD18CD18DE19DE19DE196B146B14A3125014EE17EE1755181917191728192317E317E317E3177E237E237E23DE19DE19DE19E614E614E5110B11471547150816191728192819371997199719E3177E237E237E239A1BA11EA11ECA1F631C101410145514AD16AD16C516201820182819911991199119E81DE81DE81D5E1D41224122851CB31CBE12BE12151815181518AD16E71823192319562456245624E81DE81DE81DFD1AFD1AFD1AE113851C171317131518151807170717271833183318562456245624AA1B271D8C198C19FD1AE716E716FB13FC12E316E316E316322032206B15781A16171617D21F1F1C1F1C6D1B111A111A8C19511151119C11D812CC19CC19B818322032203220C21FC21FC21FD21F7F1A7F1AA81E3E1C331133119F0F360F360F6E186E186E186E18C826C826C826C21FC21FC21F661A1221071F071F7D1C1C1B1C1B5514490E490E7A18501866236623C826C826831B831B6E18971B971B0D1C951E951E951E951E951E2E10910FC513F419121A6623662366238B2163226322711FF519F519091C5F1D951E951E951EEC0EEC0EA20ED715', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (102, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E325404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4E10DE10D670B670B1A121A12190EB414B414AB166D0F6D0F3506000E000E000E000EC10AC10AD10AAF0B040A040A100848084808760BFB0AFB0A560A8D066E066E062C0694089408160FD70AD70A0D0B300B890A890A910AC90743084308510D030F030F8408C005C0052C06FD06690C690CF60684078407A214EE0C030C030C6B09120C120C1B0D8C0C8C0C2E076F0670067006EF08BB0ABB0A2D07C706C706121612168B0A8B0AEE0CDE0C8F0C8F0C580C830D830DDC09180C180CF80C9E089E08C1089409E8096109850AEB0A4F034F034803390C390C0D0D1D0D1D0D350D3212B80AB80A9A0CCD08CD08B809AF0D3A123A121606B102B102F1027B027B026F02690345124512C711C711C711C111C111C111B6140B111A101A1022109110000700079B026A026A027102EC01EC01F901C111C111C1119213671367133B196B14E90FE90F91100D0A0D0A130AD309D3096E02EC01F101F10132075A0A5A0AC111C111C111AD1277146B146B142C0F3D0A0C0A0C0ACA09D709D7095E04CA03CA033605A50BA50B040DA50E2D0D2D0DE4145F14920D920D920D080508051909BD09BD093B02F603910691067B0BE10DE10D7B0F74137413631C23164C0A4C0A4C0A550455043A0194001E061E068E0546064606960EE10EE10E1E13E911900FB31C23164C0A480748072003E502E502E502740374037C091A0A1A0A760FF60FCF0BCF0BE50E8E07621562154E09900F900F0410410D410DBC12F81406160616B40ED80FD80FD40EE60DE60D580C7A03621562158310430F430F170D460E33113311B514121912191416DC13DC13320E730BD404D404FA03B20B670A670AB70C1D0B1D0BCE0BEB0BEB0BD80DFB18081708179E15440D440D5805880488048804BC0B7F097F09E6082D112D11B811F1112A0F2A0FF3191C151C159E10D10BD10B5F0A7B0388048804AF0B970C7B0D7B0D7B0D7F107F100F1665166516C51FB41CB41C990F8B0CD503D5037B037B037B03C5130C0E670E670E670E0D180D180D18DA12C915C915630F7D0D7D0D270C27052705D004C505C505D7159A167F167F163717FF14821382131D0F260D260D8708E20AE20ADD0CF004DB04DB041E09F50F', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (103, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4D10AD10A4A104A104A105F106312241324131315131513152F15131513152F1664193E163E164E130D0B030C030C48118B118B1133119C149C14B014DA12DA1269131F15A618A6181B18F316F316B617730BB80BB80B8914C115C115C1154E19D614D6148712AB12AB123312C711C7117215421542155618A0079D0B89148914D8157818781897157E117E11D41005110511E60F0E11ED10ED103711DD12DD12E809990D84178417F81AAC1BAC1BFB16E210FC0FFC0FF70FE60EE60EB40F461046105B10CC10CC10AF0D5811201220120415BF1ABC14BC1457111D101D10F80EAA0EAA0E5F0EF50F321132113711EE10E70CE70CC515A214A2142812311331131B118B0F8B0FAE0E010EC80EC80ECD0F1E111E113D11EB10E518E5180A19891689168910CB1343124312030EFA0CFA0C030FF00FF00F11107510751033104210C013C013E3160E1C261426146D1386138613521044104410DC0F6A10D30FD30F070F450E450E4010D010FD18FD18741B181A181AF0165216FA14FA142A12261126119A1130103010B30E2C0E2C0EB110EA10C311C31189153B1A741A741A1B175F155F15E2144D134D131611E41074107410251092109210900FA711B015B015D11325172517AB1356145614B513501441114111BB10B10FB10FA00F271027108E079B097B107B103E12B612B711B7112F12BC13BC134815BB12BB12D2116E0F6E0F320F9C0F86107A033303030EA316A316E319DC10DC105710AC11AC115D13DB1080108010DD10F50CF50CF80C1C0EA205A205390D2C1B2C1B1B1741106B106B106D1041104110BD0F5B105B10CA10B00FB00F470FC10FDE04DE04BC0D8219581158117611890F890F7B106A106A102C113F104E104E102C10E310E3102B115206C80EC80E67123E123E12A812700F700F681088119B119B112710F00FF00F830FA110A1105510FD06520D520D4613D81748164816D40E40104010491196119611B7118A0F8A0F890E9D0FA70FA70FFA0AA410541454143813A811A8118B0ED50FD50F4D1150129211921104124A0F4A0F360F7B0F7B0FF50FFD1161156115D314C612C612EB0EDD0F3D113D116012B412B41240112B112B115F11B40F9C0F', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (104, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F42713271344125F155F152A1520132013C2125611161116118D10000F000F250F6D106D104A11B311AA17AA178D155815581558152E14CF12CF1287123A113A11D31008110811ED0F051133123312C711BB1681138113D5122F122F12F2125D125D129B134B12A512A5126812641164112F113A103A10D0102B16F911F91145119410CE12CE1275130A120A12E1113213321377129A119A11D71277108D0F8D0F2C12B011381038106810951095106C11700F700FC20F081485148514A6129A129A12F20F6D0E6D0EEE10AB1160136013C90F2D102D106F0D800F9C109C100C1485148514DB11710F710F350E070E360BEB10E411EA11E20FE20F170F940D940D560EA90FA90FFA11F10FF10FCE0DCD0D0D100D10800F0F0FD110D110981189108910B50F08100810D41092103A103A10150FC10EC10E3B10FC0FFC0FBC10B61038103810D910190C190C460D1D10621262120012FE10FE10980F91119111FA0F600FCA0FCA0F7710B110AB0EAB0EC1091F0B1F0B1C1025122512BB12380FDB0FDB0F821032103210A50EFA0EFA0EDB0F940F2E0D2E0D2D0A300C300CC010BB10570F570F1710CF0ECF0ECC0E7E0D7E0D920C720BA00BA00B100F450FC00AC00A4A0C6F0E6F0EF30FEA10EA10C80FCF0ECF0E7C0CD10E070D070DBA0D6B106B1086103110490D490D540C460D460DE60FD710AC12AC12BD12A711A711661266126612AE13591159111C0E630F5A0E5A0ED10E120C410E410E94118F128F1217161214121498128D128D128D12FB118813D50CD50CA80D4F0F4F0FDB0EFD0CFD0C751086124E174E17D513AD11AD118D11A411A411770F1111F50FF50F580EAA0FAA0F4E0FC70DD20FD20FAB11931693161414CD11CD119313A411971197118E145110120F120F1B0FE40EE40EEE0EC20EC20E3811001400148415FF14171217123E124A114A1154128F0FE60FE60F620DB10DB10DF00D170C580E580E0F1182148214D314731373133B114C114C111710FD0DB20CB20C610DFA0BA50CA50CC70CED0CED0C5E10C913C9131A15441445114511F011E311E3119C0F850DCA0BCA0B3F0B240A240A4B0C230F771177112012BC10BC100510060F060FB7109B109B10', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (105, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F0004072326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4421342134A120612061298119E119E11770F9F0F9F0FEE0D0D0FD10ED10E2310FB0EFB0E560FE40D4A104A10B711CB11CB1105100C10C30EC30EE90D100C100C490CF70CF70C810EBA0EBA0E410F190FD310D31082108210C40FC40FF10E3F0E3F0ED70B300A300A750B1D0D900E900E3C0EC90DC90DF40FE10EAC0EAC0E9D0F470F470FFD0E6D0D6D0DC50B740BA50BA50BC60B010D010D7C0EC80EC80E1313E70CEB0EEB0EDC0FF00FB60FB60F250EF40CF40C890D750D750D390D190E830E830E060F18111811360BAB0CA70CA70C940E160F160FB70EE50FE50F9E0FFD0D0D0E0D0E360EA70DA70D180E4C104C100F0F360E620C620CAD0DE10EA610A6102511DD10DD10BB0E0A0E0A0EAA14EA12EA12380EE30F421DB610E9104410E710E7106210971097109710E611E611C30F18106E156E15E4107C127C122A14DA25A111A111A310AB10AB10E0104C114C114C1102101F101F104510C51AC51A651C5D1A5D1A5D1ADA25461046101711880F4D114D11E910DD10DD100210B50FB50F12114F104F144F149610CB1DCB1D461CB110420E420ECA0C1F0E1F0E5D115D115D113C151F16090F090F541001130113A513251725177D15E40D150D150D160B830BE814E8145E1A311731174515C911C9117B115E165E16A31749125E105E102F13D7129E149E140118421742177C1E7C1E7C1EFE1A2C13531253129612961296121613D913D91388137F12C112C112921809180918E918E9180C180C18D314650D650D79124E124E1271104E1087151111D41313100B170B179A189A189A18961AEF18EF180E199A18F910F91022134E104E104E104A12081308132511DD10DD10EB159A189A189A189A180E190E199A18791379130F12941094100510F70C3C103C106410A811801480149A154A154A15FC15BF14BF14BF148C148C14D4145315291529152D153D102D102D109E11D615D61578157815781547154A154015401540158C148C14D1149A1A9A1A601D7C1107120712A5120B150B150A0F590DC70CC70C030F351035109A11461A461A461A26159C1F9C1FDD106C1160116011DB11180E180E870EE90CE90CC00B9A0F191019103A12B713B713D017D019D019', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (106, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C3826404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4E40D131219177E1A7E1A6B1826182618301A57175717BD17801780172E172E17A316A316A316A31689188918BE18371C371C6D17A715A7159C162B172B172B17801780178017A817DF16DF16B516F01489188918371C371C371C5B179712B917B917E617E617E6175716801780178017931A041A041AF01968136C196C19E0192517251769168F178F176818E617E6175418D619571657169918041A041A0F1B471221122112A9166E186E18231C5C19C018C01868187D1C7D1C241E211E211EEC1A011B0019001948114F134B134B13F0142D152D15EC19B01AB01A5B1B141D141D081DE820711A711AE11B9F189F18421D421DB114B1142C13821282124616B718361E361EB71DE61AE61ABD14D418D418EE182B1C2B1CDA259A16231623167115A5148F128F12351643164316881D811B811B4D14661408190819E9194B1836123612121D9E169E16FC1424132413BD12EA12EA129A11811B33113311C213E414E4148B16E3157D157D157B1D831F831F131E44170E150E153D13A111A111371112111211920F970FF510F510FB0E7D15D314D314F4161018101890166C156C150D14DD13DD13A51113116C106C10280FFB10FB109F0F650F291429149714B81BB81B8E1B5A172E1B2E1B6813DE11DE11E6118D118D11BF117B137B13DD0DB50D1B111B11FF12F716491D491D9B203D183D18D51AFE14FE14F0154612751275122612741174118715FD157C167C164114B720B7209B208C1C8C1CA01AA01A8116811616139011901111123E133E134A124714FD15FD1584137E1D8D188D183812E809E8092810CE16CE16481B481BF210F210BC12AF18471447140322072207225D2295239523D118930E930E960B2615D314D3144A1F44184418870D6B15C117C1179F20D621D621DA21AA21AA21AA21E2205D1D5D1D931C8E148E147A1A43194319D51FBC225C175C17E81FE81F88208820AA21D621D621E220262126213F1E931C291E291EA620461D461D7922A117A117A117E81F081B081BF71BE81FC220C2202621941F941F1110AE24AE245B13BB11BB118C172A12A117A1172D036C03F409F409C517E720E7206822A020A020AE24DC10530C530C6E13A715A715', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (107, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4A31644184418441819188A110E0E0E0E590C840A840AFC0D9B089B089B08CB08CB08B70779059B04F014AB15AB15BA21BA216014F411F411390AF309F309D508D508D508D50889066009600960096608C115C115CD14D41DD41D5E1D6A106A101E0A220950075007980892079207F70C86198619FA18C50B111A111AF019F722B61AB61AB61A42114211FB0A05060506FD0585058505E6109E14FA18FA18CB0EBB1D301B301BF722B61AB61AB61AB61AB61AE910E9100806080653059D0D9D0D93121E121E12260C9F189F189F1841240826B61AB61A9214962296226A11B811B8112311551555157015DB12A018A018E718391841244124082608260826C722C722C72296229622F717F717CD192F1A2F1AC01EF31CF31C4B183918BD1EBD1E371A371A371AC722C722C722C7229622481C481CC81A1B1A1B1A621744165E1DE315D315BC10371A371A371A311831184F1857175717481C481C481C481C1B1A1B1A1B1AA0078706BF0EBF0EC60C440E440E440EBD0ABD0A5414CB13D614D614F10F620A620AF7081B1A1B1A94021505D207D2072D062D06C60CC60CE70950115011501123142314380AD40CD40C4E0DB107240324039D02C50990089008EB0D0F070F07DA088C0A8C0A730AF80AF310F310370FD60DD60D1008AC03AC032603FE094E0D4E0D550EBF0DBF0DF70D300DA10AA10AA10B080B080B7D0B1E0D1E0D5907270274027402590A030DD90AD90ACC0E2D122D1245100B100B10B20C1C0D570B570B5B0B37073707B90602090209AF18B115DA11DA111517FA15FA150615C50F880B880B420ABA0ABA0A4908C504C5042008FF19F5126B15CB1A341A0F1C0F1C5D15D40DD40D52109611961152105210521027071F040C040C04FF19981B771C771C40228D248D24B4238C138C1370134618521052105210F912F912561756175617EF0B5014FA1BFA1BD825D825D825D8251F248F238F239823F120F1207B1FE321E321E32134100D0E0D0E6214041603160316CB1CD820D820F525FD24FD24E32372250F250F253225E321E321E3217A117A118E1E9812F70CF70C191A191A191A0D20E421AB1EAB1EA7248B1A8B1A8B1A481D481DD4136F16051B051B', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (108, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C0066326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F49B04C904C00D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46608E306960548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C00BC00B220A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FA09FA097B0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C9060509050948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AE0E060F060F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45E1D3E1D3E1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45E1D4B1EBE2148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F487069E0FBE2148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F459045904750C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4110211028B0248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B701B701BC0248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F447029C059C0548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48D07F00CF00C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F512B920C12548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4981B44216B1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F450147A1D8D2248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4361B361B1F0D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4651A651A481048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D914EA15EA1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_01_17.tif', '[2005-01-17,2005-02-02)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (109, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540989537D1F812474000000000000000000000000000000000E6100000140014004548F4E800E8007F007F004700E1FFE1FF0400D3FF30003000240066FF66FF6FFF6DFF6DFFA5FF4DFF4DFFD0009C009100920092008C00D4FFD4FF2C002C002C0062FF69FF69FF96FE22FF19FF19FF31FFA4009200920094009200920082006B006B006B0008FF97FF97FF600069006900CDFED0FED0FEC4FED9FF740074008E008E00850085008B007C007C000900F2FFF2FF4600740074006900C1FF0400040028007400400040008E008F008F00A600B200B200A2001000F1FFF1FF3F003F003F00E0FFA5FFA5FF04009E003F003F00EAFFBD00BD00A600C500620062001000CDFFCDFFF6FF31003100ECFF34FE34FEDB009600030099FF99FF4B006000600026018B018B0183005F005F00790012013000300026004C024C024201B1003600360027005E005E006000650165016501C200FA01FA014401D600D600DE00890189016E0080011C01310031006A00D500D5001401280128011501D501D5014101160116011601EB00AA00C300C300D2004300430038015801580131015001500126014F01D701D701A2FFFDFFFDFFAA00C40065016501B001D201D201FA01FD01310231029E0249024902BE01E000E0001FFF26FF26FF4EFFAE00480098009800D20223022302DC02560256021103F500F500990553017E007E0051006B006B00330126011B011B01F5024B024B0203051C06F905F905FB0A180B180B4B0427002700F900F500F50047019502B402B4023E03DF031D071D071C06320C320C9C0D2B0F2B0F7E080702500250023402AB01AB015002BF01F501F5010E04170517055904390E390EC80FB50F400D400DD002B405B40553030503050350011502F501F501F5012408E701E7011103ED0AED0ADB11890F890FD60758095809890361035C03D4FED4FE16FFC602C602BD06F704F704D402620862080F0AFC09D50AD50ABF0448044804F302A5016F026F02500117021702A8030C048D048D04EE0738103810400CF106F106DE03B304B304340324027D037D0348025D013A003A00D90564056405B60FB60FB60FF3121E101E109708B30248034803C702B502E602E602D8019801980118092A072A071809ED094E0C4E0CDA10BD0ABD0AF0028B028B028A03', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (110, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540989537D1F812474000000000000000000000000000000000E6100000140014004548F4A400A400C001C001D601270227022C02B102B10251038A038A038A03F502000B000BB912A512A512A400A400090109010501430143011903190354035403BD03BD03BD03AA0D830F830F0613AB11AB11D9FFD9FFDCFFDF00DF00F70120032003190351035103B604F404F404AA0D4C1213101310E40A9606E8FFE8FFE1FF4E014E01FD008E038E0377036A04AD07AD0716078D108D10A311A311A311FC0E240617001700A8001B012602260253022E032E03D504D60BD60BA20B8D108D1081125914D414D4142513DB002D022D022D0283018301BE00E606E606E406540E540E4011DE1309140914571419131913ED144C024E034E034E03FA01FA01D601FF0336063606A80CB511B511021597189718BE162D142D143015890136024E034E03D501D501D50106FFA102A102D7046A086A08F911791B1C191C19681962176217AA003E02840184011E02880188015603A7038D028D022604560D560D0313C01AC01A331D8A208A20C4009DFF4E01B801B801FC020A030A03A4033805380528052507250791074E11FA11FA11F11DF11DFC01FC013B014C014C010A01160216024D03A004A00413053405230B230B36113E113E11B5195F1A3A023A0284034F034F0356015502A000A000B503E004E004FD06F10BF10BBB1092119211911704182E0155015501F701C002C00229028B028B02CD0243044304140AE91025132513901B8B148B149117D901DA01DA01C201F601F6017202D8016501650153032B042B044208941294128421B116B1160013D102D102D1022A012302110211020102EC01EC01FE0308040804EA04A50D811E811EFC18851185115C0345035E015E011E02B702B7020203EB03EB03BF04D70456065606BA10C61EC61E721650115011A5014603600260020A02CA0109020902B90285058505A0050C060C065411EE19EE19DB18B8160312890289027D013B023B02EC0003020302E90286058605D408E60925122512BA16DB18DB18C914F312B001B001BD019E019E01EA01A401CA02CA022108780B780BCF0AA915A915A9154C164C165A181914A604A6045900790090019001A1017F027F021307700B700B1D1475187518221622162C162C16DF15', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (111, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540989537D1F812474000000000000000000000000000000000E6100000140014004548F4090EA417A417A417F3176B1A6B1A3019251925193615DA11DA119514691B691BF4185A1AE520E520560BD10DA417A4175915A112A1121513621662163D0F750D750D041269182B212B21A0246E216E219606D70959155915A71384158415EF0E7D0D180D180D3312D810D81048192F202F20051D7B257B25240685077212DA18DA18F018CE0FCE0F460D730C730C7111C313C313392127210F1E0F1EA91E3A1D15131513B916D51BD51B221950145014DC0EB71324132413E119621C621CB2207C237C23DA25DA25291629164E17181BB21BB21BD318E60AE60ABB1679127912E1199C209C205F239A20C91EC91E8F1FE615FA16FA1672133F163F16E90D240A240A7A0C01130113311950241022102291224E1D4E1D571DE316F214F214720AEB09EB0915087A05C712C71201156D186D18101E6E236E235022A726A726F91FC11DD516391D391D311A511851186317E30EE30E9212AE15AE15DD20ED20A921A921A72658265826F11DF9155C1D5C1D5C1D261426148C040F1A111511152518D321D3214620982298224820AA21AA215F1A5C17DC15DC152B14C11C6A186A186A18070F070F541DE21DE21D76203320C01CC01C291B291B041804181B12721272124F0C630D630DFB11070D070D071B2C1D0C200C20BD1E781A781A141A291BD312D3124C113A123A12460C650C160F160FB6113C1A3C1A211EE51DE51D9D1D0E170E176C146F0E4E14F20EF20E2A0F650C650C510C3E103E109E19E71CE71CCC23C01AE914E914A2140A120A12C90C85115D0F5D0F270BBE0BBE0B230A2D0A8C198C196F216E216E21AA17271727179B13F20FF20F620CC2116F0D6F0D660CE909B80BB80BFD19261B261BEC1E181618166317DC1156135613C610A706A7060312B80F361136118313F20DF20D592259225922D2219A18531853189F103B0D3B0D680AB405B405F3122811FE17FE173618991AE818E818EA15281728178713531B531B420D3B0D3B0DC201680147014E144E14D016071807180416D918D918601C9116911691163A0BCF08CF08B60B3F023F025D01390137163716751794179417BF167D1AA11CA11C301F301F301F7A0D060806081305C602C6020301FB00', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (112, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540989537D1F812474000000000000000000000000000000000E6100000140014004548F4F820A821A821BC1F4D1E4D1E701EB319B3190D22951FAA1DAA1D4D1E9C0D9C0D820EE107E107300498206B226B224E22AF24AF24AF249A1B6D1B6D1BAF1FE41BE41B781A500C500CB107C705ED04ED047B256922C122C1221F241F241F247C1DDF1DDF1DB518F117F117AB14720A530553058304DF04DF043A1D6C26D526D52633231F241F240A24B71D091B091BBC16A116A1169508C804C804C7029E039E03DA250525EE20332333231123522052202F1A651B651BD31B520F520F390A05042A012A01D7013C034F214F21E9200E1C0E1C7C1AB819B8194813C40ED00ED00E380E390739073805F501F5010C02F201FD17FD17DD19BD17BD11BD114D0DCF07CF07D5077D0D7D0D1607030303032C04C302FB03FB03300561180A170A178D129E0C9E0C9E078C078C07FF04DF03DF03CF037503920292024D051B091B09C706F3184C144C148F0C4F0B4F0BB0042603E802E8020803A102A1023601DC02DC02DB051B091B09750ADF15A015B90BB90B3408D404D404BF01380238029601BB00BB00A300110118051805380A5B085B08291B9116250C250C2606A303A30399018E011E011E01D3007F007F00E901110511053E0B11091109291B8915F20CF20C060EA601F400F40038017400740061000C000C00E3013507360B360B99076C086D066D066800060E060E0901D500D500CF00640064007600F1FF29022902B608990B990B790793070203020368009F009F00A400A200AA00AA0020012F032F034C02010201027108350B350B820453086D08A105A105C10105010501CA00B202B20233038C038C032002E0026E036E036F03601060103815A508140614062A02990199011A01BB009C049C04E802CA02CA02030491059105E6048B108B1038158D021F031F03DB023F02C501C5010301C104C104CF02610361036B07CE0760136013DC12DC12DC124701E602D502D50288024002400217022E082E084605ED03D610D610D61094129412C6138C118C113901A701A701A70129037E081B061B060C06FE04FE040A037D147D14A716A716A71617178B143D12BB01BB01A606AD07AD07370FBB0DBB0D95100B0E0B0E0B0E5714F314F3149415FE14FE143B14D412', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (113, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F4D603D603A2037201450145014501F105F105740619091909430CB10DB10D7711FD1447164716AC0E040214031403A40102010201280282018201E903C709C50CC50C650E911191119D11C611C611740EEA0303020302E500E500E5008301EA01E702E702D703BC05BC05D60E1A131A13F013FF1156105610D8041C03D702D7029E02BF02BF02CB03BC04BC045D049C059C05980E9612E814E814E8142B112B113C031C040C040C0414057F047F043004DF04FA05FA05070611121112C210C512C512001031133113F201BF05A2052A062A06A6071A091A090F0993119311520A2C142C149A159514A614A614AE13AC153B053B05E205460B460B220CF111F111431493113F0F3F0F27140C150C15AC102A122A124412DC1293069306390794089408C40CA213431443143410BD11BD11A612071307131E0FEF110E110E112C11690AD10ED10EB20CAA0DAA0DF6123B153B15DA1171127112BF137B13ED0FED0F0110D814D8145819F90AF511F511E5115B135B139F13A2155D115D117D1313151315F3155712571252132418241894191509CC11E312E3123C13E013E013F2147D117D114D13141414148D155415121A121A8019B518B5186C085B11F311F3111D11C710C71056116D118311831135113F193F19D019491D491D341CE91AE91A93076211BB11BB1190117910E510E510AE0F7E0F7E0F280EE811E8117E1C131FDA1ADA1A0E1B541A36133613A313281128112610870F870F840F3E103E102B10CB120F1B0F1BC31A401A401A591BBC1AE912E912F6122112211231116F0E3B0E3B0E460F151215128715451A451AC118E419E419B61AFB1A2912A413A4134C136A116A11DB0E6B0E6B0E9C0F11121112B9157B17151A151AC119531A531ABA17A211B212B212F610870E870EA20E8E0F4D104D10F4146C186C18F7187B177B174A163E163E163813F1108610F30EF30EEC0C780D780D820F7B0F7B0FB614431643163A1C6B1729142914F6120B130B133D12E010FD0EFD0E900EEE0CEE0C4E0E9410941061163516EF18EF18B4139B129B12361216131613D4120212A713A713E013B3102A0F2A0F3B13D116D116F117A817A81711149B139B136112D9116F12', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (114, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640989537D1F812474000000000000000000000000000000000E6100000140014004548F489128912601121102110A90D350F350F200F970C4C0A4C0AC607660866088E0E771077103D12D5103511351196109C0D180F180FBD0EAE0EAE0E220CCE07CE077409000C000CA30CB010BB0FBB0FE00ECB0E6B0E6B0E880D0D0E0D0E600F540E540EBB0A7E0B2C0F2C0F2C0E590E590E1A0DCB0ACB0AD90ADB0FBA0EBA0E210E890F890FB00FB80CA40CA40C020DDA0EDA0E2A0FA70FA70F760C300B3A0A3A0AAC15E30E750E750EC30FE00FE00F0A0EB40CB40C410DA20EA20E6D0D210EC00CC00C110B6F0A6F0AAC152210BD0DBD0DDB0D8E0E8E0EE90CC809B50CB50CE80A680968092509DB0DDB0D430F870A870ADC1250114C10140E140EE40E1B0E1B0E3B080E060E068F057906790660081D0C300F300F490CB90A3B123B12C30FAD0FAD0F9B0FD50CD50C1D07CD05230923092C097F097F09050BAA0EAA0E470DDF0CF417F4170014571257122A10FF0CDC08DC082D0750095009760C210C210C080DA20E460E460EA7099419681468149B11EF10EF10A20E880A880A1F09B70BB70B560FDE0E8B0E8B0ECE0E700A700AEB0774181A161A1688141C111C11360F390CB40BB40B1A0D090F090F461023102310E00D7709770988090F1AAF185E165E1699100B100B10DD0B1B0B1B0BB60AAA0BAA0B490C1A0D9D0A9D0A4A08C00BC00B541AB41A80198019531211101110A50CB90A070C070CCA09D306D306DF04E805E805B208EA0CEA0CBC1A3E1A2C1849144914D20F690B690B540A930993094F0983068306A1063B07FD0AFD0A6A0C2B10F719F719FB1786178617B60FA10AA10AB409A209A209B606080631093109760B080A080A1B0CD70FF313F3134215BB12BB12EE0E030B34083408400933063306B4061E0A1E0A200BE60BE60BBA0C72101E119D109D108B10220F220F960CDB08DB08ED061A071A078D0ABE0DAE0DAE0D760E640D640D0C10921030113011B01077117711520B2609FE07FE07E207F807F807090EBE0FBE0FA010E90BE90B310D57118113811381139D111A0E1A0EA10CA109A1095F0956085608A30D3710BB10BB10140D190819086F126A12C713C713B413921092100214C50BC50B460AA806B107B107160B760A760A370AD208D208', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (115, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640989537D1F812474000000000000000000000000000000000E6100000140014004548F4D51005110F11541054104F0F7D0E7D0E710D740C740C560CE10EE10EF91202104610461097112A1AB60EB60E8F10701070102010BB11BB11A81090102E112E11D613C614C6142818061806187E14031A250C250C3A0EB90FFA0FFA0F8510B110B1102512A913A91325141E1A1E1A261B0417321632169615340C130F130FD50F541054105E0F2110211046110612841684160B1859145914EA12001500159B15A409720B720B6C0DA70DA70DC40F16102510251059103E133E13E411E710E7105E115C145C14B8174D0A390B170D170D3A0EE70FE70FF60FD30FD30FC60EA30DA30D1110D11104110411A8168F1C8F1CB90A4610AC0FAC0F5810C910C9102710450FBB0EBB0E100F231123111F11D20DD20D8D12DB1ADB1ADF0C180AD50FB90FB90F3B107C107C10F90ECC0ECC0E170D030F030F6314961833113311B4120C180F0C0F0C8D0FEC0FEC0FCF10E70EE70E240DAF0E450F450F2910F818F818F91D591E591E8019AF133B0D3B0D8C0D450D450DC40E76111B0D1B0D370E3B103B10C312DA16DA169A171C17241B241B241B840E280D280DF40B030E030E850F360F360F6D0EC90EC90E6A13B91C511C511CE11B4C174C17F518B90ED20ED20E4B0B8D0C8D0CA60EC00F0B0F0B0F8D0D6B0F6B0F1A1B191C191C131C761A761A4A191A127E11B90FB90F4D0E400F400FBB0FF00EF00E9A11F510F5107518131BC01CC01CBB1A0B180B182B10E311EF10EF10E80D21102110E21075108A108A106C110C120C1249146F166F16211B51185118D70F29111511E711E7118B11C011C0111011E311E3110E161D181D18A7178D15461346134716401579107910770FAF0FAF0F1111D010D010E90F011201120B173A183E173E170F16F512F5123817BB16AE10AE10BB0F0A110A118F10471121112111F814AB12AB125317E315E315B612FC14FC14521A6E1DF50EFB0FFB0F4D11D413D4130914B513B5131C15CF14CF147F10580F550E550E480F831783170F20B30BBB0EBB0EA2101D131D131A14D017821882189E15A312A312D60EB70EB70EDB0CD812D8126E195209D20C1C111C111D13D615D615361BD61DD61D8519AE13AE13A51183101D0F1D0F6A11B115B115', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (116, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F42A1ACA1AA21DA21D24146C136C130A10B511D50FD50F250967086708EB0A910C910C230C020B020B031ACA1A0B1B9C169C16FE16A215A215701744154415BE12FD06FD065E083E0B000D000D960A410A96159615541515131513A718F116F116B0127012E313E3130A0ACF07CF07B009680A680AFA0B150C1F171F17EC172C1522152215CC1528152815331569156915E90CB107B107D2073709C00DC00D840CFD1667196719F81A17191719E91520172017BD18DD16CF11CF113B0DC507C5071A08D40AD40A690C1F1BAF19AF19921B921B921B65160415D819D819C418A717A7172412CD0ECD0ED209E30DE30D22104B1B101849184918C91A911891185719E218E21808190B160B168E169218E918E9187016561956190C1817175F165F16E71A91189118411504174A194A19151D851B851BCF1F061E061E041DED1CED1CAF1325157D18591C591C701116101610F2127B1F7B1F64204D1F4D1F731E7D1D091E091E191F871D781278122515EA1BEA1B4612A00CA00CDC114321172017203C2056205620881DEA1BEA1BE31BFD1EF514F5143316CB1CCB1C9A1D6415D714D7147B1AD01FD01F311E411D411D151A4B1DCA1ACA1A521E5E186C156C155416FC17FC1782174F154F15D618871B871B871B5B1949194919D6197C187C1829133F16471447142C14F217F2170C18F61656165616D618BD1BBD1B281B881888189B1923152315FE111515871312151215CD164B134B13C1158A168A16861A0E1C0E1CB81B5E16A215A21598148312831240152E14AC13AC1327187E197E198B191917721B721BD21BB316B3168912821482148D1633133313BB1629167415C011C011DE17151A151AF216EA15EA15FA15C312C3123312A41758185818471CC31B73197319F915EF14EF14631498169816F6173511591159119E10FA0FFA0FA115701F701FB21CCC1C491F491F6B19C712C7121910AA0F6B106B10C10EA510A510F30E730E730E9C11AC1CAC1C65184C1C6418EE16EE166514FA11FA117816B316B3166A11C110C110940E140FE40FE40FA715EE18EE184E0EBF16B115B115381568126812BE17BE171514151488101B101B102E10C90EC90E970C0B100B109C0D', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (117, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640989537D1F812474000000000000000000000000000000000E6100000140014004548F4E70A7D0E000E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4410A800E9C0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4150CBC0B221048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4860C860C2E0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CB0BCB0B610E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F487107210721048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40C124717471748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E21D311BDE1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4871D261D8A1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FD1EFD1EA31F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4521E521E2D1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BE19BE192D1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448158A1C8A1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D2114A144A1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45111420FCB1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C31BD31B001B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CC1C3219A91648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DE12DE12001048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DF0DDF0D8B0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E90D2310231048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (118, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4D1013D033D03EA012803280428041809180918091809C110C110C1109F0E9F0E6302C0040205020503025202770277022F0379037903EF02A203A203F30A6412C110C110DE0EE00DE00DE00D0B0B0B0BDF002301410141015E025F035F03A2027003D80BD80B8F0ED011D011171A5B125B125B125A0BD908DF00DF00F50043024302120312031203C203780978096B07020F020F7226FF143D143D14B20AC409D101D101EB01400140010403C103C1031F0B270A810A810AD20C1A261A26DE14DE14DE14D90AB00A9E059E05530A39053705370561044C0B4C0BAD0ED00ED00E1B0F231C231C29112911981498142B153E05B106B1062C0B2C0B2C0B0508C20BC20B360FAD0E4B104B105D131E131E13291131153115D7173E03B106B106DC068B068B06590BF308621662166216561756170614301230129112E1126F196F19D20529062F0B2F0B7907A109A109EE104415441548155617561710146911AE13AE13F0154D174D178B03F706FA0BFA0B440E86118611C810211048154815EF153B183B18B312C713C713701364186418A8032B04EA072D092D092D12B912B9126E115511551186121E161E165317BF160B150B154617C9180B020B0263042E032E03F506B70EB70E5115E311551155119C1432193219F61A971297124613BC16C401C4019603A00AA00AB40A9A14A614A6146614E613E613B91350175017981A3412341227120C127902B1FFB1FFAE015C025C02CF064E0E4E0ED711AB15AB158C14C915891989196312F210F210D50EEA0187038703CC04AD02AD027F0513094B0F4B0F81131818181882182D192D19E713CA10CA10420F02088F0AAE0AAE0AA9066B086B080A0A370E370E76114916491604172D1909160916710D5E0F5E0FA00B7E0C8E0D8E0D560EF70BF70B430E150F150FA60FE60F261926192619AD15AD151D0E75117511F408760C750E750EAC0E1410301130116F103F0F3F0F27104C124C12CD16BB15BB154412BC11F311930B930B4B103B103B1008122E122E12F010931093101A12A315A315A315FA11540F540FE911E1117011701175126C136C13BA14EE1226112611AB11841584151E156F146F14A70E850D850DA112E111', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (119, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4BB05DE01DE01F201CE01CE015A03520252020D08180BFC11FC118A12ED16ED16D316A712A71241168A08E605E6052801D00295049504880301080108AE0A3A103A106E11761376135F14A115FE17FE17D908D7076D066D065C066A066A061108680A680AEE0C8712C511C511141200130013CE1779177917C4090D061205120508050906D408D408920A290C290C4813140F140FCA10CB0FCB0FDC15A117AE19B00A8C07DB09D80AD80A490ADE09DE09F50EED0EED0E2F0FF90EF90E15105A14C214C214E3165119BC18BC18BC18E40FE40F4913AD16AD16A316E8135E105E105E100F140F145E14291529153F1556158B178B17F31ABC1CA516A516F7175B165B161E14B614B614B917451A451A3B15BD14111811180616111CF31AF31ABC1CBC1CBC1C4D1B2A1A2A1A1215A81523182318AC18BB15BB154618571857189918DA1B381B381B781A581A581A971B2A1A07180718221B2F1C2F1C9C18721872186019601915171517CC18DB19381B381B5E1A781A781A0816081608163F1BF41BF41B5A1D2D1C4E1F4E1F6D194F164F16C918711971197119641ACF18CF181718E816FC16FC16E91820192019DC1D4E1F4E1F4E1F0B200B20BC16D9156616011701170117D615D615A916A916A9165E182F192F19FD19531E751C751C0B20041DB511B511B511C413C413C41368166816A916A916C819C819101EFD19FD190E1E761C761C761E9B1A490E490EE40DC011C0113011961896189618C719E81AE81A101E101E101ED01ED01E761E761EDC19390E2D0D2D0D8E0D99109910C01043154315FC171E191E19D018EE18D01ED01ED01EA31DA31D021A630FEE0FEE0F760F83118311681292129D179D17821931193119EE1803190319031979177917C2195011EE0FD210D210BE1265146514A415CA18CA18CA18AF19AF196218031870187018B9157B167B16F311121112111211A912011501156615B716BF16BF169018CC17CC17851750185018E21822162216E111B11147124712D012EF13221422147C1452165216AD192D182D187D19551A551ACD1917183816B211B211861186118611CB13CB13CB13551332133213F5140515BF18BF18F71AEB1AEB1AA915A013', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (120, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E3254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4991999198719351A751A751A90199D119D11F616990C990CAD07D90406060606C601C601C601E200CC17871987193B19B719B71990188F0C8F0C9E0A770A950495046303A901A9011D016801680160026719DB19DB19D41819193E1B3E1B47125C0A5C0AEA0A18061806710484018401D002F30050045004AE19671A8D1B8D1B181B3A1A3A1A5512D004D0044C06FE0391049104E201A003A0039003180418045119A0192A1C2A1CDF1A3A1A480E480EAB00DE00DE001301030103011103B304B30483068D04660456156F179A196618661823190F080F085B04AE01AE018706FC010C020C02EF080F0C0F0CA004B104841A841AD11C25192519C414291029107F0B9D0CA309A3096D02080608068D0BF20EF20E4D0C4D0C54195419AC1DD11C7D127D12DE0F6E0B6E0B310A2D052D052402E809E809310D210FB810B810A810C615C117C117A517031303135710180C180C570C4D073A053A05BC067E067E062E05C20FC20F4E0F36161B161B1636174C18C30DC30D4A0DCC0FCC0F47086C066C06E106420742074B082910C00FC00FFC1C111DEF16EF165C17041704178E13090F090F3B10E812E812A71222099F069F063E12C00FC00F041D5D1DAB1DAB1DA3197F177F178215A405E80DE80D3F1345154515A712671367130E11BB0EBB0E9B1A6A1B8C1981198119C616D515D51508117C137C13FB14FB14FB146A16B9100E110E11F00E240F8F198F192D1A631A631A0916A514A5149E11BA14B116B11603176A166A1610128B108B108B105211F419F419C018AC18AC18DC16C815C315C3154A19231A231A0317DD14DD141012DA101C121C12A612EC19DD17DD17DD17DD17DD171B179A179A17D9179A199A19AD17A714DD12DD124412B813B813F1148216FD14FD14E513A418A418E6188B19B319B319C8197E167E1643156E146E145E1452135213EF147016E11377137713FB15BC15BC15FA16B319B319B015F713F713B11343137D157D1568149E129E123816CF12091309130F14A312A3123512A412461146111B12A711A711551224142414C1137A127A12A01394134E144E14AA123914071307135E12AD12AD123C115C115C1170115C14461346130F124E14', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (121, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4AD01AD016B07F411F411C211F311511251129111A914A9146D149A149A1496127F147F142913631110041004600C8911C211C211F110591259120712A114A1149D15FF123B133B13F813B612B61267127D0C761276127612DB12DB12C511C412C4128C140214F613F6139413F214F214DE12EE10EE109411140B4C104C1073137313541554151C15CA12CA122312C711C711C91250135013AF12A70FFD0FFD0F6604000E890E890E2F13441244125B137214721412134313A414A4149E12510F510F8F0FFF10FF10B1041F0800110011481025144614461413140F150F156013B212B212C9100C110C11CF10D8130F184D0CCA040011FD10FD1027146214621483122A122A12F612FE12161216120212F411F4118F13C517210E210EFF0D831083106414B213B213AE12FF1185128512AA121813181345126D136D139B132A16970F970FB10FF20F2B122B128811A910A91085121B121B1283108D138D139714AE142E142E14ED14E4103D103D10DA0E9912991292118F108F10EE10EB100F120F12E613EC14EC141E15061406144213900FDC10DC104E0FD810B911B911F610BB11BB118511D411D41106136E146E1467151513C313C313630EC510611161115D1020112011C910C910C9106C134014E013E013B6147E147E14CC132A132A13240F1710BE0FBE0FB00F3D103D1070107010881488140514221322130D116D116D111F12BD12C71252118311EA18E50FE50FA510831083108310F813F8139913B713B7137212E612A311A3116E115B11A612A612E312EA18EA18EA18EA18EA18EC16E5170E170E179210BD12BD129E13041D041DA011491883138313B1133E12E211E211F71A2F232F23EB1961196119A0180E160E16D01357163E153E15FD1A16162E142E14081421132113481CAE1CAE1CBE1C8619CA18CA18C11B4B1C4B1CF51A911B911B5F1BD915FC15FC152D167E167E160219831A831C831C2C1AC619C619BB1C821E821E651D811C811CC11CAA14E319E919E919B3195318531896182C1A2C1A2C1A0D1A0D1AE519301AD91AD91A121ED41FD41F4E146A1B661B661B7D1CB81AB81A0918C317EB17EB171618E217E2179A189B1A9B1AF91D851F851F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (122, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F48911891145138914891471126E146E14C91511181118DF17F515D813D8139E10BB11BB11E811CF1486128612EA12FE11FE11091365140F160F16B513021402147215200F200F760FEA11EA114D121514E9123E133E1338113712371239111C111C11FA1075107510AD0FB60E9A0F9A0FF413F413F413CD14200FDB0EDB0E3D1176117611FC0EAE0DAE0DED0E810FDE0FDE0FF10E080F080FA213E813E813FE14C91717181718D416731657105710FE0FDC0CDC0CCA0DD00DD00D470D7F0F7F0F8D128012881388130F1835185E185E182818D113D113EE0F280D280DEE0D410DEF0CEF0C7B0EDA11DA1174119F119F11C517B417CD18CD18681664135A115A11CC0E210D210D040DEF0BEF0B6F0D350C350CDA0EA8105613ED15ED15CD15A814A814871317111711E90E4D104D103E108C111813181367109A0D9A0DA10DF10FEA15EA153C16A812A812181171107110C40FC00F4D0F4D0F351233133313B0135C105C10420E490C24132413FF1350120D110D110D11421042103A0E161116119A11DE11DE117611840E1E0F1E0F5C0EE613141314134F123D113D113311E80FE80F6D0F7810F810F8102A1110111011CA0F551055104D0FB7131D141D145511C110AB10AB10FF0E1A101A1033101411141134104D104D10EE0FC51107110711C712A41248124812AB10C010C0104D0FE00FE00F9C0F8612D110D1100811B10FB10F0111781078105B11B311C211C211AF11CF11CF1127114D0F511051108A1046104610281032103210440F1D11281949188B15D713B511B5118D11A611A6110710E111E11189114B114B117010D60D590E590E690E2D134A184A18D91512151215F911B711B7118C1126112311231189117B0D7B0D700C460F460F350FBA0E38193819101771147116711608147511751143105D105D10650FBB0DBB0DF20CF20CB30EB30E470EEA1BAA17AA177A1699169916381741104110BD0E540FC20EC20EE40D990C990C7A0E7F0E7F0E7F0EEE1DC91EC91EA41EC81EC81E6915E51060116011F40EE20DE20D8F0D8F0D8F0D540FF60EE10FE10FC0239A1EC91EC91E171D99109910DE10DE10DE103E0F0A0E0A0E1F0D1F0DDD10DD10021234103410', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (123, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F00040723264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4CF140F15B215B21532154C13181218121812940B940B98060E080E084906CC07CC073E0C470C5609CD13CD1353177E177E171819151515151812CC0CCC0CF90CBA0B1B081B08EC09F10DF10DAD0E3F0A701570157D1A81198119E51ACB17AA14AA14AA14A30EA30EE00F940F940F8B0DBC0CBC0CB10DA30B671967196719C11A1F1B1F1B4517BD14BD14BD14E00EE00EFA0FCB0F6D0E6D0EBC0A040E040EE10A161333153315C11AC11AC11A1F181F181D161D16F70EC70EC70E9A0D600E600EA50EF90EF90E430E3413E412E41269142A15D218D2189F18231723173312DF0DDF0D720EBC0DBC0DCA0D230E140C140C56138015821482148616DA17DA176919B71AB71A7D162910240E240E5C0F150C150C5708F105F105F10FA6121714171448185917B817B817421ABB13BB13B412B50EB50EB00E350E350E95092B063B07AF0DAF0D841163196319F6187B197B19FD17C516C5160D15050ED80DD80DDA0F850D850D95073707B80EB80EB0124B174B1765194D19FC17FC1724173D163D16E110E40CE40C1C0EE50EE50E530D200BBC0FBC0F1B103415A216A2169C179815981522157E157E15FA104A104A108B0E710CB50DB50DB70D0711621762172014AC13AC13C8131F131F1391141B13EB12EB12EC103B103B10AE0D750E750EC00B91113319331933192F1F9F159F15CA1508150815DF12DC11DC11A21066106610B90EFA0F470F470F2819A818A317A317091CBC20BC20DE1ACC18CC1857177313E70FE70FA10F810E810E2A0EAC0EAC0E2D135418AB18AB189419E1198B158B15EE16A815A8154013591259122811950F950F0A0ECC0C140BBA0ECF10DF162B192B197F17571457145016CA15CA150D14E913FB11FB117C10620D620DCC0D501030103010D2148E158E154014021502154E11DD102F112F112F11590D590D9311D70CD70C800E5C0E7D107D10D115D40B55135513D70FEE10EE10B3102D112D11990E530F530F4910D110FB0FFB0FC90DF81515171517880CBC0BBC0BBC09F011F0113D129B11AB10AB10480E61106110D110D110D110060E610F8F178F170610660C9E079E07000D37123712B5132F102F10270EB00BB00BF30F4A10DA0FDA0F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (124, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C38264023E87981A30D474000000000000000000000000000000000E6100000140014004548F456092A0D4A104A10DE13F415F4154217151C4B1B4B1B7017FA19FA19451D451D451DBA16D513D5133F0AE20C340D340DFA0FE613D217D2177D1B3D193D19FB1D881A881AAA1B8A1C8A1CA21C9816B115DB0ADB0AFD0CBC0EBC0E6D11EC13EC13CA16A119A119741A9E1929192919E21A8A1A8A1A5D16A20E1E0B1E0BBE0C8E0C8E0C58112512BE14BE149B1742194219901BA819A81993187E137E1375152313850B070B070B880B9B0F9B0F1B11E014E014841584178417E3140519C817C817DD13B912B912CF12930A760A760AF909300E300E2F102B10C012C01234168914891415156B176B1747161B171B177314B5069807980745085F0A550E550E3E0F200E200E450FC50FC50F98147F13461746177217781478143B074F07D306D3063E0AC30AC30A830E9B0E9B0E790C970D071107115A1263146314F8133813381337078F09E308E308260AE60D100F100F460F030D030D1B0C0F100F10390F87118711B411200EA30AF10AF10A7F0BE509E509B20C200E200E530F4F0F4F0FD20E360F9C0F9C0F770F4D124D1278135B12A80BA80B3A09F307F3075F0A920CBB0EBB0E590FC810C810FE0F66106610C00ED00FD00FA81203121F0B1F0BC90AD108DC07DC07AC0C940D940D290F201020104F109010C90EC90E5F0FC30FC30F6D11610BEE09EE09A8095B095B09DC0B080E080E6F0EA40FBD10BD107F103E0F3E0F7C0FA70FA70F8810730B9F099F093B085709D80BD80BBE0D7A0E7A0E9B10BA11BA11F40EAF0DAF0DC80E6C0E88108810140BC509A007A00775075C095C09B10BD00ED00E3A0F930F470E470E290C220D220DA80D1E0F1E0F50100E0FE909E9094207D807BD0ABD0AC40C930E930EE10E700F700FB90D360D360D500EE60EFC0D5C0EB60F310B0B080B08FB07EE08EE089E0C1A101A106510040E500D500D2D0F2F112F115E10E10F9B0C9B0C300894079407D807330733071C0A820FDC0EDC0EE50ECB0DCB0D5D10BB10BB1025113E0E1C091C0940082A0884068406DE0699099909460C260F260F750FC40DC40D2610AA119B119B11EC0DBA0BC009C009060814071407820765086508380AA40C550D550D630D8E0D8E0DB510AE0FAE0F6F0E', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (125, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F493178F172713271336124E104E108713BC12BC12EF11461146114510C10F620D620DBF0D680C680CB115D214B414B414A010181118117012B110990F990F25118F108F100D11380F380F320F830E830EA20ED811A411A411FA1046121C111C111312780E780E3810541054109F108A0F60106010D60F5110CB0FCB0F0311FB0EFB0EA50D9C0B9C0BC20DF411F411F5141C10AB0FAB0FDE0F48124812A2117C103717371703191B101B107F08FD0BB90FB90FE1101F131F13381103100310DE0F3110311022107610E315A41BA41B261CED16ED16C916DE13DE134510831283123D112D0FFA0EFA0EF20FDE0FDE0F7910F21448164816C2167D167D16E116E315E315E31559118C0E8C0E910D9C0F9C0F4010851185119A0F381073117311EC10361409110911BB11CC12CC121511170F170F9C0D790D410F410FCA10CF10CF10A30AE10D560F560FE00F00100010940F590F590FD50F560F1A0D1A0D290BA20EA20ECC0F6D106D105B12C710790E790E5C0D350FC90FC90F240FC20FC20F990F110C110C230E140D140D410FFD0EA10FDF11DF11240FFE0EFE0E240ED20DD20DD40DAB0FAB0F2D109D0F630C630C7F0C1B0B1B0BFA0FC20FCB0FCB0FA40E890E890EFF0EC30F22112211870F821182116810AD0AAD0A2709430B430BF00C860C150F4F0F4F0F570F270F270F311033113311DB119E119E119012C112B616B616140D220D220D7E099D0EB00FB00F240E8D0D8D0D940BB30D120F120F7A12E114E1144E13D516D516001370127012800B3E0CDC0DDC0D8C0D1E0D290D290D7F0E050F050FC2153F163F169D1822182218AA18AD133E143E14FC0D560E1B0E1B0EE40ECB0DCB0D090F650E650E6113751309130913061735183518D61807160716E10F6C11DC11DC11C80ED00D5F0F5F0FA51099109910F5130E130E13E114A013A0139F14B112C11438103810A211AE10AE10210F510F510F0310451045100211C510FD10FD101D12A814A814CD12DB12DF0FDF0F650F130F130F360E590D300D300DEE0E610C610C42106C116C1178116B126B121E14A10FF00BF00B310CBC0DE90FE90F630E040F040F5F0EE514E5142D187716D913D9133F141E141E145414', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (126, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C00663264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4620D0414041448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43C0D16150B1648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45110AA14051548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47C107210371248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F438113811E51248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4171017104D0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4970F9D0F9D0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45A11440E440E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42E10CA0FC90D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A10FBF10DE0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C20F3F0F850E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EF0BEF0B310B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40D0C0D0C070A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45807AE0BAE0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40A08F607F60748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46312470A470A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C1147316F70B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DB12A014500F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B610B6100A0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46E156E15C31548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (127, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4AD13AD1310198A1877177717DB15A014A0145D161117111764151711F710F710030E2F122F123413BB16F919F9199E1DE21AE21A7918961A961A461CF718B017B0174C12581258122A12E313E313FD13F01591199119C51B521A221922190C19141B141B7B19B017B01714113A123A121A13691466146614EE1531193B193B19521A7B137B13E914BE1ABE1AE718B40F7C0D7C0DAF0EA80EA80E6F10E411E411C31598173A163A161A133211A10FA10F61114F124F12600F510E510EDD0E1F0F1F0F05105E11CB16D310C012BE1008110811C3102E102E10E00FAA0FAA0F341034113411A40FB6109F109F104E139A171C101C10670FAA11AA114C132D132D13AB120E125F105F1022100A0F0A0F5C0F7C117C1118132D17470F470F0D0FCC0EAE0FAE0FC4113D123D12D410EA0FEA0F910F5B0F5B0FE60E1C12D614D6148817A60D400F400FD30FBA0FBA0FB4100F0F0F0F090F670F430F430F0910C80EC80EBB12201320139B14A30D290E290EBE0E8C0F8C0F21139312F712F712FB12E30FE30F1C11641264128A123F1635143514900C750C5E0E5E0E59108D138D135316621662167413AB10AB10AF11AF117A137A1382182F192F191A132613881088107811851385132E15FD14FC10FC10EE106E126E12D510A411A4115318001C001CDB0F1213CF1137113711BB1156105610E7107D0F7D0F1D0F1513151308131611E314E314D5190C1E290C290C570CC40CC40C030D090E090E120F590F1C0F1C0F9A0F05100510EB126814681440196319230F230F3C0D4B0D4B0D2F0E2510351035101C109C0F9C0FCA10BB10BB1064120115101A101AF01A460D590D590D0C0E520F520FC510071007106510EE0FEE0F05117111401340133918C71CC71C5E1EE10FE50FE50FDC0E190F190F4C10A10F3B0E3B0E8F0E5513551377126B156B15731ADA1ADA1A401EAA11E111C40EC40ED610DD11DD116911C40EC40E7B0EA112A1127916D016141B141BC3199119911907168E127A0D7A0DA70EBF10BF1026111F11E80EE80EA61254155415A518821A821A241D801B801BA5126A10BA0DBA0D7A0C7C0D90109010F50F221122119113311431140418FF1AFF1AA91BE11AD619', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (128, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F40D140D14EA145A145A1462120211FB11FB114111BF12BF120314C213C2133414E316E316F6139D1466147714771478145315531576133A103A106510AE10AE10B911E912FD13FD136713F113F113DB12D91B841B841BE91C9A1C9A1CCD13451545155615C010E111E111D212D212D212E81510171017C4152819EB1BEB1B7C1D9817981798178916BB16BB166316A512A512B412FE11FE115316C5161F171F17CB16081B201C201C14193B173B17C015811681165916791615151515CC15D011D0119115701670169A178F1A8F1A8F1AC119BB15FD16FD16E715E715E715C01BB916B916EA13911491146C12761260122D172D17BE19511851184B17091709174A15671B671B321B421BE115E1152C16871387132213A612881788178B198B198B197016D517D5174A168D152B152B1574187D167D16AA15191319130A13C3107B177B17111B5E1BB014B014281824182418A616DB16DB16DB162617261764162D1539143914F81312166F1A6F1A111BF915F915F9155D135D130D15EB167F177F1753155D145D14D71601170117161778196B166B16B915AD158D158D158D15D016D016D61AD61AD61A06145D145D1410170916631663160C1E4A1D061D061D051D251725172517461646164E19B517841784171017101710171017101710170C1E171DC81CC81CC81C731673166F16A7143C1B3C1B3C1B10181018981B981B981B981B5B1765146319211C8321911F911F911F3E183E186416DB1BDB1B3C1B5F195F19491A981B981B981BBE1576122D1B2D1B351D721D721D0423CF1BCF1B371551234B264B265F195F195F195F195F195F19A1130A14681868182D1B2D1B5F145F141923091A091A5123512351230E110E110E115F195B1014101410A1165B18201D201DE11933213321FB14BE14BE14F209DB09DD13DD130E110E110E113607030B030B33108F1C981B981BAE123E153E151E113C11F80DF80D081501150115CF09CF09CF093502410541058A0ADC1BA218E915E9153913810E810EE20F611F611F160F160F160FCF09CF09910891080F0E8C108C10D6194A1786138613FC13350F350F150FE9117B0E7B0E840C080E080E080EFE0EFE0E641050145014', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (129, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4C311C3118212CD15CD15FB14E315E3152516D516D5165E179B16DE12DE12B814E716E716E813DD13C311C31168147B177B17ED167917B818B81897185D1A5D1A3C1A711A711AE51C441844188C14CB137914DD15DD15DB18A818A818D1187B1A7B1A061BF91AF91A711AB81AB41DB41DFD186C146C146C14CE1BC61DC61DE61925192519681D821DAE1DAE1DEB1A77187718B8157D167D16E7165C145C149512861B101A101AE619FC1A1D1D1D1DB31D6F1C6F1CEA1822152215D217A81B50175017E113A412A4126012E8129F199F19B0189E1B9E1B131A601860181E155F13001900191A1948184818A812B912B912A612D61491159115151A5D1AE817E817AB14170A170A000A08100810C7122710271001123E12B410F712F712BA16D715D715981498149814FA1004100410FE14E513CE12CE12B311451245122C110E10951095103B1259125912F8109F107B0F7B0F381186118611CE1112131213771201110111CC0E810DF813F813E5115510F70EF70EA30F7F0F7F0F30107F0F7F0F2E0FEC11A912A9123612BC11BC11940D1A15551255126810A210A2102E1171107110080D450DD60CD60CE00EF012F0127212DC12DC129B145B14631363136310C01050115011A60F0F0D0F0D900B150D150D1E0F4B134B1339126A125E155E1565142813421142116311F211F2115611F00EF00E830E1C0FD00FD00FA310721372139F13C913C913761253127F127F12D813F213E212E212A61103100310A70F17101710B010C912C91293124F132B170A145D13E712291329137513861386130111A311A311A511411141115811DA12E511E5114612EE122D142D1437120C130C139A12FD12FD12C4105411821082107B127D137D138C138E118E11E0116F12571057100A0F8913D011D01133113F113F110D13ED12ED125F1311141114F0123B12091109117013760B1B121B120613041104115410281128112913C015981598157C162C122C124112D00ED00ED510010DA212A2126212091009104010410E3A123A12C9149115911512151A121A123D10CA0E2A0F2A0FE610B60F680F680FEE1024112411DD11FA12FA126314061306130413571304120412D00F650F650F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (130, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4DD13151A721ADB1DDB1D0B199B1A9B1AFB18E917E91793172417241730152A179C1A9C1AB11C2722CE18CE18141B541D541DEE1C291A291A181A43179A189A18211534153415F614D415D4158C1AA91C461846187118951C951C8119C91CBE19BE199C19F116F1163B15DA15DA15BD141B151B15E417931C2819F41AF41AA71DEC1AEC1A1A19A218A21845195717571727177A1309130913E1155C155C156015051851185118401940194019A218A218261A261AED170B150B1524127C117C113711620F620F03113E139C1640194019771644164416AF17E317E3176B1818151815901232119C109C10C50ED30ED30EB410DE15C418C418761976197619AF17AF176D176D179217F912F912FB10CE0FCE0F3F0EAC0EAC0E0E10741683198319991976197D167D16E917D617D617AB17611361132613D80FD80FA00FF80D1C16541754172719911C911C911CDF19DF198C18DD17DD17C8175E15C512C5129E0F800F800F020DEB15811881188419611C611C9E1CDC1CBD19BD198C18801680163017C512C5124E10C60DC60DDA0B0C0DFB17C51CC51CED1C611D611D5C1D9A1A9A1A3316B515B515C5159D0E960F960F3E0A410841086907DD19C31CC31CBB1CAC1CAC1CFF1C621B621B3414EB13EB13EB136911DC0DDC0D6E0BFD06FD06FB06C318D21BD21B471BAC1CAC1CAC1CE217F516F516DD16E313E313491469116911A907FC06720672062B170D1A701B701BF41A9A1A9A1A5319AE17AE170B170B17231423142A10D407D407A70682078207EE12BF16D51AD51A921BFB1AB21AB21A541BF617F6171A18651965192C12A609A6095E077006C5086F12D214DA1BA91AA91AA91A0A1B0A1BD71C641B641BEF1AEF1A77137713E60E7F0C7F0C93061907F515F515DA1BAA1BAA1B961C1E1B1E1BBC1B681B391D391D671D31153115B710831083108C076607CE15CE150817411A901C901C181BCB1ACB1A4E1C391D391D391DB718B71873135509590859087208BB16D316D316C018A619A619A6194E1A4E1AA11B791CE41DE41D501B98199819B6071408140894076F1521172117E31AEF196C1A6C1A391B441C441C5A1D1A1D1A1D561B561B561B2F081B073C073C07', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (131, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F42722471C531C531C2216D010D010471047101F111F11DF0F2E0F2E0F3A133A133A136A1A681A681AA91C252020200D210D21871270147014E813941494145E147A127A12A5138125AF11AF11681A5C21251C251CF71F4620462083211E1F1E1F1D171D175D175D1798156E0E6E0EDB0B500C500C250E8F1E571357132E1AEB15EB15AB1F94194E184E1893190A1A0A1AC111FB10FB104510420FCF0FCF0F3122900D07080708D109E70AE70A9E1392209220301FAD1EAD1ECD1D771BD112D112841546134613681ED70FA517A517CB0969106910E01DE01DAB1CAB1CF324F324F324AA250A200A209625681E681E681E0711BE1DB919B9196D216D216D21E01D2A1E2A1E8421031E031EAA250921992099209920A11AA11A1C16D217FD17FD176D216D216D216D21142543234323C223F322F322271C1A1C1A1C3112C30CC30CEB15EB1531163116571E571EBC1FBC1FE021AC25AC253623FE17FE17571E2815220B220B1F072E10D40CD40C100F571E571E571E301F301FC321AC25AC252C23B12347104710D60B480748075C064005A90AA90A560E181518157E23AF187E237E233C2464246424311A781B781B6A08A906A9064005580523091F0E1F0EB519B519B519AF187E237E23AE17741D741DF718B318DA0ADA0AEA05480248024802C407B30AB30AB519381A381AEA2409168B168B16741D56125612D90C0A050A05E302BF04BF04CA08E30A191119111411CE11BB15BB15A20E6B0C6B0C770786088608C80B8803550355039C071E0C1E0CC5083C100F110F117D0DA709A709060C28042804DC04510229032903E401D904D9046104310E310E1907CD09F10CF10C440B46084B044B04A302790479046C02DA00DA0081004C024C02C0066D0BDC109A079A07C606600660063B05B902B902E302A303A30355033C01450045002C0285078507050C1E0F370737071107AF06AF061504B202010201020F04E703E703B202B202B202370170037003160D4A0CE305E3052F08AB0714071407B8020302030223046C056C05E9040A049B029B02F105770677067506C50563066306B706AA04AA04F00229012901BB05DF063D063D06FA09CA09CA09B80B430A430ACA07', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (132, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F45C19D41A201520157210AA0EAA0E4A10E714E714B413F810D70DD70D020EA20FA20FF010D510D5105C215C214E184E185F0DF00FF00FEA13A7144E154E151811250D250DFD0DED0FED0F8E10B10F400F8F1E8E19A11167126712C11533163316DA1434163416680EB80DB80D1E12A510BB11BB119F10680F92139213E413A516A516691964186418B018C41C2D172D17C415BB12BB12CF11C810C810630F1B0E7514751453144D1D851E851ECE1DD61BD61BFB1C511851189416671167117510E00EAB0DAB0DB30CBA14EA13EA13E11C1F201F20D01FFF1DFF1D7E1F881F2E152E15B410E10FE10F520F6A0E6A0E9E0DEC141F0C1F0C0213A320A3200620C21A7E1F7E1F7E1F7E1F7E1FC710870F870F960FC70FDC0EDC0E2E10E01234153415EC1787158715581858185818581858185818A8108D0FE711E711E30F7B107B102E10FC11651465147C144D124D124915491558185818581891109110C60F74107410E80FF910F91040057E05510EC40FC40F7C16321632169714491549154915480F480F25109F114611461128122C11660566057A05EE08EE08D80DC110C110150EA610A910A9105510B610B610CD124D114D113E12C81298059805AC06820A820A740ED70FC90FC90FF610241124115910561056104B131710860E860E3A0AC908790B790B4B104B104B10E010251125118712361136112411EA103A103A10840F170D170D180A940CBC10BC10D810D810D810A1119F16E312E312C412C412C4120613100F100FC90DEE0DEE0DCE0E730DBC106C126C126C120C120C122317351735171216E014E014E014DB1238113811E40F650F650FDC10DC108E138E13EA15EA17EA17F5172317C913C91348155C155C15101562116211E60FED0FED0F1E0F630FA013A0130F10DF170A180A18DA17D414D4146B147B147B146E154213FC12FC12B910B50F810D810DC112441144119E10FD10FD105412DA16DA16C81468158C158C157512C411C411F310DB10750C750C8B0E8C0D8C0D9B0E07112711271182115D135D1381155A165A166C14BB10BB101210A00FB705A905A9052612E512E512A311091509153E13C011C011AB124E146515651507111E111E111011', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (133, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4150F360A360A110A2D08E507E5074B0A5D095D09320A6C0C6C0C9A0E5C0F5C0F57103610C10DC10D400FE208530753071D086708670887095B095B09D30A1F0D381038102C117610761024105D0F5D0F680FFF07920892086C08ED08ED082F095F08820C820C980B650F650F361293129312D0112C10A90F1B0EA209E4085B085B086D09550B550BD908D808D808FD091A121A12A2128712371037107F0F06105A0B5A0B06093E093E09E908A00AA00AB20A9F09FB0AFB0A59102911291142106F106F10B70F2810C80BC80BC50B29094C094C090209710A710AB00A130B130B030E11101110810F000FB310B3108B106B0BE809E8092E09980998095609490B490B5F0A970A4F0E4F0EE80F28102810A80F490F490F6D10970CBB09BB09420B980C020D020D9509200A200A2609770B770B0F0FF010F0100D0F210F31103110580E500CC90AC90AC80C230F230F490CA807A807D00B3C0CD20CD20CF20E490F490F470F031103112C11DF0C9E0A9E0AEA0AE80AE80A510EDD094F0C4F0C1E0E7E0D7E0DAC0D88108810530F1A0F3C0EC812160C0609A308A308170A8D0D8D0D5B095F0C5F0C820E400D400DF20DD40FB710B7105B129610BF09BF09B607A008A0087508A40AA40A3A0B820C2F0D2F0D0110080E080E30109E119E116E12BE128D0B8D0BE007BB09CC0ACC0A5809B308B308B00C550E550ECD0E890E890EC51008121E131E133217B30FBA0ABA0AB90AE10CE10C430C500A500A200CF40D950F950FD10E72117211B1104B124B12EC172C0F8E108E1052104B104B10520E310ABF0BBF0B9C0DFB0FFB0FA5106B126B128E12C110DB1BDB1B480F640F5E115E11D511E610E6105B0E4A0D4A0DC20DEC0EEC0E591062134A134A13F6116B1C6B1CB50F030F4D114D118311B311B3119D0DAB0C981098107C122A132A134014D413D413DB11E718E718DB106E0E270FE710E710B410E40DE40D850E9410941018105A125A12551340149014901419150C1602100210DA10C713C713ED10E40EE40EB30E9610E20FE20F9D11BC11BC111D1358125812A714AD148B108B10D4101E121E12B513BA0FFF0DFF0D021025132513DB1256115611F112EC112F122F12F811', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (134, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4330C860B860B640DB111B1119512B116B11611153316761D761D741BD517D51752173C173C17DB135C0F340D340DB30F5012961396134F173E183E186C17731C731C8620481B481B8E163C1670187018A90F9B107C117C1137126B176B17B917C01BC01B751FEF20261E261ECD1FC915C915B0195117511706102511401240125517E017F217F217101896199619AC1D21182118D817BD14BD148A189313E1182810C311F61236183618D81A33163316FC152C162C162C168913140F140F5A10F312F31229148315F011F0119F16B11CB11C1B1B6B176B17C2130216E715E7159515101010101D0E29112911AE166A159D149D14F21BC11B6818681894128F148F140715891489141014D311D311200EA60FBE12BE124A10AB1240154015B51720182018CD141B111B110B15A11444144414BA13FD10FD10A90F50125012CF131912A114A114EE152A17E013E013D31143114311AC12E112E112FA10AF10AF10EC0F9E10851285123C0EB012491549157D15091609163312951295125D136214680F680FF3114D104D102C0F6111611196100513E718E7187915141514152311DF12DA16DA162514861286128813B611B611E70FB5102713BE12E01658196716671660135D135D132D1514181418C117A317A31788134610A20FA20F9F0F6E13441B441BE31C09170917A01378137813FB1572180B180B187117AF17AF17B81165116511C010471C441B441BCF18E318601560159516851585158617B51BB51B411C591A591A821B0717F418F418A915DB1B9119911931172B172B17971610191019AF11AD1ADD1ADD1AAF1CB417B417071625182518BD136B1CE218E218E3174619851885186A18DE12DE12E317781E781EF21A641564153916BC1648174817E7183F191C181C188C17E716E71617189E159E1522209020931C931C701663156315A317761776170C160C163B173B173716C318C3181F17DD17731F731F491DCE1CCE1CD51572187218F215E5160C1AAD1481140D160D160D164313DB0EDB0E5C12E31FE31F101EDE1EDE1E9F19931BC617C6173A19091A451345130D167B157B150D16631663165317B4184A184A18601E871A871AD61A6A1B6A1BB21A6019', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (135, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F417149A199A1948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F41B181C181C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44019F11CF11C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E118211F731E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F483151914211F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F317F317481848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F480148014F91648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C913C9136A1648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A515B315B31548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40A165715571548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4271306188B1E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46E13351E991B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4471C361EC81D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DB1BDB1B7C1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43F103F10CA1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45B162C172C1748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A115AE19AE1948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40C1AB417771548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4091AA4191E1948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46019F81ABA1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (136, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4490E490E1A0DB60CB60CB30D041004105C10811181111B13E612FF16FF162B1AFB19FB191818D119190D190DE60CE20CE20C730EF30E46104610221059135913BB13A416A416CA17F41BF41BFF17A01CE10BE10B3B0C760CA80FA80FC50F5B0E5B0E320F550F550FF610C315661866184D1D3D193D196618480BDE0BDE0B4F0C0D0F0D0FD910231023100810F20DE80DE80DEE10CA17CA17961B741874182121AF0B780B780B1E0B950DDA0EDA0EAF10B410B410DE0EBD0EBD0EF712D118D118491BAF1ABF1CBF1C7D0B2A0BE40BE40B190E680F680F0C10930F930FDC0E12100D140D14E81A411B411BD219D21CD21CFE0D7A0C690D690D690D1F0F17111711B010741074108012F514F514AD1AC61BC61B9F17001C01191A0F950E650DD30DD30D780E771377138712D611D611E7148B1861176117B5177E177E17001C1015A40EA40E640D930D930D990EEF15EF15621342172F192F19EF176E186E18D8171A171A170A1A5910F80EF80E280CC80D2F0F2F0FCB16511A511A511E781D781D791F791F791F9E193117CB11CB11B910630F4B0E4B0EC90DB50EB50EC114E11BE11BFE1C3C1D641E641E5B20971997199717001A001A0A126B11CF0ECF0E880E301330137913C019F119F119701B971B971BDC1D711B711B2A1A70165E165E16F3130A1131113111C513F715F715A419671F671F7F19B31AB31A991AE317B913B9136B12AB0DAB0DC2142E143B163B16D214762376237623B31724202420471B4F1A4F1AF818C515C5158416DA0FDA0FC8162D144F19BF17BF176718361736179C172420242024209F199F19DB19E41DE41DE41D1213AF11B214B21484175C1D5C1DB817E816E8167B160C1EE71EE71E211ABD1ABD1AA11DA11DA11DC616EE1544144414A712F41CF41C7A1B381C95159515F41B261B261BBF19211821185417A11D03180318C7170F138A138A135D189A1C9A1C281E5D175D173617341834183D19691831173117C01709170917301556197C197C197417211C211C9C16E313251725170C1889158915EA15EE16EE16A0147F177F171B1626142F1634183418EE174F134F13BE11B40FB40F8615C913C9139F166017D315D315B91405160516', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (137, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4D1194517E214E2141713E115C20EC20E8C0E8C0E8C0EDB0F090F090FB40FA01058145814D5132011981A981ADC17FD13FD13E115E115E115E712E712E712DB0FDB0F361236122911C514C514D510000F2E1A2E1A111E111E111E111EF5151B131B13E712431443146A130E150E15F612AF16AF16CF137E11301CF713F7135514111E111E111E7415741556153E143E14FD153616EE15EE155B159E129E12E9119E1AE314E314221047104710D210A812821582151E15881588157B15DB15DB156C151112111279117A176C166C1663118511B510B51071138B148B142015AB14AB1450156A167F147F148612B312B31201195A17CB18CB18DA168F128F12B5131F141F149F14C3146A136A13A11339123912F712CD12CD121015BC0C3E0D3E0DD10E3E12F914F91423198E168E164E16AB17AB17ED14291229126B109811B7127D0B7D0B281AA10EA10E2810FB0FFB0F301588168816F718D7185C165C165614CB12CB12A811E1119A0A9A0A180E650D650D360EAB0B970F970F4A166E176E17041745154515FC139F149F1446129511440D440D2B0E650D090F090F410E8E108E10AB130C140C14C0157B140E130E137B13FD12FD1262139312BE14BE14CC115A0F5A0F9311F217F217A7132719B414B4148812F111F1114A1267126712C2126F0CEA0DEA0D420D3C0F10151015A115511951191D1C2E172E1761148B118B11CE10CC1375147514EC0F5D0EF010F010931047124712C116A11CA11CF5195415171417148B11611061102814CE14CE14AF114E11121212127B17421A421A1B1D2E1E081B081B34153F113F11C40F841084109D1224182815EE1500127F13A915A9156B1C0A1C0A1C931DEF1CEF1CA3182612261226124E107F117F11371222144D174D174D1706160616861B071D071DBE1CB71BCE19CE192E1330113011C40EB10EB10E17116B116D176D172C178815CC16CC16391814181418181B39193919CA12421442147214340F890E890E1E0F451571167116A9161C151C154315A117A117F217D117D117D117231523152315A8103F0F3F0FB210AE13B016B0168A169E149E14E117E616571357138613D710D710AC122315231523150417F911F911', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (138, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F42011790E650F650FD410AD10AD10EF0F1A1350125012E6126713671338123E123E12C90EBA0EBA0E000F490E6E10FB10FB1003117E107E108714AD14AD14EC12DC12DC1285144C14B50FB50F630D0D11CE0DCE0D820FD110D110A410871087103411B411CA13CA1326153F133F1389131C101C10D0108112DC11DC111110981098107F100110F70FF70F98121D151D15EC15611261125A13FB11511151116A118C12001200123612601260124711991099109216D017D01794179F12E011E011C411B50EB50E1C0EBE1364116411AF10971097108E10DC106C176C17B817981798170813B312B3124512AA0FAA0F1C0E29154F1582138213E41117101710F010F314F3142D16511851181715E91232123212CF0E710D710DB71227141F131F13DD0F870F870F4A12A112511551150A181419141985107A107A10F00D900C900CE111DD103C103C10840F020FCC12CC127413C614C6149F18AA15AA15C611710D4D0D4D0D280CBD0CEB11EB11EB11F10FF10FA9113C123C129711D312D312E414121277117711F90DAD0CAD0C3B0D7C0C5C165C165C16D910D9102B112612E212E2122A13651465149C0F9E119E1168100C0E0C0E650D2710A2135712571241129610961053134A134A13B311B811B811501042110C0F0C0FA30C6C0D6C0DF00FC5128B118B114A11C312C31292149E143F133F133F13351035104F0ED10BD10BA80AE40DE40D2E16F013E911E911D711A0127A147A147A14171517152115FD15FD15DB0FEE0AF50DF50D3E0E7512751228155D131E131E132B144A154A151314EF15EF150716DE169A109A10D00D550E550E54115A155A152214F1119D119D119E124A156A166A163F1B3F1B3F1B0D191E151E152E0E020F020F4D10BA1214146B11FA11EE1142124212CD13051405147D17F317F31782192F1799109910970EB30EB30E1410631499119911A012371337131515FD16FD1603172418351635162F17E113E113230EC90EC90E13104A146C126C12AD12961596159615D814841684161916FE13FE1362168812630E630E230E490F490F35131817861786177216721672169316951695167616C014471547152B13F20EF20E630F950F950F1D0F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (139, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F493125115C016C016AC18AC18AC185D1AAF1AAF1ABA1B641DFC15FC157A119E069E063807540754070D111814A616A616F617D815D815D815F018B117B117C6178E118E11B20EEF06EF061E06DF03E5038112FF19FF190D1A0D1AD91702160216B915FE14FE147D14C913C913F10DEF06ED05ED055604C8020D170D17D217EC19EC192E18F817F81707192F167D147D14C91381118111C608FF04FF0444047C02231323139D16B7195F195F195F19E118E118B71740174017931521112111A1094E065B035B0330022313F315F315F1185C1A5C1A5C1A10181018C5174F17BE16BE169E139E099E091D07E104E1043002C11082158215541A35193519EC19EC1972197219E018E018E018E514750C750CD3067F05390339039311C719C719C719181932183218321888198819CA182718271828176F11F007F007470639033903BD0C9A14C719C71955194818481848181E199A189A18E617771777179314E70EE70E2705270527057C0C32127C17DD17DD17FD175D175D17BE18AA19AA196D19A518A518D714330DB00CB00C6E047E01041804181318411941196F18C617C61713179919751975192117D814D814D80FE109E109B405D202841684164119411941191E16EF175617561723197B197B190618251625160814F30FB405B4052B088115C418C418C418F114F1143D1632183218CB18CB18CB1859175917081408140C10390C390CAF042A16C615C6158D1511181118A315C8156A186A18B614451745178419C418C418820D720372034803CF14C81573157315E6141D151D15B215131513157B1600120012640D260DAE08AE085D047E037E0314142014A015A0152715AC13AC13D315D514D514D514E312151015101510F405F40530027803780363143214671567158C15DA13D315D315AF14AF14AF142611CF10CF10AC089405DA02DA02DA02B202C517C517A915B814B814AD14A314A3141815A713A7132011BF0D830A830A7008640A640A9F0748038E158E15E0165F155F1564154415F314F3149D10901090100C0D950F950F950F640A640A8708D5087515CF18CF18D815E115E1150515F314F3147D137D137D136410950F03100310C010CD08CD084307', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (140, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4CC02DC02DC026F05F103980298022901F403F4039206C509C5090E0E151215122014B60DE910E910E503DA039E039E03B603DB03DB03820185018501D00027082F0C2F0C9D11F611F611571496139613C802D302B503B5032C048A04210121017B01C4FFC4FF6C01300B300B610E130F130F0C149F13B0127C02E40278011D011D0127023A023A029900EF00EF008301EF002201220115021A101A10A310B012300230029F01F201F201CA01CA01CA0121029101800180017401210221025302A409A409B20B241630023002AA005D02EB01EB01CC0178027802680179017901FF02C007C007230ADF0BB70AB70ACD15CF02900190014F01BF01BF017702B102B102BC019102D505D5055D076E076E07E20ED90ED90E8016AD014C024C02AC026A02CF07CF07D60A430C430CEE0ACE07CE074906460846084D0FEA155F175F176902A80268046804E8042B092B090D0ECB0DCB0D6510F70F7B0F7B0FC80BD10ED10E6113E51DE51D7E011903FE03FE0370079A089A080D0E0D0E901090102910B30EB30E3310450C450C170FA920631ED2024A015A039C039C0313045D075D07C6079C089C08B50F161016102C1332127E117E11EF0B4C0AAC05AC05AC05010401046103510351038B03260404050405C305DA08DA085812E610E6102F0CD107800180017D05B800FD03FD0301032E032E0356034E054E05C005520B520BF00A8C0B9B0D9B0D2C08F1021D041D048C03880488041F048C028C026C01B006360736073A0B280E280E4B1220102010F70EB403800280023F0204020402ED013F0402060206B006C20AC20AE60EE80EE80EFC109211B712B712CC0378039CFF9CFF9FFF380238027602180518053D01DA0CDA0C8F0FD00D9C109C10711213141314B2023B0236003600560256025602A405BB06FD00FD00B81005170517571393119311CE113512351248033A0373034D034D035602CB09CB09040AB605B6059A1266176617F1125E128B128B123512AE10E103E1036704780378037803FD0DFD0D370EC205121112114D11901190112C11D811D811D811640FCE08CE08E108A606A60643072A0477047704ED0BAF0CAF0C0A11A110A1108C12540F250F250F3A0F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (141, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4970F8A0D8A0DD70E1D121D121D120915591559156512961196119F118A128A123A1177117711B710AE1204120412AF113E151117111792132D172D1712115E115E112C116D11DF10DF109410710F710FB012791395179517B6177B1B7B1B531A5D195D19A2139F149A119A116D1193119311CC109F0F9F0FB0124015F016F016DF17A719811781178517C914C914C613B011B011DA11F010F010E61008108A101E171E17BE13EA13EA13210F690F690FDE12FD11FD11CD1296124312431270128E108E108A0F960F7619761943131F131F13E0127A123911391105129B129B12F812321132111B11B710B7103810960FB118B11862178E135F125F12C912E412E412FB12EB14EB148313B512051105114F10A80FA80F5A0FB818851D851D2714921492145D15A813A813B10D3413421542150D171F141F14CF125D0F5D0F08105B1CAA19AA19FF1AF8198B168B1625119D0E9D0E890FD113D113E6134A104A106E11501009110911631EE414F517F5173F1A4D164D16C2153F0E3F0EA70F831043124312D50FB20FB20FF10F031103114C0A8E0DB414B414B415741385148514EC0DAA0FAA0F0211611161111E10AB0FAB0FB210B0112912D1073B0C2E0E76107610F5118B128B121913091009103512D01214101410770F401140112A12C911C209C209560E890E890EB212AD14AD144C141B11E812E812831463106310ED0F0811081177113F11C20BC20BE110A80E4D114D1148119E109E10741295129512B713DE11DE113D117511F810F81042129A1494129412B80FB211B2110E11E30FE30F3F11E91197139713E110A112A1129A12E611E611D1111314F311F31110107B102C0E2C0E0D10481048101212180F180FB410651165114812FE112F122F12151324110A100A10940F600F600F5D0FE50FE50F6B117111CC10CC107510801180114613C010C010AE10191170117011F31149104910730F0D0FC310C3105E1106100610B110F410F410821315110411640F1B0F1F0F800F800F060FBB0FBB0FA30FB011B0112C1102100210B0104210C711C711A8100610D10ED10E820FB60FB60F2A112D112D11CF0F3010D70FD70F950D150E150E9310A310A3107F10C10D', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (142, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F40010351235120D1263126312D50E9B0D9B0DE00F4011401131125012C810C810F213D813D813BE13D20F991299127811B80FB80F180F250E430F430F2710E810E8107E11B610B6104C120013001354149E106C104B104B10160F890F890F8E0DAA0EAA0E5210E80FE80F2B10B70EBB11BB1116122A152A158A10DD11171217122C10FA0FFA0F820F870E7E0F7E0F300E7D0F7D0F670D051105110413D713D713960F3B104B114B113810910F090F090FB00D8D0E8D0E510E1A0E1A0E1C0F9E11AC12AC123B13DA14810F810F0110F50EF50ED20EB00EB00ED70B070D070D740D0A0F220F220FA4116A126A120113D412101010101210B70FB70FFE0EED0BB30AB30A330D3F0E3F0E340F210F210FFE0E40114011801147114B10321132110C10080F080F440B3A0C3A0C6F0C31103110FE106510C90DC90D9C12D211D2117D115211511351130D11E60EE60ED10C5F0AB10BB10B5210341134115D129712971258146E156E1565176A117D137D137B11E910CB0DCB0D5A0B440E440E9E10F411F411701670165F145F145D161C1A1C1A2912741274107410A40E0A0E0A0E420B4D0F4D0FD7118614FB15FB159715051F051F381775177517C9110412741074101A0F6C0E530D530D0F12131413149714581658166515FA14FA14A215A416C71684118411D610C00EC00EAC0DC80FC80F22124E144E14B71613173F163F16231AAD18AD184D160015F910F9101C10190F190F470E0B10951095104F1498169816B31AFA18FA185E1A0C1E0C1EE313F01302110211EC0F0B0F660E660E571010101010EC11D213D213C315C915D514D5145913900E900E1C134711761176110011030E030EAD0E70107010D4125313FA1DFA1DBB148C158C1575143E1C3E1C41129A10CE0FCE0FDF0E370E4F114F114610841D841D961C3E1E3E1EF71CD314601960190624261B261B0411FC0E6C0E6C0EB110C711C7110C132A1C2A1C2A1CB61AC21DC21DE51CB616B6162F1B591459140610A10E450E450E72101F1149144914051AC818C818B81BB81BB81BBA10F011F01159140C15C016C10D590EDD0EE610E610E61B89188918C519A01CA01C4B196519651965193F1D3F1D3F1D40131D17', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (143, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4E112E112E013EF12AF14AF143C169619961997198B198B191F19C614C6140B16FD1B71167116C21C7C15CE14CE142E164F1A4F1A4F1A4F1A4F1A8E168E16281828182A18511751175E14C016C016321B7E169A179A17DC178F1B8F1BE41AE117E316E316E31698159815E31765186518DD17DC168F178F17AA144016891789173A192A192A192C191716171695163E173E176F121514C411C411771078107810DA14DA148B148B145416081708174E177A185F175F177D179F169F164214EB14EB1403112D112D11D4127B120614C915C9152813691469146618A118A118B41866186618B917681689138913BD103C1935113511B912901290129211571357132915391749164916F015C716C716B917B513B5131D12CB14361236121B1252125212AC11F4102C122C12A212B215B215B917B215B215E412E80F140E140ECB0DC617EB1BEB1B1213CD11CD11D50DCB0FCB0F8E10D612D612D612E70F511151115E13890C890CED0BA816EB1BEB1BEB1BA913A913AC111C10C20FC20F410FEE0FEE0FE70F9C0F9C0F060FD206D2067E08731A731A7B177B17BF1451155115DD1915171517C80E1A0E1A0E9A0CA70CC903C903C903EB02EB02C71665170217021724161C181C181C18461646164616F411820D820D330B8E028E028E022D052D050015FD12181518151815231923192319E41E8D1E8D1EED1C4F0D4F0D9A0E230DBD06BD06DE06200580108010DD12DD12DD12481A3C1A3C1ABF16401B181D181D2D16391639163F14281828184B0F7409AA0CAA0CBF10E70DE70D5C15EA198F178F17241911181118B518B91AB91AFE1C0519D918D918560C0C1612161216CF0B5C155C15EA19EA19EA197F1A741B741BFF1C701DC118C118941CB219B219E21A1D19121612167917701A701A701AC517FC16FC163317071607168F15C017C017311B5A185A1818176610E50FA611A6116612701A701ADE180A190A191B19E916E9163E16A52341264126AC1ECA1BCA1BC016C0164E0F4E0FE60E380E380E680D670DAD19AD192516331A331A331B021B021BAC1EAB1AAB1A1D17C016E50BE50B410B7108360836089A0C550C550C6F0D9D189D180E17B1193E1C3E1C9C1CC11B', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (144, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F47C1D7C1DB91C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C91DC91D691D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41019971B971B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B918A217A21748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F499178C17C61748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43C193D19741948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CB14FA18581948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B30CB30CEF0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4290C290CB30C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4210BBD0BBD0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C4027206720648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46A05A502950448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42005E305FE0448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47409F003A10448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DB05DB059E0648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4430E430EBA0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45E173D0E3D0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4881C880E880E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4201F1019E31548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C11B861D431348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (145, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F48218351AA61FA61F0C1EE412E412BC11AC0BD810D810E3128B198B19A9175F155F1574160516051656187A1CF91DF91DF91D0F138B148B149E0C560D560D9B101F151F152C14D1141E181E18FF146015A218A2186E1BCC14CC148B1414131413C20C870F870F9B10BC11BC11BC114E141E181E18F914C01398189818291A69176917481291125F115F118B11B512B5123812501350132615A414A414A312CA12671767171D22221B8813881307116F106F101711221122118A11551206150615EF11AB12AB1248120D1A1D221D221D22141F141FFA12260D260D260DD610571257122C13811581156912551355130E145F19412341234123141F31143114BF0C53105310111297149714DC1231153115E31397135C155C15A41293183B183B18A71988138813C90EF610F610941423150E150E150F1375137513C214AE17AE17241FCF1BF217F21734157F201118111813110B140B1441160F150F15D313E513E51378152E189017551EBC1B841A941594153A19FC1FFC1F0017161616165917D914091309135E13BF15BF151A195E16721D721DD118A414A414E61AAE16AE1695140F14B913B913501458165816CD1459175917C918E7137E1C7E1C41183018221722172C17301630169214371437144E1482168216381540152516251698127021C316C316AF182B192B1931193C183C183E169B13A612A612671420162016D813B113B1131C13C8195D185D184E17D217D217A2161915D414D4145313F314F314C814DE14DE1447143E14D611D61169187118BF16BF165917D315D3154213AC14AC14EC134E16231623168B14BC15BC15B914081208123616E617A617A6176915961496141D14CB130C140C14EF1589138913F411AE14AE149816B71381132515DF161A16BC14BC14CD132116211604162415241511147313731304132914C213C213B713F31393169316AC141D141D14AB17531653169C161D15801380137513D113D1130913EA13EA139E14F31355135513D2135E175E176D1705189F169F166C136813681333152A152A156716C3155C145C14AC145112C812C8121B145D165D165916EB14EB1473139F139F1361154515C315C3152F16501450140A13', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (146, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4F2140115351535154214B817B8175318FC16FC163F16371237124F0F9A0F680F680F3C0F861186116015B515B713B713F512DD16DD16C916F8143F163F16C012E111E1116E0FD80ED80EC50F26122612C013BB154215EA12EA12C015A415A4159A1511121112B31101110111D80E0E0F34133413131423182E152E15F314CC12CC12C2138F148F144C13B612D411D41188113211321196118714871462178B18E112E1128C15B215B2153016DA15D013D0136413E210E210DD11ED11ED11FD126D15F013F013D5136B1750175017BC168C148C141B16AB13AB132113EA11EA113B12A112A917A9179D14571357138E12EF176F156F157614EB11EB11D511A211921092108710D211D211CD1235163516B51569136913E412D217CD13CD13901368121F111F119210D510D5105A1002120212DB173317CA14CA1469142314231490173613571257125E11281028106A10D210C010C0100014241A241A0E15A213A213E7171B161B165E161513B312B3122E102D0E0D120D126412D013D0139019931C931C8715F3145D195D191D16E71790129012A010BE10BE103910B012B012171456195619F118751BBC15BC154A1415171517D4168F1815121512351115111511AE109D1116151615AA1A301830187714F712F71201132418241811173318E812E8126C125211F310F310C01188198819521D93179317AB139514051605169518721972190718D31218121812FD1064126412A917791A791A7A13FB14721472144D14051605160B1A071A071A61188A10FE11FE118311B412531A531A261DF21BF21BA817151815181E152D142D148B1843198119811981137F1239133913031A0B1C0B1CCE1CE118E118E1165E17251525159E14CC17CC1742186C186C18F313871308140814DC19D91E0A200A208E15AF15AF1572150E150E150215501450144E14B516F011F3136C1339135F1D5F1D5F1DC21FC21F38156C126C1249122912C213C213B7134214421461148A168C118C11AA13FC18FC18091AB81DB81D3F142C1847154715D01236133613D714821582154318EF168C128C1274119E1CF41BF41BF41B721B721B4719D219D219C41534173417C113A319251B251B9C10', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (147, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F41317181818187018BB13671767176617E517E517C11543144314F612D80ED80ED00DBC0D190F190FD0149A16891589152615D216D216A117D217D217F01597147B137B13AA0FBB0EBB0E300EEB0DEB0D23185615D614D61475142E142E14FB15E9173E163E1607153C123C12050FD30CD30C380DB50FA9118B1855146113CA13CA137914ED14ED148F1592159215A814B511B51174105A0F4B104B101210FD116B136B13381223142314E61364146414B0132514801280127F11140F140FA70F74107410D6117D12761176115111AC110012001223125211521193111E121E12C50FF90FF90FE91091102C132C13DD13B010DC11DC1106120E120E125C12BF12BF12F312791246104610CA10FB10FB104E11BA11BA11E013EB101E121E12271176107610FF11441136113611D512FF10FF10EE11FF12FF12D311421658135813A0132B13FD10FD107C10E110E110EA11EC10EC100513121212122F12261593179317EE16E514E514E71713135C145C14A512301530157B13C81345184518ED193E1D3E1DCD181B1B1B1BB81AF11CF11C8F18561B8115E716E7167C1843174317AB176D156D151B1A2C212C21501EAC21F920F920401EAF177B187B18C718A819A819201DB518B5185C18B918661B661BA41EF419F419791CFD1DFD1DFD1D221D75177517761AB51AB51AA71CD91A6B1A6B1AE91867186718EA1EA61EA61E7720791B701770177017CF17D919D919AB1BDD17DD17641C80198019871754185418EA1EA91B77207720EF19601860185617AA17B617B61767174D1A4D1AC618EB1CEB15EB159218851A851A401DDA1CDA1C7119571A571A3B192714C014C0149418EC1940174017181783178317141A0A180A188C1BDF16A81BA81BD31859175917F0114412321132115C11251925190218111853165316CB14B716B7160413D314D314B418D31AD31A8A16151279127912F612F515781978191519521552158F14EB13EB13450C4D18AA17AA17CC10480B461B461BD410B30FB30F621A861A861A9119FD16FD162E147C0FA90DA90DD513D517D517F714070F25112511B90F5B0F5B0F360F970FBA16BA163C17D013D0137E0CFC0AFC0A3E0B730A730AF8169B0A', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (148, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F43016D615D615B6176A196A190515D014EF14EF14EF145C105C101A12651265120215BA0ABA0A3409FD136E146E14C8169116331433144415C414C4144913331433144515451568176817B10D210B210BA9114C14751675161A1670147014DC15DC15DC15A215CF1384108410DD1968176817F60E960B960BFD11D3149A159A1563151D150E160E16CE155C165C163C13BF10BF10D218BF18BF1885181F0BB20D7D127012A51586188618CB1808180818E715BC14BC145B147813C121C121DB1842174217430FD20F761376130D14D016D016D016EA15EA156F16A61586158615A9175E115E11D30E450F450FFA0FCF0F2C132C1328177E15D515D51512161B151B15CE1A5E195E190016EB12BE0FBE0F23101A101A10B010FD16B613B613EC15961796176E1B971B971BC218C21886138613A0130B100B100E11EA10EA10ED11EB1473157315D1164E1A611861189E1B231B231B821ADE15DE15DE0FF310F3102B117E1257175717F61B9F16C919C9193317A31BA31B4C1D7F1D7F1DE515161051105110CA11F211F211B81309150915AF1783198616861641140C170C17F513B312A112A1120111A111A111C6111D121D12E515AE17AE17221D6F19D01A35153515E20F121712173F10920E920E65107A117A119D11941549194919E61211109F1A9F1AD01AA612A612360F12171217940E780EB110B110D511B012B0123015C614C614EC1625189B179B17BB1CBF0F8F0D8F0D1E0DB70EB70EA40E760F760FAB13651465144A170C1628182818271C1319281A281A4E15351135111C144510451045106410121612161914FC11FC11540FC613C613B413DA13E30FE30F050DAF0CAF0C3716C30C480D480D3111321832184F16081308137A14B0136B126B123810F40C0F0A0F0A920D920D920DC80C390F390F7F1132183218931940199F159F15C515B013B013480B640FBE09BE098D095D0A5D0A800F71102F112F11B91330183018EA1ADE1ADE1A011A6C15DF12070F070FBE09380B380BC40EC30FC30FF2107D137D138A1484178417B718A41990199019BB1A6917D00BD00B390DDC0EDC0EC610EA11EA11CF100B145B145B14601448164816CA199E179E170B18AB16', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (149, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4AE0EE009E0096107940794077508A303A3031F0864046404A00BAC0C8C128C12BD0F4B0F4B0F650E5D12120C120C8909380A380AE108EE03B503B503B903C109C109B8094A114A116E0FC20EC20EAF0C2E0ED7091E0A1E0AFB09F00AF00A9F05B205B205EA05D410D4109B1132113B153B15FF12E01CE01CB20D120E621062106210F110F110F110B006C50DC50D0E127E117E118C1616161616EB140A1A0A1AD20FA90EF911F91162108611861186116D0D9D109D109D109810981050119F11771677161E182E1B36103610E811E413E4136814241324136E133813381338135213371437145613F218F2181E18F118021102118914141A141ACE199E1233133313AE1386148614E013B816B8165F154F164F161211441205126D186D18551A8D138D131F12D611D611A610A110A110DD15BA16751675168E0C4B0D4B0DCB0C5717AB18AB189E144A104A10B50D6F0D400E400E2B11E115E1151716F314F314F215E215E2156E166615571157113B143E13C312C3126F1175107510340FCD0FCD0FFD0E20156F186F184A18E716E716AE17EF0FDF12DF121B144B144B14FA143D133D13B10E620EA00EA00ECB0E520F520F1A0F701170111110080F31113111F613AB1BD61BD61B471F6F1D6F1D6F1D191519159B0C240F240F1C0FA60FFB10C917C9174713F519F5192B1C7F137F13E717D61ED61E01144312761076109611F212F212F212AD0F4D1F4D1F9518BA16BA1625132213201420144F148414841443122C112C119212F212F212F212221337193719A815A61680168016B714401440147813C014C01473146D1319131913E812D812D812AE115A1571147114221666166616FC149F139F13AB141014BC14BC144B11151315130813751275124A12ED118F148F1431161416E314E31477131B131B13B912E412E4120111331233121C151C1517141714DF126F15CB16CB168E1457155715F910EA11EA117F12A8123511351158122F122F121C15E715E7156917DE14E314E314CB164613DE10DE106A106B106B103510CB10CB10A01183128312D4114710E60FAB164614011701170117C7158811881171108A108A10901002100C100C104910980E980E920F560F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (150, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F42A0E2A0EF60F0010AE11AE11B7113211321160106E0F6E0F790BD710D7108E1063107A117A11C70E0C0B2C0F2C0F9A0F371037103F11E210E210A20F790C640A640A430F2A102A102911E011E011B80E13171B1B1B1B29143B133B139B0FAC0F960E960EE90B700B700BAC0E900F900F47109010B50FB50FC31C791D691669167713F910F9100410E30EE30E700ECA0ECA0E100FEF0F64126412BC10FA0FFA0F2E1BF915AB19AB19A718F512F5120811400F340F340FC20F18111811670FE511E5115111970F3D11F1184C1D641321132113211303110311B30FB30FB30F7E0F1D101D108D103412AA11AA111710680FAC12AC12BB11AD11AD1121131F111F11C40F1E0F950F950FDA1039103910B211EC0FEC0F010F820EC212C212421298109810F40F1110B40FB40F7E0F7A107A101210611061108210A810790F790FA70F1713980F980F4A104010401066117E117E114510C90FC90F8E11A510EC10EC10BF101F0F1F0F500F251188108810991086108610F712AE13201220121612411241121F1315131513D0105C10BE0FBE0F8710971111141114B21001110111FB118E128E127E119B119B11AD1129111D101D104210FD0FFD0FFB1057159D119D117B0F220F220FFD0F2A11A312A3129F11C110C1103B10F90FF90F2E10E00FE00FAD0FE40F401040101E0FCD0E7E0E7E0E09105C125C1265116B106B10B110760F7C0E7C0EFE0EEA0E9C129C129C12CD10CD10600EF90EF90E9F0FC110C1104E100010701170118211BC0EBC0EDB0B100E9C129C125B12F011F011F40C850F430F430FDC0FC10FC10F7811FD0FFD0FF10F1B0F140B140B9E0AC610C811C811100F520D520D7C0F810E810E710FC40FC40FC40F780F370F370FA30D260C260CB90C11154317431728146410641066102310AB10AB102C1182128212090FC40EC40E720C720B720B3F0E5A143E143E1428147D10741074105710BA10BA108D12B911B91196108B0E850E850EEE0DBD0CBD0CE60FB60FBE0FBE0FAD10C211C211E5103B123B126B111110F70FF70F6A0E3C0F3C0F410EFA0CFA0C560FE70F62106210EC112D1218121812901097109710600F0F0F0F0F9A0EC9105C105C10600DBF0B', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (151, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4B20DB20D800E4B114B114B108E13B71CB71CC316DF16DF16CB16D516D516361836183618A6123512380E380E060EAC11321132114416C715C7151E196F176F17A31A0B16B416B416290AAA09AA0981058C0F1E0E1E0EA610D513D513F4124D114D115F0E930EF80FF80F350D6A0E6A0E9C0DA70BA70BA2092F0EF80DF80DD50FD710C512C512BA0FAF0FAF0FD210661066106D0C6C0D6C0D200CE50D0B100B103D11640F811081106A12D010D0105111AF0FAF0F85114B11F018F018F018CF0BCF0BEE1712191219680FC910DF10DF10E210C71098129812F113B714B714CD141D141D14131F8E178E17771D771D2F1F820EA00FD310651165113911C212C2127A14F113F113C9145D14131F131F3C18F81FF81F382096225813581353136A116A11C111611161114013BB12061106113B125E135E135E1BEA1BEA1B1B1DDD1E5913591354149F145018501856120E130E1307141015101577134E104E106317BC18860C860C2A172111871087108F1164146414A813051605165D164C16FB13FB135A11761A761A631B201220126B0B4911D312D3129C119F115F125F123B164C174C17AA15CD14CD140E15371637164A16911AD314D314610FD51067116711A115661466143315FA17FA17B3150F181F181F180F154F154F153F1ABA14BA14EA0E5110FD11FD117613F213F2134A14A716B014B014FB15851685167D17BF15BF1593169F147413100E9A0E1B133B123B127312FE13FE138314661666169318981998196D19C218A915A915C31ABD1BE30CE30C1310951395130F12DA12DA1290124D14D716D716B519D119D119B31979177917C81AAD1B3A0C3A0C310F8B1435173517E714781378130E13E816E8169119691869186F1915161B171B17E4186D0E87128712471695169516D715C215C2153E150F1584158415B11589158915C612C013C013DF12B30E9B109B106715BD14BD146816AE16C417C4173018BD19BD19BA15751575157013E21254145414770D4610A619A619351608190819B71C811D811D4A1B851885184C15F615C615C6157414A114A114BF0B1B0FDF19DF19A81996199619E61B8D1D6F186F1859184D154D151515A615A6157B14C213C213', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (152, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4610F610F4C04C502C502FA028C068C06650765076507820B9D189D189D18AA16931693161B19C6183E033E038C01E701E7013B035E02980198018403560356039503811381139516951695163B144D14BC021F031F035502BB02BB02190020FF20FF6AFEB102B102980160031E061E06DD0CCD10CD108F1C0B15D809D809DF0791059105EF011C01C201C201C501470047002901C901C901E505E706E706E40C4E155E125E12A90F5906D804D804D202100210023D02F901F9017B00BA0012011201CB03720572052F1F641957145714290AAF04AF040A0469026902DB01F9017F007F00820090019001BD01DE03DE0396226F224F1D4F1D960E7A06900490041802190219023B012A012A019E0093009300A300F10083FF9C209C20FA1D50115011E1092A072A07B402C202C202C7013701D600D600730135013501E7FFD4FE861B861BCB1F0B1C0B1C3512440B90039003C40214021402E20215031503750183018301E100E100161B161B9822F31E50145014CB10D509D509CA048A028A0291029702500250022D022B012B017C013117F31CF31C3D1DAC17AC17D4133E0B3E0B3509F104290429042604F103F103F501AF01AF014D028010881B881BFE1AFD1B071507151D0CD70AD70AA604150415041E042E045E035E035E031B051B057413E118C419C419D5192D142D142F0EBE0ABE0AD903A203B403B403C403A703A70334058F078F07BD1B471B771A771A631B7B1F571857181C14D705D7056A04A203A203520484038403AA0499050C0BAD1BDB1BC51AE51AE51A7B1FC61AC61A0E15D80ED80EEB07ED05390739073104280328033C04FD07AB17AB17991A151A151A321A3D1BD116D116D116CB0ECB0ECE06ED04ED04ED047605760510062E09BC12BC1280150415AD17AD17281BD518D518D116951495140113D90A350B350B1106760576056F0D1314F113F1132412541754170A190A190A190A19381647144714860ADC0ADC0ADC0A1C0A1C0A1F0F7D14F613F6134614CE18CE18CE18F616F116F1163A177F117F11841184118411F109C1090A0C0A0CA813E514FE14FE14FA14F616F616F616031703174714C40F841184118411E70AE70A660A0A0C0A0C', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (153, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4C618D81B951B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C116C116071E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DE20DE20E51F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D20DFE1CFE1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41F081712171248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48F05E605320848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F483FFE5FF930148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D4FE70FF72FF48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B502B502990148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CD03CD03940248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F903B004B00448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43A09C208C20848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4060ABA13440F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40C0BBA13BA1348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FD071F1ABA1348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4510E510E5A0D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BA0DBA0D980E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BA0D8410841048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40A0C9415941548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F494159415941548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (154, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F420125612561284130A140A14A4147014A613A613A1148E168E16C3166214621428161E141E1462134D1232138413841317127B137B13641400150015641586158615E616B614E314E3141012D613D613BA149415131413146D12041404145617C917721772171616D415D4155B130B130B1393125F165F16E9155615F912F912EF122813E016E016B8175A175A17AE1603160316E013F1125D125D129718131C3E163E16AC13A213A213F312C014C01437171017101735153D15DA13DA13DE13C612C612F3159D19E216E2164C14E713E7131116D416C917C9172F17C815C81534154813481381121A121A12FB15E616FC17FC1772167A16201420145B16B116B1168A175E175E17231721159D139D13DB11101610167E18821670177017FF16A318A3188516B718B718D818161916181618C617801380137C11A710A710C912B513A014A014BB15CA181C181C185A1A2D1A2D1AC517FB14FB14A9139F0F990E990E010F1D0F1D0F2C0F0F10F80FF80FC410D113D113C11239123912F80FB812AB11AB11380ED90ED90EE60E670F670F7D0E750D110E110E490F0811651065109710800E800E420E9D0B9D0B160C730D730D0D0F6C106B10530B6F0ECD0E590F590F740E990E990E7E0DAB0EAB0E500E690E4C104C109210530F530FFD106210BB0ABB0AA40E3C103C109410460D460D5C0B190CBE0FBE0F8B0F7A0F7A0F1D1061106110FE0F5A107D097D09D10C5B0D7E0E7E0E510D6E0F6E0FCE0DD70ED70EBB0FDC0FDC0F3110A710B20EB20E120F210D290D290DB10BE30CE30C2F0D720F720F4810F70D6F0E6F0EA90F171017105210771077101A11510E230E230E890B2F0F2F0FD30FAB0EFF0CFF0C750D131013103110A710A710ED0F3C10FA10FA10180F320ECA0CCA0CD60D3C103C10FC0C51095109B90B6E10170F170FA40F0E110E11A2112D122D12AB0DDF0C7D0D7D0DC20D430C430C760B550A3C0A3C0AB91001100110580F70117011ED1165131317740C3D0C410D530D530DB10DE30AE30A230C970D970DEE116E116E118C0F4110B810B81075138B16960E960E420EAE0EAE0E3E0FF40CF40CAD0B890D8A108A10A610A610A610F510A614A614B8144817', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (155, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD25404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4471194189418591EFA15FA15CA14E714E714041AA91C5B1A5B1A421732193219251BFB17FB1754138F1ABB20BB20591EEF19EF19F5197819461A461A871A921B921B9A185F1B5F1B671A301A001300131C1DBF1C721B721BBD1B14161416BF149F169F163D17201795159515B317B117B117C31783118311131CFD1CF51BF51BAF165C135C13A31539187F1A7F1AB81A1A181A187F1B721B721B9618C917FF0F9D1970185818671367136E14831683163918BF1CBF1CD61CF023F023F81CC819BA18BA187518FB16F216F2164D16941594159814BC18BC18DB1AF217C01BC01B461B831A831ACF0DF212F212A010FF0B31173117E615811881188118D619621362139814751575152D15280F280FE40E350EFC0FFC0F460F7B161C1A1C1A7719FC16FC163915881388132D14781578150713BF12431343135413FB10FB102610CC141C1C1C1C091CEF1DEF1DA717AE149A149A14FD13ED13ED13CF157E137E13D91398135C135C13F610F1141F181F18EF1D2D182D180D15011501153415F914F914F515F115061506155016C112C1126B10C812D117D1170E18FE15FE15E7169C145C145C145416FC1AFC1A741A0A170A178C15921392136210D210E814E8146B15D015AD16AD167E17BB17BB176F162B1C2B1CB420951BA11CA11CBD1FDB1D5A115A115C135F155F156715C914C9149B161A171A177C17CA187F197F19CC1EDC20DC209F1ADB1D161116111816491549153F15B41417171717971CB21BB21BCE174B194B198418951595154A133517A011A0111816D1150015001568147A147A14B118921E921E73182D24F017F017E71739173917C9132E11A215A21509152A1B2A1B5613F5171A161A16F21ACD17CD17A91877197719F0172711271184102D12BC16BC16DC15791BBB18BB18871A531B531BF21ADC18DC185020341BF413F413420E1F0E1F0E13171317F115F115FE20D41ED41E4C19531B531B66186C1C502050205020791979197A166F0E6F0E8B160417501C501CFC20FE1B3B193B19E618EF16EF16DA19A61CA61C611D5D1B5D1B1511DC0F061248177B165C1BC01BC01B6D1ACA1ACA1AEF1BE616E6169418AE1A4B194B19A3187F0E7F0E520E9314', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (156, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E325404F3283424EF8464000000000000000000000000000000000E6100000140014004548F44B0C4B0CD00ED00EC50FC50FFD0E96179617501E8E0C8E0CA204190B620B620B9D0CAA09AA09030AD40BA00AA00A2C0A2C0A2C0A870BAD0BAD0B9C0D1A07A204A2042F076C086C08190FC30CC30CB50A0E0DF40AF40A2C0A2C0A2C0A2C0AC80B5B0E5B0E1607DA03DA0369056905930C930C8908E608E608FF0F610C610C610C610C750C750C250D6D0A6D0A1606090662066206190A070B070B97090B090B09FB16FB169C0D9C0D610C610C7E0C7E0C1B09A30AA30AA609BD0ABD0A090C92099209FD09240A230BFF0B360E360EE205E205DE05BB0ABB0A990671087108BB08040BD209D209F10BF909F9093E0AFA0E800F800FCD06A504A504F70412021202C5018A026C0C6C0C600A96099609240F570F570F1511A318AE0FAE0FBB0FB512B708B7080C057E027E02CB01D401D4019202F60AF60AAA0C040FBB10BB10A318AF1308100810B5123E0A3E0A7C0C5F0C5F0C4203E502330233026307B20DB20D480F061106111711E913E713E713FD11C90BE30CE30C5F0CD205D205320426072607D8051D0C1D0CE40CE90FC90CC90CD7130A13B90FB90F430AE407E407F4065B075B079F032607B207B207620CDA0DDA0D3E11FD18FD18DB1D9813100A100AB607DD07DD0754014001A006A006F9065A075A07080FB90FB90F16173915CC0CDB1D8E131F08EE05EE058C04A304A304EB03D106D1067A0B870B870B1A113310530B530B3E0F7E09981398136309C313C313C313B20DB20DA8111B18601B601B2A113511351136119E0B9E0BB00AF4031D141D1468148C0D8C0D140F140FF212F2124D15601B601B601BBE18BE1801107F0F020602069E04850B360B360B980CB90AB90A920B700C700C070E731B731B731BAA1A3D0E3D0EFD051604160416048E0BB208B208F108511051107F10E910ED0FED0F4B1A4B1A4B1AF5119A0B9A0BA008CC0374047404350BA50B0C0C0C0C320C1A0E1A0EC914491549157222722272223E0F290B04060406A904CC03CC0306122C0D600C600C050C341734173A17AD160B130B13D20EA20EA20E590C8F058F058C043C053C0593144A165F175F177D18A817A817A8178E118D0D8D0DAF094B0A4B0AF10B1105220522054A06B710', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (157, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4030A030AF20EFA0EFA0E3910EA11BC11BC1169135B145B145B14BD13BD13E6144F1843164316C312530BED0BED0BEC11BF12BF120811541454146B14821282124E131C14A916A916E817B016B0161F17920B8B0C8B0C2F15891889188918EC1982148214C111831183118411C211C2116E15C415C41578179F08E70B2F152F15DB1694189418AC136D116D115310F80FF80FCC0E891097109710F610DC12DC12230BEC0D7D197D198B1D8B1D8B1D36178710080F080F170F8A0E8A0EC80E570F570F860FD70FD70FFA0E7D130E100E107C140E1F861486144A11560F560F270F9D0E9D0E7F0E08103C103C1056101310FA0EFA0ECB17931493149E12B611B6114E105E0F5E0F5A0EFD0D170E170E700F2010201040101210941994199419E514E51464118E12FB10FB10600D120D120D950E860F860FAB0F310F310F0A0F900F9311931125169C1FBF14BF147813DB12DB12D4108C108C10CC0F3A108C0F8C0F840DFC0DFC0D750F7C0FAF1AAF1A9C1F9C1F9C1FC418BB1722152215E9108C108C101111D20FD20F600D090E090E980F7D0F781078103B141B1D1B1D1B1DD11695159515D014DD13DD133E1020103F0F3F0FA50FA30FA30FCC0C9910AB17AB179015F216F2160713271527151C140114C110C110A0101E0F1E0F030FDA0EDA0E7E092B0A6A126A1278123612DB11DB113712251325133814481148114711840E840E160EDA0E6E0FF403DC034A0B821A821A061FA810A810041037103710FB12E30F151015100910CA0CCA0C2A0D790D56075607FB0961206120821AA40F5C105C1011102C102C10520FC90FC90FF60F6E0F6E0FD30E590E1B071B07B30B6120B510B510A6109C0F9C0FE30F281028108A10B90F170F170FF60E8A108A100810B1056E0B6E0B5B13B412B412B4124A0F4A0F470FB81090109010750FD90ED90EE50EE70FE70F420F1409650A650AC014001835153515E50E330F330F2210A410A410A410DF0EDF0E360EDD0EEC0EEC0EDE0B2F102211221101126A0F6A0F500EC40EC40E5210A911021102117011C50EC50E5C0EDE0DDE0DB710F0124A134A137412F60FF60FB60E3D0F4E104E10B011D111D1110611F80FF80F1510170ED80D', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (158, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F47C127C12C211C814C8149C14731273121D12FC10661066101C10A80EA80ED00EDF0FDF0FC1102E11E416E4165B13DE135F135F1381133D123D122B12F710F710EE0FE20FE20F7A0F541046114611D5105E158E128E1287121E121E1276126511651101134F11BD11BD115D11FC10FC109910E00FE00F521046165B115B110C119B0F39123912B9121E111E11DE10DD11DD112A12CF10CF10FD11BD0F4B0F4B0FC9104911340F340F3310F00FF00F8410310F310F0C0FDE1211131113D011BD10BD10860F170E170E13102F1059125912A60F950E950E930C530ED10FD10FD512741374135710F00EF00E670E530DB50A121085106411E60FE60FCD0D1E0E1E0ECC0DFA0EFA0E2F119B0F9B0F750D7D0D3E103E1030101A10C70FC70F8D105A0F5A0F9D0E3A0F3A0F8F0F9D0F3B0F3B0F4B0FFE0DFE0D13102D102D104F109510730F730FFC0FAA0BAA0BFA0C4B0FDE11DE116A1034103410540FE610E610920F380F570F570FCC0F2410B70EB70E420A200A200AF10E861086109F11990E7C0F7C0FCB0F700F700FAA0D460F460FC40F500F9A0C9A0C270A950B950B640FE70F3B0E3B0EB10EC70EC70EE40DB50DB50D6A0CC40A460B460BCE0E530ECA0ACA0AC90B770D770DB60F3B0F3B0F5C0FCB0DCB0D2F0D0110270D270DC50CFB0FFB0F6E0F380E4F0C4F0CD30B8C0C8C0CAC0FBA0FB111B111BD10C510C5104911491149117A12E911E911790D280FD70DD70D200E760B750E750E53109B119B1104143D123D1292118911B611B61195108212B10CB10C5D0C6B0E6B0E390E4A0C4A0C60100512F214F21431125C115C11F510F510F510C00FE110150F150F8C0D1A0F1A0F9C0E8A0DDA0FDA0F9A114B144B14F1127A117A11D0122C11171117119F13AF0F840E840EB90E750E750EA50E620E620E9210D213D2137414F11381118111FF1123102310ED11120EAA0EAA0E300CCC0CCC0CA60CF30B100D100D5D1172137213A613D812D812B910D70FD70FF00FBB0D740C740CC50CE30A160C160C0C0C960C960C50107B137B130D14DB12171117117011E810E810D80D2E0D260B260B7E0A0E0A0E0AFA0B310DC310C310CF1027102710180FEF0EEF0EAA0F7A0F7A0F', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (159, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F0004072326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F41D121D129B115A115A11C41144114411C30F4E104E10FC0D3F0EAE0CAE0CEF0F400F400FEC0EBE0CE50FE50FE31027112711FA0F2010C30FC30F940E970B970B3B0BE50BE50B340E840E840E990E700EA010A010EB0FD60FBC0FBC0F780FFC0EFC0E1F0C95099509900AAE0C8C0E8C0EA40EF80DF80DF30EF40D1B0E1B0EF50FB80FB80F2E0F400D400DD10A590AA30AA30AB80AC60CC60CA90D2A0E2A0EC610D00BF20DF20D980F16108A0F8A0F7A0D8D0D8D0DF10D260D260D2A0C640DA40DA40D830EC10FC10FB50AD20BE50CE50C360E5B0E5B0E5A0D101010103E0F0E0E7E0D7E0D1D0E200D200DF40D470F470F1A10C50DF80CF80C490D330F3A103A100510A411A4119F0F7C0D7C0D7611F010F0101F0E980EA9189510DB0F1510351135116B0F221022102E10A711A7119F0F880FBC12BC12AA0E691069109510EE1A001100117010ED10ED100E11F210F210E3103D10950F950FD40F051605168D199B139B133312FB1AF90FF90FCB106110EF10EF1058101510151043100A0F0A0FF00EDA0F000F000F790FB017B0170013300F300F300F920C0A0F0A0F8710C110C1108211E3128D0E8D0E6E12B10FB10F5611E811E8116611350F7D0D7D0D8A0B8C0CBF14BF14EF12AC14AC14CA11AE0FAE0F6C0FFA15FA1546124812990F990F9E122C12E212E2127C14AD14AD1415191F191F1914187C11C70FC70F541254125412F111261226128212EC11C511C511DB15B714B7143917F716601360137F116A0D6A0D861268126812611030107213E11042128D10EF14EF14A114741674163516FC15FC159315B41517111711C1121510151030104611E711E711D210EF10EF109A130A160A1695169516951695162515C811C811C20E290F290FF90E7F0C1A101A10E80F62100913091358143614361436143D133D13E1120A130A13111329134A144A145A132E100B100B10E80F6C136C13F013F013F0132E142713FE13FE133D1363146314C1148B1A8B1A1620FE106B116B11C6117D137D13F20D270CDC0CDC0CF00DF20FF20F8310881688167E1657179A1C9A1CEF0F21113C113C11CC11880E880E3C0E980C980C9F0C4C11701170116F121912191257173A103A10', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (160, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C3826404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4BE0CEB0E3517B416B4162E17C116C116621512151215CC16691569153D15E8155F155F15D214A81101150115DF18221822180316291329133B13FE14A914A914131513151315BC169B159B15F4140D1364156415AA1A6C1A6C1AC715FA10A215A215CB15DE14DE146515FA15FA15BC169217341734177B173111241824186F17521552154615FD15FD15101688168816CA176515671567159C15EB14EB140818B2117B117B113314071707177D1AB71776177617D317501A501A501ABE18BE18D015EE168D178D17DE0F5812551255121315381538157B18281A281A4C19AA1CAA1CA71C451D87178717D2178D178D17A918AD1576137613B911B010B01005176519501C501C2E1CE819E8195415A817A817CF179F199F19EE1AB3156C156C153D148013AF11AF1146154E154E157F19D418D4183815D814E817E8177219F5175B105B100A173A153A15E213C712C712E811031203127F116614A611A611CB1409160916C8156814FC10FC10DA17EA1BEA1BE718E016D713D7133C137E117E11F210E810E8108E0EC60F121112118B0E22113F123F121C15711571153715F913F9137013281328133411D210DD0FDD0F400FF110F1101F0E0C0E2E122E12FC124B154B15BD183A14D516D516691204110411EB10661066108E1199129912770DFA0DF510F5107312AE12B217B217F118B412B4121915B913B9132D15E21118121812A7117C107C1072133314551455149D12B01AB01A66198217821736156B15AD14AD14DB12461046106111941194114611131347134713E711811C2118211848143C0A3C0A1A127F137F137A1767153E103E101A12D51392119211D518EF19EF191F1B251E251E7618580C580C5009851088108810581EE914E914900F4615B716B716C4120E150E15FB16D61F941C941CCE19FB11FB11F21347174717B11862166216BE20922216201620A9152318501850180120201B201BB31D651D651D1A1AD419FD19FD19D31D851C851C0C22F218E50FE50FEE137E117E11231EDA1CBF1FBF1FCF1DCD1BCD1B160E3D193D1981152B162B165816150E090809083303B1037E117E11441E0A210A213320571D571D2A1EB00F030E030E1E1629162916', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (161, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4A81182148C148C14B215FA14C10FC10FBC0EA30FA30F4910860C860C860C2D0C2D0C550BA00893070D134513EC13501950197313C714C714DA0B690B690BDE0B120C810C810C810B6E0C6E0C360CEC0C5115511551153816381635160D140D145B0C260B690A690AA30BA70BA70BEB135A165A16E312F20E6E1A6E1A6E1A07188B158B157F17410F410F460A420842083F09060A060A06140815A317A317C00FFF1A6E1A6E1A2E1CC418C418491CFB14FB14800D2C0A9F089F081F08D70CD70C9510781078106B0E021702170217261AD61B491C491C5A1AE41EE41E5A1A640E640E7514951895185912EA11721172115517D815031D031DAD1CC61CC61C7B1F7A1F7A1F7F1C491DA419A4194E198D198D19B721B721B721F5171517DE19DE19F718871987191A1C391FC51BC51B671BA31CA31C901ADB17DB17B721B721B7216814CE122B189B189B182518061906190C19FA18FA187E1A511851187419DB17641364130F09DA08830E830E290DD112D112D1128D0D8D0D3313001214111411590E16101610CE0C201220126B044808490A490A5008D507E40BE40B7C0A990E990EBE0F5A135A13E409700B700BB20B64084D054D05BF03410A22092209B50C1C0A1C0A7B0A070A070A0C0A770AB10EB10EE50D540C540C310CF605F6056A05670AD60BD60B1E0D910C910C8D0C970B740A740A6F0AA30AA30AC60A590C590C6A07AF0499059905F50A910B380B380BD40E6A106A10F10F2C0F2C0F4D0BCC0B7C0A7C0AF10AE908E908F508D809D809D513C01453145314DC1B6E156E151E154D105A0A5A0A7D0A300A300A75099B089B08A9092C1D2C1D4615A3176A25DA1CDA1CB6130B0C0B0C8D0C591159116F0B6F0B6F0B7607C006C706C7062C1D2C1D291F291F1B22E124E124E722E212E212EB1191139411941106110F160F16120F810E810E9B0F6A11061B061BDE23482448241E247C2298229822B021871E871E3219CC19CC199719660E610D610D281151166B166B167B230C230C232C26002500255D255D253D253D253D258B1E8B1ECC1B870C870CC41BFD11720D720DB51691159115A22095226C226C223D253D253D253D25B71CB71CA813DD13971E971E', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (162, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C0066326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4930775076D0C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EC0C7208D10748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F00EF00EA30C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4980E980E980E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49D0DF90AF90A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48E0EB30FB30F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B721B721B72148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B721B721DC1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DA083718DC1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47A077A077D0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B003B003F30348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F497039703140448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41E05DF06DF0648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47B09A10BA10B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42C1DEF1F0E2348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42C1DBA1E711E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46A114B1BAD2248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A417A417940A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FB15FB15711348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45F121B141B1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_02.tif', '[2005-02-02,2005-02-18)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (163, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540989537D1F812474000000000000000000000000000000000E6100000140014004548F4C000C000840084007500C800C800C100FEFF3900390024004CFF4CFF55FF88FF88FF94FF3FFF3FFFBA00AC00DF00D800D800B200B500B500380039003900D5FF88FF88FF01FF9EFF94FF94FF7FFF91009D009D00BD00CA00CA00B200F7FFF7FFDEFF6BFFEBFFEBFF83008900890009FF2CFF2CFF23FF15007D007D0093007D009C009C00A100A500A500D6FFFDFFFDFF2000890089002800DFFF0D000D002F007D004C004C000E0031003100A100DB00DB00BA002100220022004D00210021000400BFFFBFFFF3FF90004B004B00E2FF720072004500FA00500050000F002500250051003A003A00A6FF8DFE8DFE5E008E000000DEFFDEFFB000B000B000B201DE02DE0263004E004E005D00B801B500B5006500E701E7010E018000540054004A007D007D00C700FF02670267027001290329033501F300F30004018601860140001C01DE00610061007E0021012101D8012F012F01C0014A034A033E012501250125010A01BC00BC00BC00990061006100F600470147012D01570157011F01740198019801460054005400F0003101B001B001B001330333032E01F20131023102D2030F030F037D027301730164FF75FF75FFDAFF3D015E009D009D00F90363036303A502750275020803850185010008C301940094007900AB00AB007B0156011A011A0163036A036A039003C106D406D4062B0A580B580BD105B300B3000401FD00FD009001EC0107020702E303DE030E070E07FE06DD0BDD0BCF0B500B500BF907D40379027902FB01110211021F02A901C201C201E0030B040B04F7046E0C6E0C1D0E1F0D960B960BDB049D069D06CF041D041D04370175014501450148026105DD02DD0217068B0B8B0BE011990D990D1C08E308E3081D05B4046C05D400D400D40047024702C90655075507A004F208F2086F0D0D09BF0ABF0AFE058304830447045D038A028A0247016E026E02F5044A05620662062508EC11EC114C0C8A0A8A0A1E0592059205F1058204AF03AF03E1025402D901D90192067F057F05F00E5C0F5C0F5D1151105110C309BF04F105F1057005A202E603E603A1026C036C032D08DE07DE074508B20BA70CA70CF610B30BB30B8C06B205B2055006', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (164, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540989537D1F812474000000000000000000000000000000000E6100000140014004548F4820077009200920024016D026D027302780378032A029C016A026A024103720C720C01130414041491004B00A200A2007100AC01AC011E031F021E021E02530236023602310E9010901089139313931315001B00000082008200E7029303930387036A036A03280433043304310ED213C611C611C20DB9060E000E00F9FF29012901F90162046204F1031B04990599054905681068105D125C125C12F6104805F3FFF3FF45003502CA02CA02190380038003DB0440064006F70CEE0EEE0E91110B13151315136C115E00420142014F023002300229011A071A07CA07D10BD10BE7102213091309130913F011F0112D13E701DE02DE02DF025D025D021302FE03CB04CB04180921122112B515B915B915B915C013C013BA1386016402DE02DE0212021202120222002402240237038007800799119F19E017E01754165E145E14BC001C02BA01BA01A40240024002AB045D04D802D80298056D0D6D0D361688178817621ABD1EBD1E310160009901540254024E034E034E03DA04470747074707C808C808300AC4126B136B136C1E0F1D70037003DD0110021002CB0119021902E5033005300509070F06B40EB40E0A120D120D12A519A519A703A7033D04D303D30343028C02C801C8012904FF05FF05FA08B40EB40E0812AA11AA113D152318F201A102A1029B03050405045803E002E0024D03970597056308EF1125182518F5184A144A14CD153F023E023E0267026B026B029C021003E502E502A1048A058A05BA0825182518991D49144914FF111D042B032B03D4016A02300230027503770377030405F104F1044E066C0E491949194817741074106C056C05F801F80161039A029A02E902F104F1046406E206FA07FA079B12381B381BF6156C106C105D037E04820482044804D502DF02DF0292034907490785077D067D068211571A571A7817CD137C106D056D054704FC03FC038E01470447049F046007600796086A0ADA12DA12B315081708170314A211580458046303410441043D03B6020D040D041A09970B970BF60A461546159114F714F7146216A6121206120639004D00230323033703610461044809AE0CAE0CC211D217D2176F156F1505150515CF14', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (165, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540989537D1F812474000000000000000000000000000000000E6100000140014004548F4890F05170517D216A61796199619D41867156715C5139B119B115110C715C71563160E1BDA20DA20960C040E8A158A15E314311231128312061406142F105A0D5A0D550F4118B31FB31F6925A122A122B906330C6E166E161313B714B714390FA90CC40BC40BD50B2B0F2B0F0419EE20EE20871F45244524480584081A1418171817231560106010DB0C9E0B9E0BF00A290E290E721D7E21D71FD71F6A20F01D6E116E110414711971197F170C150C15580D7E0B580A580ACD15751575156C1F1223122334264E221113111384169917E618E61871174409440943112B092B09CD15921F921FE720211E8C1F8C1F5B1D6514FC15FC15B410CE13CE133F0E37083708F905080A080A6B144B236D206D20F11FAB1BAB1BB91B9E15B914B914790BF008F0084E080B04A60CA60C620E411341135A1DA721A7212F2075227522A91D071C0417261B261BF718361536153C0F7B057B054E0CA910A910431DE41EA91FA91FD9214F214F210F1D4817B21AB21A341A3D153D151206B013180A180A9815DC1EDC1E2C1D5E205E20981E6E1E6E1EA51971180A140A149413BB187F117F117F1188088808FA195C1C5C1C0E1E1C1E871A871A5B195B190B150B15C41120102010B707670867082D08A805A8058E15D518F01DF01DC31CC218C2189717D2164A124A121F0FE70FE70F72073D076F0A6F0AA50EB81EB81E6E1C4D1B4D1B751ADF14DF147A13220C9C12AE0EAE0EE00D2A092A09B604860A860AD8193B1F3B1FDE23A319121312138412F910F9107D0CE60D0D0E0D0E4E07B502B5021104AA04D319D319A721281F281FB617AF16AF166B13C20FC20F3A10C00F20082008CA064E05D407D407C31BC31BC31B311E00180018FA18D2100D100D10F70BB109B1097C10750C030F030F960F400C400C651D36203620B6221B1C16181618DC0E3E093E09BC05D106D106A2110D0EA613A613541647197E177E177715ED15ED15A118CB19CB19DB0D3E093E094D0360042E049F139F13C9146F176F17F1155C175C176518CF16CF160F161D0C100810080108AE03AE033C03B602C815C81527170E170E17E8160619A61AA61A981A351A351AC711A309A309E104470347032A03D202', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (166, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540989537D1F812474000000000000000000000000000000000E6100000140014004548F495206F226F229E1EBF1DBF1DDA1EAE17AE17061F521FF51EF51ED01A260E260E6C0ECF05CF052C04A122C721C721E3213223D022D022021950165016611BF01BF01B5919C90BC90B9E074B044E044E047523DD216C226C223E23AA22AA22B11CAB17AB17FB1471137113021183087E047E0458040B060B06F01D3026562356238423AA22AA223B22F219FF18FF184E146C126C126E08D503D50310024C034C034E221F22602084238423D51F1E1E1E1E5C19FE16FE168318A60BA60B7B09650215011501AF019204C41EC41EDC1DD11AD11ACD19CA16CA162C132A0E750C750CFB0A1D061D06A80441024102AA02C1016918691820176E13E711E711100E99089908CA08210C210CA605A702A7020A04C10374067406B4076918B216B2165D0FA80EA80E4406490749070C05B804B8040106E606280328037208750A750A5409D31832123212BF0F5F0D5F0DAA0503046303630337031C031C03E501180418047707EE08EE084A0D5614D813C50EC50EF50A2308230872032E032E030F03E601E601CB012601E504E504DA0B720B720B5B19F6129C0F9C0FE907CC04CC041A037A025B025B02C901C501C501690322052205460C950B950BD21663110C0F0C0F6C0EC403A602A60259029E019E017C0123012301520393055D0B5D0B2F07730BDD07DD073903670E670EDF025B025B024102B601B601BC011C01230323035806760C760C70078707AF05AF053C034A034A0309036E02700270027C01AA04AA044F03730373032C09390C390CCC073509720D5B0B5B0B540476037603ED027B067B062905680568056F056B061C071C07D308B111B1118015940D0D090D092D053604360470032D0133073307A605730673060B07D706D7061208441244122D164F05B705B7056505090564046404C8035709570995049F059F050F0AD70A94129412E113A113A1132E04AB05A905A9051A05CB04CB04BF043B0C3B0C210A2B06541054104E112B122B12C913F311F311B6026203620362031205420CBC0ABC0A6C0AF909F9094D061F141F146E166E166E16A216FD13701270037003FE080D0A0D0A8A0F220F220F5E10B210B2104010CB13941594152814BF14BF142C143913', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (167, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F4510351031D03C3018F018F01A5017A037A03C1053E093E09D20DFF0FFF0F17122314AE14AE1490101A015E025E020002F100F100CC01A201A2014706180BBE0DBE0DB70FAA11AA11AA11CB10CB10EF0FF004D701D701C700D700D7002502F503730573058007F307F307860F431243129313E41194109410CF06FE0452035203F302A003A003F104110811086E07B107B107080FD511591359139414721172119204D905CC05CC057D07E106E106D306080AA607A607DE0AB912B912CD11E212E212C6104F114F11C10192076A09680A680A890AC90AC90AC90A70127012E50D2F142F14C314A714B114B1143913BD13600660062808790E790E530D82118211FC135C133F113F11F8131C151C15EA1055125512B2121A13C308C308A109A909A909410D5E13A314A3140312D312D31209135F125F12660F7C116E116E11C611A50C100E100E530EEA0EEA0E95122D152D155C135C135C134E146E12BE0FBE0F030F6D146D14E3184A0DD211D211CF116813681328149A146B136B136B13FC13FC13B712B610B6101312001800185C199C0B9411AF12AF12531388138813AA13EC10EC106B131E141E1471112B111F191F19F11797189718730BC911B011B011F21062106210D31070113F113F11E71005120512E3124C1C4C1CBC1BC91AC91A8707A511CE11CE1133111310B910B910911001100110E40E841284129817321C5F1A5F1AB81A561A8F138F1325144F114F11EF0F880F880F2B10D310D310A50F3B137419741950198A198A19101BAC1A0F140F14931216121612DD10470EB00EB00E850FBB12BB12AA15F118F118DE18291929195F1A8819A313861486144614DF10DF10550F950F950F0D0F901290127E156A17DE18DE18F1183A193A1921189912DF13DF13A212A60FA60FBA0E1210AC10AC1088148E178E174318DA16DA166A16E816E81608149311A0110F100F10F80CCB0DCB0DB50FA50FA50F6F14B615B615961927174B144B14FE129B129B12701221116D0F6D0F030ED90DD90DE10E641064108B154716F718F718DC134D124D12F611E111E11139139A110E120E12AD124211960F960F5312F315F3153A179D189D18261414141414E111FD112A12', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (168, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640989537D1F812474000000000000000000000000000000000E6100000140014004548F46A126A1260116D0F6D0F5E0F710F710F410FD70DA509A509C70735093509A00FCE10CE107512B211DD11DD113011920DE70FE70F1F0FE20EE20EF40B3C083C08F808E60BE60B8C0DDA103D103D101B0F880F7F0F7F0F5A0EF00DF00D690FB30DB30D9B0BAB0A0E0E0E0E950EB50EB50EDF0D570B570BD70AD10F850F850F480EF70EF70E6E0FD90C9F0C9F0C170DCE10CE10E3109C109C10F10BEF0A320A320AD6140910B30EB30EA20F04110411BF0E810D810D710D190F190FBC0E8F0F650D650DE20B230B230BBD138D10E70DE70DEA0EBE0FBE0FE50D510A0A0D0A0D550BF709F7091F0A430F430F730FBB0BBB0B1A1366117B0FD80DD80DD90E750E750EC908CC05CC05DF056F076F074A09440C380F380F560D520B80118011540FD20ED20E9D0F170E170E7607BE06C009C0092709060A060AAC0BBF0EBF0E990DDF0DDB16DB162213401140114E0F980D790879080907FB09FB09EC0CD80CD80C350E8C0FB20EB20E7E0A5C198F148F144C12901090104A0F480B480B3109170D170D1C11DB10691069100010810A810A3F090718AE16AE161913CD0FCD0F090FE50C410D410DF00EF70EF70E1E122D112D115E0ED909D909CC0AF919EC18001600166D0FE30FE30F310C250C250C300CDA0CDA0C1D0D650D8D0B8D0BC6086B0C6B0C561A5B1A9D199D19D0111C111C11EC0C690A990C990C730978077807E204EE05EE0582080A0D0A0DAC1ADF196D182C122C12F810890B890B160B4A094A096309B806B80682067A072B0B2B0BBE0CC010751975191318C716C716B30EF10AF10A750976097609E106EA06600A600A7E0CF90AF90ACB0CC01045134513C813761176114C0EAD0BF708F708CC085A065A06DB06FF0AFF0AB10CA90CA90C020E5311D3108710871038102E0E2E0E3B0CE808E8081407050805084A0A660F480E480E880F0D0E0D0EEF103810BC10BC109C1047104710A20B0D0944084408C007F107F107660FDB10DB104412F10CF10CFE0DF010C3127F117F11BC11EA0EEA0E7D0DFC09FC0909096A076A076F0EDD0F55115511D50D8C088C082A1226128913891373139D119D112F13E40CE40C330A1907DC06DC06840AC00AC00AEC0A9C089C08', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (169, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640989537D1F812474000000000000000000000000000000000E6100000140014004548F4B211ED0FD810151015108E0E8A0E8A0E9F0C390B390B660C5B0E5B0E2111151088118811A6114017E20EE20E4F10C30FC30FE8108411841118103710381138115613CD14CD14B617A016A0168C141318250C250CD50D8B0FCD10CD106A116A116A119E125113511326146F186F1889190117AB15AB15B114CE0B890D890DCB0F4E104E106D0FAA10AA10B910AE127D167D16DB17171417148D1320142014D7148F097E0B7E0B920D2C0E2C0E4A0F641027102710BD104813481334125A125A12FB1101150115BC16030B8C0D160C160C9A0F5D105D10FD0F551055108A0F350E350E121026114B114B111317011B011B520B1511DA0FDA0FD90FBD10BD10E50FB50EA70EA70E5F0E80118011F3108E0F8E0F5F1266186618DF0D0C0B2B0F0C100C1068108E108E104E0FC20EC20EC20D960F960F8A157B167912791213143D17F20CF20CF30EA30FA30F9710770E770EB30DEC0E800F800FE60F7F187F18541C8B1C8B1CEA183F147B0D7B0DFF0C4D0D4D0DD50E4A116D0D6D0D600EA00FA00FB111901690167117FC16DD1ADD1AD419900F4E0C4E0CA70B0D0E0D0EF20F9C0F9C0F450F17101710FE12AA1B6A1B6A1BF21A74177417AA17760F390E390E8F0BB60CB60C7D0E2E10C90FC90F6C0EB80FB80F791A891A891A4C1B211A211A491873122211360E360EC60D5A0F5A0FFD0F4B0F4B0FA012E10FE10FFD178F19381B381B7E1967176717C0101712E510E510280E4D104D101D118710A511A511A0115B125B12B315AD16AD16E81967176717C0102911E510FD11FD119B11EC11EC11B41178127812251512171217A0170915F712F71244169D14F10FF10FF70FA40FA40F61110E110E115F110E120E1235154316091609167815751275121B16221651105110101076107610BA10581109120912D9144A134A13E415D414D414E012B315B315C019521B9A0E401040104211AC13AC13D21390139013A914AC14AC142A11A510350E350E43114D184D18891EA30BC60EC60E9910AE13AE134C14711747164716B31591129112E70EA90EA90E580DF312F312E918B3092C0D9B109B10B813FB15FB15B3191F1C1F1C391719131913DA117610F110F110B311CF15CF15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (170, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F440178017971597155D1353125312120F9F0ED90DD90D4009BD08BD085C0BB90DB90D8E0D770B770B1318321853187E157E15681502150215F215C314C314D411210721077A071C0C5B0E5B0E1A0B490AE614E61482147812781295162E162E16C312481246144614BB0C42074207AE09FE0AFE0AF30C2B0CB116B1160D17A914DE13DE1359140F150F15291585158515420FF206F2064707E809DA0DDA0DE70C9316D017D017B819C718C71887161E171E176618BA160A140A149C09320632067307630B630BBD0D4C1A6D196D1956195A195A19E9153A15261926195418841784177311430E430EB907BE0CBE0CBA1047190A1707170717C2182818281878184F184F186C182B132B135E14C817141914199C15271827183D170D16E115E11560192C182C1897156F1625182518C01A9A1D9A1DF31D081D081D6B1BDE1ADE1A3F14B0148A16C519C519FE14BC12BC121815C51FC51FC51FA91FA91FB31C501CF51BF51B941D291CD712D71221166E1B6E1BB11488108810E113E81FD51FD51FA91FA91FA91F861CA119A1193D1CE61D681568152315131A131A0A1CFD1487148714FF19D51FD51FBB1D2A1C2A1C4B1A4C1C871A871A3F1EC617FC14FC147A16C917C9179D17721572151019B31BB31BB31B0119B818B818A419FD19FD19E3134B165D135D137214BC17BC179E173A172A162A166A1A961B961B8F1B5F185F187A190A160A161512C4151D13461546150D17691469146A1605170517E2191F1B1F1B1B1BA818121612162216311231129D145814691469147318E418E418E218BA162F1B2F1B2F1BFF16FF162314811481149516921492142216B01520164E134E13D4174C194C1983160816081669150313031323149516A119A119C31BE71AF518F518341658155815C314701570155B17C211DB10DB10BA11ED11ED11E715FF1DFF1DCF1DA51BC71CC71CCC18AA12AA12E6113E1043104310EB0DA710A710D40ED40ED40EFC12F91BF91BCF1DA91BC5180E170E179614AA13AA136616D916D916EC1174117411E50F051073107310261724182418DF0F6F16621562153F150D130D13E016E0167A137A139B11750F750F5910150F150F420D5E125E123F0D', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (171, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640989537D1F812474000000000000000000000000000000000E6100000140014004548F45C0BD00DBC0C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4490A530DFF0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42B0C990BDE0F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EA0CEA0C2C0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4ED0CED0C011048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F462113F113F1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F440134A154A1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4351B441A2B1C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4291CAF1B221D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E61D071E1E1F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43F1E3F1EF41D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F421192119F41D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C416931A931A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BC11E915E91548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46E11F110751B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E71A961AE71A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A51BEF18B11548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B013B013FD1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45B0E5B0E2B1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4320E4310431048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (172, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F4B0022A042A04EB01DE03FA04FA04B708620762075A08F90EF90EC810330F330FCC027B05A607A607D102DA02B703B7036F030F040F042A04C803C803FB09AD133A123A12AE0F9B0E9B0E150E440B440BFF002701FB01FB018E03E103E103E10339043D0A3D0A640D1E101E10DD178B128B129E11130C250AFF00FF00D801C403C403650494049404BC04C40AC40A050A770E770E7E1FBF128D128D126E0CD509AF02AF0211022F022F029B03B504B504E80E1B0C130D130DBC0D192019202114621262124F0BDB0BA305A305F807730563056305F405E80EE80E3B0D2A102A105410B517B5172F10B00FF910F91036120607480648068608BA09BA09B708CB0FCB0F1A0FC41002140214021485118511890F19121912C31573057B087B08E50730083008960E780987148714CE15211721171B1400120012D4100E125A175A17DE0778080D0D0D0D4B09A10AA10A1E11DE14DE14E014BF15BF158A13A011EB12EB122B1493159315C105BA07820D820DB00EEB11EB1165115110E014E0149F15571557157813781378138A1281168116C3059606080B2B0B2B0B3112BA12BA12EB1112111211B614B715B7156716EC15D314D314141714187B027B02D308920592051C0A4A104A101D159A1262116211CB15821982198C1AAF13AF13AA13A116A702A7022D05F70BF70B730DAA14CF12CF12B113D613D613DB12A116A11681186013601333120512140334013401620503030303190BC40EC40E36122216221622168E15DB17DB17FD1285118511BA0F5B02B504B5049405460546053609B70D300F300F541107140714A218061706173F143A113A11FA0E8208B90AE80AE80A63082C092C09F90AA60EA60E6811DD14DD14BA17BE16DB14DB149A0D380F380FE00B660DFA0CFA0C5A0D230C230C760D000F000FF40F6A1046174617551702150215450F57105710EB09210D840E840E420EE10F6D106D10890F230F230FEA0F0C120C126F152A142A140212A1113012D20BD20B4A0F521052101F1218121812AD10EB10EB106011F513B114B1144F131F0F1F0FD81121126F126F12CB1235133513A8145F12EC10EC10081145154515F11416131613500FBC0DBC0DF9101612', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (173, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD254023E87981A30D474000000000000000000000000000000000E6100000140014004548F455071403140319031103110335046D046D04C509F00BFD10FD103A12C815C815AE150B110B114B15B309B007B0072C01E303EA05EA0526060A090A09670CE40FE40F6D107611761119114F13D216D216250ADD08AB07AB077A07CF07CF070809C50AC50AD30E801242114211080F3B103B1004163F163F16D5097A064C064C0638069008CE09CE09190BC20DC20D5D12280E280EB80C380C380C9112AE154818DB0B440ABD0AD20ED20EAD0BF60BF60B800DE80EE80EC20CCF0CCF0C3C0EFC12CA13CA13FC143B1818171817D4166A116A110F14A416A416E115BC12B20DB20DED0D33113311E4121F141F14A014BA14BB16BB16A21CA21CEE16EE164518BF15BF1569134E154E159C156A156A1507137812FB15FB15A214A61BA21CA21CA21CA21CA21C571A571A571A0D1522162D172D178917E014E0140E174F174F170D17F41BAF1AAF1AAF1AAF1AAF1A6C1A571A571A571AFB1CFB1CFB1C9917B117B11771182918451645162518D318AF1AAF1AAF1AAF1AAF1A9A169A169A16FB1CFB1CFB1CFB1CFB1CF21EF21E1B18311631161418BE18BE18BE18D61877187718FA16AF169A169A16851985198519501DF21EF21EF21EF21EF21EA116851654185418541854181E161E16531662166216851985198519CE19C31DB81DB81DF21ED819A711A711A711471247124712831683160D1651163D1C3D1C3D1CCE19CE19B81DB81DB81DB81D1319CA0DCA0D1C0DFD0DFD0D500F591A591A591A591A3D1C3D1C3D1C3D1C3D1CBC1DBC1DB81DB81DB418B60DD30BD30B950CB30EB30E180F0A140A14F716101910199D186C18BC1DBC1DBC1DBC1DBC1DB5183D0F2C102C10B50F710F710F0411E2110315031537182F192F198C18971897189718DE16DE165C180D102C10101010104D129F129F12F4144B174B174B17921892187F173317E016E016D014C515C5153012DE10DE10DE102D13B314B314E3141A151A151A155B1639173917BF161B171B178E17C414C4142112E412AF12AF125F136013891389137D14A114A1146A18B116B116F5189A199A19EB18CF165F14E512E51297127C127C120D13201320131E13D812D812B513DB13CE17CE17EB19721A721A28131512', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (174, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E3254023E87981A30D474000000000000000000000000000000000E6100000140014004548F41D181D181B18F418291A291A0E18A013A0133812850E850EA1089A07D408D4089B03B403B4037B02F11628182818F417161816184517890F890FD70DF80D6E036E03A705FB01FB018E0233013301A903FD17D818D81855183617041804180F13AC0DAC0DFE0D3A063A06B402F300F3004D0692021506150648187F19DB19DB197F17E917E9177C13A306A306D206CF029A039A03EB02B804B80407075F065F063B1861188019801960177317EB0FEB0F450252025202CC014A024A020104B907B907B00842086E09BA149E165F17FF16FF169A1771097109CA022F012F01C3091A041D041D04A40B580D580D25093B09BF17BF17B717121812189B14731273123C0BDA0C930B930BDB04FE09FE09570D5A0F5A0FF70DDC0D501750177319401987148714FA11120B120B060909080908B304720C720C460E740F721072107F10C515971697163C160F140F14A712FE0BFE0B630C3B0A360736070F0AA709A7093B09AE0FAE0F2B0FF815BE15BE157D1624174C144C14C60D7F0D7F0DD30AE307E3079007EF0AEF0A520BA30F720F720FB0194018D616D6164316851585153710570F570FB80FC911C911AF11A70CAB08AB086811D50ED50ED819FC19821A821AB31A6B166B1678138B0AC70CC70CB2129E139E135E1374137413B7100B0E0B0E1319371A331A1D1A1D1AAF14CA13CA13D011B212B2122C142414241418155A11A510A510EC0D0B0F7C187C18CF193B1A3B1AFA136A116A11BD1112149B159B153A1504150415B7124B104B102911C911DD17DD174017531753175C146A1415151515BF17B018B018EE15EB14EB14AD1193109C119C1122130D1821162116D415F714F71471152C172C170819961A961ACE166414021302130113E012E0120E1518155A145A14BB13621862187D18E61814191419961AE116E116DA130713071334137A127A12B214F9147513AF12AF12F11582158215D115511951194F14571357132F130312F613F613DB13951295125F14CF119B119B11CD12E811E811A9116011210F210FF80F27112711DD116B136B13D1134A114A111512A6113E123E123F110112AA11AA11AA11E212E2125711471147110C11E71315131513F4110D13', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (175, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF8254023E87981A30D474000000000000000000000000000000000E6100000140014004548F46B036B038D0990119011A9110E12FF11FF11BC1098129812AF13A314A3141B12701470140E13CC1080068006520EE511B011B0110D117E117E11F110CF14CF1426158D136C136C13EB13D611D6114312180E9D109D100B133D133D13511254125412EF135914401440149B1355145514111309110911E1111D0D94109410B112BB12661466145E134B124B123D1197119711471289128912D111720E0D100D106E09B90E28102810671227122712301336133613A012031384128412E711EB0DEB0D0B0F421042103B09540F73107310BF107C1394139413501363146314251399119911B10FAF0FAF0FEB0F2D144F17DC0D7E099A10D310D310D21351135113DF11041104110B12C611EC10EC10E810C810C810AF123E17140E140E2C0E51105110D413A113A11308111D11041104117B11F110F1102B11851285123A126E14FD0EFD0E3C0F660FCF11CF117211CD10CD109910F90FF90F630F5E125E127313A41317131713A813B00F3E103E10DE0E741174119B11A310A310DE10F2108F118F114E121014101481147D127D12F812A10E6E106E10B80F651099119911FB105A115A114911DC11DC113812F113F1132F154612CF12CF12AB0D281057115711641050115011CD10B310B31050138813A813A8135F14E413E413AA129E129E120B0FD90F14101410B10F0E100E100310EF0F2314231404148513851319117B117B113611E311F211C9110312071632103210231003100310861014131413AB121813181310122B12F610F610A810131123122312EB12E515E515DB14C712C7129B14F016A314A3145C1201130113CF11E618E61809118715011301131F14A0121B111B112914BB1DBB1D9F19D018D018501703160316A2148315191619162F1887154C144C14B913FD12FD12F616CB18CB18B918721858175817D317CC17CC17B118EC19EC19CA190A142F152F15EB1408150815C216C417CA18CA183819D317D3179C18F318F3183B19B819B819C51B1712CC12CD16CD167F1541164116B216A416A4169F18351835189E170A1870197019D31C4E1C4E1C0D130916A015A015E516901690162C164616D116D116FD16901690164D179C179C17C51CC31CC31C', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (176, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F47B117B11F9115B135B133C123E143E14CE155D175D17C516F6168B138B13761018121812F311E115901290128D127D117D1188120A140A150A150814D013D013DF14670F670FB90F6C126C1240127814CE12E412E412B71032113211BC10CE10CE102F1131103110E20FB60FFB0FFB0F0414D613D6135D156A10510F510F8B11DF11DF11AC0F5F0E5F0EC40E0B0FE70FE70FAC0FEB0FEB0FB513F413F4135D15E8157E167E16301654157B107B1005104D0D4D0D390E7A0E7A0E970D460F460F8F125312A613A6134F17CE170C170C17B416261326133E10A50DA50D8C0E130EB90DB90D4B0E311231124E11881188113E171C17FB17FB172315BA124D104D10830EBC0DBC0D020DF40BF40B820D5A0C5A0CB60E0C10051210151015C5143A133A137112A710A710780E860E860E9C0DA60F11131113C410C00DC00D700DEB0E64156415BE153012301215111E101E10110F0A0EA90EA90E3411F112F112D812030F030F750C180BA212A21239131E12771077109E100A100A10BA0C8B108B101A11A511A511A011C70D310E310EDA0C4F120A120A121612A410A410D010710F710FF00E7E104C114C113A114E104E10580FAD0FAD0F980E95129B139B13CE10191043104310590E680D680DD50FF10FF10F160F700F700F670FE10FDA0FDA0FF211AC1113121312421177107710D80EE40DE40D650FA7118A108A107C10670F670F980FAA0FAA0F1311D010E510E510901073107310C00ED80EA50EA50ECC0FC60FC60FE80E200F200FB10FC70F3E1687151214F3109B109B109C1049104910E40E3B103B10DC1015101510940E5F0DA70CA70C020EC610591759179314471347132D117E107E106B115810961096106810C30DC30DD60BB80DB80D6B0EED0E6E186E18EE1482134A154A151E1387108710D10F06110611520FA10EA10E990C8C0C060E060E370E091A6F166F1633155715571596161B111B11780E360FA00EA00E170EE90CE90CA10DB00DB00DB00DFC1C321D321D331D6F1D6F1D0A14F312B912B912D30E4D0E4D0EBF0EBF0EBF0E120F5B0E3C0F3C0F381D221D301D301D2C1AA611A611CA12B712B7121B104D0E4D0E9B0EA30EDE0FDE0FC71264106410', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (177, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F00040723264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4E115E115C614C61446143414E212E2124110570A570A10072F062F06AC068E088E08F30CD30CDB09461346139415C015C015A616E615E6152A11310C310C960CFD0902090209D50A630E630E9B0FD30ACD14CD14B318311831187B197619B113B113B1132D0F2D0F6710060F060F430E290E290EF60E8A0CBB160D170D17BE1876197619FE14BE13BE136D145E0E5E0E661076102C0E2C0EAD0A470E470E870A0A12E013E013BA18BA18BA185216401663136313D90EA30EA30EEA0D4D0F4D0F440F2F0F2F0F3D0F0A12C311C311AE13C514DB16DB16CA17D114D114C812550F550F110ECA0DCA0D6B0DDC0EB30CB30C0512DE13AB13AB138F140A160A16761757185718DD17DE0FA40EA40EA90F2F0C2F0C470836063606EB0EAE11C912C9122E16E31542164216A817CF12CF12C110C70EC70E960FD10ED10EB809AC06D106CA0CCA0C93119E179E176717701870182F18071607161313BA0D170D170DF80F7D0D7D0D6F075307590D590DA20F3A153A157F17F617A018A018C616AB15AB1583109D0C9D0C210E4A0F4A0F860D7D0B510F510F8F0E2515371737179315AB14AB14C4130D140D14DA1025102510BF0D930C1A0E1A0EF20DDA0F1D131D13C413121312133313091209120013721249134913E3100A100A10530EBA0EBA0E640B46116B156B156B153A204B144B14B314A813A813E5110B120B124210BC10BC10A50F7A0F5C0E5C0E3E16441529152915B8193A203A203D1B3D1B3D1B2417FC14281028105A10B30FB30F410F350F350FC6108A154516451638183618EC15EC15BB166816681620133D123D121F1213101310D20E020E560BED0E0E109916431743179915A711A7110612681668166813E113E412E4123911CA0DCA0D4A0DF20E050F050F2C1365146514C9117D137D130C1150108D118D11B51122122212A312B40DB40DBD0EFD0E031103110115170E55105510BC0F88118811261169116911D910DB11DB11FE11FB10751075104E0E49147E157E15E40EDF0DDF0D140B071207121E13B31184118411570F6B106B1001117B107B10110C2212CD15CD159611A80E42094209FD0D1E131E139E13E410E410EC0DD60CD60C6F102B10140F140F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (178, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C38264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4DB092D0D3B103B100A140D160D164E18591A6B1A6B1AFA163D193D19911A661B661B3A17B915B915D30A1C0DD80DD80DAA0F74121F171F17141B57185718371B071907192219E819E819FB1991176E16240C240C0E0ED40ED40E5F12281328131017CF1ACF1AD118F8187F187F1843194F194F19B8162710310B310B370C6C0D6C0DB411EC12C815C81599178F178F17171AA618A61858180F140F14F5148814FE0A2F0A2F0A770C5A105A107911FA12FA12CB14181718171E15F918621862181E145C125C12AC138709A909A909A008520E520EB810AF104C134C13661444144414CE146218621834162B162B163E14BD06F707F7073608A409470E470ED00F510E510E8C0FC610C61071141D13D215D215A516EA13EA13D106E40699069906BA09480A480A9F0E230F230FA90C590D1D111D112112011401144914EA12EA125307320A1E091E099C09DD0E590F590F450FB40DB40DE80CC60FC60F790F771277129411660F0F0C8C0B8C0BE70BED08ED083B0C480F480F490F470F470FD20F2C0F9E109E10850F1D131D1349139612AC0BAC0BAF09E007E007330B0D0D730E730E680F1211121107106A0F6A0F8D0F240F240F3E1343121B0C1B0CA60BF308CD08CD08200D5B0E5B0E640F031103113911CA0F160E160E0C0F54105410B80FE50B080A080A7409E208E208F20B040E040EB70D2D108E108E10D1108E0E8E0E100F84108410E110140CA809A8097608D208C409C4090C0D810E810EF110E411E411430E9E0D9E0D920EDF0E1C0F1C0F560B450AD607D6073507FF07FF07E80BFD0DFD0D260F610FD30DD30D960B8D0C8D0C980DB20DB20DF20E930E78097809920784061F0A1F0A360C3B0E3B0E930E270F270FB70D030C030C930CAE0DF50DFD0E8B10BA0A450845084F0863086308090D930F930F8A102A0EE70CE70C630F3E103E10780F880F610B610B23084F074F073207400740077D0A480FC20EC20ED40E580D580DC510581158112E11850DF108F108F807B20776067606DC06B809B809270C4D0E4D0E3B0FBA0DBA0D4A105012841184112E0D2A0C260A260ACB08C706C706910734083408C708D80AA70CA70C840DDA0EDA0E0D1101100110080E', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (179, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D264023E87981A30D474000000000000000000000000000000000E6100000140014004548F46E176E17F713F713E4127F117F11DE13DB12DB12C312D211D211C5106810630E630E980DC10CC10C6E16A9146E146E1431118C128C12F312D611221122116811481148113D11121012109B0F000E000E2710FE12151215124C11F7122B122B12D3113C0E3C0EEE0F431043103D11F60F6C106C106D0FFD0FD710D710BC11B710B710B710BD10BD10D210F313F313BF14CD10291029104B10A411A4113A11D010511651164A1629112911400B010E4A104A105711A912A912E710D90FD90F4310DA0FDA0F3C116411F914131813188C1824172417A21637143714E110B012B012EF10BA0F4C0F4C0FD410C00FC00F06111C15ED15ED15C5161F161F166F15E915E915E91548125D0F5D0F8E0DFE0EFE0E221124122412911007100C110C11DB10B915E311E3113C1285138513F711780F780FA10DA80D270F270F0312A511A5110F0CAC0EFA0FFA0F1B0F6B0F6B0F6D0F090F090F9510850FFA0DFA0DD20B800E800E82104310431096123111A20EA20EE90D0D108E108E106A0F800F800F6410A70DA70DB50EDB0EDB0EFD0FBF0F0D11341234122810EE0FEE0F5F0F3A0E3A0EED0D200F200F85100C103D0C3D0C960D9F0C9F0C62103C100F0F0F0F0D0F610F610FCB0E5F0E2E102E10F90E6C106C103C0FCB0BCB0B470A850B850B3C0D540CAB0F510F510F660F380F380F950F20102010BA10C010C0106B11BA119E149E14260FE60EE60EFF09580F4E104E10E10EEE0DEE0D040C750D6B0F6B0F63117E137E13CB12DB14DB14651347134713EF0C130CBB0DBB0DD00D6B0D110E110EEA0DC90EC90EA5142D152D151A14E413E4133B15E613F913F913F50D250E6A0E6A0E720F290E290E6C10950F950F23124D12771277128D12561456140D15ED14ED14880F6C1197119711170FE90DDF0FDF0F1A11D210D21046136C126C128B1399129912B513AE125A14AA0FAA0F011269106910830E57105710A910F710F710B4113D11251125111511D013D0131913CE12870F870F660F7C0E7C0E940E870D3C0D3C0DAD0E810D810DF60F011101118411AA12AA12DB13D20FE00BE00BB60BB00DBE10BE104D0FF00DF00D690E7F137F13DB14C612C612C6121214DB13DB133514', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (180, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C00663264023E87981A30D474000000000000000000000000000000000E6100000140014004548F4710D7114711448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A80D47155D1648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FD0FBF14601448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D0103711761148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E311E311AE1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D10FD10FBA0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F488109A109A1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40A12B80EB80E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F493107D10410E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40D111F11121048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43C10A30FC70F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4010C010CA30C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D50BD50B8D0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B40AE00BE00B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B80BA30AA30A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42D13BB0EBB0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F45A140416530E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CE12E414691048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CB10CB10341048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40B160B16951448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (181, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4A213A2132A1948183B173B172115E814E8140416A116A11679155C118A118A11DA0E36103610EE11D61638193819C51CA21AA21A8418801980191D1B4418081708170F1288128812E0114A124A1210139F1684188418F21A53197D187D18A5187D1A7D1AE818131713179E110C120C12C31254145614561444167F18BA18BA18EE18A412A4120514C618C618A5161610680E680EB90EBA0EBA0EE50FE311E311AD157416901590159D102110E20FE20F5E11890F890FA20F240F240F340F8D0E8D0EFD0F1912C819EB1094122711041104110011F70FF70FCF1086108610F410DF0FDF0FB110DE0E29102910BC13C81937103710C20F86108610481330133013BC129F1076117611ED10330F330F8E0FFB10FB1058137915F70EF70E5C0F310F221122119C11EC10EC1016107B107B102710C40EC40E560E1812AB15AB159015700ECF0ECF0EB70F850F850F0911410F410F730F3A0F130E130EF70EFD0DFD0D2713F713F7137813F20DAA0EAA0EFD0F841084107814BC123D123D1268121C0F1C0F68103B113B11C4123816D613D613B20C9D0C010E010EFD0FE313E313D4153115311523132811281184114C118A138A1338153B173B171F122512FC0FFC0F391259135913A614B11487108710F410201120112B108A118A115D15A819A819B90F25125810D710D7107B11101010107210E50FE50FE50F251025100910860FD412D412C117DB1B090C090C910C780C780CBF0B430D430DB50F770EFA0EFA0EDD0EFB0FFB0F1B10231323136518CD18970E970EBB0CC80CC80C640CF60F051005105C10781078107711B810B8109610F612EA18EA18B81AB20C600D600DAE0D820E820EA211B60FB60F7110A70FA70F5011D610FF11FF1173147E1A7E1A341BE00FE60FE60F070F210F210FA2101E0FDC0DDC0DC70E151215128B118614861492160F190F19F91A6112FB11040F040F6D10641164116B10520F520F6E0EE210E21004151E16A517A5177C17B916B916D914A212CD0DCD0DFF0D700F700F6111C810330F330F9E10D813D8139815BB18BB18271BE218E21875122C10760E760E6B0B0A0D21102110960F64116411E91112131213FC15FF18FF1878191D188817', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (182, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F476137613B11553155315B312F70F1A111A11F20F37123712F91264126412031521152115DA114C11F713F314F3144B159E159E154712F20FF20F4B10B010B010E210841294139413FB11EF12EF124E116B1F6B1F6B1F6B1F6B1F6B1F5412171117113A11B010341134112D12221222129614B816B816F915491C491C491C6B1FF214C914C9144213A613A6132613B511B5112612E411E411E7155316F416F416C819491C491C491C2B17B516B5167614F217F217F21795189518951810160E110E112C14E615E615C819C819C819C819BB199A161E191E191E191E191E19D319951895180813E513E51395125212EA117915CA15BB19BB19BB19BB19361936193619071A071AF919C919CF14CF14B5142113211396127B12DD15DD15F917F917F9176618301B301B301B36197A157A15CA17C117C1173E16E012E0129712DD107E157E154F186E1866186618301B301B301BCE18CE18CE18CE18C117C1173E16B514AA13AA137F13FF15601760174F18F81AF81AF81A761576157B15CE18CE18CE189714891489140516C514C5144215EB17A515A515EE17F81A7C197C197C197C197C193718BF18BF18331439143914AB150B15AF15AF15BA1B591BFE1AFE1A6E1B9B1C9B1C9B1C8F188F18F9187F16021602164016DE15DE15371597159715DB1B5D1B561A561A621A3B1C3B1C9B1CF918641964196719F918F918D7189B189B180017A5166714CD186D1B2020FC1CFC1C2A1D9A1B9A1B371A951A951A40199B189B189A180A190A190A19A415B7116D1B6D1B7A1B321C321C222306210621371A5D1C201C201CC2167313731370187018701894121513A217A2177F1C421D22232223982106210621082106210621960FF50EF50ED7138A0F120F120F751687176A196A197D20B020B0208D17AA0FAA0F560BD40A570F570F630FB00DB00D1D0B220B220BE20EBF1AD71AD71A9714B614B614AA0E6111ED0BED0BD50DC50DC50DAC0AFE0AFE0AC90331063106550A8E19A418441544153012010E010EFD0ECA19CA199B0D9B0D9B0D7F0B8A0CB30CB30CE00DAF10AF1088179316631263129F109A0F9A0F4F0F5311A80CA80C770DD20DD20D1B0EA10FA10FD80F08130813', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (183, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F40F110F110112F614F6148B130F150F152017201720175617ED15A912A9120A147414741420140B14491149117014B017B01706176B17D217D217E217B91AB91AF319191A191AD41BB119B1198314E713EC14B515B515C61888188818301814191419AA1AB51AB51A1A1A2B1A3C1C3C1CB9199C149C149D14F51B3E1C3E1C391A5B195B19631C141D681D681D481B7C177C174015DA16DA160B17CB14CB14AA12191A54195419391ABA1AB61DB61DC91DCE1BCE1BBB19791479145518611A56165616EA13D512D512EA11A21258185818E9187F1C7F1CF819A417A41771146B11801580151A19221722179112B312B3127B12451419141914C31A891701170117A311250A250A580B22102210D912FB10FB10451285125310361236126E168F158F154715471547156810CE0ECE0E411342129E129E127E101D121D12A011FF0F6E106E102512211121110711D110B010B01062108F118F11D411B412B4125B12C410C410210F110E0D130D13D611A50F830E830E4F1049104910EB0F870F870F270F741108120812C51190119011730D7414511251127210910F910F64114F0F4F0FE30C490D2F0C2F0CB00FCC11CC11D8115F125F123A13F313891289125610B310A610A610C20EC20CC20CC10AB20CB20C1710DD12DD12CB112012E314E3146714AC12D011D011F211791279124C11780E780ED40D590E120F120F7310FF12FF121713AC13AC13B711A812C212C212D013D4136212621282100A100A1031107D0F7D0F6910801280124D1232135C17151328138513D913D913A31308130813A711B610B610BB11501150116A115F12C911C911541226132A132A13AC12F513F513081308130813E210A110BA10BA108312E012E012F912A511A5119A12ED12DF0EDF0E390FFA13FA11FA11961144114411FC12091309135C13E814E8143C133112421142114C130D0C721172117D12EE10EE100810C911C9110D13B5155E155E15B415A411A4112812630F630FE610B30C761276126112051005100311D10E881288127F1496159615A114011201125310FD0ECA0FCA0FE71009104B104B10E81080118011AA111B131B133314361336139F126D1321122112700F6B0F6B0F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (184, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F40B14F915B51696189618D417F317F317E0166D166D161F16EA15EA15221450173C183C18DA19D71DE814E814A3164C184C18B218C317C317261719169916991671139A139A135714EF15EF154118D81963146314E614A715A71503177019BC17BC176B17031603166D14E614E6145E14351535153E17151ACF159D169D16C5167A177A17A917631763175F17CA16CA1651162914C613C6134915071507155F16151515161516AB16661766178617711798179817ED16581558154C138A128A125E122510251001132712D214F716F7166E16D415D415BF1605170517581758155815ED12671129112911A1105F125F12531073148F168F16A01777177717A9161117201720171217701370136412F910F910740F7A0F7A0FFF0F25139118911863197717C315C3152617F416F416011747144714E21336113611F010D00E9B15DB15DB158B18F11AF11A3C19451845180A18EB16EB1659166F14AC13AC13510F35103510100EEC120F150F15F618C31AC31AAC1ADB1A551855180A1886158615EB15AC13AC131A11EF0EEF0E120DBF0DD018C21AC21AD61A911C911C851C271927190A1867156715A715340F32103210C50EE209E209EA075E19521B521B4E1B5A1B5A1B761C0E190E19D3149E138F138F13AA10E20EE20EAA0C46084608E3089418BD1ABD1AEE1A301BC21AC21A7E17281728176E159B139B13721303120312200A8309C808C8085C174818E919E919D819B018B018A417411741176E1657163A143A141211490A490AB809350A350A26136F15FE1AFE1A151B871A8B198B193818D117D117E716B915B9153A14770B770BA3099E09070AED1203147119BE19BE19BE192B1A2B1A71190818081862175417781378139A0EE50CE50C7F09EC097E147E146319361A361ADC1A981A981A8E18CF179F199F198919B614B614E70FCB0FCB0FB009B90AEF15EF15CF167C193A1B3A1B701A12181218E8189F199F19111A671867187912E708F108F108A80A0E1663166316E41635183518FC174F174F171D19D71AA41CA41C6C18D317D317220806080608580B73148A168A160E1858176F176F17C217AE1AAE1A001BA21BA21B6917411741174D0988080A090A09', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (185, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4D71D8C1BA01BA01BD717AC11AC110F110F1196119611631071107110CF13CF13CF13351856185618D819341D311DD01DD01DD311F615F615CF13EF13EF139813551255122413D21FB012B012AB197922AB19AB196C1CEF1CEF1C691FF41AF41AAF18AF189C169C167616480F480FF60D9E0C9E0CE210B71F1A151A15C51817191719BA1BF318AF18AF18AF18EA18EA183D14DC11DC11060F8C0EE00FE00F8821AF0C2B0A2B0AFE0CD50BD50BD0131F1A1F1AC91D611C611CFE1B651929182918831795159515C1206E11081708172D0C0C0E0C0EB11824190F1D0F1D36233623362381232F202F207A23FB20FB203D216E12851EAF1CAF1C9A1D4E1B4E1BEE19E91AE91AEF20BD20BD20BD238A268A268A26B321DD1ADD1A9B15B518F518F518CF1AC41AC41A951BA324A324A324A324002300238A268A268A264B138F0B8F0BEC123C130D160D162B1A2B1AC11EC11EF823192419248B22B91AB91AB520E018D108D108EC07EC09DA0DDA0D970FEE1AEE1A5F1B1E1E1E1EF82316241624BB226B2241124112160DC105C105020515036E096E09970F25142514AA16A018D024D024A524D822D8226B22A817A817850749054905F7023803940AB80FB80FE717F315F3155C195A215A214519491F491FB01EED15400540050407120312031203B709E10AE10A87178A178A17581F1D14BA13BA13531A64116411270A3B053B05CA02150415041108A20AA210A210240F890EA314A314AC0CDC08DC082405E206E206FD078F034D034D03D107730A730A070A2C0ECF0ECF0E950E04090409E40828032803D5034E026803680360019B049B040E05150F150FEC09B00C780C780CB30B3A09660666062802460346036E02790179017E00F402F402FE05F30A340EE30AE30A7E08D508D508CC074B054B053305C703C7031102CB01940094003004440744071A0CFF0F2C0A2C0AB709620962092006410550035003B6063E063E064A020F020F022B037D057D05AB0B4D0E8E098E096F0BE708210721078E04AC02AC0276065C065C0691048906660566054C076A086A080109BF0A000B000BD3071B061B067604030203027905BF0792069206D10D5F0C5F0C090D800A800A5E0A', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (186, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4541B9D1BDF13DF13DE0F0E0F0E0FE01020152015A4137C118A0D8A0D500E15101510D2100F110F117922D31D931793172A11E20FE20FEC1420157E157E15C711E70DE70D7C0EAF0FAF0F9E10DD0F630FB71F7B1C38134F184F18CC152B162B16A015CB15CB159F0F7A0E7A0EA8120E1145114511DD0FE50EAE13AE137B131E181E18CF188B188B180919DE1A92179217FD15551355136E11B510B5103110630EF117F117F117C620F320F320C6206D1B6D1B771CEF18EF18F915121312133D103D0FF70DF70D410DF117F117F117C6207821782178219F1F9F1FBD1FCB1D351535150712EA0FEA0F4D0F710E710E960EEF118F0F8F0FF6167821782174208D1F451C451CE51DA71BA71B0710C10FC10FB30F2110AA0FAA0F000B3F0F02130213C911131213123B13DC15DC1590184F164F164610890F8B118B11FF0F5A115A11EC09A40DA811A8114611C90EC90E54132913D014D014F215F90FF90FC60F850F850FF70FA110A110150386033F07880988092515A914A9148D116E136E132313850F850F8510A5113E113E1193113C11210321032305650965098E0E5D0E5D0E850FD5105C115C112C10DE10DE10FF1276117611A01282100C040C041B061C0C1C0CEC0DB50F53105310FD10411141116110E510E510ED127510600F600FE70A7609AC09AC09E310960F960FBA10531153118B124911491162103A10591059100B105E0D5E0D120A710C700E700E3D12C510C510BB12A714AD13AD13CB13811281122E11680F680FD00DEE0DEE0DA80E150F2012A713A713FD13951295125416F818F8181D161F141F14C113AF123E113E11230F7C0F7C0F340EC413481448148E14301530151D1622156813681360143015301500153F123F12F71021102110FF0F1C139114911404111C152116211661157E127E12CB1316141614B9147113FC12FC120D110010F912F912C514431443146F137F137F13CD1263136313E71322153315331532120E120E121B11C710DC10DC104B11F111F1115210321144124412BB11121312136A14561556158614B810B8100F11970F7F09070A070A1911B611B6113D12A614A614FA12ED11ED117F1273130A150A15AB106F106F105210', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (187, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4E10E670A670A670AD107DA07DA079A0A90099009480A550C550C5C0E811081107B11C810A40DA40D630F8B0867076707C8073D083D08030980098009EE0A3F0C4E0F4E0F6F11361136117110BF10BF10E50E8A087E087E0808088408840838094208640C640CA30BCB0ECB0E2212281328130E12E8108210630E000A5509740874082C091F0B1F0BF008990899084B0A531253127512D4126A106A102F111511000B000B380920092009EC08A00AA00AC40A5B09C00BC00B3110C310C3109A10F910F9105110340FB50CB50C050C42098409840912096A0A6A0AFE090D0B0D0B0F0EA410A410930FB20F301130110212840B6A0A6A0A40095E095E098909E90AE90A1A0AC10A740D740D970F711071107B10091009108012700C8E098E09C60B240CFC0BFC0B080AF709F7092809930B930BB70FF810F810291048108C128C12140EDC0CA80AA80AEE0D160F160F1B0D1F081F08800A2C0D500D500DD50EBC0FBC0F380F9B129B123C117B0C080A080A490BE90AE90A400F7D0A570B570B200EC60CC60CE20DA710A7107A0F0810370E82106A0B830885088508970A1D0E1D0EFD09AB0CAB0C310F070D070DD80D89103C113C11B01259104F0A4F0ACE07AD08AD08CC08950B950B130CEC0CFC0DFC0DB50FE80DE80DA9103F123F12B6121A13630B630B2E081E0A340B340B710A220922091A0D6B0F6B0FA40FD90ED90E82116D13CA13CA13CA16AD0F450B450B390C010D010D010DDD09DD094D0CCC0EFD10FD1013108212821210121A131A13D017560FBD10BD102110D610D610E40EA9097B0C7B0C500D751075105A10B311B31119136112891B891B820FEA0F6A116A11F01138113811F70E520C520C410C0E0F0E0F8E10CB13781378132113351A351A0010E50FA810A810891115121512910E0E0D160E160EED1052125212DC1447144714AC13D117D117C710120F820F0F110F110F11A50DA50DB00D280F280F3710DF10DF10A6123E14E214E214D1146715581058102E11A313A313C511320E320E0C0E76109B109B102511421142118B12E012E012D1147014EB10EB103111B512B5126E135210CC0DCC0DBF0FCD12CD12AF11F810F8108012B312A712A7125912', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (188, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F4640C4E0C4E0CBB0E35113511B411531253123F14D215F719F7199416D114D114EA15C615C61547137D10390D390D8A109A121A121A122616B116B116DF16BE19BE19E41D0F1C0F1C4715D313B417B4178210DA10381238123F12CA14CA1468166A186A18291CE41EC11CC11C631DA115A1152516BD15BD151511D511D812D8124516A816ED16ED16761717191719E31EA818A81809181F141F14671628111D17340FA4128412AE17AE17501903170317FD15941694169416B512A811A8110B103D123D1261117815A512A5124B159F1B9F1BBF19251825182814E61500160016FF149E109E10510F990F990FB9153113F413F413C719AD19791779179B13BB14BB148114A713A713F81403120312E80EC40E721172114B115B1386148614851656165616E414501250127D149B14381438143E135B115B11F40E7C117C11F9110213E613E613EB14E215AD13AD13A7127C127C125A13A813A813BA11EA10EA102F102D11AC12AC12370E291268146814D0143B153B159812B612B6120F148C14321132114912FE10FE106A0F0711071159100913661866180E15581458148A123D13B415B4151114FC12FC12D413AB11AB118C10AF0F8E131A139F166618CF15CF158813EE13EE13D8149716971695169E169E1695134910930F930FE80F2414F019F019F61A611661167D137E137E13BB145D16821782177B16BC16BC1655125C115C1152111417C919C9199417A917611461147E153A143A1423169B1A9B1AE01B341934197819F712A813A8135C13891BC318C3187D1773177317711613161316D411DB18E319E319F41A72167216D50F5F125F128211D11B39183918D017C418491849188A16581458146218351A351AA518211421144E102E0F7610761008172F181B171B17A517901690161C17391939193B1FD71DDF18DF18E311A50FA50FA910621062106715EF15AB16AB164F163E183E185215D419F61DF61D7C1A6A186A186011E80FE80F5710C011E9147014F213F214F014F0143D13FE10FE10AD14E91BE91B421894199419DD149D157F137F13CC15A4178B138B13F014A714A71430156F156F1516155D152115211563194F124F121F1364146414E615DE15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (189, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640AE3ABC314E08474000000000000000000000000000000000E6100000140014004548F40B158119811948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40A19151B151B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C616341C341C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41D17141EFC1D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F478159015D61B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49F159F15451648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F423132313031648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4ED11ED11EE1348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F444130B150B1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C7139314931448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48E138A15BD1B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F424145C18461B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41417961AE91A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A916A916331748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F457105710A01848F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41A0FB010B01048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4ED112D142D1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4E914CD14221548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A41784165C1A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DE155118031B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (190, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4DE0CDE0C7A0C450C450C780DBF0FBF0F630F6411641102131012BE14BE149C18F018F01816161B16DC0CDC0CE80BCB0CCB0C7B0EEC0FC810C8109B0F591259121D131D141D143017441B441BF915A119960B960B6B0B550C8D0E8D0E430F450E450EDB0FBA0EBA0EC0107B14B517B517AE1D4C184C18C317FA0A330B330B820B990E990E13105F105F1018110E0E370E370ECF0FAA15AA15BE195F175F17D91C0F0BBC0ABC0A5E0BB00CEF0EEF0EF3106D0F6D0F780ED60DD60D6E12331733173B191D18DC19DC19F70B4A0ACF0BCF0B960DD20FD20F6A0F3A0F3A0F780E29108B138B130519B719B71944179D1A9D1AC50DD70CFB0BFB0B780C710F231023108910D90FD90F5F103E153E15AB18C11AC11A3017AA19CB18ED0FE00D730D130D130D360E2D122D12E810991199119814F7170B170B178E179A169A16561A6414110E110EEC0D9F0D9F0D4E0EDA13DA13D91227176E186E186218621862186017061706174F195410C30EC30EBE0B240D7C0E7C0E56141F1A1F1A391B9D199D194C1B2B1B2B1B6918DF162C0E2C0E870DE50E1E0E1E0E430D380D380D2C14381A381ABD1A47195A1A5A1A1F1E51185118C716471747172C11A911CC0ECC0ECE0E6A0F6A0FA51114184A184A18331973197319751BB919B9192818C9141F141F149113631192119211AD1067146714FB15EC18EC187417DF19DF19B4196117781378136E0EE80BE80BC214BF13311531159B12E31CE31C141EDF15981A981AF6191E191E19DC18611561159C14550D550D1F14DA13A2162E162E16C2172A162A1612153C1B3C1BCE1B6E186E18F018271A781978194012531079147914EB15491A491A5616331533153014A01A0D1B0D1B51197F197F19781A861B861B79134512521352139B11401A401ADB17CD16EF13EF13AF19AF19AF19E9182B172B1729167F1B42154215A114BD123812381230156217621780190B160B16B216B016B0165418D616EE15EE155A154014401416132616201520155C152817281711168914E116E116AD178E158E15F4155E175E1722140115011500153113BE14781578152B154813481371123A103A10E1135D145D141316A217341534157A1420152015', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (191, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F41B168B161A141A145712A0140C0F0C0F110E110E110E330F570E570EA70F6712371337136612CF10461846185B153C133C130515E813E813341234123412330F330FDC11DC118011BE14BE148210460FDE17DE17D31A4A1B4A1B8B194E14021302133512E213E21305132B152B158A131C161C163E145A10291ADB12DB126D10EB19EB19E1173614361407150D140D14C3165F179C159C15BB133A123A123A11EE1794139413600F700F700F4D10E8126715671522154B144B14FF159C159C15DB13AC11AC1172117E16C114C114090F240F521052107514191519153415C014C0145515701578127812C51215121512CB181C16031B031B62160D110D11AE1236143614E615A9153114311447124D114D1174125F125F126414620D970B970B420E231194139413331B951795171B16BA17BA17B313E110E1109A1044129612400B400BB316E70DE70D9810721072103C14801780174519B1182E152E15E112BC12BC126C11D211170B170B280F031103119810890F1B121B121018DA17DA172F17FD13FD139012BC13BC135F125A110B0E0B0ED80F0311EA11EA11DC0FDE11DE11911470137013DE153E158D128D121E13C812C81228130312561356136D13DF0FDF0FB11249174917A015B817EC14EC1418140211021125122A122A127A12F70B7B0E7B0E950DA60F7E167E168216F518F5182A1AAC16AC16BD140F100F10A4109F135F145F14FF0CED0D2E132E131A11F312F3122A16D81CD81CD81C5216421442140F10F20FF20FB51343154315531018106315631599185E1D5E1D3C1E741DD81CD81C6315B711B7114C0FF80EF80E55122017B814451259103A1378177817FA1CCD1BCD1B8C1DA61EA61E9217781178117611B50EC410C4109611A113C513C513D1165A165A16581C1D1D1D1D321DCB1A76187618861309110911400E5D0E5D0EB51013112C142C149D15B513C616C6169218B818B818B11B56185618B912931393132114E10E080E080E9D0E431348154815B815B012B0129D15A218A21833170717FA15FA15A314C314C314B5102E0E2E0E1C107613BA15BA15F915F413F41354170817931393130A14D011D0113913FC14FC14B3124E10F710F710', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (192, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4CF10C00E551055103911C510C51018101B1362126212F1120A130A1323121B121B128A0EF10EF10E460FEB0DCC1050115011F810A010A0108C140F140F14C812C812C812CB13B913040F040F210E7312A50DA50D22100D110D119710101110116C110E12DA13DA13721491129112C712571057101311401384128412AD0F15121512BF119910F010F010E5121115111594156F126F1200131512B210B210B011EF11D512D5123F12E112E11266139B109B1011162917291795165712BA11BA11D5110D0F0D0F6C0E2513071207128A10B310B310BC11A6101A161A162217F216F216C312B012B0124212670F670F970EBD14BD1419131913FB1199109910CC1055145514B315DE17DE17BD14A61240124012560E1F0E1F0E96125A1459125912890F300F300F67120C131F141F14F9175A165A16C91052105210340D5F0D5F0DD2117B1034103410720F920ECA12CA12691356155615B517331633168410810D350D350D7B0C0A0D98119811D211371037100E11AF11AF11E41164136413AC142E1282118211C50DA80CA80C250D730D33153315EA14F110F1105C11F011F512F512B612E913E9133D10AE11AE11F10F080E080E150EE90FAE134B114B111711311131110913C212C21278118D118D11B3102511060F060FE60BA80CA80CEF0E59124C104C10E10F3912391286136C14B812B8126D128B108B105B0DE00BE00BB40A390D390DD515161424112411DC101A12AB14AB14BE1414151415C51473157315470E0E0B2D0E2D0E020E7F127F12B8149D12B512B51262134A144A14A8132E152E155215B915B10FB10F590EDE0EDE0EA011AB13AB13A113D0114B114B116A126914F315F315A8172A192A19F9180D150D15920E540F540F1F10BE12CC131311B8119411FB11FB11D513B915B9156616A217A217FC182F1669116911BD0E280F280FD60F5A14DB11DB11DD11B812B812D913C915C915EE15A117C315C31577167C137C13930E4D0F4D0FB010AB13D812D8129612AD14021402140E13E314E3141714DD13DD1322161E12360E360E360E9B0F9B0FFD128716A916A9163B15231423144E1457155715EB14AF1406150615C512310F310FAB0FAB0FAB0FAD0F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (193, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4131402152A162A1679171A171A175417E318E318B819711B9B149B14EA11FC07FC07A2086409640973126E139C159C150517521552157C152017DC17DC17DC17D411D411F90E3A083A0835085106A1064013E217E717BC17BC1743163F153F1568156F146F14E513C913C913540E92097408740882060705671667166018B519B519F917D817D817BD17B914E513E513C9134D114D11C30A370537054D062C0391129112F715A01868186818631840174017D9157315731515156C116C113D0CE2066E056E05B802B811D315D3158E1889198919E11850175017F1162E162E162E16C8132F0C2F0C7E08C306C306B902F90F371537150019621862188418CF18DD17DD171D171D171D172814280E280E2D07AE07800380036611DB18AB18AB180C17AB16AB16401759175917F1164E164E164E167111500A500AAE07200320030A0D1915AB18AB187D18D216D2164D178616C116C1160E160E160E160814260C260CC504FE03FE03730DAC12F816F616F6163E17831683161B1652175217151715171517BB13280E280A280AB2043E02B216B216AB17D517D5170A17321632163216E016411641163314B413B4137310810981096406DE02C715C715D618AC17AC17AB155C160D160D167716871687164E157814781493134E0F0506050633064E15221722173C17C614C61433158E168E16FA14FA14FA14EC13E813BD13BD132F0F900B900B7504FA154E144E14F31431163116B61499149E149E14D911FA13FA132C16DE16DE16EB0DEA04EA045305A314EC140C150C155A144B144B14411437133713E413381238128C0E320E9D0B9D0BE0043D053D05CC13F013FF14FF148A144E134E13D5141F14F513F513EC1251105110B70EA107A1077A021E031E035A14A013D714D714C7143F13BC14BC1437143714371460114C114C1184081D0765046504C5037504871487147D13871387134A13CA13CA134914A313A3135711E60D040B040B330A2E0A2E0A0F0931056D136D13B1141E131E13EE1245143B143B140F1297109710760DDE0FDE0F4E10B50DB50DF10AC809101354165416891394139413031413141314FD12A512A512A60F831010121012A612460A460A3508', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (194, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4350442044204900788071303130319023E033E03DD08020B020BCA0D5513551321121E0E57115711A1068006EB05EB05F505DA04DA04A802EC03EC03AD00120ACF0BCF0B9B122F112F111414BE12BE120705D603FA05FA057F05B7045D015D018C0292019201E0020F0D0F0D230E240F240F9B13C212CD122C033205D102AB02AB029C03470347037701250225022703A902390139018604C510C5102A11CD12FE02FE02980218041804850485048504CD04DB02270327032D031604160451047B0B7B0B560BDA15B902B902A400400586048604EF03AE04AE04B402C702C7022405FE07FE077D0A660D5E0C5E0C0015E002B501B501F801270327032F046A046A044602FB02550655060A096E0A6E0A4E0F400F400F15165D02640364034A030103B306B3066909DE0BDE0BF30B9B069B061E076A096A09021004179D179D17D303BD02E903E903210448084808630C6B0E6B0EA8105E0F400D400DFC0BF30FF30F8816A31CA31C3E023C0389038903E10690089008630C850F59105910990F8F0E8F0EC6106E0D6E0DE811621D9219DE0297016B02F903F9032C04510651068E07170617061A0E730F730FEA106C11571257120A0F710C1C031C03E90255055505B50487048704A504D904430543053106C507C507C3100F120F12110F8D0831023102B802A3012E042E045504030503051E05240724074806130C130C550CEF0DE40DE40D3209FF044E034E035704C302C3020304EF04EF04CC019308D608D608A20BDA0CDA0C45111A111A116810D9045C035C037705BD04BD0479041504C906C9064A08520A520AED0D340F340FEE0F0412781378135D05E802C701C701C70177057705C7021A051A053003FD0DFD0D3610DA0DC910C9101413DD13DD137504E602BC01BC019C029C029C021E064C08A102A102811174177417EA120411041199112D122D1231056D05E405E405E4059C02090B090B7B0B300A300AAE10AD17AD177E125112231223122D129F1026052605DB07230623069B074F0C4F0CD50C5C0B4C104C104110201120118C11AA11AA119711BB0F2E082E085408B507B507AB08C70861096109810DC00EC00ED70F960F960FDD12480F4E0F4E0F2D0F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (195, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F4E80DF90DF90D8F0FF011F0112F135515FD13FD139A12C211C211BB112A132A135E113F123F121F111E1249124912F0112C144A174A17FB135E145E142A1146114611BB113111B710B7109B0F060F060FCD12051302160216CF16B418B4184C189B169B163814E8125E115E114711661166111910EB0DEB0DCD120A15F514F5144317821847174717B716FA12FA122A13F611F6113C12671167118210F70E4911D016D01625137E137E13880FE20EE20EBD11021202128B125612291229123C12041104117B10EF0F8C188C18ED133B133B1338124312F710F710E8115B125B12731292119211C1107E107E101811FB0F7B187B18CC15CC12251225127F12A212A212DA123A133A133D12C8119B109B1057102C102C109B0F7B18C918C918E61307140714DB158E128E12E90EF812F112F1128B13C612C6124E108E0F8E0F640FFD19471747178617AF15D115D1159610E80EE80EEC0E061306133113F90FF90FD30FEF0E2810281092194A15C119C119551E9A159A156514910E910E29106F10EE10EE105310A30FA30FA30EA60FA60F710C6E0EBC13BC138015A913471347138C0EB70FB70FBC108D108D10590FBF0FBF0F0710EB1038128D089A0D3C10F810F810F91282128212EE114E104E1068114511BA0FBA0F960FE50FE50FDC116012A50AA50A4D10BD10BD10B615C014C014BB141A11411241123F11790E790E860E031103110D12D7111C0D1C0D7C11A710441244127412B010B0101F126C126C127A1309100910F40FB11144114411DF12A614DD12DD12B70F201120117211061006103E1134114912491267107A137A13F2131D131D137812DD130E120E125F10A810F50EF50E261046104610CC11700E700E5A0F6C126C1270137A1378137813D1127D1106100610F80E750F750FAF0F8A0F8A0F26117810600F600FFA10FF12FF124614D511D5119F1004117B117B11DB113D103D104E0F1B0FD410D410E8115D105D109A11461246120F141F121911BB0FDB0FFD0E860F860F700F9A0F9A0F960F09110911EB104B104B10F8109211A212A2120A11D8104B0F4B0FA30FDE0FDE0FED10C010C010CA0F0410720F720FCC0CFC0DFC0D30120F120F12F810730E', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (196, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F474102313231308129D129D127F0E660E660E810FE110E1104712C211701070102513401340131A134610D712D712DB111C101C108D0F080DFB0EFB0EDD0F2E112E115E1053105310671276137613FB13F610BA109C109C10670F04100410080E1B0F1B0F5E10571057109E10B00E0B110B11C912FD13FD1349119112F112F1122011761076102F10FA0EFB0FFB0F630E340E340E8E0E80108010F312EB13EB13EF0FE4105A125A1212117B10C20FC20F360EBA0EBA0EEC0D7D0D7D0DC90F9C114E134E139E1315159D109D101A10651065108C0FFB0EFB0E940B1E0B1E0BA30B5F0E0D0F0D0FF211971297121213F912B410B4102D117C107C109F0EE90B9E099E09B60BFD0CFD0CBD0EB80FB80FB4107811781174112F11A9109F119F112811F60FF60F750B660B660B230BAA0FAA0F30107210C00EC00E4C1130123012C5125B11301330137C11B60EB60E9B0DCF09220B220BA70E0B100B10D811931293128813EB14EB143A156C12D113D11355121610A40EA40EF80AFF0CFF0CBF0F4F124F126A156A15331433147816121812183812B0124D114D11BE0EE40EE40EE10A6B0D6B0DEF109D1380148014FA147A1A7A1A8617001800186012D8124D114D11780F750ECD0BCD0BA51085138513BA13D313D313181422142214F5147916C016801280129111860F860FC90D6C0F6C0FEC10C213C2130D14C2158415841507183A153A151615F713D211D211FC10A40FA40FD00E03107E107E10BB13A713A713361736163616B7177519751985131B14EA11EA1111115B0F740E740E370F0F100F10E111EA13EA13AA15AA152714271416140B130B1305141712531253128B10B90DB90D3D0F8E108E1042121613C819C819C514FE14FE14E912801980194A13C710D910D910FF0FD10DA10FA10F3910C318C318D216F919F91932179C14D315D3156A1EAA18AA181911530F840D840D180FA20EA20EC710621662162A16BD122B192B190E174D144D14BE18CE14CE14D810630FB50EB50E730F5C100213021357164A154A155918DC19DC199D1080118011CA143F15BF16730E9C0E1F0FA710A710B91671127112A614F716F7162817131710161016B01A701A701A6A13C117', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (197, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F400130013151415143015301569157016701670163A163A163C161B101B100811F4172A132A13551941154D144D146916E619E619E0194C184C186B16FD15B614B6143A140E140E1446126A136A131C16DC15F516F5169F171E1B1E1BA51A6C184C184C184C18F215F2157F16561656160E166814491549159A144D15FE16FE1665196519651941194F174F174C187C167C16F215E1149A129A122913111311131515FF14CA14CA14E514A015A01551171F184F174F179C1710171017C4149F149F14A91269126912F912BF12C413611461146713A714A71474171F181F18C117C117C1179C176916EA14EA14A6117C1414111411E21232133213F0115D145D142715F016CD16CD16CD16EC16EC162316231523159112F4118B128B127A12D212D2129212E8107B127B12A612B016B016B117BC16BC1663130611DC0EDC0EF20EBD168B198B194A1409120912150F55115511A911321132115011871092119211BC11541054106B0DE416271B271B271B00130013E013C30F2C0F2C0F2B0E781078105C10F70EF70ED80D52095209DC0BF618111A2F182F18A414C614C614F6148A0F8A0F770D6B0C6B0C1B0D310DF008F008F00895069506C016B916681668160815D817D817D8177916791679164213330C330CE80AB304B304B304C506C506F71364134F134F134F138C178C178C17F118601860186317C70FC70FC70FC70FE008E0080A098205BF10BF102F132F132F1386185F185F1837174517BD17BD17F812271327133E13D713D713280D7E0B8B0C8B0C2A12E611E611E415B418851685168916D014D014E0147A157A15A019AB145E145E14220EA01504160416BF0E8E158E152719E418E418131703180318F018AD19BA15BA15A619A915A9156815F8160C160C167F15981798176A175416F315F315F2154D154D152D15C318C318A6194918491870167E105E110D110D113110EC13EC13F5165E165E16D216F615F61599158C20DA21DA21701AFE19FE19BF16C61634103410580E5A0D5A0D870B840B331533151013FE19FE19A81AD81AD81AB51AB119B119C1170115950C950CC3076606B405B405440AC10BC10B770BA117A117201689192C1A2C1A3C194119', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (198, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640398DFEE1F802474000000000000000000000000000000000E6100000140014004548F47A197A19C91B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4231B231BF71A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A3168E178E1748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46F17D415D41548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48615D415971648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47C148B14851748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4F4115813CF1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43E0F3E0F391048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F43D0E3D0EAE0E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4780CC80CC80C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F40D06280A280A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F428079B064C0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4820551076E0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F47E0BBE07E30748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D608D608600748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F432103210730D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41417F40DF40D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4321A2B102B1048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4461C9517DF1548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44119BD19601448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (199, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F497169C172C1B2C1B0A1A881388139D12280E0A120A126813181918199617AA15AA156C153E153E157B18C31A681C681C681CDE1374157415650F170E170EC0105B165B169C15BE14E716E71607152C155A175A17FB187F157F150B15871487149E0FE110E110C010A112A112A1125014CC15CC15CB1408147B167B16BD172F162F16CB139E139E119E11021299139913C8120E140E14F9140F140F143B124B13FA16FA16AB1C011965146514C312271227121811F910F910A6115212C614C614CA113E123E12F5118819091D091D721FB01CB01C9E13AE0CAE0CAE0C7811FD11FD115712071507156912D812D81254147B19ED1DED1D691E331C1B141B14970DB90FB90FF011D814D814121332143214BB1350137C157C1526143A1815181518E31872137213250F951095108E134D15921492142D12B113B1132B15E716E716C01D3D1A4F184F183A15E01A82168216C01341154115F11531143114E513B113B113B41506178B16C01D5A185618BF15BF151D17D31CD31C771512141214EF153013E112E112B013281628160018EA147D197D19D91702140214141AC116C116A8145611F611F61191120C140C143415EB16EB16D917461251195119CD17EB16441444148916C114C114EE121B131B13A71258145814A1144715571457141E127A1C321732175417CE15CE15BF179C169C166A155011BD11BD1149137C157C152414541354131E13AD173817381766150B170B17D4134C13121312133712C214C214D11486148614281499137A117A11C2162F1748144814C216661466142612971297122D13DE1583158315441564156415DA131C121C12EE1475160B160B16C31363126312C213E512C313C313AA151F131F136011C414C41445160A14DA12C51386142013CD12CD1236135015501543155C155C1575135013501300138B13561356134E1387132A142A14FB12C412C4127C154516451617162C1548124812381388138813D8123C143C149314C313A913A9136113E515E5157217CC17FD16FD16DF13BD12BD126715671567159C15341538143814C4146B12661266126213CE15CE15AD1566146614B513BC12BC1286156715851585158115FB14FB144D12', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (200, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4A314AD14D314D3145E13691569159B17A816A816AD16FD11FD114E0FBF0F470F470F9E0EDA10DA102C156915FE13FE137612141414146A149514AD16AD166312EC11EC116E0FBF0EBF0E9D0FE10FE10F0814E11456141C121C123E1427142714781343124312E5117C117C11AB0ECE0E8B138B13C7125117671467142C143C123C1298132B142B14A712791384128412A611F70FF70F4B11F013F01379174A18CE12CE1216155E155E1586162B16ED12ED123B124D114D119111D510D5102C134A15CC13CC13EE13ED16B016B01690168D148D14EE15A412A412CB11511151116B120113CD17CD17D315D912D912F4114217771577153F148912891245113211F410F410D510A411A411B012D416D4164816FC12FC1200138D171914191429137F12701170119F10F610F6107110C110C11049160F177315731532145C135C138B16EC12E511E5110A11240F240F9A10F410FB10FB10181249184918B815A713A7133B13DF14DF14EA148312A711A711F60F8A0D511051106512AF11AF117A14091B091B6D16D11328162816D71230131C121C129010C30FC30FA20EBB10BB1052127B157B155A156818581558150E130E150E1546134015E611E611541124112411A30FEC10741274126514711971195E146C136C130013FA15FA15771443160E130E1391122211C50FC50FCB1075157515AA18BF18BF185313AE138F148F14BD16F117F117DE16D11160126012D11043114311F512DC17DC179B1307143E133E134F13AC14AC144C19411941199B17BB109A119A112812BF1121182118571B7F1A7F1A38165E165E1660132F132F139816D717CA16CA16DA12FC10331233125916171A171AA01BDD18DD182015E915FD13FD13FA127914791430157815781587139A119B139B13C915691E301F301F6F17101510159113851385137F137513751393127612C311C31375112C12B61CB61CB61CAD1EAD1EDF16F510F510F910F810BF12BF12521251125112371239131712171241125B165B16B316981A981A6016C31607150715F711BB11BB11D3129A139A138514D113461246127411D816A019A019A019451945193E18B619B619CE12C714C7145113BB15DA17DA17E411', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (201, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F471166716671641169F12201620169815411641166915FE13FE133B12E00EE00EC70DA50E480F480F61141515071407146813171517158B16FA16FA16BA152B1426122612F00F970D970D160E2B0C2B0C5117CA14F613F613181310131013C0146E160B160B167E14F911F9112B0F090C090C390DDF0E05114A18D6133B12931293124213CD13CD13CD14451545155213D011D011360F8F0E4F0F4F0F410F2C111D131D135011DF12DF124C138B138B13C2137F13DE10DE10D010310E310E960EE80EE80E830F071275107510DA107D118B118B1102110311031183116D116D11070F1D0E1D0E0C108A0E65106510401305114D114D11C6119B119B112512BF11BF110F129311650F650FE90F4B0F4B0FF10F2F102F104E139A10551155118110900F900F32119310951095105D12691069103211731173110711891444134413AE11AE11EC0FEC0F550F051005104010F70FF70FBD109F119F11841100128A158A157B15EA14EA143013E9103A123A12B8106E126E12E212971265156515D318BE1ABE1A9E16441A441A6F189D199D194015F2155A13A315A31518179D149D14B813311231125F18211E211E801CC921FB1EFB1EF91BE717DB14DB1457156A156A154118D916D916511633164B184B18AD19721C721CAF1A801D801DE51BEF19B214B214F616D015D015F7153D16F815F8150516BF17BF17631971197119F019B7188E158E15FF15D5157B167B16731692139213BF16FA15FA15E31454165416FC17BD1626192619E9167B157B15FD15FC152516251689130B150B15BD14B116BA14BA14DF15C516C51625198218821882154E164E16A3190014691369131116C5164F144F143F142615261539166F146F140816261377177717261706200620C3115B1219111911401135153515D5146D15171517153E14FC12FC122312E512E512AC16F316F31639133F12821282122C1319149F159F155217A914A9144213441244126F0FAE1283118311C20BC30D441844181911BB0FBB0F64163F173F17B417241624164D1368106E0D6E0DCE0E2C122C12670DE10DE910E9105710D50ED50E9C0F4310D315D3155416EF12EF12710C90099009A40A320B320BBB108007', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (202, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F48411E213E213AD15A117A11743140614021402143B135E105E100012DB13DB13FE135B0C5B0C8E07B410F811F811AF146714BA13BA132615971597150A13681368133C1470157E157E15260F870B870B051178137F167F167F164314431497159715971597159613AE11AE1168194D164D16170ED109D1092C11DA13231523150515F7148E148E14641483158315F41353125312EB19931D931D931DD60ADE0C071209122015CF15CF150B166C166C16D014DE14DE142D1612176B206B20CF16931D931D240E600F1D131D13B513621562155615871587155C152C168B158B15EC180F120F12C40D420E420E3D0F560F1C131C135716E414CC14CC144015E514E5149F1A9C199C198A1C8014110F110F930F700F700FFC0F261512131213DE14D214D214FE16F917F9172A180F192F162F163E16D80ED80E7D10DC0FDC0F5411D4146F146F14AF14D215B516B5168217A818A8183B18F214F2148C0F461046108D108A117915791530185C167415741509154E164E16AC16461A461ACF143D0FA00FA00F1F11CA10CA10761329152915E7176A18991599157C13FC12FC12F010CF1175127512B50FF810F8101E1129122912591631183118EF19601845188D148D149911781578151B10670F670FB20FE110E110871182151319131999129912F717F717CB18B014B0141910781578154A0F170F860F860F6311B212B2123E15B813B8135917E3171117111750196414140F140F200F410F410F2F0F10101010E11199159915AD18F615D317D317121A53197D1B7D1BD8170D120D12E9113F103F103F10C2109014901466143E133E13830D13151315C51372207D1B7D1BC30E890E890EF6101F0E450E450E99107B177B179B16B213B213B312D113CF12CF12CF18FB15360C360C1110111011106D0D7A0F7A0F8A108916891627192719FE14FE1438157D137D13C30DFB15C30DC30DC30DE410E410E410B2102F112F11A1135D165D16541997199719BD18C415FA13E10DDE0DC30DC30DC30DE410E410E410F81040124012F5133F163F162F186618041804189218AA164C094C096B0BE90EE90E42117A117A11DD105C13AD13AD1338142C152C152318F716F71603173C15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (203, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4CA1065086508F40831093109A20A36093609500C3D0A3D0AFC0C960FA712A712DE0F940F940FAD0E9E160C0C0C0CD109ED0AED0A7A0B13097F097F098E09480D480D0D0DC210C210570FB80FB80F6E0DA90C580BBD0BBD0B640B370C370C0D0A720A720A520B7D117D1176114B11571157114B11AE15AE15DE0C8A0D37103710D6109F109F10E110B80A830F830FCB113A123A124613D012D0128D1351175117600FD70E70107010BF105E115E115E110110461146110D115B105B10C4102C114C134C135B165C19521052101C113C133C13DE1389128912EF12EC11EC11C3118412A312A3121913A716A716AF1696177D117D113313E517E5173018D312181318133213741374135612291529155A14BB11BB11C6112712651287188718C91987128712AE125E115E11181014101410A914AB1553155315190E220F220F4F11D51623182318F414A710A710F80E5A0D870E870E4F125E145E14B714D012D012791427142714B5157815A812A812F6134F1282118211D612271227127E107C107C10220F1C13B618B618B618DD16DD1631186B11B211B211C113111411143415B912B912A40E050E870E870E4F0F41104110FF0F401240129912D610DB0FDB0FB114CB17551E551E551E41194119B41939143914BB0B15101510871087101B13B417B417C113691769178418DE14DE14C816AF19AF19D0182613F60FF60F63129A129A12E5125E10A81AA81A09172E152E154413DF11671367136213521352132613551155113114D312D31270126F12A217A217F513F314CF14CF149A134A134A13C111901390131B13821235133513E212E212E21251126E1360136013E114BC14BC14AA13D513D5136A13401310141014F01179127912B812831283124512C3119B149B1462152815EE13EE13D7127A127A122D1283128312F510421242120D14F9136D136D13FA134D15551655166214A014A014B110F510F510A3114012751075108311441244121D146E146E14AA16BC14141514151D164C129B109B105810511051103710ED10ED102011DA11DA112F11011041103C157E14CB15D915D915F014C411C4115C108A108A109B1029101B101B100311800F800F1D10DC0F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (204, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4320E320E5A0F0410761176117C11A911A9110510DB0EDB0E0E0CC210C21086113C116D126D12F10F810B560E560EF50F81108110FE10E310E310910F050D580A580AE80EF610F6102E128512851221106012B817B8177313C011C0112B10CC0FBA0FBA0F0C0CDC0BDC0B0E0E6D106D10EE10D011A610A610A7182F19D815D815D91211111111ED0F570F570F410E930E930EBD0F8210BD12BD12D711C010C0105C1954178C178C17E4169D129D122711AC0F9F0F9F0F2011D511D511690F051305132F12F30FD1119617C21A8B147D137D13B81201110111A90F5C105C103F1055115511E7103A13931293121A11A70F041304134012251225129012221122115F0F8D0F841084106212DF11DF1104136A116A11200EA40F7A147A14B9143611361115108610A90FA90FDF0E631163117F11E811E811C8115C1291109110410F90146C126C129F108F108F109E11711171110B11B60FB60F171264125F125F126612FB0FFB0FA20E1213E512E512F310C910C9107112D412CE12CE1296116B136B13F913FA13FA13C511AE0F901090101B139413E015E01517118A118A1124125D125D126712D511D51188120012DD10DD10CA10A310A3101B13021C171817184210D50FD50F69104B11EF12EF12A212D111D111231073117311FA100E110E115E104F11481148117911400F1C0F1C0F4010C112C11298125B105B1051102B0F7B0F7B0F7B0FA60E6B126B123C1376117611370FF00EF00EB60FFD11FD11CF117F1085128512C811860F860F9A0CE70CB612B6122712BE11BE11AA0D0310DA0FDA0FC70FA510A510C9128C118C111711EF10600B600BA10BA210F810F810EA0EC90DC90DA30FA20FA20FDD0FDE10DE10D310E60FC210C210700E920C920C910CA6148C178C17E513CC10CC105D107210D810D810911119131913A20F321032104D0D100B100B980E10143B153B156C14741141104110C910FD10FD1046130D110D11F910B40E250F250F2D0E1C0D1C0D4110A80F4E0F4E0F02119D129D12BD11C811C8115A12671112111211010F9F109F10B30E210D210DDC0F7810C410C4104512BD122813281340116D116D115210600F600F290E131203110311180EAB0B', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (205, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4CD0ECD0E670F2111211164108C0F7B177B17EB13D914D91459164B154B1570177D197D1951161814170F170FC50E0A11E610E610C613D514D514B01750165016B4177F1680168016C90FE50CE50CF907D90F3B0F3B0F0A11681368134710A311A311E20F7F10AE11AE119210D910D910040EE30EE30EF809150F130F130F6610EB106D116D11D8129211921154117B117B11AE0E670F670FAF0DB80FC40EC40ED1116B0ECB0FCB0FF8116B106B107E11C411C411C412C711DF17DF178917B80EB80ECF1647154715A70FC610F010F01083104D10C012C0127214E514E514D1142F152F15C71B7B187B18641B801CCB1FA40F1C0FA00F1B111B11D710471247120414DC14DC14C5140615291C291C4A19721F721FBE1F69209B129B128E122C112C1196111C111C11EB120A136B126B126D120B140B14F01A5D1C5D1C0D1D7E1D3E123E121013E813CF15CF152E12871287120914EE14EE1476134F114F1170146B17D20CD20C481D2211D80FD80FC31198139813DF136015601529168215B413B4130812BB16BB16C01982108210A80C2811AC12AC12E811AC11561256128A15D916D916DB153B153B15DE14C015C015EF1553190E150E158E0F94103E113E11AA140A140A14B614491749177E15171714181418FB15DC15DC158D1BD214D214A60E3B106E116E1171130D140D14E1135D16301530153F1502160216F516ED15ED156B17DE177115E70C980E6D11AC12AC128B12AA13AA136114501650162217AF18AF1864182B18CE15CE15001A7E1B5C0D5C0D26103413341310122F122F1227126D14A416A416D4189518951838197D167D16831A671BA30CA30CE30F501320162016B91384138413D71103150315A118A117A117D4172E15CE15CE15FE182A0E8D108D103616671667164A153414341434144C13591359139514971397138A11AE11AE117F11F30E071107112E13781278125415D61592169216121700180018891433133313B112F411B812B812050D6910F617F6174C15B516B5161618F219F2195D192C172C17D9131D14BD13BD13DE1225132513AB0BED109E189E18591745174517E819B21A2E182E181218EC13EC1358147C147C14DE12A111A111', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (206, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F4CB0FCB0F440585028502EA027E047E042C062C062C060D0A600E961796177F14471447144F187018EA05EA057803A103A1037102FC018D028D0274035003500371053B0F3B0F3F133F133F133F134515F503750375033203C003C0037B00D200D200D20031023102F5010604630C630C630C060E060E47159512280828085F094A074A07EF0128018B018B0158017E007E00E2018E028E026D0520072007970F8C127A127A12A50D060727062706E103E902E902B8033F033F03A500CB00AA01AA01EB0289058905CB1FB91662116211F60872057205CD04D102D1026F021A030E010E01D900EF01EF01AC01E802E8026920631FD018D018D60C6B062705270581022A022A027C01780178012901520152018F01AE0057002C1D2C1DEC19641264123609F808F808380351035103A7017B014E014E0131022E022E0294005200B51CB51C891CC61BC61BB71279074B044B046803640264021B03EE03EE030C02D801D801660155017118711871208D1B70127012A60E2A0B2A0BD506FD02FD0207030C03EB02EB029702CB01CB01F3017318061A061A0E1A821682161514F40AF40A5207BB0472047204D2032E042E04A6027A027A02BC02C110B217B2176B16EA17A514A514760E7E097E09E90357045704D703CB03D803D803D8034F054F057115AC19F619F619161A9F179F17FF0F930B930BDE03B903F102F102D802F603F6038806540854087E1B941A321B321B701BC61E3B173B17801066076607500478037803C003D903D9034A071F05210B671BC81B1F174E174E17301A91199119BD14770D770DD00756046D086D0871053F033F03D6049509F016F01692173B163B16B71851196B186B180417961496143606370637063706D207D207B408310B9510951086168E1536173617C5193D193D19E216851485145C11F20BE40CE40C4907D207D207FC0E2213E211E2118E1136163616C217371937190918CB17E813E813E20B420D420D420D1F0C1F0CAB0EFA12F812F81283124515A516A516A41537173717C81787118711171224112411E50A6E0B550C550C7811541285128512CF13691669169B16741774177B12AE11B213B213BD14410D410D360B460C460C', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (207, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C006632640C4DF4092A3FD464000000000000000000000000000000000E6100000140014004548F47018F819361948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F499169916B71C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44B1C4B1C271C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4970F361A361A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42309010D010D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EA046705CD0648F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4570081002F0248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F452001C00730048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D902D902880248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F407040704150448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4EF03BA05BA0548F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4B8079F099F0948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F46B092A14C90F48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4210B2A142A1448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F495097415991348F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F49C0F9C0F600E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FC0EFC0EC61148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4FC0E4E114E1148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4BE0C9514951448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F488104F124F1248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (208, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F41712381238121D13FB13FB134314EB13021402148514E616E616EA1678147814FA15931493142C138E12C6125A135A13D81183138313AB14581558156F155A155A1514177314391539158D129F139F13E5140615F813F8134E13D813D813C81691175E175E170816711571159613BB12BB12C2127A157A150F16C7150A130A1306130813D616D61677175717571755161D161D162214BA12981298128A18291A431643160314CC13CC1354138E148E14CF16171717178D154715341434142B148312831282155A180B170B177E144314431493156A1682178217511753155315D614A913A913D2128F128F124D167816E817E81703161316381438140B163D173D17BB176E176E17F5166B145E135E13551253165316B917C1155617561702176E176E17D315251925191A19CD18CF17CF178E1716131613B71074117411EF12E213C713C7136A15F5175A175A174B1AF219F2197817781478143514F20F3C0E3C0EDE0F110F110F730F1A101A101A1087106E136E13041322122212A90FB1124C104C10B20DD20FD20FBF0EAB0FAB0FC70E4D0D990D990DB80E0710E60FE60F7D0F130E130E040E5A0B5A0B260CA10CA10C0B0F071157106D0BAC0E9B0EDE0EDE0E920E6B0E6B0E960C2E0E2E0E860E390FC00FC00F8410780F780F7B110710700A700A9F0E970F970F7510360D360DEE0BAA0C0C100C100010B40FB40F451041104110A50FEA0FB408B4089F0C490D5B0F5B0FCC0CBA0EBA0ECD0EC50EC50EAF102D102D107C10FD10860E860E150E6B0C0C0D0C0D1C0B020C020CFF0DF00EF00E9B0F300ED40ED40EF60F291029105A10A410A410BA0F220E6C0D6C0D520B120E120E1D0F5D0E2F0D2F0D720DE90FE90F190F570F570F2A103710E410E4102E0E670DBD0CBD0C470D230F230F450C00090009740C5010360F360F8B0FC911C911CA11E312E312BF0C170CB80CB80C310D630C630CE30BE10AE90BE90B04110D100D10DB0ECC11CC11DA11FF13A316220CF30B4A0D820D820D820D6B0A6B0A8A0BA60DA60DC7119E119E11801097104F104F10D5136915550E550E140EEE0EEE0E6F0E990D990D980B8E0D491149118F11641164117D11A315A315E9159C15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (209, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD25404F3283424EF8464000000000000000000000000000000000E6100000140014004548F48410771B771B111C76177617B115EC15EC1541184F1A561856188C14D916D9169E188C168C160D121D13A41CA41C351C731973199B185C180E190E190716231623163B16011901192C18EB17D712D712BA1A391CD11AD11A7F1ABF15BF15D51375157515B0150F152F122F1258145F145F14531381118111291AA31BE01AE01AFF15141314135813FC14881688168D16F616F616681953195319E413041614105A18E2179617D012D012BA132E142E14F014E719E719041AEB17EB17131B2816C614C614A6147713D116D116A5157E157E15461322162216FA17A2172D182D188B17161716170E108F108F104210641251175117BA153D163D160B1659177513751317146A136A13B91399119911E70F870E850F850F850E1B15961996198E18861586154814161316138B139B139B137711801139123912E411DC10DC107210A214861C861C861C781C781C9716F313191419146913031303138D13D612D612A6121B1291129112880F701371187118781CBF17BF17D3147514751440142B142B14D413C413531353139013FD11FD115710AE12231823183818EB15EB1566155F140F140F141D15A214A214CB16F714F714FC13951295120710841121152115DB15EF14E314E31439163816381677150D180D18FA1B71186B196B19501BBE1ACA0FCA0FA413691469147914D014D0145F16BA16BA1618166A16871787172F1AAF1AAF1ACE18EE180610061005151C151C154415B21466166616F916A115A115E31581168116DF1699139913E212FD142F112F113A1520143A153A152C14B614B614E014B91AB91A3816AC1DFF16FF166B1671167116C613171240154015A613CF17CF17151392155B155B15EE16E315E315A4176A196A191E1798119811E810F212351635162C1552187F137F130A179F179F172417E617E617031DF21A15181518950F950F950FA31689161F141F14211D86198619B41458175817D217FB18341D341DD51EA219A219BE16950F950F69152F169B189B181F1DFA17621862188E176C156C156218F71AF71A931C371937198912FC10D9109C15CC134D15F915F9151E15251825184C19911491144616711818151815AC14F80EF80EAE0E1C15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (210, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E325404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4920C920CB70F8D0ED20ED20E681178167816751BF40CF40C1705E509240B240BAC0B360A360A8409620EA70BA70B680AF609F6090A0DB90EB90E1C10A407B104B1042A097E097E09B10F2C0D2C0DAC0A500F340C340C8C095B09EE0CEE0C140DFF0DFF0DF6076C036C03FE05FE05BC0DBC0DF6096B0A6B0A1410990B990B990B5C0BB80CB80CA20D130B130B11060806B606B606250CEA0BEA0B430BA10AA10A77132D107E0B7E0B9409FC09B00CB00CCD07AB0AAB0A9A0A820A820A340CB50AB50A430B5A0BB70C64129F0CE2093D093D093D09870987095B05D507D507D408E006E909E909370CFF0AFF0A790B5D0EAE0DAE0D5508BB07BB070D0882038203B801750286088608CC048A058A052D0D940D940D3810B011850F850F970FA111060B060BEC068C028C02B001F701F701F702DA05DA05040A590D7310731095163213311031109813C00BC00B810E6C0C6C0C9904FC03820282024007D60DD60D1A0F5D115D11841189138D138D131E14180EC00FC00F5F0C82068206C905F508F5085008DE0CDE0CBA0D5811A40DA40D0013E11140124012300AC70AC70A8605C405C40560066309630963099F0D081008106E1286168616BE1ABD11E809E809A5082D0A2D0A0303C5039B079B074D096F096F095912941294120F179B15470CEE18AA11C707D707D70755076E076E07FF055F0A5F0A490C490C490C6D13FD12D50FD50FFC119D0B661266129109731373137313EF0FEF0F9F118716271A271A2314D713D713E5133B0B3B0BEB0AFB057D137D13EF12130E130E09103710FD12FD12A116CD1ACD1AC6198E188E18BF12FF11D008D00809054C0C110D110D8D0D310B310B760CAE0EAE0E4610561A4F1B4F1BEB1AA710A710F606C903C903C9030D0CE008E00849092B102B10AA105511CE11CE11E2195C1A5C1AE915280C280C7707010466046604790B840B290C290CE50C211021109914931493142F1F701E701EA710570AB507B507720501040104D9100D0DCF0BCF0BEE0B9B169B16AD16EE163E113E119410371137118C0C1107110761050B080B081C150F17981898187C165517AF16AF164311280E280E6C0BE20AE20AE00B4B06CD07CD07A007D90F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (211, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF825404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4840984090C0E220F220F16106A115C115C115512C013C013FD13DA12DA1225157D1755165516B612E90BBE0CBE0C8C116E126E128A110B140B143D14831283126E135B13C015C0159817731673167D16470CC20DC20DDD13C115C115C115CE17791479147811FB10FB1082111A121A127416741674169B16E80A2F0EEB13EB1338165B165B1674139C119C112D10B610B610DF0EA9100D110D113511C112C112B70CC40F79177917141CB41AB41ABF16D1106E0E6E0E020FE70EE70EF20E620F620F810FF70FF70F5D0EFF11241024109213A0199E149E14BB11B80FB80FBB10F50FF50FE90F03111F101F1069104110D310D310A9167C147C14FC11471147115C1010101010B50E180F730D730D271076107610521041104B194B194B19E413E413D410F911C810C810A30DA20DA20DB90FBF0FBF0FC80F480E480E2A0EDE0FC611C6112417311CCE15CE15BD137D127D121A116F116F11BD108510CC0FCC0F620D8D0E8D0E740F791092189218331C551A551AA616D8196B146B14A0107F107F10F210DF0FDF0F410D740E740E530F671084108410D613D019D019D0199C1662166216B614B614B61425101710FE0EFE0E940FEE0FEE0F470CEA1054165416D914BE16BE16D21262166216B614B2132D112D11F610840F840F150F9D0E9D0E9D0B930B28122812D012E211211221124812CA12CA12D113DC10DC102311A40EA40EF60DD80E6B0FFB05C5054A0BAB17AB170918C810C8101C10FD0FFD0F3E12CE0FDC0FDC0F67102D0D2D0DDC0D2F0E3E0A3E0A220A091B091B2B17080FCF10CF104310E810E8107B0FD80FD80F6810F30FF30F130F300EF008F008D50C021B7B107B108A107A107A10FF0F14111411AC10F00F2A0F2A0F110F7A117A112B107606E80BE80BA114761276129412B00FB00F860E21116B106B10560FC30EC30E340FA710A710520F320B020B020B24145516871487143A0F3A0F3A0F0310C90FC90F2A11470F470F960E230F2D0F2D0F750EA410B010B010BF11100F100F530E940E940EB00F6B113511351189116C0F6C0F0D0F740E740ED90FB5122912291277115B0F5B0F300FCC0F4110411075113F123F12CC11DD10DD10A410EE0DF80D', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (212, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F43112311211126C146C145D144E124E122E126B11AE10AE104B10B30FB30F230F38103810E310931152165216B9111513D812D81209134F124F124F126A116A111C10290F290FC70F3010A411A4117F1131150A120A1251122112211272128F118F117F12761197119711041177117711D2108910891016116E16B811B8111B116B0F3A123A12481293109310F1101D121D120A12C910C9103112001078107810CB107711ED0EED0E5010CA0FCA0FCF0F4F0F4F0F8D0FDC12071307134512C510C510A410E30EE30E41100F10EA11EA11E50F410E410E450DCE0D03100310C612AE12AE122E10A90EA90E130F420D100B41107510B21170107010AD0DD70FD70FAA0D9C0F9C0FBD10D20FD20F4C0F6B0E481248121312FD115E105E1087103C0F3C0F910EA00FA00F900F4D104B0F4B0FBE108A0F8A0FB011701270123212B512AC0FAC0F0D107D0B7D0B6C0D8C108E128E122011FC10FC103611BA11BA116D117F11F810F810161236104B0F4B0F040BC50AC50A540F2C102C109011860F3F113F11C910F40FF40FF60DC010C010EC11620FB80CB80CB30A500C500C270F53104E0E4E0EA40EB40FB40F530E300E300EE20C980A800B800B360F8D0D190B190B960CBE0DBE0D47104B0F4B0FBF0F6B0E6B0E3D0EAB11320E320E950C941094106B0FD70D960B960B3D0CBA0CBA0C5F10311034123412F3108D108D10F81175117511BB13321332132F0E720F7E0E7E0E420E2A0C920F920FF910CF11CF118D13CB11CB11CB114812D112D112C010C313870D870D9B0C400E400E480FBD0BBD0B3810921264146414D111B411B4117712AB12AB1273108512F60FF60F820E900F900F1A0FF60DB50FB50F8E12BF13BF1332131412141210146F11511151114014F90FF10FF10F600F2A0F2A0F8B0E530E530E86111414141463148114D712D7123713C710C71090123D0EDB0EDB0E340C630D630DA60C840C750D750D251248134813BA13051305139E10260F260FE90F470E030E030EC90CA60A660C660C060C550D550D6A109513951308144E13FE11FE11131288108810F80D390D310B310B630A6B0A6B0ADA0BAE0D8F118F11E410881188112D0F1A0F1A0FF70F380F380F', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (213, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F0004072326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F45E115E114C11551255121F12A111A111FC105F115F11DC0EB70DAD0CAD0C8011201120119C0F560D431043102E11E211E2114311061131113111AF0E8F0C8F0C230C7C0C7C0C650EA50FA50F210F4C0FFA10FA103A10DD0FF310F3109210750F750FA40C190A190AA60A690D630F630F900F6C0E6C0E500FFB0DD00ED00E92118A108A10AC0FD40DD40DF50AEF0AFC0AFC0A190A870D870D3F0D5E0E5E0E440FEF0BAF0DAF0D231021118F108F106E0EBA0EBA0E000F830D830D2B0BC60C1A0D1A0D890E1A0F1A0F100B8B0CB10DB10D660E6C0E6C0E790CF211F211EC0F250F030D030D170F340D340D5A0EDE0EDE0EFD117D0ED20DD20DB40D5A101A121A128510121312139611370D370DD60E280F280FEB0E030E9815B5124711E911F512F5120F1007120712761267126712D01013101C101C10610F951295125811DB1207130713E712641364139C12651265127F12C8113C103C106E1012151215821A7A177A1786110513371237127C12711275127512F711AD11AD11A811A70FA70FF70EBB10EE0FEE0FB60F201420144C11EB0E58115811450E0A110A118E11B811B8113B11D60FC00EC00E3715B010B0105613E012E0129411ED101D0E1D0EEC0BD50D6A136A13A611FD12FD123B11E910E9106510281528157611B612E210E210DB133113F012F012C912CE13CE136F16D516D51660166111CE10CE10E01257135713F612F612F612C313971219121912DC13D813D8134716C514EF11EF116511091009105213F912F9125611BC10E91185121B12DF113D133D133F13281528158114B514B51438137B13F811F8113513FA10FA101C112812DA11DA119511CE11CE11B211541454143D153D153D153D152512B612B612EE0F6F106F102B0FB40E981098103410AE10791379131414DA13DA13DA13E911E9113B11BA11BA11D4110612901590157F12F11067106710401096129612651365136513CE13BE1176137613F3112B142B14E614351A351A571EAA10D111D111A511A712A7121D0FE60D500E500EF50F5F115F117811C613C613BC131617A51BA51BFB0F5C1198119811F1118E0F8E0F470FA40DA40D900DDB12DE12DE125813DC12DC12661474107410', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (214, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C3826404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4560D4A0E4616F614F6146916AA15AA157014791379135E16CA13CA13C5131E156A146A14651331103312331209190B150B15341546124612F01183139D139D13611387138713351635153515D813E3114B134B136619EB18EB1881150F11BB14BB14D414C213C2139614BC14BC140216B0151B151B1550158B0F631763176216191519151915CF14CF1441148A158A152218BC14C314C31465142E132E137917E711621162111B12191619166419C8169F169F16441774187418C9183317331770149716D616D616B30E7A11161216126115E015E01541176A1A6A1A10188B1C8B1C8B1C741B651665162517D616D61698157E13DB12DB1228117810781025186A1ACD1ACD1A281BEA19EA192117E316E3167C1706190619DB121515081508158513EC121A111A116814DF14DF146C16BE16BE16EB155715A017A0177F19B5172A102A10C714321532159313EA12EA12A911B211B2117911C112C112C112F4134B154B15AB1506135510551084169318931897179717A313A3135613D011D011021102110211AF0E1D1006120612250F5C1014111411B114DB14DB14B714B913B9135613D312D312B211BE10F30FF30F01106C116C11560E790FD010D01039139A139A13C9167B13D313D31371128A118A11C6103B103B10AB1137123712E10D9B0FE211E2112E126A137C157C152013881188115313571357137114E71106120612C51186108610E911EE121C131C13B812131613168F16A315A315D213D013DA13DA13F6121810181018112A112A112812B4130613061353115319341834181615A90CA90CE312BF12BF1290140F134D104D10BD114712980F980F1A12D013D0137A180E1A0E1A5918A00CA00C2509340F4C104C10631BC111C111F9119D15AE17AE1795107E117E11DE13111EC719C7192717620F620F8B1151185118BF18D415D4152E202E201D1F1D1FFC137312411241127D1EEB18EB18001C791B791BED17FF17FB17FB17AC1BBF1BBF1B341F6A1A940E940E2B0C8F0B8F0B3520381CCE1ECE1EB81C41194119BA0E5315531562169E179E170316C50B8E058E056205F505211A211A821DB81FB81FFA1E3A1C3A1CD119E710B410B4103B130E150E15', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (215, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D26404F3283424EF8464000000000000000000000000000000000E6100000140014004548F43110AE11C111C111CA1439151B111B1175104F114F113B13E610E610E61047104710E10F050D260AE31108125613EB12EB12461423172317770F3C0D3C0DC60E2A10AB10AB10761076107610CE0F28106D146D143114F212F21288130A160A16040EB40CA10DA10DD90D640E640E321704170417B013C411661966195A19CD14881388131315900D900DED09F50AF50A120CB40EB40E7F163E16DA16DA164C126619B318B318E318CF17CF173C1DC510C510630BC409AE0BAE0BAE0B5E0D5E0DCC128E128E122011C016C016C0167117A4183C1D3C1D1D1D161C161C5E19FD0DFD0D59153918391819144415DB10DB109416D115A417A417261A701A701AF21CED1CED1C9D1AC019141A141AD9197A1A7A1AFF1F7E1E7E1EB51793168E178E177F1892199219001B791C021A021AB119D21CD21C931A0F160F16CC1CD81BD81B06138F105817DE17DE17A916E619E619DB19461A461A20194B154B1592170F16600E600EB60A650B380F380F890D9B119B1104136A0F6A0F8A12CB10A30EA30E2F0E1F111F11D70F100C100CB607800A140C140C3D0B690A310B310BB90AEE0CEE0CE10F71137113760BFF0BFF0BCA0BFC086C086C08E306270C4E0B4E0B1B0D650C650CCD0B2D0A2D0A4F0AB90AC00DC00D8E0D2D0C2D0C320D310931091909280CDF0CDF0CC40D620D620D1A0DED0B940B940B700B030B030BCB0AF30CF30CD808E00835093509EA0C5D0D930D930DC30F3B113B118E1167106710D90CE50C9A0A9A0A440BB10BB10BB10BA00BA00B47128A14E715E7159C1ABA15BA1539163712910C910C230CB90BB90B7C0B210C210C210C791ADA1C9D158016A4215E1D5E1DC212590E590E4E0EE411E411E90BB20BB20B000A480A700A700AA1174D1DE421E421E4219C239C23C720981398136B13CA13CA13CA131B11D013D013D20A88098809930F8111C91AC91AE421432243229A227A212F222F220220D21DD21D6818681868181A16A40D660D660DAC101E17E016E016151FF720F720A7237824782457242824482348235B22B91DB91DE316840A840A6D181A11000F000FEE1423142314171EBD21DF22DF224724C022C022C0220C1D0C1D73149013E71BE71B', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (216, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF3D61E4C0066326404F3283424EF8464000000000000000000000000000000000E6100000140014004548F4260A0C09E90B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42810AB0BC00948F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4C411C411450C48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4CE0FCE0FE40E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41E0F750D750D48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42B0F9211921148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4671ED61ED61E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4D81BE21DDC1A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4650B8516DC1A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4320A320A0B0B48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F41D071D077B0748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4AE07AE07D60748F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F425098D0A8D0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A00B500E500E48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4DA1C1E20612148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F44D1D4C1F022148F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F481112C1A2A2048F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F42E162E165F0A48F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4A716A716451248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F48111B012B01248F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F448F4', 'MODIS_NDVI_2005_02_18.tif', '[2005-02-18,2005-03-06)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (217, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF9AB22EC95CB82540989537D1F812474000000000000000000000000000000000E6100000140014004548F4BB00BB00990099009600C100C100BE0042005D005D004800A2FFA2FFB1FFC3FFC3FFAFFF89FF89FFBB00BB00DB00D700D700AE00A000A0005800660066000E00E200E20075FFE200CA00CA006F009000AE00AE00C600C500C500A100ABFFABFFBDFFF7FF230023009B009E009E00E7FF4400440044006200800080008C006D00A800A800AB00AE00AE002300430043002F009E009E0022000C004F004F005B00790060006000D8FF56005600A400DC00DC00D3007C00BA00BA004F000D000D005300060006001200A700730073001D006800680068003001A800A8009600E100E100AE005700570073FF3EFF3EFF3900A500530030003000C300BF00BF001F022E042E04D8012D012D0137018302080108019000B801B801F90097008F008F0087009F009F008101A204700370034104940494043102470147015001910191016000CD00AF00910091009C008401840169044E014E010504CB04CB04A102DE02DE02DE02FC01DC01B400B400AD009A009A00DE00840184018401800180016E0195018A038A0395029B029B025A02000273017301730167046704E6008C02B602B60273080C070C073605B401B4014E00530253024901EA01B400C300C300F20499049904AC03C502C502F205DE04DE045E0AB102F900F900F0000B010B01AC01AF01B301B30199049A049A04CB030F071C081C08A009450B450BE106FA01FA016A01500150016D02FF010B020B023E0533057E087E084409DF0DDF0D9C0A400840086B078C050B030B037402A102A10254023302ED03ED03F504200520050B06B70DB70D660E650B790A790A7B063807380723060205020506021502F501F50103062004D503D503F80A880D880D0B134E0C4E0C630896089608A106CF05F9068D018D01A201DC02DC020E0950095009E90694099409D912920BDB0BDB0BDB076E056E05F504BA045B035B0390012F032F03FF06F0074F084F08480AB314B3141311BE11BE11B20615071507D407B306600460043D041304A806A8061907230723076B0E1F0F1F0F4F102C112C11B00B1C07D407D40786079203B104B1041C0445054505900718091809B9076F0D820D820D5811120D120D1A0AB208B2081109', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (218, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF6E682508B2CD2540989537D1F812474000000000000000000000000000000000E6100000140014004548F47A009100620062000F01990299029A029503950366012D02AF03AF033005550E550E6C131C151C15900044005C005C008700170217022203C601470247025402260326038A0EC213C2134514D814D814620078008500880088008704DF04DF045D0462046204AB04AA04AA048A0E3D1540144014D310C209670067007000430143019504CE05CE055205F1049005900519054F104F1031132F132F1347127008200020005C00C103C103C103F9038C058C055406960596054B0E660D660DDE113C12C911C91118113E00D800D8008D02CD02CD028E02F307F307C10AE70DE70D581056123A123A123A12661166113C12D301A002A002A402CD02CD02E302EC04EC04EC04BF08161316134E166B146B146B14661366136012BE01B102A702A7029502A902A9027002E202E202B705D208D208FD1176180618061834146A126A12DC0136026E026E02220309030903FF05F80474037403FE067D0D7D0D1214FB14FB149A17331D331D00021D022C02FB02FB028703870387031B060C090C090C09710A710AFF0D1A1468156815901E4A1C9D059D05D402F302F302C702CD02CD02840491059105A408DE085B115B11A112A712A7128A186E19CE05CE05D605D605D6055003240313041304AB0449084908680B4A0D4A0DD01237123712C814F518DF026C046C04FA05EB05EB052005A304A3041105E307E307C0098610AA17AA17FA17D114D1143215D802D502D502FA0214041404BA03540467046704B305010701070D0AE717E717981A771277121212020584038403D302A60362036203B1062F052F052306140614060A08E010191519152516BE0FBE0FF906F906C904C9048C048C038C03D304820782072C08FD08E40CE40C721423182318A415D40FD40FBA04E805BE06BE06B006B00554055405DD05EB08EB084E09060906092E119E1A9E1A8B16DB11AF0FC307C3071307DD05DD051505A506A5064D061F091F096C09480BDF13DF130415D015D0157E131511DF06DF06210510071007CC06A20509060906BE09AC0BAC0BC10B04150415D61313141314111599123B073B0788038F03C407C407D207B307B307780B810D810DC80F65176517FC14FC14401440142014', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (219, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF431E1C4707E32540989537D1F812474000000000000000000000000000000000E6100000140014004548F404115C175C171017761707190719AB1854135413D012E911E911FD10CE14CE145917D41E052105218B0E940E24142414941442124212F813E812E812A911960D960DD20E0419032003203126AA24AA24C209B10E14181418B1122F142F14900F3F0CA30BA30BDF0B680F680F9B1A3F233F23FB23762376237008C20C1418281728177513BF11BF11040D9E0B9E0B1E0B800E800E2C222C22962396232A22C41E2C112C11E715BA18BA18A3160E160E16410DFA088A088A08AF18CA15CA15DB1ED322D3228E26BF1F7110711000172617191719175216B709B709850B43074307AF18881E881ED91EA01DBE20BE20B11CF712DA15DA152E125C115C115C11E406E40673054307430737174522F41FF41F561FA61BA61BB81B4F142D152D15880C0F080F086D09A403850785077D0834143414A21D672167212B1FA81FA81FE61BA71A1D18FF18FF18261820172017CC076C036C032807C511C511381C3F1E0E1F0E1F7E1E591D591D4A1CC0186319631989180B170B17850A720CFB07FB076514CE1ECE1E991B2B1F2B1F361F361F361F6E190A19A812A8123C148414130A130A750A840884086918D11CD11CA81DBB1C1A1A1A1A28182818B912B9126A12400E400E3A06C005C00573038505850571124616A81DA81D721B0B190B1918170213D111D1116F0F9C0C9C0C8D05D102F708F708E40F11241124611B241A241ABC18B414B4148D1489105E114D0F4D0F0F0B030803089A0093079307A21A112411242924DF18BD12BD12BB12D912D91264114E0CF50CF50C2105110011006F02A103A21AA21A8F22A41DA41DDF18A018A018BD148B128B12DE137C0E0A060A0600050D06540754078C1D8C1D8C1D221EF21AF21A361BB51140114011290CDA0DDA0DAF0F6E09760E760E760E960E960E171AC91EC91E9A233A22FB18FB18720F070A070A5809AA0EAA0E1511A80FC010C010101565188D168D162A15EF15EF153A22D618D6182F10670967096809C90DA40D99129912B813E816E8162F16B916B9169B15F816F816B915200F100B100B6F09500A500A6D090D08A215A2150917FA16FA164617ED17A719A7190B17791679167D14C80EC80E900B87088708CC099408', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (220, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF17D412865CF82540989537D1F812474000000000000000000000000000000000E6100000140014004548F4A320732373234F1E7F1D7F1DB51FA316A316651DC820C720C72044198D0F8D0F110F400540054D04AA2418221822B5210422242124216D170A150A15151A981C981C4C18C60BC60BAC085704F903F9031D22BB215C225C22C72295219521291CCC15CC15D21229122912911013080D040D04A40456075607C41E0826FD20FD20F72335213521B322A118CF18CF184812370E370EBA08D403D4039501FC02FC02BF1FDE1FF71FF723F723AD1DA51CA51C8818CF18CF189414A609A6091B092D02690169010C021106381F381FAC1BCF1ACF1A0E1A901690161516F512160C160CF907EA05EA05F205A004A004C4055D03DD18DD18BD16BB12B413B4136511E50FE50F0B10160C160CF8041D031D0346058907EC08EC08230ADD1877167716AE0F49124912A209470D470DE20681068106AE08E10ACD08CD08A50B5B0B5B0BD60BD018621262125012640F640F270A29099F069F06CF05CE05CE051E058C058C051C0974097409C410DF142114FA11FA11910EE90BE90BF1087B067B069F06CF05CF0585055103D704D704790E470F470F28184812EB11EB11990D170917092708B105A405A405550592059205D00458055805790E7C0F7C0F0213DD10281028105A0E4009EF07EF07540676057605FF04F003F003A9043B04080C080C080CB10F580E580ED60B150E150E1309E306E3067F06CD05CD05D4050404010401047204380E380E380E2B0EF20DF20DE40B280B280B1D0AB507B507B507A604FB05FB054D08DB04DB04E7093C0D3C0D3710E311681226112611150CA80AA80ACF08280A280A04071A071A079008A50ACA0ACA0A440E8C128C12B115B512170C170C0F0EFC0BFC0B500AE8056E096E095C082C0B2C0B080B220A220A34106A136A13D016F50DD10DD10D930D510D390C390C7D0B400D400D340B160D160D210E480E471247128E1425142514A40D920D9D0D9D0D9A0C710C710C770CBE0FBE0FE00E160DFD0FFD0FC511E411E411CC13661266120D08950895089508710C680F210F210F3B0FCD0ECD0EAE0F0514051446164616461655169F13B0126F086F08D90A150C150CC20F2010201031107E127E12BB11C513351635163513951495143414A313', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (221, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFEC8909C5B10D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F4330333031C039E02EC01EC0144021D031D031609CD0ACD0AA00F671267120F13E6149E139E139A12C9008301830189023F013F01FA01FA01FA0152086D0C640E640E98108B128B121C1307100710BD11170623022302DB0017011701C102AB060A080A08A40B700B700BFC0FB411B4115513FD1147114711A6081607B403B40336036D046D04AB06A40BA40BA40B560B560BBD0F5511141214125D14051205121106A3079B079B07C50932093209AE098F10000B000B6113611361136113EC12EC12EF11361136115D039209320DBB0EBB0E1A0DF00BF00BF00B03130313EC13321432144814B214B914B91477133F1487078707240AE510E510FA0EB310B310A0132415141314131014551555150F12E512E51278139814E30AE30AA00C160B160B7B10DC12DA14DA14AE13DA13DA13DA1363126312BC0F5E11C712C712FB131E105F0F5F0FDB101C111C112912101510159C143D153D153D153F12141014101D1057155715FA194011FA11FA111912FA13FA135915D213CE13CE13641344134413E9116A116A11FB1239183918311AB10FBE11B212B212FA138D138D13D01224112411CD132514251467104410DF17DF17D41779197919B10FBF12C711C7117511DE10DE101C11C61142114211F910FB0FFB0F8411221B221B6D1BC21AC21A2B0E34121B121B125D117A10DA10DA10AB11C310C3100C116B136B13A416701BD61AD61A501BA31A3C143C14EE140E120E12A810111011103B11BB11BB11DC12B913611861183E19511951191E1B6C1BD314D314B1128F128F123611D60EA70FA70F21100E140E144316B818B8182A191A191A19621A84196A141D151D15EC145E115E113F1045114511D50FA913A9133A167517461846183A187B187B1867193313A714A714BF13B410B4103D0FED103D113D1140147E177E173F18D216D216BF162018201805155012CD1290119011020E6B0E6B0E67104A104A10461447174717D9181D17E014E0146F13D512D512B012C2111B101B10720E690F690FF50F5110511016158B166E196E190914641264121E12B411B411A313EA1117121712F412E0122D102D100F12BD15BD15C116961796175B1480148014B0113C123112', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (222, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BFC03F000407232640989537D1F812474000000000000000000000000000000000E6100000140014004548F49B129B12B911921192114B11F80FF80FDB0FFC0FAE0AAE0A7908A30AA30A77107A117A11CA12E312CD12CD12C5111D0F08110811A70F820F820F880D46094609CE09FD0BFD0B9A0E0911351135119E0F2F112F112F110B0F2C0E2C0EB70F680D680D170D9C0BE10EE10E5E0F9D0F9D0F3F0FC10CC10C140B311038113811E80E130F130FD90F280DC20CC20CDD0D121312132413191219120F0C430C190B190B47143811EF0EEF0E06109B129B12B50F850F850FF60DFD0FFD0F69101B11720E720EC10CAF0BAF0B3F148D13310E310E45116F116F11540FF20A950D950D0F0CFB0AFB0A300CFF10FF100910CE0DCE0D981489138F10D50DD50D7C0F95109510380A750675060A07AC08AC08900AB50DA60FA60F850EF40B65136513820F340F340FF10FFC10FC10CE088008930A930A0B0BA70AA70ABE0DB90FB90F1C0EE00E1F161F16E91261116111E80EBB0F050A050A0B08AF0AAF0A580DA60DA60DC70FED108A0F8A0F530C231A7F167F162E142A122A120012F10BF10B480AAF0EAF0E1F13F8127E127E1247110C0C0C0CA40A4E1878187818A214351135117810540F3B0F3B0FA111221022101C14571257124D0F550B550BDD0B1B1AD919461746172711E80FE80F900C650D650DB70D2C0E2C0E530EFC0D540D540DDE09210D210DA31AA41A4E194E192612051305131B0DBA0A9B0D9B0D7A09DF08DF08F80418061806C108A80DA80D6C1BD8191019D212D212ED11DA0BDA0B8A0CD60AD60A590AD407D4070E07CF08DD0BDD0B8E0DB911531953195C1801170117890F2E0B2E0B9909170A170A0F08AA07150D150D020E820C820CD10DB911C413C413CC1283128312D00F420C270A270A8C08B606B606A407100C100CD00E180E180E520FB2120611931093109B10EC0DEC0D470CE708E7087B07D909D909370B3711990F990FF610FE0EFE0E02122910B510B51084117C0F7C0FD70B100A9B089B0835077A087A0837115512551256145F0F5F0FA20ED51045128812881288138B0F8B0F150F760B760B4F09E707E7071F0F0412A612A612DE0E270A270A31121D12A914A9148A14E312E3122A13960E960E5C0AC407450645063C0C0C0D0C0D690CC908C908', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (223, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF94F5F6425C382640989537D1F812474000000000000000000000000000000000E6100000140014004548F4E312F70F21114E104E107F0ED60ED60EC10C6C0B6C0BA30C320F320F191153111414141461126815EA0FEA0F6710BA0FBA0FFD11C511C511ED0F241052115211271407150715B718AB15AB156D16C916BC0CBC0CB00FFD0FDC11DC11CC125E125E123513D813D813DC14421842181A1909188C168C164316D80B1C0E1C0E3E10A010A010071057115711C510FE13AF16AF16A318A515A5158414771477140C15D709CC0BCC0BF30E86108610610FD010C110C110741193149314AB12851385139A12AE16AE16B116C40BF20FE90BE90B2911241124111C100411041192105C105C10B2110911DC12DC12F0185B1A5B1AF40B1112A010A010E10FF510F5101D10A70E1C0F1C0F4B0EE511E511851168116811CA16CC18CC18E00EF10DBB0F7C107C109D1025112511A50FE60EE60E740E8210821052174417691469141616B416080E080E380F0D100D10BA10B70EB70EEB0E700FED0FED0F3B102F182F18691B971B971B231A2517E60DE60DD00C4E0E4E0E610F6A11730E730E9F0EC60FC60F22136817681771172418AE1AAE1AF4189C10C00CC00C690C3A0E3A0E7610101010100F10B911B9118413F51A0C1B0C1BB31A451845183F187010F60DF60DF30BE30CE30CBD0E0511071107111E1134123412541A2C1A2C1AD31AEC19EC190C181D138812FE0DFE0DB40ED90FD90FDA10B10FB10F6A13EE11EE11A717F718551A551A81195E175E17B91177120F110F11890ECF10CF10F211F810D512D5125312951395130A18121812185B1910181018B9113F110E11461246122E12691269122E13A713A713E81572167216D117B21488148814A116A114F10FF10FBA109C0F9C0FCD11FF12FF12AE12441244126615BF15191619166B156F126F123915AD1676107610A910A510A51047117F111E131E13CD140C150C15F314FF14FF145A13711671169719DA1AA60E821182116811BB13BB13BC13F614F6148714981498140C125612AC0FAC0F14132A192A194E1D2A0C2B0F2B0FD8105B145B14051664172B162B16EF16A212A212270FFF0EFF0EE50E2A142A14A118920A130EB810B8104D14C516C5164E191A1B1A1B6A17811381132014D91035123512E81241164116', 'MODIS_NDVI_2005_03_06.tif', '[2005-03-06,2005-03-22)');
INSERT INTO ndvi_modis (rid, rast, filename, acquisition_range) VALUES (224, '0100000100E9A9C42BFF10613FE6A9C42BFF1061BF69ABED81B14D2640989537D1F812474000000000000000000000000000000000E6100000140014004548F468154E15CF13CF13EB125B135B133410FD0D6A0C6A0C2A0B150A150AD90B920F920F26100A0E0A0EC916B417C317DC14DC14B7149F149F14F3146E146E143911BA07BA072D07140F6F0F6F0FA60C910A98169816A8162C132C137A17F515F5155214C21388148814121118071807E809310D310D6B10070DFA17FA1778167A160F140F141715A316A316BF15101710178B120C070C071B07C50A590E590E700D6216CA17CA177219411