% Advection-diffusion with stochastic viscosity coefficient, SBP-SAT
% implementation
% Time integration with 4th order Runge-Kutta

clear all;

% Choose mod == 'herm' for Hermite polynomials representing mu w. lognormal
% distribution, or choose mod == 'lege' for Legendre polynomials and
% uniform distribution of mu.

mod = 'lege';

% Spatial interval end points
left = 0.4;
right = 0.6;

% Problem input parameters
v = 2;
M = 1;
rho_0 = 0.5;

c1= 0.02;
c2= 0.01;

t = 0;
t0 = 0.005;
x0 = 0.5;


p = 4; % Number of gPC coefficients
m = 50; % Number of discretization points in space
T = 0.001; % End time
order = 6; % Order of accuracy of SBP operators, should be 2, 4 or 6.

% Setup parameters determined as functions of other parameters
dx = (right-left)/(m-1); % Spatial step length
dt = 0.2*dx^2/v; % Time step, here dependent on (Delta x)^2 but changes with ratio viscosity/velocity
V = v*eye(m*p); % Advective velocity, here assumed spatially uniform
x = linspace(left,right,m);


I_p=eye(p);
I_m=eye(m);

old = 1;
new = 2;

u = zeros(p*m,2);
%Compute inner triple products of the chosen basis functions
if mod == 'lege'
    C = Legendre_chaos(p-1);
end
if mod == 'herm'
    C = Hermite_chaos(p-1);
end

%Difference operators Dx=P^(-1)Q

[D1,D2,BD,D,Pnorm] = SBP_operators(m,dx,order);
P_inv = inv(Pnorm);
D1 = kron(D1,eye(p));
D2 = kron(D2,eye(p));


%Initialization and forcing terms
u_init = zeros(m*p,1);
if t0>0
    if mod == 'herm'
        u(:,old) = init_Hermite(rho_0,c1,c2,v,t,t0,x,x0,m,p);
    end
    if mod == 'lege'
        u(:,old) = init_Legendre(rho_0,c1,c2,v,t,t0,x,x0,m,p);
    end
end

% This is a first order approximation of the delta function
if t0 == 0
    u((m-1)/2*p+1,old) = rho_0/(dx);
end


if mod == 'herm'
    B1 = mu_SG_lognormal(c1,c2,p);
end
if mod == 'lege'
    if p>1
        B1 = C(:,:,1)*c1+C(:,:,2)*c2/sqrt(3);
    end
    if p==1
        B1 = C(:,:,1)*c1;
    end
end

% Compute force terms
E_0 = sparse(zeros(m));
E_0(1,1) = 1;
E_M = sparse(zeros(m));
E_M(m,m) = 1;
Sig_I_L = -v/2*eye(p);
Sig_V_L = B1;
Sig_V_R = -B1;
force_left_1 = sparse(kron(P_inv,I_p)*kron(E_0,Sig_I_L));
force_left_2 = kron(P_inv*D',I_p)*kron(E_0,Sig_V_L);
force_right = sparse(kron(P_inv,I_p)*kron(E_M,Sig_V_R));

g0 = zeros(m*p,1);
g1 = zeros(m*p,1);


B = kron(I_m,B1);

%Iterate over time with fourth order Runge-Kutta until time T is reached

while (t<T)
    if T-t<dt
        dt = T-t;
    end
    t = t+dt;
    if mod == 'herm'
        [g0 g1] = bdy_cond_Hermite(rho_0,c1,c2,v,t,t0,x,x0,m,p);
    end
    if mod == 'lege'
        [g0 g1] = bdy_cond_Legendre(rho_0,c1,c2,v,t,t0,x,x0,m,p);
    end
    
    % Fourth order Runge-Kutta in time
    k1 = dt*((-V*D1+B*D2)*u(:,old)+force_left_1*(u(:,old)-g0)+force_left_2*(u(:,old)-g0)+force_right*(kron(D,I_p)*u(:,old)-g1));
    k2 = dt*((-V*D1+B*D2)*(u(:,old)+k1/2)+force_left_1*(u(:,old)+k1/2-g0)+force_left_2*((u(:,old)+k1/2)-g0)+force_right*(kron(D,I_p)*(u(:,old)+k1/2)-g1));
    k3 = dt*((-V*D1+B*D2)*(u(:,old)+k2/2)+force_left_1*(u(:,old)+k2/2-g0)+force_left_2*((u(:,old)+k2/2)-g0)+force_right*(kron(D,I_p)*(u(:,old)+k2/2)-g1));
    k4 = dt*((-V*D1+B*D2)*(u(:,old)+k3)+force_left_1*(u(:,old)+k3-g0)+force_left_2*((u(:,old)+k3)-g0)+force_right*(kron(D,I_p)*(u(:,old)+k3)-g1));
    
    % Update the solution vector
    u(:,new)=u(:,old)+1/6*(k1+2*k2+2*k3+k4);
    u(:,old)=u(:,new);
    
    
    % Plot the solution
    for k=1:p
        u_plot(:,k) = u(k:p:end,new);
    end
    plot(x,u_plot,'-*');
    title(['gPC coefficients, t=' num2str(t,'%.4f')])
    leg_strs = {};
    for k=1:p
        legstrs{k} = ['gPC coe. ' num2str(k-1)];
    end
    
    legend(legstrs);
    drawnow;
end

u = u(:,new);

% Statistics are readily obtained from the gPC coefficients
u_num_exp = u(1:p:p*(m-1)+1);
u_num_var = zeros(m,1);

for i=1:m
    u_num_var(i,1) = sum(u((i-1)*p+2:i*p).^2);
end

% Compute reference statistics based on numerical quadrature of the exact
% solution
if mod == 'herm'
    [u_exp u_var] = statistics_adv_diff_lognorm(m,t,x,x0,c1,c2,rho_0,t0,v);
end
if mod == 'lege'
    [u_exp u_var] = statistics_adv_diff_uniform_mu(m,t,x,x0,c1,c2,rho_0,t0,v);
end

% Plot the computed and reference expectation and variance as a function of
% space (at time T)

figure;
plot(x,u_num_exp,'-r',x,u_exp,'--k','LineWidth',1.5);
h_legend = legend('Num','Ref');;
set(h_legend,'FontSize',12);
title('Mean','FontSize',16);

figure;
plot(x,u_num_var,'-r',x,u_var,'--k','LineWidth',1.5);
h_legend = legend('Num','Ref');;
set(h_legend,'FontSize',12);
title('Variance','FontSize',16);


