

% Stochastic Galerkin formulation of Burgers' equation, one spatial dimension
% Finite difference discretization in space (summation by parts + SAT) for SG
% Weak imposition of boundary conditions
% Runge-Kutta (4th order accurate) in time
% The assigned fourth and second order dissipation operators => one-sided diff. op.


clear all;

left = 0; right = 1; % Spatial end points
p = 4;% Number of polynomial chaos terms (order+1)
m = 100; % Discretization points in space
T = 0.7; % End time of simulation
% Solution parameters
mean_left = 0.9;
mean_right = -1.1;
sig_h_left = 0.3; % Assumed to be positive for correct analytical solution
% Assume same standard deviation everywhere for comparison with analytical
% solution
x0 = 0.5; % Initial location of shock, x0 in [left, right]


dx = (right-left)/(m-1);
dt = 0.05*dx;

sig_h_right = sig_h_left;
sig_h = sig_h_left;

x = linspace(left,right,m);

% Analytical solution of the truncated problem for p=1,2, else the exact
% coefficients of the infinte expansion problem
if p == 1
    u_ref = exact_solution_determ(mean_left,mean_right,T,m,left,right,x0,x);
elseif p == 2
    u_ref = exact_solution_2x2(mean_left,mean_right,sig_h,T,m,left,right,x0,x);
else
    u_ref = exact_solution_p_inf(mean_left,mean_right,sig_h,T,m,left,right,x0,x,p);
end



I_p = eye(p);
I_m = eye(m);
old = 1;
new = 2;
u = zeros(p*m,2);
order = 4; % Order of accuracy of SBP operators - SHOULD MATCH DISSIPATION OPERATORS

%Compute inner triple products
% Here we assume Gaussian distribution of the states, which causes no
% problem in terms of numerical stability
[C] = Hermite_chaos(p-1);

%Difference operators (D1 1st derivative)
[D1 D2,BS,S,P] = SBP_operators(m,dx,order);

% Invert the norm matrix P (assuming it is diagonal)
for j=1:m
    P_temp(j,j) = 1/P(j,j);
end
P_inv=sparse(P_temp);

kron_Dx = kron(D1,I_p);

%Initialization
u(:,old) = initial_conditions(m,p,C,x0,mean_left,sig_h_left,mean_right,sig_h_right,left,right);

%Compute force terms
E_1 = sparse(zeros(m));
E_1(1,1) = 1;
E_m = sparse(zeros(m));
E_m(m,m)=1;
[Sig_left,Sig_right] = penalty(p,u(1:p,1),u(p*(m-1)+1:p*m,1),C);
force_left = kron(I_m,I_p)*kron(P_inv,I_p)*kron(E_1,Sig_left);
force_right = kron(I_m,I_p)*kron(P_inv,I_p)*kron(E_m,Sig_right);

% Time dependent boundary conditions
t_tol = 1e-5; % Tolerance to avoid division with zero at t=0
if p == 1
    [g_left g_right] = boundary_cond_determ(mean_left,mean_right,t_tol,x0,left,right,m);
elseif p == 2
    [g_left g_right] = boundary_cond_2x2(mean_left,mean_right,sig_h,t_tol,x0,left,right,m);
else
    [g_left g_right] = boundary_cond_p_inf(mean_left,mean_right,sig_h,t_tol,x0,left,right,p,m);
end

%Iterate over time with fourth order Runge-Kutta
t=0;
while t<T
   if T-t<dt
      dt = T-t; % Make sure we end at t=T
   end
    t = t+dt;% Update the time
    
    % Compute system eigenvalues for dissipation. May be replaced by the
    % approximate expression in Chapter 6.
    
    max_ev = 0;
    for i=0:length(u)/p-1
        u_part = u(p*i+1:p*(i+1),1);
        eig_temp = max(abs(eig(I_p*A_matrix(u_part,p,C))));
        if eig_temp>max_ev
            max_ev = eig_temp;
        end
    end
    
    % These dissipation operators are chosen for the 4th order operators -
    % need to be adapted for different order of accuracy
    
    diss_2nd_ord = dissipation_2nd_der(m,p,P_inv,I_p,2*max_ev/(6*dx),dx);
    diss_4th_ord = dissipation_4th_der(m,p,P_inv,I_p,2*max_ev/(24*dx),dx);
    
    % Time-stepping by 4th order Runge-Kutta
    
    F1 = flux_func(u(:,old),p,C);
    k1 = dt*(-kron_Dx*F1+(diss_4th_ord+diss_2nd_ord+force_left+force_right)*u(:,old)-force_left*g_left-force_right*g_right);
    
    F2 = flux_func(u(:,old)+k1/2,p,C);
    k2 = dt*(-kron_Dx*F2+(diss_4th_ord+diss_2nd_ord+force_left+force_right)*(u(:,old)+k1/2)-force_left*g_left-force_right*g_right);
    
    F3 = flux_func(u(:,old)+k2/2,p,C);
    k3 = dt*(-kron_Dx*F3+(diss_4th_ord+diss_2nd_ord+force_left+force_right)*(u(:,old)+k2/2)-force_left*g_left-force_right*g_right);
    
    F4 = flux_func(u(:,old)+k3,p,C);
    k4 = dt*(-kron_Dx*F4+(diss_4th_ord+diss_2nd_ord+force_left+force_right)*(u(:,old)+k3)-force_left*g_left-force_right*g_right);
    
    u(:,new) = u(:,old)+1/6*(k1+2*k2+2*k3+k4);
    u(:,old) = u(:,new);
    
    % Update penalties for imposition of boundary conditions
    [Sig_left, Sig_right] = penalty(p,u(1:p,1),u(p*(m-1)+1:p*m,1),C);
    force_left = kron(I_m,I_p)*kron(P_inv,I_p)*kron(E_1,Sig_left);
    force_right = kron(I_m,I_p)*kron(P_inv,I_p)*kron(E_m,Sig_right);
    
    % Update the time dependent boundary conditions
    
    if p == 1
        [g_left g_right] = boundary_cond_determ(mean_left,mean_right,t,x0,left,right,m);
    elseif p == 2
        [g_left g_right] = boundary_cond_2x2(mean_left,mean_right,sig_h,t,x0,left,right,m);
    else
        [g_left g_right] = boundary_cond_p_inf(mean_left,mean_right,sig_h,t,x0,left,right,p,m);
    end
   
    % Plot the coefficients
    
    for k=1:p
        u_plot(:,k) = u(k:p:end,new);
    end
    plot(x,u_plot,'-*');
    title(['gPC coefficients, t=' num2str(t,'%.2f')])
    leg_strs = {};
    for k=1:p
        legstrs{k} = ['gPC coe. ' num2str(k-1)];
    end
    legend(legstrs);
    drawnow;
end

u=u(:,new);

for i=1:m
    u_var(i,1) = sum(u((i-1)*p+2:i*p,old).^2);
end


% Plot the numerical and analytical solution for the truncated 2x2 problem

if p == 2
    subplot(1,2,1);
    plot(x,u(1:p:end),'-','LineWidth',2,'Color','r');
    hold on;
    plot(x,u_ref(:,1),'--','LineWidth',2,'Color','b')
    legend('Numerical','Analytical, 2x2');
    title('u_0','fontsize',14,'fontweight','b');
    
    subplot(1,2,2);
    plot(x,u(2:p:(m-1)*p+2),'-','LineWidth',2,'Color','r');
    hold on;
    plot(x,u_ref(:,2),'--','LineWidth',2,'Color','b')
    legend('Numerical','Analytical, 2x2');
    title('u_1','fontsize',14,'fontweight','b');
end

% Plot the numerical approximation of order p and the analytical
% coefficients

if p ~= 2
    for k=1:p
        figure;
        plot(x,u(k:p:end),'-r',x,u_ref(:,k),'--b','LineWidth',2)
        legend('Numerical','Analytical');%,'Analytical expected value');
        title(['Solution gPC coefficient ' num2str(k-1)],'fontsize',14,'fontweight','b');
    end
end

