
function [g_left g_right] = boundary_cond_2x2(mean_left,mean_right,sig_h,t,x0,left,right,m)

% Compute the boundary conditions of the 2x2 stochastic Galerkin form of
% Burgers' equation for the Riemann problem

% Indata:
% mean_left,mean_right - Mean (u_0) of the left and right states
% sig_h - Standard deviation (u_1), assumed uniform over the spatial domain
% t - Time
% x0 - Initial shock location
% left - Lower limit of spatial interval
% right - Upper limit of spatial interval
% m - Number of spatial grid points

% Outdata:
% g_left - Dirichlet condition for left boundary
% g_right - Dirichlet condition for right boundary


% Exact solution for the 2 x 2 case

s1 = (mean_left+mean_right)/2-sig_h; % Shock speed 1
s2 = (mean_left+mean_right)/2+sig_h; % Shock speed 2

g_left = zeros(2*m,1);
g_right = zeros(2*m,1);
g_left(1) = mean_left;
g_left(2) = sig_h;
g_right(2*(m-1)+1) = mean_right;
g_right(2*(m-1)+2) = sig_h;

% One wave propagating to the left, the other to the right
if s1<=0 && s2>=0
    if t>(left-x0)/s1
        g_left(1)=(mean_left+mean_right)/2
        g_left(2) = (mean_left-mean_right)/2+sig_h;
        
    end
    if t>(right-x0)/s2
        g_right(2*(m-1)+1) = (mean_left+mean_right)/2;
        g_right(2*(m-1)+2) = (mean_left-mean_right)/2+sig_h;
    end
end

% Both waves propagating to the left
if s1<=0 && s2<=0
    if t>(left-x0)/s1 && t<(left-x0)/s2
        g_left(1)=(mean_left+mean_right)/2;
        g_left(2) = (mean_left-mean_right)/2+sig_h;
        
    end
    if t>(left-x0)/s2
        g_left(1) = mean_right;
        g_left(2) = sig_h;
    end
end

% Both waves propagating to the right
if s1>=0 && s2>=0
    if t>(right-x0)/s2 && t < (right-x0)/s1
        g_right(2*(m-1)+1) = (mean_left+mean_right)/2;
        g_right(2*(m-1)+2) = (mean_left-mean_right)/2+sig_h;
    end
    if t > (right-x0)/s1
        g_right(2*(m-1)+1) = mean_left;
        g_right(2*(m-1)+2) = sig_h;
    end
end